#!/usr/bin/env node
/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import esbuild from "esbuild";
import path from "path";
import fs from "fs";

// eslint-disable-next-line no-undef
const BASE = process.cwd();

let GIT_ROOT = BASE;
while (!fs.existsSync(path.join(GIT_ROOT, ".git")) && GIT_ROOT !== "/") {
  GIT_ROOT = path.join(GIT_ROOT, "../");
}
if (GIT_ROOT === "/") {
  // eslint-disable-next-line no-undef
  console.log("not found");
  // eslint-disable-next-line no-undef
  process.exit(1);
}
const GIT_HASH = GIT_ROOT === "/" ? undefined : git_hash();

let _package = JSON.parse(fs.readFileSync(path.join(BASE, "package.json")));

function git_hash() {
  const rev = fs
    .readFileSync(path.join(GIT_ROOT, ".git", "HEAD"))
    .toString()
    .trim()
    .split(/.*[: ]/)
    .slice(-1)[0];
  if (rev.indexOf("/") === -1) {
    return rev;
  } else {
    return fs.readFileSync(path.join(GIT_ROOT, ".git", rev)).toString().trim();
  }
}

/**
 * Problem: 
 *   No loader is configured for ".node" files: ../../node_modules/.pnpm/fsevents@2.3.3/node_modules/fsevents/fsevents.node
 * 
 * Reference:
 *   https://github.com/evanw/esbuild/issues/1051#issuecomment-806325487
 *
 * kept as reference, need to be tested on MacOs
 */
const nativeNodeModulesPlugin = {
  name: 'native-node-modules',
  setup(build) {
    // If a ".node" file is imported within a module in the "file" namespace, resolve 
    // it to an absolute path and put it into the "node-file" virtual namespace.
    build.onResolve({ filter: /\.node$/, namespace: 'file' }, args => ({
      path: require.resolve(args.path, { paths: [args.resolveDir] }),
      namespace: 'node-file',
    }))

    // Files in the "node-file" virtual namespace call "require()" on the
    // path from esbuild of the ".node" file in the output directory.
    build.onLoad({ filter: /.*/, namespace: 'node-file' }, args => ({
      contents: `
        import path from ${JSON.stringify(args.path)}
        try { module.exports = require(path) }
        catch {}
      `,
    }))

    // If a ".node" file is imported within a module in the "node-file" namespace, put
    // it in the "file" namespace where esbuild's default loading behavior will handle
    // it. It is already an absolute path since we resolved it to one above.
    build.onResolve({ filter: /\.node$/, namespace: 'node-file' }, args => ({
      path: args.path,
      namespace: 'file',
    }))

    // Tell esbuild's default loading behavior to use the "file" loader for
    // these ".node" files.
    let opts = build.initialOptions
    opts.loader = opts.loader || {}
    opts.loader['.node'] = 'file'
  },
}

const buildConfigBase = {
  outdir: "lib",
  bundle: true,
  minify: false,
  target: ["es2020"],
  loader: {
    ".key": "text",
    ".crt": "text",
    ".node": "file",
    ".html": "text",
    ".svg": "dataurl",
  },
  sourcemap: true,
  define: {
    __VERSION__: `"${_package.version}"`,
    __GIT_HASH__: `"${GIT_HASH}"`,
  },
  //plugins: [nativeNodeModulesPlugin],
};

/**
 * Build time libraries, under node runtime
 */
const buildConfigBuild = {
  ...buildConfigBase,
  entryPoints: ["src/index.build.ts"],
  outExtension: {
    ".js": ".mjs",
  },
  format: "esm",
  platform: "node",
  external: ["esbuild"],
  // https://github.com/evanw/esbuild/issues/1921
  // How to fix "Dynamic require of "os" is not supported"
  // esbuild cannot convert external "static" commonjs require statements to static esm imports
  banner: {
    js: `
    import { fileURLToPath } from 'url';
    import { createRequire as topLevelCreateRequire } from 'module';
    const require = topLevelCreateRequire(import.meta.url);
    const __filename = fileURLToPath(import.meta.url);
    const __dirname = path.dirname(__filename);
`,
  },
};

/**
 * Development libraries, under node runtime
 */
const buildConfigTesting = {
  ...buildConfigBase,
  entryPoints: ["src/index.testing.ts"],
  outExtension: {
    ".js": ".mjs",
  },
  format: "esm",
  platform: "browser",
  external: ["preact", "@gnu-taler/taler-util", "jed", "swr", "axios"],
  jsxFactory: "h",
  jsxFragment: "Fragment",
};

/**
 * Testing libraries, under node runtime
 */
const buildConfigNode = {
  ...buildConfigBase,
  entryPoints: ["src/index.node.ts", "src/cli.ts"],
  outExtension: {
    ".js": ".cjs",
  },
  format: "cjs",
  platform: "node",
  external: ["preact"],

};

/**
 * Support libraries, under browser runtime
 */
const buildConfigBrowser = {
  ...buildConfigBase,
  entryPoints: [
    "src/tests/mock.ts",
    "src/tests/swr.ts",
    "src/index.browser.ts",
    "src/live-reload.ts",
    "src/stories.tsx",
  ],
  outExtension: {
    ".js": ".mjs",
  },
  format: "esm",
  platform: "browser",
  external: ["preact", "@gnu-taler/taler-util", "jed", "swr", "axios"],
  jsxFactory: "h",
  jsxFragment: "Fragment",
};

[
  buildConfigNode,
  buildConfigBrowser,
  buildConfigBuild,
  buildConfigTesting,
].forEach((config) => {
  esbuild.build(config).catch((e) => {
    // eslint-disable-next-line no-undef
    console.log(e);
    // eslint-disable-next-line no-undef
    process.exit(1);
  });
});
