/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 * @author Nic Eigel
 */

import { useLang, useTranslationContext } from "@gnu-taler/web-util/browser";
import { VNode, h } from "preact";
import { FormErrors, FormProvider } from "../../components/forms/FormProvider.js";
import { LangSelector } from "../../components/menu/LangSelector.js";
import { Settings, useSettings } from "../../hooks/useSettings.js";

function getBrowserLang(): string | undefined {
  if (typeof window === "undefined") return undefined;
  if (window.navigator.languages) return window.navigator.languages[0];
  if (window.navigator.language) return window.navigator.language;
  return undefined;
}

export function Settings({ onClose }: { onClose?: () => void }): VNode {
  const { i18n } = useTranslationContext()
  const borwserLang = getBrowserLang()
  const { update } = useLang(undefined, {})

  const [value, updateValue] = useSettings()
  const errors: FormErrors<Settings> = {
  }

  function valueHandler(s: (d: Partial<Settings>) => Partial<Settings>): void {
    const next = s(value)
    const v: Settings = {
      advanceOrderMode: next.advanceOrderMode ?? false,
      dateFormat: next.dateFormat ?? "ymd"
    }
    updateValue(v)
  }

  return <div>
    <section class="section is-main-section">
      <div class="columns">
        <div class="column" />
        <div class="column is-four-fifths">
          <div>

            <FormProvider<Settings>
              name="settings"
              errors={errors}
              object={value}
              valueHandler={valueHandler}
            >
              <div class="field is-horizontal">
                <div class="field-label is-normal">
                  <label class="label">
                    <i18n.Translate>Language</i18n.Translate>
                    <span class="icon has-tooltip-right" data-tooltip={"Force language setting instance of taking the browser"}>
                      <i class="mdi mdi-information" />
                    </span>
                  </label>
                </div>
                <div class="field field-body has-addons is-flex-grow-3">
                  <LangSelector />
                  &nbsp;
                  {borwserLang !== undefined && <button
                    data-tooltip={i18n.str`generate random secret key`}
                    class="button is-info mr-2"
                    onClick={(e) => {
                      update(borwserLang.substring(0, 2))
                    }}
                  >
                    <i18n.Translate>Set default</i18n.Translate>
                  </button>}
                </div>
              </div>
            </FormProvider>
          </div>
        </div>
        <div class="column" />
      </div>
    </section >
    {onClose &&
      <section class="section is-main-section">
        <button
          class="button"
          onClick={onClose}
        >
          <i18n.Translate>Close</i18n.Translate>
        </button>
      </section>
    }
  </div >
}