--
-- This file is part of TALER
-- Copyright (C) 2023 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--
-- @author Özgür Kesim

CREATE FUNCTION create_table_age_withdraw(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'age_withdraw';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(age_withdraw_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',h_commitment BYTEA NOT NULL CONSTRAINT h_commitment_length CHECK(LENGTH(h_commitment)=64)'
      ',max_age SMALLINT NOT NULL CONSTRAINT max_age_positive CHECK(max_age>=0)'
      ',amount_with_fee taler_amount NOT NULL'
      ',reserve_pub BYTEA NOT NULL CONSTRAINT reserve_pub_length CHECK(LENGTH(reserve_pub)=32)'
      ',reserve_sig BYTEA NOT NULL CONSTRAINT reserve_sig_length CHECK(LENGTH(reserve_sig)=64)'
      ',noreveal_index SMALLINT NOT NULL CONSTRAINT noreveal_index_positive CHECK(noreveal_index>=0)'
      ',h_blind_evs BYTEA[] NOT NULL CONSTRAINT h_blind_evs_length CHECK(cardinality(h_blind_evs)=cardinality(denom_serials))'
      ',denom_serials INT8[] NOT NULL CONSTRAINT denom_serials_array_length CHECK(cardinality(denom_serials)=cardinality(denom_sigs))'
      ',denom_sigs BYTEA[] NOT NULL CONSTRAINT denom_sigs_array_length CHECK(cardinality(denom_sigs)=cardinality(denom_serials))'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Commitments made when withdrawing coins with age restriction and the gamma value chosen by the exchange. '
     'It also contains the blindly signed coins, their signatures and denominations.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'The gamma value chosen by the exchange in the cut-and-choose protocol'
    ,'noreveal_index'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'The maximum age (in years) that the client commits to with this request'
    ,'max_age'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Commitment made by the client, hash over the various client inputs in the cut-and-choose protocol'
    ,'h_commitment'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Reference to the public key of the reserve from which the coins are going to be withdrawn'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature of the reserve''s private key over the age-withdraw request'
    ,'reserve_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Array of references to the denominations'
    ,'denom_serials'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Array of the blinded envelopes of the chosen fresh coins, with value as given by the denomination in the corresponding slot in denom_serials'
    ,'h_blind_evs'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Array of signatures over each blinded envelope'
    ,'denom_sigs'
    ,table_name
    ,partition_suffix
  );
END
$$;


CREATE FUNCTION constrain_table_age_withdraw(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'age_withdraw';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD PRIMARY KEY (h_commitment);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_h_commitment_reserve_pub_key'
    ' UNIQUE (h_commitment, reserve_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_age_withdraw_id_key'
    ' UNIQUE (age_withdraw_id);'
  );
END
$$;


CREATE FUNCTION foreign_table_age_withdraw()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'age_withdraw';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub)'
    ' REFERENCES reserves(reserve_pub) ON DELETE CASCADE;'
  );
END
$$;


INSERT INTO exchange_tables
  (name
  ,version
  ,action
  ,partitioned
  ,by_range)
VALUES
  ('age_withdraw', 'exchange-0002', 'create',   TRUE ,FALSE),
  ('age_withdraw', 'exchange-0002', 'constrain',TRUE ,FALSE),
  ('age_withdraw', 'exchange-0002', 'foreign',  TRUE ,FALSE);

