/* 
 *	FIG : Facility for Interactive Generation of figures
 *
 *	Copyright (c) 1985 by Supoj Sutanthavibul (supoj@sally.UTEXAS.EDU)
 *	January 1985.
 *	1st revision : Aug 1985.
 *
 *	Zoom function implemented by Frank Schmuck (schmuck@svax.cs.cornell.edu)
 *	July 1988.
 *
 *	%W%	%G%
*/
#include <suntool/tool_hs.h>
#include <stdio.h>
#include <math.h>
#include "func.h"
#include "object.h"
#include "const.h"

extern int		canvas_swfd;
extern struct cursor	null_cursor;
extern struct cursor	crosshair_cursor;
extern F_compound	objects;
extern struct pixwin	*sideruler_pixwin;
extern struct pixwin	*topruler_pixwin;
extern			(*canvas_kbd_proc)();
extern			(*canvas_locmove_proc)();
extern			(*canvas_leftbut_proc)();
extern			(*canvas_middlebut_proc)();
extern			(*canvas_rightbut_proc)();
extern			null_proc();
extern			set_popupmenu();

/*
 *	Zoom function is implemented by translating and scaling the whole
 *	figure according to the currently selected zoom factor and position.
 *	Advantage of this approach: most other codes works unchanged in zoomed
 *	mode.  Drawbacks: 1. The figure must be unzoomed before storing it to
 *	a file.  2. There is a danger that rounding errors in the scaling cause
 *	the figure to change if you zoom in and out several times; currently 
 *	this is no problem, because all parameters that are changed by scaling
 *	are integers.
 *
 *	Zooming makes it necessary to distinguish between the real (or absolute)
 *	coordinates of an object and its current screen (or zoomed) coordinates.
 *	This file provides a set of functions 
 *	    zoomed_x(), zoomed_y(), and unzoomed_x(), unzoomed_y()
 *	that translate between the two representations.  There are two other
 *	translation function for length-values that are not coordinates (e.g.
 *	circle radius):
 *	    zscaled(), unzscaled();
 *	these functions only multiply/divide by the current zoom factor without
 *	adding an offset.
 *
 *	The current zoom setting is given by the two global variables
 *	    zoom_factor (current zoom magnification)
 *	    zoom_fix	(absolute coordinates of the point currently displayed
 *			at the center of the canvas window)
 *	In other words, following equalities will always hold:
 *	    zoom_factor == zscaled(1)
 *	    zoom_fix == {unzoomed_x(CANVS_CENTER_X), unzoomed_y(CANVS_CENTER_Y)} 
 *
 *	Other files that had to be changed to implement zooming:
 *
 *	    ruler.c, grid.c:
 *		Top and siderulers as well as the canvas grid must be adjusted
 *		according to current zoom factor and position.
 *	
 *	    canvas.c: 
 *		Mouse coordinates must be rounded to multiples of the current
 *		zoom factor.
 *
 *	    change.c:
 *		Coordinates and other parameters (e.g. circle radius) must 
 *		be translated to absolute values before being displayed in
 *		the change window.  Before applying changes absolute values
 *		must be translated back to zoomed values.
 */


int			zoom_factor = 1;
F_pos			zoom_fix = {CANVS_CENTER_X, CANVS_CENTER_Y};
			zoom_left_proc(), zoom_middle_proc();

zoom_selected()
{
	canvas_kbd_proc = null_proc;
	canvas_locmove_proc = null_proc;
	canvas_leftbut_proc = zoom_left_proc;
	canvas_middlebut_proc = zoom_middle_proc;
	canvas_rightbut_proc = set_popupmenu;
	set_cursor(&crosshair_cursor);
	reset_action_on();
	}

static
zoom_left_proc(x, y, xm, ym, event)
int	x, y, xm, ym;
Event  *event;
{
	if (event->ie_shiftmask & SHIFTMASK)
	    zoom_out();
	else
	    zoom_in(x, y);
	}

static
zoom_middle_proc(x, y, xm, ym, event)
int	x, y, xm, ym;
Event  *event;
{
	if (event->ie_shiftmask & SHIFTMASK)
	    zoom_back();
	else
	    zoom_position(x, y);
	}

unzoomed_x(x)
/* translate x from screen coordinates to absolute coordinates */
{
	return (x-CANVS_CENTER_X)/zoom_factor + zoom_fix.x;
	}
unzoomed_y(y)
/* translate y from screen coordinates to absolute coordinates */
{
	return (y-CANVS_CENTER_Y)/zoom_factor + zoom_fix.y;
	}

zoomed_x(x)
/* translate x from absolute coordinates to screen coordinates */
{
	return (x-zoom_fix.x)*zoom_factor + CANVS_CENTER_X;
	}
zoomed_y(y)
/* translate y from absolute coordinates to screen coordinates */
{
	return (y-zoom_fix.y)*zoom_factor + CANVS_CENTER_Y;
	}

unzscaled(x)
/* translate zoomed value to absolute value */
{
	return x/zoom_factor;
	}
zscaled(x)
/* translate absolute value to zoomed value */
{
	return x*zoom_factor;
	}

redisplay_canvas_and_rulers()
{
	redisplay_canvas();
	pw_batch_on(sideruler_pixwin);
	pw_batch_on(topruler_pixwin);
	show_sideruler();
	show_topruler();
	pw_batch_off(sideruler_pixwin);
	pw_batch_off(topruler_pixwin);
	put_msg("zoom factor = %d", zoom_factor);
	}

zoom_in(x, y)
int	x, y;
{
	translate_compound(&objects, CANVS_CENTER_X-x, CANVS_CENTER_Y-y);
	zoom_fix.x = unzoomed_x(x);
	zoom_fix.y = unzoomed_y(y);
	if (zoom_factor < 4) {
	    scale_compound(&objects, 2.0, 2.0, CANVS_CENTER_X, CANVS_CENTER_Y);
	    zoom_factor *= 2;
	    }
	win_setmouseposition(canvas_swfd, CANVS_CENTER_X, CANVS_CENTER_Y);
	set_rulermark(CANVS_CENTER_X, CANVS_CENTER_Y);
	clean_up();
	redisplay_canvas_and_rulers();
	}

zoom_out()
{
	if (zoom_factor == 1)
	    return;
	scale_compound(&objects, 0.5, 0.5, CANVS_CENTER_X, CANVS_CENTER_Y);
	zoom_factor /= 2;
	clean_up();
	redisplay_canvas_and_rulers();
	}

zoom_position(x, y)
int	x, y;
{
	translate_compound(&objects, CANVS_CENTER_X-x, CANVS_CENTER_Y-y);
	zoom_fix.x = unzoomed_x(x);
	zoom_fix.y = unzoomed_y(y);
	win_setmouseposition(canvas_swfd, CANVS_CENTER_X, CANVS_CENTER_Y);
	set_rulermark(CANVS_CENTER_X, CANVS_CENTER_Y);
	clean_up();
	redisplay_canvas_and_rulers();
	}
	
zoom_back()
{
	double	scale_factor;
	int	change = 0;

	if (zoom_factor > 1) {
	    scale_factor = 1.0 / (double)zoom_factor;
	    scale_compound(&objects,
	      scale_factor, scale_factor, CANVS_CENTER_X, CANVS_CENTER_Y);
	    zoom_factor = 1;
	    change = 1;
	    }
	if (zoom_fix.x != CANVS_CENTER_X  ||  zoom_fix.y != CANVS_CENTER_Y) {
	    translate_compound(&objects,
	      zoom_fix.x - CANVS_CENTER_X, zoom_fix.y - CANVS_CENTER_Y);
	    zoom_fix.x = CANVS_CENTER_X;
	    zoom_fix.y = CANVS_CENTER_Y;
	    change = 1;
	    }
	if (change) {
	    clean_up();
	    redisplay_canvas_and_rulers();
	    }
	}
