/*****************************************************************
**
** MathSpad 0.60
**
** Copyright 1996, Eindhoven University of Technology (EUT)
** 
** Permission to use, copy, modify and distribute this software
** and its documentation for any purpose is hereby granted
** without fee, provided that the above copyright notice appear
** in all copies and that both that copyright notice and this
** permission notice appear in supporting documentation, and
** that the name of EUT not be used in advertising or publicity
** pertaining to distribution of the software without specific,
** written prior permission.  EUT makes no representations about
** the suitability of this software for any purpose. It is provided
** "as is" without express or implied warranty.
** 
** EUT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
** SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL EUT
** BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
** DAMAGES OR ANY DAMAGE WHATSOEVER RESULTING FROM
** LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
** OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
** OF THIS SOFTWARE.
** 
** 
** Roland Backhouse & Richard Verhoeven.
** Department of Mathematics and Computing Science.
** Eindhoven University of Technology.
**
********************************************************************/
/*
**  File  : sources.c
**  Datum : 27-3-92
**  Doel  : initialiseren en eindigen van windowomgeving en windowmanager
**          registreren van windows en bepalen van windowtype
**          beschikbaar maken van fonts en GC's
**          enkele algemene functies
*/

#include "mathpad.h"
#include "system.h"
#include "funcs.h"
#include "sources.h"
#include "message.h"
#include "notatype.h"
#include "output.h"
#include "remark.h"
#include "intstack.h"
#include <dirent.h>
#include <sys/stat.h>
#include <ctype.h>
#include <signal.h>
#include <pwd.h>
#include <unistd.h>
#include "config.h"

#define GENERALINFOFILE "fonts.mpt"

FUNCTIONS nowindowfunc = { NULL } ;
FUNCTIONS *eventfunc[MAXWINDOWTYPE] =
    { &nowindowfunc, &buttonfuncs,       &scrollbarfuncs, &stringfuncs,
      &menufuncs,    &maineditfuncs,     &editfuncs,      &mainbufferfuncs,
      &bufferfuncs,  &mainnotationfuncs, &notationfuncs,  &notadeffuncs,
      &symbolfuncs,  &defaultfuncs,      &remarkfuncs,    &findfuncs,
      &fileselcfuncs,&popupfuncs,        &checkboxfuncs };

Display *display;
Visual *visual=NULL;
Colormap colormap;
static int usegray=0;
static int def_depth=1;
static int visualclass=-1;
static char *displayname = NULL;
Bool iconic=False, output_mode = MPTEX;
unsigned long save_time = 0;
int save_minute = 20;
char *wmselection = NULL;
char *latexselection = NULL;
unsigned long message_time = 0, last_time = 0;
int screen_num;
Window root_window;
unsigned int display_height, display_width;
unsigned long white_pixel, black_pixel;
XClassHint class_hints;
Atom WM_SAVE_YOURSELF, WM_DELETE_WINDOW;
Atom protocol[2];
XWMHints wm_hints;
Pixmap icon_pixmap;
static char * generalinfofile=NULL;
static Cursor wait_cursor=None;

int font_height_max, font_ascent_max, font_descent_max;
short font_width_max;

int font_height_ar[NR_SIZE], font_ascent_ar[NR_SIZE], font_descent_ar[NR_SIZE];
short font_width_ar[NR_SIZE];

char *project_name = NULL;

#define BUFSIZE 1000
static char buffer[BUFSIZE];

#include "progicon"

/*
**  Functies en types die fonts toegankelijk maken:
*/

typedef
struct {
    short width, ascent, descent, left, right;
} CHARINFO;

typedef
struct {
    char *name;
    char *code;
    short normal[256], math[256];
    short locks;
} LATEXINFO;

typedef
struct {
    char *name;
    Font f_id;
    unsigned min_char, max_char;
    short f_width;
    int f_height, f_ascent;
    CHARINFO *char_info;
    short locks;
} MXFONTINFO;

typedef
struct {
    short *size;
    short *orsize;
    char *loaded;
    char *name;
    short normal;
    short max;
    short label;
    short fnr;
} SIZEINFO;

typedef
struct {
    short max;
    char *loaded;
    SIZEINFO **info;
    int lock;
    char *filename;
} FONTSIZEINFO;

typedef
struct {
    short tnr;
    char *fontname, *opentextfont, *openmathfont, *closefont, *texfilename;
} GENFONTINFO;

typedef
struct {
    short fnr, lnr, fsnr, number, page[2], nrml, max;
    short *size;
    char *loaded;
    char *name;
} FONTINFO;

PAGELIST pagelist;

static int grnr=0;  /* which group of fonts is used?  */

static INTSTACK *font_stack = NULL;

static LATEXINFO linfo[30];
static MXFONTINFO xfinfo[200];
static FONTSIZEINFO fsinfo[200];

static int nr_latex=0, nr_xfont=0, nr_fsinfo=0;

static FONTINFO finfo[NR_SIZE][MAXFONTS];
static GENFONTINFO gfinfo[256];
static unsigned char fnr2anr[NR_SIZE][256];
static int number_of_fonts[NR_SIZE];
static int alternative[NR_SIZE] = {0};
static int script[NR_SIZE] = {0};

static void remove_return(char *str)
{
    int i;

    if (str && (i=strlen(str)) && (str[i-1] == '\n')) {
	str[i-1] = '\0';
    }
}

static int read_integer_short(char **txt)
{
    int n=0;
    Bool neg=False;

    if (**txt=='-') { neg=True; (*txt)++;}
    while (isdigit(**txt)) {
	n=n*10+ (**txt)-'0';
	(*txt)++;
    }
    return (neg? -n:n);
}

static int read_integer(char **txt)
{
    int n=read_integer_short(txt);

    while (isspace(**txt)) (*txt)++;
    return n;
}

static void make_pagelist(int nr)
{
    int i,j,h,n;

    n=0;

    for (i=0; i<256; i++) {
	h = fnr2anr[nr][i];
	if (finfo[nr][h].number==i && i!=BUTTONFONT)
	    if (finfo[nr][h].page[0]== -1) {
		pagelist.page[n].fnr = i;
		pagelist.page[n++].spos = 0;
		if (xfinfo[finfo[nr][h].fnr].max_char > 127) {
		    pagelist.page[n].fnr = i;
		    pagelist.page[n++].spos = 128;
		}
	    } else
		for (j=0; j<2; j++)
		    if (finfo[nr][h].page[j]!= -2) {
			pagelist.page[n].fnr = i;
			pagelist.page[n++].spos = finfo[nr][h].page[j];
		    }
    }
    pagelist.maxpage = n;
}

static void calc_size(void)
{
    int i,j,h;

    for (j=0; j<NR_SIZE; j++) {
	font_descent_ar[j] = 0;
	font_ascent_ar[j] = 0;
	font_width_ar[j] = 0;
	for (i=0; i<number_of_fonts[j]; i++) {
	    if (finfo[j][i].number!=BUTTONFONT) {
		h = xfinfo[finfo[j][i].fnr].f_height -
		    xfinfo[finfo[j][i].fnr].f_ascent;
		if (font_descent_ar[j] < h) font_descent_ar[j] = h;
		if (font_ascent_ar[j] < xfinfo[finfo[j][i].fnr].f_ascent)
		    font_ascent_ar[j] = xfinfo[finfo[j][i].fnr].f_ascent;
		if (font_width_ar[j] < xfinfo[finfo[j][i].fnr].f_width)
		    font_width_ar[j] = xfinfo[finfo[j][i].fnr].f_width;
	    }
	}
	font_height_ar[j] = font_ascent_ar[j] + font_descent_ar[j];
    }
    if (number_of_fonts[POPUPFONT]) {
	script[0] = POPUPFONT;
	i=1;
    } else i=0;
    for (; i<NR_SIZE; i++) {
	int k=1;
	h=0;
	for (j=0; j<NR_SIZE; j++)
	    if (font_height_ar[j]<font_height_ar[i] && font_height_ar[j]>h) {
		h= font_height_ar[j];
		k=j;
	    }
	if (h)
	    script[i] = k;
	else
	    script[i] = i;
    }
    font_width_max   = font_width_ar[grnr];
    font_ascent_max  = font_ascent_ar[grnr];
    font_descent_max = font_descent_ar[grnr];
    font_height_max  = font_height_ar[grnr];
}

static Bool read_fontsize(char *name, FONTSIZEINFO *fsi)
{
    int i=0,j=0;
    FILE *f;
    char *dirs[2];
    char *fname;
    char b[500];
    short lb[50];
    SIZEINFO *tmp=NULL;

    if (!name) return False;
    dirs[0] = userdir;
    dirs[1] = program_dir;
    concat_in(b, name, ".mpm");
    fname = search_through_dirs(dirs, 2, b);
    if (!fname || !(f=fopen(fname,"r"))) {
	message2(ERROR, "Can't open fontsize file ", name);
	myfree(name);
	return False;
    }
    fsi->filename = name;
    fsi->info = NULL;
    fsi->max = 0;
    fsi->lock=0;
    while (fgets(buffer, BUFSIZE, f)) {
	char *c=buffer;
	while (isspace(*c)) c++;
	i=0;
	while (isdigit(*c)) {
	    i = i*10+ *c-'0';
	    c++;
	}
	if (i) {
	    if (!tmp) tmp = (SIZEINFO*) malloc(sizeof(SIZEINFO));
	    tmp->label=i;
	    c++;
	    while (isspace(*c)) c++;
	    i=0;
	    while (*c && *c!=':') {
		j=0;
		if (*c=='*') {
		    lb[i]=tmp->label;
		    tmp->normal = i;
		    i++;
		    c++;
		} else {
		    while (isdigit(*c)) {
			j = j*10+*c-'0';
			c++;
		    }
		    if (j) lb[i++]=j;
		}
		while (*c && !isdigit(*c) && *c!=':' && *c!='*') c++;
	    }
	    if (*c) {
		c++;
		while (isspace(*c)) c++;
	    }
	    if (*c && i) {
		SIZEINFO **t;
		tmp->size = (short*) malloc(sizeof(short)*i);
		tmp->loaded = (char*) malloc(sizeof(char)*i);
		tmp->orsize = (short*) malloc(sizeof(short)*i);
		for (j=0; j<i; j++) {
		    tmp->orsize[j] = lb[j];
		    tmp->loaded[j] = False;
		    tmp->size[j] = 0;
		}
		tmp->max = i;
		remove_return(c);
		tmp->name = concat(c,"");
		tmp->fnr = 0;
		t = (SIZEINFO**) malloc(sizeof(SIZEINFO*)*(fsi->max+1));
		for (j=0; j<fsi->max; j++) t[j] = fsi->info[j];
		t[j] = tmp;
		myfree(fsi->info);
		fsi->info = t;
		tmp = NULL;
		fsi->max++;
	    }
	}
    }
    fclose(f);
    myfree(tmp);
    if (!fsi->max) {
	message2(ERROR, "Strange fontsize file: ", name);
	myfree(name);
	return False;
    } else {
	fsi->loaded = (char*) malloc(sizeof(char) * fsi->max);
	for (i=0; i<fsi->max; fsi->loaded[i++] = False);
	return True;
    }
}

static void read_general_fontinfo(void)
{
    int i=0;
    FILE *f;
    char *dirs[6];
    char *ind;
    char *fname;
    char b[1000];

    dirs[0]=b;
    concat_in(b+i, userdir, "/%.mpt");
    i+=strlen(b+i)+1;
    dirs[1]=userdir;
    dirs[2]=b+i;
    concat_in(b+i, homedir, "mathspad/%.mpt");
    i+=strlen(b+i)+1;
    dirs[3]=b+i;
    concat_in(b+i, homedir, "mathspad/");
    i+=strlen(b+i)+1;
    dirs[4]=b+i;
    concat_in(b+i, program_dir,"%.mpt");
    dirs[5]=program_dir;
    fname = search_through_dirs(dirs, 6, generalinfofile);
    if (!fname)
	message2(EXIT-1, "Unable to find font information file ",
		 generalinfofile);
    f = fopen(fname, "r");
    if (!f)
	message2(EXIT-1, "Unable to open font information file ",
		 generalinfofile);
    for (i=0; i<256; i++) {
	gfinfo[i].fontname = gfinfo[i].texfilename = gfinfo[i].opentextfont
	    = gfinfo[i].openmathfont = gfinfo[i].closefont = NULL;
	gfinfo[i].tnr = 0;
    }
    buffer[0]='\0'; fgets(buffer,BUFSIZE,f);
    while (buffer[0]) {
	if ((ind = begins_with("FONTNR:", buffer))) {
	    int fnr=0;
	    remove_return(ind);
	    if (*ind) fnr = read_integer(&ind);
	    fnr = fnr&0xFF;
	    buffer[0]='\0'; fgets(buffer, BUFSIZE, f);
	    if ((ind = begins_with("FONTNAME:", buffer))) {
		remove_return(ind);
		myfree(gfinfo[fnr].fontname);
		gfinfo[fnr].fontname = concat(ind, "");
		buffer[0]='\0'; fgets(buffer,BUFSIZE,f);
	    }
	    if ((ind = begins_with("LATEX:", buffer))) {
		remove_return(ind);
		myfree(gfinfo[fnr].texfilename);
		gfinfo[fnr].texfilename = concat(ind, "");
		buffer[0]='\0'; fgets(buffer,BUFSIZE,f);
	    }
	    if ((ind = begins_with("OPENFONT:", buffer))) {
		remove_return(ind);
		myfree(gfinfo[fnr].opentextfont);
		gfinfo[fnr].opentextfont = concat(ind, "");
		buffer[0]='\0'; fgets(buffer,BUFSIZE,f);
	    }
	    if ((ind = begins_with("OPENMATHFONT:", buffer))) {
		remove_return(ind);
		myfree(gfinfo[fnr].openmathfont);
		gfinfo[fnr].openmathfont = concat(ind, "");
		buffer[0]='\0'; fgets(buffer,BUFSIZE,f);
	    }		
	    if ((ind = begins_with("CLOSEFONT:", buffer))) {
		remove_return(ind);
		myfree(gfinfo[fnr].closefont);
		gfinfo[fnr].closefont = concat(ind, "");
		buffer[0]='\0'; fgets(buffer,BUFSIZE,f);
	    }
	} else {
	    buffer[0]='\0';
	    fgets(buffer, BUFSIZE, f);
	}	
    }
    fclose(f);
}

static int read_fontpart(FILE *f, FONTINFO finf[MAXFONTS])
{

#define get_new_string (not_finished = ((fgets(buffer, BUFSIZE, f)!=NULL) && \
					!begins_with("STOP", buffer)))

    int nr, i;
    Bool not_finished;
    char *ind;
    unsigned char last_fontnr = 0;

    nr=0;
    get_new_string;
    while (not_finished && nr<MAXFONTS) {
	if ((ind = begins_with("FONT:", buffer)) ||
	    (ind = begins_with("FONTGROUP:", buffer))) {
	    remove_return(ind);
	    finf[nr].name = (char *) malloc(strlen(ind)+1 );
	    strcpy(finf[nr].name,ind);
	    if (buffer[4]!=':') finf[nr].nrml = 1;
	    else finf[nr].nrml = 0;
	    finf[nr].page[0]= -1;       /* alle karakters in symbol window */
	    finf[nr].page[1]= -2;
	    finf[nr].lnr=0;
	    finf[nr].fnr=0;
	    finf[nr].max = 0;
	    finf[nr].size = NULL;
	    finf[nr].number = last_fontnr;
	    if ((int)(last_fontnr+1)< BUTTONFONT) last_fontnr++;
	    if (get_new_string && (ind = begins_with("NUMBER:", buffer))) {
		remove_return(ind);
		if (*ind) {
		    i = read_integer(&ind);
		    if (0<=i && i<BUTTONFONT)
			finf[nr].number = i;
		}
		get_new_string;
	    }
	    if (not_finished && (ind = begins_with("PAGE:", buffer))) {
		remove_return(ind);
		if (begins_with("NONE", ind))
		    finf[nr].page[0]= -2;  /* geen symbolen in symboolwindow */
		else {
		    for (i=0; i<2 && ind; i++) {
			while (*ind && !isdigit(*ind)) ind++;
			if (*ind)
			    finf[nr].page[i]=read_integer(&ind);
		    }
		}
		get_new_string;
	    }
	    if (gfinfo[finf[nr].number].texfilename) {
		nr++;
	    } else {
		free(finf[nr].name);
	    }
	} else
	    if ((ind = begins_with("BUTTONFONT:", buffer))) {
		remove_return(ind);
		finf[nr].name= (char *) malloc(strlen(ind)+1);
		strcpy(finf[nr].name,ind);
		finf[nr].page[0] = finf[nr].page[1] = -2;
		finf[nr].number = BUTTONFONT;
		nr++;
		get_new_string;
	    } else 
		get_new_string;
    }
    return nr;
}

static short load_latexfile(char *name)
{
#define BIGBUFSIZE 20480
    
    char *tbuf=NULL, *pos;
    char *dirs[3];
    char *fname;
    int count=0, i;
    FILE *tf;

    if (!name) return 0;

    for (i=0;
	 i<nr_latex && (!linfo[i].name || strcmp(name, linfo[i].name));
	 i++);
    if (i<nr_latex) {
	linfo[i].locks++;
	myfree(name);
	return i;
    }
    dirs[0] = userdir;
    dirs[1] = program_dir;
    dirs[2] = "";
    fname = search_through_dirs(dirs, 3, name);
    if (!fname || !(tf=fopen(fname, "r"))) {
	message(ERROR,  "Can't open file with LaTeX macros.");
	myfree(name);
	return 0;
    }
    if (!(tbuf = (char *) malloc(BIGBUFSIZE))) {
	message(ERROR, "Out of memory loading LaTeX macros.");
	myfree(name);
	return 0;
    }
    count=1;
    pos=tbuf+1;
    tbuf[0] = '\0';
    for (i=0; i<256; i++)
	linfo[nr_latex].normal[i] = linfo[nr_latex].math[i] = 0;
    while (fgets(tbuf+count, BIGBUFSIZE-count, tf)!=NULL) {
	pos = tbuf+count;
	while (isspace(*pos)) pos++;
	if (isdigit(*pos)) {
	    i = read_integer_short(&pos);
	    tbuf[count] = i%256;
	    if (*pos=='$' || *pos=='&' || isspace(*pos)) {
		count++;
		tbuf[count]= (isspace(*pos) ? ' ' : *pos);
		count++;
		switch (*pos) {
		case '&': linfo[nr_latex].normal[i%256]=count;
		case '$': linfo[nr_latex].math[i%256]=count;
		    break;
		default:  linfo[nr_latex].normal[i%256]=count;
		}
		pos++;
		while (isspace(*pos)) pos++;
		if (!*pos)
		    tbuf[count++] = ' ';
		else
		    while (*pos && *pos!='\n') tbuf[count++] = *pos++;
		tbuf[count++] = '\0';
	    }
	}
    }
    fclose(tf);
    if (count>1) {
	if ((linfo[nr_latex].code = (char *) malloc( count))) {
	    for (i=0; i<count; i++)
		linfo[nr_latex].code[i] = tbuf[i];
	    myfree(tbuf);
	} else
	    linfo[nr_latex].code = tbuf;
    } else
	linfo[nr_latex].code = concat("","");
    linfo[nr_latex].name = name;
    linfo[nr_latex].locks++;
    nr_latex++;
    return nr_latex-1;
}

static int load_xfont(char *name)
{
    CHARINFO *tinf;
    XFontStruct *tempinfo;
    XCharStruct *perchar;
    unsigned i,j;

    for (i=0;
	 i<nr_xfont && (!xfinfo[i].name || strcmp(name, xfinfo[i].name));
	 i++);
    if (i<nr_xfont) {
	xfinfo[i].locks++;
	myfree(name);
	return i;
    }
    if (!(tempinfo = XLoadQueryFont(display, name))) {
	message2(ERROR, "Cannot load font ", name);
	myfree(name);
	return 0;
    }
    for (i=0; i<nr_xfont && xfinfo[i].f_id!=tempinfo->fid; i++);
    if (i<nr_xfont) {
	xfinfo[i].locks++;
	myfree(name);
	XFree((void*)tempinfo);
	return i;
    }
    xfinfo[nr_xfont].min_char = tempinfo->min_char_or_byte2;    
    xfinfo[nr_xfont].max_char = tempinfo->max_char_or_byte2;
    xfinfo[nr_xfont].f_width  = tempinfo->max_bounds.width;
    xfinfo[nr_xfont].f_id     = tempinfo->fid;
    xfinfo[nr_xfont].f_ascent = tempinfo->ascent;
    xfinfo[nr_xfont].f_height = tempinfo->ascent + tempinfo->descent;
    xfinfo[nr_xfont].name     = name;
    j = xfinfo[nr_xfont].max_char - xfinfo[nr_xfont].min_char + 1;
    if ((tinf = xfinfo[nr_xfont].char_info =
	 (CHARINFO *) malloc( 256*sizeof(CHARINFO)))==NULL) {
	message(ERROR, "Out of memory at font loading.");
	myfree(name);
	XFree((void*) tempinfo);
	return 0;
    }
    for (i=0; i<xfinfo[nr_xfont].min_char; i++,tinf++) tinf->width = 0;
    if ((perchar = tempinfo->per_char))
	for (i=0; i<j; i++,perchar++,tinf++) {
	    tinf->width   = perchar->width;
	    tinf->ascent  = perchar->ascent;
	    tinf->descent = perchar->descent;
	    if (perchar->lbearing<0)
		tinf->left = -perchar->lbearing;
	    else
		tinf->left = 0;
	    if (tinf->width<perchar->rbearing)
		tinf->right = perchar->rbearing - tinf->width;
	    else
		tinf->right = 0;
	}
    else
	for (i=0; i<j; i++,tinf++) {
	    tinf->width   = xfinfo[nr_xfont].f_width;
	    tinf->ascent  = xfinfo[nr_xfont].f_ascent;
	    tinf->descent = xfinfo[nr_xfont].f_height-
		xfinfo[nr_xfont].f_ascent;
	    tinf->left = tinf->right = 0;
	}
    for (i=xfinfo[nr_xfont].max_char+1; i<256; i++, tinf++) tinf->width = 0;
    XFree((void*) tempinfo);
    xfinfo[nr_xfont].locks++;
    nr_xfont++;
    return nr_xfont-1;
}

static void free_xfont(int nr)
{
    if (xfinfo[nr].locks) {
	xfinfo[nr].locks--;
	if (!xfinfo[nr].locks) {
	    myfree(xfinfo[nr].name);
	    xfinfo[nr].name = NULL;
	    myfree(xfinfo[nr].char_info);
	    xfinfo[nr].char_info = NULL;
	    XUnloadFont(display,xfinfo[nr].f_id);
	    xfinfo[nr].f_id = 0;
	}
    }
}

static void free_xfontsize(int nr)
{
    if (fsinfo[nr].lock) {
	fsinfo[nr].lock--;
	if (!fsinfo[nr].lock) {
	    int i;
	    myfree(fsinfo[nr].filename);
	    fsinfo[nr].filename=NULL;
	    for (i=0; i<fsinfo[nr].max; i++) {
		if (fsinfo[nr].loaded[i])
		    free_xfont(fsinfo[nr].info[i]->fnr);
		myfree(fsinfo[nr].info[i]->size);
		myfree(fsinfo[nr].info[i]->orsize);
		myfree(fsinfo[nr].info[i]->loaded);
	    }
	    myfree(fsinfo[nr].loaded);
	}
    }
}

static Bool add_xfontsize(int fsnr, short *size, int nr)
{
    FONTSIZEINFO *fs = fsinfo+fsnr;
    int i, tl,p,j,k;

    i=0;
    for (i=0; fs->info[i]->size!=size && i<fs->max; i++);
    if (i==fs->max || fs->info[i]->loaded[nr]) return True;
    tl = fs->info[i]->orsize[nr];
    for (i=0; fs->info[i]->label!=tl && i<fs->max; i++);
    if (i==fs->max) return False;
    p = fs->info[i]->fnr = load_xfont(fs->info[i]->name);
    fs->loaded[i] = True;
    if (!p) {
	/* Unable to load font. Search alternative */
	k=1;j=fnr2anr[grnr][fsnr];
	while  (!p && k<finfo[grnr][j].max) {
	    if (i-k>=0 && finfo[grnr][j].loaded[i-k])
		p = finfo[grnr][j].size[i-k];
	    else if (i+k<finfo[grnr][j].max && finfo[grnr][j].loaded[i+k])
		p = finfo[grnr][j].size[i+k];
	    k++;
	}
	if (!p) p = finfo[grnr][j].fnr;
	fs->info[i]->fnr=p;
    }
    for (i=0; i<fs->max; i++) {
	for (j=0; j<fs->info[i]->max; j++) {
	    if (fs->info[i]->orsize[j]==tl) {
		fs->info[i]->size[j] = p;
		fs->info[i]->loaded[j] = True;
	    }
	}
    }
    return True;
}

static short load_xfontsize(char *name, short **sizes, char **loaded,
			    short *max, short *nrml, short *fsnr)
{
    int i,j=0,k;
    char *c = name;

    while (isdigit(*c)) {
	j = j*10+*c-'0';
	c++;
    }
    c++;
    while (isspace(*c)) c++;
    i=0;
    while ((name[i++]=*c++));
    for (k=0; k<nr_fsinfo && fsinfo[k].lock; k++);
    for (i=0;
	 i<nr_fsinfo && (!fsinfo[i].lock || strcmp(fsinfo[i].filename,name));
	 i++);
    if (i==nr_fsinfo) {
	i=k;
	if (read_fontsize(name, fsinfo+i)) {
	    if (i==nr_fsinfo) nr_fsinfo++;
	} else
	    i=nr_fsinfo;
    }
    if (i<nr_fsinfo) {
	fsinfo[i].lock++;
	*fsnr = i;
	for (k=0; k<fsinfo[i].max && fsinfo[i].info[k]->label<j; k++);
	if (k==fsinfo[i].max) k--;
	add_xfontsize(i, fsinfo[i].info[k]->size, fsinfo[i].info[k]->normal);
	*sizes = fsinfo[i].info[k]->size;
	*max = fsinfo[i].info[k]->max;
	*nrml = fsinfo[i].info[k]->normal;
	*loaded = fsinfo[i].info[k]->loaded;
	return (*sizes)[*nrml];
    }
    return 0;
}

static Bool load_font(FONTINFO *finf)
{
    finf->max = 0;
    if (finf->nrml)
	finf->fnr = load_xfontsize(finf->name, &finf->size, &finf->loaded,
				   &finf->max, &finf->nrml, &finf->fsnr);
    else
	finf->fnr = load_xfont(finf->name);
    if (finf->fnr) {
	finf->lnr = gfinfo[finf->number].tnr;
	if (!finf->lnr) {
	    if (finf->max)
		free_xfontsize(finf->fsnr);
	    else
		free_xfont(finf->fnr);
	}
	return (finf->lnr); 
    }
    return 0;
}

static void remove_font(int nr, int pos, FONTINFO *finf)
{
    int i;

    if (fnr2anr[nr][BUTTONFONT] == pos) {
	finfo[nr][pos].number = BUTTONFONT;
	for (i=0; i<BUTTONFONT; i++) {
	    if (fnr2anr[nr][i]==pos) fnr2anr[nr][i] = MAXFONTS+1;
	}
    } else {
	if (finfo[nr][pos].max)
	    free_xfontsize(finfo[nr][pos].fsnr);
	else
	    free_xfont(finfo[nr][pos].fnr);
	i=pos;
	while (i+1<number_of_fonts[nr]) {
	    finfo[nr][i] = finfo[nr][i+1];
	    i++;
	}
	number_of_fonts[nr]--;
	for (i=0; i<256; i++)
	    if ((int)fnr2anr[nr][i]>pos)
		fnr2anr[nr][i]--;
	    else if (fnr2anr[nr][i]==pos)
		fnr2anr[nr][i] = MAXFONTS+1;
    }
}

static Bool can_change_font(int nr, int number)
{
    return ((number != BUTTONFONT) &&
	    ((number_of_fonts[nr]<MAXFONTS) ||
	     (fnr2anr[nr][number]!=fnr2anr[nr][TEXTFONT]) ||
	     (number==TEXTFONT &&
	      fnr2anr[nr][BUTTONFONT]!=fnr2anr[nr][TEXTFONT])));
}

static Bool must_remove_font(int nr, int number)
{
    return ((fnr2anr[nr][number]!=fnr2anr[nr][TEXTFONT])||(number==TEXTFONT));
}

static void add_font(int nr, FONTINFO *finf)
{
    finfo[nr][number_of_fonts[nr]] = *finf;
    if (finf->number == TEXTFONT) {
	int i,j=fnr2anr[nr][TEXTFONT];

	for (i=0; i<256; i++)
	    if (fnr2anr[nr][i]==j)
		fnr2anr[nr][i]=number_of_fonts[nr];
    } else
	fnr2anr[nr][finf->number] = number_of_fonts[nr];
    number_of_fonts[nr]++;
}

int missing_font(FILE *f)
{
    FONTINFO tfi[MAXFONTS], comp;
    int i, n;
    Bool load_tfi[MAXFONTS], lchange=False;
    int wrong1, wrong2, good;
    int oldwidth, oldheight;

#define free_names(A)  for (i=0; i<n; i++) if ((A) || !load_tfi[i]) { \
		       myfree(tfi[i].name); }

    wrong1 = wrong2 = good = 0;
    n = read_fontpart(f, tfi);
    for (i=0; i<n; i++) {
	load_tfi[i]=False;
	comp = finfo[0][fnr2anr[0][tfi[i].number]];
	if (tfi[i].number!=BUTTONFONT) {
	    if (tfi[i].number==TEXTFONT) good++;
	    if (comp.number!= tfi[i].number) {
		wrong1++;
		load_tfi[i]=True;
	    }
	}
    }
    if (!good) {
	free_names(True);
	return EXIT;
    }
    if (wrong1) {
	i=0;
	while (wrong1 && i<n) {
	    if (load_tfi[i])
		if (can_change_font(0,tfi[i].number)) {
		    if (load_font(tfi+i)) {
			add_font(0,tfi+i);
			wrong1--;
			lchange = True;
		    }
		} else
		    load_tfi[i]=False;
	    i++;
	}
	if (lchange) {
	    oldwidth = font_width_ar[0];
	    oldheight = font_height_ar[0];
	    if (!number_of_fonts[SYMBOLFONT] && alternative[SYMBOLFONT]==0)
		make_pagelist(0);
	    calc_size();
	    if ((oldwidth!=font_width_ar[0])||(oldheight!=font_height_ar[0]))
		call_layout_change();
	}
	if (wrong1) {
	    message(CLICKREMARK, "Missing fonts for this file.");
	    free_names(False);
	    return EXIT-1;
	}
    }
    free_names(False);
    return 0;
}

static void make_fonts(void)
{
    int i,j;
    int num;
    FILE *f = NULL;
    char *dirs[2];
    char fname[500];
    char *name;

    dirs[0] = userdir;
    dirs[1] = program_dir;

    for (j=0; j<NR_SIZE; j++) {
	number_of_fonts[j]=0;
	for (i=0; i<256; fnr2anr[j][i++]=MAXFONTS+1);
	alternative[j]=j;
	if ((!fontfile[j] || !fontfile[j][0]) && j) {
	    alternative[j]=0;
	} else {
	    concat_in(fname, fontfile[j], ".mpf");
	    name = search_through_dirs(dirs, 2, fname);
	    if (!name) {
		fname[strlen(fname)-4]='\0';
		name = search_through_dirs(dirs, 2, fname);
	    }
	    if (!name && !j) {
		concat_in(fname, program_fontfilename, ".mpf");
		name = search_through_dirs(dirs, 2, fname);
		if (!name) {
		    fname[strlen(fname)-4] = '\0';
		    name = search_through_dirs(dirs, 2, fname);
		}
	    }
	    if (name)
		f = fopen(name, "r");
	    else
		f = NULL;
	    if (!f) {
		if (!j) {
		    message(EXIT-1, "Fontfile not found.");
		    /* ending program */
		} else
		    alternative[j]=0;
	    } else {
		num = read_fontpart(f, finfo[j]);
		fclose(f);
		for (i=0; i<256; i++)
		    fnr2anr[j][i]=255;
		for (i=0; i<num; i++) {
		    if (load_font(&finfo[j][i])) {
			add_font(j, &finfo[j][i]);
		    }
		}
		if (!fnr2anr[j][TEXTFONT]==255)
		    message(EXIT -1, "Fontfile does not contain a textfont.");
	    }
	}
    }
    calc_size();
    make_pagelist(alternative[SYMBOLFONT]);
}

Bool new_fontfile(int nr, char *newname)
{
    if (((!newname||!*newname) && (!fontfile[nr]||!fontfile[nr][0])) ||
	(newname && fontfile[nr] && !strcmp(newname, fontfile[nr])))
	return False;
    else {
	int i;
	Bool fileok;
	int num;
	FILE *f;
	FONTINFO newfonts[MAXFONTS];
	char *dirs[2];
	char fname[500];
	char *name;

	dirs[0] = userdir;
	dirs[1] = program_dir;
	concat_in(fname, newname, ".mpf");
	name = search_through_dirs(dirs, 2, fname);
	if (!name) {
	    fname[strlen(fname)-4] = '\0';
	    name = search_through_dirs(dirs, 2, fname);
	}
	if (!name) {
	    message(CLICKREMARK, "Unable to access new fontfile.");
	    return False;
	}
	f = fopen(name, "r");
	num = read_fontpart(f, newfonts);
	fclose(f);
	fileok = False;
	for (i=0; !fileok && i<num; i++)
	    fileok = (newfonts[i].number == TEXTFONT);
	if (!fileok) {
	    message(CLICKREMARK, "Fontfile does not contain a textfont.");
	    for (i=0; i<num; i++) {
		myfree(newfonts[i].name);
	    }
	    return False;
	}
	for (i=0; i<num; i++) {
	    if (can_change_font(nr, newfonts[i].number)) {
		if (load_font(newfonts+i)) {
		    if (must_remove_font(nr,newfonts[i].number))
			remove_font(nr, fnr2anr[nr][newfonts[i].number],
				    newfonts+i);
		    add_font(nr, newfonts+i);
		}
	    } else {
		myfree(newfonts[i].name);
	    }
	}
	alternative[nr] = nr;
	calc_size();
	make_pagelist(alternative[SYMBOLFONT]);
	myfree(fontfile[nr]);
	fontfile[nr] = newname;
	return True;
    }
}

static void destroy_fonts(int nr)
{
    int i;
    for (i=0;i<number_of_fonts[nr];i++) {
	if (finfo[nr][i].max)
	    free_xfontsize(finfo[nr][i].fsnr);
	else
	    free_xfont(finfo[nr][i].fnr);
    }
}

void push_fontgroup(int nr)
{
    int i;
    i = (nr<0||nr>NR_SIZE ? 0 : alternative[nr]);
    push_int(&font_stack, i);
    if (grnr!=i) {
	grnr = i;
	font_height_max  = font_height_ar[grnr];
	font_ascent_max  = font_ascent_ar[grnr];
	font_descent_max = font_descent_ar[grnr];
	font_width_max   = font_width_ar[grnr];
	setsimpletabsize(8*char_width(Font2Char(TEXTFONT,'n'),0));
    }
}

void pop_fontgroup(void)
{
    int i = pop_int(&font_stack);
    i = head_int(font_stack);
    if (i!=grnr) {
	grnr = i;
	font_height_max  = font_height_ar[grnr];
	font_ascent_max  = font_ascent_ar[grnr];
	font_descent_max = font_descent_ar[grnr];
	font_width_max   = font_width_ar[grnr];
	setsimpletabsize(8*char_width(Font2Char(TEXTFONT,'n'),0));
    }
}

#define FNR(A,B) fnr2anr[A][B]
#define FINFO(A,B) finfo[A][FNR(A,B)]

static int get_size(int granr, int fontnr, int size)
{
    int j = fnr2anr[granr][fontnr];
    int i = size+finfo[granr][j].nrml;

    if (i<0) i=0; else if (i>=finfo[granr][j].max) i=finfo[granr][j].max-1;
    if (!finfo[granr][j].loaded[i]) {
	message(MESSAGE, "Loading extra font ...");
	add_xfontsize(finfo[granr][j].fsnr, finfo[granr][j].size, i);
	message(MESSAGE, "Loading extra font ... done.");
    }
    return i;
}

#define XINFO(A,B,C) xfinfo[(FINFO(A,B).max?FINFO(A,B).size[get_size(A,B,C)]:FINFO(A,B).fnr)]

Font font_ID(int fontnr, int size)
{
    return XINFO(grnr,fontnr,size).f_id;
}

int font_height(int fontnr, int size)
{
    return XINFO(grnr,fontnr,size).f_height;
}

char *font_opentexttex(int fontnr, int size)
{
    return gfinfo[fontnr].opentextfont;
}

char *font_openmathtex(int fontnr, int size)
{
    return gfinfo[fontnr].openmathfont;
}

char *font_name(int fontnr)
{
    return gfinfo[fontnr].fontname;
}

char *font_closetex(int fontnr, int size)
{
    return gfinfo[fontnr].closefont;
}

int font_ascent(int fontnr, int size)
{
    return XINFO(grnr,fontnr,size).f_ascent;
}

int font_descent(int fontnr, int size)
{
    return XINFO(grnr,fontnr,size).f_height -
	XINFO(grnr,fontnr,size).f_ascent;
}

short font_width(int fontnr, int size)
{
    return XINFO(grnr,fontnr,size).f_width;
}

int char_width(Char data, int size)
{
    int h = FINFO(grnr,Char2Font(data)).lnr;
    int j = Char2ASCII(data);

    if (!linfo[h].math[j] &&
	!linfo[h].normal[j]) return 0;
    else
	return XINFO(grnr,Char2Font(data),size).char_info[j].width;
}

int char_ascent(Char data, int size)
{
    int j = Char2ASCII(data);
    return XINFO(grnr,Char2Font(data), size).char_info[j].ascent;
}

int char_descent(Char data, int size)
{
    int j = Char2ASCII(data);
    return XINFO(grnr,Char2Font(data), size).char_info[j].descent;
}

int char_left(Char data, int size)
{
    return XINFO(grnr,Char2Font(data), size).char_info[Char2ASCII(data)].left;
}

int char_right(Char data, int size)
{
    return XINFO(grnr,Char2Font(data), size).char_info[Char2ASCII(data)].right;
}

char *char_latex(Char data, Bool math)
{
    int h = FINFO(grnr,Char2Font(data)).lnr;
    int j = Char2ASCII(data);
    int idx = (math? linfo[h].math[j] : linfo[h].normal[j]);

    if (idx && XINFO(grnr,Char2Font(data), 0).char_info[j].width)
	return linfo[h].code+idx;
    else
	return NULL;
}

static int linfoused[30];
/* for parsing: loop through all the available symbols to add them
** to the parser. linfoused indicates which (latex)maptables are already used
** and fonts using that table are skipped. Undefined fonts are also
** skipped. (otherwise it would result in 256*256*2=131072 loops, now the
** maximum is 256*30*2=15360 loops, but usually much less (standard 2048).
*/
char *char_latex_next(Char *data, int *math)
{
    int i,j,h,m,id;
    char *lh;
    if (!*data)	for (i=0; i<30; linfoused[i++]=0);
    if (!*math) (*math)++; else { *math=0; (*data)++; }
    i=Char2Font(*data);
    j=Char2ASCII(*data);
    h = FINFO(grnr, i).lnr;
    lh=linfo[h].code;
    m=*math;
    while (i<256) {
	if (!h || linfoused[h]) {
	    /* font is not available or uses same coding: skip it */
	    i++; j=0;m=0;
	    h = FINFO(grnr, i).lnr;
	    lh=linfo[h].code;
	} else if (((m  && (id=linfo[h].math[j])) ||
		    (!m && (id=linfo[h].normal[j]))) &&
		   ((lh[id]!=j) || (lh[id+1]!=0))) {
	    /* found something of interest */
	    *data=Font2Char(i,j);
	    *math=m;
	    return lh+id;
	} else {
            /* increase (i,j,m) */
	    if (!m) m=1;
	    else {
		m=0;
		if (j<255) j++;
		else {
		    j=0;
		    linfoused[h]=1;
		    i++;
		    h = FINFO(grnr,i).lnr;
		    lh=linfo[h].code;
		}
	    }
	}
    }
    *data=0;
    *math=0;
    return 0;
}

int string_width(int fontnr, char *string, int nr)
{
    int i;
    fontnr = finfo[grnr][fnr2anr[grnr][fontnr]].fnr;
    for (i=0; (nr && *string && *string!='\n') ;nr--,string++)
	i+= xfinfo[fontnr].char_info[*string].width;
    return i;
}

/*
**
**  Functies die werken met GC's, fonts en colors:
**
*/

#define MAX_COLORS 2
#define MAX_PLANES 3
#define MAX_CELLS 16

unsigned long colorlist[MAX_CELLS/2][2];
static unsigned long swap_plane[MAX_PLANES];
static int use_underline[MAX_CELLS/2];
static int maxcolors=MAX_CELLS;

/* this should be available through a file */
static char *colorname[MAX_CELLS] = {
/* foreground   background    selection */
    "Black",     "White",     /* normal */
    "Yellow",    "Blue",      /* source */
    "Green",     "Magenta",   /* argument */
    "Red",       "Cyan",      /* source+argument */
    "White",     "Black",     /* target */
    "Blue",      "Yellow",    /* target+source */
    "Magenta",   "Green",     /* target+argument */
    "Cyan",      "Red" };     /* target+source+argument */

static char *grayname[MAX_CELLS] = {
/* foreground   background    selection */
    "Black",     "White",     /* normal */
    "Gray25",    "Gray75",    /* source */
    "Gray12",    "Gray88",    /* argument */
    "Gray37",    "Gray63",    /* source+argument */
    "White",     "Black",     /* target */
    "Gray75",    "Gray25",    /* target+source */
    "Gray88",    "Gray12",    /* target+argument */
    "Gray63",    "Gray37" };  /* target+source+argument */



static void make_colors(void)
{
    Colormap def_cmap;
    int ncolors = maxcolors;
    int np=0,i,j;
    XColor exact_defs[MAX_CELLS];
    unsigned long plane_masks[MAX_PLANES];
    unsigned long colors[MAX_COLORS];
    char **cname;

    for (i=0; i<MAX_PLANES; plane_masks[i++]=0);
    for (i=0; i<MAX_COLORS; colors[i++]=0);
    def_cmap = colormap;
    cname=colorname;
    if (def_depth == 1) {
	ncolors=2;
	np=0;
    } else {
	np=MAX_PLANES;
	while (np>0 &&
	       !XAllocColorCells(display,def_cmap,False,plane_masks,np,
				 colors,2)) {
	    np--;
	    ncolors=ncolors/2;
	}
	if (visualclass==GrayScale || usegray) cname =grayname;
    }
    for (i = 0; i < ncolors; i++) {
	if (!XParseColor (display, def_cmap, cname[i], &exact_defs[i])) {
	    fprintf(stderr,"%s: color name %s not in database",
		    progname, colorname[i]);
	    exact_defs[i].red=exact_defs[i].green=exact_defs[i].blue=0;
	}
	exact_defs[i].flags = DoRed | DoGreen | DoBlue;
    }
    if (ncolors==2) {
	j=0;
	if (XAllocColor(display, def_cmap, &exact_defs[0])) {
	    j++;
	    colorlist[0][0]=exact_defs[0].pixel;
	}
	if (XAllocColor(display, def_cmap, &exact_defs[1])) {
	    j=j+2;
	    colorlist[0][1]=exact_defs[1].pixel;
	}
	if (j!=3 || colorlist[0][1]==colorlist[0][0]) {
	    if (j)
		XFreeColors(display, def_cmap,
			    colorlist[0]+(j==2), 1+(j==3), 0);
	    colorlist[0][0]=black_pixel;
	    colorlist[0][1]=white_pixel;
	}
	colorlist[4][0]=colorlist[0][1];
	colorlist[4][1]=colorlist[0][0];
	for (i=1; i<4; i++)
	    for (j=0; j<2; j++) {
		colorlist[i][j]=colorlist[0][j];
		colorlist[i+4][j]=colorlist[4][j];
	    }
    } else {
	int k;
	exact_defs[0].pixel = colors[0];
	exact_defs[1].pixel = colors[1];
	colorlist[0][0]=colors[0];
	colorlist[0][1]=colors[1];
	j=2;
	for (i=0; i<np; i++) {
	    for (k=0; k<j; k++) {
		colorlist[(j+k)/2][(j+k)%2]=
		exact_defs[j+k].pixel = exact_defs[k].pixel | plane_masks[i];
	    }
	    j=j*2;
	}
	XStoreColors (display, def_cmap, exact_defs, ncolors);
	if (ncolors<MAX_CELLS) {
	    if (ncolors==4) {
		for (i=2; i<4; i++)
		    for (k=0; k<2; k++)
			colorlist[i][k]=colorlist[i-2][k];
	    }
	    for (i=4; i<8; i++)
		for (k=0; k<2; k++)
		    colorlist[i][k]=colorlist[i-4][1-k];
	}
    }
    plane_masks[0]=3;
    plane_masks[1]=1;
    plane_masks[2]=0;
    for (i=0; i<MAX_CELLS/2; use_underline[i++]=0);
    for (i=0,j=1; i<MAX_PLANES; i++,j*=2) {
	swap_plane[i]=colorlist[0][0]^colorlist[j][0];
	if (!swap_plane[i]) {
	    int k;
	    for (k=0; k<MAX_CELLS/2; k++)
		if (k&j) use_underline[k] ^= plane_masks[i];
	}
    }
    /* set back and foreground colors */
    black_pixel=colorlist[0][0];
    white_pixel=colorlist[0][1];
}


#define MAXGCS 3

static GC gca[MAXGCS];
static int gcfont[MAXGCS];
static int gccolor[MAXGCS];

static void make_GCs(void)
{
    XGCValues values;
    unsigned long mask;
    int i;

    values.foreground = colorlist[0][0];
    values.background = colorlist[0][1];
    values.fill_style = FillSolid;
    values.fill_rule  = WindingRule;
    values.font       = font_ID(TEXTFONT,0);
    values.function   = GXcopy;
    gccolor[Normal]=0;

    mask = (GCForeground | GCBackground | GCFillStyle | GCFunction |
	    GCFillRule | GCFont);

    gca[Normal]   = XCreateGC(display, root_window, mask, &values);
    values.foreground = colorlist[0][1];
    values.background = colorlist[0][0];
    gccolor[Reverse]=0;
    gca[Reverse] = XCreateGC(display, root_window, mask, &values);

    values.background = 0;
    values.foreground = colorlist[0][0]^colorlist[4][0];
    gccolor[Xor]=4;
    values.function = GXxor;
    gca[Xor] = XCreateGC(display, root_window, mask, &values);
    for (i=0; i<MAXGCS; i++) gcfont[i]=TEXTFONT;
}

static void destroy_GCs(void)
{
    int i;

    for (i=0; i<MAXGCS; i++) XFreeGC(display,gca[i]);
}

GC get_GC(TextMode gcnr, int colortype)
{  /* add a check if XSet is needed. */
    if (gccolor[gcnr]!=colortype) {
	XSetForeground(display, gca[gcnr], colorlist[colortype][gcnr]);
	XSetBackground(display, gca[gcnr], colorlist[colortype][1-gcnr]);
	gccolor[gcnr]=colortype;
    }
    return gca[gcnr];
}

GC get_GCXor(int colortype)
{   /* add a check if XSet is needed. */
    if (gccolor[Xor]!=colortype) {
	XSetForeground(display, gca[Xor],
		       colorlist[colortype][0]^colorlist[0][0]);
	gccolor[Xor]=colortype;
    }
    return gca[Xor];
}

int must_underline(int colortype)
{
    return use_underline[colortype];
}

GC get_GC_font(TextMode gcnr, int colortype, int fontnr, int size)
{
    (void) get_GC(gcnr, colortype);
    if (gcfont[gcnr]!=fontnr) {
	XSetFont(display,gca[gcnr],font_ID(fontnr,size));
	gcfont[gcnr] = fontnr;
    }
    return gca[gcnr];
}

void undefined_font(TextMode gcnr)
{
    gcfont[gcnr]= -1;
}

/*
**
**  Functies die gegevens van windows verwerken:
**
*/

typedef struct {
    int state,mapped;
    Window win_id, pwin_id;
    WINDOWTYPE type;
    void *data;
    char *helpfile;
} EXTENDEDINFO;

#define EXPAND_SIZE  10
static EXTENDEDINFO *winfo = NULL;
static int number_of_windows = 0;
static int max_usable = 0;

static int get_pos(Window win)
{
    int i,j,h;
    
    i= -1;
    j= number_of_windows;
    while (i<j-1) {
	h= (i+j)/2;
	if (winfo[h].win_id < win)      i=h;
	else if (winfo[h].win_id > win) j=h;
	else return h;
    }
    return i;
}

static int expand_windows(int nr)
{
    if (number_of_windows+nr>max_usable) {
	int newsize = (((number_of_windows+nr)+EXPAND_SIZE-1) / EXPAND_SIZE) *
	    EXPAND_SIZE;
	EXTENDEDINFO *temp =
	    (EXTENDEDINFO *) malloc(newsize * sizeof(EXTENDEDINFO));
	int i;
	if (!temp) return False;
	for (i=0; i<number_of_windows; i++) {
	    temp[i] = winfo[i];
	}
	myfree(winfo);
	winfo = temp;
	max_usable = newsize;
	return True;
    }
    return True;
}

static void shift_left(int i)   /* schuif alles na i 1 plaats naar links */
{
    while (i<number_of_windows) {
	winfo[i]=winfo[i+1];
	i++;
    }
    number_of_windows--;
}

static void shift_right(int i)  /* schuif alles na i 1 plaats naar rechts */
{
    int j=number_of_windows-1;
    
    while (j>i) {
	winfo[j+1] = winfo[j];
	j--;
    }
    number_of_windows++;
}

WINDOWTYPE get_window_type(Window win, Window *pwin, void **wdata)
{
    int i;
    
    i = get_pos(win);
    if (i==-1 || winfo[i].win_id != win || !exist_window(winfo[i].pwin_id)) {
	*pwin = 0;
	*wdata = NULL;
	return NOWINDOW;
    }
    *wdata = winfo[i].data;
    *pwin = winfo[i].pwin_id;
    return winfo[i].type;
}

Bool exist_window(Window win)
{
    int i;

    i = get_pos(win);
    return (i!=-1 && winfo[i].win_id == win);
}

int add_window(Window win, WINDOWTYPE wtype, Window pwin, void *wdata,
	       char *helpfile)
{
    int i;

    if (!expand_windows(1)) {
	message(ERROR, "Not enough windows available.");
	return 0;
    }
    i = get_pos(win);
    if (i!=-1 && winfo[i].win_id == win) return 0;
    shift_right(i);
    i++;
    winfo[i].state = VisibilityFullyObscured;
    winfo[i].mapped = False;
    winfo[i].win_id = win;
    winfo[i].pwin_id = pwin;
    winfo[i].type = wtype;
    winfo[i].data = wdata;
    winfo[i].helpfile = helpfile;
    return 1;
}

char* window_help(Window win)
{
    int i;
    i = get_pos(win);
    if (i!=-1 && winfo[i].win_id == win) return winfo[i].helpfile;
    else return NULL;
}

void change_visibility(Window win, int state)
{
    int i;
    i = get_pos(win);
    if (i==-1 || winfo[i].win_id != win) return;
    winfo[i].state = state;
}

void change_mapped(Window win, int mapped)
{
    int i;
    i = get_pos(win);
    if (i==-1 || winfo[i].win_id != win) return;
    winfo[i].mapped = mapped;
}

void refresh_all(void)
{
    int i,k;
    for (i=0; i<number_of_windows; i++) {
	k=i;
	if (winfo[i].state != VisibilityFullyObscured &&
	    winfo[i].type !=BUTTONWINDOW &&
	    winfo[i].mapped && eventfunc[winfo[i].type]->draw) {
	    if (!winfo[i].data) {
		int j;
		j = get_pos(winfo[i].pwin_id);
		if (j!=-1 && winfo[j].win_id == winfo[i].pwin_id &&
		    winfo[j].data) k=j;
	    }
	    if (winfo[k].data)
		(*(eventfunc[winfo[i].type]->draw))(winfo[k].data);
	    else
		(*(eventfunc[winfo[i].type]->draw))((void*) &winfo[i].win_id);
	}
    }
}

void *next_data_with_type(WINDOWTYPE wt, int *i)
{
    if (*i<0) *i=0;
    while (*i<number_of_windows && winfo[*i].type != wt) (*i)++;
    if (*i==number_of_windows)
	return NULL;
    return winfo[*i].data;
}

void *remove_window(Window win)
{
    int i;
    void *temp;
    
    i = get_pos(win);
    if (i==-1 || winfo[i].win_id != win) return NULL;
    temp=winfo[i].data;
    shift_left(i);
    while (i<number_of_windows) {
	if (winfo[i].pwin_id==win) winfo[i].mapped=False;
	i++;
    }
    return temp;
}

void destroy_window(Window win)
{
    void *temp;
    
    if ((temp = remove_window(win))) myfree(temp);
}

static Atom targets[5], target_string=0, data_prop=0, clipboard=0,
            target_targets=0, target_length=0;
static Window select_window = 0;

static void make_atoms(void)
{
    if (!target_string) {
	target_string = XInternAtom(display, "STRING", True);
	targets[0] = target_string;
    }
    if (!target_targets) {
	target_targets = XInternAtom(display, "TARGETS", True);
	targets[1] = target_targets;
    }
    if (!target_length) {
	target_length = XInternAtom(display, "LENGTH", True);
	targets[2] = target_length;
    }
    if (!clipboard) clipboard = XInternAtom(display, "CLIPBOARD", True);
}

void set_selection_window(Window win)
{
    select_window = win;
}

void get_wm_selection(void)
{
    if (!select_window) return;
    if (!data_prop) data_prop = XInternAtom(display, "STRING_ASK", False);
    make_atoms();
    if (data_prop && target_string)
	XConvertSelection(display, XA_PRIMARY, target_string, data_prop,
			  select_window, last_time);
}

Bool set_wm_selection(void)
{
    if (!select_window) return False;
    make_atoms();
    if (clipboard) {
	XSetSelectionOwner(display, clipboard, select_window, last_time);
	(void) XGetSelectionOwner(display, clipboard);
    }
    XSetSelectionOwner(display, XA_PRIMARY, select_window, last_time);
    return (XGetSelectionOwner(display, XA_PRIMARY)==select_window);
}

void set_clipboard(void)
{
/*    if (latexselection && target_string && clipboard)
	XChangeProperty(display, select_window, clipboard,
			target_string, 8, PropModeReplace,
			latexselection, strlen(latexselection)); */
}

void send_selection(XSelectionRequestEvent *event)
{
    XSelectionEvent sev;
    make_atoms();
    if (event->target==target_string) {
	if (latexselection)
	    XChangeProperty(event->display, event->requestor, event->property,
			    event->target, 8, PropModeReplace,
			    (unsigned char*) latexselection,
			    strlen(latexselection));
	else
	    XChangeProperty(event->display, event->requestor, event->property,
			    event->target, 8, PropModeReplace,
			    "", 0);
	sev.property = event->property;
    } else if (event->target==target_targets) {
	XChangeProperty(event->display, event->requestor, event->property,
			event->target, 32, PropModeReplace,
			(unsigned char*) targets, 3);
	sev.property = event->property;
    } else if (event->target==target_length) {
	int l;
	if (latexselection) l=strlen(latexselection); else l=0;
	XChangeProperty(event->display, event->requestor, event->property,
			event->target, 32, PropModeReplace,
			(unsigned char*) &l, 1);
	sev.property = event->property;
    } else
	sev.property = None;
    sev.type = SelectionNotify;
    sev.serial = event->serial;
    sev.send_event = True;
    sev.display = event->display;
    sev.target = event->target;
    sev.requestor = event->requestor;
    sev.selection = event->selection;
    sev.time = event->time;
    XSendEvent(event->display, event->requestor, True, 0, (XEvent*) &sev);
}

/*
**
**  Functies die de gegevens initialiseren of verwijderen:
**
*/

static void make_fontpath(void)
{
    char *cp, *fp, **cdirs, **adirs, **ndirs;
    int i, j, ol, nl, ncdirs, nadirs, nndirs;

    if (!(cp = getenv("MATHPADFONTPATH"))) cp = MATHPADHOME "/fonts";

    if ((i = strlen(cp)) == 0) return;

    if (!(cdirs = XGetFontPath(display, &ncdirs))) {
        message(ERROR, "Cannot get old font path.");
        return;
    }

    if (!(fp = (char*) malloc(i+1))) message(EXIT-1, "Out of memory.");
    strcpy(fp, cp);
    nadirs = i = 1;
    for (cp = fp; (cp = strchr(cp,',')) != NULL; cp++) nadirs++;
    if (!(adirs = (char**) malloc(nadirs*sizeof(char*))))
	message(EXIT-1, "Out of memory.");
    adirs[0] = fp;
    for (cp = fp; (cp=strchr(cp,',')) != NULL; *cp++ = '\0') adirs[i++] = cp+1;

    if (!(ndirs = (char**) malloc((ncdirs+nadirs)*sizeof(char*))))
	message(EXIT-1, "Out of memory.");
    for (i = 0; i != ncdirs; i++) ndirs[i] = cdirs[i];
    nndirs = ncdirs;
    for (i = 0; i != nadirs;  i++) {
	nl = strlen(adirs[i]);
        for (j=0; j!=nndirs; j++) {
	    ol = strlen(ndirs[j]);
	    if (ol-nl<2 && nl-ol<2) {
		int dl;
		dl = (ol<nl ? ol : nl);
		if ((dl==nl || adirs[i][nl-1]=='/') &&
		    (dl==ol || ndirs[j][ol-1]=='/') &&
		    !strncmp(adirs[i],ndirs[j],dl-1))  break;
	    }
	}
        if (j == nndirs) ndirs[nndirs++] = adirs[i];
    }

    if (nndirs != ncdirs) XSetFontPath(display, ndirs, nndirs);

    free((char*)ndirs);
    free((char*)adirs);
    free(fp);
    XFreeFontPath(cdirs);
}

static Bool in_dump = False;
static Bool in_auto_save = False;
static Bool alarm_works = True;
static Bool alarm_went_off = False;
Bool can_auto_save = True;

static void do_auto_save(void)
{
    int i;
    int j=0;

    if (!in_dump) {
	message(MESSAGE,"AutoSave in progress ... ");
	XFlush(display);
    }
    in_auto_save = True;

    for (i=0; i<number_of_windows; i++) {
	if (eventfunc[winfo[i].type]->auto_save) {
	    if (in_dump) j++;
	    (*(eventfunc[winfo[i].type]->auto_save))(winfo[i].data, j);
	}
    }
    if (!in_dump) {
	message(MESSAGE,"AutoSave done");
	XFlush(display);
    }
    in_auto_save = False;
    alarm_went_off = False;
}

static Bool alarm_set = False;

typedef struct {
    int signum;
    char *sigcomment;
    void (*oldhandle)(int);
    int sigtype; /* 1=dump backups,reset old: 2=alarm: 3=backups,quit */
} SIGTYPE;

static SIGTYPE catchsig[] = {
#ifdef SIGHUP
                              { SIGHUP, "Hanging up.", NULL, 3 },
#endif
#ifdef SIGINT
                              { SIGINT, "Interrupt.", NULL, 3 },
#endif
#ifdef SIGQUIT
                              { SIGQUIT, "Quit.", NULL, 1 },
#endif
#ifdef SIGILL
                              { SIGILL, "Illegal instruction.", NULL, 1 } ,
#endif
#ifdef SIGBUS
                              { SIGBUS, "Bus error.", NULL, 1 } ,
#endif
#ifdef SIGSEGV
                              { SIGSEGV, "Segmentation fault.", NULL, 1 },
#endif
#ifdef SIGALRM
                              { SIGALRM, NULL, NULL, 2 },
#endif
#ifdef SIGTERM
                              { SIGTERM, "Terminated.", NULL, 3 },
#endif
#ifdef SIGLOST
                              { SIGLOST, "Resource lost.", NULL, 3 },
#endif
                              { 0, NULL, NULL, 0 } };

static void handle(int sig)
{
    int i=0;

    while (catchsig[i].sigtype && catchsig[i].signum!=sig) i++;
    switch (catchsig[i].sigtype) {
    case 1:
	signal(sig, handle);
	if (!in_dump) {
	    in_dump = True;
	    do_auto_save();
	    fprintf(stderr, "%s: %s Dumps made.\n",
		    progname, catchsig[i].sigcomment);
	    in_dump = False;
	} else
	    fprintf(stderr, "%s: %s\n", progname, catchsig[i].sigcomment);
	signal(sig, catchsig[i].oldhandle);
	break;
    case 3:
	signal(sig, handle);
	if (!in_dump) {
	    in_dump = True;
	    do_auto_save();
	    in_dump = False;
	    fprintf(stderr, "%s: %s Dumps made.\n",
		    progname, catchsig[i].sigcomment);
	}
	exit(1);
	break;
    case 2:
	signal(sig,handle);
	if (in_dump) break;
	alarm_works = True;
	alarm_went_off = !in_auto_save;
	if (can_auto_save && alarm_went_off) {
	    do_auto_save();
	    alarm_went_off = False;
	}
	alarm(save_minute*60);
	alarm_set = True;
	break;
    default:
	break;
    }
}

static Bool first_error = True;

static int error_handler(Display *d, XErrorEvent *err)
{
    char msg[80];
    int i;

    if (err->request_code == 51) {
	fprintf(stderr, "%s: Unable to add fontpath.", progname);
	return 0;
    }
    fprintf(stderr, "%s: Error from server:\n", progname);
    if (d!=display)
	fprintf(stderr, "  ? ");
    XGetErrorText(d, err->error_code, msg, 80);
    i = 0;
    while (!isspace(msg[i])) i++;
    msg[i] = '\0';
    fprintf(stderr, "   (%s) %i.%i 0x%lx ",
	    msg, err->request_code, err->minor_code, err->resourceid);
    if (err->error_code == BadWindow || err->error_code == BadDrawable) {
	WINDOWTYPE wt;
	void *data;
        Window par;
	wt = get_window_type((Window) err->resourceid, &par, &data);
	if (wt)
	    fprintf(stderr, "%i 0x%lx %p",wt,par,data);
    }
    fprintf(stderr, "\n");
    if (first_error) {
	in_dump = True;
	do_auto_save();
	in_dump = False;
	fprintf(stderr, "%s Dumps made.\n", progname);
	first_error = False;
    }
    return 0;
}

#ifdef IOERROR_HANDLE
static int ioerror_handler(Display *d, XErrorEvent *err)
#else
static int ioerror_handler(Display *d)
#endif
{
    if (d!=display)
	fprintf(stderr, "%s Error from strange display.\n", progname);
    fprintf(stderr, "%s: XIO-error. Program has to terminate.\n", progname);
    in_dump = True;
    do_auto_save();
    fprintf(stderr, "%s: Dumps made.\n", progname);
    exit(2);
}

void server_init(void)
{
    int i;
    if ((display=XOpenDisplay(displayname)) == NULL) {
	fprintf(stderr, "%s: Cannot connect to server %s\n",
		progname, XDisplayName(displayname));
	exit(-1);
    }
    screen_num = DefaultScreen(display);
    root_window = RootWindow(display, screen_num);
    display_width = DisplayWidth(display,screen_num);
    display_height = DisplayHeight(display,screen_num);
    if (visualclass>=StaticGray && visualclass<=DirectColor) {
	XVisualInfo vinfo, *vinforet;
	int vinfonr;
	vinfo.class=visualclass;
	vinfo.screen=screen_num;
	vinforet=XGetVisualInfo(display,VisualScreenMask|VisualClassMask,
				&vinfo, &vinfonr);
	if (vinfonr && vinforet) visual=vinforet[0].visual;
	def_depth=vinfo.depth;
	colormap = XCreateColormap(display, root_window, visual, AllocNone);
	if (vinforet) XFree(vinforet);
    }
    if (!visual) {
	XVisualInfo vinfo, *vinforet;
	VisualID vid;
	int vinfonr;
	visual=DefaultVisual(display,screen_num);
	vid=XVisualIDFromVisual(visual);
	vinfo.visualid=vid;
	vinforet=XGetVisualInfo(display, VisualIDMask, &vinfo, &vinfonr);
	if (vinfonr && vinforet) visualclass = vinforet[0].class;
	if (vinforet) XFree(vinforet);
	def_depth=DefaultDepth(display, screen_num);
	colormap=DefaultColormap(display,screen_num);
    }
    black_pixel = BlackPixel(display,screen_num);
    white_pixel = WhitePixel(display,screen_num);
    icon_pixmap = XCreateBitmapFromData(display, root_window,
					progicon_bits, progicon_width,
					progicon_height);
    WM_SAVE_YOURSELF = XInternAtom(display, "WM_SAVE_YOURSELF", True);
    WM_DELETE_WINDOW = XInternAtom(display, "WM_DELETE_WINDOW", True);
    XSetErrorHandler(error_handler);
    XSetIOErrorHandler(ioerror_handler); /* Add option -DIOERROR_HANDLE */
    if (WM_SAVE_YOURSELF==None || WM_DELETE_WINDOW == None) 
	message(MESSAGE, "No save_yourself or delete_window available.");
    protocol[1]=WM_SAVE_YOURSELF;
    protocol[0]=WM_DELETE_WINDOW;
    make_fontpath();
    nr_xfont = 1;
    xfinfo[0].f_id=0;
    xfinfo[0].name = NULL;
    xfinfo[0].locks = 1;
    xfinfo[0].char_info = (CHARINFO*) malloc(256*sizeof(CHARINFO));
    nr_latex = 1;
    linfo[0].name = NULL;
    linfo[0].code = concat("","");
    for (i=0; i<256; i++)
	linfo[0].normal[i]=linfo[0].math[i]=xfinfo[0].char_info[i].width = 0;
    read_general_fontinfo();
    for (i=0; i<256; i++)
	if (gfinfo[i].texfilename)
	    gfinfo[i].tnr = load_latexfile(gfinfo[i].texfilename);
	else
	    gfinfo[i].tnr = gfinfo[0].tnr;
    make_fonts();
    make_colors();
    make_GCs();
    wait_cursor = XCreateFontCursor(display, XC_watch);
    (void) add_window(root_window, NOWINDOW, root_window, NULL, NULL);
    user_id = getuid();
    group_id = getgid();

    i=0;
    while (catchsig[i].sigtype) {
	catchsig[i].oldhandle = signal(catchsig[i].signum, handle);
	i++;
    }
}

void server_close(void)
{
    int i;
    for (i=0; i<NR_SIZE; i++)
	destroy_fonts(i);
    destroy_GCs();
    XCloseDisplay(display);
    exit(0);
}

Window motion_window = 0, when_motion_window=0;
Bool   motion_set = False, motion_wait = False, is_click = True;
unsigned long press_time = 0, drag_time = 0, release_time = 0;
Bool   double_click = False;
unsigned int mouse_button = 0;
unsigned int press_state = 0;

#define ANYBUTTON (Button1Mask | Button2Mask | Button3Mask | \
		   Button4Mask | Button5Mask)

Bool mouse_press(unsigned int status, unsigned int bnr)
{
    if (!(status & ANYBUTTON)) {
	press_state = status;
	if (status&ControlMask) bnr++;
	if (status&Mod1Mask) bnr+=2;
	mouse_button=bnr;
	is_click = True;
	return True;
    } else
	return False;
}

Bool mouse_release(unsigned int status, unsigned int bnr)
{
    return ((status & ANYBUTTON) ^ (Button1Mask<<(bnr-1)))==0;
}

void set_save_period(int minute)
{
    char s[50];
    if (minute>0) {
	alarm(60*minute);
	alarm_set = True;
	save_minute = minute;
	sprintf(s, "AutoSave every %i minutes.", minute);
	message(MESSAGE, s);
    } else {
	do_auto_save();
    }
}

Bool is_drag(unsigned long  motion_time)
{
    if (motion_set && motion_wait) {
	if (motion_time > drag_time) {
	    motion_wait = False;
	    return True;
	} else
	    return False;
    }
    return motion_set;
}

Bool motion_get_pos(int *x, int *y)
{
    Window root, child;
    int rx, ry;
    unsigned int buttons;

    if (motion_set && motion_window)
	return XQueryPointer(display, motion_window, &root, &child,
			     &rx, &ry, x, y, &buttons);
    else
	return False;
}

void get_motion_hints(Window win, int ms)
{
    motion_window = win;
    motion_set = True;
    if (ms == -1) ms = wait_time;
    motion_wait = (ms>0);
    drag_time = press_time+ms;
}

void stop_motion_hints(void)
{
    motion_set = False;
    motion_window = 0;
    motion_wait = False;
}

static Window wc_window=0;

void set_wait_cursor(Window win)
{
    XSetWindowAttributes swat;

    swat.cursor=wait_cursor;
    wc_window=win;
    XChangeWindowAttributes(display, win, CWCursor, &swat);
    XFlush(display);
}

void remove_wait_cursor(void)
{
    XSetWindowAttributes swat;
    swat.cursor=None;
    XChangeWindowAttributes(display, wc_window, CWCursor, &swat);
    XFlush(display);
    wc_window=0;
}


static void (*create_func)(int,Bool) = 0;
static int create_arg = 0;

static void handle_directory_creation(void *data, int bnr)
{
    Bool made = False;
    char *c;
    void (*f)(int,Bool) = create_func;
    int a = create_arg;

    c = (char*)data + strlen((char*)data)-1;
    if (*c=='/') *c = '\0';
    else c = NULL;
    if (!bnr) {
	made = !mkdir((char*)data,511);
    }
    create_func = NULL;
    create_arg = 0;
    if (f) (*f)(a, made);
    if (c) *c = '/';
}

static Bool create_directory(char *data, void (*func)(int,Bool), int arg)
{
    DIR *d;
    FILE *f;
    char *c;

    if ((d = opendir(data))) {
	closedir(d);
	return True;
    }
    if ((f = fopen(data, "r"))) {
	fclose(f);
	return False;
    }
    c = concat(data,
	       "\nis a directory that MathSpad needs.\nMust it be created ?");
    if (c) {
	create_func = func;
	create_arg = arg;
	remark_make(root_window, (void*) data, handle_directory_creation,
		    REMARK_CENTRE, char2Char(c), " Create \n Cancel ",
		    NULL,0, NULL);
	free(c);
    }
    return False;
}

static char **needed_dir[3] = { &userdir, &notationdir, &latexdir }; 

void create_needed_directories(int number, Bool made)
{
    if (made) {
	while (number<3) {
	    if (create_directory( *needed_dir[number],
				  create_needed_directories, number+1))
		number++;
	    else
		return;
	}
    }
}

/*
**  Defaults loading
*/

static char *defaultfilename = NULL;

#define DEFAULTFILENAME  ".mpdefaults"
#define NR_FILEDESC 19
#define NR_FILEDESC_OLD 21
static char *filedesc[NR_FILEDESC_OLD] =
    { "ScreenLine", "ScreenTab",    "ScreenSpace",  "LatexLine",
      "LatexTab",   "LatexSpace",   "DirUser",      "DirStencils",
      "DirLatex",   "Fonts",        "StencilFont",  "SymbolFont",
      "PopupFont",  "KeyPath",      "AutoSave",     "LeftMargin",
      "TextDots",   "ClickTime",    "Position",     "DirNotations",
      "NotationFont", };

static char **filechar[NR_FILEDESC_OLD] =
    { NULL, NULL, NULL, &latex_line_unit, &latex_tab_unit,
      &latex_space_unit, &userdir, &notationdir, &latexdir,
      &fontfile[0], &fontfile[1], &fontfile[2], &fontfile[3],
      &keypath, NULL, NULL, NULL, NULL, NULL, &notationdir,
      &fontfile[1] };
static int *fileint[NR_FILEDESC_OLD] =
    { &line_space, &screen_tab, &screen_space, NULL, NULL,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, &save_minute, &latex_side, &textdots, &wait_time,
      NULL, NULL, NULL };

static char *windowdesc[MAXWINDOWTYPE+2] =
    { "No Window", "Button", "Scrollbar", "String", "Menu", "Edit",
      "SubEdit", "Buffer", "SubBuffer", "Stencil", "StencilFile",
      "Define", "Symbol", "Default", "Remark", "Find", "FileSelc",
      "Popup", "CheckBox", "Notation", "NotationFile" };

static char *no_return_copy(char *str)
{
    char *tmp,*ind;

    tmp = ind = (char *) malloc(strlen(str)+1);
    while (*str && *str!='\n')
	*ind++ = *str++;
    *ind='\0';
    return tmp;
}

extern void default_update(void);

static void load_defaults(FILE *file)
{
    int i,x,y,w,h,k;
    char *id;

    while (fgets(buffer, BUFSIZE, file) && strcmp(buffer, "STOP\n")) {
	if (buffer[0]!='#') {
	    id = NULL;
	    for (i=0;
		 i<NR_FILEDESC_OLD && !(id=begins_with(filedesc[i],buffer));
		 i++);
	    if (id) {
		if (filechar[i]) {
		    myfree(*filechar[i]);
		    *filechar[i] = no_return_copy(id);
		} else if (fileint[i])
		    sscanf(id, "%i", fileint[i]);
		else {
		    sscanf(id, "%i %i %i %i %i", &k, &x, &y,&w,&h);
		    if (eventfunc[k]->set_last_pos)
			(*(eventfunc[k]->set_last_pos))(x,y,w,h);
		}
	    } else {
		i=MAXWINDOWTYPE;
		while (i && !(id=begins_with(windowdesc[--i],buffer)));
		if (id) {
		    sscanf(id, "%i\t%i\t%i\t%i", &x, &y, &w, &h);
		    if (eventfunc[i]->set_last_pos)
			(*(eventfunc[i]->set_last_pos))(x,y,w,h);
		}
	    }
	}
    }
    userdir = standard_dir(userdir);
    notationdir = standard_dir(notationdir);
    latexdir = standard_dir(latexdir);
    default_update();
}

static void write_defaults(FILE *f)
{
    int i;

    fprintf(f,"#\n#  Default values for MathSpad.\n#\n");
    for (i=0; i<NR_FILEDESC; i++) {
	if (filechar[i])
	    fprintf(f, "%s\t%s\n", filedesc[i], *filechar[i]);
	else if (fileint[i])
	    fprintf(f, "%s\t%i\n", filedesc[i], *fileint[i]);
	else {
	    int x,y,w,h,j;
	    int dx,dy;
	    Bool add;

	    fprintf(f, "#\n#  Default window positions.\n#\n"
		       "#Type\t\tx\ty\twidth\theight\n");
	    window_manager_added(select_window, &dx, &dy);
	    for (j=NOWINDOW; j<MAXWINDOWTYPE; j++) {
		if (eventfunc[j]->last_pos) {
		    add = (*(eventfunc[j]->last_pos))(&x,&y,&w,&h);
		    /* virtual window managers move windows */
		    while (x<0) x+=display_width;
		    while (y<0) y+=display_height;
		    while (x>display_width) x-=display_width;
		    while (y>display_height) y-=display_height;
		    if (add) { x -= dx; y -= dy; }
		    fprintf(f, "%s", windowdesc[j]);
		    add = strlen(windowdesc[j]);
		    while (add<16) {
			fprintf(f, "\t");
			add = add + (8-(add&7));
		    }
		    fprintf(f, "%i\t%i\t%i\t%i\n", x, y, w, h);
		}
	    }
	}
    }
    fprintf(f, "#  Don't remove the next 'STOP'\nSTOP\n");
}

void save_defaults(void)
{
    FILE *file;

    if ((file = fopen(defaultfilename, "w"))) {
	write_defaults(file);
	fclose(file);
    }
}

static FILE* open_project_file(char *name)
{
    FILE *f;
    int i,j;

    concat_in(buffer, homedir, "mathspad/");
    i = strlen(buffer);
    strcat(buffer, strip_name(name));
    j = strlen(buffer);
    if (strcmp(buffer+(j-4), ".mpj")) strcat(buffer, ".mpj");
    if ((f = fopen(buffer, "r"))) return f;  /* $HOME/mathspad/name.mpj */
    if ((f = open_file(program_dir, buffer+i, "r")))
	return f;                            /* $MATHPAD/name.mpj */
    return NULL;
}

static char *visname[6] = { "StaticGray",  "GrayScale", "StaticColor",
			    "PseudoColor", "TrueColor", "DirectColor" };
static int viscode[6] = { StaticGray,  GrayScale, StaticColor,
			  PseudoColor, TrueColor, DirectColor };
#define MAXVISUAL 6


static int parse_colorstring(char *c)
{
    int n=0;
    if (!c) return 0;
    while (*c) {
	switch (*c) {
	case 'T': case 't': case '1': n=n|4; break;
	case 'S': case 's': case '2': n=n|1; break;
	case 'A': case 'a': case '3': n=n|2; break;
	default: break;
	}
	c++;
    }
    return n;
}

void make_defaults(int argc, char *argv[])
{
    FILE *f=NULL;
    int i;
    Bool isopt;
    char *c;
    Bool noopt=False;

    get_currentwd();
    arguments = argv;
    number_of_arguments = argc;
    progname = strip_name(argv[0]);
    for (i=1; i<argc; i++) {
	if (argv[i][0]=='-') {
	    isopt=True;
	    switch (argv[i][1]) {
	    case 'd':
		if (!strcmp(argv[i]+2,"isplay") && i<argc-1)
		    displayname = argv[++i];
		else
		    isopt=False;
		break;
	    case 'v':
		if (!strcmp(argv[i]+2, "isual") && i<argc-1) {
		    int n;
		    i++;
		    for (n=0;n<MAXVISUAL && strcmp(argv[i],visname[n]);n++);
		    if (n==MAXVISUAL) {
			fprintf(stderr,
				"%s: Unknown visual class \"%s\"."
				"Using default.\n",
				progname, argv[i]);
		    } else
			visualclass=viscode[n];
		} else
		    isopt=False;
		break;
	    case 'i':
		if (!strcmp(argv[i]+2,"conic"))
		    iconic = True;
		else
		    isopt=False;
		break;
	    case 'a':
		if (!strcmp(argv[i]+2,"scii"))
		    output_mode = ASCII;
		else
		    isopt=False;
		break;
	    case 'g':
		if (!strcmp(argv[i]+2, "ray")) {
		    usegray=1;
		} else isopt=False;
		break;
	    case 'b':
		if (argv[i][2]=='g' && i<argc-1) {
		    int nr;
		    nr = parse_colorstring(argv[i]+3);
		    colorname[nr*2+1]=argv[++i];
		} else isopt=False;
		break;
	    case 'f':
		if (argv[i][2]=='g' && i<argc-1) {
		    int nr;
		    nr = parse_colorstring(argv[i]+3);
		    colorname[nr*2]=argv[++i];
		} else if (!strcmp(argv[i]+2, "onts") && i<argc-1) {
		    if (generalinfofile) free(generalinfofile);
		    generalinfofile = concat(argv[++i],"");
		} else
		    isopt=False;
		break;
	    case 'c':
		if (!strcmp(argv[i]+2, "olors") && i<argc-1) {
		    i++;
		    sscanf(argv[i], "%i", &maxcolors);
		    if (maxcolors==0) maxcolors=MAX_CELLS;
		} else
		    isopt=False;
		break;
	    case 'p':
		if (!strcmp(argv[i]+2, "roject") && i<argc-1) {
		    if (project_name) free(project_name);
		    project_name = concat(argv[++i], "");
		} else if (!strcmp(argv[i]+2, "lain"))
		    output_mode = PLAINTEX;
		else
		    isopt=False;
		break;
	    default:
		isopt=False;
		break;
	    }
	    if (!isopt) {
		if (noopt) {
		    c=program_keypath;
		    program_keypath = concat(c, argv[i]+1);
		    myfree(c);
		} else {
		    myfree(program_keypath);
		    program_keypath = concat(argv[i]+1, "");
		    noopt=True;
		}
	    }
	} else {
	    if (project_name) free(project_name);
	    project_name = concat(argv[i],"");
	}
    }
    if (!generalinfofile)
	generalinfofile=concat(GENERALINFOFILE,"");
    defaultfilename = concat( homedir, DEFAULTFILENAME);
    if (project_name)
	f = open_project_file(project_name);
    if (!f && !(f = fopen(defaultfilename, "r")))
	f = open_file(program_dir, DEFAULTFILENAME, "r");
    if (f) {
	load_defaults(f);
	fclose(f);
    }
}

/*
** Functions and variables for saving and loading a save-state-file (.mpj)
*/

static FILE *ssf = NULL;

static void set_save_state_file(char *name)
{
    concat_in(buffer, homedir, "mathspad/");
    if (!is_directory(buffer)) mkdir(buffer,511);
    strcat(buffer, strip_name(name));
    strcat(buffer, ".mpj");
    ssf = fopen(buffer, "w");
    if (!ssf) {
	message(CLICKREMARK, "Unable to create project file.");
	failure=True;
    } else {
	char *c = project_name;
	project_name = concat(strip_name(name), "");
	myfree(c);
	fprintf(ssf, "#\n#  MathSpad project file.\n");
	write_defaults(ssf);
	fprintf(ssf, "#\n#  Opened windows and loaded stencils.\n#\n"
		"#Type\t\tx\ty\twidth\theight\ticon?\tpos.\tData\n");
    }
}

static void set_save_entry(WINDOWTYPE type, int xpos, int ypos, int width, int height,
		    Bool as_icon, int sbpos, char *string)
{
    int i;
    if (ssf) {
	fprintf(ssf, "%s", windowdesc[type]);
	i = strlen(windowdesc[type]);
	while (i<16) { fprintf(ssf, "\t"); i = i+ 8-(i&7); }
	/* to work with virtual window managers on different resolutions */
	i=0;
	while (xpos<0) { xpos+=display_width; i--; }
	while (xpos>display_width) { xpos-=display_width; i++; }
	if (i) fprintf(ssf, "%i&%i\t", xpos, i);
	else fprintf(ssf, "%i\t", xpos);
	i=0;
	while (ypos<0) { ypos+=display_height; i--; }
	while (ypos>display_height) { ypos-=display_height; i++; }
	if (i) fprintf(ssf, "%i&%i\t", ypos, i);
	else fprintf(ssf, "%i\t", ypos);
	fprintf(ssf, "%i\t%i\t%i\t%i\t%s\n", width, height,
		as_icon, sbpos, (string ? string : ""));
    }
}

void close_save_state_file(void)
{
    if (ssf) fclose(ssf);
    ssf = NULL;
}

Bool get_save_state_file(char *name)
{
    ssf = open_project_file(name);
    if (!ssf) {
	message2(CLICKREMARK, "Unable to open project file ", name);
	return False;
    } else {
	char *h = project_name;
	load_defaults(ssf);
	project_name = concat(strip_name(name), "");
	myfree(h);
	return True;
    }
}

Bool get_save_entry(WINDOWTYPE *type, int *xpos, int *ypos, int *width,
		    int *height, Bool *as_icon, int *sbpos, char **string)
{
    char *idx=NULL;
    int i=0;

    if (!ssf) return False;
    while (!idx && fgets(buffer, BUFSIZE, ssf)) {
	if (buffer[0]!='#') {
	    if (isdigit(buffer[0])) {
		idx = buffer;
		i = read_integer(&idx);
	    } else {
		i=MAXWINDOWTYPE+2;
		while (i && !(idx=begins_with(windowdesc[--i], buffer)));
	    }
	}
    }
    if (!idx) return False;
    if (i==MAXWINDOWTYPE) i=MAINNOTATIONWINDOW;
    else if (i==MAXWINDOWTYPE+1) i=NOTATIONWINDOW;
    *type = (WINDOWTYPE) i;
    *xpos = read_integer(&idx);
    if (*idx=='&') {
	idx++;
	i = read_integer(&idx);
	*xpos += i*display_width;
    }
    *ypos = read_integer(&idx);
    if (*idx=='&') {
	idx++;
	i = read_integer(&idx);
	*ypos += i*display_height;
    }
    *width = read_integer(&idx);
    *height = read_integer(&idx);
    *as_icon = read_integer(&idx);
    *sbpos = read_integer(&idx);
    if (idx[0]) idx[strlen(idx)-1] = '\0';
    if (idx[0])
	*string = standard_dir(concat(idx,""));
    else
	*string = concat("", "");
    return True;
}

void load_project(char *name)
{
    if (get_save_state_file(name)) {
	int x,y,w,h,i,s;
	WINDOWTYPE wt;
	char *str;
	Bool menu_opened = False;
	while (get_save_entry(&wt, &x, &y, &w, &h, &i, &s, &str)) {
	    menu_opened |= (wt==MENUWINDOW);
	    if (eventfunc[wt]->use_state)
		(*(eventfunc[wt]->use_state))(x,y,w,h,i,s,str);
	}
	close_save_state_file();
	clear_file_ref();
	if (!menu_opened) {
	    message(ERROR, "Strange project file.");
	    (*(eventfunc[MENUWINDOW]->use_state))(0,0,0,0,0,0,NULL);
	}
    } else
	(*(eventfunc[MENUWINDOW]->use_state))(0,0,0,0,0,0,NULL);
}

void save_project(char *name)
{
    int x,y,w,h,i,s,j;
    char *str;
    char dir[1000];

    set_save_state_file(name);
    if (failure) return;
    (*(eventfunc[MENUWINDOW]->state))(NULL, &x, &y, &w, &h, &i, &s,&str);
    set_save_entry(MENUWINDOW, x, y, w, h, i, s, str);
    j=0;
    while ((j=get_next_filename(j, &str, &i))>=0) {
	if (i) { /* only open notation files are listed. */
	    concat_in(dir, get_notation_dirname(j), str);
	    strcat(dir, ".mps");
	    set_save_entry(NOTATIONWINDOW, 0,0,0,0,0,0,dir);
	}
    }
    j=0;
    while (j<number_of_windows) {
	if (eventfunc[winfo[j].type]->state && winfo[j].type != MENUWINDOW) {
	    (*(eventfunc[winfo[j].type]->state))(winfo[j].data, &x, &y, &w,
						 &h, &i, &s, &str);
	    set_save_entry(winfo[j].type,x,y,w,h,i,s,str);
	}
        j++;
    }
    close_save_state_file();
    message2(MESSAGE, "Project saved: ", name);
}

void call_layout_change(void)
{
    int i;

    Bool used[MAXWINDOWTYPE];

    for (i=0; i<MAXWINDOWTYPE; used[i++]=0);
    settabsize(screen_tab);
    setsimpletabsize(8*char_width(Font2Char(TEXTFONT,'n'),0));

    for (i=0; i<number_of_windows; i++) {
        if (eventfunc[winfo[i].type]->layout_change) {
	    (*(eventfunc[winfo[i].type]->layout_change))(winfo[i].data);
	    used[winfo[i].type] = True;
	    XFlush(display);
	}
    }
    for (i=0; i<MAXWINDOWTYPE; i++) {
	if (!used[i] && eventfunc[i]->layout_change)
	    (*(eventfunc[i]->layout_change))(NULL);
    }
}

void call_auto_save(unsigned long curtime)
{
    last_time = curtime;
    if (message_time && message_time+5000<curtime) {
	clear_message(False);
	message_time = 0;
    }
    if (!alarm_works)
        if (!save_time || curtime < save_time)
	    save_time = curtime;
        else if (save_time+save_minute*60000 < curtime) {
	    alarm_went_off = True;
	    save_time = curtime;
        }
    if (!alarm_set) {
        alarm(save_minute*60);
        alarm_set = True;
    }
    if (alarm_went_off) {
        can_auto_save = False;
        do_auto_save();
        can_auto_save = True;
        alarm_went_off = False;
    }
}    

void window_manager_added(Window win, int *x, int *y)
{
    unsigned int w,h,b,d;
    Window r;

    XGetGeometry(display, win, &r, x, y, &w, &h, &b, &d);
}
