/*****************************************************************
**
** MathSpad 0.60
**
** Copyright 1996, Eindhoven University of Technology (EUT)
** 
** Permission to use, copy, modify and distribute this software
** and its documentation for any purpose is hereby granted
** without fee, provided that the above copyright notice appear
** in all copies and that both that copyright notice and this
** permission notice appear in supporting documentation, and
** that the name of EUT not be used in advertising or publicity
** pertaining to distribution of the software without specific,
** written prior permission.  EUT makes no representations about
** the suitability of this software for any purpose. It is provided
** "as is" without express or implied warranty.
** 
** EUT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
** SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL EUT
** BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
** DAMAGES OR ANY DAMAGE WHATSOEVER RESULTING FROM
** LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
** OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
** OF THIS SOFTWARE.
** 
** 
** Roland Backhouse & Richard Verhoeven.
** Department of Mathematics and Computing Science.
** Eindhoven University of Technology.
**
********************************************************************/
/*
** File: symbol.c
*/

#include "mathpad.h"
#include "system.h"
#include "funcs.h"
#include "sources.h"
#include "button.h"
#include "message.h"
#include "symbol.h"
#include "keymap.h"
#include "helpfile.h"

#define SYMBOLNAME  "MathSpad Symbol"
#define ICONNAME    "Symbol"

#define SIZEX      16
#define SIZEY       8
#define DEFAULTDX   4
#define DEFAULTDY   2

enum button { PREVBUTTON, NEXTBUTTON, DONEBUTTON, NR_BUTTON };

static struct { int nr;
		int helpnr;
                char *name;
	    } symbolbutton[NR_BUTTON] =
{ {PREVBUTTON, SYMBOLPREVHELP, "Prev"},
  {NEXTBUTTON, SYMBOLNEXTHELP, "Next"},
  {DONEBUTTON, SYMBOLDONEHELP, "Done"} };

typedef
struct {
         Window symbolwin;
	 int pagenr, fnr, spos, dx, dy, sizex, sizey;
	 int xpos, ypos, width, height;
	 Bool iconized;
       } SYMBOLINFO;

static int nr_opened=0;
static int nr_iconized=0;
static int is_opened = False;
static SYMBOLINFO *selected;
static Char select_sym = 0;

static int offset_x, offset_y, selx, sely;
static int last_xpos = 0, last_ypos = 0, last_width = 0, last_height = 0;
static int last_pagenr = 0;
static Bool state_open = False, as_icon = False;
static XTextProperty symbol_name, icon_name;
static char *name = SYMBOLNAME, *iname = ICONNAME;
static XSetWindowAttributes symbol_attr;
static unsigned long symbol_mask;

static int pixel_to_pos(SYMBOLINFO *sinfo, int *x, int *y)
{
    *x = (*x - offset_x) / sinfo->dx;
    *y = (*y - offset_y) / sinfo->dy;
    if (*x>=0 && *x<sinfo->sizex && *y>=0 && *y<sinfo->sizey)
	return 1;
    else return 0;
}

static char pos_to_char(SYMBOLINFO *sinfo, int i, int j)
{
    return (char) (sinfo->spos + i + j*sinfo->sizex);
}

static void pos_to_pixel(SYMBOLINFO *sinfo, int *i, int *j)
{
    *i = offset_x + * i * sinfo->dx;
    *j = offset_y + * j * sinfo->dy;
}

static void draw_char(SYMBOLINFO *sinfo, int i, int j, int gcnr)
{
    int ischar;
    char sym = pos_to_char(sinfo,i,j);

    if ((ischar = char_width(Font2Char(sinfo->fnr, sym),0))) {
	pos_to_pixel(sinfo, &i,&j);
	XFillRectangle(display, sinfo->symbolwin, get_GC(1-gcnr,0),
		       i, j, sinfo->dx, sinfo->dy);
	XDrawString(display, sinfo->symbolwin, get_GC_font(gcnr,0,sinfo->fnr,0),
		    i+ (sinfo->dx - ischar)/2, j+font_ascent_max, &sym, 1);
    }
}

static void symbol_bad_end(void *data)
{
    SYMBOLINFO *sinfo = (SYMBOLINFO *) data;

    if (sinfo->iconized) nr_iconized--;
    nr_opened--;
    if (sinfo == selected) {
	selected = NULL;
	selx = -1;
	sely = -1;
    }
    if (!nr_opened) {
	symbol_is_open = False;
    }
    if (nr_opened == nr_iconized) symbol_iconized=True;
    destroy_window(sinfo->symbolwin);
}    

static void symbol_close(void *data)
{
    SYMBOLINFO *sinfo = (SYMBOLINFO *) data;

    XDestroyWindow(display, sinfo->symbolwin);
    symbol_bad_end(data);
}

static void symbol_draw(void *data)
{
    int i,j;
    SYMBOLINFO *sinfo = (SYMBOLINFO *) data;

    push_fontgroup(SYMBOLFONT);
    for (i=0; i<sinfo->sizex; i++)
	for (j=0; j<sinfo->sizey; j++)
	    draw_char(sinfo, i,j,
		      (sinfo==selected && i==selx && j==sely ? Reverse
		       : Normal));
    pop_fontgroup();
}

static void symbol_layout_change(void *data)
{
    SYMBOLINFO *sinfo = (SYMBOLINFO *) data;
    int x = BINTERSPACE/2;
    int y = BINTERSPACE/2;
    int i;

    if (!data) return;
    push_fontgroup(SYMBOLFONT);
    selx = sely = -1;
    selected = NULL;
    i = sinfo->pagenr;

    if (i>=(int)pagelist.maxpage || pagelist.page[i].fnr!=sinfo->fnr ||
	pagelist.page[i].spos!=sinfo->spos) {
	i=0;
	sinfo->pagenr=0;
	while (i<(int)pagelist.maxpage) {
	    if (pagelist.page[i].fnr==sinfo->fnr) {
		sinfo->pagenr=i;
		if (pagelist.page[i].spos==sinfo->spos)
		    i = pagelist.maxpage;
	    }
	    i++;
	}
    }
    sinfo->spos = pagelist.page[sinfo->pagenr].spos;
    sinfo->fnr = pagelist.page[sinfo->pagenr].fnr;
    sinfo->dx = font_width_max + DEFAULTDX;
    sinfo->dy = font_height_max + DEFAULTDY;
    sinfo->width = offset_x+ sinfo->dx * sinfo->sizex + x;
    sinfo->height = offset_y + sinfo->dy * sinfo->sizey + y;
    XResizeWindow(display, sinfo->symbolwin, sinfo->width, sinfo->height);
    if (!sinfo->iconized) {
	XClearWindow(display, sinfo->symbolwin);
	symbol_draw(data);
    }
    pop_fontgroup();
}

static void symbol_unselect(void)
{
    if (selected) {
	draw_char(selected, selx, sely, Normal);
	selx = sely = -1;
	selected = NULL;
    }
}

static void symbol_handle_button(void *data, int b_num)
{
    SYMBOLINFO *sinfo = (SYMBOLINFO *) data;

    push_fontgroup(SYMBOLFONT);
    switch (b_num) {
    case PREVBUTTON:
	symbol_unselect();
	XClearWindow(display, sinfo->symbolwin);
	if (sinfo->pagenr>0)
	    sinfo->pagenr--;
	else
	    sinfo->pagenr = pagelist.maxpage-1;
	sinfo->fnr = pagelist.page[sinfo->pagenr].fnr;
	sinfo->spos = pagelist.page[sinfo->pagenr].spos;
	last_pagenr = sinfo->pagenr;
	symbol_draw(data);
	break;
    case NEXTBUTTON:
	symbol_unselect();
	XClearWindow(display, sinfo->symbolwin);
	if (sinfo->pagenr<(int)pagelist.maxpage-1)
	    sinfo->pagenr++;
	else
	    sinfo->pagenr=0;
	sinfo->fnr = pagelist.page[sinfo->pagenr].fnr;
	sinfo->spos = pagelist.page[sinfo->pagenr].spos;
	last_pagenr = sinfo->pagenr;
	symbol_draw(data);
	break;
    case DONEBUTTON:
	if (can_close_symbol) symbol_close(data);
	break;
    }
    pop_fontgroup();
}

static void symbol_press(void *data, XButtonEvent *event)
{
    int x = event->x;
    int y = event->y;
    SYMBOLINFO *sinfo = (SYMBOLINFO *) data;

    push_fontgroup(SYMBOLFONT);
    if (pixel_to_pos(sinfo, &x,&y) && !(x==selx && y==sely)) {
	symbol_unselect();
	selected = sinfo;
	selx = x;
	sely = y;
	draw_char(selected, selx, sely, Reverse);
    }
    get_motion_hints(sinfo->symbolwin, -1);
    pop_fontgroup();
}

Char symbol_last(void)
{
    return select_sym;
}

static void symbol_release(void *data, XButtonEvent *event)
{
    int x = event->x;
    int y = event->y;
    SYMBOLINFO *sinfo = (SYMBOLINFO *) data;

    stop_motion_hints();
    push_fontgroup(SYMBOLFONT);
    if (!pixel_to_pos(sinfo, &x,&y))
	symbol_unselect();
    if (selected) {
	if (!char_width(select_sym = 
			Font2Char(selected->fnr,
				  pos_to_char(selected, selx, sely)),0))
	    select_sym = 0;
    }
    symbol_unselect();
    pop_fontgroup();
    call_func(SYMBOL_KEY);
}

static void symbol_resize(void *data, XConfigureEvent *event)
{
    SYMBOLINFO *sinfo = (SYMBOLINFO *) data;
    int maxx,maxy, fitx, fity, dx, dy, fitdx, fitdy;
    Bool tryx = True, tryy = True;

    push_fontgroup(SYMBOLFONT);
    sinfo->xpos = last_xpos = event->x;
    sinfo->ypos = last_ypos = event->y;
    if (sinfo->width!=event->width || sinfo->height!=event->height) {
	if (sinfo==selected) symbol_unselect();
	sinfo->width = last_width = event->width;
	sinfo->height = last_height = event->height;
	fitdx = dx = font_width_max;
	fitdy = dy = font_height_max;
	fitx = maxx = (sinfo->width - offset_x*2) / dx;
	fity = maxy = (sinfo->height - offset_y) / dy;
	while (maxx*maxy >=128) {
	    fitx = maxx;
	    fity = maxy;
	    fitdx = dx;
	    fitdy = dy;
	    if (tryx) {
		dx++;
		maxx = (sinfo->width -offset_x*2)/dx;
		tryx = False;
	    } else if (tryy) {
		dy++;
		maxy = (sinfo->height -offset_y)/dy;
		tryy = False;
	    } else {
		dx++;
		maxx = (sinfo->width -offset_x*2)/dx;
		tryx = tryy = True;
	    }
	}
	sinfo->dx = fitdx;
	sinfo->dy = fitdy;
	sinfo->sizex = fitx;
	sinfo->sizey = fity;
	XClearWindow(display, sinfo->symbolwin);
	symbol_draw(data);
    }
    pop_fontgroup();
}

static void  symbol_motion(void *data, int x, int y)
{
    SYMBOLINFO *sinfo = (SYMBOLINFO *) data;

    push_fontgroup(SYMBOLFONT);
    if (!pixel_to_pos(sinfo, &x,&y)) {
	symbol_unselect();
    } else if ((x!=selx || y!=sely)) {
	symbol_unselect();
	selected = sinfo;
	selx = x;
	sely = y;
	draw_char(selected, selx, sely, Reverse);
    }
    pop_fontgroup();
}

static void symbol_iconize(void *data)
{
    SYMBOLINFO *sinfo = (SYMBOLINFO *) data;

    if (!sinfo->iconized) {
	sinfo->iconized = True;
	nr_iconized++;
	if (nr_iconized == nr_opened) {
	    symbol_iconized = True;
	}
	if (selected == sinfo) {
	    selected = NULL;
	    selx = sely = -1;
	}
    }
}

static void symbol_deiconize(void *data)
{
    SYMBOLINFO *sinfo = (SYMBOLINFO *) data;

    if (sinfo->iconized) {
	sinfo->iconized = False;
	nr_iconized--;
	symbol_iconized = False;
	symbol_is_open = True;
    }
}

static void symbol_state(void *data, int *x, int*y, int *w, int *h,
		  int *i, int *s, char **str)
{
    SYMBOLINFO *sinfo = (SYMBOLINFO *) data;
    int xm,ym;

    window_manager_added(sinfo->symbolwin, &xm,&ym);
    *x = sinfo->xpos-xm;
    *y = sinfo->ypos-ym;
    *w = sinfo->width;
    *h = sinfo->height;
    *i = sinfo->iconized;
    *s = sinfo->pagenr;
    *str = NULL;
}

static void symbol_use_state(int x, int y, int w, int h,
		      int i, int s, char *str)
{
    as_icon = i;
    state_open = True;
    last_xpos = x;
    last_ypos = y;
    last_width = w;
    last_height = h;
    if (s<(int)pagelist.maxpage)
	last_pagenr = s;
    else
	last_pagenr = 0;
    symbol_open();
    state_open = False;
    as_icon = False;
}

static int symbol_last_pos(int *x, int *y, int *w, int *h)
{
    *x = last_xpos;
    *y = last_ypos;
    *w = last_width;
    *h = last_height;
    return is_opened;
}

static void symbol_set_last_pos(int x, int y, int w, int h)
{
    last_xpos = x;
    last_ypos = y;
    last_width = w;
    last_height = h;
}

FUNCTIONS symbolfuncs = {
    symbol_bad_end, symbol_draw, symbol_resize, symbol_press, symbol_release,
    symbol_motion, symbol_iconize, symbol_deiconize, NULL, NULL,
    symbol_layout_change, NULL, symbol_use_state, symbol_state, NULL, NULL,
    symbol_last_pos, symbol_set_last_pos  };

void symbol_init(void)
{
    int x = BINTERSPACE/2;
    int y = BINTERSPACE/2;
    int dx, dy;

    push_fontgroup(SYMBOLFONT);
    offset_x = x;
    offset_y = button_height+BINTERSPACE;
    selx = sely = -1;
    dx = font_width_max + DEFAULTDX;
    dy = font_height_max + DEFAULTDY;
    pop_fontgroup();
    last_width = offset_x+ dx * SIZEX + x;
    last_height = offset_y + dy * SIZEY + y;
    if (!last_xpos && !last_ypos) {
	last_xpos = (display_width - last_width) /2;
	last_ypos = (display_height - last_height) /2;
    }
    if (!XStringListToTextProperty(&name, 1, &symbol_name)) {
	message(EXIT-1, "No location for symbolname.");
	return;
    }
    if (!XStringListToTextProperty(&iname, 1, &icon_name)) {
	message(EXIT-1, "No location for symbol iconname.");
	return;
    }
    symbol_mask = (CWBackPixel | CWBorderPixel | CWBitGravity |
		   CWColormap | CWEventMask);
    symbol_attr.background_pixel = white_pixel;
    symbol_attr.border_pixel = black_pixel;
    symbol_attr.colormap = colormap;
    symbol_attr.bit_gravity = NorthWestGravity;
    symbol_attr.event_mask = (ExposureMask | ButtonPressMask |
			      ButtonMotionMask | PointerMotionHintMask |
			      ButtonReleaseMask | KeyPressMask |
			      StructureNotifyMask | VisibilityChangeMask);
}

void symbol_open(void)
{
    int x = BINTERSPACE/2, y = BINTERSPACE/2;
    int i;
    XSizeHints size_hints;
    SYMBOLINFO *sinfo;

    if (!(sinfo= (SYMBOLINFO *) malloc( sizeof(SYMBOLINFO)))) {
	message(ERROR, "Not enough memory to open window.");
	return;
    }
    push_fontgroup(SYMBOLFONT);
    sinfo->dx = font_width_max+DEFAULTDX;
    sinfo->dy = font_height_max+DEFAULTDY;
    pop_fontgroup();
    sinfo->sizex = SIZEX;
    sinfo->sizey = SIZEY;
    sinfo->width = offset_x + sinfo->dx*sinfo->sizex + x;
    sinfo->height = offset_y + sinfo->dy*sinfo->sizey + y;
    if (!state_open) {
	sinfo->xpos = last_xpos;
	sinfo->ypos = last_ypos;
    } else {
	sinfo->xpos = last_xpos;
	sinfo->ypos = last_ypos;
	sinfo->width = last_width;
	sinfo->height = last_height;
    }
    sinfo->pagenr = last_pagenr;
    sinfo->fnr = pagelist.page[sinfo->pagenr].fnr;
    sinfo->spos = pagelist.page[sinfo->pagenr].spos;
    sinfo->iconized = True;
    sinfo->symbolwin =
	XCreateWindow(display, root_window,
		      sinfo->xpos, sinfo->ypos, sinfo->width, sinfo->height,
		      BORDERWIDTH, CopyFromParent, InputOutput,
		      visual,
		      symbol_mask, &symbol_attr);
    if (!state_open)
	size_hints.flags = PPosition | PSize | PMinSize;
    else {
	size_hints.flags = USPosition | USSize | PMinSize;
	sinfo->width = sinfo->height = 0;
    }
    size_hints.min_height =
    size_hints.min_width  = 3*button_height;
    wm_hints.initial_state = (iconic || as_icon ? IconicState : NormalState);
    i=0;
    XSetWMProperties(display, sinfo->symbolwin, &symbol_name, &icon_name,
		     NULL, 0, &size_hints, &wm_hints, &class_hints);
    set_protocols(sinfo->symbolwin);
    if (add_window(sinfo->symbolwin, SYMBOLWINDOW, root_window,
		   (void *) sinfo, helpname[SYMBOLHELP]))
	while (i<NR_BUTTON && button_make(symbolbutton[i].nr, sinfo->symbolwin,
					  symbolbutton[i].name, &x, y,1,
					  (void*) sinfo,
					  helpname[symbolbutton[i].helpnr],
					  NULL, NULL, NULL,
					  symbol_handle_button, NULL, NULL))
	    i++, x+=BINTERSPACE;
    if (i<NR_BUTTON) {
	XDestroyWindow(display, sinfo->symbolwin);
	myfree(sinfo);
    } else {
	is_opened = True;
	symbol_is_open = True;
	nr_opened++;
	nr_iconized++;
	XMapSubwindows(display, sinfo->symbolwin);
	XMapWindow(display, sinfo->symbolwin);
    }
}

