/*
 * converters.c,v 2.3 1992/08/11 00:05:10 pete Exp
 * converters.c,v
 * Revision 2.3  1992/08/11  00:05:10  pete
 * Added XoShading converter.
 *
 * Revision 2.2  1992/07/11  20:03:46  pete
 * Maded ObjLabelFormat the type for text.
 *
 * Revision 2.1  1992/06/23  00:29:09  pete
 * Changed interface to _XoMenuNew and _XoMenuDone.
 *
 * Revision 2.0  1992/04/23  02:52:14  ware
 * First public release.
 *
 * Revision 1.10  1992/04/23  02:18:58  ware
 * Added several classes.  Worked on geometry management
 *
 * Revision 1.9  1992/03/03  17:19:07  ware
 * Added Orientation and double converters.
 *
 * Revision 1.8  1992/02/27  14:30:29  ware
 * Compiled with GCC 2.0 and very strict checks.  Fixed Warnings
 *
 * Revision 1.7  1992/02/20  15:11:09  ware
 * Applied new indentation
 *
 * Revision 1.6  92/02/04  21:22:46  pete
 * Release 44
 *
 * Revision 1.5  1991/12/01  16:15:21  pete
 * Added BackingStore converters.
 *
 * Revision 1.4  1991/08/26  11:58:11  pete
 * Use XoProto() for conditional prototypes.  Working on getting traversals
 * and menus to work more efficiently.  Changed to following naming
 * conventions.
 *
 * Revision 1.3  91/07/19  00:59:55  pete
 * Use shorter file names.  Various speedups.
 *
 * Revision 1.2  1991/06/01  10:03:15  pete
 * Working on menubar
 *
 * Revision 1.1  91/05/15  08:56:42  pete
 * Initial revision
 *
 */

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xmu/CharSet.h>
#include <X11/Xo/Simple.h>
#include <X11/Xo/Label.h>
#include <X11/Xo/Button.h>
#include <X11/Xo/ObjDraw.h>
#include <X11/Xo/ObjLabelF.h>
#include <X11/Xo/Obj3dB.h>
#include <X11/Xo/Obj2dB.h>
#include <X11/Xo/ObjArrow.h>
#include <X11/Xo/Grip.h>
#include <X11/Xo/Scrollbar.h>
#include <X11/Xo/text.h>
#include <X11/Xo/FontFamily.h>

XoProto (extern int, strcasecmp, (const char *s1, const char *s2));
XoProto (extern int, strncasecmp, (const char *s1, const char *s2, int len));

struct _class_list
{
	char           *cl_name;
	WidgetClass    *cl_class;
};
static WidgetClass BorderDefault = (WidgetClass) 4;
static struct _class_list ClassListArray[] =
{
	"ObjDraw", &xoObjDrawWidgetClass,
	"ObjText", &xoObjLabelFormatWidgetClass,
	"ObjLabel", &xoObjLabelWidgetClass,
	"Obj3dBorder", &xoObj3dBorderWidgetClass,
	"Obj2dBorder", &xoObj2dBorderWidgetClass,
	"ObjArrow", &xoObjArrowWidgetClass,
	"Grip", &xoGripWidgetClass,
	"Button", &xoButtonWidgetClass,
	"DefaultBorder", &BorderDefault,
	"None", NULL,
	NULL, NULL
};
static struct _class_list *ClassList;

static void
_XoGetVisualCvter (gw, size, value)
	Widget          gw;
	Cardinal       *size;
	XrmValue       *value;
{
	static Visual  *v;

	v = XoVisualGet (gw);
	value->addr = (caddr_t) v;
	value->size = sizeof (v);
}

static void
ObjGet (value, name, found)
	WidgetClass    *value;
	char           *name;
	Boolean        *found;
{
	struct _class_list *ptr;

	if (!ClassList)
		ClassList = ClassListArray;
	*found = False;
	*value = NULL;
	for (ptr = ClassList; ptr && ptr->cl_name; ptr++)
	{
		if (strcmp (ptr->cl_name, name) == 0)
		{
			if (ptr->cl_class)
				*value = *ptr->cl_class;
			else
				*value = NULL;
			*found = True;
			break;
		}
	}
}

static Boolean
StrToObjectClass (display, args, num_args, from, to, converter_data)
	Display        *display;	/* display */
	XrmValue       *args;
	Cardinal       *num_args;
	XrmValue       *from;
	XrmValue       *to;
	XtPointer      *converter_data;
{
	static WidgetClass value;
	Boolean         converted;
	Colormap        colormap;
	Visual         *visual;
	String          border;

	ObjGet (&value, (char *) from->addr, &converted);
	if (!converted)
	{
		XtDisplayStringConversionWarning (display,
						  from->addr, "WidgetClass");
		return (False);
	}
	/*
	 * If we are on a color display use ObjBorder, otherwise use
	 * Obj2dBorder.
	 */
	if (value == BorderDefault)
	{
		if (*num_args < 3)
		{
			return False;
		}
		else
		{
			/*
			 * We don't actually use colormap.  It is used as the
			 * unique value to cache the resource conversion.
			 * I.e. a colormap uniquely identifies the visual
			 * type we are using.
			 */
			colormap = (Colormap) args[0].addr;
			visual = (Visual *) args[2].addr;
			if (XoVisualColor (display, visual))
				border = "Obj3dBorder";
			else
				border = "Obj2dBorder";
			ObjGet (&value, border, &converted);
			if (!converted)
				return False;
		}
	}
	if (to->addr)
	{
		if (to->size < sizeof (WidgetClass))
		{
			to->size = sizeof (WidgetClass);
			return False;
		}
		else
		{
			to->size = sizeof (WidgetClass);
			*(WidgetClass *) to->addr = value;
		}
	}
	else
	{
		to->addr = (XtPointer) &value;
		to->size = sizeof (value);
	}
	return converted;
}

/*
 *----------------------------------------------------------------------
 * XoGravity converters
 *----------------------------------------------------------------------
 */

struct _gravity_list
{
	char           *gl_name;
	XoGravity       gl_value;
	XrmQuark        gl_quark;
};
static struct _gravity_list GravityList[] =
{
	"North", XoNORTH, 0,
	"NorthEast", XoNORTHEAST, 0,
	"East", XoEAST, 0,
	"SouthEast", XoSOUTHEAST, 0,
	"South", XoSOUTH, 0,
	"SouthWest", XoSOUTHWEST, 0,
	"West", XoWEST, 0,
	"NorthWest", XoNORTHWEST, 0,
	"Center", XoCENTER, 0,
};

static Boolean  HasGravityQuarks;

static void
InitGravityQuarks ()
{
	int             i;
	char            lower[512];

	for (i = 0; i < XtNumber (GravityList); i++)
	{
		XmuCopyISOLatin1Lowered (lower, GravityList[i].gl_name);
		GravityList[i].gl_quark = XrmStringToQuark (lower);
	}
}

String
XoGetStringFromGravity (gravity)
	XoGravity       gravity;
{
	int             i;

	for (i = 0; i < XtNumber (GravityList); i++)
	{
		if (GravityList[i].gl_value == gravity)
			return GravityList[i].gl_name;
	}
	return ((char *) NULL);
}


XoGravity
XoGetGravityFromString (str)
	String          str;
{
	int             i;
	char            lower[512];
	XrmQuark        q;
	XoGravity       gravity = (XoGravity) -1;

	if (!HasGravityQuarks)
	{
		InitGravityQuarks ();
		HasGravityQuarks = True;
	}
	XmuCopyISOLatin1Lowered (lower, str);
	q = XrmStringToQuark (lower);
	for (i = 0; i < XtNumber (GravityList); i++)
	{
		if (GravityList[i].gl_quark == q)
		{
			gravity = GravityList[i].gl_value;
			break;
		}
	}
	return gravity;
}


static Boolean
StrToGravity (display, args, num_args, from, to, converter_data)
	Display        *display;	/* display */
	XrmValue       *args;
	Cardinal       *num_args;
	XrmValue       *from;
	XrmValue       *to;
	XtPointer      *converter_data;
{
	static XoGravity gravity;

	gravity = XoGetGravityFromString ((char *) from->addr);
	if (to->addr)
	{
		if (to->size < sizeof (XoGravity))
		{
			to->size = sizeof (XoGravity);
			return False;
		}
		else
		{
			*(XoGravity *) to->addr = gravity;
			to->size = sizeof (XoGravity);
		}
	}
	else
	{
		to->addr = (caddr_t) & gravity;
		to->size = sizeof (XoGravity);
	}
	return True;
}

static Boolean
StrToDouble (display, args, num_args, from, to, converter_data)
	Display        *display;	/* display */
	XrmValue       *args;
	Cardinal       *num_args;
	XrmValue       *from;
	XrmValue       *to;
	XtPointer      *converter_data;
{
	static double value;

	value = atof ((char *) from->addr);
	if (to->addr)
	{
		if (to->size < sizeof (double))
		{
			to->size = sizeof (double);
			return False;
		}
		else
		{
			*(double *) to->addr = value;
			to->size = sizeof (double);
		}
	}
	else
	{
		to->addr = (caddr_t) &value;
		to->size = sizeof (double);
	}
	return True;
}

/*
 *----------------------------------------------------------------------
 * XoOrientation converters
 *----------------------------------------------------------------------
 */

struct _orientation_list
{
	char           *gl_name;
	XoOrientation       gl_value;
	XrmQuark        gl_quark;
};
static struct _orientation_list OrientationList[] =
{
	"Vertical", XoVERTICAL, 0,
	"Horizontal", XoHORIZONTAL, 0,
};

static Boolean  HasOrientationQuarks;

static void
InitOrientationQuarks ()
{
	int             i;
	char            lower[512];

	for (i = 0; i < XtNumber (OrientationList); i++)
	{
		XmuCopyISOLatin1Lowered (lower, OrientationList[i].gl_name);
		OrientationList[i].gl_quark = XrmStringToQuark (lower);
	}
}

String
XoGetStringFromOrientation (orientation)
	XoOrientation       orientation;
{
	int             i;

	for (i = 0; i < XtNumber (OrientationList); i++)
	{
		if (OrientationList[i].gl_value == orientation)
			return OrientationList[i].gl_name;
	}
	return ((char *) NULL);
}


XoOrientation
XoGetOrientationFromString (str)
	String          str;
{
	int             i;
	char            lower[512];
	XrmQuark        q;
	XoOrientation       orientation = XoCENTER;

	if (!HasOrientationQuarks)
	{
		InitOrientationQuarks ();
		HasOrientationQuarks = True;
	}
	XmuCopyISOLatin1Lowered (lower, str);
	q = XrmStringToQuark (lower);
	for (i = 0; i < XtNumber (OrientationList); i++)
	{
		if (OrientationList[i].gl_quark == q)
		{
			orientation = OrientationList[i].gl_value;
			break;
		}
	}
	return orientation;
}


static Boolean
StrToOrientation (display, args, num_args, from, to, converter_data)
	Display        *display;	/* display */
	XrmValue       *args;
	Cardinal       *num_args;
	XrmValue       *from;
	XrmValue       *to;
	XtPointer      *converter_data;
{
	static XoOrientation orientation;

	orientation = XoGetOrientationFromString ((char *) from->addr);
	if (to->addr)
	{
		if (to->size < sizeof (XoOrientation))
		{
			to->size = sizeof (XoOrientation);
			return False;
		}
		else
		{
			*(XoOrientation *) to->addr = orientation;
			to->size = sizeof (XoOrientation);
		}
	}
	else
	{
		to->addr = (caddr_t) & orientation;
		to->size = sizeof (XoOrientation);
	}
	return True;
}

/*
 *----------------------------------------------------------------------
 * XoPointing converters
 *----------------------------------------------------------------------
 */

struct _pointing_list
{
	char           *gl_name;
	XoPointing       gl_value;
	XrmQuark        gl_quark;
};
static struct _pointing_list PointingList[] =
{
	"Left", XoLEFT, 0,
	"Right", XoRIGHT, 0,
	"Up", XoUP, 0,
	"Down", XoDOWN, 0,
};

static Boolean  HasPointingQuarks;

static void
InitPointingQuarks ()
{
	int             i;
	char            lower[512];

	for (i = 0; i < XtNumber (PointingList); i++)
	{
		XmuCopyISOLatin1Lowered (lower, PointingList[i].gl_name);
		PointingList[i].gl_quark = XrmStringToQuark (lower);
	}
}

String
XoGetStringFromPointing (pointing)
	XoPointing       pointing;
{
	int             i;

	for (i = 0; i < XtNumber (PointingList); i++)
	{
		if (PointingList[i].gl_value == pointing)
			return PointingList[i].gl_name;
	}
	return ((char *) NULL);
}


XoPointing
XoGetPointingFromString (str)
	String          str;
{
	int             i;
	char            lower[512];
	XrmQuark        q;
	XoPointing       pointing = XoCENTER;

	if (!HasPointingQuarks)
	{
		InitPointingQuarks ();
		HasPointingQuarks = True;
	}
	XmuCopyISOLatin1Lowered (lower, str);
	q = XrmStringToQuark (lower);
	for (i = 0; i < XtNumber (PointingList); i++)
	{
		if (PointingList[i].gl_quark == q)
		{
			pointing = PointingList[i].gl_value;
			break;
		}
	}
	return pointing;
}


static Boolean
StrToPointing (display, args, num_args, from, to, converter_data)
	Display        *display;	/* display */
	XrmValue       *args;
	Cardinal       *num_args;
	XrmValue       *from;
	XrmValue       *to;
	XtPointer      *converter_data;
{
	static XoPointing pointing;

	pointing = XoGetPointingFromString ((char *) from->addr);
	if (to->addr)
	{
		if (to->size < sizeof (XoPointing))
		{
			to->size = sizeof (XoPointing);
			return False;
		}
		else
		{
			*(XoPointing *) to->addr = pointing;
			to->size = sizeof (XoPointing);
		}
	}
	else
	{
		to->addr = (caddr_t) & pointing;
		to->size = sizeof (XoPointing);
	}
	return True;
}

/*
 *----------------------------------------------------------------------
 * XoJustify Converters
 *----------------------------------------------------------------------
 */

struct _justify_list
{
	char           *jl_name;
	XoJustify       jl_value;
	XrmQuark        jl_quark;
};
static struct _justify_list JustifyList[] =
{
	"Left", XoJUST_LEFT, 0,
	"Center", XoJUST_CENTER, 0,
	"Right", XoJUST_RIGHT, 0,
};

static Boolean  HasJustifyQuarks;

static void
InitJustifyQuarks ()
{
	int             i;
	char            lower[512];

	for (i = 0; i < XtNumber (JustifyList); i++)
	{
		XmuCopyISOLatin1Lowered (lower, JustifyList[i].jl_name);
		JustifyList[i].jl_quark = XrmStringToQuark (lower);
	}
}

String
XoGetStringFromJustify (justify)
	XoJustify       justify;
{
	int             i;

	for (i = 0; i < XtNumber (JustifyList); i++)
	{
		if (JustifyList[i].jl_value == justify)
			return JustifyList[i].jl_name;
	}
	return ((char *) NULL);
}


XoJustify
XoGetJustifyFromString (str)
	String          str;
{
	int             i;
	char            lower[512];
	XrmQuark        q;
	XoJustify       justify = XoCENTER;

	if (!HasJustifyQuarks)
	{
		InitJustifyQuarks ();
		HasJustifyQuarks = True;
	}
	XmuCopyISOLatin1Lowered (lower, str);
	q = XrmStringToQuark (lower);
	for (i = 0; i < XtNumber (JustifyList); i++)
	{
		if (JustifyList[i].jl_quark == q)
		{
			justify = JustifyList[i].jl_value;
			break;
		}
	}
	return justify;
}

static Boolean
StrToJustify (display, args, num_args, from, to, converter_data)
	Display        *display;	/* display */
	XrmValue       *args;
	Cardinal       *num_args;
	XrmValue       *from;
	XrmValue       *to;
	XtPointer      *converter_data;
{
	static XoJustify justify;

	justify = XoGetJustifyFromString ((char *) from->addr);
	if (to->addr)
	{
		if (to->size < sizeof (XoJustify))
		{
			to->size = sizeof (XoJustify);
			return False;
		}
		else
		{
			*(XoJustify *) to->addr = justify;
			to->size = sizeof (XoJustify);
		}
	}
	else
	{
		to->addr = (caddr_t) & justify;
		to->size = sizeof (XoJustify);
	}
	return True;
}

/*
 *----------------------------------------------------------------------
 * BackingStore Converters
 *----------------------------------------------------------------------
 */

struct _backing_store_list
{
	char           *bs_name;
	int             bs_value;
	XrmQuark        bs_quark;
};
static struct _backing_store_list BackingStoreList[] =
{
	"NotUseful", NotUseful, 0,
	"WhenMapped", WhenMapped, 0,
	"Always", Always, 0,
};

static Boolean  HasBackingStoreQuarks;

static void
InitBackingStoreQuarks ()
{
	int             i;
	char            lower[512];

	for (i = 0; i < XtNumber (BackingStoreList); i++)
	{
		XmuCopyISOLatin1Lowered (lower, BackingStoreList[i].bs_name);
		BackingStoreList[i].bs_quark = XrmStringToQuark (lower);
	}
}

String
XoGetStringFromBackingStore (backing_store)
	int             backing_store;
{
	int             i;

	for (i = 0; i < XtNumber (BackingStoreList); i++)
	{
		if (BackingStoreList[i].bs_value == backing_store)
			return BackingStoreList[i].bs_name;
	}
	return ((char *) NULL);
}


int
XoGetBackingStoreFromString (str)
	String          str;
{
	int             i;
	char            lower[512];
	XrmQuark        q;
	int             backing_store = NotUseful;

	if (!HasBackingStoreQuarks)
	{
		InitBackingStoreQuarks ();
		HasBackingStoreQuarks = True;
	}
	XmuCopyISOLatin1Lowered (lower, str);
	q = XrmStringToQuark (lower);
	for (i = 0; i < XtNumber (BackingStoreList); i++)
	{
		if (BackingStoreList[i].bs_quark == q)
		{
			backing_store = BackingStoreList[i].bs_value;
			break;
		}
	}
	return backing_store;
}

static Boolean
StrToBackingStore (display, args, num_args, from, to, converter_data)
	Display        *display;	/* display */
	XrmValue       *args;
	Cardinal       *num_args;
	XrmValue       *from;
	XrmValue       *to;
	XtPointer      *converter_data;
{
	static int      backing_store;

	backing_store = XoGetBackingStoreFromString ((char *) from->addr);
	if (to->addr)
	{
		if (to->size < sizeof (int))
		{
			to->size = sizeof (int);
			return False;
		}
		else
		{
			*(int *) to->addr = backing_store;
			to->size = sizeof (int);
		}
	}
	else
	{
		to->addr = (caddr_t) & backing_store;
		to->size = sizeof (int);
	}
	return True;
}

#ifdef notdef
/*
 *----------------------------------------------------------------------
 * SortedList Converters
 *----------------------------------------------------------------------
 */

static int
str_list_cmp (cp1, cp2)
	char          **cp1, **cp2;
{
	return strcmp (*cp1, *cp2);
}

static Boolean
StrToSortedList (display, args, num_args, from, to, converter_data)
	Display        *display;	/* display */
	XrmValue       *args;
	Cardinal       *num_args;
	XrmValue       *from;
	XrmValue       *to;
	XtPointer      *converter_data;
{
	String          ptr;
	int             cnt;
	int             list_size;
	String         *list;
	char            buf[BUFSIZ];
	char           *bufptr;
	Boolean         status = True;

#define DEFAULT_LIST_SIZE	6
	list_size = DEFAULT_LIST_SIZE;
	list = (String *) XtMalloc (sizeof (String) * list_size);
	cnt = 0;
	for (ptr = (char *) from->addr; ptr && *ptr;)
	{
		/*
		 * Skip any leading space
		 */
		while (*ptr && isspace (*ptr))
			++ptr;
		/*
		 * Now accumulate everything up to a ','
		 */
		for (bufptr = buf; *ptr && *ptr != ','; ptr++)
		{
			*bufptr++ = *ptr;
		}
		*bufptr = '\0';
		if (*ptr)
			++ptr;	/* make sure we skip the ',' */
		/*
		 * Now it's time to trim any additional whitespace. Step
		 * backward until at the beginning of the buffer or a
		 * non-space character is found.
		 */
		while (--bufptr > buf)
		{
			if (!isspace (*bufptr))
				break;
			else
				*bufptr = '\0';
		}
		/*
		 * Add it to the list, but only if it is non-empty
		 */
		if (buf[0])
		{
			if (++cnt >= list_size)
			{
				list_size += DEFAULT_LIST_SIZE;
				list = (String *) XtRealloc ((char *) list,
							     sizeof (String)
							     * list_size);
			}
			list[cnt - 1] = XtNewString (buf);
		}
	}
	/*
	 * Sort the list of strings
	 */
	if (cnt > 1)
		qsort ((char *) list, cnt, sizeof (char *), str_list_cmp);

	/*
	 * Finally, make the list NULL terminated and short.
	 */
	++cnt;
	list = (String *) XtRealloc ((char *) list, sizeof (String) * cnt);
	list[cnt - 1] = NULL;
	if (to->addr)
	{
		if (to->size < sizeof (String *))
		{
			status = False;
		}
		else
		{
			*(String **) to->addr = list;
		}
	}
	else
	{
		static String  *l;

		l = list;
		to->addr = (caddr_t) & l;
	}
	to->size = sizeof (String *);
	return status;
}

static void
StrListDestroy (app, to, data, args, num_args)
	XtAppContext    app;
	XrmValuePtr     to;
	XtPointer       data;
	XrmValuePtr     args;
	Cardinal       *num_args;
{
	String         *list;
	String         *entry;

	list = (String *) to->addr;
	if (!list)
		return;
	for (entry = list; entry && *entry; entry++)
	{
		XtFree ((XtPointer) entry);
	}
	XtFree ((XtPointer) list);
}

#endif					/* notdef */

/*
 *----------------------------------------------------------------------
 * SortedQuarkList Converters
 *----------------------------------------------------------------------
 */

static int
quark_list_cmp (cp1, cp2)
	XrmQuark       *cp1, *cp2;
{
	return *cp1 - *cp2;
}

static Boolean
StrToQuarkSortedList (display, args, num_args, from, to, converter_data)
	Display        *display;	/* display */
	XrmValue       *args;
	Cardinal       *num_args;
	XrmValue       *from;
	XrmValue       *to;
	XtPointer      *converter_data;
{
	String          ptr;
	unsigned int	cnt;
	unsigned int	list_size;
	XrmQuark       *list;
	char            buf[BUFSIZ];
	char           *bufptr;
	Boolean         status = True;

#define DEFAULT_LIST_SIZE	6
	list_size = DEFAULT_LIST_SIZE;
	list = (XrmQuark *) XtMalloc (sizeof (XrmQuark) * list_size);
	cnt = 0;
	for (ptr = (char *) from->addr; ptr && *ptr;)
	{
		/*
		 * Skip any leading space
		 */
		while (*ptr && isspace (*ptr))
			++ptr;
		/*
		 * Now accumulate everything up to a ','
		 */
		for (bufptr = buf; *ptr && *ptr != ','; ptr++)
		{
			*bufptr++ = *ptr;
		}
		*bufptr = '\0';
		if (*ptr)
			++ptr;	/* make sure we skip the ',' */
		/*
		 * Now it's time to trim any additional whitespace. Step
		 * backward until at the beginning of the buffer or a
		 * non-space character is found.
		 */
		while (--bufptr > buf)
		{
			if (!isspace (*bufptr))
				break;
			else
				*bufptr = '\0';
		}
		/*
		 * Add it to the list, but only if it is non-empty
		 */
		if (buf[0])
		{
			if (++cnt >= list_size)
			{
				list_size += DEFAULT_LIST_SIZE;
				list = (XrmQuark *) XtRealloc ((char *) list,
							   sizeof (XrmQuark)
							       * list_size);
			}
			list[cnt - 1] = XrmStringToQuark (buf);
		}
	}
	/*
	 * Sort the list of XrmQuarks
	 */
	if (cnt > 1)
		qsort ((char *) list, (int) cnt, sizeof (char *), quark_list_cmp);

	/*
	 * Finally, make the list NULL terminated and short.
	 */
	++cnt;
	list = (XrmQuark *) XtRealloc ((char *) list, sizeof (XrmQuark) * cnt);
	list[cnt - 1] = (XrmQuark) NULL;
	if (to->addr)
	{
		if (to->size < sizeof (XrmQuark *))
		{
			status = False;
		}
		else
		{
			*(XrmQuark **) to->addr = list;
		}
	}
	else
	{
		static XrmQuark *l;

		l = list;
		to->addr = (caddr_t) & l;
	}
	to->size = sizeof (XrmQuark *);
	return status;
}

#ifdef notdef
static void
StringListDestroy (app, to, data, args, num_args)
	XtAppContext    app;
	XrmValuePtr     to;
	XtPointer       data;
	XrmValuePtr     args;
	Cardinal       *num_args;
{
	String         *list;
	String         *entry;

	list = (String *) to->addr;
	if (!list)
		return;
	for (entry = list; entry && *entry; entry++)
	{
		XtFree ((XtPointer) entry);
	}
	XtFree ((XtPointer) list);
}

#endif					/* notdef */

static Boolean 
AtCvtStringToFontSize (disp, args, nargs, from, to, data)
	Display        *disp;
	XrmValue       *args;
	Cardinal       *nargs;
	XrmValue       *from, *to;
	XtPointer      *data;
{
	static int      result;
	Boolean         Failed = False;

	if (*nargs != 0)
		XtAppErrorMsg (XtDisplayToApplicationContext (disp),
			       "cvtStringToFontSize", "wrong Parameters",
			       "XtToolkitError",
		       "String to AtFontSize conversion takes no arguments",
			       (String *) NULL, (Cardinal *) NULL);

	result = AtFontStringToSize ((char *) from->addr);
	Failed = result < 0;
	if (Failed)
	{
		XtDisplayStringConversionWarning (disp, from->addr, "AtFontSize");
	}
	else
	{
		if (!to->addr)
			to->addr = (caddr_t) & result;
		else if (to->size < sizeof (int))
			Failed = True;
		else
			*(int *) to->addr = result;
		to->size = sizeof (int);
	}
	return !Failed;
}

void 
AtRegisterFontSizeConverter ()
{
	static Boolean  registered = False;

	if (registered == False)
	{
		XtSetTypeConverter (XtRString, XtRFontSize,
				    AtCvtStringToFontSize, NULL, 0,
				    XtCacheAll, NULL);
		registered = True;
	}
}

/**********************************************************************/

static Boolean 
AtCvtStringToFontStyle (disp, args, nargs, from, to, dp)
	Display        *disp;
	XrmValue       *args;
	Cardinal       *nargs;
	XrmValue       *from, *to;
	XtPointer      *dp;
{
	static int      result;
	Boolean         Failed = False;

	if (*nargs != 0)
		XtAppErrorMsg (XtDisplayToApplicationContext (disp),
			       "cvtStringToFontStyle", "wrong Parameters",
			       "XtToolkitError",
		      "String to AtFontStyle conversion takes no arguments",
			       (String *) NULL, (Cardinal *) NULL);

	if (!strcasecmp ((char *) from->addr, "plain"))
		result = AtFontPLAIN;
	else if (!strcasecmp ((char *) from->addr, "bold"))
		result = AtFontBOLD;
	else if (!strcasecmp ((char *) from->addr, "italic"))
		result = AtFontITALIC;
	else if (!strcasecmp ((char *) from->addr, "bolditalic"))
		result = AtFontBOLDITALIC;
	else
	{
		Failed = True;
	}

	if (Failed)
	{
		XtDisplayStringConversionWarning (disp, from->addr, "AtFontStyle");
	}
	else
	{
		if (!to->addr)
			to->addr = (caddr_t) & result;
		else if (to->size < sizeof (int))
			Failed = True;
		else
			*(int *) to->addr = result;
		to->size = sizeof (int);
	}
	return !Failed;
}

void 
AtRegisterFontStyleConverter ()
{
	static Boolean  registered = False;

	if (registered == False)
	{
		XtSetTypeConverter (XtRString, XtRFontStyle,
				    AtCvtStringToFontStyle, NULL, 0,
				    XtCacheAll, NULL);
		registered = True;
	}
}

/**********************************************************************/

static Boolean 
AtCvtStringToJustify (dpy, args, nargs, from, to, dp)
	Display        *dpy;
	XrmValue       *args;
	Cardinal       *nargs;
	XrmValue       *from, *to;
	XtPointer      *dp;
{
	static int      result;
	Boolean         Failed = False;
	char           *inp = (char *) from->addr;

	if (*nargs != 0)
		XtAppErrorMsg (XtDisplayToApplicationContext (dpy),
			       "cvtStringToAtjustify", "wrong Parameters",
			       "XtToolkitError",
			"String to AtJustify conversion takes no arguments",
			       (String *) NULL, (Cardinal *) NULL);

	if (!strncasecmp (inp, "justify", 7))
		inp += 7;
	if (!strncasecmp (inp, "atjustify", 9))
		inp += 9;

	if (!strcasecmp (inp, "left"))
		result = AtTextJUSTIFY_LEFT;
	else if (!strcasecmp (inp, "top"))
		result = AtTextJUSTIFY_TOP;
	else if (!strcasecmp (inp, "center"))
		result = AtTextJUSTIFY_CENTER;
	else if (!strcasecmp (inp, "right"))
		result = AtTextJUSTIFY_RIGHT;
	else if (!strcasecmp (inp, "bottom"))
		result = AtTextJUSTIFY_BOTTOM;
	else
	{
		Failed = True;
	}

	if (Failed)
	{
		XtDisplayStringConversionWarning (dpy, from->addr, "AtJustify");
	}
	else
	{
		if (!to->addr)
			to->addr = (caddr_t) & result;
		else if (to->size < sizeof (int))
			Failed = True;
		else
			*(int *) to->addr = result;
		to->size = sizeof (int);
	}
	return !Failed;
}

/*
 *----------------------------------------------------------------------
 * Register any and all converters
 *----------------------------------------------------------------------
 */

void
_XoRegisterConverters ()
{
	static Boolean  done;
	static XtConvertArgRec objCvtArgs[] =
	{
		{XtWidgetBaseOffset,
			(XtPointer) XtOffsetOf (CoreRec, core.colormap),
		sizeof (Colormap)},
		{XtWidgetBaseOffset,
			(XtPointer) XtOffsetOf (CoreRec, core.screen),
		sizeof (Screen *)},
		{XtProcedureArg,
			(XtPointer) _XoGetVisualCvter,
		sizeof (Visual *)},
	};

	if (!done)
	{
		done = True;
		XtSetTypeConverter (XtRString, XtRObjectClass, StrToObjectClass,
				    (XtConvertArgList) objCvtArgs,
				    XtNumber (objCvtArgs), XtCacheAll,
				    (XtDestructor) NULL);
		XtSetTypeConverter (XtRString, XtRGravity, StrToGravity,
				    (XtConvertArgList) NULL, 0, XtCacheAll,
				    (XtDestructor) NULL);
		XtSetTypeConverter (XtRString, XtRJustify, StrToJustify,
				    (XtConvertArgList) NULL, 0, XtCacheAll,
				    (XtDestructor) NULL);
		XtSetTypeConverter (XtRString, XtRDouble, StrToDouble,
				    (XtConvertArgList) NULL, 0, XtCacheAll,
				    (XtDestructor) NULL);
#ifdef notdef
		XtSetTypeConverter (XtRString, XtRSortedStrList,
				    StrToSortedList,
				    (XtConvertArgList) NULL, 0, XtCacheAll,
				    (XtDestructor) StrListDestroy);
#endif
		XtSetTypeConverter (XtRString, XtRSortedQuarkList,
				    StrToQuarkSortedList,
				    (XtConvertArgList) NULL, 0, XtCacheAll,
				    (XtDestructor) NULL);
		XtSetTypeConverter (XtRString, XtRBackingStore,
				    StrToBackingStore,
				    (XtConvertArgList) NULL, 0, XtCacheAll,
				    (XtDestructor) NULL);
		XtSetTypeConverter (XtRString, XtROrientation,
				    StrToOrientation,
				    (XtConvertArgList) NULL, 0, XtCacheAll,
				    (XtDestructor) NULL);
		XtSetTypeConverter (XtRString, XtRPointing,
				    StrToPointing,
				    (XtConvertArgList) NULL, 0, XtCacheAll,
				    (XtDestructor) NULL);
		XtSetTypeConverter (XtRString, XtRFontSize,
				    AtCvtStringToFontSize, NULL, 0,
				    XtCacheAll, NULL);
		XtSetTypeConverter (XtRString, XtRFontStyle,
				    AtCvtStringToFontStyle, NULL, 0,
				    XtCacheAll, NULL);
		XtSetTypeConverter (XtRString, XtRAtJustify,
				    AtCvtStringToJustify, NULL, 0,
				    XtCacheAll, NULL);
		XoRegisterShadingConverter ();
	}
}
