/*
 * Column:	Lay children out in a vertical column with each child
 *		the width of the largest and it's natural height
 *
 * Column.c,v 2.3 1992/08/11 00:11:31 pete Exp
 * Column.c,v
 * Revision 2.3  1992/08/11  00:11:31  pete
 * Added resize_flag to accomodate other geometry handlers.
 *
 * Revision 2.2  1992/07/22  15:55:41  pete
 * Worked on getting the geometry requests to be correct.  Put most of
 * the size calculations into CalcWidth and CalcHeight.  Made sure they
 * are consistent with the layout done in DoLayout.
 *
 * Revision 2.1  1992/07/11  19:57:44  pete
 * QueryGeometry was using uninitialized fields from proposed.
 *
 * Revision 2.0  1992/04/23  02:50:39  ware
 * First public release.
 *
 * Revision 1.8  1992/04/23  02:18:31  ware
 * Added several classes.  Worked on geometry management
 *
 * Revision 1.7  1992/02/27  14:30:29  ware
 * Compiled with GCC 2.0 and very strict checks.  Fixed Warnings
 *
 * Revision 1.6  1992/02/24  01:19:31  ware
 * Changed scale to user_height.  This allows the user to interactively
 * set the height of a widget.  When this is non-zero, this is used as
 * the preferred height of the widget instead of querying for the value.
 *
 * Revision 1.5  1992/02/20  15:11:09  ware
 * Applied new indentation
 *
 * Revision 1.4  1992/02/04  21:22:46  pete
 * Release 44
 *
 * Revision 1.3  1991/11/30  15:51:19  pete
 * Cleaned up some nitpicky compile time warnings.
 *
 * Revision 1.2  1991/09/12  09:44:41  pete
 * Big changes.  Moved all the geometry management functions from Vpane.c
 * into this widget.  Also changed so it is a child of XoConsBase so the
 * constraints can be associated with the Vpane.
 *
 * Revision 1.1  91/08/30  17:40:54  pete
 * Initial revision
 *
 */

#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xo/ColumnP.h>
#include <X11/Xo/dbug.h>


#include <X11/Xo/ColumnRec.h>

XoProto (static void, DoLayout, (Widget gw));
XoProto (static void, SizeLayout, (Widget gw, Dimension *width_ret, Dimension *height_ret));
XoProto (static int, CalculateStretch, (Widget gw));
XoProto (static int, CalculateShrink, (Widget gw));
XoProto (static void, AskNewWidth, (XoColumnWidget w, int width, int height, int query));
XoProto (static void, CalcWidth, (Widget gw, int new_height, Dimension *width_ret));
XoProto (static void, CalcHeight, (Widget gw, int new_width, Dimension *height_ret));

/*
 *----------------------------------------------------------------------
 * Core Class Methods
 *----------------------------------------------------------------------
 */

/*
 * Resize -	Respond to a change in our size.  DoLayout() does the
 *		actual work.
 */
static void
Resize (gw)
	Widget          gw;
{
	XoColumnWidget  w = (XoColumnWidget) gw;

	DBUG_ENTER ("Column.Resize");
	DoLayout (gw);
	w->cons_base.resize_flag = True;
	DBUG_VOID_RETURN;
}

static XtGeometryResult
QueryGeometry (gw, proposed, desired)
	Widget          gw;
	XtWidgetGeometry *proposed;
	XtWidgetGeometry *desired;
{
	XoColumnWidget  w = (XoColumnWidget) gw;
	XtGeometryResult result;
	Dimension       width, height;

	DBUG_ENTER ("Column.QueryGeometry");
	SizeLayout (gw, &width, &height);
	DBUG_PRINT ("layout", ("----------------In %s -----------", XoName (gw)));
	/*
	 * I'm just being asked my ideal size
	 */
	if (!proposed->request_mode)
	{
		DBUG_PRINT ("layout", ("Responding with ideal size"));
		if (width == w->core.width && height == w->core.height)
			result = XtGeometryNo;
		else
		{
			desired->width = width;
			desired->height = height;
			desired->request_mode = CWWidth | CWHeight;
			result = XtGeometryAlmost;
		}
	}
	else if (XoGeoDimFlags (proposed) == CWWidth)
	{
		DBUG_PRINT ("layout", ("Calculating best height"));
		CalcHeight (gw, (int) proposed->width, &desired->height);
		desired->request_mode = CWHeight;
		if (desired->height == gw->core.height)
			result = XtGeometryYes;
		else
			result = XtGeometryAlmost;
	}
	else if (XoGeoDimFlags (proposed) == CWHeight)
	{
		DBUG_PRINT ("layout", ("Calculating best width"));
		CalcWidth (gw, (int) proposed->height, &desired->width);
		desired->request_mode = CWWidth;
		if (desired->width == gw->core.width)
			result = XtGeometryYes;
		else
			result = XtGeometryAlmost;
	}
	else
	{
		DBUG_PRINT ("layout", ("Being told my dimension"));
		if (proposed->width == width && proposed->height == height)
			result = XtGeometryYes;
		else
		{
			result = XtGeometryNo;
		}
	}
	DBUG_PRINT ("layout", ("------------- done ------------"));
	DBUG_RETURN (result);
}

/*
 *----------------------------------------------------------------------
 * Composite Class Methods
 *----------------------------------------------------------------------
 */

/*
 * GeometryManager - Responds to request for changes in size of position
 *		of children.  (See pp 228 of Asente & Swick).
 */

static XtGeometryResult
GeometryManager (gw, request, geo_ret)
	Widget          gw;
	XtWidgetGeometry *request;
	XtWidgetGeometry *geo_ret;
{
	XoColumnWidget  w;		/* this column widget */
	Dimension       new_width;	/* our new width */
	Dimension       new_height;	/* our new height */
	Dimension       old_width;	/* child's old width */
	XtGeometryResult result;	/* parents answer */

	w = (XoColumnWidget) XtParent (gw);
	if (!XoGeoDimFlags (request))
		return XtGeometryNo;
	if (XoGeoDimFlags (request) == CWWidth)
	{
		/*
		 * 1. We set the child's width to this new width.
		 */
		old_width = gw->core.width;
		gw->core.width = request->width;
		/*
		 * 2. Query all the children to determine our "optimum" size
		 * (i.e. SizeLayout).
		 */
		SizeLayout ((Widget) w, &new_width, &new_height);
		gw->core.width = old_width;
		AskNewWidth (w, (int) new_width, (int) new_height,
			     (int) (request->request_mode & XtCWQueryOnly));
		if (gw->core.width == request->width)
			result = XtGeometryDone;
		else
			result = XtGeometryNo;
	}
	else if (XoGeoDimFlags (request) == CWHeight)
	{
		/*
		 * Sure, we'll let it change it's height.
		 * FIX: Should we scale the height request or set the
		 *	scale to zero?
		 */
		if (!(request->request_mode & XtCWQueryOnly))
		{
			XtResizeWidget (gw, gw->core.width,
					request->height,
					gw->core.border_width);
			DoLayout ((Widget) w);
			result = XtGeometryDone;
		}
		else
		{
			result = XtGeometryYes;
		}
	}
	else
		result = XtGeometryNo;
	return result;
}


static void
ChangeManaged (gw)
	Widget          gw;
{
	XoColumnWidget  w = (XoColumnWidget) gw;
	Dimension       width;
	Dimension       height;

	DBUG_ENTER ("Column.ChangeManaged");
	if (w->cons_base.do_layout)
	{
		SizeLayout (gw, &width, &height);
		DBUG_PRINT ("size", ("%s: requested width = %d, height = %d",
				       XoName (gw), width, height));
		AskNewWidth (w, (int) width, (int) height, False);
	}
	DBUG_VOID_RETURN;
}

/*
 *----------------------------------------------------------------------
 * Constraint Class Methods
 *----------------------------------------------------------------------
 */

static void
ConsInitialize (request, new, arglist, num_args)
	Widget          request;	/* as first created */
	Widget          new;		/* after other parent classes */
	ArgList         arglist;	/* list of arguments */
	Cardinal       *num_args;	/* how many */
{
	XoColumnConstraint constraints;
	constraints = (XoColumnConstraint) new->core.constraints;
	constraints->column.skip = False;
	constraints->column.user_height = 0; /* sentinal value */
}

/*
 *----------------------------------------------------------------------
 * ConsBase Class Methods
 *----------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------
 * Column Class Methods
 *----------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------
 * Private Utilities
 *----------------------------------------------------------------------
 */

/*
 * Layout each of the children.  Children are all the same width and
 * arranged vertically.  Each child is asked its optimum height
 * given our current width.  This value is then multiplied by whatever
 * the associated scale factor set for this child (the scale factor can
 * be used by subclasses to allow interactive changing of the widget sizes).
 * If there is too much room for the children, then
 * a "stretchability" factor is calculated and each child height is
 * increased according to it's "stretch" factor.  If there is not enough
 * room, a similar effect takes place except the "shrink" factor is used.
 */

static void
DoLayout (gw)
	Widget          gw;
{
	XoColumnWidget  w = (XoColumnWidget) gw;
	int             i;
	Widget          child;
	Widget          last;		/* the last managed widget */
	Position        y;
	Position        x;
	Boolean         first;
	XtWidgetGeometry geo_request;
	XtWidgetGeometry geo_return;
	XoColumnConstraint constraints;

	DBUG_ENTER ("Column.DoLayout");
	DBUG_PRINT ("layout", ("-------------------------------------"));
	first = True;
	x = y = 0;
	for (i = 0; i < w->composite.num_children; i++)
	{
		child = w->composite.children[i];
		if (!XtIsManaged (child) || SKIPCHILD (child))
			continue;
		/*
		 * From Asente & Swick, pg 240: "If intended is not NULL, a
		 * return of XtGeometryYes means that the child either does
		 * not have a preferred geometry or would be happy with the
		 * geometry described in the inteneded parameter.  A return
		 * of XtGeometryNo means that the proposed changes are not
		 * acceptable to the child and that the child prefers to
		 * retain its current geometry.  A return of XtGeometryAlmost
		 * means that some change is acceptable, and the caller must
		 * examine the contents of preferred_return to find out what
		 * the preferred change is.  If all the fields that were
		 * flagged in intended have the same values in
		 * preferred_return, XtGeometryAlmost means that changes in
		 * the proposed geometry are acceptable but that the child
		 * wants some additional changes made to other fields.  If
		 * the fields have different values, it means that the child
		 * wants these fields chanaged but not to the proposed
		 * values.  In both cases the flagged fields in
		 * preferred_return hold the child's preferred values."
		 * 
		 * What this means in this section of code is if the child says
		 * yes, that's great.  If the child returns No, that's tough
		 * because we have to make the chanaged anyway.  If the child
		 * returns Almost, and it is willing to change it's height,
		 * we'll do that.  Otherwise it is set to our width and the
		 * old height.
		 */
		geo_request.request_mode = CWWidth;
		geo_request.width = w->core.width;
		constraints = (XoColumnConstraint) child->core.constraints;
		if (XtQueryGeometry (child, &geo_request, &geo_return) == XtGeometryAlmost)
		{
			if (geo_return.request_mode & CWWidth)
			{
				DBUG_PRINT ("layout", ("ignoring %s's return", XoName (child)));
				geo_return.width = w->core.width;
				geo_return.height = child->core.height;
			}
			else
			{
				DBUG_PRINT ("layout", ("using my width instead of childs, using %s's height", XoName (child)));
				geo_return.width = w->core.width;
				if (constraints->column.user_height)
					geo_return.height = constraints->column.user_height;
			}
		}
		else
		{
			DBUG_PRINT ("layout", ("%s's said yes or no, just using calculated values", XoName (child)));
			geo_return.width = w->core.width;
			geo_return.height = child->core.height;
		}
		if (first)
		{
			y += geo_return.border_width;
			first = False;
		}
		else
		{
			y += w->column.vspace;
		}
		XoConfWidget (child, x, y, 
			      (unsigned int) geo_return.width - 2 * geo_return.border_width,
			      (unsigned int) geo_return.height,
			      (unsigned int) geo_return.border_width);
		y += child->core.height + child->core.border_width;
		last = child;
	}

	/*
	 * Now we need to figure out what to do with either extra space
	 * or not enough space.
	 */

	if (y < (int) w->core.height)
	{
		Dimension       stretch;
		float           stretchable;
		float           slop;
		int             total_stretch;

		total_stretch = CalculateStretch (gw);
		if (total_stretch <= 0)
		{
			stretchable = 0.0;
		}
		else
		{
			stretchable = (w->core.height - y) / (double) total_stretch;
		}
		y = 0;
		first = True;
		slop = 0;
		for (i = 0; i < w->composite.num_children; i++)
		{
			child = w->composite.children[i];
			if (!XtIsManaged (child) || SKIPCHILD (child))
				continue;
			if (first)
			{
				y = child->core.border_width;
				first = False;
			}
			else
			{
				y += w->column.vspace;
			}
			constraints = (XoColumnConstraint) child->core.constraints;
			stretch = stretchable * constraints->column.stretch
				* child->core.height + slop;
			DBUG_PRINT ("layout", ("stretching %s by %d", XoName (child), stretch));
			slop = stretch
				- (stretchable * constraints->column.stretch
				   * child->core.height + slop);
			XoConfWidget (child, x, y,
				      (unsigned int) child->core.width,
				      (unsigned int) child->core.height + stretch,
				      (unsigned int) child->core.border_width);
			y += child->core.height + child->core.border_width;
		}
	}
	else if (y > (int) w->core.height)
	{
		Dimension       shrink;
		float           shrinkable;
		float           slop;
		int             total_shrink;

		total_shrink = CalculateShrink (gw);
		if (total_shrink <= 0)
		{
			shrinkable = 0.0;
		}
		else
		{
			shrinkable = (y - w->core.height) / (double) total_shrink;
		}
		y = 0;
		first = True;
		slop = 0;
		for (i = 0; i < w->composite.num_children; i++)
		{
			child = w->composite.children[i];
			if (!XtIsManaged (child) || SKIPCHILD (child))
				continue;
			if (first)
			{
				y = child->core.border_width;
				first = False;
			}
			else
			{
				y += w->column.vspace;
			}
			constraints = (XoColumnConstraint) child->core.constraints;
			shrink = shrinkable * constraints->column.shrink * child->core.height + slop;
			if (shrink >= child->core.height)
				shrink = child->core.height - 1;
			slop = shrink - (shrinkable * constraints->column.shrink * child->core.height + slop);
			DBUG_PRINT ("layout", ("shrinking %s by %d", XoName (child), shrink));
			XoConfWidget (child, x, y,
				      (unsigned int) child->core.width,
				      (unsigned int) child->core.height - shrink,
				      (unsigned int) child->core.border_width);
			y += child->core.height + child->core.border_width;
		}
	}
	DBUG_VOID_RETURN;
}

/*
 * Return the amount of stretchiness in all the child widgets.
 */

static int
CalculateStretch (gw)
	Widget          gw;
{
	int             i;
	Widget          child;
	XoColumnConstraint cons;
	XoColumnWidget  w = (XoColumnWidget) gw;
	int             stretch;

	stretch = 0;
	for (i = 0; i < w->composite.num_children; i++)
	{
		child = w->composite.children[i];
		if (!XtIsManaged (child) || SKIPCHILD (child))
			continue;
		cons = (XoColumnConstraint) child->core.constraints;
		stretch += cons->column.stretch * child->core.height;
	}
	return stretch;
}

static int
CalculateShrink (gw)
	Widget          gw;
{
	int             i;
	Widget          child;
	XoColumnConstraint cons;
	XoColumnWidget  w = (XoColumnWidget) gw;
	int             shrink;

	shrink = 0;
	for (i = 0; i < w->composite.num_children; i++)
	{
		child = w->composite.children[i];
		if (!XtIsManaged (child) || SKIPCHILD (child))
			continue;
		cons = (XoColumnConstraint) child->core.constraints;
		shrink += cons->column.shrink * child->core.height;
	}
	return shrink;
}

/*
 * Calculate our desired size based by querying each of the managed
 * children and determining the desired height and width.
 */

static void
SizeLayout (gw, width_ret, height_ret)
	Widget          gw;
	Dimension      *width_ret;
	Dimension      *height_ret;
{
	Dimension	width;
	Dimension	height;

	CalcWidth (gw, (int) gw->core.height, &width);
	CalcHeight (gw, (int) width, &height);
	if (width_ret)
		*width_ret = width;
	if (height_ret)
		*height_ret = height;
}

/*
 * Ask our parent for a new width and try several times
 */

static void
AskNewWidth (w, width, height, query)
	XoColumnWidget  w;
	int             width;
	int             height;
	int             query;
{
	XtWidgetGeometry request;	/* what we want */
	XtWidgetGeometry reply;		/* what the parent says */
	XtGeometryResult result;
	int             i;

	DBUG_ENTER ("Column.AskNewWidth");
	DBUG_PRINT ("size", ("new size [%d,%d], old size [%d,%d]",
			     width, height, w->core.width, w->core.height));

	if (w->core.width == width)
		return;
	/*
	 * Set this flag to True in case to avoid unnecessary resize
	 */
	w->cons_base.resize_flag = True;
	for (i = 0; i < XO_GEO_NUMREQUEST; i++)
	{
		request.request_mode = CWWidth;
		if (height != w->core.height)
			request.request_mode |= CWHeight;
		request.width = width;
		request.height = height;
		
		/*
		 * Set a flag so we know if the Resize Method has
		 * been called.  It is set to True by Resize().  If
		 * it is still False, then call Resize() by hand.
		 */
		if (query)
		{
			request.request_mode |= XtCWQueryOnly;
		}
		else
		{
			w->cons_base.resize_flag = False;
		}
		result = XtMakeGeometryRequest ((Widget) w, &request, &reply);
		if (result != XtGeometryAlmost)
			break;
		width = reply.width;
		height = reply.height;
	}
	if (!w->cons_base.resize_flag)
	{
		ThisClass(w).core_class.resize ((Widget) w);
	}
	DBUG_PRINT ("size", ("final size [%d,%d], result = %s", width, height,
			     XoGeometryResultString (result)));
		
	DBUG_VOID_RETURN;
}

static void
CalcWidth (gw, new_height, width_ret)
	Widget          gw;
	int             new_height;
	Dimension      *width_ret;
{
	XoColumnWidget	w = (XoColumnWidget) gw;
	int             i;
	Widget          child;
	Dimension       max_width;
	XtWidgetGeometry geo_ret;

	DBUG_ENTER ("Column.CalcWidth");
	/*
	 * Query the children for their preferred dimensions. Find the
	 * maximum width and use that for all the children
	 */

	max_width = 0;
	for (i = 0; i < w->composite.num_children; i++)
	{
		child = w->composite.children[i];
		if (!XtIsManaged (child) || SKIPCHILD (child))
			continue;
		/*
		 * From Asente & Swick, p 239: "When intended is NULL, a
		 * return of XtGeometryYes means that the child does not have
		 * a preferred geometry and that any geometry change is
		 * acceptable.  A return of XtGeometryAlmost means that
		 * geo_ret contains the child's preferred geometry and that
		 * at least one of the child's geometry fields current
		 * differs from its preferred geometry.  A return of
		 * XtGeometryNo means that preferred_return contains the
		 * child's preferred geometry and that this is the same as
		 * its current geometry."
		 */
		(void) XtQueryGeometry (child, (XtWidgetGeometry *) NULL,
					&geo_ret);
		geo_ret.width += 2 * geo_ret.border_width;
		if (geo_ret.width > max_width)
			max_width = geo_ret.width;
	}
	if (max_width == 0)
		max_width = 1;
	*width_ret = max_width;
	DBUG_PRINT ("size", ("%s's width = %d", XoName (gw), max_width));
	DBUG_VOID_RETURN;
}

static void
CalcHeight (gw, new_width, height_ret)
	Widget          gw;
	int             new_width;
	Dimension      *height_ret;
{
	int             i;
	XoColumnWidget  w = (XoColumnWidget) gw;
	XoColumnConstraint	cons;
	Widget          child;
	XtWidgetGeometry geo_request;
	XtWidgetGeometry geo_return;
	Dimension       height;
	Boolean		first;		/* if this is the first widget */

	DBUG_ENTER ("Column.CalcHeight");
	first = True;
	height = 0;
	for (i = 0; i < w->composite.num_children; i++)
	{
		child = w->composite.children[i];
		if (!XtIsManaged (child) || SKIPCHILD (child))
			continue;
		cons = (XoColumnConstraint) child->core.constraints;
		geo_request.request_mode = CWWidth;
		geo_request.width = new_width - 2 * child->core.border_width;
		(void) XtQueryGeometry (child, &geo_request, &geo_return);
		if (first)
		{
			first = False;
			height += geo_return.border_width;
		}
		else
		{
			height += w->column.vspace;
		}
		if (cons->column.user_height)
			geo_return.height = cons->column.user_height;
		height += geo_return.height + geo_return.border_width;
	}
	if (height_ret)
		*height_ret = height;
	DBUG_PRINT ("size", ("%s's height = %d", XoName (gw), height));
	DBUG_VOID_RETURN;
}
