/**********************************************************************/
/* window.c                                                           */
/*                                                                    */
/* Window handling routines                                           */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          */
/**********************************************************************/
#include <stdio.h>
#include <math.h>
#include <fmclient.h>
#include <gl/gl.h>
#include <gl/device.h>
#include <errno.h>
#define IRIS4D 1
#include "geo.h"
#include "walk.h"
#include "vcr.h"

/* Mouse button down status */
int WindLeftDown = 0;
int WindMiddleDown = 0;
int WindRightDown = 0;

/**********************************************************************/
/* Reset mouse button down status                                     */
/**********************************************************************/
void ResetMouseDown()
{ WindLeftDown = WindMiddleDown = WindRightDown = 0; }

/**********************************************************************/
/* Handle main window events. Keep track of mouse movement and        */
/* buttons pressed                                                    */
/**********************************************************************/
void DoEventsWind(dev, val)
  long dev;
  short val;
{
  winset(WindWid);
  switch(dev) {

  case RIGHTMOUSE : /* On down mouse down */
    if (val) {
      if (!WindLeftDown && !WindMiddleDown) {
	BuildMainMenu();                      /* Menu on right down */
	dopup(Menu);
      } else {
	WindRightDown = 1;
	view.om.x = getvaluator(MOUSEX) - WindXorigin;
	view.om.y = getvaluator(MOUSEY) - WindYorigin;
      }      
    } else 
      WindRightDown = 0;
    break;

  case LEFTMOUSE :                           
    if (val) {
      WindLeftDown = 1;
      view.om.x = getvaluator(MOUSEX) - WindXorigin;
      view.om.y = getvaluator(MOUSEY) - WindYorigin;
    } else 
      WindLeftDown = 0;
    break;
    
  case MIDDLEMOUSE :                         
    if (val) {
      WindMiddleDown = 1;
      view.om.x = getvaluator(MOUSEX) - WindXorigin;
      view.om.x = getvaluator(MOUSEY) - WindYorigin;
    } else
      WindMiddleDown = 0;
    break;

  case REDRAW:
    if (Option.debug) printf("Redrawing window\n");
    reshapeviewport();
    getorigin((int)&WindXorigin, (int)&WindYorigin);
    WindLeftDown = WindRightDown = WindMiddleDown = 0;
    reshapeviewport();
    Draw_Scene();
    break;
  }
}

/**********************************************************************/
/* Do events in main window                                           */
/**********************************************************************/
void DoWind()
{
  int moved = 0;
  winset(WindWid);

  if (Option.debug)
    if (WindMiddleDown || WindRightDown || WindLeftDown) {
      printf("Buttons pressed are: %d %d %d\n", WindLeftDown, 
	     WindMiddleDown, WindRightDown);
    }
  /* Current x,y position */
  view.m.x = getvaluator(MOUSEX) - WindXorigin;
  view.m.y = getvaluator(MOUSEY) - WindYorigin;

  if (WindLeftDown && !WindMiddleDown && WindRightDown) {
    Option.mode = ORIENT_ZOOM;      /* Right + left = Zoom */
    moved = 1;
  }
  else if (WindLeftDown && WindMiddleDown && WindRightDown) {
    Option.mode = ORIENT_SCALE;     /* All buttons = Scale */
    moved = 1;
  }
  else if (WindLeftDown && !WindMiddleDown && !WindRightDown) {
    Option.mode = ORIENT_ROT;       /* Left button = Rotate in x,z */
    moved = 1;
  }
  else if (!WindLeftDown && WindMiddleDown && !WindRightDown) {
    Option.mode = ORIENT_SPIN;     /* Middle button = Rotate in y */
    moved = 1;
  }
  else if (WindLeftDown && WindMiddleDown && !WindRightDown) {
    Option.mode = ORIENT_TRANSL;    /* Not middle button = translate */
    moved = 1;
  }

  /* Update scene depending on movement */
  if (moved) {
    if (Option.debug) 
      printf("Moved with movement mode: %d\n", Option.mode);
    Update_Scene();
    moved = 0;
  }

  /* Store current as old position */
  view.om.x = view.m.x;
  view.om.y = view.m.y;
}
  
/**********************************************************************/
/* Handle panel events                                                */
/**********************************************************************/
void DoEventsPanel(dev, val)
     long dev;
     short val;
{ printf("Panel not set up yet\n"); }

/**********************************************************************/
/* Handle all program events                                          */
/**********************************************************************/
void HandleEvents()
{
  int done = 0;
  long dev;
  short val;
  short wind_infocus = 0;
  short panel_infocus = 0;
  short help_infocus = 0;

  while(!done) {
    if ((!WindLeftDown && !WindRightDown && !WindMiddleDown) || qtest()) {
      switch(dev = qread(&val)) {
      
      case ESCKEY:
	if (val) break; 
	done = TRUE;
	
      case INPUTCHANGE:  /* Switch to appropriate window */
	wind_infocus = (val == WindWid);
	panel_infocus = (val == PanelWid);
	help_infocus = (val == HelpWid);
	break;
	
      case REDRAW: /* Redraw appropriate window */
	if (val == WindWid)
	  DoEventsWind(dev, val);
	else if (val == HelpWid) DoEventsHelp(dev, val); 
	else if (val == PanelWid) DoEventsPanel(dev, val);
	break;
	
      default: /* Send device id and device value to approp. window */
	if (wind_infocus)
	  DoEventsWind(dev, val);
	else if (help_infocus)
	  DoEventsHelp(dev,val);
	else if (System.UsePanel && panel_infocus) 
	  DoEventsPanel(dev, val);
      }
    }
    
    /* Handle geometry changes */
    if (wind_infocus) DoWind(); 
  }
  
  if (System.UseVCR)           /* Stop the video */
    vcr_end_video();
}

/**********************************************************************/
/* Initialize fonts 5,8 and 10 pt                                     */
/**********************************************************************/
void InitFonts()
{
  fminit();
  
  FontScreen = fmfindfont("Screen");
  FontScreen5 = fmscalefont(FontScreen, 5.0);
  FontScreen8 = fmscalefont(FontScreen, 8.0);
  FontScreen10 = fmscalefont(FontScreen, 10.0);
  
  fmgetfontinfo(FontScreen, &FontScreenInfo);
  fmgetfontinfo(FontScreen5, &FontScreen5Info);
  fmgetfontinfo(FontScreen8, &FontScreen8Info);
  fmgetfontinfo(FontScreen10, &FontScreen10Info); 
}

float clut_gamma = 2.20;
short gamma_rgb[256];
/**********************************************************************/
/* Load colour map for gamma correction */
/**********************************************************************/
void SetGamma()
{
  int i;
  
  /* Loop through all 256 values a color lookup table would have */
  for (i = 0; i < 256; i++) {
    gamma_rgb[i] = 
      (int)(255.0 * pow((double) i / 255.0, 1.0 / clut_gamma));
    /* if (i % 16 == 0)
       printf("\n");
       printf("%d ", gamma_rgb[i]); */
  }
  gammaramp(gamma_rgb, gamma_rgb, gamma_rgb);
}

/*********************************************************************/
/* Open a new geometry window                                        */
/*********************************************************************/
int OpenNewWindow(Wname, xsize, ysize, xorigin, yorigin) 
     char *Wname;
     int xsize, ysize, xorigin, yorigin;
{
  int newWid;

  if (System.UseVCR) {            /* Use VCR */
    vcr_setup_window(FALSE);

  } else if (System.FullScreen) {
    prefsize(XMAXSCREEN, YMAXSCREEN);
    /* prefposition(0,0,XMAXSCREEN-1,YMAXSCREEN-1); */
    noborder();

  } else {
    if (System.UsePrefsize)       /* Check to use prefered size or */
      prefsize(xsize, ysize);     /* position */
    if (System.UsePrefpos)
      prefposition(xorigin, xorigin + xsize -1,
		   yorigin, yorigin + ysize -1);
  }

  if ((newWid = winopen(Wname)) == -1) {
    fprintf(stderr, "No more windows available\n");
    return -1;
  }

  if (System.UseDoubleBuf) 
    doublebuffer();            /* Use doublebuffering */
  if (System.UseRGB) {
    RGBmode();                 /* Use RGB mode */    
    if (System.gamma)          /* Set gamma correction */
      SetGamma();
  }
  mmode(MVIEWING);
  gconfig();
  getorigin(&xorigin, &yorigin);
  getsize(&xsize, &ysize);

  if (System.UseZBuf)           /* Use z-buffer */
    zbuffer(TRUE);              
  glcompat(GLC_ZRANGEMAP, 0);   /* Set map */
  System.zfar =                 /* Find max z for zbuffering */
    getgdesc(GD_ZMAX);

  /* if (System.UseAccBuf) {   Using accumulation buffer (not used) 
     subpixel(TRUE);
     InitAccBuff(newWid, 64);
     InitKernals();
     SetKernalId(); 
     } */

  if (System.UseVCR) {          /* Set view parameters */
    vcr_start_video();
  }
  return newWid;
}

/**********************************************************************/
/* Initialize graphics                                                */
/**********************************************************************/
void InitGraphics()
{
  /* Check for z and bit-planes */
  if (System.UseDoubleBuf) 
    if  (getgdesc(GD_BITS_NORM_DBL_RED) == 0) {
      fprintf(stderr, "Double buffered RGB not available on this machine\n");
      exit(1); 
    }
  if (System.UseZBuf)
    if (getgdesc(GD_BITS_NORM_ZBUFFER) == 0) {
      fprintf(stderr, "Z buffer not available on this machine\n");
      exit(1);
    }
  if (System.UseAccBuf) 
    if (getgdesc(GD_BITS_ACBUF) == 0) {
      fprintf(stderr, "Accumlation buffer not available on this machine\n");
      exit(1);
    }
  if (System.UseVCR) {
    vcr_set_mode(System.VCRmode);
  }
  
  /* Get/set screen size/aspect */
  System.xscrnsize = getgdesc(GD_XPMAX);  
  if (System.xscrnsize == 1280) {
    keepaspect(5, 4);
    System.scrnaspect = 1.25;
  } else if (System.xscrnsize == 1024) {
    keepaspect(4, 3);
    System.scrnaspect = 1.34;
  } else {
    fprintf(stderr, "Something's extremely wrong:  ");
    fprintf(stderr, "Screen size in x=%d\n", System.xscrnsize);
    exit(-1) ;
  }

  WindWid = OpenNewWindow(WindowName, WindXsize, WindYsize, 
			  WindXorigin, WindYorigin);
  
  System.GraphInitialized = 1;
}

/**********************************************************************/
/* Initialize queue devices and reset queue */
/**********************************************************************/
void InitQueue()
{
  qdevice(ESCKEY);
  qdevice(LEFTMOUSE);
  qdevice(MIDDLEMOUSE);
  qdevice(RIGHTMOUSE); 
  qdevice(INPUTCHANGE); 
  qdevice(REDRAW);
  qreset();
}

/**********************************************************************/
/* Exit the program */
/**********************************************************************/
void DoExit(rc)
    int rc;
{
  if (System.GraphInitialized) 
    gexit();
  exit(rc);
}

