/**********************************************************************/
/* hc.c : hemicube form-factor computation.                           */
/*                                                                    */
/* Routines modified from those given by Eric Chen (Graphics Gems II) */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          */
/**********************************************************************/
#include <stdio.h>
#include <math.h>
#include "geo.h"
#include "struct.h"
#include "rad.h"
#include "io.h"
#include "ff.h"

#define USE_HARDWARE 1
extern RadParams ReadLog;
extern OptionType Option;

/**********************************************************************/
#define DFF_Index(i) \
  ((i)<hres ? i: (hres-1- ((i)%hres) ))   /* Index to delta ff array */
Hemicube hemicube;                        /* Hemicube */
double *formfactors;                      /* Form factors */

void InitHemicube();
void SumFactors();
void MakeTopFactors();
void MakeSideFactors();

/**********************************************************************/
/* Create delta ff for top face of hemicube */
/**********************************************************************/
void MakeTopFactors(hres, deltaFactors)
     int hres;
     double *deltaFactors;
{
  int j,k;
  double xSq, ySq, xylSq;
  double n=hres;
  double *wp;
  double dj, dk;
  
  wp = deltaFactors;
  for(j=0;j<hres;j++) {
    dj = (double)j;
    ySq = (n-(dj+0.5)) / n;
    ySq *= ySq;
    for (k=0;k<hres;k++) {
      dk = (double)k;
      xSq = (n-(dk+0.5)) / n;
      xSq *=xSq;
      xylSq = xSq + ySq + 1.0;
      xylSq *= xylSq;
      *wp++ = 1.0 / (xylSq * M_PI * n * n);
    }
  }
}

/**********************************************************************/
/* Create table of delta form factors for side faces of hemicube */
/**********************************************************************/
void MakeSideFactors(hres, deltaFactors)
     int hres;
     double *deltaFactors;
{
  int j,k;
  double x, y, xyl, xSq, ySq, xylSq;
  double n=hres;
  double *wp;
  double dj, dk;

  wp = deltaFactors;
  for(j=0;j<hres;j++) {
    dj = (double)j;
    y = (n-(dj+0.5)) / n;
    ySq = y*y;
    for (k=0;k<hres;k++) {
      dk = (double)k;
      x = (n-(dk+0.5)) / n;
      xSq = x*x;
      xSq *=xSq;
      xyl = xSq + ySq + 1.0;
      xylSq = xyl*xyl;
      *wp++ = y / (xylSq * M_PI * n * n);
    }
  }
}

/**********************************************************************/
/* Initialize hemicube (original 6 sided hemicube)                    */
/**********************************************************************/
void InitHemicube()
{
  int n;
  int hRes;

  if (Option.debug) printf("\t> Initializing hemicube\n");
  hemicube.view.fovx = 90;
  hemicube.view.fovy = 90;
  hRes = ((int)(ReadLog.hemicubeRes/ 2.0 + 0.5))*2;
  hemicube.view.xRes = hemicube.view.yRes = hRes;
  hemicube.buffersize = n = hRes * hRes;
  hemicube.view.buffer = (unsigned long *) malloc(n * sizeof(unsigned long));
  hemicube.view.near = ReadLog.worldSize * .00001;
  hemicube.view.far = ReadLog.worldSize;
  hemicube.view.bank = 0.0;

  hemicube.topFactors = (double *) malloc(n/4 * sizeof(double));
  hemicube.sideFactors = (double *) malloc(n/4 *sizeof(double));
  MakeTopFactors(hRes/2, hemicube.topFactors);
  MakeSideFactors(hRes/2, hemicube.sideFactors);

  if (USE_HARDWARE)
    Create_Item_Buffer(hRes, hRes); /* h/w item-buffer */
}

/**********************************************************************/
/* Initialize temporary space for ff summation                        */
/**********************************************************************/
void InitSumFactors(ffsize)
     int ffsize;
{     
  int i;

  formfactors = (double *) malloc(ffsize * sizeof(double));
  for(i=0;i<ffsize;i++) formfactors[i] = 0.0;
}

/**********************************************************************/
/* Free form factor storage.                                          */
/**********************************************************************/
void FreeSumFactors()
{
  if (Option.debug) printf("\t> Doing hemicube clean up\n");
  free(formfactors);
}

/**********************************************************************/
/* Sum up form factors for given hemicube                             */
/**********************************************************************/
void SumFactors(xRes, yRes, buf, deltaFactors)
     int xRes, yRes;
     unsigned long *buf;
     double *deltaFactors;
{
  int i,j;
  int ii,jj;
  unsigned long *ip=buf;
  int hres = xRes / 2;

  double *ff;

  for (i=0;i<yRes; i++) {
    ii= DFF_Index(i) * hres;
    for (j=0;j<xRes; j++, ip++) {
      if (((*ip) != kBackgroundItem) && (*ip) < ReadLog.num_receivers) {
	jj = DFF_Index(j);
	formfactors[*ip] += deltaFactors[ii+jj];
      }
    }
  }
}

/**********************************************************************/
/* Copy ff values back to elements                                    */
/**********************************************************************/
void UpdateElementFactors(ff)
     double *ff;
{
  int i;
  Elist *elptr;
  
  elptr = ReadLog.elements;
  for(i=0;i<ReadLog.num_elements;i++) 
    elptr->ff = ff[i];
}
