/* store.c, picture output routines                                         */

/* Copyright (C) 1994, MPEG Software Simulation Group. All Rights Reserved. */

/*
 * Disclaimer of Warranty
 *
 * These software programs are available to the user without any license fee or
 * royalty on an "as is" basis.  The MPEG Software Simulation Group disclaims
 * any and all warranties, whether express, implied, or statuary, including any
 * implied warranties or merchantability or of fitness for a particular
 * purpose.  In no event shall the copyright-holder be liable for any
 * incidental, punitive, or consequential damages of any kind whatsoever
 * arising from the use of these programs.
 *
 * This disclaimer of warranty extends to the user of these programs and user's
 * customers, employees, agents, transferees, successors, and assigns.
 *
 * The MPEG Software Simulation Group does not represent or warrant that the
 * programs furnished hereunder are free of infringement of any third-party
 * patents.
 *
 * Commercial implementations of MPEG-1 and MPEG-2 video, including shareware,
 * are subject to royalty fees to patent holders.  Many of these patents are
 * general enough such that they are unavoidable regardless of implementation
 * design.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>

#include "config.h"
#include "global.h"

/* private prototypes */
static void store_one _ANSI_ARGS_((char *outname, unsigned char *src[],
  int offset, int incr, int height));
static void store_yuv _ANSI_ARGS_((char *outname, unsigned char *src[],
  int offset, int incr, int height));
static void store_sif _ANSI_ARGS_((char *outname, unsigned char *src[],
  int offset, int incr, int height));
static void store_tga _ANSI_ARGS_((char *outname, unsigned char *src[],
  int offset, int incr, int height));
static void store_ppm _ANSI_ARGS_((char *outname, unsigned char *src[],
  int offset, int incr, int height));
static void store_yuv1 _ANSI_ARGS_((char *name, unsigned char *src,
  int offset, int incr, int width, int height));
static void putbyte _ANSI_ARGS_((int c));
static void putword _ANSI_ARGS_((int w));
static void putlong _ANSI_ARGS_((long l));
static void conv422to444 _ANSI_ARGS_((unsigned char *src, unsigned char *dst));
static void conv420to422 _ANSI_ARGS_((unsigned char *src, unsigned char *dst));

#define OBFRSIZE 4096
static unsigned char obfr[OBFRSIZE];
static unsigned char *optr;
static int outfile;

/* color space conversion coefficients
 *
 * entries are {crv,cbu,cgu,cgv}
 *
 * crv=(255/224)*65536*(1-cr)/0.5
 * cbu=(255/224)*65536*(1-cb)/0.5
 * cgu=(255/224)*65536*(cb/cg)*(1-cb)/0.5
 * cgv=(255/224)*65536*(cr/cg)*(1-cr)/0.5
 *
 * where Y=cr*R+cg*G+cb*B (cr+cg+cb=1)
 */

static int convmat[8][4] = {
  {117504, 138453, 13954, 34903}, /* no sequence_display_extension */
  {117504, 138453, 13954, 34903}, /* ITU-R Rec. 709 (1990) */
  {104597, 132201, 25675, 53279}, /* unspecified */
  {104597, 132201, 25675, 53279}, /* reserved */
  {104448, 132798, 24759, 53109}, /* FCC */
  {104597, 132201, 25675, 53279}, /* ITU-R Rec. 624-4 System B, G */
  {104597, 132201, 25675, 53279}, /* SMPTE 170M */
  {117579, 136230, 16907, 35559}  /* SMPTE 240M (1987) */
};

/*
 * store a picture as either one frame or two fields
 */
void storeframe(src,frame)
unsigned char *src[];
int frame;
{
  char outname[32];

  if (prog_seq || prog_frame || framestoreflag)
  {
    /* progressive */
    sprintf(outname,outputname,frame,'f');
    store_one(outname,src,0,coded_picture_width,vertical_size);
  }
  else
  {
    /* interlaced */
    sprintf(outname,outputname,frame,'a');
    store_one(outname,src,0,coded_picture_width<<1,vertical_size>>1);

    sprintf(outname,outputname,frame,'b');
    store_one(outname,src,
      coded_picture_width,coded_picture_width<<1,vertical_size>>1);
  }
}

/*
 * store one frame or one field
 */
static void store_one(outname,src,offset,incr,height)
char *outname;
unsigned char *src[];
int offset, incr, height;
{
  switch (outtype)
  {
  case T_YUV:
    store_yuv(outname,src,offset,incr,height);
    break;
  case T_SIF:
    store_sif (outname,src,offset,incr,height);
    break;
  case T_TGA:
    store_tga(outname,src,offset,incr,height);
    break;
  case T_PPM:
    store_ppm(outname,src,offset,incr,height);
    break;
  default:
    break;
  }
}

/* separate headerless files for y, u and v */
static void store_yuv(outname,src,offset,incr,height)
char *outname;
unsigned char *src[];
int offset,incr,height;
{
  char tmpname[32];

  sprintf(tmpname,"%s.Y",outname);
  store_yuv1(tmpname,src[0],offset,incr,horizontal_size,height);

  sprintf(tmpname,"%s.U",outname);
  store_yuv1(tmpname,src[1],offset>>1,incr>>1,horizontal_size>>1,height>>1);

  sprintf(tmpname,"%s.V",outname);
  store_yuv1(tmpname,src[2],offset>>1,incr>>1,horizontal_size>>1,height>>1);
}

/* auxiliary routine */
static void store_yuv1(name,src,offset,incr,width,height)
char *name;
unsigned char *src;
int offset,incr,width,height;
{
  int i, j;
  unsigned char *p;

  if (verbose>1)
    printf("saving %s\n",name);

  if ((outfile = open(name,O_CREAT|O_TRUNC|O_WRONLY|O_BINARY,0666))==-1)
  {
    sprintf(errortext,"Couldn't create %s\n",name);
    error(errortext);
  }

  optr=obfr;

  for (i=0; i<height; i++)
  {
    p = src + offset + incr*i;
    for (j=0; j<width; j++)
      putbyte(*p++);
  }

  if (optr!=obfr)
    write(outfile,obfr,optr-obfr);

  close(outfile);
}

/*
 * store as headerless file in U,Y,V,Y format
 */
static void store_sif (outname,src,offset,incr,height)
char *outname;
unsigned char *src[];
int offset, incr, height;
{
  int i,j;
  unsigned char *py, *pu, *pv;
  static unsigned char *u422, *v422;

  if (!u422)
  {
    u422 = (unsigned char *)malloc((coded_picture_width>>1)
                                   *coded_picture_height);
    v422 = (unsigned char *)malloc((coded_picture_width>>1)
                                   *coded_picture_height);
  }

  conv420to422(src[1],u422);
  conv420to422(src[2],v422);

  strcat(outname,".SIF");

  if (verbose>1)
    printf("saving %s\n",outname);

  if ((outfile = open(outname,O_CREAT|O_TRUNC|O_WRONLY|O_BINARY,0666))==-1)
  {
    sprintf(errortext,"Couldn't create %s\n",outname);
    error(errortext);
  }

  optr = obfr;

  for (i=0; i<height; i++)
  {
    py = src[0] + offset + incr*i;
    pu = u422 + (offset>>1) + (incr>>1)*i;
    pv = v422 + (offset>>1) + (incr>>1)*i;

    for (j=0; j<horizontal_size; j+=2)
    {
      putbyte(*pu++);
      putbyte(*py++);
      putbyte(*pv++);
      putbyte(*py++);
    }
  }

  if (optr!=obfr)
    write(outfile,obfr,optr-obfr);

  close(outfile);
}

/*
 * store as (uncompressed) Truevision TGA file
 */
static void store_tga(outname,src,offset,incr,height)
char *outname;
unsigned char *src[];
int offset, incr, height;
{
  int i, j;
  int y, u, v, r, g, b;
  int crv, cbu, cgu, cgv;
  unsigned char *py, *pu, *pv;
  static unsigned char tga24[14] = {0,0,2,0,0,0,0, 0,0,0,0,0,24,32};
  static unsigned char *u422, *v422, *u444, *v444;

  if (!u422)
  {
    u422 = (unsigned char *)malloc((coded_picture_width>>1)
                                   *coded_picture_height);
    v422 = (unsigned char *)malloc((coded_picture_width>>1)
                                   *coded_picture_height);
    u444 = (unsigned char *)malloc(coded_picture_width*coded_picture_height);
    v444 = (unsigned char *)malloc(coded_picture_width*coded_picture_height);
  }

  conv420to422(src[1],u422);
  conv420to422(src[2],v422);
  conv422to444(u422,u444);
  conv422to444(v422,v444);

  strcat(outname,".tga");

  if (verbose>1)
    printf("saving %s\n",outname);

  if ((outfile = open(outname,O_CREAT|O_TRUNC|O_WRONLY|O_BINARY,0666))==-1)
  {
    sprintf(errortext,"Couldn't create %s\n",outname);
    error(errortext);
  }

  optr = obfr;

  /* TGA header */
  for (i=0; i<12; i++)
    putbyte(tga24[i]);

  putword(horizontal_size); putword(height);
  putbyte(tga24[12]); putbyte(tga24[13]);

  /* matrix coefficients */
  crv = convmat[matrix_coefficients][0];
  cbu = convmat[matrix_coefficients][1];
  cgu = convmat[matrix_coefficients][2];
  cgv = convmat[matrix_coefficients][3];
  
  for (i=0; i<height; i++)
  {
    py = src[0] + offset + incr*i;
    pu = u444 + offset + incr*i;
    pv = v444 + offset + incr*i;

    for (j=0; j<horizontal_size; j++)
    {
      u = *pu++ - 128;
      v = *pv++ - 128;
      y = 76309 * (*py++ - 16); /* (255/219)*65536 */
      r = clp[(y + crv*v + 32768)>>16];
      g = clp[(y - cgu*u -cgv*v + 32768)>>16];
      b = clp[(y + cbu*u + 32786)>>16];
      putbyte(b); putbyte(g); putbyte(r);
    }
  }

  if (optr!=obfr)
    write(outfile,obfr,optr-obfr);

  close(outfile);
}

/*
 * store as PPM (PBMPLUS) file
 */
static void store_ppm(outname,src,offset,incr,height)
char *outname;
unsigned char *src[];
int offset, incr, height;
{
  int i, j;
  int y, u, v, r, g, b;
  int crv, cbu, cgu, cgv;
  unsigned char *py, *pu, *pv;
  char header[32];
  static unsigned char *u422, *v422, *u444, *v444;

  if (!u422)
  {
    u422 = (unsigned char *)malloc((coded_picture_width>>1)
                                   *coded_picture_height);
    v422 = (unsigned char *)malloc((coded_picture_width>>1)
                                   *coded_picture_height);
    u444 = (unsigned char *)malloc(coded_picture_width*coded_picture_height);
    v444 = (unsigned char *)malloc(coded_picture_width*coded_picture_height);
  }

  conv420to422(src[1],u422);
  conv420to422(src[2],v422);
  conv422to444(u422,u444);
  conv422to444(v422,v444);

  strcat(outname,".ppm");

  if (verbose>1)
    printf("saving %s\n",outname);

  if ((outfile = open(outname,O_CREAT|O_TRUNC|O_WRONLY|O_BINARY,0666))==-1)
  {
    sprintf(errortext,"Couldn't create %s\n",outname);
    error(errortext);
  }

  optr = obfr;

  /* PPM header */
  sprintf(header,"P6\n%d %d\n255\n",horizontal_size,height);

  for (i=0; header[i]!=0; i++)
    putbyte(header[i]);

  /* matrix coefficients */
  crv = convmat[matrix_coefficients][0];
  cbu = convmat[matrix_coefficients][1];
  cgu = convmat[matrix_coefficients][2];
  cgv = convmat[matrix_coefficients][3];
  
  for (i=0; i<height; i++)
  {
    py = src[0] + offset + incr*i;
    pu = u444 + offset + incr*i;
    pv = v444 + offset + incr*i;

    for (j=0; j<horizontal_size; j++)
    {
      u = *pu++ - 128;
      v = *pv++ - 128;
      y = 76309 * (*py++ - 16); /* (255/219)*65536 */
      r = clp[(y + crv*v + 32768)>>16];
      g = clp[(y - cgu*u - cgv*v + 32768)>>16];
      b = clp[(y + cbu*u + 32786)>>16];
      putbyte(r); putbyte(g); putbyte(b);
    }
  }

  if (optr!=obfr)
    write(outfile,obfr,optr-obfr);

  close(outfile);
}

static void putbyte(c)
int c;
{
  *optr++ = c;

  if (optr == obfr+OBFRSIZE)
  {
    write(outfile,obfr,OBFRSIZE);
    optr = obfr;
  }
}

static void putword(w)
int w;
{
  putbyte(w); putbyte(w>>8);
}

static void putlong(l)
long l;
{
  putword((int)l); putword((int)(l>>16));
}

/* horizontal 1:2 interpolation filter */
static void conv422to444(src,dst)
unsigned char *src,*dst;
{
  int i, i2, w, j, im3, im2, im1, ip1, ip2, ip3;

  w = coded_picture_width>>1;

  if (base.mpeg2)
  {
    for (j=0; j<coded_picture_height; j++)
    {
      for (i=0; i<w; i++)
      {
        i2 = i<<1;
        im2 = (i<2) ? 0 : i-2;
        im1 = (i<1) ? 0 : i-1;
        ip1 = (i<w-1) ? i+1 : w-1;
        ip2 = (i<w-2) ? i+2 : w-1;
        ip3 = (i<w-3) ? i+3 : w-1;

        /* FIR filter coefficients (*256): 21 0 -52 0 159 256 159 0 -52 0 21 */
        /* even samples (0 0 256 0 0) */
        dst[i2] = src[i];

        /* odd samples (21 -52 159 159 -52 21) */
        dst[i2+1] = clp[(int)(21*(src[im2]+src[ip3])
                        -52*(src[im1]+src[ip2]) 
                       +159*(src[i]+src[ip1])+128)>>8];
      }
      src+= w;
      dst+= coded_picture_width;
    }
  }
  else
  {
    for (j=0; j<coded_picture_height; j++)
    {
      for (i=0; i<w; i++)
      {

        i2 = i<<1;
        im3 = (i<3) ? 0 : i-3;
        im2 = (i<2) ? 0 : i-2;
        im1 = (i<1) ? 0 : i-1;
        ip1 = (i<w-1) ? i+1 : w-1;
        ip2 = (i<w-2) ? i+2 : w-1;
        ip3 = (i<w-3) ? i+3 : w-1;

        /* FIR filter coefficients (*256): 5 -21 70 228 -37 11 */
        dst[i2] =   clp[(int)(  5*src[im3]
                         -21*src[im2]
                         +70*src[im1]
                        +228*src[i]
                         -37*src[ip1]
                         +11*src[ip2]+128)>>8];

       dst[i2+1] = clp[(int)(  5*src[ip3]
                         -21*src[ip2]
                         +70*src[ip1]
                        +228*src[i]
                         -37*src[im1]
                         +11*src[im2]+128)>>8];
      }
      src+= w;
      dst+= coded_picture_width;
    }
  }
}

/* vertical 1:2 interpolation filter */
static void conv420to422(src,dst)
unsigned char *src,*dst;
{
  int w, h, i, j, j2;
  int jm6, jm5, jm4, jm3, jm2, jm1, jp1, jp2, jp3, jp4, jp5, jp6, jp7;

  w = coded_picture_width>>1;
  h = coded_picture_height>>1;

  if (prog_frame)
  {
    /* intra frame */
    for (i=0; i<w; i++)
    {
      for (j=0; j<h; j++)
      {
        j2 = j<<1;
        jm3 = (j<3) ? 0 : j-3;
        jm2 = (j<2) ? 0 : j-2;
        jm1 = (j<1) ? 0 : j-1;
        jp1 = (j<h-1) ? j+1 : h-1;
        jp2 = (j<h-2) ? j+2 : h-1;
        jp3 = (j<h-3) ? j+3 : h-1;

        /* FIR filter coefficients (*256): 5 -21 70 228 -37 11 */
        dst[w*j2] =     clp[(int)(  5*src[w*jm3]
                             -21*src[w*jm2]
                             +70*src[w*jm1]
                            +228*src[w*j]
                             -37*src[w*jp1]
                             +11*src[w*jp2]+128)>>8];

        dst[w*(j2+1)] = clp[(int)(  5*src[w*jp3]
                             -21*src[w*jp2]
                             +70*src[w*jp1]
                            +228*src[w*j]
                             -37*src[w*jm1]
                             +11*src[w*jm2]+128)>>8];
      }
      src++;
      dst++;
    }
  }
  else
  {
    /* intra field */
    for (i=0; i<w; i++)
    {
      for (j=0; j<h; j+=2)
      {
        j2 = j<<1;

        /* top field */
        jm6 = (j<6) ? 0 : j-6;
        jm4 = (j<4) ? 0 : j-4;
        jm2 = (j<2) ? 0 : j-2;
        jp2 = (j<h-2) ? j+2 : h-2;
        jp4 = (j<h-4) ? j+4 : h-2;
        jp6 = (j<h-6) ? j+6 : h-2;

        dst[w*j2] = clp[(int)(  2*src[w*jm6]
                         -10*src[w*jm4]
                         +35*src[w*jm2]
                        +242*src[w*j]
                         -18*src[w*jp2]
                          +5*src[w*jp4]+128)>>8];

        dst[w*(j2+2)] = clp[(int)( 11*src[w*jm4]
                             -38*src[w*jm2]
                            +192*src[w*j]
                            +113*src[w*jp2]
                             -30*src[w*jp4]
                              +8*src[w*jp6]+128)>>8];

        /* bottom field */
        jm5 = (j<5) ? 1 : j-5;
        jm3 = (j<3) ? 1 : j-3;
        jm1 = (j<1) ? 1 : j-1;
        jp1 = (j<h-1) ? j+1 : h-1;
        jp3 = (j<h-3) ? j+3 : h-1;
        jp5 = (j<h-5) ? j+5 : h-1;
        jp7 = (j<h-7) ? j+7 : h-1;

        dst[w*(j2+1)] = clp[(int)( 11*src[w*jp5]
                             -38*src[w*jp3]
                            +192*src[w*jp1]
                            +113*src[w*jm1]
                             -30*src[w*jm3]
                              +8*src[w*jm5]+128)>>8];

        dst[w*(j2+3)] = clp[(int)(  2*src[w*jp7]
                             -10*src[w*jp5]
                             +35*src[w*jp3]
                            +242*src[w*jp1]
                             -18*src[w*jm1]
                              +5*src[w*jm3]+128)>>8];
      }
      src++;
      dst++;
    }
  }
}
