/* 
 * Mach Operating System
 * Copyright (c) 1992 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 * 
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 * 
 * Carnegie Mellon requests users of this software to return to
 * 
 *  Software Distribution Coordinator  or  Software.Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 * 
 * any improvements or extensions that they make and grant Carnegie Mellon 
 * the rights to redistribute these changes.
 */
/*
 * HISTORY
 * $Log:	model_dep.c,v $
 * Revision 2.8  92/01/03  20:31:28  dbg
 * 	Don't use 'RB_KDB' - always trap to debugger.
 * 	[91/11/06            dbg]
 * 
 * 	Rename kdb_init to ddb_init.
 * 	[91/09/11            dbg]
 * 
 * Revision 2.7  91/06/19  11:59:18  rvb
 * 	cputypes.h->platforms.h
 * 	[91/06/12  13:46:41  rvb]
 * 
 * Revision 2.6  91/05/18  14:37:49  rpd
 * 	Changed arguments to vm_page_create.
 * 	[91/05/16            rpd]
 * 
 * Revision 2.5  91/03/16  14:58:46  rpd
 * 	Updated for new kmem_alloc interface.
 * 	[91/03/03            rpd]
 * 
 * Revision 2.4  90/08/27  22:11:43  dbg
 * 	Name for debugger flag is 'RB_KDB'.  Declare boothowto.
 * 	[90/08/14            dbg]
 * 
 * 	Don't initialize sun_kmem - it's no longer used.
 * 	[90/07/17            dbg]
 * 
 * Revision 2.3  90/06/02  15:06:41  rpd
 * 	Fixed timemmap to not require execute permission.
 * 	[90/05/01            rpd]
 * 
 * Revision 2.2  89/08/09  14:33:20  rwd
 * 	Include mach/vm_prot.h
 * 	[89/08/08            rwd]
 * 	Must define mtime as external.
 * 	[89/08/08            rwd]
 * 	Moved timemmap here.
 * 	[89/08/08            rwd]
 * 
 * Revision 2.1  89/08/03  16:51:23  rwd
 * Created.
 * 
 * 24-May-89  Randall Dean (rwd) at Carnegie-Mellon University
 *	Copyright added to MACH_KERNEL version torn from machdep.c by
 *	dbg.
 *
 */
#include <mach_kdb.h>
#include <platforms.h>

#include <mach/boolean.h>
#include <kern/time_out.h>
#include <kern/thread.h>

#include <vm/vm_kern.h>
#include <mach/vm_param.h>
#include <vm/vm_page.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/reboot.h>

#include <sun/consdev.h>

#include <machine/cpu.h>
#include <machine/frame.h>
#include <machine/eccreg.h>
#include <machine/interreg.h>
#include <machine/machparam.h>
#include <machine/memerr.h>
#include <machine/scb.h>

#include <mon/sunromvec.h>

#include <sundev/mbvar.h>

/*
 *	Pointer to thread which owns FP state.
 */
thread_t	fp_threads[NCPUS];

/*
 * Routines that differ between SUN3 models.
 */

int (*exit_vector)() = (int (*)())0;	/* where to go when halting MACH */

#define	TESTVAL 0xa55a	/* memory test value */

#ifdef SUN3_260
/*
 * Since there is no implied ordering of the memory cards, we store
 * a zero terminated list of pointers to eccreg's that are active so
 * that we only look at existent memory cards during softecc() handling.
 */
struct eccreg *ecc_alive[MAX_ECC+1];
int	vac;
#endif SUN3_260

extern void	inittodr();

extern char	version[];

extern vm_offset_t hole_start, hole_end;
int fbobmemavail;

int boothowto = 0;

machine_init()
{

	vm_offset_t	fb_start_pa, fb_end_pa, pa, phys;
	boolean_t	fbarray[FBSIZE/NBPG];
	int		dvmapage;

	/*
	 * If the first available physical address (namely, the first
	 * one handed to vm_page_startup) is already beyond the frame
	 * buffer memory, it has been used.  This means that we can't
	 * support a frame buffer if it needs that memory.
	 */

	if (fbobmemavail){
		extern vm_offset_t avail_start;

		if (trunc_page(OBFBADDR) < avail_start) {
			printf("Kernel image overlaps frame buffer memory\n");
			fbobmemavail = 0;
		}
	}

	/*
	 * Determine if anything lives in DVMA bus space.
	 * We're paranoid and go through both the 16 bit
	 * and 32 bit device types.
 	 */
	if (kmem_alloc_pageable(kernel_map, &phys, PAGE_SIZE) != KERN_SUCCESS)
		panic("dvma_probe: no space");
	disable_dvma();
        /* Access VME addresses, don't worry about cache here. */
	for (dvmapage = 0; dvmapage < btoc(dvmasize); dvmapage++) {
		pmap_enter(kernel_pmap, phys,
				       EPA_VME_D16+dvmapage,
				       VM_PROT_READ|VM_PROT_WRITE, FALSE);
		if (poke((short *)phys, TESTVAL) == 0)
			break;
		pmap_enter(kernel_pmap, phys,
				       EPA_VME_D32+dvmapage,
				       VM_PROT_READ|VM_PROT_WRITE, FALSE);
		if (poke((short *)phys, TESTVAL) == 0)
			break;
	}
	enable_dvma();
	kmem_free(kernel_map, phys, PAGE_SIZE);

	bootflags();
#if	MACH_KDB
	/*
	 *	Initialize the kernel debugger.
	 */
	ddb_init();
	if (boothowto & RB_HALT)
	    asm("trap #0xf");
#endif	MACH_KDB

	/*
	 * Good {morning,afternoon,evening,night}.
	 * When printing memory, use the total including
	 * those hidden by the monitor (mon_mem).
	 */
	printf(version);

#define MEG	(1024*1024)
	printf("physical memory = %d.%d%d megabytes.\n", mem_size/MEG,
		((mem_size%MEG)*10)/MEG,
		((mem_size%(MEG/10))*100)/MEG);

	if (dvmapage < btoc(dvmasize)) {
		printf("CAN'T HAVE PERIPHERALS IN RANGE 0 - %dKB\n",
		    ctob(dvmasize) / 1024);
		panic("dvma collision");
	}

	(void) kmem_alloc_wired(kernel_map,
				(vm_offset_t *) &iopbmap, IOPBMAPSIZE);
	(void) kmem_alloc_wired(kernel_map,
				(vm_offset_t *) &mb_hd.mh_map, DVMAMAPSIZE);

	rminit(iopbmap, (long)ctob(IOPBMEM), (long)DVMA, 
	    "IOPB space", IOPBMAPSIZE);
	rminit(mb_hd.mh_map, (long)(dvmasize - IOPBMEM), (long)IOPBMEM,
	    "DVMA map space", DVMAMAPSIZE);

	/*
	 * Configure the system.
	 */
	configure();		/* set up devices */
	get_root_device();	/* get root device for initial laod */
	inittodr();		/* start todr */

	/*
	 * If the onboard frame buffer memory is still
	 * available, use it as generic physical memory.
	 */
	if (fbobmemavail){
		/*
		 *	Memory was still available.  Take it back.
		 */

		vm_page_create(OBFBADDR, OBFBADDR + FBSIZE);
		fbobmemavail = 0;
		hole_start = 0;
		hole_end = 0;
	}
#ifdef SUN3_260
	/*
	 *	Enable cache if appropriate.
	 */
	if (cpu == CPU_SUN3_260) {
		setcacheenable(1);
		vac = 1;
	}
#endif	SUN3_260
	consconfig();
}

struct	bootf {
	char	let;
	short	bit;
} bootf[] = {
	'a',	RB_ASKNAME,
	's',	RB_SINGLE,
	'i',	RB_INITNAME,
	'h',	RB_HALT,
	'b',	RB_NOBOOTRC,
#if	MACH_KDB
	'd',	RB_KDB,
#endif	MACH_KDB
	0,	0,
};
char *initname = "/etc/init";

/*
 * Parse the boot line to determine boot flags .
 */
bootflags()
{
	register struct bootparam *bp = (*sunromp->v_bootparam);
	register char *cp;
	register int i;

	cp = bp->bp_argv[1];
	if (cp && *cp++ == '-')
		do {
			for (i = 0; bootf[i].let; i++) {
				if (*cp == bootf[i].let) {
					boothowto |= bootf[i].bit;
					break;
				}
			}
			cp++;
		} while (bootf[i].let && *cp);
	if (boothowto & RB_INITNAME)
		initname = bp->bp_argv[2];
/*	if (boothowto & RB_HALT)
		halt("bootflags"); */

}

halt_all_cpus(reboot)
	boolean_t	reboot;
{
	consdev = 0;
	startnmi();

	if (!reboot) {
	    halt((char *)0);
	    /* MAYBE REACHED */
	}
	else {
	    printf("Rebooting MACH...\n");
/*
	    (*sunromp->v_boot_me)(howto & RB_SINGLE ? "-s" : "");
 */
	    (*sunromp->v_boot_me)("");
	    /*NOTREACHED*/
	}
}

halt_cpu()
{
    halt_all_cpus(1);
}

/*
 * Initialize MACH's vector table:
 * Vectors are copied from protoscb unless
 * they are zero; zero means preserve whatever the
 * monitor put there.  If the protoscb is zero,
 * then the original contents are copied into
 * the scb we are setting up.
 */
initscb()
{
	register int *s, *p, *f;
	register int n;
	struct scb *orig, *getvbr();

	orig = getvbr();
	exit_vector = orig->scb_trap[14];
	s = (int *)&scb;
	p = (int *)&protoscb;
	f = (int *)orig;
	for (n = sizeof (struct scb)/sizeof (int); n--; s++, p++, f++) {
		if (*p) 
			*s = *p;
		else
			*s = *f;
	}
	setvbr(&scb);
}

/*
 * Halt the machine and return to the monitor
 */
halt(s)
	char *s;
{
	extern struct scb *getvbr();

	if (s)
		(*sunromp->v_printf)("(%s) ", s);
	(*sunromp->v_printf)("Mach Halted\n\n");
	startnmi();
	if (exit_vector)
		getvbr()->scb_trap[14] = exit_vector;
	asm("trap #14");
	if (exit_vector)
		getvbr()->scb_trap[14] = protoscb.scb_trap[14];
	stopnmi();
}

/*
 * Print out a traceback for the caller - can be called anywhere
 * within the kernel or from the monitor by typing "g4" (for sun-2
 * compatibility) or "w trace".  This causes the monitor to call
 * the v_handler() routine which will call tracedump() for these cases.
 */
/*VARARGS0*/
tracedump(x1)
	caddr_t x1;
{
	struct frame *fp = (struct frame *)(&x1 - 2);
	u_int tospage = btoc(fp);

	(*sunromp->v_printf)("Begin traceback...fp = %x\n", fp);
	while (btoc(fp) == tospage) {
		if (fp == fp->fr_savfp) {
			(*sunromp->v_printf)("FP loop at %x", fp);
			break;
		}
		(*sunromp->v_printf)("Called from %x, fp=%x, args=%x %x %x %x\n",
		    fp->fr_savpc, fp->fr_savfp,
		    fp->fr_arg[0], fp->fr_arg[1], fp->fr_arg[2], fp->fr_arg[3]);
		fp = fp->fr_savfp;
	}
	(*sunromp->v_printf)("End traceback...\n");
}

/*
 * if a() calls b() calls caller(),
 * caller() returns return address in a().
 */
int
(*caller())()
{

#ifdef lint
	return ((int (*)())0);
#else
	asm("   movl    a6@,a0");
	asm("   movl    a0@(4),d0");
#endif
}

/*
 * if a() calls callee(), callee() returns the
 * return address in a();
 */
int
(*callee())()
{

#ifdef lint
	return ((int (*)())0);
#else
	asm("   movl    a6@(4),d0");
#endif
}

/*
 * Buscheck is called by mbsetup to check to see it the requested
 * setup is a valid busmem type (i.e. VMEbus).  Returns 1 if ok
 * busmem type, returns 0 if not busmem type.  This routine
 * make checks and panic's if an illegal busmem type request is detected.
 */
buscheck(bufaddr, npf)
	register caddr_t bufaddr;
	register int npf;
{
	int origctx;
	register int i, pf;
	register int pt;
	long pme;
	register struct pme *pte;

	pme = getpgmap(bufaddr);
	pte = (struct pme *)&pme;
	pt = pme & PGT_MASK;
	if (pt == PGT_VME_D16 || pt == PGT_VME_D32) {
		pf = pte->pfn;
		if (pf < btoc(DVMASIZE))
			panic("buscheck: busmem in DVMA range");
		for (i = 0; i < npf; i++, pf++) {
			pme = getpgmap(bufaddr);
			pte = (struct pme *)&pme;
			if ((*(int *)pte & PGT_MASK) != pt || pte->pfn != pf)
				panic("buscheck: request not contiguous");
			bufaddr += NBPG;
		}
		return (1);
	}
	return (0);
}

/* 
 * Compute the address of an I/O device within standard address
 * ranges and return the result.  This is used by DKIOCINFO
 * ioctl to get the best guess possible for the actual address
 * set on the card.
 */
getdevaddr(addr)
	caddr_t addr;
{
	int off = (int)addr & PGOFSET;
	int pte = getkpgmap(addr);
	int physaddr = ((pte & PG_PFNUM) & ~PGT_MASK) * NBPG;

	switch (pte & PGT_MASK) {
	case PGT_VME_D16:
	case PGT_VME_D32:
		if (physaddr > VME16_BASE) {
			/* 16 bit VMEbus address */
			physaddr -= VME16_BASE;
		} else if (physaddr > VME24_BASE) {
			/* 24 bit VMEbus address */
			physaddr -= VME24_BASE;
		}
		/*
		 * else 32 bit VMEbus address,
		 * physaddr doesn't require adjustments
		 */
		break;

	case PGT_OBMEM:
	case PGT_OBIO:
		/* physaddr doesn't require adjustments */
		break;
	}

	return (physaddr + off);
}

static int (*mon_nmi)();		/* monitor's level 7 nmi routine */
static u_char mon_mem;			/* monitor memory register setting */
extern int level7();			/* Unix's level 7 nmi routine */

stopnmi()
{
	struct scb *vbr, *getvbr();

	vbr = getvbr();
	if (vbr->scb_autovec[7 - 1] != level7) {
#ifndef GPROF
		set_clk_mode(0, IR_ENA_CLK7);	/* disable level 7 clk intr */
#endif !GPROF
		mon_nmi = vbr->scb_autovec[7 - 1];	/* save mon vec */
		vbr->scb_autovec[7 - 1] = level7;	/* install Unix vec */
#ifdef SUN3_260
		if (cpu == CPU_SUN3_260) {
			mon_mem = MEMREG->mr_eer;
			MEMREG->mr_eer = EER_INTENA | EER_CE_ENA;
		}
#endif SUN3_260
#if defined(SUN3_160) || defined(SUN3_50) || defined(SUN3_110) || defined(SUN3_60)
		if (cpu == CPU_SUN3_160 || cpu == CPU_SUN3_50 ||
		    cpu == CPU_SUN3_110 || cpu == CPU_SUN3_60)
			mon_mem = MEMREG->mr_per;
#endif defined(SUN3_160) || defined(SUN3_50) || defined(SUN3_110) || defined(SUN3_60)
	}
}

startnmi()
{
	struct scb *getvbr();

	if (mon_nmi) {
		getvbr()->scb_autovec[7 - 1] = mon_nmi;	/* install mon vec */
#ifndef GPROF
		set_clk_mode(IR_ENA_CLK7, 0);	/* enable level 7 clk intr */
#endif !GPROF
#ifdef SUN3_260
		if (cpu == CPU_SUN3_260)
			MEMREG->mr_eer = mon_mem;
#endif SUN3_260
#if defined(SUN3_160) || defined(SUN3_50) || defined(SUN3_110) || defined(SUN3_60)
		if (cpu == CPU_SUN3_160 || cpu == CPU_SUN3_50 ||
		    cpu == CPU_SUN3_110 || cpu == CPU_SUN3_60)
			MEMREG->mr_per = mon_mem;
#endif defined(SUN3_160) || defined(SUN3_50) || defined(SUN3_110) || defined(SUN3_60)
	}
}

/*
 * Handler for monitor vector cmd -
 * For now we just implement the old "g0" and "g4"
 * commands and a printf hack.
 */
void
v_handler(addr, str)
	int addr;
	char *str;
{

	switch (*str) {
	case '\0':
		/*
		 * No (non-hex) letter was specified on
		 * command line, use only the number given
		 */
		switch (addr) {
		case 0:		/* old g0 */
		case 0xd:	/* 'd'ump short hand */
			panic("zero");
			/*NOTREACHED*/
		
		case 4:		/* old g4 */
			tracedump();
			break;

		default:
			goto err;
		}
		break;

	case 'p':		/* 'p'rint string command */
	case 'P':
		(*sunromp->v_printf)("%s\n", (char *)addr);
		break;

	case '%':		/* p'%'int anything a la printf */
		(*sunromp->v_printf)(str, addr);
		(*sunromp->v_printf)("\n");
		break;

	case 't':		/* 't'race kernel stack */
	case 'T':
		tracedump();
		break;

	case 'u':		/* d'u'mp hack ('d' look like hex) */
	case 'U':
		if (addr == 0xd) {
			panic("zero");
		} else
			goto err;
		break;

	default:
	err:
		(*sunromp->v_printf)("Don't understand 0x%x '%s'\n", addr, str);
	}
}

/*
 * Handle parity/ECC memory errors.  XXX - use something like
 * vax to only look for soft ecc errors periodically?
 */
memerr()
{
	u_char per, eer;
	char *mess = 0;
	int c;
	long pme;

	eer = per = MEMREG->mr_er;
#ifdef SUN3_260
	if (cpu == CPU_SUN3_260 && (eer & EER_ERR) == EER_CE) {
		MEMREG->mr_eer = ~EER_CE_ENA;
		softecc();
		MEMREG->mr_dvma = 1;	/* clear latching */
		MEMREG->mr_eer |= EER_CE_ENA;
		return;
	} 
#endif SUN3_260

	/*
	 * Since we are going down in flames, disable further
	 * memory error interrupts to prevent confusion.
	 */
	MEMREG->mr_er &= ~ER_INTENA;

#if defined(SUN3_160) || defined(SUN3_50) || defined(SUN3_110) || defined(SUN3_60)
	if ((cpu == CPU_SUN3_160 || cpu == CPU_SUN3_50 ||
	    cpu == CPU_SUN3_110 || cpu == CPU_SUN3_60) &&
	    (per & PER_ERR) != 0) {
		printf("Parity Error Register %b\n", per, PARERR_BITS);
		mess = "parity error";
	}
#endif defined(SUN3_160) || defined(SUN3_50) || defined(SUN3_110) || defined(SUN3_60)


#ifdef SUN3_260
	if ((cpu == CPU_SUN3_260) && (eer & EER_ERR) != 0) {
		printf("ECC Error Register %b\n", eer, ECCERR_BITS);
		if (eer & EER_TIMEOUT)
			mess = "memory timeout error";
		if (eer & EER_UE)
			mess = "uncorrectable ECC error";
		if (eer & EER_WBACKERR)
			mess = "writeback  error";
	}
#endif SUN3_260

	if (!mess) {
		printf("Memory Error Register %b %b\n",
		    per, PARERR_BITS, eer, ECCERR_BITS);
		mess = "unknown memory error";
	}

	printf("DVMA = %x, context = %x, virtual address = %x\n",
		MEMREG->mr_dvma, MEMREG->mr_ctx, MEMREG->mr_vaddr);

	pme = getpagemap((int)MEMREG->mr_ctx, (caddr_t)MEMREG->mr_vaddr);

	printf("pme = %x, physical address = %x\n", pme,
	    ptob(((struct pme *)&pme)->pfn) + (MEMREG->mr_vaddr&PGOFSET));

	/*
	 * Clear the latching by writing to the top
	 * nibble of the memory address register
	 */
	MEMREG->mr_dvma = 1;

	panic(mess);
	/*NOTREACHED*/
}

#ifdef SUN3_260
int prtsoftecc = 1;
int memintvl = MEMINTVL;

/* 
 * Routine to turn on correctable error reporting.
 */
ce_enable(ecc)
register struct eccreg *ecc;
{
	ecc->eccena |= ENA_BUSENA_MASK;
}

/*
 * Probe memory cards to find which one(s) had ecc error(s).
 * If prtsoftecc is non-zero, log messages regarding the failing
 * syndrome.  Then clear the latching on the memory card.
 */
softecc()
{
	register struct eccreg **ecc_nxt, *ecc;

	for (ecc_nxt = ecc_alive; *ecc_nxt != (struct eccreg *)0; ecc_nxt++) {
		ecc = *ecc_nxt;
		if (ecc->syndrome & SY_CE_MASK) {
			if (prtsoftecc) {
				printf("mem%d: soft ecc addr %x+%x=%x syn %b\n",
				    ecc - ECCREG,
				    (ecc->eccena & ENA_ADDR_MASK) <<
					ENA_ADDR_SHIFT,
				    (ecc->syndrome & SY_ADDR_MASK) <<
					SY_ADDR_SHIFT,
				    ((ecc->eccena & ENA_ADDR_MASK) <<
					ENA_ADDR_SHIFT) +
					((ecc->syndrome & SY_ADDR_MASK) <<
					SY_ADDR_SHIFT),
				    (ecc->syndrome & SY_SYND_MASK) >>
					SY_SYND_SHIFT, SYNDERR_BITS);
			}
			ecc->syndrome |= SY_CE_MASK;	/* clear latching */
			ecc->eccena &= ~ENA_BUSENA_MASK;/* disable board */
			timeout(ce_enable, (caddr_t)ecc, memintvl*hz);
		}
	}
}
#endif SUN3_260

boot(paniced, howto)
	int paniced, howto;
{
	static int prevflag = 0;
	register struct buf *bp;
	int iter, nbusy;
	int s;

	consdev = 0;
	startnmi();
	printf("done\n");
	s = spl7();				/* extreme priority */
	if (howto & RB_HALT) {
		halt((char *)NULL);
		/* MAYBE REACHED */
	} else {
		printf("Rebooting Mach...\n");
		(*sunromp->v_boot_me)(howto & RB_SINGLE ? "-s" : "");
		/*NOTREACHED*/
	}
	(void) splx(s);
}
/*
 *	This is used by the mouse and kbd.
 */
uniqtime(tv)
	register struct time_value *tv;
{
	static struct time_value last;
	static int uniq;
			
	while (last.microseconds != time.microseconds || 
	       last.seconds != time.seconds) {
		last = time;
		uniq = 0;
	}
	*tv = last;
	tv->microseconds += uniq++;
}

#include <mach/vm_prot.h>

timemmap(dev,off,prot)
	vm_prot_t prot;
{
	extern int *mtime;

	if (prot & VM_PROT_WRITE)
		return (-1);

	return getkpgmap(mtime) & PG_PFNUM;
}
