#ifdef PICL
#define _PICL_C_

#define _NO_XPRINTF_

#include "easy.h"
#include "easydefs.h"

#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <time.h>

#ifndef MAXWAITPERIOD
#define MAXWAITPERIOD 10
#endif

#define LOGFILE_BUFSIZE 64*1024         /* Buffer size of the logfile */
static char logbuf[LOGFILE_BUFSIZE];    /* I/O buffer for tracefile */

#define MSG_CLOCK_PREP 20001000
#define MSG_CLOCK_SYNC 20001001

static char path[256]="";
static struct stat stbuf;

static int tsecs = 0, tusecs = 0;       /* Cumulative seconds since trace start */
static double tsecstotal = 0;           /* Cumulative seconds & usecs since trace start */
static int idlesecs = 0, idleusecs = 0; /* Idle period (not cumulative) */
static double idlesecstotal = 0;        /* Idle period in seconds & usecs */
static double t0ref = 0;                /* Reference time since start of tracing */
static FILE *All_logfp[MAXCLUSTERS] = { 0 } ; /* All logfile pointer(s) */

#define timestrip(total,sec,usec) \
{ (sec) = (int)(total); (usec) = (int) (1000000.0 * ((total) - (sec))); }

static  /* ??? */
double clocksync()
{
  if (ME != HOSTID) {
    double t1, t2, dcl;
    FILE *tmpfp = logfp;
    int dt = setdatatype(REAL8);
    int st = setstride(1);
    double waitperiod;
    
    logfp = NULL; /* If on, shut off logging temporarely */
    
    if (ME == 0) {
      int multiplier = numnodes();

      t1 = dclock();
      broadcast(MSG_CLOCK_PREP,&t1,1);
      gsync();
      gsync();
      t2 = dclock();
      waitperiod = multiplier * (t2 - t1);
      if (waitperiod < numnodes()) waitperiod = numnodes();
      if (waitperiod > MAXWAITPERIOD) waitperiod = MAXWAITPERIOD;
      t2 = dclock() + waitperiod;
      broadcast(MSG_CLOCK_SYNC,&waitperiod,1);

      /* printf("0: t2=%lfs, wp=%lfs\n",t2,waitperiod); fflush(stdout); */
    }
    else {
      recv(0,MSG_CLOCK_PREP,&t1,1);
      gsync();
      gsync();
      recv(0,MSG_CLOCK_SYNC,&waitperiod,1);
      t2 = dclock() + waitperiod;
    }

    if (t2 > (dcl=dclock())) {
      while (t2 > dclock()) /* Do nothing, but spin */;
    }
    else {
      fprintf(stderr,"clocksync(%d): Clock sync failed. T's=%lf,%lf,%lf\n",
	      ME,dcl,t2,dcl-t2);
      killproc(ME);
    }

    logfp = tmpfp;    /* Enable logging again */
    setdatatype(dt);
    setstride(st);
    
    /* return t2; */
    return dclock(); /* Better ? */
  }
  return 0;
}


static void makedir(dir)
     char *dir;
{
  static char hostname[64]="";

  if (hostname[0] == ' ') gethostname(hostname,sizeof(hostname));

  if (stat(path,&stbuf) == -1) {
    if (errno == ENOENT) 
      mkdir(path,0700);
    else {
      fprintf(stderr,"makedir(%d): Problems with stat(%s:%s,...)\n",
	      ME,hostname,path);
      perror(path);
      killproc(ME);
    }
  }
  
  stat(path,&stbuf);
  if (!S_ISDIR(stbuf.st_mode)) {
    fprintf(stderr,"makedir(%d): '%s:%s' is not a directory\n",
	    ME,hostname,path);
    killproc(ME);
  }
}

void traceenable(name, verbose)
     char *name;
     int verbose;
{
  if (ME != HOSTID) { /* Initially restrict to NODEs only */
    char *home = getenv("HOME");
    int cl_id = getcluster();
    char hostname[128];

    sprintf(path,"%s/picl",home);

    makedir(path);

    sprintf(path,"%s/picl/%d",home,HOST_TIMESTAMP);

    makedir(path);    

    picl_me = ME;

    sprintf(path,"%s/picl/%d/%s-%d.%d",home,HOST_TIMESTAMP,name ? name : "",
	    picl_me,cl_id);

    gethostname(hostname,sizeof(hostname)/2);
    xprintf("%d: PICL-tracefile = '%s:%s'\n",ME,hostname,path);

    logfp = NULL;
    All_logfp[cl_id] = fopen(path,"w");

    /* Try to syncronize timers as much as possible */
    gsync(); 
    t0ref = clocksync();

    logfp = All_logfp[cl_id];
  
    if (logfp) {
      setvbuf(logfp,logbuf,_IOFBF,LOGFILE_BUFSIZE);
      TRACETIME();
      TRACEF(1);
      strcat(hostname,"@");
      strcat(hostname,PVM_ARCH);
      tracemsg(hostname);
    }
    else {
      fprintf(stderr,"traceenable(%d): Unable to open tracefile '%s'\n",
	      ME,path);
      killproc(ME);
    }
  }
}


void tracemsg(s)
     char *s;
{
  if (s && logfp) {
    TRACETIME();
    TRACEF(16,s);
  }
}


void traceexit()
{
  if (logfp) {
    int cl_id = getcluster();
    TRACEF(19); /* Use the latest time stamp value  */
    fflush(logfp);
    stat(path,&stbuf);
    fprintf(logfp,"%9d\n",stbuf.st_size+10);
    fclose(logfp);
    logfp = All_logfp[cl_id] = NULL;
  }
}


static double timedelta(Tsecs, Tusecs, T0ref)
     int *Tsecs;
     int *Tusecs;
     double T0ref;
{
  double secs = dclock() - T0ref;
  timestrip(secs,*Tsecs,*Tusecs);
  return secs;
}


void TRACETIME()   
{ 
  tsecstotal = timedelta(&tsecs,&tusecs,t0ref); 
}

void SETIDLETIME() 
{ 
  TRACETIME();
  idlesecstotal = tsecstotal;
  TRACEF(11,(double)0, (double)0);
}

void UPDATEIDLETIME() 
{
  int idle = tsecstotal - idlesecstotal;
  timestrip(idle,idlesecs,idleusecs);
  TRACEF(11,idlesecs,idleusecs);
}


#ifdef  __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

void 
#ifdef __STDC__
TRACEF(int i, ...)
#else
TRACEF(va_alist)
va_dcl
#endif
{
  static char *compact_fmt[1+21] = 
    { /* vfprintf()-format of compact records */
      "dummy_rec\n", /*  0: Should not be called (N/A) */
      "0 0 0\n",     /*  1: trace_start */
      "\n",          /*  2: open (N/A) */
      "\n",          /*  3: load (N/A) */
      "%d %d %d\n",  /*  4: send */
      "\n",          /*  5: probe (N/A) */
      "%d %d %d\n",  /*  6: recv */
      "%d\n",        /*  7: recv_blocking */
      "%d %d %d\n",  /*  8: recv_waking */
      "\n",          /*  9: message (N/A?)*/
      "\n",          /* 10: sync */
      "%d %d\n",     /* 11: compstats */
      "%d %d %d %d %d\n",  /* 12: commstats (N/A?) */
      "\n",          /* 13: close */
      "\n",          /* 14: trace_level (N/A) */
      "\n",          /* 15: trace_mark (N/A) */
      "%s\n",        /* 16: trace_message */
      "\n",          /* 17: trace_stop (N/A) */
      "\n",          /* 18: trace_flush (N/A) */
      " ",           /* 19: trace_exit */
      "%d %d %d\n",  /* 20: block_begin */
      "%d %d %d\n"   /* 21: block_end */
    };

  va_list therest;
#ifdef __STDC__
  va_start(therest,i);
#else
  int i;
  va_start(therest);
  i = va_arg(therest, int);
#endif

  if (i>=1 && i<=21 && logfp) {
    fprintf(logfp,"%d %d %d %d ",i,tsecs,tusecs,picl_me);
    vfprintf(logfp,compact_fmt[i],therest);
  }

  va_end(therest);

}

#else

static void dummy_routine()
 { return; }

#endif /* PICL */
