/*
 Copyright (c) 2003 RIPE

 All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of the author not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
 AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
 DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

 $Id: v6wpd.c,v 1.23 2003/04/15 12:02:04 can Exp $
*/

#include "local_error.h"
#include "supconf.h"
#include "sockinfo.h"
#include "ipmapping.h"
#include "server.h"
#include "log.h"
#include "config.h"
#include "acl.h"
#include "sig.h"
#include "conntrack.h"
#include <sys/resource.h>

#include <fcntl.h>

/*
 * Usual Help Display
 */
void printHelp(char *thisExec)
{
    g_print
        ("\nUsage:\n%s [-n?] -f <configuration file>:\n  -f: Start with the given configuration file\n  -n: Don't go background.\n  -?: This help message\n",
         thisExec);
}

/*
 * Wait for logging to finish and
 * remove the pid file upon normal termination
 */
void myExit(void)
{
    while (!logQueueEmpty())
        sleep(1);
    unlink((gchar *) getConf("PIDFILE"));
    g_warning("Exiting...");
}

void writePid(void)
{
    FILE *F;
    const gchar *pidFileName;

    pidFileName = (const char *) getConf("PIDFILE");
    F = fopen(pidFileName, "w");
    if (F == NULL) {
        local_error("fopen failed on %s: %s", pidFileName,
                    strerror(errno));
    }
    fprintf(F, "%d\n", (gint) getpid());
    fclose(F);
}

/*
 * Main function - reads the configuration and starts server
 */
int main(int argc, char **argv)
{
    gchar c, *confFileName = NULL;
    gboolean isOptErr = FALSE;
    gint f, sock;
    pid_t p;
    gboolean goBackGround = TRUE;

    /* Init threads for glib */
    if (!g_thread_supported())
        g_thread_init(NULL);
    /* Read in command line options */
    while ((c = getopt(argc, argv, "nf:?")) != EOF) {
        switch (c) {
        case 'f':
            confFileName = g_strdup(optarg);
            break;
        case 'n':
            goBackGround = FALSE;
            break;
        case '?':
            printHelp(argv[0]);
            exit(1);
        default:
            isOptErr = TRUE;
        }
    }
    if ((isOptErr) || (confFileName == NULL)) {
        printHelp(argv[0]);
        g_free(confFileName);
        exit(1);
    }
    /* Check the readability of configuration file */
    f = open(confFileName, O_RDONLY);
    if (f == -1) {
        g_critical("%s cannot be opened.", confFileName);
        g_free(confFileName);
        exit(1);
    }
    close(f);
    /* Parse configuration */
    confHash = g_hash_table_new(g_str_hash, g_str_equal);
    readConfFile(confFileName, (confFunc) & parseConf);
    g_free(confFileName);
    /* fork the server, parent exits and the child acts as the server */
    if (goBackGround)
        p = fork();
    else
        p = 0;
    if (p == -1) {
        g_free(confFileName);
        local_error("Can't fork: %s", strerror(errno));
    } else if (p > 0) {
        return (0);
    } else {
        /* set session id */
        if (goBackGround)
            if (setsid() < 0) {
                local_error("setsid() failed");
            }
        /* fork again */
        if (goBackGround)
            p = fork();
        else
            p = 0;
        if (p == -1) {
            g_free(confFileName);
            local_error("Can't fork: %s", strerror(errno));
        } else if (p > 0) {
            return (0);
        } else {
            /* Bind to given socket */
            sock = myBind();
            /* Change to a safer user */
            switchUser();
            /* Write down pid file */
            writePid();
            /* For cleaning up the pid file and log queue */
            g_atexit(myExit);
            /* Init mysql */
            my_init();
            /* Start logging to file
             * WARNING: stdin, stdout and stderr is closed after logging
             * starts. So no standard output is possible beyond this point.
             */
            startLogging();
            /* Start connection tracking */
            connTrack = connTrackInit();
            /* Read acl table into hash */
            aclHash = g_hash_table_new(g_str_hash, g_str_equal);
            readAclList((gchar *) getConf("PROXYTABLE"));
            /* Start signal handler */
            sigListen();
            /* Start the listener */
            startServer(sock);
        }
    }
    g_free(confFileName);
    return 0;
}
