/* wmesa.c */

/*
 * Mesa 3-D graphics library
 * Version:  1.2 beta
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Windows driver by: Mark E. Peterson (markp@ic.mankato.mn.us)
 */

/*
$Id: wmesa.c,v 1.4 1995/06/21 15:39:26 brianp Exp $

$Log: wmesa.c,v $
 * Revision 1.4  1995/06/21  15:39:26  brianp
 * removed #include "wmesaP.h"
 *
 * Revision 1.3  1995/06/21  15:38:10  brianp
 * added struct_wmesa_context, don't need wmesaP.h anymore
 *
 * Revision 1.2  1995/06/09  18:53:39  brianp
 * removed compute_mult_and_shift()
 * dd_read/write_color_span/pixels() take GLubyte arrays instead of GLints
 *
 * Revision 1.1  1995/06/09  18:52:29  brianp
 * Initial revision
 *
 */

/* changed by Gerbert Orasche for use with VRweb 
   June 1995
   Institute For Information Processing And Computer Supported New Media (IICM)
   Graz University Of Technology

   This source only works with Win32s!!!!
*/

#include <stdlib.h>
#include <windows.h>
#include "gl\wmesa.h"
#include "bresenhm.h"
#include "context.h"
#include "dd.h"
#include "quikpoly.h"
#include "xform.h"
#ifndef _MAC
#include "wing.h"
#endif

/* Bit's used for dest: */
#define FRONT_PIXMAP	1
#define BACK_PIXMAP	2
#define BACK_XIMAGE	4


// these tables are used for color lookup with WinG
char unsigned const aDividedBy51Rounded[256] =
{
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
    2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
    2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
};

char unsigned const aDividedBy51[256] =
{
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
    2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
    2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 5, 
};

char unsigned const aModulo51[256] =
{
    0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19,
    20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37,
    38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 0, 1, 2, 3, 4, 5, 6,
    7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
    26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43,
    44, 45, 46, 47, 48, 49, 50, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12,
    13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30,
    31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48,
    49, 50, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17,
    18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35,
    36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 0, 1, 2, 3,
    4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22,
    23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
    41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 0, 
};

//Dither matrix for 8 bit to 2.6 bit halftones.
char unsigned const aHalftone8x8[64] =
{
     0, 38,  9, 47,  2, 40, 11, 50,
    25, 12, 35, 22, 27, 15, 37, 24,
     6, 44,  3, 41,  8, 47,  5, 43,
    31, 19, 28, 15, 34, 21, 31, 18,
     1, 39, 11, 49,  0, 39, 10, 48,
    27, 14, 36, 23, 26, 13, 35, 23,
     7, 46,  4, 43,  7, 45,  3, 42,
    33, 20, 30, 17, 32, 19, 29, 16,
};

// Translates a 2.6 bit-per-pixel halftoned representation into the
// slightly rearranged WinG Halftone Palette.
char unsigned const aWinGHalftoneTranslation[216] =
{
    0,
    29,
    30,
    31,
    32,
    249,
    33,
    34,
    35,
    36,
    37,
    38,
    39,
    40,
    41,
    42,
    43,
    44,
    45,
    46,
    47,
    48,
    49,
    50,
    51,
    52,
    53,
    54,
    55,
    56,
    250,
    250,
    57,
    58,
    59,
    251,
    60,
    61,
    62,
    63,
    64,
    65,
    66,
    67,
    68,
    69,
    70,
    71,
    72,
    73,
    74,
    75,
    76,
    77,
    78,
    79,
    80,
    81,
    82,
    83,
    84,
    85,
    86,
    87,
    88,
    89,
    250,
    90,
    91,
    92,
    93,
    94,
    95,
    96,
    97,
    98,
    99,
    100,
    101,
    102,
    103,
    104,
    105,
    106,
    107,
    108,
    109,
    110,
    111,
    227,
    112,
    113,
    114,
    115,
    116,
    117,
    118,
    119,
    151,
    120,
    121,
    122,
    123,
    124,
    228,
    125,
    126,
    229,
    133,
    162,
    135,
    131,
    132,
    137,
    166,
    134,
    140,
    130,
    136,
    143,
    138,
    139,
    174,
    141,
    142,
    177,
    129,
    144,
    145,
    146,
    147,
    148,
    149,
    150,
    157,
    152,
    153,
    154,
    155,
    156,
    192,
    158,
    159,
    160,
    161,
    196,
    163,
    164,
    165,
    127,
    199,
    167,
    168,
    169,
    170,
    171,
    172,
    173,
    207,
    175,
    176,
    210,
    178,
    179,
    180,
    181,
    182,
    183,
    184,
    185,
    186,
    187,
    188,
    189,
    190,
    191,
    224,
    193,
    194,
    195,
    252,
    252,
    197,
    198,
    128,
    253,
    252,
    200,
    201,
    202,
    203,
    204,
    205,
    206,
    230,
    208,
    209,
    231,
    211,
    212,
    213,
    214,
    215,
    216,
    217,
    218,
    219,
    220,
    221,
    222,
    254,
    223,
    232,
    225,
    226,
    255,
};


// lookuptables for multiplying with 6 and 36 (from 0 to 5)
// also used for color lookup
char unsigned const aTimes6[6] =
{
    0, 6, 12, 18, 24, 30
};

char unsigned const aTimes36[6] =
{
    0, 36, 72, 108, 144, 180
};


// calculates palette index for given RGB value
BYTE GetIndex(const GLfloat color[4]);
//BOOL GetPen(HDC DC,COLORREF color);

/* array for pen cache */
#define PEN_CACHE_SIZE 16
/*struct pen_cache 
{
  COLORREF PenColor;  
  HPEN PenHandle;
} pPenCache[PEN_CACHE_SIZE];
GLuint iLastUsed;
HPEN penOrig;*/

struct wmesa_context
{
  struct gl_context *gl_ctx;	/* the main library context */
  HWND Window;
  HDC Compat_DC;                /* Display context for double buffering. */
  HBITMAP Old_Compat_BM,Compat_BM;            /* Bitmap for double buffering */
  GLuint width, height,ScanWidth;
  GLboolean db_flag;	/* double buffered? */
  GLboolean rgb_flag;	/* RGB mode? */
  GLuint depth;		/* bits per pixel (1, 8, 24, etc) */
  unsigned long pixel;	/* current color index or RGBA pixel value */
  unsigned long clearpixel; /* pixel for clearing the color buffers */
  GLint rmult, gmult, bmult, amult;	/* Multiplier */
	GLint rshift, gshift, bshift, ashift;	/* Bit shifts */
  char *ScreenMem; // WinG memory
  BITMAPINFO *IndexFormat;
  // changed by Gerbert Orasche 170795
  HPALETTE hPal; // Current Palette
  HGLOBAL hRecMem; // memory handle for bitmapinfo structure
};


#ifdef NDEBUG
  #define assert(ignore)	((void) 0)
#else
  void Mesa_Assert(void *Cond,void *File,unsigned Line)
  {
    char Msg[512];
    sprintf(Msg,"%s %s %d",Cond,File,Line);
    MessageBox(NULL,Msg,"Assertion failed.",MB_OK);
    exit(1);
  }
  #define assert(e)	if (!e) Mesa_Assert(#e,__FILE__,__LINE__);
#endif
#define DD_GETDC ((Current->db_flag) ? Current->Compat_DC : GetDC(Current->Window))
#define DD_RELEASEDC if (!Current->db_flag) ReleaseDC(Current->Window,DC)
static WMesaContext Current = NULL;
#ifdef __SYMANTEC_BUGS
  struct dd_function_table DD; 
  #include "vb.h"
  struct vertex_buffer VB;
#endif
#define FLIP(Y)  (Current->height-(Y)-1)
/* Finish all pending operations and synchronize. */
void dd_flush(void)
{
}
/* Return characteristics of the output buffer. */
void dd_buffer_info( GLuint *width, GLuint *height,GLuint *depth)
{
  int New_Size;
  RECT CR;
  int iNewWidth;

  GetClientRect(Current->Window,&CR);
  *width=CR.right;
  *height=CR.bottom;
  *depth = Current->depth;
  New_Size=((*width)!=Current->width) || ((*height)!=Current->height);
  if (New_Size)
  {
    Current->width=*width;

    // align bitmap to 32 bit
    iNewWidth=Current->width;
    if(iNewWidth&0x03)
    {
      iNewWidth&=0xfffffffc;
      iNewWidth+=4;
    }

    Current->ScanWidth=iNewWidth;
    Current->height=*height;

    if (Current->db_flag)
    {
      if (Current->rgb_flag==GL_TRUE)
        Current->Compat_BM=CreateCompatibleBitmap(Current->Compat_DC,Current->width,Current->height);
      else
      {
        Current->IndexFormat->bmiHeader.biWidth=iNewWidth;
        // we are only drawing bottom up WinG bitmaps!!!
        // has to be changed!! (perhaps it is faster, because of not flipping all y!)
        /*if (Current->IndexFormat->bmiHeader.biHeight<0)
          Current->IndexFormat->bmiHeader.biHeight=-Current->height;
        else*/
          Current->IndexFormat->bmiHeader.biHeight=Current->height;
#ifdef _MAC
//#ifdef __PC__
        // allocate bitmap buffer for Mac
        Current->ScreenMem=malloc(iNewWidth*Current->height);
#else
        Current->Compat_BM=WinGCreateBitmap(Current->Compat_DC,Current->IndexFormat,&((void *) Current->ScreenMem));
        if(Current->Compat_BM==NULL)
        {
           MessageBox(NULL,"Error allocating WinG bitmap.","Fatal Error",MB_OK);
           exit(1);
        }
#endif
      }
#ifndef _MAC
//#ifndef __PC__
      DeleteObject(SelectObject(Current->Compat_DC,Current->Compat_BM));
#endif
    }
  }
}

/*
 * Specify which buffer to read from.
 * Input:  mode - as passed to glReadBuffer
 * Return:  new mode or GL_FALSE if error.
 */
GLenum dd_read_buffer(GLenum mode)
{
  GLenum m;
  switch (mode)
  {
    case GL_FRONT:
    case GL_FRONT_LEFT:
    case GL_LEFT:
      m = mode;
	    //Current->Readable = Current->Window;
    break;
    case GL_BACK:
    case GL_BACK_LEFT:
	    m = mode;
      //if (Current->backpixmap)
	    //  Current->readable = Current->backpixmap;
      //else if (Current->backimage)
	    //  Current->readable = None;
    break;
    case GL_AUX0:
    case GL_FRONT_RIGHT:
    case GL_RIGHT:
    case GL_BACK_RIGHT:
    default:
	    return GL_FALSE;
  }
  return m;
}
/*
 * Specify which buffer(s) to draw into.
 * Input:  mode - as passed to glDrawBuffer
 * Return:  new output mode or GL_FALSE if error.
 */
GLenum dd_draw_buffer(GLenum mode)
{
  GLenum m;
  switch (mode)
  {
    case GL_NONE:
      m = GL_NONE;
      //Current->dest = 0;
    break;
    case GL_FRONT:
    case GL_FRONT_LEFT:
	    m = mode;
	    //Current->dest = FRONT_PIXMAP;
	  break;
    case GL_BACK:
    case GL_BACK_LEFT:
	    m = mode;
      //Current->dest = 0;
    break;
    case GL_LEFT:
    case GL_FRONT_AND_BACK:
	    m = mode;
      //Current->dest = FRONT_PIXMAP;
    break;
    case GL_RIGHT:
    case GL_FRONT_RIGHT:
    case GL_BACK_RIGHT:
    case GL_AUX0:
    default:
	   /* Don't change drawbuffer, return error signal */
	   return GL_FALSE;
  }
  return m;
}
/*
 * Set the color index used to clear the color buffer.
 * This implements glClearIndex().
 */
void dd_clear_index(GLuint index)
{
  Current->clearpixel = index;
}
/*
 * Set the color used to clear the color buffer.
 * This implements glClearColor().
 */
void dd_clear_color(const GLfloat color[4])
{
  if(Current->rgb_flag)
    Current->clearpixel=RGB(color[0],color[1],color[2]);
  else
    Current->clearpixel=GetIndex(color);
}

/*
 * Clear the specified region of the color buffer using the clear color
 * or index as specified by one of the two functions above.
 */
void dd_clear(GLboolean all,GLint x, GLint y, GLint width, GLint height )
{
  if (all)
  {
    x=y=0;
    width=Current->width;
    height=Current->height;
  }

  if(Current->rgb_flag)
  {
    HDC DC=DD_GETDC;
    HPEN Pen=CreatePen(PS_SOLID,1,Current->clearpixel);
    HBRUSH Brush=CreateSolidBrush(Current->clearpixel);
    HPEN Old_Pen=SelectObject(DC,Pen);
    HBRUSH Old_Brush=SelectObject(DC,Brush);
  
    Rectangle(DC,x,y,x+width,y+height);
    SelectObject(DC,Old_Pen);
    SelectObject(DC,Old_Brush);
    DeleteObject(Pen);
    DeleteObject(Brush);
    DD_RELEASEDC;
  }
  else
  {
    memset(Current->ScreenMem+y*Current->ScanWidth+x,Current->clearpixel,
           Current->ScanWidth*height);
  }
}

/*
 * Set the pixel logic operation.  Return GL_TRUE if the device driver
 * can perform the operation, otherwise return GL_FALSE.  If GL_FALSE
 * is returned, the logic op will be done in software by Mesa.
 */
GLboolean dd_logicop( GLenum op )
{
  return GL_FALSE;
}

/**********************************************************************/
/*****                Simple rendering functions                  *****/
/**********************************************************************/
/* Set the current color index. */
void dd_index(GLuint index)
{
  Current->pixel=index;
}
/* Set the index mode bitplane mask. */
void dd_index_mask(GLuint mask)
{
  assert(0);
}
/* Set the current RGBA color. */
void dd_color( const GLfloat color[4] )
{
  if(Current->rgb_flag)
    Current->pixel=RGB(color[0]*Current->rmult,
                     color[1]*Current->gmult,
                     color[2]*Current->bmult);
  else
    Current->pixel=GetIndex(color);
}

/* Set the RGBA drawing mask. */
void dd_color_mask( GLboolean rmask, GLboolean gmask,GLboolean bmask, GLboolean amask)
{
  assert(0);
}

/* Write a pixel using current index or color. */
void dd_draw_truepixel( GLint x, GLint y )
{
  HDC DC=DD_GETDC;
  y=FLIP(y);
  SetPixel(DC,x,y,Current->pixel);
  DD_RELEASEDC;
}

void dd_draw_ditherpixel( GLint x, GLint y )
{
  if(y>=Current->height)
    y--;
  Current->ScreenMem[(y*Current->ScanWidth)+x]=Current->pixel;
}

/* Draw a line using the current index or color. */
void dd_draw_line_truecolor( GLint x0, GLint y0, GLint x1, GLint y1 )
{
  HDC DC=DD_GETDC;
  //GetPen(DC,Current->pixel);
  HPEN Pen=CreatePen(PS_SOLID,1,Current->pixel);
  HPEN Old_Pen=SelectObject(DC,Pen);
  y0=FLIP(y0);
  y1=FLIP(y1);
  MoveToEx(DC,x0,y0,NULL);
  LineTo(DC,x1,y1);
  SelectObject(DC,Old_Pen);
  DeleteObject(Pen);
  DD_RELEASEDC;
}

#define BRESENHAM_PLOT(x,y) dd_draw_ditherpixel(x,y);
void dd_draw_line_dithercolor( GLint x0, GLint y0, GLint x1, GLint y1 )
{
  /* horizontal line */
  if(y0==y1)
  {
    if(x1<x0)
    {
      GLint t=x1;
      x1=x0;
      x0=t;
    }
    memset(Current->ScreenMem+(y0*Current->ScanWidth+x0),Current->pixel,x1-x0+1);
  }
  else
    BRESENHAM(x0,y0,x1,y1);
}

/* Draw a convex polygon using the current index or color. */
void dd_draw_polygon_truecolor( GLuint n, GLint x[], GLint y[] )
{
  POINT *Pts=(POINT *) malloc(n*sizeof(POINT));
  GLuint i;
  HDC DC=DD_GETDC;
  HPEN Pen=CreatePen(PS_SOLID,1,Current->pixel);
  HBRUSH Brush=CreateSolidBrush(Current->pixel);
  HPEN Old_Pen=SelectObject(DC,Pen);
  HBRUSH Old_Brush=SelectObject(DC,Brush);
  //GetPen(DC,Current->pixel);
  for (i=0; i<n; i++)
  {
    Pts[i].x=x[i];
    Pts[i].y=FLIP(y[i]);
  }
  Polygon(DC,Pts,n);
  //SelectObject(DC,Old_Pen);
  SelectObject(DC,Old_Brush);
  //DeleteObject(Pen);
  DeleteObject(Brush);
  DD_RELEASEDC;
  free(Pts);
}

void dd_draw_polygon_dithercolor( GLuint n, GLint x[], GLint y[] )
{
  gl_quick_polygon(n,x,y);
}

/**********************************************************************/
/*****             Span-based pixel drawing and reading           *****/
/**********************************************************************/
/* Write a horizontal span of color-index pixels with a boolean mask. */
void dd_write_index_span( GLuint n, GLint x, GLint y,const GLuint index[], const GLubyte mask[] )
{
  GLuint i;
  char *Mem=Current->ScreenMem+y*Current->ScanWidth+x;
  assert(Current->rgb_flag==GL_FALSE);
  for (i=0; i<n; i++)
    if (mask[i])
      Mem[i]=index[i];
}

/*
 * Write a horizontal span of pixels with a boolean mask.  The current
 * color index is used for all pixels.
 */
void dd_write_monoindex_span(GLuint n,GLint x,GLint y,const GLubyte mask[])
{
  GLuint i;
  char *Mem=Current->ScreenMem+y*Current->ScanWidth+x;
  assert(Current->rgb_flag==GL_FALSE);
  for (i=0; i<n; i++)
    if (mask[i])
      Mem[i]=Current->pixel;
}

/* Read a horizontal span of color-index pixels. */
void dd_read_index_span( GLuint n, GLint x, GLint y, GLuint index[])
{
  GLuint i;
  char *Mem=Current->ScreenMem+y*Current->ScanWidth+x;
  assert(Current->rgb_flag==GL_FALSE);
  for (i=0; i<n; i++)
    index[i]=Mem[i];
}

/* Write a horizontal span of color pixels with a boolean mask. */
void dd_write_truecolor_span( GLuint n, GLint x, GLint y,
			  const GLubyte red[], const GLubyte green[],
			  const GLubyte blue[], const GLubyte alpha[],
			  const GLubyte mask[] )
{
  GLuint i;
  HDC DC=DD_GETDC;
  y=FLIP(y);
  for(i=0; i<n; i++)
    if(mask[i])
// on the Mac we take the slow one!
#ifdef _MAC
      SetPixel(DC,x+i,y,RGB(red[i],green[i],blue[i]));
#else
      SetPixelV(DC,x+i,y,RGB(red[i],green[i],blue[i]));
#endif
  DD_RELEASEDC;
}

void dd_write_dithercolor_span( GLuint n, GLint x, GLint y,
			  const GLubyte red[], const GLubyte green[],
			  const GLubyte blue[], const GLubyte alpha[],
			  const GLubyte mask[] )
{
  GLuint i;
  char *pMem=Current->ScreenMem+(y*Current->ScanWidth+x);
  for(i=0; i<n; i++)
  {
    if (mask[i])
    {
      GLubyte cRed=red[i];
      GLubyte cGreen=green[i];
      GLubyte cBlue=blue[i];
      GLubyte cHalfIndex=aHalftone8x8[(y%8)*8+(i%8)];
      GLubyte PaletteIndex;

      // Now, look up each value in the halftone matrix
      // using an 8x8 ordered dither.
      GLubyte RedTemp=aDividedBy51[cRed]+
              (aModulo51[cRed] > cHalfIndex);
      GLubyte GreenTemp=aDividedBy51[cGreen]+
              (aModulo51[cGreen] > cHalfIndex);
      GLubyte BlueTemp=aDividedBy51[cBlue]+
              (aModulo51[cBlue] > cHalfIndex);

      // Recombine the halftoned RGB values into palette indeces
      PaletteIndex=RedTemp+aTimes6[GreenTemp]+aTimes36[BlueTemp];

      // And translate through the WinG Halftone Palette
      // translation vector to give the correct value.
      *pMem=aWinGHalftoneTranslation[PaletteIndex];
    }
    pMem++;
  }
}

/*
 * Write a horizontal span of pixels with a boolean mask.  The current color
 * is used for all pixels.
 */
// changed by Gerbert Orasche 170795 for VRweb
void dd_write_monotruecolor_span( GLuint n, GLint x, GLint y,const GLubyte mask[])
{
  GLuint i;
  HDC DC=DD_GETDC;
  y=FLIP(y);
  for (i=0; i<n; i++)
    if(mask[i])
#ifdef _MAC
      SetPixel(DC,x+i,y,Current->pixel);
#else
      SetPixelV(DC,x+i,y,Current->pixel);
#endif
  DD_RELEASEDC;
}

void dd_write_monodithercolor_span( GLuint n, GLint x, GLint y,const GLubyte mask[])
{
  GLuint i;
  // decide, if WinG needs flipping!
  //y=FLIP(y);
  for (i=0; i<n; i++)
  {
    if(mask[i])
    {
      dd_draw_ditherpixel(x+i,y);
    }
  }
}

/* Read an array of color pixels. */
void dd_read_truecolor_pixels( GLuint n, const GLint x[], const GLint y[],
				  GLubyte red[], GLubyte green[],
				  GLubyte blue[], GLubyte alpha[] )
{
  GLuint i;
  COLORREF Color;
  HDC DC=DD_GETDC;
  assert(Current->rgb_flag==GL_TRUE);
  for (i=0; i<n; i++)
  {
    Color=GetPixel(DC,x[i],FLIP(y[i]));
    red[i]=GetRValue(Color);
    green[i]=GetGValue(Color);
    blue[i]=GetBValue(Color);
    alpha[i]=255;
  }
  DD_RELEASEDC;
  memset(alpha,0,n*sizeof(GLint));
}

void dd_read_dithercolor_pixels( GLuint n, const GLint x[], const GLint y[],
				  GLubyte red[], GLubyte green[],
				  GLubyte blue[], GLubyte alpha[] )
{
  GLuint i;
  COLORREF Color;
  HDC DC=DD_GETDC;
  assert(Current->rgb_flag==GL_TRUE);
  for (i=0; i<n; i++)
  {
    Color=GetPixel(DC,x[i],FLIP(y[i]));
    red[i]=GetRValue(Color);
    green[i]=GetGValue(Color);
    blue[i]=GetBValue(Color);
    alpha[i]=255;
  }
  DD_RELEASEDC;
  memset(alpha,0,n*sizeof(GLint));
}

/* Read a horizontal span of color pixels. */
void dd_read_color_span( GLuint n, GLint x, GLint y,
		         GLubyte red[], GLubyte green[],
			 GLubyte blue[], GLubyte alpha[] )
{
  GLuint i;
  COLORREF Color;
  HDC DC=DD_GETDC;
  assert(Current->rgb_flag==GL_TRUE);
  y=FLIP(y);
  for (i=0; i<n; i++)
  {
    Color=GetPixel(DC,x+i,y);
    red[i]=GetRValue(Color);
    green[i]=GetGValue(Color);
    blue[i]=GetBValue(Color);
    alpha[i]=255;
  }
  DD_RELEASEDC;
  memset(alpha,0,n*sizeof(GLint));
}

/**********************************************************************/
/*****            Array-based pixel drawing and reading           *****/
/**********************************************************************/
/* Write an array of pixels with a boolean mask. */
void dd_write_index_pixels( GLuint n, const GLint x[], const GLint y[],
				  const GLuint index[], const GLubyte mask[] )
{
  GLuint i;
  assert(Current->rgb_flag==GL_FALSE);
  for (i=0; i<n; i++)
  {
    if (mask[i])
    {
      dd_index(index[i]);
      dd_draw_truepixel(x[i],y[i]);
    }
  }
}
/*
 * Write an array of pixels with a boolean mask.  The current color
 * index is used for all pixels.
 */
void dd_write_monoindex_pixels( GLuint n,
				       const GLint x[], const GLint y[],
				       const GLubyte mask[] )
{
  GLuint i;
  assert(Current->rgb_flag==GL_FALSE);
  for (i=0; i<n; i++)
  {
    if (mask[i])
      dd_draw_truepixel(x[i],y[i]);
  }
}
/* Read an array of color index pixels. */
void dd_read_index_pixels( GLuint n,
				  const GLint x[], const GLint y[],
				  GLuint indx[] )
{
  GLuint i;
  //assert(Current->rgb_flag==GL_FALSE);

  char* pMem=Current->ScreenMem;
  int width=Current->ScanWidth;
  for(i=0;i<n;i++)
    indx[i]=*(pMem+y[i]*width+x[i]);
}

/* Write an array of pixels with a boolean mask. */
void dd_write_truecolor_pixels( GLuint n, const GLint x[], const GLint y[],
				   const GLubyte r[], const GLubyte g[],
				   const GLubyte b[], const GLubyte a[],
				   const GLubyte mask[] )
{
  GLuint i;
  HDC DC=DD_GETDC;
  for(i=0; i<n; i++)
    if(mask[i])
      SetPixel(DC,x[i],FLIP(y[i]),RGB(r[i],g[i],b[i]));
  DD_RELEASEDC;
}

void dd_write_dithercolor_pixels( GLuint n, const GLint x[], const GLint y[],
				   const GLubyte r[], const GLubyte g[],
				   const GLubyte b[], const GLubyte a[],
				   const GLubyte mask[] )
{
  GLuint i;
  int width=Current->ScanWidth;
  char* pMem=Current->ScreenMem;
  for(i=0;i<n;i++)
    if(mask[i])
      *(pMem+y[i]*width+x[i])=aWinGHalftoneTranslation[
        aDividedBy51Rounded[r[i]]+
        aTimes6[aDividedBy51Rounded[g[i]]]+
        aTimes36[aDividedBy51Rounded[b[i]]]];
}

/*
 * Write an array of pixels with a boolean mask.  The current color
 * is used for all pixels.
 */
void dd_write_monotruecolor_pixels( GLuint n,
				       const GLint x[], const GLint y[],
    		       const GLubyte mask[] )
{
  GLuint i;
  HDC DC=DD_GETDC;
  for(i=0;i<n;i++)
    if (mask[i])
      SetPixel(DC,x[i],FLIP(y[i]),Current->pixel);
  DD_RELEASEDC;
}

void dd_write_monodithercolor_pixels( GLuint n,
				       const GLint x[], const GLint y[],
    		       const GLubyte mask[] )
{
  GLuint i;
  int width=Current->ScanWidth;
  char* pMem=Current->ScreenMem;
  for(i=0;i<n;i++)
    if(mask[i])
      *(pMem+y[i]*width+x[i])=Current->pixel;
}

#define PAL_SIZE 256
void GetPalette(HPALETTE Pal,RGBQUAD *aRGB)
{
	int i;
  LOGPALETTE* pBuffer;
  pBuffer=malloc(256*sizeof(PALETTEENTRY)+4);

  //pBuffer->palVersion=0x300;
  //pBuffer->palNumEntries=256;

  GetPaletteEntries(Pal,0,256,pBuffer->palPalEntry);

  for (i=0; i<PAL_SIZE; i++)
  {
    aRGB[i].rgbRed=pBuffer->palPalEntry[i].peRed;
    aRGB[i].rgbGreen=pBuffer->palPalEntry[i].peGreen;
    aRGB[i].rgbBlue=pBuffer->palPalEntry[i].peBlue;
    aRGB[i].rgbReserved=pBuffer->palPalEntry[i].peFlags;
  }
  free(pBuffer);
}

WMesaContext WMesaCreateContext(HWND hWnd,HPALETTE Pal,GLboolean rgb_flag,GLboolean db_flag)
{
  BITMAPINFO *Rec;
  HDC DC;
  HBITMAP hbmWinG;
  RECT CR;
  WMesaContext c;
  int iMallocSize;
  int iNewWidth;

  // allocate a new Mesa context */
  c = (struct wmesa_context *) calloc(1,sizeof(struct wmesa_context));
  if (!c)
    return NULL;
  c->gl_ctx = gl_new_context(NULL);
  c->Window=hWnd;
  // changed by Gerbert Orasche for VRweb
  c->gl_ctx->RGBAflag = GL_TRUE;
  db_flag=GL_TRUE; // WinG requires double buffering
  c->pixel = 0;
  c->gl_ctx->RedScale   = c->rmult = 255;
  c->gl_ctx->GreenScale = c->gmult = 255;
  c->gl_ctx->BlueScale  = c->bmult = 255;
  c->gl_ctx->AlphaScale = c->amult = 255;
  c->rshift=0;
  c->gshift=0;
  c->bshift=0;
  c->ashift=0;
  if (rgb_flag==GL_FALSE)
  {
    //mode for true color on <= 256 color displays
    c->rgb_flag = GL_FALSE;

    /* set function pointers */
    DD.draw_pixel=dd_draw_ditherpixel;
    DD.draw_line=dd_draw_line_dithercolor;
    DD.draw_polygon=dd_draw_polygon_dithercolor;
    DD.write_color_span=dd_write_dithercolor_span;
    DD.write_monocolor_span=dd_write_monodithercolor_span;
    DD.write_color_pixels=dd_write_dithercolor_pixels;
    DD.write_monocolor_pixels=dd_write_monodithercolor_pixels;
    DD.write_index_span=dd_write_index_span;
    DD.write_monoindex_span=dd_write_monoindex_span;
    DD.write_index_pixels=dd_write_index_pixels;
    DD.write_monoindex_pixels=dd_write_monoindex_pixels;
    DD.read_index_span=dd_read_index_span;
    DD.read_color_span=dd_read_color_span;
    DD.read_index_pixels=dd_read_index_pixels;
    DD.read_color_pixels=dd_read_dithercolor_pixels;
  }
  else
  {
    //GLuint iCount;

    // mode for truecolor
    c->rgb_flag = GL_TRUE;

    /* set function pointers */
    DD.draw_pixel=dd_draw_truepixel;
    DD.draw_line=dd_draw_line_truecolor;
    DD.draw_polygon=dd_draw_polygon_truecolor;
    DD.write_color_span=dd_write_truecolor_span;
    DD.write_monocolor_span=dd_write_monotruecolor_span;
    DD.write_color_pixels=dd_write_truecolor_pixels;
    DD.write_monocolor_pixels=dd_write_monotruecolor_pixels;
    DD.write_index_span=dd_write_index_span;
    DD.write_monoindex_span=dd_write_monoindex_span;
    DD.write_index_pixels=dd_write_index_pixels;
    DD.write_monoindex_pixels=dd_write_monoindex_pixels;
    DD.read_index_span=dd_read_index_span;
    DD.read_color_span=dd_read_color_span;
    DD.read_index_pixels=dd_read_index_pixels;
    DD.read_color_pixels=dd_read_truecolor_pixels;
  }

  GetClientRect(c->Window,&CR);
  c->width=CR.right;
  c->height=CR.bottom;
  if (db_flag)
  {
    c->db_flag = 1;
    c->gl_ctx->Color.DrawBuffer = GL_BACK;
    c->gl_ctx->Pixel.ReadBuffer = GL_BACK;
    /* Double buffered */
    if (c->rgb_flag==GL_TRUE)
    {
      DC=GetDC(c->Window);
      c->Compat_DC=CreateCompatibleDC(DC);
      c->Compat_BM=CreateCompatibleBitmap(DC,c->width,c->height);
      ReleaseDC(c->Window,DC);
      c->Old_Compat_BM=SelectObject(c->Compat_DC,c->Compat_BM);
    }
    else
    {
#ifndef _MAC
//#ifndef __PC__
      c->Compat_DC=WinGCreateDC();
      if(c->Compat_DC==NULL)
      {
         MessageBox(NULL,"Fatal Error","Error getting WinG device context.",MB_OK);
         exit(1);
      }
#endif

      // allocating Bitmap Info Header
      iMallocSize=sizeof(BITMAPINFO)+(PAL_SIZE-1)*sizeof(RGBQUAD);
      c->hRecMem=GlobalAlloc(GMEM_FIXED,iMallocSize);
      if(c->hRecMem==NULL)
      {
         MessageBox(NULL,"Fatal Error","Error allocating bitmapinfoheader.",MB_OK);
         exit(1);
      }

      Rec=(BITMAPINFO*)GlobalLock(c->hRecMem);
      if(Rec==NULL)
      {
         MessageBox(NULL,"Fatal Error","Error locking bitmapinfoheader.",MB_OK);
         exit(1);
      }

      memset((void*)Rec,0,iMallocSize);

      // set up header
      Rec->bmiHeader.biSize=sizeof(BITMAPINFOHEADER);
      Rec->bmiHeader.biCompression=BI_RGB;
      c->hPal=Pal;
      GetPalette(Pal,Rec->bmiColors);

      // align bitmap to 32 bit
      iNewWidth=c->width;
      if(iNewWidth&0x03)
      {
        iNewWidth&=0xfffffffc;
        iNewWidth+=4;
      }

#ifdef _MAC
//#ifdef __PC__
      Rec->bmiHeader.biWidth=iNewWidth;
      Rec->bmiHeader.biHeight=c->height;
      c->ScreenMem=malloc(iNewWidth*c->height);
      if((c->ScreenMem)==0)
        MessageBox(NULL,"Could not allocate image buffer\n","Fatal Error.",MB_OK);
      Rec->bmiHeader.biPlanes=1;
      Rec->bmiHeader.biBitCount=8;
      Rec->bmiHeader.biCompression=BI_RGB;
      Rec->bmiHeader.biClrUsed=PAL_SIZE;
#else
      // we will ignore bitmap orientation for now...
      WinGRecommendDIBFormat(Rec);

      Rec->bmiHeader.biWidth=iNewWidth;
      Rec->bmiHeader.biHeight=c->height;
      //Rec->bmiHeader.biHeight*=c->height;
      //Rec->bmiHeader.biClrUsed=PAL_SIZE;
      if (Rec->bmiHeader.biPlanes!=1 || Rec->bmiHeader.biBitCount!=8)
      {
        MessageBox(NULL,"This code presumes a 256 color, single plane, WinG Device.\n","Warning.",MB_OK);
        //exit(1);
      }


      // now allocate the double buffer
      hbmWinG=WinGCreateBitmap(c->Compat_DC,Rec,(void *)&(c->ScreenMem));
      if(hbmWinG==NULL)
      {
         MessageBox(NULL,"Error allocating WinG bitmap.","Fatal Error",MB_OK);
         DeleteDC(c->Compat_DC);
         exit(1);
      }
      c->Compat_BM=hbmWinG;

      c->Old_Compat_BM=SelectObject(c->Compat_DC,c->Compat_BM);
      WinGSetDIBColorTable(c->Compat_DC,0,PAL_SIZE,Rec->bmiColors);
      SetStretchBltMode(c->Compat_DC,COLORONCOLOR);
#endif

      c->IndexFormat=Rec;
      c->ScanWidth=iNewWidth;
    }
  }
  else
  {
    /* Single Buffered */
    c->db_flag = 0;
    c->gl_ctx->Color.DrawBuffer = GL_FRONT;
    c->gl_ctx->Pixel.ReadBuffer = GL_FRONT;
  }
  c->gl_ctx->DBflag = db_flag;
  return c;
}

void WMesaDestroyContext( WMesaContext c )
{
   gl_destroy_context( c->gl_ctx );
   if (c->db_flag)
   {
     // release resources
#ifdef _MAC
//#ifdef __PC__
     free(c->ScreenMem);
#else
     DeleteObject(SelectObject(c->Compat_DC,c->Old_Compat_BM));
#endif
     DeleteDC(c->Compat_DC);
     GlobalUnlock(c->hRecMem);
     GlobalFree(c->hRecMem);
   }
   free( (void *) c );
}
void WMesaMakeCurrent( WMesaContext c , HDC draw_dc)
{
   gl_set_context( c->gl_ctx );
   Current = c;
   if (Current->gl_ctx->Viewport.Width==0)
      /* initialize viewport to window size */
      gl_viewport( 0, 0, Current->width, Current->height );
}

void WMesaSwapBuffers(HDC draw_dc)
{
  // changed 130795 Gerbert Orasche
  if (Current->db_flag)
  {
    if (Current->rgb_flag)
      BitBlt(draw_dc,0,0,Current->width,Current->height,Current->Compat_DC,0,0,SRCCOPY);
    else
#ifdef _MAC
//#ifdef __PC__
      SetDIBitsToDevice(draw_dc,        // handle of device context 
      0,                                // x-coordinate of upper-left corner of dest. rect. 
      0,                                // y-coordinate of upper-left corner of dest. rect. 
      Current->width,                   // source rectangle width 
      Current->height,                  // source rectangle height
      0,                                // x-coordinate of lower-left corner of source rect. 
      0,                                // y-coordinate of lower-left corner of source rect. 
      //Current->height,                  // first scan line in array 
      0,                  // first scan line in array 
      Current->height,                  // number of scan lines 
      (const void*)Current->ScreenMem,  // address of array with DIB bits
      Current->IndexFormat,             // address of structure with bitmap info.
      DIB_RGB_COLORS);                  // RGB or palette indices 
#else
      WinGBitBlt(draw_dc,0,0,Current->width,Current->height,Current->Compat_DC,0,0);
#endif
  }
}

void WMesaPaletteChange(HPALETTE Pal)
{
  if (Current && Current->rgb_flag==GL_FALSE)
  {
    Current->hPal=Pal;
    GetPalette(Pal,Current->IndexFormat->bmiColors);
#ifndef _MAC
    WinGSetDIBColorTable(Current->Compat_DC,0,PAL_SIZE,Current->IndexFormat->bmiColors);
#endif
  }
}

// Gerbert Orasche
// IICM 170795
// calculates palette index for given RGB value
// assuming a 6x6x6 rgb color cube
BYTE GetIndex(const GLfloat color[4])
{
  return aWinGHalftoneTranslation[aDividedBy51Rounded[(BYTE)(color[0]*255.0F)]+
         aTimes6[aDividedBy51Rounded[(BYTE)(color[1]*255.0F)]]+
         aTimes36[aDividedBy51Rounded[(BYTE)(color[2]*255.0F)]]];
}
