/* draw.c */

/*
 * Mesa 3-D graphics library
 * Version:  1.2
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
$Id: draw.c,v 1.23 1995/06/20 16:29:36 brianp Exp $

$Log: draw.c,v $
 * Revision 1.23  1995/06/20  16:29:36  brianp
 * don't scale Z to integer values here
 * introduced new triangle, polygon area code but not used yet
 *
 * Revision 1.22  1995/06/02  18:59:31  brianp
 * added special case for computing plane equation of 3-sided polygon
 *
 * Revision 1.21  1995/06/02  13:55:09  brianp
 * implemented vertex/primitive buffering
 *
 * Revision 1.20  1995/05/26  19:32:02  brianp
 * replace many assignments in gl_color() with macros
 *
 * Revision 1.19  1995/05/22  21:02:41  brianp
 * Release 1.2
 *
 * Revision 1.18  1995/05/22  21:00:36  brianp
 * tried initial vertex buffering
 *
 * Revision 1.17  1995/05/17  13:17:22  brianp
 * changed default CC.Mode value to allow use of real OpenGL headers
 * removed need for CC.MajorMode variable
 *
 * Revision 1.16  1995/05/15  16:07:33  brianp
 * moved StippleCounter init to new place
 *
 * Revision 1.15  1995/05/12  16:29:18  brianp
 * added #define for NULL
 *
 * Revision 1.14  1995/03/27  20:31:26  brianp
 * new Texture.Enabled scheme
 *
 * Revision 1.13  1995/03/24  15:31:23  brianp
 * introduced VB
 *
 * Revision 1.12  1995/03/23  17:10:12  brianp
 * changed render_point() to render_points(n)
 *
 * Revision 1.11  1995/03/09  21:42:09  brianp
 * new ModelViewInv matrix logic
 *
 * Revision 1.10  1995/03/09  20:08:04  brianp
 * introduced TRANSFORM_POINT and TRANSFORM_NORMAL macros
 *
 * Revision 1.9  1995/03/04  19:29:44  brianp
 * 1.1 beta revision
 *
 * Revision 1.8  1995/03/02  19:10:00  brianp
 * fixed a texgen bug
 *
 * Revision 1.7  1995/02/27  22:48:48  brianp
 * modified for PB
 *
 * Revision 1.6  1995/02/27  15:08:04  brianp
 * added Vcolor/Vindex scheme
 *
 * Revision 1.5  1995/02/26  22:58:43  brianp
 * more zero-area polygon work
 *
 * Revision 1.4  1995/02/26  21:59:43  brianp
 * *** empty log message ***
 *
 * Revision 1.3  1995/02/25  22:07:56  brianp
 * relaxed test for zero-area polygons, still not perfect though
 *
 * Revision 1.2  1995/02/24  15:23:36  brianp
 * removed initialization of d from render_point()
 * added RasterColor code to gl_rasterpos()
 *
 * Revision 1.1  1995/02/24  14:20:43  brianp
 * Initial revision
 *
 */


/*
 * Draw points, lines, and polygons.
 */


#ifdef DEBUG
#  include <stdio.h>
#endif
#include <assert.h>
#include <math.h>
#include "clip.h"
#include "context.h"
#include "draw.h"
#include "feedback.h"
#include "fog.h"
#include "light.h"
#include "lines.h"
#include "list.h"
#include "macros.h"
#include "pb.h"
#include "points.h"
#include "polygons.h"
#include "texture.h"
#include "vb.h"
#include "xform.h"



#ifndef NULL
#  define NULL 0
#endif



/*
 * Render the points stored in VB.Eye[0..n-1].
 */
static void render_points( GLuint n )
{
   register GLuint i;

   for (i=0;i<n;i++) {
      register GLfloat d;
      register GLfloat ndc_x, ndc_y, ndc_z;

      /* Clip against user clipping planes */
      if (CC.Transform.AnyClip) {
	 if (gl_userclip_point(VB.Eye[i])==0)
	   continue;
      }

      /* Transform vertex from eye to clip coordinates:  clip = Proj * eye */
      TRANSFORM_POINT( VB.Clip[i], CC.ProjectionMatrix, VB.Eye[i] );

      /* Clip against view volume */
      if (GL_VIEWCLIP_POINT( VB.Clip[i] )==0)
	continue;

      /* Transform from clip to ndc:  ndc = clip / W */
#ifdef DEBUG
      assert( VB.Clip[i][3] != 0.0 );
#endif
      d = 1.0F / VB.Clip[i][3];
      ndc_x = VB.Clip[i][0] * d;
      ndc_y = VB.Clip[i][1] * d;
      ndc_z = VB.Clip[i][2] * d;

      /* Transform from NDC to window coords */
      VB.Win[i][0] = MAP_X( ndc_x );
      VB.Win[i][1] = MAP_Y( ndc_y );
      VB.Win[i][2] = MAP_Z( ndc_z );

#ifdef NEW
      VB.WinX[i] = (GLint) ((ndc_x * CC.Viewport.Sx + CC.Viewport.Tx) * SUB_PIX_SCALE);
      VB.WinY[i] = (GLint) ((ndc_y * CC.Viewport.Sy + CC.Viewport.Ty) * SUB_PIX_SCALE);
      VB.WinZ[i] = (GLint) ((ndc_z * CC.Viewport.Sz + CC.Viewport.Tz) * DEPTH_SCALE);
#endif
      (*CC.PointFunc)( i );
   }
}



/*
 * Render a line segment from VB.Eye[v1] to VB.Eye[v2].
 */
static void render_line( GLuint v1, GLuint v2 )
{
   GLfloat d;
   GLfloat ndc_x, ndc_y, ndc_z;
   GLuint provoking_vertex;

   /* which vertex dictates the color when flat shading: */
   provoking_vertex = v2;

   /*
    * Clipping may introduce new vertices.  New vertices will be stored
    * in the vertex buffer arrays starting with location VB.Free.  After
    * we've rendered the line, these extra vertices can be overwritten.
    */
   VB.Free = VB_MAX;

   /* Clip against user clipping planes */
   if (CC.Transform.AnyClip) {
      if (gl_userclip_line( &v1, &v2 )==0)
	return;
   }

   /* Apply projection matrix:  clip = Proj * eye */
   TRANSFORM_POINT( VB.Clip[v1], CC.ProjectionMatrix, VB.Eye[v1] );
   TRANSFORM_POINT( VB.Clip[v2], CC.ProjectionMatrix, VB.Eye[v2] );

   /* Clip against view volume */
   if (gl_viewclip_line( &v1, &v2 )==0)
      return;

   /* Transform from clip coords to ndc:  ndc = clip / W */
#ifdef DEBUG
   assert( VB.Clip[v1][3] != 0.0 );
   assert( VB.Clip[v2][3] != 0.0 );
#endif
   d = 1.0F / VB.Clip[v1][3];
   ndc_x = VB.Clip[v1][0] * d;
   ndc_y = VB.Clip[v1][1] * d;
   ndc_z = VB.Clip[v1][2] * d;

   /* Map ndc coord to window coords. */
   VB.Win[v1][0] = MAP_X( ndc_x );
   VB.Win[v1][1] = MAP_Y( ndc_y );
   VB.Win[v1][2] = MAP_Z( ndc_z );

   /* Transform from clip coords to ndc:  ndc = clip / W */
   d = 1.0F / VB.Clip[v2][3];
   ndc_x = VB.Clip[v2][0] * d;
   ndc_y = VB.Clip[v2][1] * d;
   ndc_z = VB.Clip[v2][2] * d;

   /* Map ndc coord to window coords. */
   VB.Win[v2][0] = MAP_X( ndc_x );
   VB.Win[v2][1] = MAP_Y( ndc_y );
   VB.Win[v2][2] = MAP_Z( ndc_z );

   (*CC.LineFunc)( v1, v2, provoking_vertex );
}




/*
 * Render a polygon.  The vertices of the polygon are in the VB.Eye array.  
 * Input:  n - number of vertices
 *         vlist - list of vertices in the polygon.
 *         odd_flag - if non-zero, reverse the orientation of the polygon
 */
static void render_polygon( GLuint n, GLuint vlist[], GLuint odd_flag )
{
   GLuint i, j;
   GLuint provoking_vertex;
   GLuint facing;

   /* which vertex dictates the color when flat shading: */
   provoking_vertex = (CC.Mode==GL_POLYGON) ? vlist[0] : vlist[n-1];

   /*
    * Clipping may introduce new vertices.  New vertices will be stored
    * in the vertex buffer arrays starting with location VB.Free.  After
    * we've rendered the polygon, these extra vertices can be overwritten.
    */
   VB.Free = VB_MAX;

   /* Clip against user clipping planes in eye coord space. */
   if (CC.Transform.AnyClip) {
      n = gl_userclip_polygon( n, vlist );
      if (n<3)
	return;
   }

   /* Transform vertices from eye to clip coordinates:  clip = Proj * eye */
   for (i=0;i<n;i++) {
      j = vlist[i];
      TRANSFORM_POINT( VB.Clip[j], CC.ProjectionMatrix, VB.Eye[j] );
   }

   /* Clip against view volume in clip coord space */
   n = gl_viewclip_polygon( n, vlist );
   if (n<3)
     return;

   /* Transform vertices from clip to ndc:  ndc = clip / W */
   for (i=0;i<n;i++) {
      GLfloat d, ndc_x, ndc_y, ndc_z;
      j = vlist[i];
#ifdef DEBUG
      assert( VB.Clip[j][3] != 0.0 );
#endif
      d = 1.0F / VB.Clip[j][3];
      ndc_x = VB.Clip[j][0] * d;
      ndc_y = VB.Clip[j][1] * d;
      ndc_z = VB.Clip[j][2] * d;

      /* Transform ndc coord to a window coord.  Note that window Z values */
      /* are scaled to integer Z buffer values. */
      VB.Win[j][0] = MAP_X( ndc_x );
      VB.Win[j][1] = MAP_Y( ndc_y );
      VB.Win[j][2] = MAP_Z( ndc_z );

#ifdef NEW
      VB.WinX[j] = (GLint) (MAP_X( ndc_x ) * SUB_PIX_SCALE);
      VB.WinY[j] = (GLint) (MAP_Y( ndc_y ) * SUB_PIX_SCALE);
      VB.WinZ[j] = (GLint) (MAP_Z( ndc_z ) * DEPTH_SCALE);
#endif
   }

#ifdef AREA
   /* compute area of polygon to find orientation */
   {
      GLfloat area = 0.0F;
      GLuint prevj;

      prevj = vlist[n-1];
      for (i=0;i<n;i++) {
	 j = vlist[i];
	 area += (VB.Win[j][0]-VB.Win[prevj][0]) * (VB.Win[j][1]+VB.Win[prevj][1]);
/*	 area += (VB.Win[prevj][0]*VB.Win[j][1]) - (VB.Win[j][0]*VB.Win[prevj][1]);*/
	 prevj = j;
      }
      area *= 0.5;
      if (odd_flag) {
	 area = -area;
      }
      printf("area=%g\n", area);
   }
#endif

   /* Compute the plane equation of polygon: ax + by + cz = d */
   if (CC.ComputePlane) {
      if (n==3) {
	 /* a triangle */
	 GLuint j0 = vlist[0];
	 GLuint j1 = vlist[1];
	 GLuint j2 = vlist[2];
	 GLfloat ex = VB.Win[j1][0] - VB.Win[j0][0];
	 GLfloat ey = VB.Win[j1][1] - VB.Win[j0][1];
	 GLfloat ez = VB.Win[j1][2] - VB.Win[j0][2];
	 GLfloat fx = VB.Win[j2][0] - VB.Win[j0][0];
	 GLfloat fy = VB.Win[j2][1] - VB.Win[j0][1];
	 GLfloat fz = VB.Win[j2][2] - VB.Win[j0][2];
	 CC.PlaneA = ey*fz-ez*fy;
	 CC.PlaneB = ez*fx-ex*fz;
	 CC.PlaneC = ex*fy-ey*fx;
	 if (CC.PlaneA==0.0F && CC.PlaneB==0.0F && CC.PlaneC==0.0F) {
	    /* zero-area, don't render */
	    return;
	 }
      }
      else {
	 /* n-sided polygon */
	 /* This is tricky because of co-located points and parallel edges! */
	 GLfloat ex, ey, ez;  /* e and f are edge vectors */
	 GLfloat fx, fy, fz;
	 GLuint j0, j1, j2;

	 j0 = vlist[0];

	 /* find edge e such that:  e != <0,0,0> */
	 i = 1;
	 while (1) {
	    j1 = vlist[i++];
	    ex = VB.Win[j1][0] - VB.Win[j0][0];
	    ey = VB.Win[j1][1] - VB.Win[j0][1];
	    ez = VB.Win[j1][2] - VB.Win[j0][2];
	    if (i==n) {
	       /* degenerate polygon */
	       return;
	    }
	    if (ex!=0.0F || ey!=0.0F /*  || ez!=0.0F  */  )  break;
	 }

	 /* find edge f such that:  e X f != <0,0,0> */
	 while (1) {
	    j2 = vlist[i++];
	    fx = VB.Win[j2][0] - VB.Win[j0][0];
	    fy = VB.Win[j2][1] - VB.Win[j0][1];
	    fz = VB.Win[j2][2] - VB.Win[j0][2];
	    if (fx!=ex && fy!=ey) {
	       CC.PlaneA = ey*fz-ez*fy;
	       CC.PlaneB = ez*fx-ex*fz;
	       CC.PlaneC = ex*fy-ey*fx;
	       if (CC.PlaneA>0.001F || CC.PlaneA<-0.001F
		   || CC.PlaneB>0.001F || CC.PlaneB<-0.001F
		   || CC.PlaneC>0.001F || CC.PlaneC<-0.001F) {
		  break;
	       }
	    }
	    if (i==n) {
	       /* degenerate polygon */
	       return;
	    }
	 }
      }
      
      if (CC.PlaneC==0.0F) {
	 /* polygon is perpindicular to view plane, don't draw it */
	 return;
      }

      /* compute facing:  0=front, 1=back */
      facing = (CC.PlaneC<0.0F) ^ odd_flag ^ (CC.Polygon.FrontFace==GL_CW);

      if (CC.Polygon.CullFlag && facing==(CC.Polygon.CullFaceMode==GL_BACK)) {
	 /* culled */
	 return;
      }
      j = vlist[0];
      CC.PlaneD = CC.PlaneA*VB.Win[j][0]
	        + CC.PlaneB*VB.Win[j][1] + CC.PlaneC*VB.Win[j][2];
   }
   else {
      CC.PlaneA = CC.PlaneB = CC.PlaneD = 0.0F;
      CC.PlaneC = 1.0F;
      facing = 0;
   }

   if (facing==1 && CC.Light.Enabled && CC.Light.Model.TwoSide) {
      /* use back color or index */
      VB.Color = VB.Bcolor;
      VB.Index = VB.Bindex;
   }
   else {
      /* use front color or index */
      VB.Color = VB.Fcolor;
      VB.Index = VB.Findex;
   }

   (*CC.PolygonFunc)( n, vlist, provoking_vertex );
}



#ifdef NEW
/*
 * This is an experiment, not currently in use!
 */
static void render_triangle( GLuint v0, GLuint v1, GLuint v2,
			     GLuint provoking_vertex, GLuint odd_flag )
{
   GLuint i, j, n;
   GLuint facing;
   GLuint vlist[1000];

   /*
    * Clipping may introduce new vertices.  New vertices will be stored
    * in the vertex buffer arrays starting with location VB.Free.  After
    * we've rendered the polygon, these extra vertices can be overwritten.
    */
   VB.Free = VB_MAX;

   /* Clip against user clipping planes in eye coord space. */
/* TODO:
   if (CC.Transform.AnyClip) {
      n = gl_userclip_polygon( n, vlist );
      if (n<3)
	return;
   }
*/
   /* Transform vertices from eye to clip coordinates:  clip = Proj * eye */
   TRANSFORM_POINT( VB.Clip[v0], CC.ProjectionMatrix, VB.Eye[v0] );
   TRANSFORM_POINT( VB.Clip[v1], CC.ProjectionMatrix, VB.Eye[v1] );
   TRANSFORM_POINT( VB.Clip[v2], CC.ProjectionMatrix, VB.Eye[v2] );

   /* Clip against view volume in clip coord space */
/* TODO:
   n = gl_viewclip_polygon( n, vlist );
   if (n<3)
     return;
*/

   {
      GLfloat d;

      /* Transform vertices from clip to ndc:  ndc = clip / W */
#ifdef DEBUG
      assert( VB.Clip[v0][3] != 0.0 );
      assert( VB.Clip[v1][3] != 0.0 );
      assert( VB.Clip[v2][3] != 0.0 );
#endif
      d = 1.0F / VB.Clip[v0][3];
      VB.Win[v0][0] = MAP_X( VB.Clip[v0][0] * d );
      VB.Win[v0][1] = MAP_Y( VB.Clip[v0][1] * d );
      VB.Win[v0][2] = MAP_Z( VB.Clip[v0][2] * d );
      d = 1.0F / VB.Clip[v1][3];
      VB.Win[v1][0] = MAP_X( VB.Clip[v1][0] * d );
      VB.Win[v1][1] = MAP_Y( VB.Clip[v1][1] * d );
      VB.Win[v1][2] = MAP_Z( VB.Clip[v1][2] * d );
      d = 1.0F / VB.Clip[v2][3];
      VB.Win[v2][0] = MAP_X( VB.Clip[v2][0] * d );
      VB.Win[v2][1] = MAP_Y( VB.Clip[v2][1] * d );
      VB.Win[v2][2] = MAP_Z( VB.Clip[v2][2] * d );
   }


   /* Compute the plane equation of polygon: ax + by + cz = d */
   if (CC.ComputePlane) {
      GLfloat ex, ey, ez;  /* e and f are edge vectors */
      GLfloat fx, fy, fz;

      ex = VB.Win[v1][0] - VB.Win[v0][0];
      ey = VB.Win[v1][1] - VB.Win[v0][1];
      ez = VB.Win[v1][2] - VB.Win[v0][2];
      fx = VB.Win[v2][0] - VB.Win[v0][0];
      fy = VB.Win[v2][1] - VB.Win[v0][1];
      fz = VB.Win[v2][2] - VB.Win[v0][2];
      CC.PlaneA = ey*fz-ez*fy;
      CC.PlaneB = ez*fx-ex*fz;
      CC.PlaneC = ex*fy-ey*fx;
      /* TODO: verify */
      if ((CC.PlaneA==0.0 && CC.PlaneB==0.0) || CC.PlaneC==0.0) {
	 /* polygon is perpindicular to view plane, don't draw it */
	 /* or polygon has zero area */
	 return;
      }

      /* compute facing:  0=front, 1=back */
      facing = (CC.PlaneC<0.0F) ^ odd_flag ^ (CC.Polygon.FrontFace==GL_CW);

      if (CC.Polygon.CullFlag && facing==(CC.Polygon.CullFaceMode==GL_BACK)) {
	 /* culled */
	 return;
      }
      CC.PlaneD = CC.PlaneA*VB.Win[v0][0]
	        + CC.PlaneB*VB.Win[v0][1] + CC.PlaneC*VB.Win[v0][2];
   }
   else {
      CC.PlaneA = CC.PlaneB = CC.PlaneC = CC.PlaneD = 1.0;
      facing = 0;
   }

   if (facing==1 && CC.Light.Enabled && CC.Light.Model.TwoSide) {
      /* use back color or index */
      VB.Color = VB.Bcolor;
      VB.Index = VB.Bindex;
   }
   else {
      /* use front color or index */
      VB.Color = VB.Fcolor;
      VB.Index = VB.Findex;
   }

   vlist[0] = v0;
   vlist[1] = v1;
   vlist[2] = v2;
   n = 3;

   (*CC.PolygonFunc)( n, vlist, provoking_vertex );
}
#endif



/*
 * Either the vertex buffer is full (VB.Count==VB_MAX) or glEnd() has been
 * called.  Render the primitives defined by the vertices and reset the
 * buffer.
 * Input:  alldone - GL_TRUE = caller is glEnd()
 *                   GL_FALSE = calling because buffer is full.
 */
static void flush_vb( GLboolean alldone )
{
   GLuint vlist[VB_MAX];
   GLuint i;

   switch (CC.Mode) {
      case GL_POINTS:
         render_points( VB.Count );
	 VB.Count = 0;
	 break;

      case GL_LINES:
	 VB.Count = VB.Count - (VB.Count & 1);  /* Make sure count is even */
	 for (i=0;i<VB.Count;i+=2) {
	    render_line( i, i+1 );
	    CC.StippleCounter = 0;
	 }
	 VB.Count = 0;
	 break;

      case GL_LINE_STRIP:
	 if (VB.Count>1) {
	    for (i=1;i<VB.Count;i++) {
	       render_line( i-1, i );
	    }
	    if (VB.Count==VB_MAX) {
	       /* Copy last vertex to first pos */
	       COPY_4V( VB.Eye[0], VB.Eye[VB_MAX-1] );
	       COPY_4V( VB.Fcolor[0], VB.Fcolor[VB_MAX-1] );
	       VB.Findex[0] = VB.Findex[VB_MAX-1];
	    }
	 }
	 VB.Count = 1;
         break;

      case GL_LINE_LOOP:
	 if (VB.Flushes==0) {
	    /* start at 0th vertex */
	    for (i=1;i<VB.Count;i++) {
	       render_line( i-1, i );
	    }
	 }
	 else {
	    /* start at 1th vertex */
	    for (i=2;i<VB.Count;i++) {
	       render_line( i-1, i );
	    }
	 }
	 if (alldone) {
	    render_line( VB.Count-1, 0 );
	 }
	 else {
	    assert(VB.Count==VB_MAX);
	    /* recycle the vertex list */
	    COPY_4V( VB.Eye[1], VB.Eye[VB_MAX-1] );
	    COPY_4V( VB.Fcolor[1], VB.Fcolor[VB_MAX-1] );
	    VB.Findex[1] = VB.Findex[VB_MAX-1];
	    VB.Count = 2;
	 }
	 VB.Flushes++;
         break;

      case GL_TRIANGLES:
	 for (i=2;i<VB.Count;i+=3) {
	    vlist[0] = i-2;
	    vlist[1] = i-1;
	    vlist[2] = i-0;
	    render_polygon( 3, vlist, 0 );
	 }
	 VB.Count = 0;
	 break;

      case GL_TRIANGLE_STRIP:
	 for (i=2;i<VB.Count;i++) {
	    vlist[0] = i-2;
	    vlist[1] = i-1;
	    vlist[2] = i-0;
	    render_polygon( 3, vlist, i&1 );
	    /*render_triangle( i-2, i-1, i-0, i, i&1 );*/
	 }
	 if (VB.Count==VB_MAX) {
	    /* Buffer was filled, recycle the vertex list */
	    COPY_4V( VB.Eye[0], VB.Eye[VB_MAX-2] );
	    COPY_4V( VB.Eye[1], VB.Eye[VB_MAX-1] );
	    COPY_4V( VB.Fcolor[0], VB.Fcolor[VB_MAX-2] );
	    COPY_4V( VB.Fcolor[1], VB.Fcolor[VB_MAX-1] );
	    COPY_4V( VB.Bcolor[0], VB.Bcolor[VB_MAX-2] );
	    COPY_4V( VB.Bcolor[1], VB.Bcolor[VB_MAX-1] );
	    VB.Findex[0] = VB.Findex[VB_MAX-2];
	    VB.Findex[1] = VB.Findex[VB_MAX-1];
	    VB.Bindex[0] = VB.Bindex[VB_MAX-2];
	    VB.Bindex[1] = VB.Bindex[VB_MAX-1];
	    VB.Edgeflag[0] = VB.Edgeflag[VB_MAX-2];
	    VB.Edgeflag[1] = VB.Edgeflag[VB_MAX-1];
	    VB.Count = 2;
	 }
	 break;

      case GL_TRIANGLE_FAN:
	 for (i=2;i<VB.Count;i++) {
	    vlist[0] = 0;
	    vlist[1] = i-1;
	    vlist[2] = i;
	    render_polygon( 3, vlist, 0 );
	 }
	 if (VB.Count==VB_MAX) {
	    /* Buffer was filled, copy last vertex to position [1] */
	    COPY_4V( VB.Eye[1], VB.Eye[VB_MAX-1] );
	    COPY_4V( VB.Fcolor[1], VB.Fcolor[VB_MAX-1] );
	    COPY_4V( VB.Bcolor[1], VB.Bcolor[VB_MAX-1] );
	    VB.Findex[1] = VB.Findex[VB_MAX-1];
	    VB.Bindex[1] = VB.Bindex[VB_MAX-1];
	    VB.Edgeflag[1] = VB.Edgeflag[VB_MAX-1];
	    VB.Count = 2;
	 }
	 break;

      case GL_QUADS:
	 for (i=3;i<VB.Count;i+=4) {
	    vlist[0] = i-3;
	    vlist[1] = i-2;
	    vlist[2] = i-1;
	    vlist[3] = i-0;
	    render_polygon( 4, vlist, 0 );
	 }
	 VB.Count = 0;
	 break;

      case GL_QUAD_STRIP:
	 for (i=3;i<VB.Count;i+=2) {
	    vlist[0] = i-2;
	    vlist[1] = i-3;
	    vlist[2] = i-1;
	    vlist[3] = i-0;
	    render_polygon( 4, vlist, 1 );
	 }
	 if (VB.Count==VB_MAX) {
	    /* recycle the vertex list */
	    COPY_4V( VB.Eye[0], VB.Eye[VB_MAX-2] );
	    COPY_4V( VB.Eye[1], VB.Eye[VB_MAX-1] );
	    COPY_4V( VB.Fcolor[0], VB.Fcolor[VB_MAX-2] );
	    COPY_4V( VB.Fcolor[1], VB.Fcolor[VB_MAX-1] );
	    COPY_4V( VB.Bcolor[0], VB.Bcolor[VB_MAX-2] );
	    COPY_4V( VB.Bcolor[1], VB.Bcolor[VB_MAX-1] );
	    VB.Findex[0] = VB.Findex[VB_MAX-2];
	    VB.Findex[1] = VB.Findex[VB_MAX-1];
	    VB.Bindex[0] = VB.Bindex[VB_MAX-2];
	    VB.Bindex[1] = VB.Bindex[VB_MAX-1];
	    VB.Edgeflag[0] = VB.Edgeflag[VB_MAX-2];
	    VB.Edgeflag[1] = VB.Edgeflag[VB_MAX-1];
	    VB.Count = 2;
	 }
	 break;

      case GL_POLYGON:
	 for (i=0;i<VB.Count;i++) {
	    vlist[i] = i;
	 }
	 render_polygon( VB.Count, vlist, 0 );
	 if (VB.Count==VB_MAX) {
	    COPY_4V( VB.Eye[1], VB.Eye[VB_MAX-1] );
	    COPY_4V( VB.Fcolor[1], VB.Fcolor[VB_MAX-1] );
	    COPY_4V( VB.Bcolor[1], VB.Bcolor[VB_MAX-1] );
	    VB.Findex[1] = VB.Findex[VB_MAX-1];
	    VB.Bindex[1] = VB.Bindex[VB_MAX-1];
	    VB.Edgeflag[1] = VB.Edgeflag[VB_MAX-1];
	 }
	 VB.Count = 2;
	 break;

      default:
	 assert(1==0);
   }
}



/*
 * Process a glVertex* call.
 * Input:  v - an (x,y,z,w) object coordinate
 */
void gl_vertex( const GLfloat v[4] )
{
   GLfloat normal[3];

   if (VB.Count==VB_MAX) {
      flush_vb( GL_FALSE );
   }

   /* transform vertex to eye coords and save in Veye:  eye = ModelView * v */
   TRANSFORM_POINT( VB.Eye[VB.Count], CC.ModelViewMatrix, v );

   /* compute vertex color */
   if (CC.Light.Enabled==GL_TRUE) {
      TRANSFORM_NORMAL( normal, CC.Current.Normal );

      /* compute lighting */
      if (CC.RGBAflag) {
	 gl_color_shade( VB.Eye[VB.Count], normal, CC.LightTwoSide,
			 VB.Fcolor[VB.Count], VB.Bcolor[VB.Count] );
      }
      else {
	 GLuint front_index, back_index;
	 gl_index_shade( VB.Eye[VB.Count], normal, CC.LightTwoSide,
			 &front_index, &back_index );
	 VB.Findex[VB.Count] = (GLfloat) front_index;
	 VB.Bindex[VB.Count] = (GLfloat) back_index;
      }
   }
   else {
      /* copy current color or index */
      if (CC.RGBAflag) {
	 COPY_4V( VB.Fcolor[VB.Count], CC.Current.Color );
      }
      else {
	 VB.Findex[VB.Count] = (GLfloat) CC.Current.Index;
      }
   }

   if (CC.Fog.Enabled && CC.Hint.Fog!=GL_NICEST) {
      /* apply per-vertex fog */
      GLfloat z = VB.Eye[VB.Count][2];
      if (z<0.0F)  z = -z;
      if (CC.RGBAflag) {
	 gl_fog_color_vertex( z, VB.Fcolor[VB.Count] );
	 gl_fog_color_vertex( z, VB.Bcolor[VB.Count] );
      }
      else {
	 gl_fog_index_vertex( z, &VB.Findex[VB.Count] );
	 gl_fog_index_vertex( z, &VB.Bindex[VB.Count] );
      }
   }

   if (CC.Texture.Enabled) {
      GLfloat tc[4];
      /* automatic texture coordinate generation */
      /* TODO: xform normal if sphere map */
      gl_texgen( v, VB.Eye[VB.Count], normal, CC.Current.TexCoord );

      /* transform current texture coordinate by texture matrix and save it */
      /* tc = TexMat * TexCoord */
      TRANSFORM_POINT( tc, CC.TextureMatrix, CC.Current.TexCoord );
      /* Divide by w??? */
      VB.Vs[VB.Count] = tc[0];
      VB.Vt[VB.Count] = tc[1];
   }

   VB.Edgeflag[VB.Count] = CC.Current.EdgeFlag;
   VB.Count++;
}



/*
 * Process a vertex produced by an evaluator.
 * Input:  v - vertex
 *         n - normal
 *         c - color
 *         i - color index
 *         t - texture coordinate
 */
void gl_eval_vertex( const GLfloat v[4], const GLfloat n[3],
		     const GLfloat c[4], GLfloat i, GLfloat t[4] )
{
   GLfloat normal[3];

   if (VB.Count==VB_MAX) {
      flush_vb( GL_FALSE );
   }

   /* transform vertex to eye coords and save in vertex list:  eye = M * v */
   TRANSFORM_POINT( VB.Eye[VB.Count], CC.ModelViewMatrix, v );

   /* compute vertex color */
   if (CC.Light.Enabled==GL_TRUE) {
      TRANSFORM_NORMAL( normal, n );

      /* compute lighting */
      if (CC.RGBAflag) {
	 gl_color_shade( VB.Eye[VB.Count], normal, CC.LightTwoSide,
			 VB.Fcolor[VB.Count], VB.Bcolor[VB.Count] );
      }
      else {
	 GLuint front_index, back_index;
	 gl_index_shade( VB.Eye[VB.Count], normal, CC.LightTwoSide,
			 &front_index, &back_index );
	 VB.Findex[VB.Count] = (GLfloat) front_index;
	 VB.Bindex[VB.Count] = (GLfloat) back_index;
      }
   }
   else {
      if (CC.RGBAflag) {
	 VB.Fcolor[VB.Count][0] = c[0];
	 VB.Fcolor[VB.Count][1] = c[1];
	 VB.Fcolor[VB.Count][2] = c[2];
	 VB.Fcolor[VB.Count][3] = c[3];
      }
      else {
	 VB.Findex[VB.Count] = i;
      }
   }

   if (CC.Fog.Enabled && CC.Hint.Fog!=GL_NICEST) {
      /* apply per-vertex fog */
      GLfloat z = VB.Eye[VB.Count][2];
      if (z<0.0F)  z = -z;
      if (CC.RGBAflag) {
	 gl_fog_color_vertex( z, VB.Fcolor[VB.Count] );
	 gl_fog_color_vertex( z, VB.Bcolor[VB.Count] );
      }
      else {
	 gl_fog_index_vertex( z, &VB.Findex[VB.Count] );
	 gl_fog_index_vertex( z, &VB.Bindex[VB.Count] );
      }
   }

   if (CC.Texture.Enabled) {
      GLfloat tc[4];
      /* automatic texture coordinate generation */
      /* TODO: xform normal if sphere map */
      gl_texgen( v, VB.Eye[VB.Count], normal, t );

      /* transform current texture coordinate by texture matrix and save it */
      TRANSFORM_POINT( tc, CC.TextureMatrix, t );
      /* divide by w??? */
      VB.Vs[VB.Count] = tc[0];
      VB.Vt[VB.Count] = tc[1];
   }

   VB.Edgeflag[VB.Count] = CC.Current.EdgeFlag;
   VB.Count++;
}




void gl_rasterpos( const GLfloat v[4] )
{
   GLfloat eye[4], clip[4], ndc[3], d;

   /* transform v to eye coords:  eye = ModelView * v */
   TRANSFORM_POINT( eye, CC.ModelViewMatrix, v );

   /* compute lighting */
   if (CC.Light.Enabled==GL_TRUE) {
      GLfloat eyenorm[3];
      if (!CC.ModelViewInvValid) {
	 gl_compute_modelview_inverse();
      }
      TRANSFORM_NORMAL( eyenorm, CC.Current.Normal );
      if (CC.RGBAflag) {
	 gl_color_shade( eye, eyenorm, 0, CC.Current.RasterColor, NULL );
      }
      else {
	 gl_index_shade( eye, eyenorm, 0, &CC.Current.RasterIndex, NULL );
      }
   }
   else {
      /* copy current color or index */
      if (CC.RGBAflag) {
	 COPY_4V( CC.Current.RasterColor, CC.Current.Color );
      }
      else {
	 CC.Current.RasterIndex = CC.Current.Index;
      }
   }

   /* clip to user clipping planes */
   if (gl_userclip_point(eye)==0) {
      CC.Current.RasterPosValid = GL_FALSE;
      return;
   }

   /* compute raster distance */
   CC.Current.RasterDistance = (GLfloat)
                      sqrt( eye[0]*eye[0] + eye[1]*eye[1] + eye[2]*eye[2] );

   /* apply projection matrix:  clip = Proj * eye */
   TRANSFORM_POINT( clip, CC.ProjectionMatrix, eye );

   /* clip to view volume */
   if (gl_viewclip_point( clip )==0) {
      CC.Current.RasterPosValid = GL_FALSE;
      return;
   }

   /* ndc = clip / W */
#ifdef DEBUG
   assert( clip[3]!=0.0 );
#endif
   d = 1.0F / clip[3];
   ndc[0] = clip[0] * d;
   ndc[1] = clip[1] * d;
   ndc[2] = clip[2] * d;

   CC.Current.RasterPos[0] = MAP_X( ndc[0] );
   CC.Current.RasterPos[1] = MAP_Y( ndc[1] );
   CC.Current.RasterPos[2] = MAP_Z( ndc[2] );
   CC.Current.RasterPos[3] = clip[3];    /* TODO: verify */

   CC.Current.RasterPosValid = GL_TRUE;

   if (CC.Texture.Enabled) {
      /* TODO: Current.RasterTexCoord */

   }

   if (CC.RenderMode==GL_SELECT) {
      /* TODO: is this correct? */
      CC.HitFlag = GL_TRUE;
      if (CC.Current.RasterPos[2] < CC.HitMinZ) {
	 CC.HitMinZ = CC.Current.RasterPos[2];
      }
      if (CC.Current.RasterPos[2] < CC.HitMaxZ) {
	 CC.HitMaxZ = CC.Current.RasterPos[2];
      }
   }

}



void gl_index( GLuint index )
{
   CC.Current.Index = index;
}



void gl_color( const GLfloat color[4] )
{
   COPY_4V( CC.Current.Color, color );

   if (CC.Light.ColorMaterialEnabled) {
      switch (CC.Light.ColorMaterialMode) {
	 case GL_EMISSION:
	    if (CC.Light.ColorMaterialFace==GL_FRONT ||
		CC.Light.ColorMaterialFace==GL_FRONT_AND_BACK) {
	       COPY_4V( CC.Light.Material.Emission[0], color );
	    }
	    if (CC.Light.ColorMaterialFace==GL_BACK ||
		CC.Light.ColorMaterialFace==GL_FRONT_AND_BACK) {
	       COPY_4V( CC.Light.Material.Emission[1], color );
	    }
	    break;
	 case GL_AMBIENT:
	    if (CC.Light.ColorMaterialFace==GL_FRONT ||
		CC.Light.ColorMaterialFace==GL_FRONT_AND_BACK) {
	       COPY_4V( CC.Light.Material.Ambient[0], color );
	    }
	    if (CC.Light.ColorMaterialFace==GL_BACK ||
		CC.Light.ColorMaterialFace==GL_FRONT_AND_BACK) {
	       COPY_4V( CC.Light.Material.Ambient[1], color );
	    }
	    break;
	 case GL_DIFFUSE:
	    if (CC.Light.ColorMaterialFace==GL_FRONT ||
		CC.Light.ColorMaterialFace==GL_FRONT_AND_BACK) {
	       COPY_4V( CC.Light.Material.Diffuse[0], color );
	    }
	    if (CC.Light.ColorMaterialFace==GL_BACK ||
		CC.Light.ColorMaterialFace==GL_FRONT_AND_BACK) {
	       COPY_4V( CC.Light.Material.Diffuse[1], color );
	    }
	    break;
	 case GL_SPECULAR:
	    if (CC.Light.ColorMaterialFace==GL_FRONT ||
		CC.Light.ColorMaterialFace==GL_FRONT_AND_BACK) {
	       COPY_4V( CC.Light.Material.Specular[0], color );
	    }
	    if (CC.Light.ColorMaterialFace==GL_BACK ||
		CC.Light.ColorMaterialFace==GL_FRONT_AND_BACK) {
	       COPY_4V( CC.Light.Material.Specular[1], color );
	    }
	    break;
	 case GL_AMBIENT_AND_DIFFUSE:
	    if (CC.Light.ColorMaterialFace==GL_FRONT ||
		CC.Light.ColorMaterialFace==GL_FRONT_AND_BACK) {
	       COPY_4V( CC.Light.Material.Ambient[0], color );
	       COPY_4V( CC.Light.Material.Diffuse[0], color );
	    }
	    if (CC.Light.ColorMaterialFace==GL_BACK ||
		CC.Light.ColorMaterialFace==GL_FRONT_AND_BACK) {
	       COPY_4V( CC.Light.Material.Ambient[1], color );
	       COPY_4V( CC.Light.Material.Diffuse[1], color );
	    }
	    break;
      }
   }

}



void gl_begin( GLenum p )
{
   CC.Mode = p;
   VB.Count = VB.Flushes = 0;

   if (!CC.ModelViewInvValid) {
      gl_compute_modelview_inverse();
   }

   switch (CC.Mode) {
      case GL_POINTS:
         if (CC.PointFunc==NULL) {
	    gl_set_point_function();
	 }
	 CC.LightTwoSide = 0;
	 VB.Color = VB.Fcolor;
	 VB.Index = VB.Findex;
	 gl_init_pb( GL_POINT );
	 break;
      case GL_LINES:
      case GL_LINE_STRIP:
      case GL_LINE_LOOP:
         if (CC.LineFunc==NULL) {
	    gl_set_line_function();
	 }
	 CC.LightTwoSide = 0;
	 VB.Color = VB.Fcolor;
	 VB.Index = VB.Findex;
	 CC.StippleCounter = 0;
	 gl_init_pb( GL_LINE );
         break;
      case GL_TRIANGLES:
      case GL_TRIANGLE_STRIP:
      case GL_TRIANGLE_FAN:
      case GL_QUADS:
      case GL_QUAD_STRIP:
      case GL_POLYGON:
         if (CC.PolygonFunc==NULL) {
	    gl_set_polygon_function();
	 }
	 CC.LightTwoSide = (GLuint) CC.Light.Model.TwoSide;
	 gl_init_pb( GL_POLYGON );
         break;
      default:
	 gl_error( GL_INVALID_ENUM, "glBegin" );
	 CC.Mode = GL_BITMAP;
   }
}



void gl_end( void )
{
   if (CC.Mode==GL_BITMAP) {
      /* glEnd without glBegin */
      gl_error( GL_INVALID_OPERATION, "glEnd" );
      return;
   }

   flush_vb( GL_TRUE );
   gl_flush_pb();
   PB.primitive = CC.Mode = GL_BITMAP;  /* Default mode */
}




/*
 *
 * Public functions
 *
 */

void glBegin( GLenum p )
{
   if (CC.CompileFlag) {
      gl_save_begin( p );
   }
   if (CC.ExecuteFlag) {
      gl_begin( p );
   }
}



void glEnd( void )
{
   if (CC.CompileFlag) {
      gl_save_end();
   }
   if (CC.ExecuteFlag) {
      gl_end();
   }
}
