/*******************************************************************************
*
* University of Western Australia
* Department of Computer Science
* Copyright (c) University of Western Australia
*
* SYSTEM :              VIP
* RELEASE:		3
* SUBSYSTEM:            VIP
* MODULE:		localhisteq.c - Program for localized histogram equalization
*					Reference:     Gonzalez and Woods  - 
*					      "Digital Image Processing" pp 173-183.
* REVISION:             3.2
* AUTHOR:               PK 
* CREATION DATE:        01 January 1994
* REVISION DATE:	3/3/94
*
********************************************************************************
*
* REVISION LOG
*
* REVISION:		3.2
* REVISION DATE:	03 March 1993
* COMMENT:		Fixed casts from Allocate_Image() etc.
* BY:			CFF
*
* REVISION:		3.1
* REVISION DATE:	03 March 1993
* COMMENT:		ANSIfied and SCCS'd
* BY:			CFF
*
* "@(#)localhisteq.c	3.2 3/3/94"
*
*******************************************************************************/

#include <stdio.h>
#include "vip.h"
#include "vipiofn.h"
#include "misc.h"


/*- Calculate_Histogram -----------------------------------------------

Function to calculate a local histogram for a region in an image.

input: image  - pointer to image data.
       top   } - coordinates of top, left corner of local region.
       left  }
       window - size of local square region in the image.

output: hist - histogram of image data in region.

Warning:  No checks of image bounds are made.

----------------------------------------------------------------------*/

void    Calculate_Histogram(IMAGE * image, long hist[256], int top, int left, int window)
{
    int     row, col, i, bottom, right;

    bottom = top + window - 1;
    right = left + window - 1;

/* clear histogram */

    for (i = 0; i < 256; i++)
	hist[i] = 0;

/* now set up histogram */

    for (row = top; row <= bottom; row++)
	for (col = left; col <= right; col++)
	    hist[image->i.c[row][col]]++;

    return;
}


/*- Move_Histogram_Right1 ----------------------------------------------

Function to recursively update a local histogram for a region that has
been moved one column to the right.

input: image  - pointer to image data.
       hist   - existing histogram of region.
       top   } - coordinates of top, left corner of existing local region.
       left  }
       window - size of local square region in the image.

output: hist is updated to correspond to the histogram of the region
        shifted one column to the right.

Warning:  No checks of image bounds are made.

----------------------------------------------------------------------*/

void    Move_Histogram_Right1(IMAGE * image, long hist[256], int top, int left,
            int window)
{
    int     row, bottom, newright;

    bottom = top + window - 1;
    newright = left + window;

if(bottom >= image->rows) printf("bottom %d\n",bottom);
if(newright >= image->rows) printf("newright %d\n",newright);

/* eliminate left column from histogram and add in data for the new column
on the right */

    for (row = top; row <= bottom; row++) {
	hist[image->i.c[row][left]]--;
	hist[image->i.c[row][newright]]++;
    }
    return;
}


/*- Make_LUT --------------------------------------------------------

Function to form a cumulative histogram that can be used (after
rescaling) as a look up table for reassigning grey levels in an image
when performing histogram equalization.

--------------------------------------------------------------------*/

void    Make_LUT(long hist[256], long LUT[256])
{
    int     i;

    LUT[0] = hist[0];
    for (i = 1; i < 256; i++)
	LUT[i] = LUT[i - 1] + hist[i];

    return;
}

/*- Local_Histogram_Equalize ------------------------------------------

Function to perform local histogram equalization.  A square windowed
region is indexed across the image and histogram equalization applied
to the centre element of the region.

---------------------------------------------------------------------*/

IMAGE  *Local_Histogram_Equalize(IMAGE * image, int window)
{
    int     row, col, r, c, halfwindow, grey;
    long    hist[256], LUT[256];
    float   rescale;
    IMAGE  *outimage;

    if (image->type != BYTETYPE) {
	VIP_Error_Msg("Local_Histogram_Equalize: image must be BYTETYPE");
	return (NULL);
    }


    if (window > MIN(image->rows, image->cols)) {
	VIP_Error_Msg("Local_Histogram_Equalize: window is larger than the image!");
	return (NULL);
    } else     if (window < 2) {
	VIP_Error_Msg("Local_Histogram_Equalize: window size must be 2 pixels or more");
	return (NULL);
    }

    outimage = (IMAGE * )Allocate_Image(0, 0, image->rows, image->cols, BYTETYPE);

    rescale = 255.0 / (float) (window * window);
    halfwindow = window / 2;

/* special code for top row */

    Calculate_Histogram(image, hist, 0, 0, window);
    Make_LUT(hist, LUT);
    for (r = 0; r <= halfwindow; r++) {
	for (c = 0; c <= halfwindow; c++) {
	    grey = (int) ((float) LUT[image->i.c[r][c]] * rescale + 0.5);
	    outimage->i.c[r][c] = grey > 255 ? 255 : grey;
	}
    }

    for (col = 1; col < image->cols - window; col++) {
	c = col + halfwindow;
	Move_Histogram_Right1(image, hist, 0, col - 1, window);
	Make_LUT(hist, LUT);
	for (r = 0; r <= halfwindow; r++) {
	    grey = (int) ((float) LUT[image->i.c[r][c]] * rescale + 0.5);
	    outimage->i.c[r][c] = grey > 255 ? 255 : grey;
	}
    }

    for (r = 0; r <= halfwindow; r++) {
	for (c = image->cols - halfwindow - 1; c < image->cols; c++) {
	    grey = (int) ((float) LUT[image->i.c[r][c]] * rescale + 0.5);
	    outimage->i.c[r][c] = grey > 255 ? 255 : grey;
	}
    }



/* now the centre of the image */

    for (row = 1; row < image->rows - window; row++) {
	r = row + halfwindow;
	Calculate_Histogram(image, hist, row, 0, window);
	Make_LUT(hist, LUT);

/* values on the left hand side */

	for (c = 0; c <= halfwindow; c++) {
	    grey = (int) ((float) LUT[image->i.c[r][c]] * rescale + 0.5);
	    outimage->i.c[r][c] = grey > 255 ? 255 : grey;
	}

/* values across the middle */

	for (col = 1; col < image->cols - window; col++) {
	    c = col + halfwindow;
	    Move_Histogram_Right1(image, hist, row, col - 1, window);
	    Make_LUT(hist, LUT);
	    grey = (int) ((float) LUT[image->i.c[r][c]] * rescale + 0.5);
	    outimage->i.c[r][c] = grey > 255 ? 255 : grey;
	}

/* values on the right hand side */

	for (c = image->cols - halfwindow - 1; c < image->cols; c++) {
	    grey = (int) ((float) LUT[image->i.c[r][c]] * rescale + 0.5);
	    outimage->i.c[r][c] = grey > 255 ? 255 : grey;
	}

    }

/* now special code for the bottom row */

    row = image->rows - window;
    Calculate_Histogram(image, hist, row, 0, window);
    Make_LUT(hist, LUT);
    for (r = image->rows - halfwindow - 1; r < image->rows; r++) {
	for (c = 0; c <= halfwindow; c++) {
	    grey = (int) ((float) LUT[image->i.c[r][c]] * rescale + 0.5);
	    outimage->i.c[r][c] = grey > 255 ? 255 : grey;
	}
    }

    for (col = 1; col < image->cols - window; col++) {
	c = col + halfwindow;
	Move_Histogram_Right1(image, hist, row, col - 1, window);
	Make_LUT(hist, LUT);
	for (r = image->rows - halfwindow - 1; r < image->rows; r++) {
	    grey = (int) ((float) LUT[image->i.c[r][c]] * rescale + 0.5);
	    outimage->i.c[r][c] = grey > 255 ? 255 : grey;
	}
    }

    for (r = image->rows - halfwindow - 1; r < image->rows; r++) {
	for (c = image->cols - halfwindow - 1; c < image->cols; c++) {
	    grey = (int) ((float) LUT[image->i.c[r][c]] * rescale + 0.5);
	    outimage->i.c[r][c] = grey > 255 ? 255 : grey;
	}
    }

    return (outimage);
}

/*----------------------------------------------------------------*/

void    main(int argc, char *argv[])
{
    IMAGE  *inimage, *outimage;

    if (argc != 4) {
	printf("Usage:histeq inimage window_size outimage\n");
	return;
    }

    inimage = (IMAGE *)Read_Image(argv[1]);
    if (!inimage) {
	printf("could not open %s \n", argv[1]);
	return;
    }

    outimage = Local_Histogram_Equalize(inimage, atoi(argv[2]));
    if (!outimage) {
	Free_Image(inimage);
	return;
    }

    Write_Image(outimage, argv[3]);
    Free_Image(inimage);
    Free_Image(outimage);
    return;
}

