/*******************************************************************************
*
* University of Western Australia
* Department of Computer Science
* Copyright (c) University of Western Australia
*
* SYSTEM :              VIP
* RELEASE:		3
* SUBSYSTEM:            VIP
* MODULE:		histe.c - Apply histogram equalization on an image.
* REVISION:             3.2
* AUTHOR:               DH
* CREATION DATE:        31 Oct 1989
* REVISION DATE:	8/5/93
*
********************************************************************************
*
* REVISION LOG
*
* REVISION:             3.2
* REVISION DATE:        5 August 1993
* COMMENT:              Changes to Get_Intensity to avoid grey level wraparound
* BY:                   PK
*
* REVISION:		3.1
* REVISION DATE:	11 July 1992
* COMMENT:		ANSIfied and SCCS'd
* BY:			CFF
*
* REVISION:
* REVISION DATE:	24 Jan 1992
* COMMENT:		NEWVIP
* BY:			DH
*
*******************************************************************************/

#ifndef lint
static char *sccs_id = "@(#)histe.c	3.2 8/5/93";

#endif



#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "vip.h"

void            Usage();
void            Pperror( char *, char * );
static int      Freq( struct struct_vip_IMAGE *, float[] );
static int      Get_Intensity( float[], int[] );
static IMAGE    *Change_Intensity( struct struct_vip_IMAGE *, 
                                   int[] );


#define EPSILON 0.005


/*- Usage -----------------------------------------------------------

Print error message and exit program.

--------------------------------------------------------------------*/

void    Usage()
{
    (void) fprintf(stderr, "usage: histe [input_image] [-o output_image]\n");
    exit(1);
}


/*- Pperror ---------------------------------------------------------

Print error message and exit program.

--------------------------------------------------------------------*/

void    Pperror(m1, m2)
char   *m1, *m2;
{
    (void) fprintf(stderr, "histe: %s %s.\n", m1, m2);
    exit(1);
}


/*- Freq ------------------------------------------------------------

Calculate frequency, relative frequency, and cumulative frequency
of an image.

--------------------------------------------------------------------*/

static int Freq(im, cumf)
IMAGE  *im;
float   cumf[256];
{
    register int r, c;
    int     f[256];
    float   relf[256];
    long    imsiz;
    float   sum;

    for (r = 0; r < 256; r++)
	f[r] = 0;
    if (!im) {
	VIP_Error_Msg("Freq: input image is NULL");
	return (0);
    }
    switch (im->type) {
    case BYTETYPE:
	imsiz = (long) im->rows * (long) im->cols;
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--)
		f[(int) im->i.c[r][c]]++;
	for (r = 255; r >= 0; r--)
	    relf[r] = (float) f[r] / (float) imsiz;
	for (r = 0, sum = 0; r < 256; r++) {
	    sum += relf[r];
	    cumf[r] = sum;
	    if (sum > 1.0)
		break;
	}
	for (; r < 256; r++)
	    cumf[r] = 1.0;

	if (fabs(cumf[255] - 1.0) > EPSILON) {
	    VIP_Error_Msg("Freq: Error in calculating cumulative frequency");
	    return (0);
	}
	break;
    default:
	VIP_Error_Msg("Freq: only able to process images of BYTETYPE");
	return (0);
    }
    return (1);
}


/*- Get_Intensity ---------------------------------------------------

Calculate the mapping to the new intensity value.
both 'cumf' and 'inten' should be arrays of size 256.

--------------------------------------------------------------------*/

static int Get_Intensity(cumf, inten)
float   cumf[];
int     inten[];
{
    register int i;

    for (i = 0; i < 256; i++) {
	inten[i] = (int) (cumf[i] * 255.0 + 0.5);	/* scaleup and round to
							 * nearest integer */
	if (inten[i] > 255)
	    inten[i] = 255;
    }
    return (1);
}


/*- Change_Intensity ------------------------------------------------

Modify intensity value in input image.

--------------------------------------------------------------------*/

static IMAGE *Change_Intensity(im, inten)
IMAGE  *im;
int     inten[];
{
    register int r, c;
    int     pix;
    IMAGE  *outim;

    if (!(outim = (IMAGE *) Allocate_Image(0, 0, im->rows, im->cols, im->type))) {
	VIP_Error_Msg("Change_Intensity: out of memory");
	return (NULL);
    }

    Copy_Header(im, outim);

    switch (im->type) {
    case BYTETYPE:
	for (r = im->rows - 1; r >= 0; r--)
	    for (c = im->cols - 1; c >= 0; c--) {
		pix = im->i.c[r][c];
		outim->i.c[r][c] = inten[pix];
	    }
	break;
    default:
	VIP_Error_Msg("Change_Intensity: only able to process images of BYTETYPE");
	Free_Image(outim);
	outim = NULL;
    }
    return (outim);
}


/*- Main ------------------------------------------------------------

Main body of the program.

--------------------------------------------------------------------*/

main(argc, argv)
char   *argv[];
{
    int     arg;
    char   *infile = NULL, *outfile = NULL;
    IMAGE  *im, *outim;
    int     inten[256];
    float   cumf[256];

    for (arg = 1; arg < argc; arg++)
	if (argv[arg][0] == '-')
	    switch (argv[arg][1]) {
	    case 'o':
		if (++arg < argc)
		    outfile = argv[arg];
		else
		    Usage();
		break;
	    default:
		Usage();
	    }
	else if (!infile)
	    infile = argv[arg];
	else
	    Usage();

    if (!(im = (IMAGE *) Read_Image(infile)))
	Pperror("Error in reading image from file", infile);

    if (!Freq(im, cumf))
	exit(1);
    if (!Get_Intensity(cumf, inten))
	exit(1);
    if (!(outim = (IMAGE *) Change_Intensity(im, inten)))
	exit(1);
    Write_Image(outim, outfile);
    exit(0);
}
