/*******************************************************************************
*
* University of Western Australia
* Department of Computer Science
* Copyright (c) University of Western Australia
*
* SYSTEM :              VIP
* RELEASE:		3
* SUBSYSTEM:            VIP
* MODULE:		convolve.c - Convolve an image with a mask 
*				     (also an image).		
* REVISION:             3.1
* AUTHOR:               CA
* CREATION DATE:        9 Apr 1987
* REVISION DATE:	7/10/92        
*
********************************************************************************
*
* REVISION LOG
*
* REVISION:		3.1
* REVISION DATE:	11 July 1992
* COMMENT:		ANSIfied and SCCS'd
* BY:			CFF
*
* REVISION:		
* REVISION DATE:	28 Jan 1992
* COMMENT:		NEWVIP
* BY:			DH
*
*******************************************************************************/

#ifndef lint
static char *sccs_id = "@(#)convolve.c	3.1 7/10/92";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "vip.h"

void Usage();
void Pperror(char *e);
IMAGE  *Convolve_Binary_Mask(IMAGE *im, IMAGE *mask);
IMAGE  *Convolve_Mask(IMAGE *im, IMAGE *mask);

/*- Usage -----------------------------------------------------------

Print error message and exit program.

--------------------------------------------------------------------*/

void Usage()
{
    (void) fprintf(stderr, "usage: convolve mask_file [-b] [-i image_file] [-o matrix_file]\n");
    exit(1);
}


/*- Pperror ---------------------------------------------------------

Print error message and exit program.

--------------------------------------------------------------------*/

void Pperror(e)
char   *e;
{
    (void) fprintf(stderr, "convolve: %s\n", e);
    exit(1);
}


/*- Convolve_Binary_Mask --------------------------------------------

Output the convolution of an image with a binary mask.

--------------------------------------------------------------------*/

IMAGE  *Convolve_Binary_Mask(im, mask)
IMAGE  *im, *mask;
{
    register int i, j, k, l;
    int     imnr, imnc, mnr, mnc, mnr_2, mnc_2, mnrr, mncc;
    float   val;
    IMAGE  *out;

    if (!im || !mask) {
	VIP_Error_Msg("Convolve_Binary_Mask: NULL input images");
	return (NULL);
    }
    else if (mask->type != BYTETYPE) {
	VIP_Error_Msg("Convolve_Binary_Mask: the mask must be of BYTETYPE");
	return (NULL);
    }
    else if (!(out = ( IMAGE * ) Allocate_Image(0, 0, im->rows, im->cols, im->type))) { 
        VIP_Error_Msg("Convolve_Binary_Mask: out of memory");
	return (NULL);
    }

    Copy_Header(im, out);
    out->type = FLOATTYPE;

    imnr = im->rows;
    imnc = im->cols;
    mnr = mask->rows;
    mnc = mask->cols;
    mnr_2 = mnr / 2;
    mnc_2 = mnc / 2;
    mnrr = (mnr % 2 ? mnr_2 : mnr_2 - 1);
    mncc = (mnc % 2 ? mnc_2 : mnc_2 - 1);

    switch (im->type) {
    case BYTETYPE:
	for (i = mnr_2; i < imnr - mnr_2; i++)
	    for (j = mnc_2; j < imnc - mnc_2; j++) {
		for (k = -mnr_2, val = 0.0; k <= mnrr; k++)
		    for (l = -mnc_2; l <= mncc; l++)
			if (mask->i.c[k + mnr_2][l + mnc_2])
			    val += im->i.c[i + k][j + l];
		out->i.c[i][j] = val;
	    }
	break;
    case SHORTTYPE:
	for (i = mnr_2; i < imnr - mnr_2; i++)
	    for (j = mnc_2; j < imnc - mnc_2; j++) {
		for (k = -mnr_2, val = 0.0; k <= mnrr; k++)
		    for (l = -mnc_2; l <= mncc; l++)
			if (mask->i.c[k + mnr_2][l + mnc_2])
			    val += im->i.s[i + k][j + l];
		out->i.s[i][j] = val;
	    }
	break;
    case LONGTYPE:
	for (i = mnr_2; i < imnr - mnr_2; i++)
	    for (j = mnc_2; j < imnc - mnc_2; j++) {
		for (k = -mnr_2, val = 0.0; k <= mnrr; k++)
		    for (l = -mnc_2; l <= mncc; l++)
			if (mask->i.c[k + mnr_2][l + mnc_2])
			    val += im->i.l[i + k][j + l];
		out->i.l[i][j] = val;
	    }
	break;
    case FLOATTYPE:
	for (i = mnr_2; i < imnr - mnr_2; i++)
	    for (j = mnc_2; j < imnc - mnc_2; j++) {
		for (k = -mnr_2, val = 0.0; k <= mnrr; k++)
		    for (l = -mnc_2; l <= mncc; l++)
			if (mask->i.c[k + mnr_2][l + mnc_2])
			    val += im->i.f[i + k][j + l];
		out->i.f[i][j] = val;
	    }
	break;
    case DOUBLETYPE:
	for (i = mnr_2; i < imnr - mnr_2; i++)
	    for (j = mnc_2; j < imnc - mnc_2; j++) {
		for (k = -mnr_2, val = 0.0; k <= mnrr; k++)
		    for (l = -mnc_2; l <= mncc; l++)
			if (mask->i.c[k + mnr_2][l + mnc_2])
			    val += im->i.d[i + k][j + l];
		out->i.d[i][j] = val;
	    }
	break;
    default:
	VIP_Error_Msg("Convolve_Binary_Mask: only able to process images of BYTETYPE, SHORTTYPE, LONGTYPE, FLOATTYPE, and DOUBLETYPE");
	Free_Image(out);
	out = NULL;
    }
    return (out);
}


/*- Convolve_Mask ---------------------------------------------------

Output the convolution of an image with a mask (an image of FLOATTYPE).

--------------------------------------------------------------------*/

IMAGE  *Convolve_Mask(im, mask)
IMAGE  *im, *mask;
{
    register int i, j, k, l;
    int     imnr, imnc, mnr, mnc, mnr_2, mnc_2, mnrr, mncc;
    float   val;
    IMAGE  *out;

    if (!im || !mask) {
	VIP_Error_Msg("Convolve_Mask: NULL input images");
	return (NULL);
    }
    else if (mask->type != FLOATTYPE) {
	VIP_Error_Msg("Convolve_Mask: the mask must be of FLOATTYPE");
	return (NULL);
    }
    else if (!(out = ( IMAGE * ) Allocate_Image(0, 0, im->rows, im->cols, FLOATTYPE))) {
	VIP_Error_Msg("Convolve_Mask: out of memory");
	return (NULL);
    }

    Copy_Header(im, out);
    out->type = FLOATTYPE;

    imnr = im->rows;
    imnc = im->cols;
    mnr = mask->rows;
    mnc = mask->cols;
    mnr_2 = mnr / 2;
    mnc_2 = mnc / 2;
    mnrr = (mnr % 2 ? mnr_2 : mnr_2 - 1);
    mncc = (mnc % 2 ? mnc_2 : mnc_2 - 1);

    switch (im->type) {
    case BYTETYPE:
	for (i = mnr_2; i < imnr - mnr_2; i++)
	    for (j = mnc_2; j < imnc - mnc_2; j++) {
		for (k = -mnr_2, val = 0.0; k <= mnrr; k++)
		    for (l = -mnc_2; l <= mncc; l++)
			val += im->i.c[i + k][j + l] *
			    mask->i.f[k + mnr_2][l + mnc_2];
		out->i.f[i][j] = val;
	    }
	break;
    case SHORTTYPE:
	for (i = mnr_2; i < imnr - mnr_2; i++)
	    for (j = mnc_2; j < imnc - mnc_2; j++) {
		for (k = -mnr_2, val = 0.0; k <= mnrr; k++)
		    for (l = -mnc_2; l <= mncc; l++)
			val += im->i.s[i + k][j + l] *
			    mask->i.f[k + mnr_2][l + mnc_2];
		out->i.f[i][j] = val;
	    }
	break;
    case LONGTYPE:
	for (i = mnr_2; i < imnr - mnr_2; i++)
	    for (j = mnc_2; j < imnc - mnc_2; j++) {
		for (k = -mnr_2, val = 0.0; k <= mnrr; k++)
		    for (l = -mnc_2; l <= mncc; l++)
			val += im->i.l[i + k][j + l] *
			    mask->i.f[k + mnr_2][l + mnc_2];
		out->i.f[i][j] = val;
	    }
	break;
    case FLOATTYPE:
	for (i = mnr_2; i < imnr - mnr_2; i++)
	    for (j = mnc_2; j < imnc - mnc_2; j++) {
		for (k = -mnr_2, val = 0.0; k <= mnrr; k++)
		    for (l = -mnc_2; l <= mncc; l++)
			val += im->i.f[i + k][j + l] *
			    mask->i.f[k + mnr_2][l + mnc_2];
		out->i.f[i][j] = val;
	    }
	break;
    case DOUBLETYPE:
	for (i = mnr_2; i < imnr - mnr_2; i++)
	    for (j = mnc_2; j < imnc - mnc_2; j++) {
		for (k = -mnr_2, val = 0.0; k <= mnrr; k++)
		    for (l = -mnc_2; l <= mncc; l++)
			val += im->i.d[i + k][j + l] *
			    mask->i.f[k + mnr_2][l + mnc_2];
		out->i.f[i][j] = val;
	    }
	break;
    default:
	VIP_Error_Msg("Convolve_Mask: only able to process images of BYTETYPE, SHORTTYPE, LONGTYPE, FLOATTYPE, and DOUBLETYPE");
	Free_Image(out);
	out = NULL;
    }
    return (out);
}


/*- Main ------------------------------------------------------------

Main body of the program.

--------------------------------------------------------------------*/

main(argc, argv)
int     argc;
char   *argv[];
{
    int     argn, binary_arg = 0;
    char   *mask_arg = NULL;	/* mask file name */
    char   *image_arg = NULL;	/* image file name */
    char   *matrix_arg = NULL;	/* output image file name */
    IMAGE  *im;			/* input image */
    IMAGE  *mask, *outim;	/* mask image and output image */
    
    for (argn = 1; argn < argc; argn++)
	if (argv[argn][0] == '-')
	    switch (argv[argn][1]) {
	    case 'b':
		binary_arg = argn;
		break;
	    case 'i':
		image_arg = argv[++argn];
		break;
	    case 'o':
		matrix_arg = argv[++argn];
		break;
	    default:
		Usage();
	    }
	else if (!mask_arg)
	    mask_arg = argv[argn];
	else
	    Usage();


    if (!(im = ( IMAGE * ) Read_Image(image_arg)))
	exit(1);
    if (!(mask = ( IMAGE * ) Read_Image(mask_arg)))
	exit(1);

    if (binary_arg) {
	if (!(outim = ( IMAGE * ) Convolve_Binary_Mask(im, mask)))
	    exit(1);
    }
    else if (!(outim = ( IMAGE * ) Convolve_Mask(im, mask)))
	exit(1);

    Write_Image(outim, matrix_arg);

    exit(0);
}
