/*******************************************************************************
*
* University of Western Australia
* Department of Computer Science
* Copyright (c) University of Western Australia
*
* SYSTEM :              VIP
* RELEASE:		3
* SUBSYSTEM:            LIB
* MODULE:		vip1D.c - One dimensional edge detection library		
* REVISION:             3.4
* AUTHOR:               PK
* CREATION DATE:        October 1991
* REVISION DATE:	3/21/94        
*
********************************************************************************
*
* REVISION LOG
*
* REVISION:             3.4
* REVISION DATE:	21 March 1994
* COMMENT:		Removed vector.h
* BY:			CFF
*
* REVISION:             3.3
* REVISION DATE:	25 February 1994
* COMMENT:		Added call to Image_OK
* BY:			CFF
*
* REVISION:             3.2
* REVISION DATE:	16 August 1993
* COMMENT:		Fixed up for DEC build
* BY:			CFF
*
* REVISION:             3.1
* REVISION DATE:	9 July 1992
* COMMENT:		ANSIfied and SCCS'd
* BY:			CFF
*
* REVISION:             
* REVISION DATE:	October 1991
* COMMENT:		Version 2.0
* BY:			
*
*******************************************************************************/

#ifndef lint
static char *sccs_id = "@(#)vip1D.c	3.4 3/21/94";
#endif


#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <math.h>
#include "vip.h"
#include "vipiofn.h"
#include "vip1Dfn.h"
#include "misc.h"
#include "vipspatfn.h"


/*-  Edge_1D ----------------------------------------------------------

One Dimensional edge detection function

----------------------------------------------------------------------*/

int      Edge_1D(image, ustart, vstart, uend, vend, mask, lookup, updown,
	               span, threshold, cu, cv)
IMAGE   *image;
int      ustart, vstart, uend, vend;
INT_MASK mask;
int     *lookup;		/* lookup table to modify intensity values */
int      updown;		/* +ve for upward transition -ve for downward */
int      span;
int      threshold;
int     *cu, *cv;
{
    int     *buffer, *pointer;
    double   du, dv, Npixels;
    double   uscale, vscale;
    int      u, v, i, max;
    int      found = 0;
    char     message[80];


    if (!Image_OK(image))	/* Test for address of an image */
	return (0);

    /* Test that end points of line being scanned are in the image */

    if (!(Pixel_In_Image(image, ustart, vstart) AND Pixel_In_Image(image, uend, vend)))
	return (0);

    du = uend - ustart;
    dv = vend - vstart;

    if (abs((int)du) > abs((int)dv))	/* line being scanned is more horizontal */
	Npixels = abs((int)du);
    else			/* line is more vertical */
	Npixels = abs((int)dv);

    if (fabs(Npixels) == 0)
	Npixels = 1;

    pointer = buffer = (int *) malloc((int) Npixels * sizeof(int));	/* allocate buffer */
    if (pointer == NULL) {
	VIP_Error_Msg("Edge_1D: Could not allocate space for buffer");
	return (0);
    }
    /* Copy pixels in image into buffer */

    uscale = du / Npixels;
    vscale = dv / Npixels;

    for (i = 0; i < Npixels; i++, pointer++) {
	u = ustart + (double) i *uscale;
	v = vstart + (double) i *vscale;

	*pointer = lookup[(int) (image->i.c[v - image->vmin][u - image->umin])];
    }

    (void) Convolve_Array_i(buffer, (int) Npixels, mask);    /* Perform convolution */

    /*
     * Pick out maximum point if expected transition was down or minimum if
     * the expected transition was up
     */

    i = Find_1st_Peak(buffer, (int) Npixels, updown, span, threshold);
    if (i == ERROR) {
	max = Max_i(buffer, Npixels, &i);
	(void) sprintf(message, "Edge_1D: warning: peak below threshold, largest peak value is %d\n", max);
	VIP_Error_Msg(message);
	found = 0;
    } else
	found = 1;


    /* Find pixel coordinates corresponding to point of maximum response */

    *cu = ustart + i * uscale;
    *cv = vstart + i * vscale;

    free((char *) buffer);
    return (found);
}


/*- MEdge_1D  -----------------------------------------------------------------

Multiple Line Search for an Edge

------------------------------------------------------------------------------*/

int      MEdge_1D(image, uend1, vend1, uend2, vend2, width,
	              ntimes, mask, lookup, updown, span, threshold, cu, cv)
IMAGE   *image;
int      uend1, vend1, uend2, vend2, width, ntimes;
INT_MASK mask;
int     *lookup;		/* lookup table to modify intensity values */
int      updown;
int      span;
int      threshold;
int      cu[], cv[];
{

    int      ustart, vstart, uend, vend, uorigin, vorigin ;
    double   axis[2], saxis[2], length;
    int      i, j, nfound;

    axis[0] = uend2 - uend1;
    axis[1] = vend2 - vend1;
    length = sqrt(axis[0] * axis[0] + axis[1] * axis[1]);

    /*
     * scan for edges from left to right as one looks along the axis (Note
     * image frame is left handed)
     */

    saxis[0] = -axis[1] / length;
    saxis[1] = axis[0] / length;

    uorigin = uend1 - width / 2 * saxis[0];
    vorigin = vend1 - width / 2 * saxis[1];

    saxis[0] = saxis[0] * width;
    saxis[1] = saxis[1] * width;


    for (i = 0, nfound = 0; i < ntimes; i++) {
	ustart = uorigin + (double) i / (double) (ntimes - 1) * axis[0];
	vstart = vorigin + (double) i / (double) (ntimes - 1) * axis[1];
	uend = ustart + saxis[0];
	vend = vstart + saxis[1];

	j = Edge_1D(image, ustart, vstart, uend, vend, mask, lookup, updown,
		 span, threshold, &cu[nfound], &cv[nfound]);

	nfound += j;
    }
    return (nfound);
}


/*-- Convolve_Array_i ----------------------------------------------------------

Function to convolve an 1D array if ints with a mask

--------------------------------------------------------------------------------*/

int      Convolve_Array_i(array, arraysize, mask)
int      array[];
int      arraysize;
INT_MASK mask;
{
    int     *buffer, *output;
    int      result;
    int      limit;
    int      i, j;

    if (mask.rows != 1) {
	VIP_Error_Msg("Convolve_Array_i: Mask is not 1D");
	return (ERROR);
    }
    if (Out_Of_Range_i(mask.cols, 1, arraysize, "")) {
	VIP_Error_Msg("Convolve_Array_i: Mask is of length 0 or longer than array");
	return (ERROR);
    }
    buffer = (int *) malloc(arraysize * sizeof(int));	/* space for output
							 * buffer */

    output = buffer + mask.cols / 2;	/* initialise output pointer */
    limit = arraysize - mask.cols;

    for (i = 0; i <= limit; i++) {
	result = 0;
	for (j = 0; j < mask.cols; j++)
	    result = result + mask.m[0][j] * array[i + j];

	if (mask.rescale != 1)
	    *output = result / mask.rescale;
	else
	    *output = result;

	output++;
    }

    (void) memcpy(array, buffer, arraysize * sizeof(int));    /* copy results back */

    /*
     * Set end data points which could not be convolved to the 1st and last
     * convolved values
     */

    for (i = 0; i < mask.cols / 2; i++) {
	array[i] = array[mask.cols / 2];
	array[arraysize - 1 - i] = array[arraysize - mask.cols / 2 - 1];
    }

    free((char *) buffer);	/* free the buffer */

    return (0);
}



/*- SD_Gauss_Mask_1D  --------------------------------------------------

Function to set up 2nd derivative of a Gaussian mask
Note a negative mask is generated (ie it goes down then up rather than
vice versa) so that upward intensity gradients are marked by upward
peaks in the convolution result.

-----------------------------------------------------------------------*/

INT_MASK *SD_Gauss_Mask_1D(isigma)
int      isigma;
{
    int      i, j;
    double   AREA = 80.0;
    double   sigma, var, scale;
    INT_MASK *mask;

    sigma = (double) isigma;
    scale = AREA * sqrt(exp((double)1)) / 2.0;

    mask = ( INT_MASK * )Allocate_Int_Mask(1, 6 * isigma + 1);	/* mask is 1 x
							 * (6*sigma+1) in size */

    for (i = 0, j = -3 * isigma; j <= 3 * isigma; i++, j++) {
	var = (double) j;
	mask->m[0][i] = Round(-scale / sigma * exp(-(var * var)
	      / (2 * sigma * sigma)) * ((var * var) / (sigma * sigma) - 1));
    }
    mask->rescale = (int) AREA;
    return (mask);
}

/*- D_Gauss_Mask_1D  ------------------------------------------------------

Function to set up 1st derivative of a Gaussian mask
Note a negative mask is generated (ie it goes down then up rather than
vice versa) so that upward intensity gradients are marked by upward
peaks in the convolution result.

-------------------------------------------------------------------------*/

INT_MASK *D_Gauss_Mask_1D(isigma)
int      isigma;
{
    int      i, j;
    double   AREA = 80;
    double   sigma, var;
    INT_MASK *mask;

    sigma = (double) isigma;

    mask = ( INT_MASK * ) Allocate_Int_Mask(1, 6 * isigma + 1);	/* mask is 1 x
							 * (6*sigma+1) in size */

    for (i = 0, j = -3 * isigma; j <= 3 * isigma; i++, j++) {
	var = (double) j;
	mask->m[0][i] = Round(AREA * var / (sigma * sigma) * exp(-(var * var)
						    / (2 * sigma * sigma)));
    }
    mask->rescale = (int) AREA;
    return (mask);
}


/*- Gauss_Mask_1D  ----------------------------------------------------

Function to set up a Gaussian mask

-----------------------------------------------------------------------*/

INT_MASK *Gauss_Mask_1D(isigma)
int      isigma;
{
    int      i, j;
    double   AREA = 100;
    double   sigma, var;
    double   PHI = 3.14159265;
    INT_MASK *mask;

    sigma = (double) isigma;

    mask = (  INT_MASK * ) Allocate_Int_Mask(1, 6 * isigma + 1);	/* mask is 1 x
							 * (6*sigma+1) in size */

    for (i = 0, j = -3 * isigma; j < 3 * isigma; i++, j++) {
	var = (double) j;
	mask->m[0][i] = Round(AREA / (sqrt(2. * PHI) * sigma) * exp(-(var * var)
						   / (2. * sigma * sigma)));
    }
    mask->rescale = AREA;	/* rescale parameter */
    return (mask);
}


/*- Set_Look_Up_Table -----------------------------------------------------

Function to setup a lookup table for modifying intensity values

Intensity values are first normalised so that they vary between 0 an 1
then raised to some power before being rescaled back to range between
0 and 255.
A power less than one will exaggerate edges at low intensities and suppress
edges at higher intensities (similar to human eyes).  A power greater than
one will have the opposite effect.

---------------------------------------------------------------------------*/

int      Set_Look_Up_Table(LUT, power)
int      LUT[];
double   power;
{
    int      i;
    double   x;

    for (i = 0; i < 256; i++) {
	x = (double) i / 255.0;
	LUT[i] = pow(x, power) * 255.0;
    }
    return (OK);
}

/*--  Find_1st_Peak  -------------------------------------------------------

Function to find the first peak in an array of integers

The flag 'UpDown' is used to specify a search for a peak or a trough.
If we are looking for a trough the values are multiplied by -1 so that
only a +ve peak has to be found.  Only peaks above 'threshold' are
flagged.  If no peaks above threshold are found ERROR is returned.  It
is assumed that the first point is not the peak.

---------------------------------------------------------------------------*/

int      Find_1st_Peak(buffer, Npts, UpDown, span, threshold)
int     *buffer, Npts, UpDown, span, threshold;
{
    int      i, *backptr, *midptr, *frontptr, value, slope, lastslope;
    int      Up = 1, Down = -1, Level = 0;
    int      found = FALSE;

    UpDown = (UpDown > 0) ? 1 : -1;	/* Make sure UpDown is +-1 */

    lastslope = Down;

    for (i = span, backptr = buffer, midptr = buffer + (span / 2), frontptr = buffer + span;
	 i < Npts; i++, backptr++, midptr++, frontptr++) {

	slope = UpDown * (*frontptr - *backptr);
	value = UpDown * *midptr;

	if (slope > 0)
	    slope = Up;
	else if (slope < 0)
	    slope = Down;
	else
	    slope = Level;

	if ((lastslope == Up) AND((slope == Level) OR(slope == Down))
	    AND(value > threshold)) {
	    found = TRUE;
	    break;		/* We have found the first peak */
	}
	lastslope = slope;
    }

    if (found) {
	if (VIP_VERBOSE)
	    (void) printf("peak of %d at %d \n", value, i - (span / 2));
	if (VIP_LOG_FILE)
	    (void) (void) fprintf(VIP_LOG_FILE, "peak of %d at %d \n", value, i - (span / 2));
	return (i - (span / 2));/* Peak assumed to be between front and back
				 * points */
    } else {
	if (VIP_LOG_FILE)
	    (void) (void) fprintf(VIP_LOG_FILE, "No peak found\n");
	return (ERROR);
    }

}



/*-  Find_Closest_Peak ------------------------------------------------------

Function to find the peak in an array of integers that is closest to a
specified position in the array.

The flag 'UpDown' is used to specify a search for a peak or a trough.
If we are looking for a trough the values are multiplied by -1 so that
only a +ve peak has to be found.  Only peaks above 'threshold' are
flagged.  If no peaks above threshold are found ERROR is returned.  It
is assumed that the first point is not the peak.

----------------------------------------------------------------------------*/

int      Find_Closest_Peak(buffer, Npts, UpDown, span, threshold, position)
int     *buffer, Npts, UpDown, span, threshold, position;
{
    int      PreviousPeak, NextPeak;
    int     *forwardbuf, *backwardbuf;
    int      PosnValue;

    /* First check whether the closest peak is at position */

    PosnValue = *(buffer + position - 1);

    if ((PosnValue > threshold)
	AND(*(buffer + position - 2) < PosnValue)
	AND(*(buffer + position) < PosnValue))
	return (position);


    forwardbuf = buffer + position;
    backwardbuf = (int *) malloc(position * sizeof(int));

    Move_i((buffer + position - 1), -1, backwardbuf, 1, position);

    NextPeak = Find_1st_Peak(forwardbuf, (Npts - position), UpDown, span, threshold);
    PreviousPeak = Find_1st_Peak(backwardbuf, position, UpDown, span, threshold);

    free((char *) backwardbuf);

    if ((NextPeak == ERROR) AND(PreviousPeak == ERROR))	/* No peaks found */
	return (ERROR);

    if (NextPeak == ERROR)	/* No peaks forward of position */
	NextPeak = 32000;

    if (PreviousPeak == ERROR)	/* No peaks before position */
	PreviousPeak = 32000;

    if (NextPeak < PreviousPeak)
	return (position + NextPeak);
    else
	return (position - PreviousPeak);

}

/*-  Select_Peaks ----------------------------------------------------------

Function to select an number of peaks in an array of integers.  The search
can be specified from either end of the array.

The flag 'UpDown' is used to specify a search for a peak or a trough.
If we are looking for a trough the values are multiplied by -1 so that
only a +ve peak has to be found.  Only peaks above 'threshold' are
flagged.  If no peaks above threshold are found ERROR is returned.  It
is assumed that the first point is not the peak.

----------------------------------------------------------------------------*/

int      Select_Peaks(buffer, Npts, FirstPeakWanted, LastPeakWanted,
	              UpDown, Direction, span, threshold, peak, NpeaksFound)
int     *buffer, Npts, FirstPeakWanted, LastPeakWanted, UpDown, Direction;
int      span, threshold, *peak, *NpeaksFound;
{
    int     *buf, *bufptr;
    int      peakNo;
    int      LastPeak, NextPeak;

    bufptr = buf = (int *) malloc(Npts * sizeof(int));

    if (Direction > 0)
	Move_i(buffer, 1, buf, 1, Npts);
    else
	Move_i(&buffer[Npts - 1], -1, buf, 1, Npts);


    LastPeak = 0;
    peakNo = 0;
    *NpeaksFound = 0;

    do {
	NextPeak = Find_1st_Peak(bufptr, Npts - LastPeak, UpDown, span, threshold);
	peakNo++;
	if (peakNo >= FirstPeakWanted AND NextPeak != ERROR) {
	    peak[peakNo - FirstPeakWanted] = LastPeak + NextPeak;
	    *NpeaksFound++;
	}
	LastPeak = LastPeak + NextPeak;
	bufptr = buf + LastPeak;
    } while (peakNo < LastPeakWanted AND LastPeak != ERROR);

    free((char *) buf);
    return (OK);
}


#ifdef CRAP
/** Function to find selected peaks in an image slice ***************/

/*
 * Whether one is searching for a peak or a trough is set by 'UpDown', if we
 * are looking for a trough the values are multiplied by -1 so that only a
 * +ve peak has to be found. Forward or backward searching specified by
 * ForwBack. Only peaks above 'threshold' are flagged. Returns no of
 * requested peaks that it found above threshold. It is assumed that the
 * first point is not a peak.
 */

int      FindSlicePeaks(SLICE * slice, int FirstPeakWanted, int LastPeakWanted,
          int UpDown, int ForwBack, int span, int threshold, int *u, int *v)
{
    int      i, *backptr, *midptr, *frontptr, value, slope, lastslope;
    int      Up = 1, Down = -1, Level = 0;
    double   du, dv, frac;
    int      Ust, Vst;
    int      npeaks = 0, nfound = 0;

    UpDown = (UpDown > 0) ? 1 : -1;	/* Make sure UpDown is +-1 */
    ForwBack = (ForwBack > 0) ? 1 : -1;	/* Check ForwBack too */
    lastslope = Down;

    if (ForwBack > 0) {		/* search forwards */
	backptr = slice->array;
	midptr = backptr + (span / 2);
	frontptr = backptr + span;
	Ust = slice->Ustart;
	Vst = slice->Vstart;
	du = slice->Uend - slice->Ustart;
	dv = slice->Vend - slice->Vstart;
    } else {			/* search backwards */
	backptr = &slice->array[slice->npts - 1];
	midptr = backptr - (span / 2);
	frontptr = backptr - span;
	Ust = slice->Uend;
	Vst = slice->Vend;
	du = slice->Ustart - slice->Uend;
	dv = slice->Vstart - slice->Vend;
    }

    for (i = span; i < slice->npts; i++) {
	slope = UpDown * (*frontptr - *backptr);
	value = UpDown * *midptr;

	if (slope > 0)
	    slope = Up;
	else if (slope < 0)
	    slope = Down;
	else
	    slope = Level;

	if ((lastslope == Up) AND((slope == Level) OR(slope == Down))
	    AND(value > threshold)) {

	    npeaks++;
	    if (npeaks >= FirstPeakWanted AND npeaks <= LastPeakWanted) {	/* we want this peak */
		frac = (double) (i - span / 2) / (double) slice->npts;
		u[nfound] = Ust + frac * du;
		v[nfound] = Vst + frac * dv;
		if (VIP_VERBOSE OR DISPLAY) {
		    DrawX(u[nfound] + status.pan, v[nfound] + status.scroll, 12);
		}
		if (VIP_VERBOSE)
		    (void) printf("value at peak %d is %d\n", FirstPeakWanted + nfound, value);
		if (VIP_LOG_FILE)
		    (void) fprintf(VIP_LOG_FILE, "value at peak %d is %d\n",
			    FirstPeakWanted + nfound, value);
		nfound++;
	    }
	    if (npeaks == LastPeakWanted)
		break;
	}
	lastslope = slope;
	backptr += ForwBack;
	midptr += ForwBack;
	frontptr += ForwBack;
    }

    if (nfound < (LastPeakWanted - FirstPeakWanted + 1)) {
	if (UpDown < 0) {
	    value = Min_i(slice->array, slice->npts, &i);
	} else {
	    value = Max_i(slice->array, slice->npts, &i);
	}
	if (VIP_VERBOSE)
	    (void) printf("No peak above threshold, max value was %d at element %d\n",
		   value, i);
	if (VIP_LOG_FILE)
	    (void) fprintf(VIP_LOG_FILE, "No peak above threshold, max value was %d at element %d\n",
		    value, i);
    }
    return (nfound);
}

/** Function to find peaks in curvature along a contour ***************/


int      FindPeakCurvature(CONTOUR_2D contour, int StartPt, int threshold,
			            int ForwBack, int *u, int *v)
{
    int      i, *ptr, value, slope, lastslope, index;
    int      Up = 1, Down = -1, Level = 0;
    int      found = FALSE;
    int      UpDown;


    ForwBack = (ForwBack > 0) ? 1 : -1;	/* Check ForwBack is +/- 1 */
    lastslope = Down;

    if (threshold < 0)
	UpDown = -1;
    else
	UpDown = 1;

    if (StartPt == 0)
	index = 0;
    else if (StartPt == 1)
	index = contour.MidIndex;
    else if (StartPt == 2)
	index = contour.npts - 1;
    else {
	(void) printf("FindPeakCuvature: StartPt must be 0 1 or 2 not %d\n", StartPt);
	if (VIP_LOG_FILE)
	    (void) fprintf(VIP_LOG_FILE, "FindPeakCuvature: StartPt must be 0 1 or 2 not %d\n", StartPt);
	return (ERROR);
    }

    if ((ForwBack > 0 AND StartPt == 2) OR(ForwBack < 0 AND StartPt == 0)) {
	(void) printf("FindPeakCurvature: Searching off the end of the contour\n");
	if (VIP_LOG_FILE)
	    (void) fprintf(VIP_LOG_FILE, "FindPeakCurvature: Searching off the end of the contour\n");
	return (ERROR);
    }
    fg_setind(255);
    do {
	slope = UpDown * (contour.curvature[index + ForwBack] - contour.curvature[index]);
	value = UpDown * contour.curvature[index];

	if (slope > 0)
	    slope = Up;
	else if (slope < 0)
	    slope = Down;
	else
	    slope = Level;

	if ((lastslope == Up) AND((slope == Level) OR(slope == Down))
	    AND(value > threshold)) {
	    /* was UpDown * threshold - mistake? (PK) */
	    found = TRUE;
	    if (VIP_VERBOSE OR DISPLAY) {
		DrawX(contour.u[index] + status.pan, contour.v[index] + status.scroll, 12);
	    }
	    if (VIP_VERBOSE)
		(void) printf("curvature at peak is %d\n", contour.curvature[index]);
	    if (VIP_LOG_FILE)
		(void) fprintf(VIP_LOG_FILE, "curvature at peak is %d\n", contour.curvature[index]);
	    break;
	}
	lastslope = slope;
	index += ForwBack;
    } while (index >= 0 AND index < contour.npts);

    if (found) {
	*u = contour.u[index];
	*v = contour.v[index];
	return (1);
    } else {
	VIP_Error_Msg("No curvature value above threshold\n");
	return (0);
    }
}

#endif















