/* $Id$
 *
 * print.c
 */

/**************************************************************************
 *     Copyright (C) 1990 by Mark B. Phillips and Robert R. Miner	  *
 * 									  *
 * Permission to use, copy, modify, and distribute this software, its	  *
 * documentation, and any images it generates for any purpose and without *
 * fee is hereby granted, provided that					  *
 * 									  *
 * (1) the above copyright notice appear in all copies and that both that *
 *     copyright notice and this permission notice appear in supporting	  *
 *     documentation, and that the names of Mark B.  Phillips, Robert R.  *
 *     Miner, or the University of Maryland not be used in advertising or *
 *     publicity pertaining to distribution of the software without	  *
 *     specific, written prior permission.				  *
 *									  *
 * (2) Explicit written credit be given to the authors Mark B.  Phillips  *
 *     and Robert R. Miner in any publication which uses part or all of	  *
 *     any image produced by this software.				  *
 *									  *
 * This software is provided "as is" without express or implied warranty. *
 **************************************************************************/

#include <stdio.h>
#include "internal.h"

#define FILENAME_DISPLAY_LENGTH 30
#define FILENAME_STORED_LENGTH 256
#define MESSAGE_LENGTH 37

/* Print frame has one panel */
static Frame		print_frame = NULL;
static Panel		print_panel = NULL;

/* Gadgets for print frame */
static Panel_item	printer_button, file_button, done_button;
static Panel_item	filename_text, message_item;

static int	file_button_proc(), printer_button_proc(), done_proc();

static FILE *psfile;		/* current PostScript file */
static int PSWrite();		/* PostScript file output function */

/* Command to use for printing PostScript files; this should be set
 * by the Makefile, but just in case: */
#ifndef SPOOL_COMMAND
#define SPOOL_COMMAND=""
#endif

/*-----------------------------------------------------------------------
 * Function:	wPrint
 * Description:	pop up the printer frame (print button notify proc)
 * Args:	(none)
 * Returns:	nothing
 * Author:	mbp
 * Date:	Mon May 28 15:43:26 1990
 */
int
  wPrint()
{
  /* If we haven't created the frame yet, do so now */
  if (print_frame == NULL)
    InitializePrintFrame();

  /* Clear any message currently in the message area */
  PrintPanelMessage(NULL);

  /* Now make it visible */
  window_set(print_frame, WIN_SHOW, TRUE, 0);
}

/*-----------------------------------------------------------------------
 * Function:	InitializePrintFrame
 * Description:	Initialize the PostScript frame
 * Args:	(none)
 */
static int
InitializePrintFrame()
{
  print_frame =
    window_create(wFrame, FRAME,
		  WIN_X,		457,
		  WIN_Y,		0,
		  WIN_FONT,		wFont1,
		  FRAME_LABEL,		"PostScript",
		  FRAME_SHOW_LABEL,	TRUE,
		  0);
  if (print_frame == NULL) {
    return(1);
  }
  
  print_panel =
    window_create(print_frame, PANEL,
		  WIN_X, 	0,
		  WIN_Y,	0,
		  WIN_FONT,	wFont1,
		  WIN_WIDTH,	352,
		  WIN_HEIGHT,	142,
		  0);
  if (print_panel == NULL) {
    return(1);
  }
  
  printer_button =
    panel_create_item(print_panel, PANEL_BUTTON, 
		      PANEL_ITEM_X, 	10,
		      PANEL_ITEM_Y, 	12,
		      PANEL_LABEL_IMAGE,
		      panel_button_image(print_panel, "Printer", 8, 0),
		      PANEL_NOTIFY_PROC, printer_button_proc,
		      0);
  
  file_button =
    panel_create_item(print_panel, PANEL_BUTTON, 
		      PANEL_ITEM_X,	10,
		      PANEL_ITEM_Y,	44,
		      PANEL_LABEL_IMAGE,
		      panel_button_image(print_panel, "File", 5, 0),
		      PANEL_NOTIFY_PROC, file_button_proc,
		      0);

  filename_text =
    panel_create_item(print_panel, PANEL_TEXT, 
		      PANEL_ITEM_X, 70,
		      PANEL_ITEM_Y, 47,
		      PANEL_VALUE_DISPLAY_LENGTH, FILENAME_DISPLAY_LENGTH,
		      PANEL_VALUE_STORED_LENGTH, FILENAME_STORED_LENGTH,
		      PANEL_LABEL_STRING, "Filename:",
		      0);

  done_button =
    panel_create_item(print_panel, PANEL_BUTTON, 
		      PANEL_ITEM_X, 258,
		      PANEL_ITEM_Y, 9,
		      PANEL_LABEL_IMAGE,
		      panel_button_image(print_panel, "Done", 7, 0),
		      PANEL_NOTIFY_PROC, done_proc,
		      0);

  message_item =
    panel_create_item(print_panel, PANEL_MESSAGE,
		      PANEL_ITEM_X, 10,
		      PANEL_ITEM_Y, 118,
		      PANEL_LABEL_STRING, "Printer Error Message Here",
		      PANEL_LABEL_FONT, wFont1,
		      0);
  
  window_fit(print_frame);
}

/*-----------------------------------------------------------------------
 * Function:	done_proc
 * Description:	close the PostScript window
 */
static int
done_proc()
{
  window_set(print_frame, WIN_SHOW, FALSE, 0);
}

/*-----------------------------------------------------------------------
 * Function:	file_button_proc
 * Description:	notify proc for "File" button
 */
static int
file_button_proc()
{
  char *filename;
  char msg[256];

  /* Get the file name */
  filename = (char *)panel_get_value(filename_text);

  /* Check for write permission */
  if (!wFileOpenable(filename, "w")) {
    sprintf(msg,"I can't write to file \"%s\".", filename);
    PrintPanelError(msg);
    return(0);
  }

  /* Write the file and signal success or failure */
  if (WritePostScriptFile(filename) == 0) {
    sprintf(msg, "Wrote \"%s\".", filename);
    PrintPanelMessage(msg);
  }
  else {
    sprintf(msg, "Couldn't write \"%s\".", filename);
    PrintPanelError(msg);
  }

  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	printer_button_proc
 * Description:	send PostScript to printer
 * Args:	(none)
 */
static int
printer_button_proc()
{
  int r;
  char *mktemp(), *getenv();
  static char template[] = "/tmp/hcXXXXXX";
  char filename[sizeof(template)+1];
  char command[256];

  /* Make sure we are able to print PostScript files */
  if (strlen(SPOOL_COMMAND) <= 0) {
    PrintPanelError("Don't know how to print PostScript files.");
    return;
  }

  /* Get name for temporary file for PostScript output */
  strcpy(filename, template);
  mktemp(filename);
  if (!wFileOpenable(filename, "w")) {
    PrintPanelError("Can't write temporary file.");
  }
  else {

    /* Write the file */
    if (WritePostScriptFile(filename) != 0) {
      PrintPanelError("Error in writing temporary file");
      return;
    }

    /* Send the file to the printer */
    sprintf(command, "%s %s", SPOOL_COMMAND, filename);
    if (system(command) == 0)
      PrintPanelMessage("PostScript image sent to printer");
    else {
      PrintPanelError("Error in sending PostScript to printer !");
    }
    
    /* Now delete the temporary file */
    unlink(filename);
  }
}

/*-----------------------------------------------------------------------
 * Function:	PrintPanelMessage
 * Description:	display a string in the message area of the ps panel
 * Args  IN:	s: the string to be displayed
 * Notes:	Only the first MESSAGE_LENGTH chars are displayed
 */
static int
PrintPanelMessage(s)
char *s;
{
  return(ShowMessage(message_item, s, MESSAGE_LENGTH));
}

/*-----------------------------------------------------------------------
 * Function:	PrintPanelError
 * Description:	display a string in the message area of the ps panel
 *		  and beep the bell
 * Args  IN:	s: the string to be displayed
 * Notes:	Only the first MESSAGE_LENGTH chars are displayed
 */
static int
PrintPanelError(s)
char *s;
{
  window_bell(print_panel);
  window_bell(print_panel);
  return(ShowMessage(message_item, s, MESSAGE_LENGTH));
}

/*-----------------------------------------------------------------------
 * Function:	WritePostScriptFile
 * Description:	Write a postscript file of the current picture
 * Args  IN:	filename: name of file to write
 * Returns:     success
 * Author:	mbp
 * Date:	Sat May 12 11:45:18 1990
 */
static int
  WritePostScriptFile(filename)
char *filename;
{
  int success;

  psfile = fopen(filename, "w");
  if (psfile == NULL) return(-1);
  success = PSOutput(PSWrite);
  fclose(psfile);
  return(success);
}

/*-----------------------------------------------------------------------
 * Function:	PSWrite
 * Description:	Write a string to the currently open PostScript file
 * Args  IN:	s: string to write
 * Returns:	success status
 * Author:	mbp
 * Date:	Sat May 12 11:46:02 1990
 * Notes:	The currently open PostScript file is *psfile.
 */
static int
  PSWrite(s)
char *s;
{
  return(fputs(s, psfile));
}

/*-----------------------------------------------------------------------
 * Function:     ShowMessage
 * Description:  display a string in a message item
 * Arguments IN: item: handle of message item to display string in
 *		 s: the string to be displayed
 *		 m: maximum number of chars that can be displayed
 *		   in this message item
 * Returns:      0 for success, 1 for failure
 * Notes:        If s has more than max chars, only the first m are
 *		 displayed and 1 is returned.
 */
static int
  ShowMessage(item, s, m)
Panel_item item;
char *s;
int m;
{
  int toobig;
  char temp;

  if (s == NULL) s = "";

  if (toobig = (strlen(s)>m)) {
    temp = s[m];
    s[m] = '\0';
    panel_set(item, PANEL_LABEL_STRING, s, 0);
    s[m] = temp;
  }
  else
    panel_set(item, PANEL_LABEL_STRING, s, 0);
  return(toobig);
}
