/* $Id$
 *
 * drawprocs.c: drawing procedures
 *
 * This file is to be included by other files, not compiled separately.
 * It expects the following macros to be defined:
 * 
 *  CLASS:	storage class for global identifiers
 *  DRAWPOINT:	identifier for point drawing procedure
 *  DRAWSEGMENT:identifier for segment drawing procedure
 *  DRAWENTITY:	identifier for entity drawing procedure
 *  DEVPOINT:	name of device point drawing procedure to call
 *  DEVSEGMENT:	name of device segment drawing procedure to call
 *  DEVARC:	name of device arc drawing procedure to call
 *
 *  It defines 3 procedures, DRAWPOINT, DRAWSEGMENT, and DRAWENTITY,
 *  which draw the corresponding objects on an output device.  These
 *  procedures are defined in terms of the device drawing procedures
 *  DEVPOINT, DEVSEGMENT, and DEVARC, which do the actual work of
 *  drawing points, line segments, and arcs on the output device.
 *
 *  This file is #included by hdraw.c to define window drawing
 *  procedures, and by postscript.c to define PostScript drawing
 *  procedures.
 *
 *  [mbp 5/12/90]
 */

/**************************************************************************
 *     Copyright (C) 1990 by Mark B. Phillips and Robert R. Miner	  *
 * 									  *
 * Permission to use, copy, modify, and distribute this software, its	  *
 * documentation, and any images it generates for any purpose and without *
 * fee is hereby granted, provided that					  *
 * 									  *
 * (1) the above copyright notice appear in all copies and that both that *
 *     copyright notice and this permission notice appear in supporting	  *
 *     documentation, and that the names of Mark B.  Phillips, Robert R.  *
 *     Miner, or the University of Maryland not be used in advertising or *
 *     publicity pertaining to distribution of the software without	  *
 *     specific, written prior permission.				  *
 *									  *
 * (2) Explicit written credit be given to the authors Mark B.  Phillips  *
 *     and Robert R. Miner in any publication which uses part or all of	  *
 *     any image produced by this software.				  *
 *									  *
 * This software is provided "as is" without express or implied warranty. *
 **************************************************************************/

/*-----------------------------------------------------------------------
 * Function:	DRAWPOINT
 * Description:	draws a point in the current model
 * Args  IN:	p: a kPoint
 * Returns:	nothing
 * Author:	mbp
 * Date:	Thu Apr 12 20:04:22 1990
 * Notes:	
 */
CLASS int
  DRAWPOINT(p)
kPoint p;
{
  R2Point x;

  switch (model) {
  default:
  case KLEIN:
    DEVPOINT(p[0], p[1]);
    break;
  case POINCARE:
    KleinToPoincare(x, p);
    DEVPOINT(x[0], x[1]);
    break;
  case UHP:
    KleinToUHP(x, p);
    DEVPOINT(x[0], x[1]);
    break;
  }
}

/*-----------------------------------------------------------------------
 * Function:	DRAWSEGMENT
 * Description:	draws a geodesic segment in the current model
 * Args  IN:	*s: a kSegment in Klein coordinates
 *      OUT:	
 * Returns:	nothing
 * Author:	lena
 * Date:	Thu Apr 12 14:34:16 1990
 * Notes:	
 */
CLASS int
  DRAWSEGMENT(s)
kSegment *s;
{
  R2Point a,b,c;
  double r,arg0,arg1,tmp;
  double x1,y1,x2,y2;

  switch(model){
  case KLEIN:
    x1 = s->p[0][0];
    y1 = s->p[0][1];
    x2 = s->p[1][0];
    y2 = s->p[1][1];
    DEVSEGMENT(x1,y1,x2,y2);
    break;
  case POINCARE:
    KleinToPoincare(a,s->p[0]);
    KleinToPoincare(b,s->p[1]);
    PoincareGeoCenter(c,a,b);
    if ((c[0]==0.0) &&(c[1]==0.0)) {
      x1 = a[0];
      y1 = a[1];
      x2 = b[0];
      y2 = b[1];
      DEVSEGMENT(x1,y1,x2,y2);
    }
    else {
      r = R2dist(c,a);
      arg0 = atan2(a[1]-c[1],a[0]-c[0]);
      arg1 = atan2(b[1]-c[1],b[0]-c[0]);
      if (arg0 > arg1) {	/* arg0 < arg1  draws the arc in the */ 
	tmp = arg0;		/* interior of the unit disc         */
	arg0 = arg1;
	arg1 = tmp;
      }
      if (arg1 - arg0 > M_PI) {	/* unless they lie on opposite sides */
	tmp = arg1;		/* of the brach cut                  */
	arg1 = arg0 + 2 * M_PI;
	arg0 = tmp;
      }
      DEVARC(c[0],c[1],r,arg0,arg1);
    }
    break;
  case UHP:
    KleinToUHP(a,s->p[0]);
    KleinToUHP(b,s->p[1]);
    UHPGeoCenter(c,a,b);
    if (c[1] != 0.0) {
      if (UHP_INFINITY(a)) {
	if (UHP_INFINITY(b))
	  return;
        else
	  DEVSEGMENT(b[0],b[1],b[0],vpy + vps);
      }
      if (UHP_INFINITY(b)) {
	if (UHP_INFINITY(a))
	  return;
	else
	  DEVSEGMENT(a[0],a[1],a[0],vpy + vps);
      }
      if ((!UHP_INFINITY(a)) && (!UHP_INFINITY(b)))
	DEVSEGMENT(a[0],a[1],b[0],b[1]);
    }
    else {
      r = R2dist(c,a);
      arg0 = atan2(a[1]-c[1],a[0]-c[0]);
      arg1 = atan2(b[1]-c[1],b[0]-c[0]);
      /* Move the branch cut from -x to -y axis so we can always
	 draw from min(arg0,arg1) to max(arg0,arg1) */
      if (arg0 < -PI_HALVES) arg0 += TWO_PI;
      if (arg1 < -PI_HALVES) arg1 += TWO_PI;
      if (arg0 > arg1) {	/* arg0 < arg1  draws the arc in the */
	tmp = arg0;		/* upper half plane                  */
	arg0 = arg1;
	arg1 = tmp;
      }    
      DEVARC(c[0],c[1],r,arg0,arg1);
    }
    break;
  }
}

/*-----------------------------------------------------------------------
 * Function:	DRAWENTITY
 * Description:	draw an entity
 * Args  IN:	entity: the entity to be drawn
 * Returns:	success status
 * Author:	mbp
 * Date:	Thu Apr 12 17:09:37 1990
 * Notes:	
 */
CLASS int
  DRAWENTITY( entity )
gdb_Entity entity;
{
  switch (gdb_entity_type(entity)) {
  case POINT:
    DRAWPOINT((kPoint*)gdb_entity_data(entity));
    break;
  case SEGMENT:
    DRAWSEGMENT((kSegment*)gdb_entity_data(entity));
    break;
  default:
    return(-1);
    break;
  }
  return(0);
}

