(* $Id: Dirichlet.m,v 1.4 90/07/11 13:08:08 mbp Exp Locker: mbp $
 *
 * Dirichlet.m: Functions for computing the Dirichlet domain in RH2
 *   for a finite array of points.
 *)

(**************************************************************************
 *     Copyright (C) 1990 by Mark B. Phillips and Robert R. Miner	  *
 * 									  *
 * Permission to use, copy, modify, and distribute this software, its	  *
 * documentation, and any images it generates for any purpose and without *
 * fee is hereby granted, provided that					  *
 * 									  *
 * (1) the above copyright notice appear in all copies and that both that *
 *     copyright notice and this permission notice appear in supporting	  *
 *     documentation, and that the names of Mark B.  Phillips, Robert R.  *
 *     Miner, or the University of Maryland not be used in advertising or *
 *     publicity pertaining to distribution of the software without	  *
 *     specific, written prior permission.				  *
 *									  *
 * (2) Explicit written credit be given to the authors Mark B.  Phillips  *
 *     and Robert R. Miner in any publication which uses part or all of	  *
 *     any image produced by this software.				  *
 *									  *
 * This software is provided "as is" without express or implied warranty. *
 **************************************************************************)

BeginPackage["Dirichlet`", "RH2`"]

DirichletRegion::usage = "DirichletRegion[p, {a1,a2,...,an}] returns the list
of segments which form the boundary of the Dirichlet region of the points
{a1,a2,...,an} relative to the base point p.  The points p,a1,...,an can be
kPoints, pPoints, uPoints, or r21Vectors.  The segments are returned in the
model of p, unless p is an r21Vector, in which case they are returned as
kSegments."

PairList::usage = "PairList[ list ] returns a list of successive pairs of
elements of list; i.e. PairList[ {a1, a2, a3, ... } ] returns { {a1,a2},
{a2,a3}, ... }."

AugmentedPairList::usage = "AugmentedPairList[ list ] returns a list
of successive pairs of elements of list, the last such pair consisting
of the last and first elements of list; i.e. PairList[ {a1, a2, a3,
..., an} ] returns { {a1,a2}, {a2,a3}, ... {an, a1} }."

HalfPlaneIntersection::usage = "HalfPlaneIntersection[ {h1, h2, ...} ] returns
the list of segments which form the boundary of the intersection of the
half-planes h1, h2, ... .  The hi are r21Vectors representing half-planes ---
the half-plane corresponding to hi is the (projection of the) set of
r21Vectors whose h21Dot product with hi is positive.  The segments are
returned as affine line segments --- each segment is a line of two affine
points, like {{x1,y1}, {x2,y2}}."

SegmentkSegment::usage = "SegmentkSegment[{a,b}] converts the affine line
segment from a to b to a kSegment.  This involves possibly clipping to the
unit disk and converting the endpoints to kPoints.  If the affine segment
{a,b} lies completely outside the unit disk, we return Null."

ClipPolygon::usage = "ClipPolygon[ vlist, h ] clips the subject polygon whose
vertices are given by vlist to the half-plane determined by h.  It returns a
list of vertices representing the clipped polygon.  The vertices are given in
affine coordinates."

ClipSegment::usage = "ClipSegment[s, p, h] returns the list of new vertices
which should be added to the output list when traversing the segment from s to
p in clipping a polygon to the half-plane h.  The vertices are given in affine
coordinates.  The output list may contain zero, one, or two vertices."

InHalfPlane::usage = "InHalfPlane[p, h] tests whether the point p lies
in the half-plane h.  p is in affine coordinates.  h is an r21vector
representing a half-plane."

SegmentCirclePoints::usage = "SegmentCirclePoints[p1,p2] returns a list of
points at which the line segment from p1 to p2 touches the unit circle.  The
points are ordered in the direction from p1 to p2, and may include p1 or p2 if
either of these lies on the unit circle.  If the segment does not touch the
unit circle we return the empty list.  p1 and p2 are in affine coordinates."

InOpenDisk::usage = "InOpenDisk[{x,y}] test whether the point (x,y) lies in
the open unit disk."

SegmentIntersection::usage = "SegmentIntersection[a, b, h] returns
the affine coordinates of point at which the line through a and b
intersects the line with projective coordinates given by h."

Begin["`private`"]

DirichletRegion[ p_r21Vector, array_List ] :=
  Select[ Map[ SegmentkSegment,
               HalfPlaneIntersection[ Map[ hBisector[p, #]&, 
					   Map[ r21Vector, array ] ] ] ],
          (# =!= Null)& ]

DirichletRegion[ p_pPoint, array_List ] :=
  Map[ pSegment, DirichletRegion[ r21Vector[p], array ] ]

DirichletRegion[ p_uPoint, array_List ] :=
  Map[ uSegment, DirichletRegion[ r21Vector[p], array ] ]

DirichletRegion[ p_kPoint, array_List ] :=
  DirichletRegion[ r21Vector[p], array ]

HalfPlaneIntersection[ hlist_List ] :=
  Block[ {vlist},
    vlist = { {1,1}, {1,-1}, {-1,-1}, {-1,1} };	(* start with unit square *)
    Scan[ (vlist=ClipPolygon[vlist, #])&, hlist ];
    Return[ PairList[ Append[ vlist, First[vlist] ] ] ]
    ]

ClipPolygon[ vlist_List, h_r21Vector ] :=
  Block[ {newlist, s},
    newlist = {};
    s = Last[vlist];
    Scan[ (newlist=Join[newlist, ClipSegment[s, #, h]]; s=#)&, vlist ];
    Return[ newlist ]
    ]

ClipSegment[s_, p_, h_] :=
  If[ InHalfPlane[p, h],
      If[ InHalfPlane[s, h],
          Return[ {p} ],
          Return[ {SegmentIntersection[s, p, h], p} ] ],

      If[ InHalfPlane[s, h],
          Return[ {SegmentIntersection[s, p, h]} ],
          Return[ {} ] ]
      ]

InHalfPlane[p_, h_] := ( h21Dot[ r21Vector[p], h ] > 0 )

(*
 * NOTE: SegmentCirclePoints works with the parametrization
 *                p[t_] := (1 - t) {x1, y1} + t {x2, y2}
 * of the line segment from {x1,y1} to {x2,y2}.  It looks for solutions in the
 * interval [0,1] of the quadratic equation |p[t]|^2 == 1, and returns the
 * points corresponding to these solutions.  The variables a,b,c are the
 * coefficients of this quadratic equation.
 *)
SegmentCirclePoints[ { {x1_, y1_}, {x2_, y2_} } ] :=
  Block[ {a,b,c,d2,d,p,tlist},
    a = x1^2 + y1^2 + x2^2 + y2^2 - 2 ( x1 x2 + y1 y2 );
    b = 2 ( x1 x2 + y1 y2 - ( x1^2 + y1^2 ) );
    c = x1^2 + y1^2 - 1;
    d2 = b^2 - 4 a c;
    If[ d2 < 0, Return[ {} ] ];

    p[t_] := (1 - t) {x1, y1} + t {x2, y2};

    If[ d2 == 0, Return[ { p[-b / (2 a)] } ] ];

    d = Sqrt[d2];
    tlist =
      Sort[ Select[ { (-b - d) / (2 a), (-b + d) / (2 a) },
            ( (#>=0) && (#<=1) )& ] ];

    Return[ Map[ p, tlist ] ]
    ]

InOpenDisk[ {x_, y_} ] := (x^2 + y^2) < 1

SegmentkSegment[ {a_, b_} ] :=
  Block[ {ain, bin, pts},
    ain = InOpenDisk[a];
    bin = InOpenDisk[b];

    If[ ain && bin, Return[ kSegment[ kPoint[a], kPoint[b] ] ] ];

    pts = SegmentCirclePoints[{a,b}];

    If[ ain, Return[ kSegment[ kPoint[a], kPoint[pts[[1]]] ] ] ];

    If[ bin, Return[ kSegment[ kPoint[pts[[1]]], kPoint[b] ] ] ];
 
    If[ Length[pts]==2,
        Return[ Apply[ kSegment, Map[ kPoint, pts ] ] ] ];

    Return[ Null ]
    ]

PairList[ v_List ] :=
  Map[ v[[{#,#+1}]]&, Range[1, Length[v]-1] ]

AugmentedPairList[ v_List ] :=
  PairList[ Append[ v, Last[v] ] ]

SegmentIntersection[a_, b_, h_] :=
  hAffine[ h21Cross[ h21Cross[ r21Vector[a], r21Vector[b] ], h ] ]

End[]

EndPackage[]
