
/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*************************************************************
*
*    file:      filmq.c
*
*    contents:  Functions for calculating energy, volume,
*               and their gradients for the QUADRATIC
*               SOAPFILM model.  Also utility functions
*               for quadratic interpolation and Gaussian
*               integration.
*/

/************************************************************************
*
*  Functions to calculate area, volume, and energy according to
*  the tessellation model.
* 
*  Quadratic patch version with 7-point quadrature.
*/

#include "include.h"

/* 7-point Gaussian quadrature on triangle.  Accurate for fifth
   degree polynomials.  Ref: Abramowitz & Stegun 25.4.63.iii.

   Triangle is: x > 0, y > 0, x + y < 2.

   Point indexing:

       |\
       |2\
       |   \
       |4 0 5\
       |       \
       |3   6   1\
       ------------


*/

#define ROOT15 3.87298334620742

/* point coordinates */
#define Q1  (18 + 4*ROOT15)/21.0
#define Q2  (18 - 4*ROOT15)/21.0
#define Q3  (12 - 2*ROOT15)/21.0
#define Q4  (12 + 2*ROOT15)/21.0

REAL tq7_coord[FACET_INTERP][2] = { { 2/3.0, 2/3.0}, { Q1, Q3 }, { Q3, Q1 },
   { Q3, Q3 }, { Q2, Q4 }, { Q4, Q4 }, { Q4, Q2 } };

/* point weights */
#define W1  270/600.0
#define W2  (155 - ROOT15)/600.0
#define W3  (155 + ROOT15)/600.0

REAL tq7_weight[FACET_INTERP] = { W1, W2, W2, W2, W3, W3, W3 };

/* volume coefficients */
REAL vcoeff[FACET_CTRL][FACET_CTRL][FACET_CTRL];

/* precalculated interpolation polynomial values */
REAL poly6[FACET_INTERP][FACET_CTRL];

/* interpolation polynomial partials */
REAL polyparts[FACET_INTERP][FACET_CTRL][2];   /* integration point, control pt, partial */

/* interpolation polynomial second partials */
REAL poly2partial[FACET_CTRL][2][2] = { { {1.0, 1.0}, {1.0, 1.0} },
    {{-2.0, -1.0}, {-1.0, 0.0} },{ {1.0,0.0},{0.0,0.0} }, 
    { {0.0,1.0},{1.0,0.0} }, { {0.0,0.0},{0.0,1.0} },
    { {0.0,-1.0},{-1.0,2.0} } };

/************************************************************************
*
*  Calculates all forces on control points due to facet and
*  accumulates them at each control point.
*  Quadratic version.
*/

void facet_force_q(f_id)
facet_id f_id;
{
  vertex_id v_id[FACET_CTRL];
  REAL *x[FACET_CTRL];
  REAL *force[FACET_CTRL];
  REAL normal[MAXCOORD];
  int i,j,k,n;
  REAL norm,summer[2][MAXCOORD];
  facetedge_id fe_id;
  REAL density = get_facet_density(f_id);
  REAL area = 0.0;
  REAL gdensity = 0.0;
  REAL z;
  body_id b_id;
  REAL forces[FACET_CTRL][MAXCOORD];  /* total forces from this facet */
  REAL *forceptr[FACET_VERTS];   /* pointers to forces */
  WRAPTYPE wraps[FACET_VERTS];
  WRAPTYPE wrap;

  memset((char*)forces,0,sizeof(forces));  /* set to 0 */
  /* get control points */
  generate_facet_fe_init();
  for ( i = 0, j = 0, wrap = 0 ; i < FACET_EDGES ; i++ )
    { 
      generate_facet_fe(f_id,&fe_id);
      wraps[j] = wrap;
      v_id[j++] = get_fe_tailv(fe_id);
      wraps[j] = wrap;
      v_id[j++] = get_fe_midv(fe_id);
      if ( web.symmetry_flag )
        wrap = (*sym_compose)(wrap,get_fe_wrap(fe_id));
    }
    
  for ( j = 0 ; j < FACET_CTRL ; j++ )
    {
      x[j] = get_coord(v_id[j]);
      force[j] = get_force(v_id[j]);
      forceptr[j] = forces[j];
    }

  if ( web.metric_flag )
    { simplex_force_metric(v_id,x,density,forceptr);
      goto cumforces;  /* assume no gravity */
    }

  if ( web.gravflag )
   { 
     b_id = get_facet_body(f_id);
     if ( valid_id(b_id) )
        gdensity += get_body_density(b_id)*web.grav_const;
     b_id = get_facet_body(facet_inverse(f_id));
     if ( valid_id(b_id) )
        gdensity -= get_body_density(b_id)*web.grav_const;
    }
  for ( i = 0 ; i < gauss_num ; i++ )  /*  integration point number */
    {
      /* calculate tangents and normals */ 
      mat_mult(gpolypartial[i],x,tang,web.dimension,ctrl_num,
	  web.sdim);
      cross_prod(tang[0],tang[1],normal);
      norm = sqrt(dot(normal,normal,web.sdim));
      for ( n = 0 ; n < 2 ; n++ ) /* parameter number */
        cross_prod(normal,tang[n],summer[n]);

      area += density*gausswt[i]/2*norm;
      for ( k = 0 ; k < FACET_CTRL ; k++ )
       for ( j = 0 ; j < web.sdim ; j++ )
        forces[k][j] -= density*gausswt[i]/2/norm*
          (gpolypartial[i][1][k]*summer[0][j] 
	       - gpolypartial[i][0][k]*summer[1][j]);

     /* add gravitational force */
     if ( web.gravflag )
       { z = 0.0;
         for ( j = 0 ; j < FACET_CTRL ; j++ )
           z += gpoly[i][j]*x[j][2];
         for ( j = 0 ; j < FACET_CTRL ; j++ )
           { 
             forces[j][0] -= 0.5*z*z*gausswt[i]/2*gdensity
                             *(gpolypartial[i][0][j]*tang[1][1] -
                                       tang[0][1]*gpolypartial[i][1][j]);
             forces[j][1] -= 0.5*z*z*gausswt[i]/2*gdensity
                             *(tang[0][0]*gpolypartial[i][1][j] -
                                      gpolypartial[i][0][j]*tang[1][0]);
             forces[j][2] -= z*gpoly[i][j]*gausswt[i]/2*gdensity
                              *(tang[0][0]*tang[1][1] -
                                       tang[0][1]*tang[1][0]);
           } 
       }
   }


cumforces:
  /* add to totals, unwrapping if necessary */
  for ( i = 0 ; i < FACET_CTRL ; i++ )  /* vertex loop */
   { REAL *force; 
     REAL wforce[MAXCOORD];  /* unwrapped forces */

     force = get_force(v_id[i]);
     if ( web.symmetry_flag )
       { (*sym_form_pullback)(get_coord(v_id[i]),forces[i],wforce,wraps[i]);
	 for ( j = 0 ; j < web.sdim ; j++ )
	   force[j] += wforce[j];
       }
     else
	 for ( j = 0 ; j < web.sdim ; j++ )
	   force[j] += forces[i][j];
   }

  set_facet_area(f_id,area);
}

/*********************************************************************
*
*  Returns energy due to facet.
*  Quadratic version.
*/

void facet_energy_q(f_id)
facet_id f_id;
{
  REAL energy = 0.0;
  body_id b_id;
  REAL *x[FACET_CTRL];
  vertex_id v_id[FACET_CTRL];
  REAL normal[MAXCOORD];
  int i,j;
  REAL norm;
  facetedge_id fe_id;
  REAL u = 0.0;  /* gravitational integral */
  REAL z;
    
  /* get control points */
  generate_facet_fe_init();
  for ( i = 0, j = 0 ; i < FACET_EDGES ; i++ )
    { 
      generate_facet_fe(f_id,&fe_id);
      v_id[j++] = get_fe_tailv(fe_id);
      v_id[j++] = get_fe_midv(fe_id);
    }
  for ( j = 0 ; j < FACET_CTRL ; j++ )
    {
      x[j] = get_coord(v_id[j]);
    }
  if ( web.metric_flag )
    { energy = simplex_energy_metric(v_id,x);
      goto skip_from_metric;
    }

  for ( i = 0 ; i < gauss_num ; i++ )  /*  integration point number */
    {
      /* calculate tangents and normals */ 
      mat_mult(gpolypartial[i],x,tang,web.dimension,ctrl_num,
	  web.sdim);
      cross_prod(tang[0],tang[1],normal);
      norm = sqrt(dot(normal,normal,web.sdim));
      energy += gausswt[i]*norm/2;

      if ( web.gravflag )
       { z = 0.0;
         for ( j = 0 ; j < FACET_CTRL ; j++ )
           z += gpoly[i][j]*x[j][2];
         u += 0.5*z*z*gausswt[i]/2*(tang[0][0]*tang[1][1] -
                                       tang[0][1]*tang[1][0]);
       }
    }

skip_from_metric:
  set_facet_area(f_id,energy);
  web.total_area   += energy;
    
  /* apportion area to vertices and midpoints to scale motion */
  /* Note apportionment ratios depend on equivalent test      */
  /* function used to get curvature; here linear used         */
    {
      generate_facet_fe_init();
      while ( generate_facet_fe(f_id,&fe_id) )
       {
	 edge_id e_id = get_fe_edge(fe_id);
         vertex_id v_id = get_edge_headv(e_id);
         add_vertex_star(v_id,energy);
         v_id = get_edge_midv(e_id);
         add_vertex_star(v_id,energy);
        }
    }

  /* calculate surface energy */
  energy *= get_facet_density(f_id);
    
  /* add gravitational energy, vector potential z*z/2*k  */
  if ( web.gravflag )
   { b_id = get_facet_body(f_id);
     if ( valid_id(b_id) )
        energy += u*get_body_density(b_id)*web.grav_const;
     b_id = get_facet_body(facet_inverse(f_id));
     if ( valid_id(b_id) )
        energy -= u*get_body_density(b_id)*web.grav_const;
   }


  web.total_energy += energy;

}

/**********************************************************************
*
*  Find triangle's contribution to volumes of neighboring bodies.
*/

void facet_volume_q(f_id)
facet_id f_id;
{ 
  body_id b_id0,b_id1;
  REAL vol = 0.0;
  REAL *x[FACET_CTRL];
  vertex_id v_id[FACET_CTRL];
  int i,j,k;
  facetedge_id fe_id;

  b_id0 = get_facet_body(f_id);
  b_id1 = get_facet_body(facet_inverse(f_id));
    
  if ( !valid_id(b_id0) && !valid_id(b_id1) ) return;

  if ( web.symmetric_content )
    error("Symmetric content not enabled for quadratic model.\n",RECOVERABLE);

  /* get control points */
  generate_facet_fe_init();
  for ( i = 0, j = 0 ; i < FACET_EDGES ; i++ )
    { 
      generate_facet_fe(f_id,&fe_id);
      v_id[j++] = get_fe_tailv(fe_id);
      v_id[j++] = get_fe_midv(fe_id);
    }
  for ( j = 0 ; j < FACET_CTRL ; j++ )
    x[j] = get_coord(v_id[j]);

  /* volume, integral of z dx dy */
  for ( i = 0 ; i < FACET_CTRL ; i++ )
    for ( j = 0 ; j < FACET_CTRL ; j++ )
      for ( k = 0 ; k < FACET_CTRL ; k++ )
        vol += vcoeff[i][j][k]*x[i][2]*(x[j][0]*x[k][1]-x[j][1]*x[k][0]);

  /* add to body volumes */
  if ( valid_id(b_id0) )
    set_body_volume(b_id0,get_body_volume(b_id0) + vol);
  if ( valid_id(b_id1) )
    set_body_volume(b_id1,get_body_volume(b_id1) - vol);
}

/****************************************************************
*
*  Function:  film_grad_q()
*
*  Purpose:   Calculate volume gradients in SOAPFILM quadratic
*              model.
*/

void film_grad_q()
{
  body_id bi_id;  /* identifier for body i */
  body_id bj_id;  /* identifier for body j */
  facetedge_id fe_id;
  facet_id f_id;
  REAL g[MAXCOORD];
  vertex_id v_id[FACET_CTRL];
  REAL *x[FACET_CTRL];
  volgrad *vgptr;
  int i,j,k,n;

  FOR_ALL_FACETS(f_id)
   { 
     bi_id = get_facet_body(f_id);
     bj_id = get_facet_body(facet_inverse(f_id));
               
     /* get control points */
     generate_facet_fe_init();
     for ( i = 0, j = 0 ; i < FACET_EDGES ; i++ )
       { 
         generate_facet_fe(f_id,&fe_id);
         v_id[j++] = get_fe_tailv(fe_id);
         v_id[j++] = get_fe_midv(fe_id);
       }
     for ( j = 0 ; j < FACET_CTRL ; j++ )
       {
         x[j] = get_coord(v_id[j]);
       }

     for ( i = 0 ; i < FACET_CTRL ; i++ )
       {
         g[0] = g[1] = g[2] = 0.0;
         for ( j = 0 ; j < FACET_CTRL ; j++ ) 
           for ( k = 0 ; k < FACET_CTRL  ; k++ )
             {
               g[0] += vcoeff[k][i][j]*x[j][1]*x[k][2] 
                         - vcoeff[j][k][i]*x[j][2]*x[k][1];
               g[1] += vcoeff[j][k][i]*x[j][2]*x[k][0] 
                         - vcoeff[k][i][j]*x[j][0]*x[k][2];
               g[2] += vcoeff[i][j][k]*(x[j][0]*x[k][1] - x[j][1]*x[k][0]);
             }

         if ( valid_id(bi_id) && (get_battr(bi_id) & (PRESSURE|FIXEDVOL)) )
           { 
             vgptr = get_bv_new_vgrad(bi_id,v_id[i]);
             for ( n = 0 ; n < web.sdim ; n++ )
               vgptr->grad[n] += g[n];
           }

         if ( valid_id(bj_id) && (get_battr(bj_id) & FIXEDVOL) )
           { 
             vgptr = get_bv_new_vgrad(bj_id,v_id[i]);
             for ( n = 0 ; n < web.sdim ; n++ )
               vgptr->grad[n] -= g[n];
           }
       }
   }  
}

REAL tq7_integral(f)
#ifdef NOPROTO
REAL (*f)();
#else
REAL (*f)(REAL,REAL);
#endif
{
  int i;
  REAL sum = 0.0;

  for ( i = 0 ; i < FACET_INTERP ; i++ )
    sum += tq7_weight[i]*(*f)(tq7_coord[i][0],tq7_coord[i][1]);

  return sum;
}

/* Interpolation polynomials for 6 point interpolation on triangle.

   Triangle is: u > 0, v > 0, u + v < 2.

   Point indexing (counterclockwise):
      4
      5  3
      0  1  2

*/

/* interpolation polynomials */

REAL intpoly6(k,u,v)
int k;  /* point index */
REAL u,v;   /* evaluation point */
{
  switch ( k )
   {
     case 0: return (1 - (u + v))*(2 - (u + v))/2.0;
     case 1: return u*(2 - (u + v));
     case 2: return u*(u - 1)/2.0;
     case 3: return u*v;
     case 4: return v*(v - 1)/2.0;
     case 5: return v*(2 - (u + v));
   }

  /* bad index */
  return 0.0;
}

/* partials of interpolation polynomials */

REAL intpoly6part(k,i,u,v)
int k;  /* point index */
int i;  /* partial number: 0 for u, 1 for v */
REAL u,v;   /* evaluation point */
{
  if ( i == 0 )
   switch ( k )
    {
      case 0: return u + v - 1.5;
      case 1: return 2 - 2*u - v;
      case 2: return u - 0.5;
      case 3: return v;
      case 4: return 0.0;
      case 5: return -v;
    }
  else if ( i == 1 )
   switch ( k )
    {
      case 0: return u + v - 1.5;
      case 1: return -u;
      case 2: return 0.0;
      case 3: return u;
      case 4: return v - 0.5;
      case 5: return 2 - u - 2*v;
    }

  /* bad index */
  return 0.0;
}


/* finding antisymmetric coefficients for volume integrals */

/* integrand function */
static int al,be,ga;
REAL vintzf(u,v)
REAL u,v;
{ 
   return 
     intpoly6(al,u,v)*intpoly6part(be,0,u,v)*intpoly6part(ga,1,u,v);
}

void vcoeff_init()
{
  int i,k,n;
  REAL t;

  for ( al = 0 ; al < FACET_CTRL ; al++ )
    for ( be = 0; be < FACET_CTRL ; be++ )
      for ( ga = 0 ; ga < FACET_CTRL ; ga++ )
        { t = tq7_integral(vintzf);
          /* round to nearest 90th */
          t = (int)(t*90.001)/90.0;
          vcoeff[al][be][ga] = t;
        }

  /* do interpolation polynomial precalculation */
  for ( i = 0 ; i < FACET_INTERP ; i++ )
    for ( k = 0 ; k < FACET_CTRL ; k++ )
        poly6[i][k] = intpoly6(k,tq7_coord[i][0],tq7_coord[i][1]);

  /* also do polyparts */
  for ( i = 0 ; i < FACET_INTERP ; i++ )
    for ( k = 0 ; k < FACET_CTRL ; k++ )
      for ( n = 0 ; n < 2 ; n++ )
        polyparts[i][k][n] = intpoly6part(k,n,tq7_coord[i][0],tq7_coord[i][1]);
}
