/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*****************************************************************
*
*  File: express.c
*
*  Purpose: To execute user functions for evolver constraints. 
*           Evaluation takes recursively.  An array of 
*           parameters is available to the evaluator.
*              
*   Internal form of expressions is tree stored in pre-order form
*   with links.
*      
*/


#include "include.h" 
#include "express.h"
#include "ytab.h"

#ifdef EXPTEST
struct web web;
#endif

/*****************************************************************
*
*  Function eval()
*
*  Purpose: runtime evaluation of function.
*
*/


REAL eval(ex,params)
struct expnode *ex;      /* expression tree */
REAL *params;    /* vector of parameters */
{
  int n;
  REAL x;

  REAL stack[100];
  register REAL *stacktop = stack;
  register struct evalnode *node;
  
  if ( ex == NULL ) return 0.0;

  if ( ex->evallist == NULL )
    { error("Trying to evaluate null expression.\n",WARNING);
      return 0.0;
    }

  *stacktop = 0.0;  /* for empty expression */
  for ( node = ex->evallist ; ; node++ )
    switch ( node->type )
      {
        case REPLACECONST:
          *stacktop = node->value;
	  break;

        case PUSHCONST:
          *++stacktop = node->value;
	  break;

        case PUSHADJUSTABLE:
	  if ( web.params[node->inx].flags & FILE_VALUES )
	    *++stacktop = web.params[node->inx].values[int_val];
	  else
	    *++stacktop = web.params[node->inx].value;
	  break;

        case PUSHPI:
          *++stacktop = M_PI;
	  break;

        case PUSHE:
          *++stacktop = M_E;
	  break;

        case PUSHG:
          *++stacktop = web.grav_const;
	  break;

        case PUSHPARAM:
          *++stacktop = params[node->inx];
	  break;

        case USERFUNC:
	  *++stacktop = (*userfunc[node->inx])(params);
	  break;

        case '>':
	  stacktop--;
	  stacktop[0] = (REAL)(stacktop[0] > stacktop[1]);
	  break;

        case '<':
	  stacktop--;
	  stacktop[0] = (REAL)(stacktop[0] < stacktop[1]);
	  break;

        case LE_:
	  stacktop--;
	  stacktop[0] = (REAL)(stacktop[0] <= stacktop[1]);
	  break;

        case GE_:
	  stacktop--;
	  stacktop[0] = (REAL)(stacktop[0] >= stacktop[1]);
	  break;

        case NE_:
	  stacktop--;
	  stacktop[0] = (REAL)(stacktop[0] != stacktop[1]);
	  break;

        case EQ_:
	  stacktop--;
	  stacktop[0] = (REAL)(stacktop[0] == stacktop[1]);
	  break;

        case AND_:
	  stacktop--;
	  stacktop[0] = (REAL)(stacktop[0] && stacktop[1]);
	  break;

        case OR_:
	  stacktop--;
	  stacktop[0] = (REAL)(stacktop[0] || stacktop[1]);
	  break;

        case NOT_:
	  stacktop[0] = (REAL)(!stacktop[0]);
	  break;

        case PLUS:
	  stacktop--;
          stacktop[0] += stacktop[1];
	  break;

        case MINUS:
        case EQUATE:
	  stacktop--;
          stacktop[0] -= stacktop[1];
	  break;

        case TIMES:
	  stacktop--;
          stacktop[0] *= stacktop[1];
	  break;

        case DIVIDE:
	  stacktop--;
          stacktop[0] /= stacktop[1];
	  break;

        case INTPOW:
          if ( node->inx == 0 ) 
	    { *stacktop = 1.0;
	       break;
	    }
          x = *stacktop;
          for ( n = 1 ; n < abs(node->inx) ; n++ )
            *stacktop *= x;
          if ( node->inx < 0 )
            *stacktop = 1/(*stacktop);
	  break;

        case POW:
	  stacktop--;
          *stacktop = pow(stacktop[0],stacktop[1]);
	  break;

        case SQR:
          *stacktop *= *stacktop;
	  break;

        case SQRT:
          *stacktop = sqrt(*stacktop);
	  break;

        case ABS:
          *stacktop = fabs(*stacktop);
	  break;

        case SIN:
          *stacktop = sin(*stacktop);
	  break;

        case COS:
          *stacktop = cos(*stacktop);
	  break;

        case TAN:
          *stacktop = tan(*stacktop);
	  break;

        case EXP:
          *stacktop = exp(*stacktop);
	  break;

        case LOG:
          *stacktop = log(*stacktop);
	  break;

        case ASIN:
          *stacktop = asin(*stacktop);
	  break;

        case ACOS:
          *stacktop = acos(*stacktop);
	  break;

        case ATAN:
          *stacktop = atan(*stacktop);
	  break;
        
        case CHS:
          *stacktop = -*stacktop;
	  break;
        
        case INV:
          *stacktop = 1/(*stacktop);
	  break;

        /* here are attributes for queries */
	case COORD_:
          *++stacktop = get_coord(q_id)[node->inx];
	  break;

        case LENGTH_:
	  calc_edge(q_id);
          *++stacktop = get_edge_length(q_id);
	  break;

        case VALENCE_:
          *++stacktop = get_edge_valence(q_id);
	  break;

	case AREA_:
	  *++stacktop = get_facet_area(q_id);
	  break;

	case DENSITY_:
	  switch ( celement )
	    { case EDGES_: *++stacktop = get_edge_density(q_id); break;
	      case FACETS_: *++stacktop = get_facet_density(q_id); break;
	      case BODIES_: *++stacktop = get_body_density(q_id); break;
	      default: error("Density of wrong type element.\n",RECOVERABLE);
	    }
	  break;

	case VOLUME_:
	  *++stacktop = get_body_volume(q_id);
	  break;

	case ID_:
	  *++stacktop = (REAL)(ordinal(q_id)+1);
	  break;

	case ORIGINAL_:
	  *++stacktop = (REAL)get_original(q_id);
	  break;

	case TAG_:
	  *++stacktop = (REAL)get_tag(q_id);
	  break;

	case FIXED_:
	  *++stacktop = get_attr(q_id) & FIXED;
	  break;

	case FINISHED:
          return *stacktop; 

	default:
	  error("Bad expression node type.",UNRECOVERABLE);
	  break;
        
    }     

/*  return *stacktop;  */

}

#ifdef TREE_EVAL
REAL tree_eval(node,params)
struct treenode *node;      /* expression tree */
REAL *params;    /* vector of paramters */
{
  REAL x,y;
  int n;

  if ( node == NULL ) return 0.0;

    switch ( node->type )
      {
        case PUSHCONST:
          return node->value;

        case PUSHADJUSTABLE:
	  if ( web.params[node->inx].flags & FILE_VALUES )
	    return web.params[node->inx].values[int_val];
	  else
	    return web.params[node->inx].value;

        case PUSHPI:
          return M_PI;

        case PUSHE:
          return M_E;

        case PUSHG:
          return web.grav_const;

        case PUSHPARAM:
          return params[node->inx];

        case USERFUNC:
	  return (*userfunc[node->inx])(params);

        case PLUS:
          return tree_eval(node+node->left,params) + tree_eval(node+node->right,params);

        case MINUS:
        case EQUATE:
          return tree_eval(node+node->left,params) - tree_eval(node+node->right,params);

        case TIMES:
          return tree_eval(node+node->left,params) * tree_eval(node+node->right,params);

        case DIVIDE:
          return tree_eval(node+node->left,params) / tree_eval(node+node->right,params);

        case INTPOW:
          if ( node->inx == 0 ) return 1.0;
          y = x = tree_eval(node+node->left,params);
          for ( n = 1 ; n < abs(node->inx) ; n++ )
            y *= x;
          if ( node->inx < 0 )
            return 1/y;
          else return y;

        case POW:
          return pow(tree_eval(node+node->left,params),tree_eval(node+node->right,params));

        case SQR:
          x = tree_eval(node+node->left,params);
          return x*x;

        case SQRT:
          return sqrt(tree_eval(node+node->left,params));

        case ABS:
          return fabs(tree_eval(node+node->left,params));

        case SIN:
          return sin(tree_eval(node+node->left,params));

        case COS:
          return cos(tree_eval(node+node->left,params));

        case TAN:
          return tan(tree_eval(node+node->left,params));

        case EXP:
          return exp(tree_eval(node+node->left,params));

        case LOG:
          return log(tree_eval(node+node->left,params));

        case ASIN:
          return asin(tree_eval(node+node->left,params));

        case ACOS:
          return acos(tree_eval(node+node->left,params));

        case ATAN:
          return atan(tree_eval(node+node->left,params));
        
        case CHS:
          return -tree_eval(node+node->left,params);
        
        case INV:
          return 1/tree_eval(node+node->left,params);
        
	default:
	  fprintf(stderr,"tree_eval: invalid node type: %d\n",node->type);
	  exit(2);
	  break;
    }
  return 0.0; /* shouldn't happen */
}
#endif

#ifdef EVAL_DERIV
/*****************************************************************
*
*  Function eval_deriv()
*
*  Purpose: runtime evaluation of partial derivative of function.
*
*/

static  struct dstack { REAL value, deriv; } stack[100];

REAL eval_deriv(ex,params,var)
struct expnode *ex;      /* expression tree */
REAL *params;    /* vector of paramters */
int var;           /* which parameter for partial */
{
  int n;
  REAL x,y;
  register struct dstack *stacktop = stack;
  register struct evalnode *node;

  if ( ex == NULL ) return 0.0;

  node = ex->evallist;

  stacktop->value = stacktop->deriv = 0.0;  /* for empty expression */
  for ( node = ex->evallist ; ; node++ )
    switch ( node->type )
      {
        case REPLACECONST:
          stacktop->deriv = 0.0;
          stacktop->value = node->value;
	  break;

        case PUSHCONST:
        case PUSHPI:
        case PUSHE:
        case PUSHG:
          (++stacktop)->value = node->value;
          stacktop->deriv = 0.0;
	  break;

        case PUSHADJUSTABLE:
	  if ( web.params[node->inx].flags & FILE_VALUES )
	    (++stacktop)->value = web.params[node->inx].values[int_val];
          else (++stacktop)->value = node->value;
          stacktop->deriv = 0.0;
	  break;

        case PUSHPARAM:
          (++stacktop)->value = params[node->inx];
	  if ( var == node->inx )
	    stacktop->deriv = 1.0;
	  else stacktop->deriv = 0.0;
	  break;

        case PLUS:
	  stacktop--;
          stacktop[0].value += stacktop[1].value;
          stacktop[0].deriv += stacktop[1].deriv;
	  break;

        case MINUS:
        case EQUATE:
	  stacktop--;
          stacktop[0].value -= stacktop[1].value;
          stacktop[0].deriv -= stacktop[1].deriv;
	  break;

        case TIMES:
	  stacktop--;
          stacktop[0].deriv = stacktop[1].value*stacktop[0].deriv
		  + stacktop[0].value*stacktop[1].deriv;
          stacktop[0].value *= stacktop[1].value;
	  break;

        case DIVIDE:
	  stacktop--;
          stacktop[0].deriv = (stacktop[1].value*stacktop[0].deriv
		  - stacktop[0].value*stacktop[1].deriv)
		  /stacktop[1].value/stacktop[1].value;
          stacktop[0].value /= stacktop[1].value;
	  break;

        case INTPOW:
	  /* power 0 and 1 cases already folded out */
          x = stacktop->value;
	  if ( node->inx > 0 )
           for ( n = 2 ; n < node->inx ; n++ )
            stacktop->value *= x;
	  else
	    { stacktop->value = 1/x;
              for ( n = 0 ; n < -node->inx ; n++ )
                stacktop->value *= 1/x;
	    }
	  stacktop->deriv = node->inx*stacktop->value*stacktop->deriv;
	  stacktop->value *= x;
	  break;

        case POW:
	  stacktop--;
          x = stacktop[0].value;
          y = stacktop[1].value;
          stacktop->value = pow(x,y);
	  stacktop->deriv = (log(x)*stacktop[1].deriv + y/x*stacktop[0].deriv)
			       *stacktop->value;
	  break;

        case SQR:
	  stacktop->deriv *= 2*stacktop->value;
          stacktop->value *= stacktop->value;
	  break;

        case SQRT:
          stacktop->value = sqrt(stacktop->value);
	  stacktop->deriv /= 2*stacktop->value;
	  break;

        case ABS:
	  stacktop->deriv *= stacktop->value > 0 ? 1 : -1;
          stacktop->value = fabs(stacktop->value);
	  break;

        case SIN:
	  stacktop->deriv *= cos(stacktop->value);
          stacktop->value = sin(stacktop->value);
	  break;

        case COS:
	  stacktop->deriv *= -sin(stacktop->value);
          stacktop->value = cos(stacktop->value);
	  break;

        case ATAN:
	  stacktop->deriv /= (1 + stacktop->value*stacktop->value);
          stacktop->value = atan(stacktop->value);
	  break;

        case EXP:
          stacktop->value = exp(stacktop->value);
	  stacktop->deriv *= stacktop->value;
	  break;

        case LOG:
	  stacktop->deriv /= stacktop->value;
          stacktop->value = log(stacktop->value);
	  break;

        case ASIN:
	  stacktop->deriv /= sqrt(1 - stacktop->value*stacktop->value);
          stacktop->value = asin(stacktop->value);
	  break;

        case ACOS:
	  stacktop->deriv /= -sqrt(1 - stacktop->value*stacktop->value);
          stacktop->value = acos(stacktop->value);
	  break;

        case TAN:
          stacktop->value = tan(stacktop->value);
	  stacktop->deriv *= 1+stacktop->value*stacktop->value;
	  break;
        
        case CHS:
	  stacktop->deriv = -stacktop->deriv;
          stacktop->value = -stacktop->value;
	  break;
        
        case INV:
	  stacktop->deriv /= stacktop->value*stacktop->value;
          stacktop->value = 1/stacktop->value;
	  break;

	case FINISHED:
          return stacktop->deriv; 

	default:
	  error("Bad expression node type.",UNRECOVERABLE);
	  break;
        
    }     

/*   return stacktop->deriv; */
}
#endif

#ifdef TREE_EVAL
REAL tree_eval_deriv(node,params,var)
struct treenode *node;      /* expression tree */
REAL *params;    /* vector of paramters */
int var;           /* which parameter for partial */
{
  REAL x,y;
  REAL t1,t2,t3,t4; /* temps to get math stack from getting too deep */
  int n;

  if ( node == NULL ) return 0.0;

    switch ( node->type )
      {
        case PUSHCONST:
        case PUSHPI:
        case PUSHE:
        case PUSHG:
	case PUSHADJUSTABLE:
          return 0.0;

        case PUSHPARAM:
          if ( node->inx == var ) return 1.0;
          else return 0.0;

        case PLUS:
          return tree_eval_deriv(node+node->left,params,var)
               + tree_eval_deriv(node+node->right,params,var);

        case MINUS:
        case EQUATE:
          return tree_eval_deriv(node+node->left,params,var)
               - tree_eval_deriv(node+node->right,params,var);

        case TIMES:
          t1 = tree_eval_deriv(node+node->left,params,var);
          t2 = tree_eval(node+node->right,params);
          t3 = tree_eval(node+node->left,params);
          t4 = tree_eval_deriv(node+node->right,params,var);
          return t1*t2 + t3*t4;

        case DIVIDE:
          x = tree_eval(node+node->right,params);
	  if ( x == 0.0 ) error("tree_eval_deriv divide by 0\n",RECOVERABLE);
          t1 = tree_eval_deriv(node+node->left,params,var);
          t2 = tree_eval(node+node->left,params);
          t3 = tree_eval_deriv(node+node->right,params,var);
          return (t1*x - t2*t3)/x/x;

        case INTPOW:
          if ( node->inx == 0 ) return 0.0;
          y = tree_eval_deriv(node+node->left,params,var);
          if ( node->inx == 1 ) return y;
          x = tree_eval(node+node->left,params);
          if ( node->inx < 0 )
            for ( n = 0 ; n <= abs(node->inx) ; n++ )
                y /= x;
          else 
            for ( n = 1 ; n < abs(node->inx) ; n++ )
              y *= x;
          return node->inx*y;

        case POW:
          x = tree_eval(node+node->left,params);
          y = tree_eval(node+node->right,params);
          return pow(x,y)*(log(x)*tree_eval_deriv(node+node->right,params,var)
             + y/x*tree_eval_deriv(node+node->left,params,var));

        case SQR:
          return 2*tree_eval(node+node->left,params)*
                     tree_eval_deriv(node+node->left,params,var);

        case SQRT:
          return 0.5*tree_eval_deriv(node+node->left,params,var)
                         /sqrt(tree_eval(node+node->left,params));

        case ABS:
          return (tree_eval(node+node->left,params)>0 ? 1 : -1)
                   * tree_eval_deriv(node+node->left,params,var);
        case SIN:
          return cos(tree_eval(node+node->left,params))
                   * tree_eval_deriv(node+node->left,params,var);

        case COS:
          return -sin(tree_eval(node+node->left,params))
                   * tree_eval_deriv(node+node->left,params,var);

        case TAN:
          x = cos(tree_eval(node+node->left,params));
          return tree_eval_deriv(node+node->left,params,var)/x/x;

        case EXP:
          return exp(tree_eval(node+node->left,params))
                   * tree_eval_deriv(node+node->left,params,var);

        case LOG:
          return  tree_eval_deriv(node+node->left,params,var)
                    /tree_eval(node+node->left,params);

        case ASIN:
          x = tree_eval(node+node->left,params);
          return tree_eval_deriv(node+node->left,params,var)/sqrt(1 - x*x);

        case ACOS:
          x = tree_eval(node+node->left,params);
          return -tree_eval_deriv(node+node->left,params,var)/sqrt(1 - x*x);

        case ATAN:
          x = tree_eval(node+node->left,params);
          return tree_eval_deriv(node+node->left,params,var)/(1 + x*x);
        
        case CHS:
          return -tree_eval_deriv(node+node->left,params,var);
        
        case INV:
          x = tree_eval(node+node->left,params);
          return -tree_eval_deriv(node+node->left,params,var)/x/x;
        
    }
  return 0.0; /* shouldn't happen */
}
#endif

  

/********************************************************************
*
*  Function: convert_all_expr()
*
*  Purpose:  Converts all expressions in use to stack form.
*
*/

void convert_all_expr()
{
   int n,i,j,k;

   for ( n = 0 ; n < web.concount ; n++ )
     { struct constraint *con = get_constraint(n);
       expr_convert(con->formula);
       for ( j = 0 ; j < MAXCONCOMP ; j++ )
         { expr_convert(con->envect[j]);
           expr_convert(con->convect[j]);
	 }
       for ( k = 0 ; k < web.quantity_count ; k++ )
         if ( con->quantity_map & (1<<k) )
           for ( j = 0 ; j < MAXCONCOMP ; j++ )
             expr_convert(con->quanvect[k][j]);
     }

   for ( n = 0 ; n < BDRYMAX ; n++ )
     { 
       for ( j = 0 ; j < web.sdim ; j++ )
         expr_convert(web.boundaries[n].coordf[j]);
     }

   for ( n = 0 ; n < web.surfen_count ; n++ )
     { 
       for ( j = 0 ; j < web.sdim ; j++ )
         expr_convert(web.surfen[n].envect[j]);
     }

   for ( n = 0 ; n < web.quantity_count ; n++ )
     { 
       if ( web.quants[n].attr & IN_USE )
        for ( j = 0 ; j < web.sdim ; j++ )
         expr_convert(web.quants[n].quanvect[j]);
     }

   if ( web.metric_flag )
     for ( i = 0 ; i < web.sdim ; i++ )
       for ( j = 0 ; j < web.sdim ; j++ )
	 expr_convert(&web.metric[i][j]);
}

/*******************************************************************
*
*  Function: expr_convert()
*
*  Purpose:  Convert expression tree to evaluaton stack list,
*            folding all constant expressions (including
*            adjustable parameters).  Call for all expressions
*            whenever adjustable parameters change.
*/

void expr_convert(ex)
struct expnode *ex;
{
  int n;
  struct treenode *tree;
  struct evalnode *e;

  if ( (ex == NULL) || (ex->root == NULL) ) return;

  /* allocate storage */
  if ( ex->evallist ) free((char*)ex->evallist);
  ex->evallist = (struct evalnode *)mycalloc(500,sizeof(struct evalnode));
  if ( ex->evallist == NULL )
	error("Cannot allocate evallist memory.",UNRECOVERABLE);

  /* copy over tree, which is in proper order, with constant folding */
  for ( n = 0, tree = ex->root+1, e = ex->evallist ; n < 500 ; n++,tree++,e++ )
   {  
      switch ( tree->type )
	{ 
	  case PUSHCONST:
	  case PUSHPI:
	  case PUSHE:
	    e->type = PUSHCONST;
	    e->value = tree->value;
	    break;
	
	  case PUSHG:     /* Gravity can now be folded */
	    e->type = PUSHCONST;
	    e->value = web.grav_const;
	    break;

	  case PUSHADJUSTABLE:     /* Adjustable constants can be folded */
	    if ( web.params[tree->inx].flags & FILE_VALUES )
	       e->type = tree->type;
	    else  
	      { e->value = web.params[tree->inx].value;
	        e->type = PUSHCONST;
	      }
	    break;

          case USERFUNC:
	    e->type = tree->type;
	    e->inx = tree->inx;
	    break;

	  case REPLACECONST:
	    e->type = REPLACECONST;
	    e->value = tree->value;
	    break;
	
	  case PLUS:
	    if ( (e[-1].type == PUSHCONST) && (e[-2].type == PUSHCONST) )
	      { e -= 2;
		e->value = e->value + e[1].value;
	      }
	    else
              e->type = tree->type;
            break;

	  case MINUS:
	  case EQUATE:
	    if ( (e[-1].type == PUSHCONST) && (e[-2].type == PUSHCONST) )
	      { e -= 2;
		e->value = e->value - e[1].value;
	      }
	    else
              e->type = tree->type;
            break;

	  case TIMES:
	    if ( (e[-1].type == PUSHCONST) && (e[-2].type == PUSHCONST) )
	      { e -= 2;
		e->value = e->value * e[1].value;
	      }
	    else
              e->type = tree->type;
            break;

	  case DIVIDE:
	    if ( (e[-1].type == PUSHCONST) && (e[-2].type == PUSHCONST) )
	      { e -= 2;
		e->value = e->value / e[1].value;
	      }
	    else
              e->type = tree->type;
            break;

	  case POW:
	    if ( (e[-1].type == PUSHCONST) && (e[-2].type == PUSHCONST) )
	      { e -= 2;
		e->value = pow(e->value,e[1].value);
	      }
	    else
              e->type = tree->type;
            break;

	  case EXP:
	    if ( e[-1].type == PUSHCONST )
	      { e--;
		e->value = exp(e->value);
	      }
	    else
              e->type = tree->type;
            break;

	  case LOG:
	    if ( e[-1].type == PUSHCONST )
	      { e--;
		e->value = log(e->value);
	      }
	    else
              e->type = tree->type;
            break;

	  case ABS:
	    if ( e[-1].type == PUSHCONST )
	      { e--;
		e->value = fabs(e->value);
	      }
	    else
              e->type = tree->type;
            break;

	  case SIN:
	    if ( e[-1].type == PUSHCONST )
	      { e--;
		e->value = sin(e->value);
	      }
	    else
              e->type = tree->type;
            break;

	  case ASIN:
	    if ( e[-1].type == PUSHCONST )
	      { e--;
		e->value = asin(e->value);
	      }
	    else
              e->type = tree->type;
            break;

	  case COS:
	    if ( e[-1].type == PUSHCONST )
	      { e--;
		e->value = cos(e->value);
	      }
	    else
              e->type = tree->type;
            break;

	  case ACOS:
	    if ( e[-1].type == PUSHCONST )
	      { e--;
		e->value = acos(e->value);
	      }
	    else
              e->type = tree->type;
            break;

	  case TAN:
	    if ( e[-1].type == PUSHCONST )
	      { e--;
		e->value = tan(e->value);
	      }
	    else
              e->type = tree->type;
            break;

	  case ATAN:
	    if ( e[-1].type == PUSHCONST )
	      { e--;
		e->value = atan(e->value);
	      }
	    else
              e->type = tree->type;
            break;

	  case SQR:
	    if ( e[-1].type == PUSHCONST )
	      { e--;
		e->value *= e->value;
	      }
	    else
              e->type = tree->type;
            break;

	  case SQRT:
	    if ( e[-1].type == PUSHCONST )
	      { e--;
		e->value = sqrt(e->value);
	      }
	    else
              e->type = tree->type;
            break;

	  case CHS:
	    if ( e[-1].type == PUSHCONST )
	      { e--;
		e->value = -e->value;
	      }
	    else
              e->type = tree->type;
            break;

	  case INTPOW:
	    if ( e[-1].type == PUSHCONST )  /* powers 0, 1 already folded */
	      { int k;
		REAL x;

		e--;
		x = e->value;
		for ( k = 1 ; k < abs(tree->inx) ; k++ )
                  e->value *= x;
		if ( tree->inx < 0 ) 
		  e->value = 1/e->value;
	      }
	    else
              { e->type = tree->type;
		e->inx  = tree->inx;
	      }
            break;

	  case COORD_:
	  case PUSHPARAM:
            e->type = tree->type;
	    e->inx  = tree->inx;
            break;

       case EQ_:
       case NE_:
       case GE_:
       case LE_:
       case '>':
       case '<':
       case AND_:
       case OR_:
       case NOT_:
	  case LENGTH_:
	  case VALENCE_:
	  case AREA_:
	  case VOLUME_:
	  case DENSITY_:
	  case ID_:
	  case TAG_:
	  case ORIGINAL_:
	  case FIXED_:
	    e->type = tree->type;
	    break;

          case FINISHED:
	    e->type = tree->type;
	    goto windup;

	  default: 
	    sprintf(errmsg,"Bad node type in expr_convert: %d\n",tree->type);
	    error(errmsg,RECOVERABLE);
	}

   }

windup:
  /* get rid of extra */
  ex->evallist = (struct evalnode *)kb_realloc((char *)ex->evallist,
				 (e-ex->evallist+1)*sizeof(struct evalnode));

}



/*****************************************************************
*
*  Function eval_all()
*
*  Purpose: runtime tree_evaluation of function and all of its
*  partial derivatives.
*
*/


void eval_all(ex,params,pcount,fval,partials)
struct expnode *ex;      /* expression tree */
REAL *params;    /* vector of paramters */
int  pcount;     /* number of variables */
REAL *fval;      /* function value */
REAL *partials;  /* values of partials */
{
  int i,n;
  REAL x,y;
  struct dstack { REAL value, deriv[MAXCOORD]; } stack[100];
  register struct dstack *stacktop = stack;
  register struct evalnode *node;

  if ( pcount > web.sdim )
     error("More variables than MAXCOORD in eval_all().\n",RECOVERABLE);

  for ( i = 0 ; i < pcount ; i++ )
     stacktop->deriv[i] = 0.0;
  stacktop->value = 0.0;  /* for empty expression */
  if ( ex == NULL ) return;

  for ( node = ex->evallist ; ; node++ )
    switch ( node->type )
      {
        case REPLACECONST:
          for ( i = 0 ; i < pcount ; i++ )
            stacktop->deriv[i] = 0.0;
          stacktop->value = node->value;
	  break;

        case PUSHCONST:
        case PUSHPI:
        case PUSHE:
        case PUSHG:
          (++stacktop)->value = node->value;
          for ( i = 0 ; i < pcount ; i++ )
            stacktop->deriv[i] = 0.0;
	  break;

        case PUSHADJUSTABLE:
	  if ( web.params[node->inx].flags & FILE_VALUES )
	    (++stacktop)->value = web.params[node->inx].values[int_val];
          else (++stacktop)->value = node->value;
          for ( i = 0 ; i < pcount ; i++ )
            stacktop->deriv[i] = 0.0;
	  break;

        case PUSHPARAM:
          (++stacktop)->value = params[node->inx];
          for ( i = 0 ; i < pcount ; i++ )
	    if ( i == node->inx )
	      stacktop->deriv[i] = 1.0;
	    else stacktop->deriv[i] = 0.0;
	  break;

        case USERFUNC:
	  stacktop++;
	  stacktop->value =
	    (*userfunc_deriv[node->inx])(params,stacktop->deriv);
          break;

        case PLUS:
	  stacktop--;
          stacktop[0].value += stacktop[1].value;
          for ( i = 0 ; i < pcount ; i++ )
            stacktop[0].deriv[i] += stacktop[1].deriv[i];
	  break;

        case MINUS:
        case EQUATE:
	  stacktop--;
          stacktop[0].value -= stacktop[1].value;
          for ( i = 0 ; i < pcount ; i++ )
            stacktop[0].deriv[i] -= stacktop[1].deriv[i];
	  break;

        case TIMES:
	  stacktop--;
          for ( i = 0 ; i < pcount ; i++ )
            stacktop[0].deriv[i] = stacktop[1].value*stacktop[0].deriv[i]
		  + stacktop[0].value*stacktop[1].deriv[i];
          stacktop[0].value *= stacktop[1].value;
	  break;

        case DIVIDE:
	  stacktop--;
          for ( i = 0 ; i < pcount ; i++ )
            stacktop[0].deriv[i] = (stacktop[1].value*stacktop[0].deriv[i]
		  - stacktop[0].value*stacktop[1].deriv[i])
		  /stacktop[1].value/stacktop[1].value;
          stacktop[0].value /= stacktop[1].value;
	  break;

        case INTPOW:
	  /* power 0 and 1 cases folded out */
          x = stacktop->value;
	  if ( node->inx > 0 )
           for ( n = 2 ; n < node->inx ; n++ )
            stacktop->value *= x;
	  else
	    { stacktop->value = 1/x;
              for ( n = 0 ; n < -node->inx ; n++ )
                stacktop->value *= 1/x;
	    }
          for ( i = 0 ; i < pcount ; i++ )
	    stacktop->deriv[i] = node->inx*stacktop->value*stacktop->deriv[i];
	  stacktop->value *= x;
	  break;

        case POW:
	  stacktop--;
          x = stacktop[0].value;
          y = stacktop[1].value;
          stacktop->value = pow(x,y);
          for ( i = 0 ; i < pcount ; i++ )
	    stacktop->deriv[i] = (log(x)*stacktop[1].deriv[i]
                 + y/x*stacktop[0].deriv[i]) *stacktop->value;
	  break;

        case SQR:
          for ( i = 0 ; i < pcount ; i++ )
	    stacktop->deriv[i] *= 2*stacktop->value;
          stacktop->value *= stacktop->value;
	  break;

        case SQRT:
          stacktop->value = sqrt(stacktop->value);
          for ( i = 0 ; i < pcount ; i++ )
	    stacktop->deriv[i] /= 2*stacktop->value;
	  break;

        case ABS:
          for ( i = 0 ; i < pcount ; i++ )
	    stacktop->deriv[i] *= (stacktop->value > 0) ? 1 : -1;
          stacktop->value = fabs(stacktop->value);
	  break;

        case SIN:
          for ( i = 0 ; i < pcount ; i++ )
	    stacktop->deriv[i] *= cos(stacktop->value);
          stacktop->value = sin(stacktop->value);
	  break;

        case COS:
          for ( i = 0 ; i < pcount ; i++ )
	    stacktop->deriv[i] *= -sin(stacktop->value);
          stacktop->value = cos(stacktop->value);
	  break;

        case ATAN:
          for ( i = 0 ; i < pcount ; i++ )
	    stacktop->deriv[i] /= (1 + stacktop->value*stacktop->value);
          stacktop->value = atan(stacktop->value);
	  break;

        case EXP:
          stacktop->value = exp(stacktop->value);
          for ( i = 0 ; i < pcount ; i++ )
	    stacktop->deriv[i] *= stacktop->value;
	  break;

        case LOG:
          for ( i = 0 ; i < pcount ; i++ )
	    stacktop->deriv[i] /= stacktop->value;
          stacktop->value = log(stacktop->value);
	  break;

        case ASIN:
          for ( i = 0 ; i < pcount ; i++ )
	    stacktop->deriv[i] /= sqrt(1 - stacktop->value*stacktop->value);
          stacktop->value = asin(stacktop->value);
	  break;

        case ACOS:
          for ( i = 0 ; i < pcount ; i++ )
	    stacktop->deriv[i] /= -sqrt(1 - stacktop->value*stacktop->value);
          stacktop->value = acos(stacktop->value);
	  break;

        case TAN:
          stacktop->value = tan(stacktop->value);
          for ( i = 0 ; i < pcount ; i++ )
	    stacktop->deriv[i] *= 1+stacktop->value*stacktop->value;
	  break;
        
        case CHS:
          for ( i = 0 ; i < pcount ; i++ )
	    stacktop->deriv[i] = -stacktop->deriv[i];
          stacktop->value = -stacktop->value;
	  break;
        
        case INV:
          for ( i = 0 ; i < pcount ; i++ )
	    stacktop->deriv[i] /= stacktop->value*stacktop->value;
          stacktop->value = 1/stacktop->value;
	  break;

	case FINISHED:
          *fval = stacktop->value;
          for ( i = 0 ; i < pcount ; i++ )
            partials[i] = stacktop->deriv[i]; 
          return;

	default:
	  error("Bad expression node type.",UNRECOVERABLE);
	  break;
        
    }     

}
  

