/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/****************************************************************
*
*  File: cnstrnt.c
*
*  Contents:  Functions dealing with boundary constraints.
*/

#include "include.h"

/*******************************************************************
*
*  function: constraint_init()
*
*  purpose:  allocate expnode structures for a constraint
*
*******/

void constraint_init(con)
struct constraint *con;
{ int i;
  con->formula = (struct expnode *)mycalloc(1+2*MAXCONCOMP,
     sizeof(struct expnode));
  if ( con->formula == NULL )
    error("Cannot allocate constraint memory.",UNRECOVERABLE);
  for ( i = 0 ; i < MAXCONCOMP ; i++ )
    {
      con->envect[i] = con->formula + 1 + i;
      con->convect[i] = con->formula + 1 + MAXCONCOMP + i;
    }
}

/*******************************************************************
*
*  function: constraint_free()
*
*  purpose:  free expnode structures and expressions for a constraint
*
*******/

void constraint_free(con)
struct constraint *con;
{ int j,m;

  if ( con->formula )
    {  for ( j = 0 ; j < MAXCONCOMP ; j++ )
         { free_expr(con->envect[j]);
           free_expr(con->convect[j]);
         }
       free((char *)con->formula);
     }
  for ( m = 0 ; m < web.quantity_count ; m++ )
    if ( con->quanvect[m][0] )
      { for ( j = 0 ; j < MAXCONCOMP ; j++ )
          free_expr(con->quanvect[m][j]);
        free((char *)con->quanvect[m]);
      }
}

/******************************************************************
*
*  Function: constr_proj()
*
*  Purpose:  projection on one or two constraints.
*
*  Explanation:  Projecting a point or vector onto constraints
*     requires finding an appropriate linear combination of
*     constraint gradients c[i]*grad(f[i]) where c[i] = (inv A)*B,
*     where B is an appropriate vector (input) and 
*            A[i][j] = <grad(f[i]),grad(f[j])>.
*     The linear combination is returned.
*     Function return value is number of hit constraints. 
*/

static REAL aaa[MAXCONSTR][MAXCONSTR];
static REAL *a[MAXCONSTR] = { aaa[0],aaa[1],aaa[2],aaa[3] };

int constr_proj(mode,concount,constr,coord,b,combo,conlist,detect_flag)
int mode;  /* TANGPROJ for vector projection of b to tangent plane,
                  (note return is normal component of b!)
              PLAINPROJ for using input b as right side 
                  (note return is offset from constraints!)*/
int concount;  /* how many constraints */
struct constraint **constr;  /* array of pointers to constraints */
REAL *coord;  /* coordinates of point for evaluating functions */
REAL *b;      /* vector to project */
REAL *combo;  /* desired linear combination */
int *conlist;  /* list of constraint numbers; for returning hits */
int detect_flag; /* set if want to detect one-sided constraints */
{
  int i,j;
  REAL grad[MAXCONSTR][MAXCOORD];    /* gradients of constraints */
  REAL c[MAXCONSTR];                 /* combination coefficients */
  REAL r[MAXCONSTR];                 /* constructed right side B */
  REAL fval;                         /* value of constraint */
  REAL *bb;                          /* pointer to right side */

  if ( concount > MAXCONSTR )
    error("Trying to project on too many constraints.\n",RECOVERABLE);

do_over: 
  if ( concount <= 0 ) 
   {
     for ( j = 0 ; j < web.sdim ; j++ )
       combo[j] = 0.0;    
     return concount;
   }

  /* calculate gradients */
  for ( i = 0 ; i < concount ; i++ )    
     eval_all(constr[i]->formula,coord,web.sdim,&fval,grad[i]);

  /* maybe construct right side for vector projection */
  if ( mode == TANGPROJ )
    {
      for ( i = 0 ; i < concount ; i++ )
        r[i] = dot(b,grad[i],web.sdim);
      bb = r;
    }
  else bb = b;

  /* construct matrix A */  
  for ( i = 0 ; i < concount ; i++ )    
    for ( j = 0 ; j < concount ; j++ )
      a[i][j] = dot(grad[i],grad[j],web.sdim);

  /* invert */
  mat_inv(a,concount);

  /* combination coefficients */
  matvec_mul(a,bb,c,concount,concount);

   /* check one-sided constraints to see if projection necessary */
  if ( detect_flag == DETECT )
   for ( j = 0 ; j < concount ; j++ )
    if ( ((constr[j]->attr&NONNEGATIVE)&& (c[j] > 0.0 )) ||
     ((constr[j]->attr&NONPOSITIVE)&& (c[j] < 0.0 )) )
      { /* don't need to project on this one */
        constr[j] = constr[--concount];
        b[j] = b[concount];
        if ( conlist ) conlist[j] = conlist[concount];
        goto do_over;
      }
 
         
  /* form combination */
  for ( i = 0 ; i < web.sdim ; i++ )
    { combo[i] = 0.0;
      for ( j = 0 ; j < concount ; j++ )
        combo[i] += c[j]*grad[j][i];
    }
  
  return concount;
}

/******************************************************************
*
*  Function:  project_v_constr()
*
*  Purpose:  project a vertex onto its constraints.
*/

void project_v_constr(v_id)
vertex_id v_id;
{
  REAL *x;
  MAP conmap;
  struct constraint *con[CONSTRMAX],*thiscon;
  int oncount = 0;
  REAL f[CONSTRMAX],delta[MAXCOORD];
  int j,itercount = 0;
  REAL diff,totaldiff;
  int walls = 0;  /* total number of constraints vertex is on */

  x = get_coord(v_id);
/*  clear_v_constraint_status(v_id);  */
  do
    {
      walls = 0;
      totaldiff = 0.0;
      conmap = get_v_constraint_map(v_id);
      for ( j = 0, oncount = 0 ; j < web.concount ; j++,conmap >>= 1 )
        {
          if ( !(conmap & 1) ) continue; /* not subject to this constraint */ 
          thiscon = get_constraint(j);
          diff = eval(thiscon->formula,x);
          if ( ((thiscon->attr & NONNEGATIVE) && ( diff > 0.0 ))
            || ((thiscon->attr & NONPOSITIVE) && ( diff < 0.0 )) )
            {
              if ( get_v_constraint_status(v_id,j) != ON_CONSTRAINT )
                continue;
            }
          f[oncount] = -diff;
          con[oncount++] = thiscon;
          totaldiff += fabs(diff);
          set_v_constraint_status(v_id,j);
          walls++;
        }
      if ( totaldiff < web.tolerance ) break;

      constr_proj(PLAINPROJ,oncount,con,x,f,delta,NULL,NO_DETECT);
      for ( j = 0 ; j < web.sdim ; j++ )
        x[j] += delta[j];
      itercount++;
    }
  while ( itercount < MAXCONITER );

  if ( itercount >= MAXCONITER )
  { sprintf(msg,
  "Vertex %d doesn't converge to constraint after %d iterations. Diff = %15.12f\n",
       ordinal(v_id)+1,MAXCONITER,diff);
    error(msg,WARNING);
  }

  if ( walls ) set_attr(v_id,HIT_WALL);
  else unset_attr(v_id,HIT_WALL);
}

/*****************************************************************
*
*  Function: calc_constr_force_v()
*
*  Purpose: calculate force on vertex due to constraint energy. (string model)
*/

void calc_constr_force_v(v_id)
vertex_id v_id;
{
  REAL *f,*coord;
  struct constraint *constr;
  int i,j;
  MAP conmap;
  int sign;
  REAL fval,deriv[MAXCOORD];

  int_val = get_original(v_id);  /* for eval  of file parameters */
  f = get_force(v_id);
  coord = get_coord(v_id);
  if ( get_vattr(v_id) & NEGBOUNDARY ) sign = -1;
  else sign = 1;
  conmap = get_v_constraint_map(v_id);
  for ( j = 0 ; j < web.concount ; j++,conmap >>= 1 )
    {
      if ( !(conmap & 1) ) continue;
      constr = get_constraint(j);
      if ( !(constr->attr & BDRY_ENERGY) || (constr->compcount != 1) )
        continue;
      eval_all(constr->envect[0],coord,web.sdim,&fval,deriv);     
      for ( i = 0 ; i < web.sdim ; i++ )
         f[i] += -sign*deriv[i];
               /* force by constraint */
    }

  return;
}

/*****************************************************************
*
*  Function: calc_constr_force_e()
*
*  Purpose: calculate force on endpoints of edge due to constraint energy.
*/

void calc_constr_force_e(e_id)
edge_id e_id;
{
  REAL *tcoord,*hcoord;
  REAL *tforce,*hforce;
  struct constraint *constr;
  int i,k,m;
  REAL side[MAXCOORD];
  REAL green[MAXCOORD];
  REAL green_deriv[MAXCOORD][MAXCOORD];
  int j,conmap,sign;
  REAL midpt[MAXCOORD];
  REAL grad;

  int_val = get_original(e_id);  /* for eval  of file parameters */
  if ( web.modeltype == QUADRATIC )
    { constr_edge_force_q(e_id);
      return;
    }
  conmap = get_e_constraint_map(e_id);
  if ( get_eattr(e_id) & NEGBOUNDARY ) sign = -1;
  else sign = 1;
  if ( inverted(e_id) ) sign = -sign;

  tcoord = get_coord(get_edge_tailv(e_id));
  hcoord = get_coord(get_edge_headv(e_id));
  for ( j = 0 ; j < web.sdim ; j++ )
    { 
      side[j] = hcoord[j] - tcoord[j];
    }

  tforce = get_force(get_edge_tailv(e_id));
  hforce = get_force(get_edge_headv(e_id));
  for ( j = 0 ; j < web.concount ; j++,conmap >>= 1 )
   {
    if ( !(conmap & 1) ) continue;
    constr = get_constraint(j);
    if ( !(constr->attr & BDRY_ENERGY) 
           || (constr->compcount != web.sdim) ) continue;
    for ( m = 0 ; m < web.gauss1D_order ; m++ )
     {
       for ( i = 0 ; i < web.sdim ; i++ )
         midpt[i] = gauss1Dpt[m]*hcoord[i] + (1 - gauss1Dpt[m])*tcoord[i];
       for ( i = 0 ; i < web.sdim ; i++ )
         eval_all(constr->envect[i],midpt,web.sdim,&green[i],green_deriv[i]);
       for ( i = 0 ; i < web.sdim ; i++ )
         { for ( grad = 0.0, k = 0 ; k < web.sdim ; k++ )
             grad += side[k]*green_deriv[k][i];
           tforce[i] -= sign*gauss1Dwt[m]*((1-gauss1Dpt[m])*grad - green[i]);
           hforce[i] -= sign*gauss1Dwt[m]*(gauss1Dpt[m]*grad + green[i]);
         }
      }
   }


  return;

}

/*****************************************************************
*
*  Function: calc_constr_energy_v()
*
*  Purpose: calculate constraint energy due to vertex. (string model)
*/

void calc_constr_energy_v(v_id)
vertex_id v_id;
{
  REAL e;
  int j;
  MAP conmap = get_v_constraint_map(v_id);
  struct constraint *constr;

  int_val = get_original(v_id);  /* for eval  of file parameters */
  for ( j = 0 ;  j < web.concount ; j++, conmap >>= 1 )
    { if ( !(conmap & 1) ) continue;
      constr = get_constraint(j);
      if ( !(constr->attr & BDRY_ENERGY) || (constr->compcount != 1) ) continue;
      e = eval(constr->envect[0],get_coord(v_id));  
      if ( get_vattr(v_id) & NEGBOUNDARY )
        web.total_energy -= e;
      else
        web.total_energy += e;
    }

}

/*****************************************************************
*
*  Function: calc_constr_energy_e()
*
*  Purpose: calculate energy due to edge on constraint.
*           Also contributions to quantities from constraints.
*/

void calc_constr_energy_e(e_id)
edge_id e_id;
{
  REAL *tcoord,*hcoord;
  struct constraint *constr;
  int i,j,k,m;
  REAL energy = 0.0;
  REAL side[MAXCOORD];
  REAL green[MAXCOORD];
  MAP quantmap,conmap;
  int sign;
  REAL midpt[MAXCOORD];

  int_val = get_original(e_id);  /* for eval  of file parameters */
  if ( web.modeltype == QUADRATIC )
    { constr_edge_energy_q(e_id);
      return;
    }
  conmap = get_e_constraint_map(e_id);
  if ( get_eattr(e_id) & NEGBOUNDARY ) sign = -1;
  else sign = 1;
  if ( inverted(e_id) ) sign = -sign;

  tcoord = get_coord(get_edge_tailv(e_id));
  hcoord = get_coord(get_edge_headv(e_id));
  for ( j = 0 ; j < web.sdim ; j++ )
    { 
      side[j] = hcoord[j] - tcoord[j];
    }

  for ( j = 0 ; j < web.concount ; j++,conmap >>= 1 )
   {
    if ( !(conmap & 1) ) continue;
    constr = get_constraint(j);
    quantmap = constr->quantity_map;
    if ( !(constr->attr & BDRY_ENERGY) && !quantmap ) continue;
    if ( constr->compcount != web.sdim ) continue;
    for ( k = 0 ; k < web.gauss1D_order ; k++ )
     {
       for ( i = 0 ; i < web.sdim ; i++ )
         midpt[i] = gauss1Dpt[k]*hcoord[i] + (1 - gauss1Dpt[k])*tcoord[i];
       if ( constr->attr & BDRY_ENERGY )
        {
          for ( i = 0 ; i < web.sdim ; i++ )
            green[i] = eval(constr->envect[i],midpt);  /* constraint energy */
          energy += sign*gauss1Dwt[k]*dot(side,green,web.sdim);
        }
       quantmap = constr->quantity_map; /* have to refresh quantmap */
       for ( m = 0 ; quantmap != 0 ; m++, quantmap >>= 1 )
        { struct quantity * quan;
          if ( !(quantmap & 1) ) continue;
          quan = get_quant(m);
          for ( i = 0 ; i < web.sdim ; i++ )
            green[i] = eval(constr->quanvect[m][i],midpt); 
          quan->value += sign*gauss1Dwt[k]*dot(side,green,web.sdim);
        }
     }

   }

  web.total_energy += energy;

}

/*****************************************************************
*
*  Function: calc_constr_content_v()
*
*  Purpose: calculate interior content due to vertex. (string model)
*/

void calc_constr_content_v(v_id)
vertex_id v_id;
{
  REAL e=0.0;
  int j;
  MAP conmap;
  struct constraint *constr;
  body_id b_id;
  facetedge_id fe_id;
  facet_id f_id;

  fe_id = get_vertex_fe(v_id);
  if ( !valid_id(fe_id) ) return;  /* not on a cell */

  int_val = get_original(v_id);  /* for eval  of file parameters */
  conmap = get_v_constraint_map(v_id);
  for ( j = 0 ; j < web.concount ; j++,conmap>>=1 )
    {
      if ( !(conmap & 1) ) continue;
      constr = get_constraint(j);
      if ( !(constr->attr & BDRY_CONTENT) ) continue;
      if ( constr->compcount != 1 ) continue;
      if ( get_vattr(v_id) & NEGBOUNDARY )
        e += -eval(constr->convect[0],get_coord(v_id));  
      else
        e += eval(constr->convect[0],get_coord(v_id));  
    }

  /* cell on plus side of edge */
  f_id = get_fe_facet(fe_id);
  if ( valid_id(f_id) ) 
    {
      b_id = get_facet_body(f_id);
      if ( valid_id(b_id) )
        set_body_volume(b_id,get_body_volume(b_id)+e);
    }

  /* cell on other side of edge */
  f_id = get_fe_facet(inverse_id(fe_id));
  if ( valid_id(f_id) ) 
    {
      b_id = get_facet_body(f_id);
      if ( valid_id(b_id) )
        set_body_volume(b_id,get_body_volume(b_id)-e);
    }
}

/*****************************************************************
*
*  Function: calc_constr_content_e()
*
*  Purpose: calculate interior content due to edge. (film model)
*/

void calc_constr_content_e(e_id)
edge_id e_id;
{
  REAL *tcoord,*hcoord;
  struct constraint *constr;
  int i,k;
  REAL content = 0.0;
  REAL midpt[MAXCOORD];
  REAL side[MAXCOORD];
  REAL green[MAXCOORD];
  body_id b_id;
  facetedge_id fe_id;
  facet_id f_id;
  int j;
  MAP conmap;
  int sign;

  if ( web.modeltype == QUADRATIC )
    { constr_edge_content_q(e_id);
      return;
    }

  int_val = get_original(e_id);  /* for eval  of file parameters */
  conmap = get_e_constraint_map(e_id);
  if ( get_eattr(e_id) & NEGBOUNDARY ) sign = -1;
  else sign = 1;
  if ( inverted(e_id) ) sign = -sign;

  tcoord = get_coord(get_edge_tailv(e_id));
  hcoord = get_coord(get_edge_headv(e_id));
  for ( j = 0 ; j < web.sdim ; j++ )
    { 
      side[j] = hcoord[j] - tcoord[j];
    }

  for ( j = 0 ; j < web.concount ; j++,conmap>>=1 )
    {
       if ( !(conmap & 1) ) continue;
       constr = get_constraint(j);
       if ( !(constr->attr & BDRY_CONTENT) ) continue;
       if ( constr->compcount != web.sdim ) continue;
       for ( k = 0 ; k < web.gauss1D_order ; k++ )
         {
           for ( i = 0 ; i < web.sdim ; i++ )
             midpt[i] = gauss1Dpt[k]*hcoord[i] + (1 - gauss1Dpt[k])*tcoord[i];
           for ( i = 0 ; i < web.sdim ; i++ )
              green[i] = eval(constr->convect[i],midpt);
           content += sign*gauss1Dwt[k]*dot(side,green,web.sdim);
         }
    }

  fe_id = get_edge_fe(e_id);

  /* cell on plus side of edge */
  f_id = get_fe_facet(fe_id);
  if ( valid_id(f_id) ) 
    {
      b_id = get_facet_body(f_id);
      if ( valid_id(b_id) )
        set_body_volume(b_id,get_body_volume(b_id)+content);
    }

  /* cell on other side of edge */
  f_id = get_fe_facet(inverse_id(fe_id));
  if ( valid_id(f_id) ) 
    {
      b_id = get_facet_body(f_id);
      if ( valid_id(b_id) )
        set_body_volume(b_id,get_body_volume(b_id)-content);
    }

  return;


}

/****************************************************************
*
*  Function: constr_springs()
*
*  Purpose:  Since only vertices are actually confined to constraints,
*            edges and faces supposedly on constraints can pull
*            away from convex constraints, and in fact do, since
*            a long edge short-cuts the constraints.  To prevent
*            this and encourage equal-length constraint edges, an
*            energy penalty is inflicted for an edge angling away
*            from its constraint.  
*            The energy penalty is 2/3 of the area of the right
*            triangle whose base is half the side and whose hypoteneuse
*            lies on the constraint tangent.  This is done for both
*            ends of the side.
*/

void  constr_springs(e_id)
edge_id e_id;
{
  REAL s[MAXCOORD],q[MAXCOORD],*f;
  REAL ss,qq; /* square lengths */
  REAL norm;
  struct constraint *constr[CONSTRMAX];
  int concount;
  MAP conmap;
  vertex_id head,tail;
  int i,j; 

  if ( get_eattr(e_id) & FIXED ) return;
  if ( !(get_eattr(e_id) & CONSTRAINT) ) return;

  tail = get_edge_tailv(e_id);
  head = get_edge_headv(e_id);

  /* find which constraints have CONVEX attribute */
  conmap = get_e_constraint_map(e_id);
  conmap &= get_v_constraint_state(tail);
  conmap &= get_v_constraint_state(head);
  for ( i = 0, j = 0 ; j < web.concount ; j++,conmap>>=1 )
    { 
      if ( !(conmap & 1) ) continue;
      constr[i] = get_constraint(j);
      if ( constr[i]->attr & B_CONVEX ) i++;   /* keep this one */
    }
  if ( i == 0 ) return;
  concount = i;  

  /* now the calculation */
  get_edge_side(e_id,s);
  ss = dot(s,s,web.sdim);

  /*tail*/
  constr_proj(TANGPROJ,concount,constr,get_coord(tail),s,q,NULL,NO_DETECT);
  for ( i = 0 ; i < web.sdim ; i++ )
    q[i] = s[i] - q[i];   /* get tangent side */
  qq = dot(q,q,web.sdim);
  norm = (1 + (ss-qq)/3/qq)*sqrt(fabs(ss-qq)/qq)/2; /* fabs due to machine inaccuracy */
  f = get_force(tail);
  for ( i = 0 ; i < web.sdim ; i++ )
    f[i] += web.spring_constant*q[i]*norm;


  /* head */
  constr_proj(TANGPROJ,concount,constr,get_coord(head),s,q,NULL,NO_DETECT);
  for ( i = 0 ; i < web.sdim ; i++ )
    q[i] = s[i] - q[i];   /* get tangent side */
  qq = dot(q,q,web.sdim);
  norm = (1 + (ss-qq)/3/qq)*sqrt(fabs(ss-qq)/qq)/2;
  f = get_force(head);
  for ( i = 0 ; i < web.sdim ; i++ )
    f[i] -= web.spring_constant*q[i]*norm;


}

/*****************************************************************
*
*  Function: constr_spring_energy()
*
*  Purpose:  Calculate energy of kludge constraint force.
*            Constant factor included to make it best approx
*            of true area.
*/

void constr_spring_energy(e_id)
edge_id e_id;
{
  REAL sprenergy;
  REAL s[MAXCOORD],q[MAXCOORD];
  REAL ss,qq;
  struct constraint *constr[CONSTRMAX];
  int concount;
  MAP conmap;
  vertex_id tail,head;
  int i,j; 

  if ( get_eattr(e_id) & FIXED ) return;
  if ( !(get_eattr(e_id) & CONSTRAINT) ) return;

  tail = get_edge_tailv(e_id);
  head = get_edge_headv(e_id);

  /* find which constraints have CONVEX attribute */
  conmap = get_e_constraint_map(e_id);
  conmap &= get_v_constraint_state(tail);  /* make sure both ends */ 
  conmap &= get_v_constraint_state(head);  /* on constraint */
  for ( i = 0, j = 0 ; j < web.concount ; j++,conmap>>=1 )
    { 
      if ( !(conmap & 1) ) continue;
      constr[i] = get_constraint(j);
      if ( constr[i]->attr & B_CONVEX ) i++;   /* keep this one */
    }
  if ( i == 0 ) return;
  concount = i;  

  /* now the calculation */
  get_edge_side(e_id,s);
  ss = dot(s,s,web.sdim);

  constr_proj(TANGPROJ,concount,constr,get_coord(tail),s,q,NULL,NO_DETECT);
  qq = dot(q,q,web.sdim);
  sprenergy = web.spring_constant*ss*sqrt(qq/(ss - qq))/12;

  constr_proj(TANGPROJ,concount,constr,get_coord(head),s,q,NULL,NO_DETECT);
  qq = dot(q,q,web.sdim);
  sprenergy += web.spring_constant*ss*sqrt(qq/(ss - qq))/12;

  web.total_energy += sprenergy;
  web.spring_energy += sprenergy;
}


/*************************************************************************
 
  Following are quadratic model version of constraint integral routines.
     
*************************************************************************/



/************************************************************************
*
*  Function: constr_edge_energy_q()
*
*  Purpose:  Returns energy due to one edge on constraint.
*
*  Quadratic version.
*/

void constr_edge_energy_q(e_id)
edge_id e_id;
{
  REAL x[EDGE_CTRL][MAXCOORD];
  REAL *pt[EDGE_CTRL];
  REAL tang[MAXCOORD];
  vertex_id v[EDGE_CTRL];
  int i,j,k;
  struct constraint *constr;
  REAL energy = 0.0;
  REAL side[MAXCOORD];
  REAL green[MAXCOORD];
  int conmap,sign;
  REAL gpt[MAXCOORD];
        
  int_val = get_original(e_id);  /* for eval  of file parameters */
  conmap = get_e_constraint_map(e_id);
  get_edge_side(e_id,side);
  if ( get_eattr(e_id) & NEGBOUNDARY ) sign = -1;
  else sign = 1;
              
  v[0] = get_edge_tailv(e_id);
  v[1] = get_edge_midv(e_id);
  v[2] = get_edge_headv(e_id);
  for ( i = 0 ; i < EDGE_CTRL ; i++ )
    {
      pt[i] = get_coord(v[i]);
    }
  for ( i = 0 ; i < EDGE_CTRL ; i++ )
    for ( j = 0 ; j < web.sdim ; j++ )
      x[i][j] = pt[i][j];
          
    
  /* calculate tangents at integration points and accumulate */
  for ( i = 0 ; i < EDGE_INTERP ; i++ )
    {
      for ( j = 0 ; j < web.sdim ; j ++ )
        {
          tang[j] = 0.0;
          gpt[j] = 0.0;
          for ( k = 0 ; k < EDGE_CTRL ; k++ )
            { tang[j] += sdip[k][i]*x[k][j];
              gpt[j] += gcombo[k][i]*x[k][j];
            }
        }
      for ( j = 0 ; j < web.concount ; j++ )
        {
          if ( !((conmap>>j) & 1) ) continue;
          constr = get_constraint(j);
          if ( !(constr->attr & BDRY_ENERGY) ) continue;
          if ( constr->compcount != web.sdim ) continue;
          for ( k = 0 ; k < web.sdim ; k++ )
            green[k] = eval(constr->envect[k],gpt);  /* constraint energy */
          energy += sign*gauss2wt[i]*dot(tang,green,web.sdim);
        }
    }
    
  web.total_energy += energy;

}
 


/************************************************************************
*
*  Function: constr_edge_force_q()
*
*  Purpose:  Calculates force due to one edge on constraint.
*
*  Quadratic version.
*/

void constr_edge_force_q(e_id)
edge_id e_id;
{
  REAL x[EDGE_CTRL][MAXCOORD];
  REAL *pt[EDGE_CTRL];
  REAL tang[MAXCOORD];
  vertex_id v[EDGE_CTRL];
  int i,j,k,m,n;
  struct constraint *constr;
  REAL side[MAXCOORD];
  REAL green[MAXCOORD];
  REAL green_deriv[MAXCOORD][MAXCOORD];
  int conmap,sign;
  REAL gpt[MAXCOORD];
  REAL *force[EDGE_CTRL];
        
  int_val = get_original(e_id);  /* for eval  of file parameters */
  conmap = get_e_constraint_map(e_id);
  get_edge_side(e_id,side);
  if ( get_eattr(e_id) & NEGBOUNDARY ) sign = -1;
  else sign = 1;
              
  v[0] = get_edge_tailv(e_id);
  v[1] = get_edge_midv(e_id);
  v[2] = get_edge_headv(e_id);
  for ( i = 0 ; i < EDGE_CTRL ; i++ )
    {
      pt[i] = get_coord(v[i]);
      force[i] = get_force(v[i]);
    }
  for ( i = 0 ; i < EDGE_CTRL ; i++ )
    for ( j = 0 ; j < web.sdim ; j++ )
      x[i][j] = pt[i][j];
          
  /* calculate tangents at integration points and accumulate */
  for ( i = 0 ; i < EDGE_INTERP ; i++ )
    {
      for ( j = 0 ; j < web.sdim ; j ++ )
        {
          tang[j] = 0.0;
          gpt[j] = 0.0;
          for ( k = 0 ; k < EDGE_CTRL ; k++ )
            { tang[j] += sdip[k][i]*x[k][j];
              gpt[j] += gcombo[k][i]*x[k][j];
            }
        }
      for ( j = 0 ; j < web.concount ; j++ )
        { REAL tangdot[MAXCOORD];

          if ( !((conmap>>j) & 1) ) continue;
          constr = get_constraint(j);
          if ( !(constr->attr & BDRY_ENERGY) ) continue;
          if ( constr->compcount != web.sdim ) continue;
          for ( m = 0 ; m < web.sdim ; m++ )
           eval_all(constr->envect[m],gpt,web.sdim,&green[m],
                                                            green_deriv[m]);
          for ( n = 0 ; n < web.sdim ; n++ )
            for ( m = 0, tangdot[n] = 0.0 ; m < web.sdim ; m++ )
              tangdot[n] += tang[m]*green_deriv[m][n];
          for ( k = 0 ; k < EDGE_CTRL ; k++ )
            { for ( m = 0 ; m < web.sdim ; m++ )
                force[k][m] -= sign*gauss2wt[i]*sdip[k][i]*green[m];
              for ( n = 0 ; n < web.sdim ; n++ )
                force[k][n] -= sign*gauss2wt[i]*gcombo[k][i]*tangdot[n];
            }           
        }
    }
}
 


/************************************************************************
*
*  Function: constr_edge_content_q()
*
   Purpose:  Finds volume due to one edge on a constraint.
*
*  Quadratic version.
*/

void constr_edge_content_q(e_id)
edge_id e_id;
{
  REAL x[EDGE_CTRL][MAXCOORD];
  REAL *pt[EDGE_CTRL];
  REAL tang[MAXCOORD];
  vertex_id v[EDGE_CTRL];
  int i,j,k;
  struct constraint *constr;
  REAL content = 0.0;
  REAL side[MAXCOORD];
  REAL green[MAXCOORD];
  int conmap,sign;
  REAL gpt[MAXCOORD];
  body_id b_id;
  facet_id f_id;
  facetedge_id fe_id = get_edge_fe(e_id);
        
  int_val = get_original(e_id);  /* for eval  of file parameters */
  conmap = get_e_constraint_map(e_id);
  get_edge_side(e_id,side);
  if ( get_eattr(e_id) & NEGBOUNDARY ) sign = -1;
  else sign = 1;
              
  v[0] = get_edge_tailv(e_id);
  v[1] = get_edge_midv(e_id);
  v[2] = get_edge_headv(e_id);
  for ( i = 0 ; i < EDGE_CTRL ; i++ )
    {
      pt[i] = get_coord(v[i]);
    }
  for ( i = 0 ; i < EDGE_CTRL ; i++ )
    for ( j = 0 ; j < web.sdim ; j++ )
      x[i][j] = pt[i][j];
          
    
  /* calculate tangents at integration points and accumulate */
  for ( i = 0 ; i < EDGE_INTERP ; i++ )
    {
      for ( j = 0 ; j < web.sdim ; j ++ )
        {
          tang[j] = 0.0;
          gpt[j] = 0.0;
          for ( k = 0 ; k < EDGE_CTRL ; k++ )
            { tang[j] += sdip[k][i]*x[k][j];
              gpt[j] += gcombo[k][i]*x[k][j];
            }
        }
      for ( j = 0 ; j < web.concount ; j++ )
        {
          if ( !((conmap>>j) & 1) ) continue;
          constr = get_constraint(j);
          if ( !(constr->attr & BDRY_CONTENT) ) continue;
          if ( constr->compcount != web.sdim ) continue;
          for ( k = 0 ; k < web.sdim ; k++ )
            green[k] = eval(constr->convect[k],gpt); /* constraint content */
          content += sign*gauss2wt[i]*dot(tang,green,web.sdim);
        }
    }
    

  /* cell on plus side of edge */
  f_id = get_fe_facet(fe_id);
  if ( valid_id(f_id) ) 
    {
      b_id = get_facet_body(f_id);
      if ( valid_id(b_id) )
        set_body_volume(b_id,get_body_volume(b_id)+content);
    }

  /* cell on other side of edge */
  f_id = get_fe_facet(inverse_id(fe_id));
  if ( valid_id(f_id) ) 
    {
      b_id = get_facet_body(f_id);
      if ( valid_id(b_id) )
        set_body_volume(b_id,get_body_volume(b_id)-content);
    }

  return;
}


/************************************************************************
*
*  Function: constr_vol_grad_q()
*
*  Purpose:  Calculates volume gradients due to one edge on constraint.
*
*  Quadratic version.
*/

void constr_vol_grad_q(e_id)
edge_id e_id;
{
  REAL x[EDGE_CTRL][MAXCOORD];
  REAL *pt[EDGE_CTRL];
  REAL tang[MAXCOORD];
  vertex_id v[EDGE_CTRL];
  int i,j,k;
  REAL green[MAXCOORD];
  REAL green_deriv[MAXCOORD][MAXCOORD];
  struct constraint *constr;
  int m,n;
  REAL grad[EDGE_CTRL][MAXCOORD];
  int conmap,sign,bodysign=1;
  REAL gpt[MAXCOORD];
  struct volgrad *vgptri;
  facet_id f_id = get_fe_facet(get_edge_fe(e_id));

  int_val = get_original(e_id);  /* for eval  of file parameters */
  /* get basic edge data */        
  conmap = get_e_constraint_map(e_id);
  if ( get_eattr(e_id) & NEGBOUNDARY ) sign = -1;
  else sign = 1;
              
  v[0] = get_edge_tailv(e_id);
  v[1] = get_edge_midv(e_id);
  v[2] = get_edge_headv(e_id);
  for ( i = 0 ; i < EDGE_CTRL ; i++ )
    {
      pt[i] = get_coord(v[i]);
    }
  for ( i = 0 ; i < EDGE_CTRL ; i++ )
    for ( j = 0 ; j < web.sdim ; j++ )
      x[i][j] = pt[i][j];
          
  memset((char *)grad,0,sizeof(grad));

  /* find content integral gradients */    
  for ( i = 0 ; i < EDGE_INTERP ; i++ )
    {
      /* calculate coords and tangents at integration point */
      for ( j = 0 ; j < web.sdim ; j ++ )
        {
          tang[j] = 0.0;
          gpt[j] = 0.0;
          for ( k = 0 ; k < EDGE_CTRL ; k++ )
            { tang[j] += sdip[k][i]*x[k][j];
              gpt[j] += gcombo[k][i]*x[k][j];
            }
        }
 
      /* accumulate gradients due to this integration point */
      for ( j = 0 ; j < web.concount ; j++ )
        { REAL tangdot[MAXCOORD];
          if ( !((conmap>>j) & 1) ) continue;
          constr = get_constraint(j);
          if ( !(constr->attr & BDRY_CONTENT) ) continue;
          if ( constr->compcount != web.sdim ) continue;
          for ( m = 0 ; m < web.sdim ; m++ )
            eval_all(constr->convect[m],gpt,web.sdim,&green[m],
                                                         green_deriv[m]); 
          for ( n = 0 ; n < web.sdim ; n++ )
            for ( m = 0, tangdot[n] = 0.0 ; m < web.sdim ; m++ )
              tangdot[n] += tang[m]*green_deriv[m][n];
          for ( k = 0 ; k < EDGE_CTRL ; k++ )
            { for ( m = 0 ; m < web.sdim ; m++ )
                grad[k][m] += sign*gauss2wt[i]*sdip[k][i]*green[m];
              for ( n = 0 ; n < web.sdim ; n++ )
                grad[k][n] += sign*gauss2wt[i]*gcombo[k][i]*tangdot[n];
            }           
        }
    }

  /* now add gradients to proper bodies */
  for ( m = 0 ; m < EDGE_CTRL ; m++ )
    { 
      vgptri = get_vertex_vgrad(v[m]);
      for  ( ; vgptri ; vgptri = vgptri->chain )
       {
         if ( !valid_id(vgptri->b_id) ) continue; /* skip quantities */
         if ( !equal_id(get_facet_body(f_id),vgptri->b_id) ) 
          if ( !equal_id(get_facet_body(inverse_id(f_id)),vgptri->b_id) )
           { outstring("Can't find vertex body.\n");
           }
          else  bodysign = -sign;
         else bodysign = sign;
         for ( k = 0 ; k < web.sdim ; k++ )
           vgptri->grad[k] += bodysign*grad[m][k];
       }
    }
    
   
}


/*****************************************************************
*
*  Function: constr_basis()
*
*  Purpose: calculate basis of constraint tangent.
*/

 
int constr_basis(v_id,basis)
vertex_id v_id;
REAL **basis;  /* for return */
{
  MAP conmap;
  struct constraint *con[CONSTRMAX];
  int oncount = 0;
  REAL ggrad[MAXCOORD][MAXCOORD];
  REAL *grad[MAXCOORD]; /* for proper matrix form */
  REAL fval;
  int i,j;

  conmap = get_v_constraint_map(v_id);
  for ( j = 0, oncount = 0 ; j < web.concount ; j++,conmap >>= 1 )
    {
      if ( !(conmap & 1) ) continue; /* not subject to this constraint */ 
      con[oncount++] = get_constraint(j);
     }

  /* first calc constraint gradients */
  for ( i = 0 ; i < oncount ; i++ )    
    { grad[i] = ggrad[i];
      eval_all(con[i]->formula,get_coord(v_id),web.sdim,&fval,grad[i]);
    }

  /* now get basis */
  kernel_basis(grad,scratch_mat,oncount,web.sdim);

  /* transpose */
  for ( i = 0 ; i < web.sdim - oncount ; i++ )
   for ( j = 0 ; j < web.sdim ; j++ )
     basis[i][j] = scratch_mat[j][i];

  return web.sdim - oncount;
}

