static char rcsid[] = "$Id: signs.c,v 1.3 1992/01/29 14:33:52 jtsillas Exp $";

/*****************************************************************************
 *
 *  Copyright 1989 The University of Texas at Austin
 *  Copyright 1990 Microelectronics and Computer Technology Corporation
 *  Copyright 1990 Thomson Consumer Electronics, Inc.
 *  Copyright 1991 Bull HN Worldwide Info Systems, Inc.
 *
 *****************************************************************************/

/*  signs.c
 *
 *  This file contains all the routines for the creation and manipulation of 
 *  symbols used in mxgdb.  There are 3 different signs:
 *    arrow  - a solid right arrow to indicate the current execution point.
 *    updown - an outlined right arrow to indicate position in stack trace.
 *    stop   - a stop hand symbol to indicate a breakpoint is set.
 *    bomb  - a bomb symbol to indicate the point of segmentation fault.
 *
 *  To display a sign on a given line in the source window, it is first
 *  created and mapped.  To undisplay it, the sign is unmapped.  It can
 *  be mapped again when the sign is needed.  Note that the sign is never
 *  moved, so that there can be as many signs created (but not mapped) as
 *  the number of lines in the source window.
 *  For arrow and updown, there can be at most one of each mapped at a time.
 *  For stop, there can be more than one mapped at the same time.
 *  
 *  signs_init():    Initialize signs data structs.
 *  DisplayStop():   Given a line number display the sign if viewable.
 *  CreateSign():    Create a symbol, calculate position, display it (static).
 *  UpdateStops():   Update all stops and display only the viewable ones.
 *  RemoveStops():   Remove the stop at a line num.
 *  ClearStops():    Remove all stops.
 *  UpdateArrow():   Unmap current arrow and redisplay.
 *  UpdateUpdown():  Unamap current updown and redisplay.
 *  UpdateBomb():    Unamap current bomb and redisplay.
 */

#include "global.h"
#include "bitmaps.h"
#include <Xm/Xm.h>
#include <Xm/Label.h>

static Widget CreateSign(Widget, Widget, int, Cardinal);

ArrowSign 	arrowsign[MAXSIGNS];
UpdownSign 	updownsign[MAXSIGNS];
static StopSign	stopsign[MAXSIGNS];
BombSign 	bombsign[MAXSIGNS];

Arrow 		arrow;
Updown 		updown;
Stops		stops[MAXSTOPS];	/* array of stops */
Bomb 		bomb;
Cardinal	nstops;			/* number of stops */

/* Initialize data structures */

void signs_init()
{
    int i;

    for (i=0; i<MAXSIGNS; i++) {
	arrowsign[i].w = NULL;
	arrowsign[i].mapped = FALSE;
    }
    for (i=0; i<MAXSIGNS; i++) {
	stopsign[i].w = NULL;
	stopsign[i].mapped = FALSE;
    }
    arrow.i = 0;
    arrow.line = 0;
    strcpy(arrow.file, "");
    updown.i = 0;
    updown.line = 0;
    strcpy(updown.file, "");
    nstops = 0;
    bomb.i = 0;
    bomb.line = 0;
    strcpy(bomb.file, "");
}


/*  Create an arrow symbol, updown symbol or stop symbol:
 *    calculate the position of the symbol based on i, the number of lines
 *    from the top line.
 *    create the pixmap of the symbol
 *    display the symbol as a bitmap in a label widget.
 */
static Widget CreateSign(parent, textW, sign, i)
     Widget	parent;
     Widget     textW;
     int        sign;
     Cardinal 	i;
{
    Arg 	args[MAXARGS];
    Dimension 	source_height, height, width; 
    char	*bits;
    Pixel       fg, bg;
    int 	horizDistance, vertDistance, height_per_line;
    int         screen;

    static Arg sign_args[] = {
      { XmNborderWidth, 0 },
      { XmNlabelType, XmPIXMAP },
      { XmNmappedWhenManaged, False },
      { NULL, 0 },
      { NULL, 0 },
      { NULL, 0 },
      { NULL, 0 },
      { NULL, 0 },
      { NULL, 0 }
    };

    if (displayedFile == NULL) return NULL;

    /* Get height and background pixel values of parent window */
    XtSetArg(args[0], XmNheight, &source_height);
    XtSetArg(args[1], XmNbackground, &bg);
    XtGetValues(textW, args, 2);

    height_per_line = source_height/displayedFile->lines;
    vertDistance = OFFSET + (i * height_per_line); 

    screen = DefaultScreen(display);
    
    switch(sign)
      {
      case _ARROW_SIGN:
	bits = arrow_bits;
	width = arrow_width;
	height = arrow_height;
	horizDistance = 0;
	fg = app_resources.arrow_color;
	break;
      case _UPDOWN_SIGN:
	bits = updown_bits;
	width = updown_width;
	height = updown_height;
	horizDistance = 0;
	fg = app_resources.updown_color;
	break;
      case _STOP_SIGN:
	bits = stop_bits;
	width = stop_width;
	height = stop_height;
	horizDistance = arrow_width;
	fg = app_resources.stop_color;
	break;
      case _BOMB_SIGN:
	bits = bomb_bits;
	width = bomb_width;
	height = bomb_height;
	horizDistance = 0;
	fg = app_resources.bomb_color;
	break;
      default:
	abort();
      }

    XtSetArg(sign_args[3], XmNwidth, (XtArgVal) width);
    XtSetArg(sign_args[4], XmNheight, (XtArgVal) height);
    XtSetArg(sign_args[5], XmNlabelPixmap, XCreatePixmapFromBitmapData (
        display, DefaultRootWindow(display), bits, width, height,
        fg, bg, DefaultDepth(display, screen)));
    XtSetArg(sign_args[6], XmNx, (XtArgVal) horizDistance+5);
    XtSetArg(sign_args[7], XmNy, (XtArgVal) vertDistance+3);

    return XtCreateManagedWidget("sourceSign", xmLabelWidgetClass, 
				 parent, sign_args, 8);
}

/*
 *  Given a line number, displays a stop sign if that line is viewable.
 *  If the stop widget for that line does not exist, create one and map it.
 *  If the stop widget exists but not mapped, map it.
 */
void DisplayStop(file, line)
FileRec *file;
int	line;
{
  Cardinal i;
  Arg args[MAXARGS];
  
  if (line >= file->topline && line <= file->bottomline) 
    {
      i = line - file->topline;
      if (!stopsign[i].w)	/* widget does not exist */ 
	{
	  stopsign[i].w = CreateSign(sourceBBW, sourceWindow, 
				     _STOP_SIGN, i);
	  XtMapWidget(stopsign[i].w);
	  stopsign[i].mapped = 1;
	}
      else if (!stopsign[i].mapped)  /* widget not mapped */
	{
	  XtMapWidget(stopsign[i].w);
	  stopsign[i].mapped = 1;
	}
    }
}

/*
 *  Unmap all stop signs and then display only those stops that are viewable.
 */
void UpdateStops(file)
     FileRec *file;
{
  int i;
  int line; 
  
  if (!file) return;
  
  for (i=0; i<file->lines; i++)
    if (stopsign[i].w && stopsign[i].mapped) 
      {
	XtUnmapWidget(stopsign[i].w);
	stopsign[i].mapped = 0;
      }
  
  for (i=1; i<=nstops; i++)
    if (stops[i].file && !strcmp(stops[i].file, file->pathname) &&
	(line=stops[i].line) && line >= file->topline &&
	line <= file->bottomline) 
      DisplayStop(file, line);
}

/*
 * Given a line number, unmap the stop sign associated with that line.
 */
void RemoveStop(line)
int line;
{
    Cardinal i;
    char command[30];

    if (displayedFile && line >= displayedFile->topline && 
			 line <= displayedFile->bottomline) {
	i = line - displayedFile->topline;
	if (stopsign[i].w && stopsign[i].mapped) {
	    XtUnmapWidget(stopsign[i].w);
	    stopsign[i].mapped = 0;

	    sprintf(command, "clear %d\n", line);
	    send_command(command);
	    Echo = False;
	    Parse = False;
	    Prompt = False;
	    while(!Prompt) read_from_gdb();
	    
	}
    }
}

void ClearStops()
{
    int i;

    for (i=1; i<=nstops; i++) {
	stops[i].file = NULL;
	stops[i].line = 0;
    }
}

/*  Unmap the current arrow sign.
 *  Display a new arrow sign if it is viewable.
 */
void UpdateArrow(file)
FileRec *file;
{
    Cardinal i;
    int	     line;

    if (file == NULL) return;
    i = arrow.i;
    if (i>=0 && i<file->lines)
	if (arrowsign[i].w && arrowsign[i].mapped) {
	    XtUnmapWidget(arrowsign[i].w);
	    arrowsign[i].mapped = 0;
	}
    line = arrow.line;
    if (arrow.file && !strcmp(arrow.file, file->pathname) &&
    	line >= file->topline && line <= file->bottomline) {
        i = line - file->topline;
	arrow.i = i;
	if (arrowsign[i].w == NULL) {
	    arrowsign[i].w = CreateSign(sourceBBW, sourceWindow, 
					_ARROW_SIGN, i);
	    XtMapWidget(arrowsign[i].w);
	    arrowsign[i].mapped = TRUE;
	}
	else if (!arrowsign[i].mapped) {
	    XtMapWidget(arrowsign[i].w);
	    arrowsign[i].mapped = TRUE;
	}
    }
}


/*  If the new updown is on the same line as the arrow, remove the updown.
 *  Unmap current updown sign.
 *  Display the updown if it is viewable.
 */
void UpdateUpdown(file)
FileRec *file;
{
    Cardinal i;
    int	     line;

    if (file == NULL) return;
    if (updown.file && !strcmp(updown.file, arrow.file) && 
	!strcmp(updown.func, arrow.func) && (updown.line == arrow.line)) {
	updown.line = 0;
	strcpy(updown.file, "");
    }
    i = updown.i;
    if (i>=0 && i<file->lines)
	if (updownsign[i].w && updownsign[i].mapped) {
	    XtUnmapWidget(updownsign[i].w);
	    updownsign[i].mapped = 0;
	}
    line = updown.line;
    if (updown.file && !strcmp(updown.file, file->pathname) &&
    	line >= file->topline && line <= file->bottomline) {
        i = line - file->topline;
	updown.i = i;
	if (updownsign[i].w == NULL) {
	    updownsign[i].w = CreateSign(sourceBBW, sourceWindow, 
					 _UPDOWN_SIGN, i);
	    XtMapWidget(updownsign[i].w);
	    updownsign[i].mapped = TRUE;
	}
	else if (!updownsign[i].mapped) {
	    XtMapWidget(updownsign[i].w);
	    updownsign[i].mapped = TRUE;
	}
    }
}

/*  Unmap the current bomb sign, if any.
 *  Display a new bomb sign.
 */
void UpdateBomb(file)
FileRec *file;
{
    Cardinal i;
    int	     line;

    if (file == NULL) return;
    i = bomb.i;
    if (i>=0 && i<file->lines)
	if (bombsign[i].w && bombsign[i].mapped) {
	    XtUnmapWidget(bombsign[i].w);
	    bombsign[i].mapped = 0;
	}
    line = bomb.line;
    if (bomb.file && !strcmp(bomb.file, file->pathname) &&
    	line >= file->topline && line <= file->bottomline) {
        i = line - file->topline;
	bomb.i = i;
	if (bombsign[i].w == NULL) {
	    bombsign[i].w = CreateSign(sourceBBW, sourceWindow, 
				       _BOMB_SIGN, i);
	    XtMapWidget(bombsign[i].w);
	    bombsign[i].mapped = TRUE;
	}
	else if (!bombsign[i].mapped) {
	    XtMapWidget(bombsign[i].w);
	    bombsign[i].mapped = TRUE;
	}
    }
}
