/*
 * Copyright (c) 2016, Mellanox Technologies, Ltd.  All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <linux/pci.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/module.h>
#include <linux/mlx5/driver.h>

#include "mlx5_core.h"

#define MLX5_EXTRACT_C(source, offset, size)	\
((((unsigned)(source)) >> (offset)) & MLX5_ONES32(size))
#define MLX5_EXTRACT(src, start, len)		\
(((len) == 32) ? (src) : MLX5_EXTRACT_C(src, start, len))
#define MLX5_ONES32(size)			\
((size) ? (0xffffffff >> (32 - (size))) : 0)
#define MLX5_MASK32(offset, size)		\
(MLX5_ONES32(size) << (offset))
#define MLX5_MERGE_C(rsrc1, rsrc2, start, len)  \
((((rsrc2) << (start)) & (MLX5_MASK32((start), (len)))) | \
((rsrc1) & (~MLX5_MASK32((start), (len)))))
#define MLX5_MERGE(rsrc1, rsrc2, start, len)	\
(((len) == 32) ? (rsrc2) : MLX5_MERGE_C(rsrc1, rsrc2, start, len))

#define MLX5_CR_SPACE_DOMAIN 0x2
#define MLX5_ICMD_SPACE_DOMAIN 0x3

#define MLX5_HWID_ADDR 0xf0014
#define MLX5_ADDR_REG 0x58
#define MLX5_DATA_REG 0x5c

#define MLX5_NUM_MST_OFFSETS_MT4117 3128
#define MLX5_NUM_MST_OFFSETS_MT4115 3577
#define MLX5_MST_DUMP_SIZE_BYTES_MT4115 (0x561e0l)
#define MLX5_MST_DUMP_SIZE_BYTES_MT4117 (0x4eed0l)

static const unsigned long
mlx5_mst_dump_regs_mt4117[MLX5_NUM_MST_OFFSETS_MT4117][2] = {{0x000000, 16388},
	{0x010084, 1},
	{0x01008c, 2},
	{0x010180, 6},
	{0x01019c, 9},
	{0x010304, 1},
	{0x0103bc, 52},
	{0x010500, 35},
	{0x010604, 1},
	{0x01060c, 1},
	{0x010624, 1},
	{0x01062c, 1},
	{0x010644, 1},
	{0x01064c, 3},
	{0x010800, 1},
	{0x010814, 3},
	{0x010844, 1},
	{0x01084c, 4},
	{0x011000, 32},
	{0x011200, 32},
	{0x011400, 32},
	{0x011600, 32},
	{0x011800, 32},
	{0x011a00, 5},
	{0x011a24, 1},
	{0x011a2c, 1},
	{0x011a34, 3},
	{0x011a44, 1},
	{0x011a4c, 2},
	{0x012000, 20},
	{0x012088, 22},
	{0x012100, 18},
	{0x012200, 36},
	{0x012300, 3},
	{0x012310, 1},
	{0x012320, 1},
	{0x012330, 1},
	{0x012340, 1},
	{0x012350, 1},
	{0x012360, 1},
	{0x012370, 1},
	{0x012400, 166},
	{0x0126a0, 1},
	{0x012700, 32},
	{0x012800, 1},
	{0x012808, 1},
	{0x012810, 1},
	{0x012818, 1},
	{0x012820, 1},
	{0x012828, 1},
	{0x012830, 1},
	{0x012838, 1},
	{0x012850, 1},
	{0x012858, 1},
	{0x012860, 1},
	{0x012868, 1},
	{0x012870, 1},
	{0x012878, 1},
	{0x012880, 1},
	{0x012888, 1},
	{0x012890, 5},
	{0x012900, 10},
	{0x012940, 6},
	{0x012960, 6},
	{0x012980, 5},
	{0x012a00, 6},
	{0x012a40, 5},
	{0x012a80, 9},
	{0x012ac0, 19},
	{0x012b10, 2},
	{0x014000, 1},
	{0x014010, 10},
	{0x014104, 1},
	{0x01410c, 4},
	{0x014200, 1},
	{0x014210, 4},
	{0x014300, 4},
	{0x014400, 3},
	{0x014410, 1},
	{0x014444, 1},
	{0x01444c, 4},
	{0x014480, 2},
	{0x014500, 35},
	{0x014590, 1},
	{0x014600, 32},
	{0x014704, 1},
	{0x01470c, 1},
	{0x014804, 2},
	{0x014814, 3},
	{0x014824, 2},
	{0x014834, 3},
	{0x014844, 2},
	{0x014854, 5},
	{0x014a00, 4},
	{0x014a14, 2},
	{0x015004, 1},
	{0x01500c, 4},
	{0x015204, 1},
	{0x015214, 4},
	{0x015228, 1},
	{0x015244, 1},
	{0x015250, 4},
	{0x015274, 1},
	{0x01527c, 1},
	{0x0152a4, 1},
	{0x015300, 4},
	{0x015314, 1},
	{0x01531c, 3},
	{0x015404, 2},
	{0x015430, 20},
	{0x015484, 2},
	{0x015498, 5},
	{0x0154bc, 1},
	{0x0154c8, 1},
	{0x0154d4, 2},
	{0x01550c, 3},
	{0x015524, 3},
	{0x015608, 3},
	{0x015618, 4},
	{0x015680, 16},
	{0x015704, 2},
	{0x015730, 20},
	{0x015784, 3},
	{0x015794, 3},
	{0x0157c0, 6},
	{0x015800, 52},
	{0x0158d4, 1},
	{0x0158dc, 1},
	{0x0158e4, 3},
	{0x0158f4, 1},
	{0x0158fc, 52},
	{0x015b04, 2},
	{0x015b14, 3},
	{0x015b24, 2},
	{0x015b38, 6},
	{0x015b60, 5},
	{0x015c04, 1},
	{0x015cb8, 18},
	{0x015d98, 1},
	{0x015e00, 1},
	{0x015e08, 1},
	{0x015e10, 1},
	{0x016000, 32},
	{0x016100, 7},
	{0x016120, 2},
	{0x016144, 3},
	{0x016204, 1},
	{0x01620c, 1},
	{0x016224, 1},
	{0x01622c, 1},
	{0x016234, 1},
	{0x01623c, 1},
	{0x016244, 1},
	{0x01624c, 1},
	{0x016254, 1},
	{0x01625c, 2},
	{0x016270, 1},
	{0x016280, 1},
	{0x016290, 1},
	{0x0162a0, 1},
	{0x0162b0, 1},
	{0x0162c0, 1},
	{0x0162d0, 1},
	{0x0162e0, 1},
	{0x0162f0, 1},
	{0x016300, 1},
	{0x016310, 1},
	{0x016320, 1},
	{0x016330, 1},
	{0x016340, 32},
	{0x016400, 19},
	{0x016454, 7},
	{0x016484, 1},
	{0x01648c, 1},
	{0x016800, 9},
	{0x016840, 16},
	{0x017800, 1},
	{0x017828, 1},
	{0x017850, 1},
	{0x017878, 1},
	{0x0178a0, 12},
	{0x0179ac, 1},
	{0x0179d8, 4},
	{0x017a00, 6},
	{0x017a20, 6},
	{0x017a40, 3},
	{0x017c00, 6},
	{0x017c20, 6},
	{0x017c40, 6},
	{0x017c60, 6},
	{0x017c80, 6},
	{0x017ca0, 6},
	{0x017cc0, 6},
	{0x017ce0, 6},
	{0x017d00, 3},
	{0x017d10, 6},
	{0x018400, 11},
	{0x018430, 2},
	{0x018440, 4},
	{0x018604, 1},
	{0x018618, 2},
	{0x018640, 4},
	{0x018660, 2},
	{0x018800, 1},
	{0x018810, 4},
	{0x018844, 1},
	{0x01884c, 4},
	{0x018880, 2},
	{0x01a400, 12},
	{0x01a444, 1},
	{0x01a44c, 1},
	{0x01a800, 1},
	{0x01a814, 3},
	{0x01a844, 1},
	{0x01a84c, 4},
	{0x01c400, 7},
	{0x01c500, 8},
	{0x01c544, 1},
	{0x01c554, 3},
	{0x01c564, 3},
	{0x01c574, 3},
	{0x01c604, 1},
	{0x01c60c, 3},
	{0x01c800, 1},
	{0x01c814, 3},
	{0x01c844, 1},
	{0x01c84c, 4},
	{0x01f000, 1},
	{0x01f400, 1},
	{0x01f408, 4},
	{0x01f41c, 3},
	{0x01f500, 2},
	{0x01f800, 1},
	{0x01f814, 3},
	{0x01f844, 1},
	{0x01f84c, 4},
	{0x01f880, 3},
	{0x020004, 1},
	{0x02000c, 4},
	{0x020080, 3},
	{0x020090, 5},
	{0x020800, 16},
	{0x020900, 192},
	{0x021004, 1},
	{0x02100c, 4},
	{0x021400, 5},
	{0x021418, 5},
	{0x021480, 1},
	{0x021704, 2},
	{0x02173c, 17},
	{0x0217fc, 2},
	{0x021844, 1},
	{0x022280, 12},
	{0x022408, 6},
	{0x022444, 1},
	{0x022454, 3},
	{0x022504, 1},
	{0x02250c, 4},
	{0x022624, 1},
	{0x02262c, 3},
	{0x02263c, 1},
	{0x022804, 1},
	{0x02280c, 4},
	{0x022904, 4},
	{0x022924, 4},
	{0x024000, 36},
	{0x0240c0, 21},
	{0x024120, 11},
	{0x024200, 5},
	{0x024220, 1},
	{0x024230, 8},
	{0x024258, 1},
	{0x024260, 1},
	{0x024270, 9},
	{0x0242a0, 1},
	{0x0242b0, 4},
	{0x0242c8, 2},
	{0x024300, 5},
	{0x024318, 2},
	{0x02439c, 25},
	{0x024424, 4},
	{0x024464, 13},
	{0x0244a4, 1},
	{0x0244ac, 3},
	{0x0244c0, 2},
	{0x0244d0, 3},
	{0x0244e0, 3},
	{0x0244f0, 3},
	{0x024500, 3},
	{0x024510, 3},
	{0x024520, 3},
	{0x024530, 3},
	{0x024884, 11},
	{0x0248b4, 3},
	{0x0248c4, 1},
	{0x0248cc, 1},
	{0x0248d4, 1},
	{0x0248dc, 1},
	{0x0248f0, 2},
	{0x024908, 6},
	{0x024928, 6},
	{0x024968, 6},
	{0x024984, 3},
	{0x024994, 1},
	{0x02499c, 6},
	{0x0249b8, 7},
	{0x024a08, 6},
	{0x024a28, 6},
	{0x024a68, 6},
	{0x024a84, 1},
	{0x024a8c, 1},
	{0x024a94, 1},
	{0x024a9c, 13},
	{0x024ae0, 6},
	{0x024b00, 7},
	{0x024b20, 6},
	{0x024b40, 8},
	{0x024c00, 2},
	{0x024c24, 3},
	{0x024c34, 3},
	{0x025004, 6},
	{0x025800, 37},
	{0x025904, 1},
	{0x02590c, 1},
	{0x026000, 99},
	{0x026200, 1},
	{0x026800, 7},
	{0x026824, 6},
	{0x026840, 2},
	{0x026864, 1},
	{0x02686c, 1},
	{0x026874, 3},
	{0x026884, 1},
	{0x0268a4, 7},
	{0x026904, 1},
	{0x02690c, 4},
	{0x026940, 1},
	{0x026980, 33},
	{0x026a0c, 7},
	{0x026a30, 1},
	{0x026a44, 4},
	{0x026a60, 1},
	{0x026a70, 1},
	{0x026b00, 9},
	{0x026b44, 2},
	{0x026b68, 6},
	{0x026b84, 2},
	{0x026ba8, 14},
	{0x026c00, 16},
	{0x026c44, 1},
	{0x026c4c, 1},
	{0x026c84, 1},
	{0x026c8c, 4},
	{0x026cc4, 1},
	{0x026ccc, 4},
	{0x026d00, 2},
	{0x028800, 2},
	{0x028844, 1},
	{0x02884c, 4},
	{0x029004, 7},
	{0x029b30, 2},
	{0x029b50, 4},
	{0x02a004, 1},
	{0x02a00c, 4},
	{0x02a040, 6},
	{0x02a200, 1},
	{0x02a210, 11},
	{0x02a240, 8},
	{0x02a484, 1},
	{0x02a4c0, 16},
	{0x02a780, 1},
	{0x02a7a0, 4},
	{0x02a7c0, 1},
	{0x02a900, 1},
	{0x02aa04, 1},
	{0x02aa0c, 4},
	{0x02ab00, 40},
	{0x02aba4, 1},
	{0x02abac, 1},
	{0x02abb4, 1},
	{0x02abbc, 1},
	{0x02abc4, 1},
	{0x02abcc, 1},
	{0x02abe0, 4},
	{0x02abf4, 2},
	{0x02ac00, 2},
	{0x02ac10, 3},
	{0x02ac44, 3},
	{0x02ad00, 1},
	{0x02ad08, 1},
	{0x02ad10, 1},
	{0x02ad18, 1},
	{0x02ad20, 1},
	{0x02ad28, 1},
	{0x02ad30, 1},
	{0x02ad38, 1},
	{0x02ad40, 1},
	{0x02ad48, 1},
	{0x02ad50, 1},
	{0x02ad58, 1},
	{0x02ad60, 1},
	{0x02ad68, 1},
	{0x02ad70, 1},
	{0x02ad78, 1},
	{0x02ad80, 1},
	{0x02ad88, 1},
	{0x02ad90, 1},
	{0x02ad98, 1},
	{0x02ada0, 1},
	{0x02ada8, 1},
	{0x02adb0, 1},
	{0x02adb8, 1},
	{0x02adc0, 1},
	{0x02adc8, 1},
	{0x02add0, 1},
	{0x02add8, 1},
	{0x02ade0, 1},
	{0x02ade8, 1},
	{0x02adf0, 1},
	{0x02adf8, 1},
	{0x02ae00, 6},
	{0x02ae20, 5},
	{0x02ae40, 4},
	{0x02c000, 6},
	{0x02c100, 5},
	{0x02c204, 1},
	{0x02c214, 3},
	{0x02c224, 1},
	{0x02c22c, 4},
	{0x02c244, 2},
	{0x02c250, 5},
	{0x02c400, 2},
	{0x02c428, 2},
	{0x02c450, 2},
	{0x02c478, 2},
	{0x02c4a0, 21},
	{0x02c5ac, 1},
	{0x02c5d8, 4},
	{0x02c600, 6},
	{0x02c620, 6},
	{0x02c640, 6},
	{0x02c660, 6},
	{0x02c680, 3},
	{0x02c800, 3},
	{0x02c820, 6},
	{0x02c840, 6},
	{0x02c860, 6},
	{0x02c880, 6},
	{0x02c8a0, 6},
	{0x02c8c0, 6},
	{0x02c8e0, 6},
	{0x02c900, 6},
	{0x02c920, 6},
	{0x02c940, 6},
	{0x02c960, 6},
	{0x02c980, 6},
	{0x02c9a0, 6},
	{0x02c9c0, 6},
	{0x02c9e0, 6},
	{0x02ca00, 6},
	{0x02ca20, 6},
	{0x02ca40, 6},
	{0x02ca60, 6},
	{0x02ca80, 6},
	{0x02caa0, 6},
	{0x02cac0, 6},
	{0x02cae0, 6},
	{0x02cb00, 6},
	{0x02cb20, 6},
	{0x02cc48, 5},
	{0x02cd00, 9},
	{0x02cd40, 18},
	{0x030004, 2},
	{0x030034, 19},
	{0x030084, 2},
	{0x0300bc, 17},
	{0x030104, 2},
	{0x030138, 27},
	{0x030310, 3},
	{0x030340, 2},
	{0x03034c, 2},
	{0x030384, 1},
	{0x0303c0, 16},
	{0x030404, 1},
	{0x03040c, 4},
	{0x030804, 1},
	{0x03080c, 4},
	{0x030c04, 1},
	{0x030c0c, 4},
	{0x030c40, 4},
	{0x031000, 11},
	{0x031100, 11},
	{0x031200, 17},
	{0x031280, 6},
	{0x031304, 1},
	{0x03130c, 5},
	{0x031400, 6},
	{0x031420, 1},
	{0x031444, 2},
	{0x031454, 3},
	{0x031464, 2},
	{0x031474, 11},
	{0x031500, 7},
	{0x031520, 6},
	{0x031540, 8},
	{0x031600, 13},
	{0x031640, 6},
	{0x031700, 2},
	{0x034200, 24},
	{0x034280, 10},
	{0x0342ac, 2},
	{0x0342c0, 6},
	{0x0342f0, 39},
	{0x034600, 24},
	{0x034680, 10},
	{0x0346ac, 2},
	{0x0346c0, 6},
	{0x0346f0, 39},
	{0x034c00, 6},
	{0x034c20, 4},
	{0x034c40, 9},
	{0x034c80, 9},
	{0x034cc0, 3},
	{0x034cd0, 3},
	{0x034e04, 2},
	{0x034e10, 4},
	{0x034e44, 1},
	{0x034e4c, 4},
	{0x034e80, 6},
	{0x034ea0, 4},
	{0x034f04, 1},
	{0x034f18, 11},
	{0x034f80, 2},
	{0x035000, 2},
	{0x035010, 3},
	{0x035044, 3},
	{0x035100, 66},
	{0x035210, 3},
	{0x035244, 3},
	{0x035300, 64},
	{0x035404, 1},
	{0x03540c, 8},
	{0x037000, 6},
	{0x03702c, 7},
	{0x037080, 10},
	{0x0370ac, 4},
	{0x037100, 4},
	{0x037200, 2},
	{0x037210, 3},
	{0x037244, 3},
	{0x037300, 70},
	{0x03742c, 7},
	{0x037480, 10},
	{0x0374ac, 4},
	{0x037500, 4},
	{0x037600, 2},
	{0x037610, 3},
	{0x037644, 3},
	{0x037700, 69},
	{0x037818, 4},
	{0x038000, 3},
	{0x038104, 3},
	{0x03813c, 2},
	{0x038150, 3},
	{0x038400, 1},
	{0x038428, 1},
	{0x038450, 1},
	{0x038478, 1},
	{0x0384a0, 20},
	{0x0385ac, 1},
	{0x0385d8, 4},
	{0x038600, 6},
	{0x038620, 6},
	{0x038640, 3},
	{0x038800, 3},
	{0x038820, 6},
	{0x038840, 6},
	{0x038860, 6},
	{0x038880, 6},
	{0x0388a0, 6},
	{0x0388c0, 6},
	{0x038900, 28},
	{0x038978, 2},
	{0x038a40, 25},
	{0x038ac0, 16},
	{0x039000, 35},
	{0x039090, 3},
	{0x039100, 35},
	{0x039190, 3},
	{0x039200, 35},
	{0x039290, 3},
	{0x039300, 35},
	{0x039390, 3},
	{0x039400, 35},
	{0x039490, 3},
	{0x039500, 1},
	{0x039800, 3},
	{0x039884, 1},
	{0x0398c0, 16},
	{0x039904, 2},
	{0x039934, 20},
	{0x039a04, 2},
	{0x039a10, 4},
	{0x039a24, 2},
	{0x039a30, 4},
	{0x039a44, 2},
	{0x039a50, 4},
	{0x039a64, 2},
	{0x039a70, 8},
	{0x039c00, 7},
	{0x039c20, 6},
	{0x039c40, 8},
	{0x039d00, 11},
	{0x039d40, 11},
	{0x039d84, 1},
	{0x039dc0, 26},
	{0x039e30, 2},
	{0x039e44, 3},
	{0x039e54, 1},
	{0x039e5c, 2},
	{0x039e80, 2},
	{0x039e90, 3},
	{0x039ea4, 1},
	{0x039eac, 3},
	{0x039ec0, 3},
	{0x039f00, 9},
	{0x039f40, 22},
	{0x039fa0, 6},
	{0x039fc0, 8},
	{0x03c000, 3},
	{0x03c010, 3},
	{0x03c020, 3},
	{0x03c040, 9},
	{0x03c068, 6},
	{0x03c090, 2},
	{0x03c0a0, 3},
	{0x03c0c0, 12},
	{0x03c0f4, 1},
	{0x03c100, 2},
	{0x03c110, 3},
	{0x03c120, 1},
	{0x03c130, 11},
	{0x03c160, 2},
	{0x03c180, 4},
	{0x03c194, 3},
	{0x03c1a4, 2},
	{0x03c1b0, 4},
	{0x03c2e0, 5},
	{0x03c2f8, 2},
	{0x03c30c, 13},
	{0x03c34c, 77},
	{0x03c48c, 18},
	{0x03c500, 32},
	{0x03c800, 16},
	{0x03c84c, 18},
	{0x03c8e0, 2},
	{0x03c8ec, 3},
	{0x03ca78, 34},
	{0x03cb3c, 18},
	{0x03cb94, 3},
	{0x03cba4, 3},
	{0x03cbf0, 1},
	{0x03cbf8, 10},
	{0x03cc30, 1},
	{0x03cc44, 4},
	{0x03cc60, 1},
	{0x03cc80, 1},
	{0x03cc90, 1},
	{0x03d004, 6},
	{0x03e004, 1},
	{0x03e00c, 4},
	{0x03e404, 1},
	{0x03e40c, 4},
	{0x03e604, 1},
	{0x03e60c, 4},
	{0x03e800, 1},
	{0x03f034, 19},
	{0x03f084, 2},
	{0x03f0c0, 16},
	{0x03f200, 1},
	{0x03f210, 1},
	{0x03f300, 6},
	{0x03f320, 6},
	{0x03f380, 9},
	{0x03f3c0, 16},
	{0x050000, 1},
	{0x050008, 2},
	{0x050044, 8},
	{0x050104, 1},
	{0x050178, 34},
	{0x050204, 1},
	{0x05020c, 1},
	{0x050214, 1},
	{0x050224, 1},
	{0x05022c, 1},
	{0x050234, 1},
	{0x05023c, 1},
	{0x050244, 1},
	{0x05024c, 1},
	{0x050254, 1},
	{0x050264, 3},
	{0x050280, 2},
	{0x050290, 2},
	{0x0502b4, 1},
	{0x0502bc, 1},
	{0x0502c4, 1},
	{0x0502cc, 1},
	{0x0502d4, 1},
	{0x0502dc, 1},
	{0x0502e4, 1},
	{0x0502ec, 1},
	{0x0502f4, 1},
	{0x0502fc, 1},
	{0x050304, 1},
	{0x05030c, 1},
	{0x050314, 3},
	{0x050324, 3},
	{0x050334, 6},
	{0x050380, 32},
	{0x050404, 1},
	{0x050438, 18},
	{0x050500, 2},
	{0x050544, 1},
	{0x05054c, 4},
	{0x050584, 2},
	{0x050598, 2},
	{0x0505a4, 1},
	{0x0505b0, 4},
	{0x0505c4, 1},
	{0x0505cc, 1},
	{0x0505d4, 1},
	{0x0505e0, 1},
	{0x0505f0, 17},
	{0x050700, 2},
	{0x050800, 1},
	{0x050820, 9},
	{0x050850, 5},
	{0x050870, 5},
	{0x0508c4, 3},
	{0x0508d4, 3},
	{0x0508e4, 3},
	{0x0508f4, 4},
	{0x050910, 5},
	{0x050930, 4},
	{0x050944, 1},
	{0x05094c, 47},
	{0x050a40, 1},
	{0x050a50, 28},
	{0x050ac4, 1},
	{0x050acc, 2},
	{0x050ad8, 10},
	{0x050b04, 1},
	{0x050b0c, 1},
	{0x050b14, 1},
	{0x050b20, 9},
	{0x050b50, 4},
	{0x050b64, 1},
	{0x050b70, 6},
	{0x050b8c, 4},
	{0x050ba4, 4},
	{0x050bc0, 1},
	{0x050bd0, 1},
	{0x050be0, 1},
	{0x050bf0, 1},
	{0x050c00, 1},
	{0x050c14, 1},
	{0x050c1c, 1},
	{0x050c24, 1},
	{0x050c2c, 1},
	{0x050c34, 1},
	{0x050c3c, 1},
	{0x050c44, 1},
	{0x050c4c, 1},
	{0x050c54, 4},
	{0x050c70, 1},
	{0x050c80, 1},
	{0x050c90, 1},
	{0x050ca0, 1},
	{0x050cb0, 1},
	{0x050cc0, 1},
	{0x050cd0, 1},
	{0x050ce0, 1},
	{0x050cf0, 1},
	{0x050d00, 1},
	{0x050d10, 1},
	{0x050d20, 1},
	{0x050d30, 1},
	{0x050d40, 1},
	{0x050d50, 1},
	{0x050d60, 1},
	{0x050d70, 1},
	{0x050d80, 1},
	{0x050d90, 1},
	{0x050da0, 1},
	{0x050db0, 1},
	{0x050dc0, 1},
	{0x050dd0, 1},
	{0x050de0, 1},
	{0x050e00, 64},
	{0x052000, 192},
	{0x052800, 2},
	{0x053000, 51},
	{0x053400, 3},
	{0x053420, 6},
	{0x053440, 6},
	{0x053460, 6},
	{0x053480, 6},
	{0x0534a0, 6},
	{0x0534c0, 6},
	{0x0534e0, 6},
	{0x053500, 6},
	{0x053520, 6},
	{0x053540, 6},
	{0x053560, 6},
	{0x053614, 11},
	{0x053654, 11},
	{0x0536d4, 11},
	{0x053704, 1},
	{0x053740, 18},
	{0x0537a0, 10},
	{0x0537e0, 29},
	{0x053860, 10},
	{0x054204, 1},
	{0x05420c, 1},
	{0x054214, 1},
	{0x054224, 1},
	{0x05422c, 1},
	{0x054234, 1},
	{0x054244, 1},
	{0x05424c, 3},
	{0x054304, 1},
	{0x054340, 16},
	{0x0543a0, 1},
	{0x0543b0, 1},
	{0x0543c0, 1},
	{0x0543d0, 4},
	{0x054400, 8},
	{0x054440, 10},
	{0x054820, 3},
	{0x054928, 9},
	{0x054954, 4},
	{0x054970, 1},
	{0x054980, 1},
	{0x054990, 5},
	{0x0549c4, 13},
	{0x054a04, 1},
	{0x054a0c, 4},
	{0x054a40, 1},
	{0x054a50, 5},
	{0x054a70, 1},
	{0x054a80, 1},
	{0x054a90, 2},
	{0x054c04, 1},
	{0x054c0c, 4},
	{0x054c40, 1},
	{0x054c50, 4},
	{0x054d40, 1},
	{0x054d80, 7},
	{0x054e04, 1},
	{0x054e0c, 4},
	{0x054e40, 1},
	{0x054e50, 4},
	{0x054f40, 1},
	{0x054f80, 7},
	{0x055000, 11},
	{0x055050, 7},
	{0x055070, 16},
	{0x0550b8, 8},
	{0x055200, 13},
	{0x055240, 11},
	{0x055300, 16},
	{0x055400, 3},
	{0x055420, 6},
	{0x055440, 6},
	{0x055460, 6},
	{0x055480, 6},
	{0x0554a0, 6},
	{0x0554c0, 6},
	{0x0554e0, 6},
	{0x055500, 6},
	{0x055520, 6},
	{0x055540, 6},
	{0x055560, 6},
	{0x055580, 6},
	{0x0555a0, 6},
	{0x0555c0, 6},
	{0x0555e0, 6},
	{0x055600, 6},
	{0x055620, 6},
	{0x055640, 6},
	{0x055660, 6},
	{0x055680, 6},
	{0x056000, 2},
	{0x056044, 1},
	{0x05604c, 4},
	{0x056800, 2},
	{0x056844, 1},
	{0x05684c, 4},
	{0x057004, 7},
	{0x057b20, 2},
	{0x057b30, 2},
	{0x057b50, 5},
	{0x058004, 1},
	{0x05800c, 4},
	{0x058800, 7},
	{0x058820, 6},
	{0x058844, 2},
	{0x058864, 9},
	{0x058890, 1},
	{0x0588a0, 1},
	{0x0588b0, 1},
	{0x0588c0, 1},
	{0x0588d0, 1},
	{0x058904, 3},
	{0x058978, 66},
	{0x058a84, 1},
	{0x058a8c, 1},
	{0x058a94, 1},
	{0x058aa0, 2},
	{0x058ff0, 4},
	{0x059004, 1},
	{0x05900c, 4},
	{0x059080, 39},
	{0x059120, 1},
	{0x059130, 1},
	{0x059140, 1},
	{0x059150, 1},
	{0x059160, 1},
	{0x059800, 1},
	{0x059810, 1},
	{0x059820, 1},
	{0x059830, 1},
	{0x059840, 1},
	{0x059850, 1},
	{0x059860, 1},
	{0x059870, 1},
	{0x059880, 1},
	{0x059890, 1},
	{0x0598a0, 1},
	{0x0598b0, 1},
	{0x0598d8, 3},
	{0x059900, 3},
	{0x059980, 9},
	{0x0599c0, 32},
	{0x059a44, 3},
	{0x059a54, 1},
	{0x059a5c, 1},
	{0x059c00, 2},
	{0x059c28, 2},
	{0x059c50, 2},
	{0x059c78, 2},
	{0x059ca0, 18},
	{0x059dac, 1},
	{0x059dd8, 4},
	{0x059e00, 6},
	{0x059e20, 6},
	{0x059e40, 6},
	{0x059e60, 6},
	{0x059e80, 3},
	{0x05a000, 4},
	{0x05a020, 2},
	{0x05a030, 1},
	{0x05a04c, 2},
	{0x05a070, 4},
	{0x05a088, 3},
	{0x05a0a0, 1},
	{0x05a0b0, 1},
	{0x05a0c0, 1},
	{0x05a0d4, 1},
	{0x060020, 4},
	{0x060120, 4},
	{0x060144, 2},
	{0x060154, 7},
	{0x060174, 1},
	{0x06017c, 3},
	{0x06018c, 1},
	{0x0601a0, 4},
	{0x0601b8, 2},
	{0x0601c4, 2},
	{0x0601e4, 7},
	{0x060204, 1},
	{0x06020c, 4},
	{0x060240, 16},
	{0x060304, 1},
	{0x06030c, 1},
	{0x060400, 2},
	{0x060420, 3},
	{0x060430, 1},
	{0x060440, 1},
	{0x060484, 2},
	{0x0604b8, 18},
	{0x060504, 1},
	{0x06050c, 1},
	{0x060514, 1},
	{0x060900, 1},
	{0x060914, 1},
	{0x06091c, 2},
	{0x060930, 1},
	{0x060a00, 32},
	{0x060a84, 1},
	{0x060a8c, 1},
	{0x060a94, 1},
	{0x060a9c, 1},
	{0x060bf0, 3},
	{0x060c00, 11},
	{0x060c30, 3},
	{0x061004, 1},
	{0x061010, 1},
	{0x061018, 4},
	{0x061044, 2},
	{0x061054, 3},
	{0x061100, 32},
	{0x061200, 1},
	{0x061404, 1},
	{0x06140c, 4},
	{0x061440, 2},
	{0x061468, 2},
	{0x061478, 2},
	{0x06180c, 8},
	{0x061908, 6},
	{0x061928, 6},
	{0x061968, 6},
	{0x061988, 6},
	{0x0619a8, 6},
	{0x0619e8, 14},
	{0x062008, 5},
	{0x062024, 3},
	{0x062034, 1},
	{0x06203c, 1},
	{0x062044, 1},
	{0x06204c, 1},
	{0x062054, 1},
	{0x06205c, 3},
	{0x06206c, 1},
	{0x062080, 2},
	{0x062094, 1},
	{0x06209c, 2},
	{0x0620b0, 1},
	{0x0620c0, 1},
	{0x0620d0, 1},
	{0x0620e0, 1},
	{0x0620f4, 4},
	{0x0621c0, 2},
	{0x0621e4, 3},
	{0x0621f4, 3},
	{0x062404, 1},
	{0x06240c, 4},
	{0x062800, 12},
	{0x062834, 1},
	{0x06283c, 5},
	{0x062854, 1},
	{0x062900, 3},
	{0x062914, 1},
	{0x06291c, 1},
	{0x062924, 1},
	{0x06292c, 1},
	{0x062934, 1},
	{0x06293c, 2},
	{0x062950, 3},
	{0x062960, 2},
	{0x063000, 1},
	{0x063010, 4},
	{0x063024, 1},
	{0x06302c, 1},
	{0x063034, 1},
	{0x063044, 1},
	{0x06304c, 1},
	{0x063054, 1},
	{0x06305c, 2},
	{0x063070, 1},
	{0x063080, 1},
	{0x063090, 1},
	{0x0630a4, 3},
	{0x063100, 2},
	{0x063144, 1},
	{0x06314c, 1},
	{0x063154, 1},
	{0x063164, 1},
	{0x06316c, 1},
	{0x063174, 1},
	{0x063180, 16},
	{0x063208, 1},
	{0x063210, 1},
	{0x063218, 1},
	{0x063224, 3},
	{0x063240, 10},
	{0x063300, 8},
	{0x064004, 1},
	{0x06400c, 4},
	{0x064200, 3},
	{0x064300, 1},
	{0x064308, 6},
	{0x064324, 2},
	{0x064338, 2},
	{0x064380, 2},
	{0x064394, 1},
	{0x06439c, 2},
	{0x064400, 2},
	{0x064420, 3},
	{0x064430, 1},
	{0x064440, 1},
	{0x064484, 2},
	{0x0644b8, 18},
	{0x064504, 1},
	{0x06450c, 1},
	{0x064514, 1},
	{0x064804, 1},
	{0x064884, 2},
	{0x0648c8, 14},
	{0x065018, 2},
	{0x06507c, 10},
	{0x0650c0, 16},
	{0x065104, 3},
	{0x065114, 1},
	{0x065200, 3},
	{0x065220, 6},
	{0x065240, 6},
	{0x065260, 6},
	{0x065280, 6},
	{0x065300, 1},
	{0x065f00, 2},
	{0x066000, 2},
	{0x066028, 2},
	{0x066050, 2},
	{0x066078, 2},
	{0x0660a0, 27},
	{0x0661ac, 1},
	{0x0661d8, 4},
	{0x066200, 6},
	{0x066220, 6},
	{0x066240, 6},
	{0x066260, 6},
	{0x066280, 3},
	{0x066400, 6},
	{0x066500, 17},
	{0x066550, 8},
	{0x066574, 3},
	{0x070004, 1},
	{0x07000c, 4},
	{0x070400, 7},
	{0x070420, 11},
	{0x070500, 5},
	{0x070524, 1},
	{0x07052c, 1},
	{0x070534, 1},
	{0x070540, 4},
	{0x070600, 142},
	{0x070884, 2},
	{0x0708b4, 19},
	{0x070904, 1},
	{0x070940, 16},
	{0x071000, 1},
	{0x071094, 1},
	{0x071104, 2},
	{0x071140, 16},
	{0x071204, 1},
	{0x07120c, 4},
	{0x071404, 1},
	{0x07140c, 1},
	{0x071414, 1},
	{0x07141c, 1},
	{0x071424, 1},
	{0x07142c, 1},
	{0x071434, 1},
	{0x07143c, 1},
	{0x071800, 2},
	{0x071904, 1},
	{0x071940, 16},
	{0x071a04, 1},
	{0x071a40, 16},
	{0x071b04, 1},
	{0x071b40, 16},
	{0x072000, 3},
	{0x072804, 1},
	{0x07280c, 4},
	{0x072880, 1},
	{0x072888, 3},
	{0x073000, 1},
	{0x073020, 1},
	{0x073040, 1},
	{0x073060, 1},
	{0x073080, 1},
	{0x0730a0, 1},
	{0x0730c0, 1},
	{0x0730e0, 1},
	{0x073100, 1},
	{0x073120, 1},
	{0x073140, 1},
	{0x073160, 1},
	{0x073180, 1},
	{0x0731a0, 1},
	{0x0731c0, 1},
	{0x0731e0, 1},
	{0x073200, 1},
	{0x073220, 1},
	{0x073240, 1},
	{0x073260, 1},
	{0x073280, 1},
	{0x0732a0, 1},
	{0x0732c0, 1},
	{0x0732e0, 1},
	{0x073300, 1},
	{0x073320, 1},
	{0x073340, 1},
	{0x073360, 1},
	{0x073380, 1},
	{0x0733a0, 1},
	{0x0733c0, 1},
	{0x0733e0, 1},
	{0x073800, 16},
	{0x074830, 4},
	{0x074884, 1},
	{0x074890, 4},
	{0x074900, 3},
	{0x074920, 6},
	{0x074940, 6},
	{0x074980, 9},
	{0x0749c0, 16},
	{0x075000, 3},
	{0x075010, 2},
	{0x075020, 4},
	{0x075100, 2},
	{0x080000, 29},
	{0x080340, 14},
	{0x08037c, 3},
	{0x08038c, 1},
	{0x0803c0, 2},
	{0x0803e4, 3},
	{0x0803f4, 3},
	{0x080404, 6},
	{0x080804, 2},
	{0x080874, 35},
	{0x081000, 129},
	{0x081210, 4},
	{0x081228, 3},
	{0x081240, 2},
	{0x081264, 2},
	{0x081274, 3},
	{0x081284, 2},
	{0x081298, 2},
	{0x0812a4, 1},
	{0x0812b4, 21},
	{0x081310, 8},
	{0x081344, 1},
	{0x08134c, 1},
	{0x081354, 1},
	{0x081364, 2},
	{0x081370, 4},
	{0x081384, 2},
	{0x081390, 4},
	{0x081404, 2},
	{0x081478, 34},
	{0x081504, 2},
	{0x081518, 14},
	{0x081580, 5},
	{0x081598, 2},
	{0x0815a4, 10},
	{0x082000, 29},
	{0x082340, 14},
	{0x08237c, 3},
	{0x08238c, 1},
	{0x0823c0, 2},
	{0x0823e4, 3},
	{0x0823f4, 3},
	{0x082404, 1},
	{0x08240c, 4},
	{0x082804, 2},
	{0x082874, 35},
	{0x082904, 2},
	{0x082974, 35},
	{0x083000, 129},
	{0x083210, 6},
	{0x083244, 2},
	{0x083254, 7},
	{0x083284, 1},
	{0x08328c, 1},
	{0x083294, 1},
	{0x0832a4, 1},
	{0x0832b4, 19},
	{0x083304, 2},
	{0x083310, 4},
	{0x083324, 2},
	{0x083330, 14},
	{0x084000, 29},
	{0x084340, 14},
	{0x08437c, 3},
	{0x08438c, 1},
	{0x0843c0, 2},
	{0x0843e4, 3},
	{0x0843f4, 3},
	{0x084404, 1},
	{0x08440c, 4},
	{0x084804, 2},
	{0x084874, 35},
	{0x084904, 2},
	{0x084974, 35},
	{0x085000, 32},
	{0x085200, 1},
	{0x085210, 7},
	{0x085240, 12},
	{0x085280, 2},
	{0x0852a4, 1},
	{0x0852b4, 3},
	{0x085304, 1},
	{0x08530c, 1},
	{0x085314, 1},
	{0x085324, 2},
	{0x085334, 3},
	{0x085344, 2},
	{0x085358, 2},
	{0x085364, 2},
	{0x085378, 2},
	{0x085384, 2},
	{0x085398, 2},
	{0x0853c0, 23},
	{0x086000, 2},
	{0x086020, 2},
	{0x086040, 1},
	{0x086400, 11},
	{0x086800, 3},
	{0x086820, 6},
	{0x086840, 6},
	{0x086860, 6},
	{0x086880, 6},
	{0x0868a0, 6},
	{0x0868c0, 6},
	{0x0868e0, 6},
	{0x086900, 9},
	{0x086940, 17},
	{0x087000, 26},
	{0x087100, 1},
	{0x087108, 1},
	{0x087110, 1},
	{0x087118, 1},
	{0x087120, 1},
	{0x087128, 1},
	{0x087130, 1},
	{0x087138, 1},
	{0x087140, 1},
	{0x087148, 1},
	{0x087150, 1},
	{0x087158, 1},
	{0x087160, 1},
	{0x087168, 1},
	{0x087170, 1},
	{0x087178, 1},
	{0x087180, 10},
	{0x0871b0, 9},
	{0x087200, 1},
	{0x087208, 1},
	{0x087210, 3},
	{0x090000, 17},
	{0x090060, 2},
	{0x09006c, 1},
	{0x090104, 1},
	{0x090140, 25},
	{0x0901a8, 2},
	{0x0901c0, 9},
	{0x0901e8, 2},
	{0x090204, 1},
	{0x090220, 24},
	{0x090300, 6},
	{0x090320, 9},
	{0x090348, 1},
	{0x090350, 1},
	{0x090400, 6},
	{0x090420, 9},
	{0x090448, 1},
	{0x090450, 1},
	{0x090500, 6},
	{0x090520, 6},
	{0x090540, 2},
	{0x090564, 2},
	{0x090578, 3},
	{0x091004, 3},
	{0x091800, 8},
	{0x091824, 2},
	{0x091830, 10},
	{0x091860, 6},
	{0x092000, 32},
	{0x093000, 1},
	{0x093020, 1},
	{0x093040, 1},
	{0x093060, 1},
	{0x093080, 1},
	{0x0930a0, 1},
	{0x0930c0, 1},
	{0x0930e0, 1},
	{0x093100, 1},
	{0x0931a0, 1},
	{0x0931c0, 1},
	{0x093200, 3},
	{0x093404, 1},
	{0x093440, 16},
	{0x093504, 1},
	{0x09353c, 28},
	{0x0935b0, 2},
	{0x0935c0, 3},
	{0x094000, 9},
	{0x094040, 19},
	{0x0940c0, 2},
	{0x094800, 1},
	{0x094828, 1},
	{0x094850, 1},
	{0x094878, 1},
	{0x0948a0, 8},
	{0x0949ac, 1},
	{0x0949d8, 4},
	{0x094a00, 6},
	{0x094a20, 6},
	{0x094a40, 3},
	{0x096000, 1},
	{0x096010, 4},
	{0x096028, 3},
	{0x096104, 1},
	{0x09610c, 7},
	{0x096204, 1},
	{0x09620c, 4},
	{0x096488, 1},
	{0x096498, 3},
	{0x0964b0, 4},
	{0x096504, 1},
	{0x09650c, 4},
	{0x096584, 4},
	{0x096600, 16},
	{0x096644, 2},
	{0x096658, 10},
	{0x096684, 2},
	{0x0966bc, 51},
	{0x096800, 11},
	{0x096884, 3},
	{0x0968a0, 12},
	{0x097100, 6},
	{0x097120, 1},
	{0x0a2004, 1},
	{0x0a200c, 4},
	{0x0a2400, 2},
	{0x0a240c, 6},
	{0x0a2440, 1},
	{0x0a2450, 4},
	{0x0a2468, 3},
	{0x0a2480, 1},
	{0x0a24a0, 10},
	{0x0a24d0, 9},
	{0x0a2804, 1},
	{0x0a280c, 4},
	{0x0a2c00, 2},
	{0x0a2c0c, 2},
	{0x0a2c40, 1},
	{0x0a2c50, 4},
	{0x0a2c68, 3},
	{0x0a2c88, 2},
	{0x0a2cf0, 1},
	{0x0a3004, 1},
	{0x0a300c, 4},
	{0x0a3040, 2},
	{0x0a3064, 3},
	{0x0a3074, 3},
	{0x0a3200, 9},
	{0x0a3230, 2},
	{0x0a323c, 2},
	{0x0a3248, 4},
	{0x0a3400, 1},
	{0x0a3408, 3},
	{0x0a3418, 4},
	{0x0a3430, 2},
	{0x0a343c, 1},
	{0x0a3480, 1},
	{0x0a3490, 1},
	{0x0a3504, 1},
	{0x0a3510, 76},
	{0x0a4000, 48},
	{0x0a4100, 3},
	{0x0a4110, 6},
	{0x0a412c, 4},
	{0x0a4140, 1},
	{0x0a4304, 1},
	{0x0a4318, 10},
	{0x0a4804, 1},
	{0x0a480c, 4},
	{0x0a4840, 2},
	{0x0a4864, 3},
	{0x0a4874, 3},
	{0x0a4c04, 2},
	{0x0a4c10, 2},
	{0x0a4c1c, 6},
	{0x0a4c38, 2},
	{0x0a4c50, 8},
	{0x0a4c78, 19},
	{0x0a4d04, 2},
	{0x0a4d40, 21},
	{0x0a4da4, 1},
	{0x0a4dac, 1},
	{0x0a4db4, 1},
	{0x0a4dc0, 1},
	{0x0a5000, 14},
	{0x0a6000, 1},
	{0x0a6028, 1},
	{0x0a6050, 1},
	{0x0a6078, 1},
	{0x0a60a0, 18},
	{0x0a61ac, 1},
	{0x0a61d8, 4},
	{0x0a6200, 6},
	{0x0a6220, 6},
	{0x0a6240, 3},
	{0x0a6400, 3},
	{0x0a6420, 6},
	{0x0a6440, 6},
	{0x0a6460, 6},
	{0x0a6480, 6},
	{0x0a64a0, 6},
	{0x0a64c0, 6},
	{0x0a6500, 9},
	{0x0a6540, 18},
	{0x0a65c0, 4},
	{0x0a8000, 10},
	{0x0a802c, 15},
	{0x0a806c, 5},
	{0x0a8700, 17},
	{0x0a8750, 4},
	{0x0a8800, 4},
	{0x0a8880, 1},
	{0x0a88a0, 1},
	{0x0a88c0, 1},
	{0x0a8900, 1},
	{0x0a8960, 1},
	{0x0a8980, 4},
	{0x0a89a4, 3},
	{0x0a89c0, 1},
	{0x0a8a00, 4},
	{0x0a8ad4, 10},
	{0x0a8b00, 2},
	{0x0a8b80, 4},
	{0x0a9804, 1},
	{0x0a980c, 4},
	{0x0a9f00, 6},
	{0x0a9f20, 6},
	{0x0a9f40, 6},
	{0x0a9f60, 6},
	{0x0a9f80, 9},
	{0x0a9fc0, 16},
	{0x0aa020, 1},
	{0x0aa028, 2},
	{0x0b0000, 2},
	{0x0b0010, 4},
	{0x0b1000, 15},
	{0x0b1040, 10},
	{0x0b1080, 20},
	{0x0b1100, 2},
	{0x0b1110, 2},
	{0x0b1120, 2},
	{0x0b1160, 2},
	{0x0b116c, 1},
	{0x0b1180, 3},
	{0x0b1190, 3},
	{0x0b11a0, 3},
	{0x0b11e0, 2},
	{0x0b11ec, 1},
	{0x0b1200, 2},
	{0x0b1210, 2},
	{0x0b1260, 2},
	{0x0b126c, 1},
	{0x0b1280, 3},
	{0x0b1290, 3},
	{0x0b12e0, 2},
	{0x0b12ec, 1},
	{0x0b1300, 2},
	{0x0b1310, 2},
	{0x0b1320, 2},
	{0x0b1360, 2},
	{0x0b136c, 1},
	{0x0b1380, 3},
	{0x0b1390, 3},
	{0x0b13e0, 2},
	{0x0b13ec, 1},
	{0x0b1400, 3},
	{0x0b1410, 3},
	{0x0b1460, 2},
	{0x0b146c, 1},
	{0x0b1480, 3},
	{0x0b1490, 7},
	{0x0b14b0, 4},
	{0x0b14e0, 2},
	{0x0b14ec, 1},
	{0x0b1500, 3},
	{0x0b1510, 3},
	{0x0b1560, 2},
	{0x0b156c, 1},
	{0x0b1580, 2},
	{0x0b1590, 2},
	{0x0b15e0, 2},
	{0x0b15ec, 1},
	{0x0b1600, 3},
	{0x0b1610, 3},
	{0x0b1660, 2},
	{0x0b166c, 1},
	{0x0b1680, 3},
	{0x0b1690, 3},
	{0x0b16e0, 2},
	{0x0b16ec, 1},
	{0x0b1700, 8},
	{0x0b1760, 2},
	{0x0b176c, 1},
	{0x0b1780, 3},
	{0x0b1790, 3},
	{0x0b17e0, 2},
	{0x0b17ec, 1},
	{0x0b1800, 9},
	{0x0b1840, 16},
	{0x0b2000, 9},
	{0x0b2040, 48},
	{0x0b2104, 7},
	{0x0b2200, 9},
	{0x0b2240, 48},
	{0x0b2304, 7},
	{0x0b2400, 9},
	{0x0b2440, 48},
	{0x0b2504, 7},
	{0x0b2600, 9},
	{0x0b2640, 48},
	{0x0b2704, 7},
	{0x0b2800, 9},
	{0x0b2840, 48},
	{0x0b2904, 7},
	{0x0b2a00, 9},
	{0x0b2a40, 48},
	{0x0b2b04, 7},
	{0x0b2c00, 9},
	{0x0b2c40, 48},
	{0x0b2d04, 7},
	{0x0b2e00, 9},
	{0x0b2e40, 48},
	{0x0b2f04, 7},
	{0x0b3000, 65},
	{0x0b3140, 35},
	{0x0b31d0, 1},
	{0x0b4000, 132},
	{0x0b4240, 13},
	{0x0b4278, 26},
	{0x0b4300, 9},
	{0x0b4380, 32},
	{0x0b4404, 1},
	{0x0b440c, 4},
	{0x0b4444, 1},
	{0x0b444c, 4},
	{0x0b4484, 1},
	{0x0b448c, 4},
	{0x0b44c4, 3},
	{0x0b4800, 73},
	{0x0b4940, 28},
	{0x0b49c4, 5},
	{0x0b4a00, 1},
	{0x0b4a20, 16},
	{0x0b5000, 7},
	{0x0b5040, 12},
	{0x0b5100, 9},
	{0x0b5140, 16},
	{0x0b6000, 3},
	{0x0b6028, 3},
	{0x0b6050, 3},
	{0x0b6078, 3},
	{0x0b60a0, 7},
	{0x0b61ac, 1},
	{0x0b61d8, 4},
	{0x0b6200, 6},
	{0x0b6220, 6},
	{0x0b6240, 6},
	{0x0b6260, 6},
	{0x0b6280, 6},
	{0x0b62a0, 6},
	{0x0b62c0, 3},
	{0x0b7800, 3},
	{0x0b7820, 6},
	{0x0b7840, 6},
	{0x0b7860, 6},
	{0x0b7880, 6},
	{0x0b78a0, 6},
	{0x0b78c0, 6},
	{0x0b78e0, 6},
	{0x0b7900, 6},
	{0x0b7a00, 1},
	{0x0d0000, 5},
	{0x0d0038, 4},
	{0x0d0050, 4},
	{0x0d0080, 3},
	{0x0d00a0, 6},
	{0x0d00c0, 6},
	{0x0d0100, 3},
	{0x0d0120, 6},
	{0x0d0140, 6},
	{0x0d0180, 14},
	{0x0d01c0, 6},
	{0x0d1000, 2},
	{0x0d100c, 1},
	{0x0d1020, 13},
	{0x0d1058, 1},
	{0x0d1060, 6},
	{0x0d1080, 1},
	{0x0d1100, 2},
	{0x0d110c, 1},
	{0x0d1120, 13},
	{0x0d1158, 1},
	{0x0d1160, 6},
	{0x0d1180, 1},
	{0x0d1200, 2},
	{0x0d120c, 1},
	{0x0d1220, 13},
	{0x0d1258, 1},
	{0x0d1260, 6},
	{0x0d1280, 1},
	{0x0d1300, 2},
	{0x0d130c, 1},
	{0x0d1320, 13},
	{0x0d1358, 1},
	{0x0d1360, 6},
	{0x0d1380, 1},
	{0x0d1400, 3},
	{0x0d1410, 10},
	{0x0d1440, 1},
	{0x0d1450, 6},
	{0x0d1484, 6},
	{0x0d14a0, 6},
	{0x0d1504, 25},
	{0x0d1580, 14},
	{0x0d15c0, 4},
	{0x0d1600, 6},
	{0x0d1620, 6},
	{0x0d1640, 3},
	{0x0d1660, 6},
	{0x0d1700, 2},
	{0x0d170c, 3},
	{0x0d1720, 6},
	{0x0d1800, 11},
	{0x0d1830, 7},
	{0x0d1850, 7},
	{0x0d1880, 5},
	{0x0d18a0, 3},
	{0x0d18c0, 7},
	{0x0d1a00, 1},
	{0x0d1a08, 13},
	{0x0d1a40, 11},
	{0x0d1c00, 24},
	{0x0d1c64, 5},
	{0x0d1c80, 3},
	{0x0d1c90, 2},
	{0x0d1d00, 15},
	{0x0d1d40, 16},
	{0x0d1d90, 4},
	{0x0d1dc0, 12},
	{0x0d1e00, 1},
	{0x0d1e20, 1},
	{0x0d1e28, 12},
	{0x0d1e5c, 1},
	{0x0d1e80, 6},
	{0x0d1f00, 14},
	{0x0d2000, 2},
	{0x0d200c, 1},
	{0x0d2020, 13},
	{0x0d2058, 1},
	{0x0d2060, 6},
	{0x0d2080, 1},
	{0x0d2100, 2},
	{0x0d210c, 1},
	{0x0d2120, 13},
	{0x0d2158, 1},
	{0x0d2160, 6},
	{0x0d2180, 1},
	{0x0d2200, 2},
	{0x0d220c, 1},
	{0x0d2220, 13},
	{0x0d2258, 1},
	{0x0d2260, 6},
	{0x0d2280, 1},
	{0x0d2300, 2},
	{0x0d230c, 1},
	{0x0d2320, 13},
	{0x0d2358, 1},
	{0x0d2360, 6},
	{0x0d2380, 1},
	{0x0d2400, 3},
	{0x0d2410, 10},
	{0x0d2440, 1},
	{0x0d2450, 6},
	{0x0d2484, 6},
	{0x0d24a0, 6},
	{0x0d2504, 25},
	{0x0d2580, 14},
	{0x0d25c0, 4},
	{0x0d2600, 6},
	{0x0d2620, 6},
	{0x0d2640, 3},
	{0x0d2660, 6},
	{0x0d2700, 2},
	{0x0d270c, 3},
	{0x0d2720, 6},
	{0x0d2800, 11},
	{0x0d2830, 7},
	{0x0d2850, 7},
	{0x0d2880, 5},
	{0x0d28a0, 3},
	{0x0d28c0, 7},
	{0x0d2a00, 1},
	{0x0d2a08, 13},
	{0x0d2a40, 11},
	{0x0d2c00, 24},
	{0x0d2c64, 5},
	{0x0d2c80, 3},
	{0x0d2c90, 2},
	{0x0d2d00, 15},
	{0x0d2d40, 16},
	{0x0d2d90, 4},
	{0x0d2dc0, 12},
	{0x0d3000, 2},
	{0x0d3400, 1},
	{0x0d3428, 1},
	{0x0d3450, 1},
	{0x0d3478, 1},
	{0x0d34a0, 5},
	{0x0d35ac, 1},
	{0x0d35d8, 4},
	{0x0d3600, 6},
	{0x0d3620, 6},
	{0x0d3640, 3},
	{0x0d3e00, 3},
	{0x0d3e20, 6},
	{0x0d3e40, 6},
	{0x0d3e80, 3},
	{0x0d3e90, 2},
	{0x0d4000, 29},
	{0x0d4078, 4},
	{0x0d4090, 2},
	{0x0d40a0, 7},
	{0x0d40c0, 11},
	{0x0d4100, 14},
	{0x0d4140, 14},
	{0x0d4180, 61},
	{0x0d4278, 4},
	{0x0d4290, 2},
	{0x0d42a0, 7},
	{0x0d42c0, 11},
	{0x0d4300, 14},
	{0x0d4340, 14},
	{0x0d4380, 61},
	{0x0d4478, 4},
	{0x0d4490, 2},
	{0x0d44a0, 7},
	{0x0d44c0, 11},
	{0x0d4500, 14},
	{0x0d4540, 14},
	{0x0d4580, 61},
	{0x0d4678, 4},
	{0x0d4690, 2},
	{0x0d46a0, 7},
	{0x0d46c0, 11},
	{0x0d4700, 14},
	{0x0d4740, 14},
	{0x0d4780, 69},
	{0x0d4c00, 6},
	{0x0d4c40, 14},
	{0x0d4c80, 9},
	{0x0d4d00, 9},
	{0x0d4d2c, 1},
	{0x0d4d40, 3},
	{0x0d4d60, 1},
	{0x0d4d80, 3},
	{0x0d4e00, 2},
	{0x0d4e0c, 1},
	{0x0d4e14, 5},
	{0x0d4e2c, 1},
	{0x0d4e34, 5},
	{0x0d4e4c, 1},
	{0x0d4e54, 5},
	{0x0d4e6c, 1},
	{0x0d4e74, 5},
	{0x0d4e8c, 1},
	{0x0d4e94, 5},
	{0x0d4eac, 1},
	{0x0d4eb4, 3},
	{0x0d6000, 15},
	{0x0d6070, 3},
	{0x0d6080, 6},
	{0x0d6100, 9},
	{0x0d6204, 8},
	{0x0d6240, 13},
	{0x0d6280, 16},
	{0x0d6400, 8},
	{0x0d6424, 15},
	{0x0d6464, 15},
	{0x0d64a4, 15},
	{0x0d64e4, 30},
	{0x0d6580, 10},
	{0x0d65ac, 1},
	{0x0d65b4, 5},
	{0x0d65cc, 1},
	{0x0d65d4, 5},
	{0x0d65ec, 1},
	{0x0d65f4, 13},
	{0x0d6680, 4},
	{0x0d6694, 2},
	{0x0d66a0, 5},
	{0x0d66c0, 5},
	{0x0d66e0, 4},
	{0x0d6800, 19},
	{0x0d6850, 10},
	{0x0d6880, 19},
	{0x0d68d0, 10},
	{0x0d6900, 19},
	{0x0d6950, 10},
	{0x0d6980, 19},
	{0x0d69d0, 10},
	{0x0d6a00, 19},
	{0x0d6a50, 10},
	{0x0d6a80, 19},
	{0x0d6ad0, 10},
	{0x0d6c00, 19},
	{0x0d6c60, 6},
	{0x0d6c84, 1},
	{0x0d6c94, 8},
	{0x0d6cb8, 9},
	{0x0d6ce0, 4},
	{0x0d7000, 9},
	{0x0d7040, 16},
	{0x0d8000, 6},
	{0x0d8020, 3},
	{0x0d8030, 3},
	{0x0d8040, 6},
	{0x0d8060, 17},
	{0x0d80c0, 38},
	{0x0d8180, 2},
	{0x0d8400, 2},
	{0x0d8428, 2},
	{0x0d8450, 2},
	{0x0d8478, 2},
	{0x0d84a0, 16},
	{0x0d85ac, 1},
	{0x0d85d8, 4},
	{0x0d8600, 6},
	{0x0d8620, 6},
	{0x0d8640, 6},
	{0x0d8660, 6},
	{0x0d8680, 3},
	{0x0d8800, 2},
	{0x0d9000, 36},
	{0x0d9100, 26},
	{0x0d916c, 7},
	{0x0d91a0, 1},
	{0x0d91c0, 9},
	{0x0d91e8, 1},
	{0x0d9200, 6},
	{0x0d9220, 6},
	{0x0d9248, 4},
	{0x0d9280, 6},
	{0x0d929c, 1},
	{0x0d92a4, 2},
	{0x0d92b8, 9},
	{0x0d9304, 4},
	{0x0d9328, 3},
	{0x0d9340, 6},
	{0x0d9400, 1},
	{0x0d9408, 1},
	{0x0d9410, 2},
	{0x0d9424, 2},
	{0x0d9444, 1},
	{0x0d9480, 27},
	{0x0d9500, 6},
	{0x0d9520, 12},
	{0x0d9700, 7},
	{0x0d9744, 9},
	{0x0d976c, 2},
	{0x0d9780, 6},
	{0x0d97a0, 2},
	{0x0d9800, 36},
	{0x0d9900, 26},
	{0x0d996c, 7},
	{0x0d99a0, 1},
	{0x0d99c0, 9},
	{0x0d99e8, 1},
	{0x0d9a00, 6},
	{0x0d9a20, 6},
	{0x0d9a48, 4},
	{0x0d9a80, 6},
	{0x0d9a9c, 1},
	{0x0d9aa4, 2},
	{0x0d9ab8, 9},
	{0x0d9b04, 4},
	{0x0d9b28, 3},
	{0x0d9b40, 6},
	{0x0d9c00, 1},
	{0x0d9c08, 1},
	{0x0d9c10, 2},
	{0x0d9c24, 2},
	{0x0d9c44, 1},
	{0x0d9c80, 27},
	{0x0d9d00, 6},
	{0x0d9d20, 12},
	{0x0d9f00, 7},
	{0x0d9f44, 9},
	{0x0d9f6c, 2},
	{0x0d9f80, 6},
	{0x0d9fa0, 2},
	{0x0db000, 1028},
	{0x0dc018, 18},
	{0x0dc100, 4},
	{0x0dc118, 18},
	{0x0dc200, 12},
	{0x0dc300, 6},
	{0x0dc320, 5},
	{0x0dc340, 6},
	{0x0dc360, 5},
	{0x0dc380, 6},
	{0x0dc400, 9},
	{0x0dc440, 26},
	{0x0dc4c4, 1},
	{0x0dc4cc, 1},
	{0x0dc4d4, 1},
	{0x0dc50c, 7},
	{0x0dc544, 2},
	{0x0dc55c, 9},
	{0x0dc584, 7},
	{0x0dc5a4, 2},
	{0x0dc5b8, 2},
	{0x0dc5c4, 2},
	{0x0dc5d8, 2},
	{0x0dc600, 2},
	{0x0dcfbc, 15},
	{0x0dd000, 7},
	{0x0dd020, 6},
	{0x0dd040, 8},
	{0x0dd104, 1},
	{0x0dd10c, 1},
	{0x0dd200, 8},
	{0x0dd244, 2},
	{0x0dd268, 18},
	{0x0dd404, 1},
	{0x0dd440, 40},
	{0x0dd504, 3},
	{0x0dd514, 3},
	{0x0dd524, 3},
	{0x0dd534, 3},
	{0x0dd544, 3},
	{0x0dd554, 3},
	{0x0dd564, 3},
	{0x0dd574, 3},
	{0x0dd584, 3},
	{0x0dd594, 3},
	{0x0dd5a4, 3},
	{0x0dd5b4, 3},
	{0x0dd604, 2},
	{0x0dd640, 16},
	{0x0dd684, 3},
	{0x0dd704, 2},
	{0x0dd740, 18},
	{0x0ddc00, 4},
	{0x0ddc80, 1},
	{0x0ddd00, 6},
	{0x0ddd20, 6},
	{0x0ddd40, 6},
	{0x0ddd80, 1},
	{0x0dde00, 3},
	{0x0dde20, 10},
	{0x0dde50, 6},
	{0x0dde80, 9},
	{0x0ddec0, 16},
	{0x0de000, 135},
	{0x0de300, 2},
	{0x0de30c, 3},
	{0x0de320, 2},
	{0x0de32c, 3},
	{0x0de340, 2},
	{0x0de34c, 3},
	{0x0de360, 2},
	{0x0de36c, 3},
	{0x0de380, 2},
	{0x0de38c, 3},
	{0x0de3a0, 2},
	{0x0de3ac, 3},
	{0x0de3c0, 2},
	{0x0de3cc, 3},
	{0x0de3e0, 2},
	{0x0de3ec, 3},
	{0x0de400, 2},
	{0x0de40c, 3},
	{0x0de420, 2},
	{0x0de42c, 3},
	{0x0de440, 2},
	{0x0de44c, 3},
	{0x0de460, 2},
	{0x0de46c, 3},
	{0x0de480, 2},
	{0x0de48c, 3},
	{0x0de4a0, 2},
	{0x0de4ac, 3},
	{0x0de4c0, 2},
	{0x0de4cc, 3},
	{0x0de4e0, 2},
	{0x0de4ec, 3},
	{0x0de500, 135},
	{0x0de800, 2},
	{0x0de80c, 3},
	{0x0de820, 2},
	{0x0de82c, 3},
	{0x0de840, 2},
	{0x0de84c, 3},
	{0x0de860, 2},
	{0x0de86c, 3},
	{0x0de880, 2},
	{0x0de88c, 3},
	{0x0de8a0, 2},
	{0x0de8ac, 3},
	{0x0de8c0, 2},
	{0x0de8cc, 3},
	{0x0de8e0, 2},
	{0x0de8ec, 3},
	{0x0de900, 2},
	{0x0de90c, 3},
	{0x0de920, 2},
	{0x0de92c, 3},
	{0x0de940, 2},
	{0x0de94c, 3},
	{0x0de960, 2},
	{0x0de96c, 3},
	{0x0de980, 2},
	{0x0de98c, 3},
	{0x0de9a0, 2},
	{0x0de9ac, 3},
	{0x0de9c0, 2},
	{0x0de9cc, 3},
	{0x0de9e0, 2},
	{0x0de9ec, 3},
	{0x0dea00, 135},
	{0x0ded00, 2},
	{0x0ded0c, 3},
	{0x0ded20, 2},
	{0x0ded2c, 3},
	{0x0ded40, 2},
	{0x0ded4c, 3},
	{0x0ded60, 2},
	{0x0ded6c, 3},
	{0x0ded80, 2},
	{0x0ded8c, 3},
	{0x0deda0, 2},
	{0x0dedac, 3},
	{0x0dedc0, 2},
	{0x0dedcc, 3},
	{0x0dede0, 2},
	{0x0dedec, 3},
	{0x0dee00, 2},
	{0x0dee0c, 3},
	{0x0dee20, 2},
	{0x0dee2c, 3},
	{0x0dee40, 2},
	{0x0dee4c, 3},
	{0x0dee60, 2},
	{0x0dee6c, 3},
	{0x0dee80, 2},
	{0x0dee8c, 3},
	{0x0deea0, 2},
	{0x0deeac, 3},
	{0x0deec0, 2},
	{0x0deecc, 3},
	{0x0deee0, 2},
	{0x0deeec, 3},
	{0x0def00, 135},
	{0x0df200, 2},
	{0x0df20c, 3},
	{0x0df220, 2},
	{0x0df22c, 3},
	{0x0df240, 2},
	{0x0df24c, 3},
	{0x0df260, 2},
	{0x0df26c, 3},
	{0x0df280, 2},
	{0x0df28c, 3},
	{0x0df2a0, 2},
	{0x0df2ac, 3},
	{0x0df2c0, 2},
	{0x0df2cc, 3},
	{0x0df2e0, 2},
	{0x0df2ec, 3},
	{0x0df300, 2},
	{0x0df30c, 3},
	{0x0df320, 2},
	{0x0df32c, 3},
	{0x0df340, 2},
	{0x0df34c, 3},
	{0x0df360, 2},
	{0x0df36c, 3},
	{0x0df380, 2},
	{0x0df38c, 3},
	{0x0df3a0, 2},
	{0x0df3ac, 3},
	{0x0df3c0, 2},
	{0x0df3cc, 3},
	{0x0df3e0, 2},
	{0x0df3ec, 3},
	{0x0df400, 135},
	{0x0df700, 2},
	{0x0df70c, 3},
	{0x0df720, 2},
	{0x0df72c, 3},
	{0x0df740, 2},
	{0x0df74c, 3},
	{0x0df760, 2},
	{0x0df76c, 3},
	{0x0df780, 2},
	{0x0df78c, 3},
	{0x0df7a0, 2},
	{0x0df7ac, 3},
	{0x0df7c0, 2},
	{0x0df7cc, 3},
	{0x0df7e0, 2},
	{0x0df7ec, 3},
	{0x0df800, 2},
	{0x0df80c, 3},
	{0x0df820, 2},
	{0x0df82c, 3},
	{0x0df840, 2},
	{0x0df84c, 3},
	{0x0df860, 2},
	{0x0df86c, 3},
	{0x0df880, 2},
	{0x0df88c, 3},
	{0x0df8a0, 2},
	{0x0df8ac, 3},
	{0x0df8c0, 2},
	{0x0df8cc, 3},
	{0x0df8e0, 2},
	{0x0df8ec, 3},
	{0x0df900, 2},
	{0x0e0000, 3},
	{0x0e0010, 4},
	{0x0e0028, 3},
	{0x0e0048, 2},
	{0x0e0058, 2},
	{0x0e0064, 32},
	{0x0e00f0, 1},
	{0x0e00fc, 35},
	{0x0e019c, 15},
	{0x0e01e0, 1},
	{0x0e01e8, 5},
	{0x0e0204, 5},
	{0x0e021c, 1},
	{0x0e0300, 16},
	{0x0e0400, 3},
	{0x0e0410, 4},
	{0x0e0428, 3},
	{0x0e0448, 2},
	{0x0e0458, 2},
	{0x0e0464, 32},
	{0x0e04f0, 1},
	{0x0e04fc, 35},
	{0x0e059c, 15},
	{0x0e05e0, 1},
	{0x0e05e8, 5},
	{0x0e0604, 5},
	{0x0e061c, 1},
	{0x0e0700, 16},
	{0x0e0800, 3},
	{0x0e0810, 4},
	{0x0e0828, 3},
	{0x0e0848, 2},
	{0x0e0858, 2},
	{0x0e0864, 32},
	{0x0e08f0, 1},
	{0x0e08fc, 35},
	{0x0e099c, 15},
	{0x0e09e0, 1},
	{0x0e09e8, 5},
	{0x0e0a04, 5},
	{0x0e0a1c, 1},
	{0x0e0b00, 16},
	{0x0e0c00, 3},
	{0x0e0c10, 4},
	{0x0e0c28, 3},
	{0x0e0c48, 2},
	{0x0e0c58, 2},
	{0x0e0c64, 32},
	{0x0e0cf0, 1},
	{0x0e0cfc, 35},
	{0x0e0d9c, 15},
	{0x0e0de0, 1},
	{0x0e0de8, 5},
	{0x0e0e04, 5},
	{0x0e0e1c, 1},
	{0x0e0f00, 16},
	{0x0e1000, 3},
	{0x0e1010, 4},
	{0x0e1028, 3},
	{0x0e1048, 2},
	{0x0e1058, 2},
	{0x0e1064, 32},
	{0x0e10f0, 1},
	{0x0e10fc, 35},
	{0x0e119c, 15},
	{0x0e11e0, 1},
	{0x0e11e8, 5},
	{0x0e1204, 5},
	{0x0e121c, 1},
	{0x0e1300, 16},
	{0x0e1400, 3},
	{0x0e1410, 4},
	{0x0e1428, 3},
	{0x0e1448, 2},
	{0x0e1458, 2},
	{0x0e1464, 32},
	{0x0e14f0, 1},
	{0x0e14fc, 35},
	{0x0e159c, 15},
	{0x0e15e0, 1},
	{0x0e15e8, 5},
	{0x0e1604, 5},
	{0x0e161c, 1},
	{0x0e1700, 16},
	{0x0e1800, 3},
	{0x0e1810, 4},
	{0x0e1828, 3},
	{0x0e1848, 2},
	{0x0e1858, 2},
	{0x0e1864, 32},
	{0x0e18f0, 1},
	{0x0e18fc, 35},
	{0x0e199c, 15},
	{0x0e19e0, 1},
	{0x0e19e8, 5},
	{0x0e1a04, 5},
	{0x0e1a1c, 1},
	{0x0e1b00, 16},
	{0x0e1c00, 3},
	{0x0e1c10, 4},
	{0x0e1c28, 3},
	{0x0e1c48, 2},
	{0x0e1c58, 2},
	{0x0e1c64, 32},
	{0x0e1cf0, 1},
	{0x0e1cfc, 35},
	{0x0e1d9c, 15},
	{0x0e1de0, 1},
	{0x0e1de8, 5},
	{0x0e1e04, 5},
	{0x0e1e1c, 1},
	{0x0e1f00, 16},
	{0x0e20c0, 9},
	{0x0e20ec, 5},
	{0x0e2108, 3},
	{0x0e2200, 5},
	{0x0e2218, 36},
	{0x0e2300, 6},
	{0x0e2330, 4},
	{0x0e2500, 3},
	{0x0e2510, 12},
	{0x0e26e0, 6},
	{0x0e2700, 6},
	{0x0e2720, 6},
	{0x0e2740, 3},
	{0x0e2780, 6},
	{0x0e27a0, 6},
	{0x0e27c0, 3},
	{0x0e2800, 67},
	{0x0e2a00, 6},
	{0x0e2a20, 6},
	{0x0e2a40, 3},
	{0x0e2a50, 3},
	{0x0e2a60, 1},
	{0x0e2a80, 17},
	{0x0e3020, 10},
	{0x0e3070, 2},
	{0x0e3080, 2},
	{0x0e308c, 1},
	{0x0e3440, 21},
	{0x0e34e4, 13},
	{0x0e3520, 6},
	{0x0e3540, 6},
	{0x0e3560, 6},
	{0x0e3580, 6},
	{0x0e35a0, 6},
	{0x0e35c0, 6},
	{0x0e35e0, 6},
	{0x0e3600, 16},
	{0x0e3804, 3},
	{0x0e3900, 33},
	{0x0e3a00, 6},
	{0x0e3a20, 2},
	{0x0e3a30, 1},
	{0x0e3a40, 8},
	{0x0e3a64, 5},
	{0x0e3c00, 1},
	{0x0e3c28, 1},
	{0x0e3c50, 1},
	{0x0e3c78, 1},
	{0x0e3ca0, 2},
	{0x0e3dac, 1},
	{0x0e3dd8, 4},
	{0x0e3e00, 6},
	{0x0e3e20, 6},
	{0x0e3e40, 3},
	{0x0e4010, 12},
	{0x0e4044, 3},
	{0x0e4084, 2},
	{0x0e40bc, 84},
	{0x0e4240, 18},
	{0x0e45f0, 4},
	{0x0e4604, 1},
	{0x0e4640, 16},
	{0x0e46f0, 4},
	{0x0e4704, 1},
	{0x0e4740, 16},
	{0x0e5000, 8},
	{0x0e6000, 9},
	{0x0e6040, 16},
	{0x0e8000, 9},
	{0x0e8080, 6},
	{0x0e80a0, 3},
	{0x0f0000, 3},
	{0x0f0014, 11},
	{0x0f004c, 3},
	{0x0f0060, 8},
	{0x0f00f0, 3},
	{0x0f0100, 1},
	{0x0f010c, 2},
	{0x0f0118, 1},
	{0x0f0130, 4},
	{0x0f01a8, 1},
	{0x0f01c0, 2},
	{0x0f01d0, 10},
	{0x0f0200, 62},
	{0x0f0404, 9},
	{0x0f0440, 8},
	{0x0f0480, 5},
	{0x0f04b8, 21},
	{0x0f0520, 1},
	{0x0f0528, 1},
	{0x0f0540, 2},
	{0x0f0580, 4},
	{0x0f05a0, 1},
	{0x0f05c0, 8},
	{0x0f0800, 17},
	{0x0f0850, 9},
	{0x0f0880, 9},
	{0x0f0920, 4},
	{0x0f093c, 5},
	{0x0f095c, 5},
	{0x0f097c, 5},
	{0x0f099c, 1},
	{0x0f0a90, 3},
	{0x0f0c00, 128},
	{0x0f0e04, 1},
	{0x0f0e14, 9},
	{0x0f0e3c, 1},
	{0x0f1000, 16},
	{0x0f1080, 10},
	{0x0f10c0, 1},
	{0x0f10e0, 2},
	{0x0f10ec, 1},
	{0x0f10f4, 3},
	{0x0f1400, 6},
	{0x0f1420, 6},
	{0x0f1440, 6},
	{0x0f1460, 6},
	{0x0f1480, 6},
	{0x0f14a0, 6},
	{0x0f14c0, 6},
	{0x0f14e0, 6},
	{0x0f1500, 6},
	{0x0f1520, 6},
	{0x0f1540, 6},
	{0x0f1560, 6},
	{0x0f1580, 6},
	{0x0f15a0, 3},
	{0x0f1800, 3},
	{0x0f1840, 4},
	{0x0f1854, 3},
	{0x0f1864, 3},
	{0x0f1874, 3},
	{0x0f2000, 2},
	{0x0f200c, 3},
	{0x0f2020, 10},
	{0x0f2060, 6},
	{0x0f2080, 2},
	{0x0f208c, 3},
	{0x0f20a0, 10},
	{0x0f20e0, 6},
	{0x0f2100, 2},
	{0x0f210c, 3},
	{0x0f2120, 10},
	{0x0f2160, 6},
	{0x0f2180, 2},
	{0x0f218c, 3},
	{0x0f21a0, 10},
	{0x0f21e0, 6},
	{0x0f2200, 2},
	{0x0f220c, 3},
	{0x0f2220, 10},
	{0x0f2260, 6},
	{0x0f2280, 2},
	{0x0f228c, 3},
	{0x0f22a0, 10},
	{0x0f22e0, 6},
	{0x0f2300, 2},
	{0x0f230c, 3},
	{0x0f2320, 10},
	{0x0f2360, 6},
	{0x0f2380, 2},
	{0x0f238c, 3},
	{0x0f23a0, 10},
	{0x0f23e0, 6},
	{0x0f2400, 2},
	{0x0f240c, 3},
	{0x0f2420, 10},
	{0x0f2460, 6},
	{0x0f2480, 2},
	{0x0f248c, 3},
	{0x0f24a0, 10},
	{0x0f24e0, 6},
	{0x0f2500, 2},
	{0x0f250c, 3},
	{0x0f2520, 10},
	{0x0f2560, 6},
	{0x0f2580, 2},
	{0x0f258c, 3},
	{0x0f25a0, 10},
	{0x0f25e0, 6},
	{0x0f2600, 2},
	{0x0f260c, 3},
	{0x0f2620, 10},
	{0x0f2660, 6},
	{0x0f2680, 2},
	{0x0f268c, 3},
	{0x0f26a0, 10},
	{0x0f26e0, 6},
	{0x0f2700, 2},
	{0x0f270c, 3},
	{0x0f2720, 10},
	{0x0f2760, 6},
	{0x0f2780, 2},
	{0x0f278c, 3},
	{0x0f27a0, 10},
	{0x0f27e0, 6},
	{0x0f2800, 2},
	{0x0f280c, 3},
	{0x0f2820, 10},
	{0x0f2860, 6},
	{0x0f2880, 2},
	{0x0f288c, 3},
	{0x0f28a0, 10},
	{0x0f28e0, 6},
	{0x0f2900, 2},
	{0x0f290c, 3},
	{0x0f2920, 10},
	{0x0f2960, 6},
	{0x0f2980, 2},
	{0x0f298c, 3},
	{0x0f29a0, 10},
	{0x0f29e0, 6},
	{0x0f4000, 7},
	{0x0f4020, 4},
	{0x0f4204, 1},
	{0x0f4280, 35},
	{0x0f4310, 4},
	{0x0f4404, 1},
	{0x0f4480, 34},
	{0x0f4510, 10},
	{0x0f453c, 3},
	{0x0f4800, 7},
	{0x0f4820, 4},
	{0x0f4a04, 1},
	{0x0f4a80, 35},
	{0x0f4b10, 4},
	{0x0f4c04, 1},
	{0x0f4c80, 34},
	{0x0f4d10, 10},
	{0x0f4d3c, 3},
	{0x0f5000, 7},
	{0x0f5020, 4},
	{0x0f5204, 1},
	{0x0f5280, 35},
	{0x0f5310, 4},
	{0x0f5404, 1},
	{0x0f5480, 34},
	{0x0f5510, 10},
	{0x0f553c, 3},
	{0x0f5800, 7},
	{0x0f5820, 4},
	{0x0f5a04, 1},
	{0x0f5a80, 35},
	{0x0f5b10, 4},
	{0x0f5c04, 1},
	{0x0f5c80, 34},
	{0x0f5d10, 10},
	{0x0f5d3c, 3},
	{0x100000, 1},
	{0x100008, 1},
	{0x100010, 2},
	{0x100020, 1},
	{0x100028, 1},
	{0x100030, 2},
	{0x100040, 1},
	{0x100048, 1},
	{0x100050, 2},
	{0x100060, 1},
	{0x100068, 1},
	{0x100070, 2},
	{0x100080, 21},
	{0x1000d8, 2},
	{0x100100, 21},
	{0x100158, 2},
	{0x100180, 21},
	{0x1001d8, 2},
	{0x100200, 21},
	{0x100258, 2},
	{0x100284, 1},
	{0x1003b0, 5},
	{0x100400, 13},
	{0x100440, 13},
	{0x100480, 13},
	{0x1004c0, 13},
	{0x100500, 68},
	{0x100618, 1},
	{0x100804, 1},
	{0x10080c, 4},
	{0x100820, 9},
	{0x1008a0, 24},
	{0x100920, 24},
	{0x100a00, 48},
	{0x100adc, 68},
	{0x100d00, 1},
	{0x100d08, 2},
	{0x100d80, 4},
	{0x100e00, 4},
	{0x100e20, 1},
	{0x100e28, 7},
	{0x100e48, 7},
	{0x100e68, 7},
	{0x100e88, 6},
	{0x100ebc, 9},
	{0x100f00, 6},
	{0x100f1c, 10},
	{0x100f70, 8},
	{0x100f94, 4},
	{0x101404, 1},
	{0x10141c, 26},
	{0x101504, 1},
	{0x10151c, 30},
	{0x101600, 1},
	{0x101628, 6},
	{0x101648, 6},
	{0x101680, 16},
	{0x1016e0, 16},
	{0x101780, 2},
	{0x101790, 16},
	{0x101a00, 14},
	{0x101a3c, 4},
	{0x101a50, 2},
	{0x101a60, 2},
	{0x101a70, 2},
	{0x101a80, 2},
	{0x101a90, 1},
	{0x101a9c, 11},
	{0x101b0c, 5},
	{0x101c00, 34},
	{0x101d00, 3},
	{0x102000, 1},
	{0x102028, 1},
	{0x102050, 1},
	{0x102078, 1},
	{0x1020a0, 5},
	{0x1021ac, 1},
	{0x1021d8, 4},
	{0x102200, 6},
	{0x102220, 6},
	{0x102240, 3},
	{0x102560, 1},
	{0x102584, 10},
	{0x1025b0, 1},
	{0x1025fc, 1},
	{0x102604, 1},
	{0x1026ec, 69},
	{0x103000, 32},
	{0x103084, 5},
	{0x1030f8, 3},
	{0x103108, 3},
	{0x103118, 7},
	{0x103144, 1},
	{0x103160, 10},
	{0x103200, 32},
	{0x103284, 5},
	{0x1032f8, 3},
	{0x103308, 3},
	{0x103318, 7},
	{0x103344, 1},
	{0x103360, 10},
	{0x103400, 32},
	{0x103484, 5},
	{0x1034f8, 3},
	{0x103508, 3},
	{0x103518, 7},
	{0x103544, 1},
	{0x103560, 10},
	{0x103600, 32},
	{0x103684, 5},
	{0x1036f8, 3},
	{0x103708, 3},
	{0x103718, 7},
	{0x103744, 1},
	{0x103760, 10},
	{0x103800, 1},
	{0x10380c, 1},
	{0x10397c, 97},
	{0x104000, 3},
	{0x104020, 3},
	{0x104040, 3},
	{0x104060, 3},
	{0x104084, 1},
	{0x104090, 4},
	{0x1040a4, 1},
	{0x1040b0, 4},
	{0x1040c4, 1},
	{0x1040d0, 4},
	{0x1040e4, 1},
	{0x1040f0, 21},
	{0x104148, 18},
	{0x1041f0, 4},
	{0x104204, 3},
	{0x104308, 9},
	{0x104330, 2},
	{0x104340, 16},
	{0x1043b4, 4},
	{0x1043c8, 4},
	{0x1043dc, 4},
	{0x1043f0, 4},
	{0x104404, 1},
	{0x104470, 36},
	{0x104504, 1},
	{0x104570, 36},
	{0x104604, 1},
	{0x104670, 36},
	{0x104704, 1},
	{0x104770, 50},
	{0x104840, 2},
	{0x10484c, 1},
	{0x104900, 1},
	{0x104908, 1},
	{0x104984, 1},
	{0x1049a0, 24},
	{0x104a04, 7},
	{0x104a24, 7},
	{0x104a44, 7},
	{0x104a64, 7},
	{0x104a84, 7},
	{0x104aa4, 7},
	{0x104ac4, 7},
	{0x104ae4, 16},
	{0x104b40, 16},
	{0x104c00, 6},
	{0x104c20, 6},
	{0x104c40, 6},
	{0x104c60, 6},
	{0x104c80, 6},
	{0x104ca0, 6},
	{0x104cc0, 6},
	{0x104ce0, 6},
	{0x104d00, 3},
	{0x104d20, 6},
	{0x104d40, 6},
	{0x105000, 448},
	{0x105704, 3},
	{0x105734, 1},
	{0x106000, 62},
	{0x106100, 35},
	{0x1061c0, 6},
	{0x1061e0, 6},
	{0x106200, 6},
	{0x106220, 6},
	{0x106240, 6},
	{0x106260, 6},
	{0x106280, 6},
	{0x1062a0, 6},
	{0x1062c0, 6},
	{0x1062e0, 6},
	{0x106300, 6},
	{0x106320, 6},
	{0x106340, 6},
	{0x106360, 6},
	{0x106380, 6},
	{0x1063a0, 6},
	{0x107010, 1},
	{0x110000, 7},
	{0x110020, 7},
	{0x110040, 5},
	{0x110060, 6},
	{0x110080, 5},
	{0x110098, 1},
	{0x1100a0, 4},
	{0x1100b8, 8},
	{0x1100e0, 6},
	{0x110200, 7},
	{0x110220, 7},
	{0x110240, 5},
	{0x110260, 6},
	{0x110280, 5},
	{0x110298, 1},
	{0x1102a0, 4},
	{0x1102b8, 8},
	{0x1102e0, 6},
	{0x110400, 7},
	{0x110420, 7},
	{0x110440, 5},
	{0x110460, 6},
	{0x110480, 5},
	{0x110498, 1},
	{0x1104a0, 4},
	{0x1104b8, 8},
	{0x1104e0, 6},
	{0x110600, 7},
	{0x110620, 7},
	{0x110640, 5},
	{0x110660, 6},
	{0x110680, 5},
	{0x110698, 1},
	{0x1106a0, 4},
	{0x1106b8, 8},
	{0x1106e0, 6},
	{0x110800, 21},
	{0x110880, 15},
	{0x1108c0, 3},
	{0x1108d0, 2},
	{0x110900, 1},
	{0x111000, 1},
	{0x111028, 1},
	{0x111050, 1},
	{0x111078, 1},
	{0x1110a0, 5},
	{0x1111ac, 1},
	{0x1111d8, 4},
	{0x111200, 6},
	{0x111220, 6},
	{0x111240, 3},
	{0x111400, 6},
	{0x111420, 3},
	{0x111480, 9},
	{0x1114c0, 25},
	{0x111540, 25},
	{0x1115c0, 25},
	{0x111640, 20},
	{0x118000, 29},
	{0x118078, 4},
	{0x118090, 2},
	{0x1180a0, 7},
	{0x1180c0, 11},
	{0x118100, 14},
	{0x118140, 14},
	{0x118180, 61},
	{0x118278, 4},
	{0x118290, 2},
	{0x1182a0, 7},
	{0x1182c0, 11},
	{0x118300, 14},
	{0x118340, 14},
	{0x118380, 61},
	{0x118478, 4},
	{0x118490, 2},
	{0x1184a0, 7},
	{0x1184c0, 11},
	{0x118500, 14},
	{0x118540, 14},
	{0x118580, 61},
	{0x118678, 4},
	{0x118690, 2},
	{0x1186a0, 7},
	{0x1186c0, 11},
	{0x118700, 14},
	{0x118740, 14},
	{0x118780, 69},
	{0x118c00, 6},
	{0x118c40, 14},
	{0x118c80, 9},
	{0x118d00, 9},
	{0x118d2c, 1},
	{0x118d40, 3},
	{0x118d60, 1},
	{0x118d80, 3},
	{0x118e00, 2},
	{0x118e0c, 1},
	{0x118e14, 5},
	{0x118e2c, 1},
	{0x118e34, 5},
	{0x118e4c, 1},
	{0x118e54, 5},
	{0x118e6c, 1},
	{0x118e74, 5},
	{0x118e8c, 1},
	{0x118e94, 5},
	{0x118eac, 1},
	{0x118eb4, 3},
	{0x119000, 29},
	{0x119078, 4},
	{0x119090, 2},
	{0x1190a0, 7},
	{0x1190c0, 11},
	{0x119100, 14},
	{0x119140, 14},
	{0x119180, 61},
	{0x119278, 4},
	{0x119290, 2},
	{0x1192a0, 7},
	{0x1192c0, 11},
	{0x119300, 14},
	{0x119340, 14},
	{0x119380, 61},
	{0x119478, 4},
	{0x119490, 2},
	{0x1194a0, 7},
	{0x1194c0, 11},
	{0x119500, 14},
	{0x119540, 14},
	{0x119580, 61},
	{0x119678, 4},
	{0x119690, 2},
	{0x1196a0, 7},
	{0x1196c0, 11},
	{0x119700, 14},
	{0x119740, 14},
	{0x119780, 69},
	{0x119c00, 6},
	{0x119c40, 14},
	{0x119c80, 9},
	{0x119d00, 9},
	{0x119d2c, 1},
	{0x119d40, 3},
	{0x119d60, 1},
	{0x119d80, 3},
	{0x119e00, 2},
	{0x119e0c, 1},
	{0x119e14, 5},
	{0x119e2c, 1},
	{0x119e34, 5},
	{0x119e4c, 1},
	{0x119e54, 5},
	{0x119e6c, 1},
	{0x119e74, 5},
	{0x119e8c, 1},
	{0x119e94, 5},
	{0x119eac, 1},
	{0x119eb4, 3},
	{0x11c000, 19},
	{0x11c050, 10},
	{0x11c080, 19},
	{0x11c0d0, 10},
	{0x11c100, 19},
	{0x11c150, 10},
	{0x11c180, 19},
	{0x11c1d0, 10},
	{0x11c200, 19},
	{0x11c250, 10},
	{0x11c280, 19},
	{0x11c2d0, 10},
	{0x11c300, 19},
	{0x11c350, 10},
	{0x11c380, 19},
	{0x11c3d0, 10},
	{0x11c400, 19},
	{0x11c460, 6},
	{0x11c484, 1},
	{0x11c494, 8},
	{0x11c4b8, 9},
	{0x11c4e0, 4},
	{0x11c500, 3},
	{0x11d000, 7},
	{0x11d020, 15},
	{0x11d060, 15},
	{0x11d0a0, 15},
	{0x11d0e0, 15},
	{0x11d120, 15},
	{0x11d160, 15},
	{0x11d1a0, 15},
	{0x11d1e0, 8},
	{0x11d400, 9},
	{0x11d428, 3},
	{0x11d440, 5},
	{0x11d480, 9},
	{0x11d4a8, 3},
	{0x11d4c0, 5},
	{0x11d500, 9},
	{0x11d528, 3},
	{0x11d540, 5},
	{0x11d580, 9},
	{0x11d5a8, 3},
	{0x11d5c0, 5},
	{0x11d600, 6},
	{0x11d620, 6},
	{0x11d640, 6},
	{0x11d660, 6},
	{0x11d680, 6},
	{0x11d6a0, 6},
	{0x11d6c0, 6},
	{0x11d6e0, 6},
	{0x11d708, 2},
	{0x11d718, 4},
	{0x11d734, 1},
	{0x11d73c, 4},
	{0x11d750, 4},
	{0x11d764, 1},
	{0x11d76c, 2},
	{0x11d800, 8},
	{0x11d840, 8},
	{0x11d880, 8},
	{0x11d8c0, 8},
	{0x11d900, 8},
	{0x11d940, 8},
	{0x11d988, 4},
	{0x11da00, 3},
	{0x11da18, 2},
	{0x11da24, 16},
	{0x11da80, 3},
	{0x11da98, 2},
	{0x11daa4, 16},
	{0x11db00, 3},
	{0x11db18, 2},
	{0x11db24, 16},
	{0x11db80, 3},
	{0x11db98, 2},
	{0x11dba4, 16},
	{0x11dc00, 8},
	{0x11dc40, 1},
	{0x11e000, 72},
	{0x11e200, 72},
	{0x11e400, 72},
	{0x11e600, 72},
	{0x11e800, 6},
	{0x11e820, 6},
	{0x11e840, 6},
	{0x11e860, 6},
	{0x11e880, 6},
	{0x11e8a0, 6},
	{0x11e8c0, 6},
	{0x11e8e0, 6},
	{0x11e900, 6},
	{0x11e920, 6},
	{0x11e940, 6},
	{0x11e9fc, 7},
	{0x11ea20, 6},
	{0x11ea40, 6},
	{0x11ea60, 6},
	{0x11ea80, 6},
	{0x11eaa0, 6},
	{0x11eac0, 6},
	{0x11eae0, 6},
	{0x11eb00, 6},
	{0x11eb20, 6},
	{0x11eb40, 6},
	{0x11ebfc, 7},
	{0x11ec20, 6},
	{0x11ec40, 6},
	{0x11ec60, 6},
	{0x11ec80, 6},
	{0x11eca0, 6},
	{0x11ecc0, 6},
	{0x11ece0, 6},
	{0x11ed00, 6},
	{0x11ed20, 6},
	{0x11ed40, 6},
	{0x11edfc, 7},
	{0x11ee20, 6},
	{0x11ee40, 6},
	{0x11ee60, 6},
	{0x11ee80, 6},
	{0x11eea0, 6},
	{0x11eec0, 6},
	{0x11eee0, 6},
	{0x11ef00, 6},
	{0x11ef20, 6},
	{0x11ef40, 6},
	{0x11effc, 15},
	{0x11f040, 2},
	{0x11f080, 14},
	{0x11f0c0, 2},
	{0x11f100, 14},
	{0x11f140, 2},
	{0x11f180, 14},
	{0x11f1c0, 2},
	{0x11f400, 17},
	{0x11f448, 5},
	{0x11f460, 3},
	{0x11f470, 3},
	{0x11f480, 16},
	{0x11f500, 17},
	{0x11f548, 5},
	{0x11f560, 3},
	{0x11f570, 3},
	{0x11f580, 16},
	{0x11f600, 17},
	{0x11f648, 5},
	{0x11f660, 3},
	{0x11f670, 3},
	{0x11f680, 16},
	{0x11f700, 17},
	{0x11f748, 5},
	{0x11f760, 3},
	{0x11f770, 3},
	{0x11f780, 16},
	{0x11f800, 8},
	{0x11f824, 15},
	{0x11f864, 15},
	{0x11f8a4, 15},
	{0x11f8e4, 30},
	{0x11f980, 10},
	{0x11f9ac, 1},
	{0x11f9b4, 5},
	{0x11f9cc, 1},
	{0x11f9d4, 5},
	{0x11f9ec, 1},
	{0x11f9f4, 13},
	{0x11fa80, 4},
	{0x11fa94, 2},
	{0x11faa0, 5},
	{0x11fac0, 5},
	{0x11fae0, 4},
	{0x11fc04, 8},
	{0x11fc40, 13},
	{0x11fd00, 6},
	{0x11fd20, 3},
	{0x11fd30, 3},
	{0x11fd44, 1},
	{0x11fd4c, 1},
	{0x11fd54, 1},
	{0x11fd5c, 8},
	{0x11fd80, 11},
	{0x11fdc0, 3} };

static const unsigned long
mlx5_mst_dump_regs_mt4115[MLX5_NUM_MST_OFFSETS_MT4115][2] = {{0x000000, 16388},
	{0x010084, 1},
	{0x01008c, 2},
	{0x010180, 6},
	{0x01019c, 9},
	{0x010304, 1},
	{0x0103bc, 52},
	{0x010500, 35},
	{0x010604, 1},
	{0x01060c, 1},
	{0x010624, 1},
	{0x01062c, 1},
	{0x010644, 1},
	{0x01064c, 3},
	{0x010800, 1},
	{0x010814, 3},
	{0x010844, 1},
	{0x01084c, 4},
	{0x011000, 32},
	{0x011200, 32},
	{0x011400, 32},
	{0x011600, 32},
	{0x011800, 32},
	{0x011a00, 5},
	{0x011a24, 1},
	{0x011a2c, 1},
	{0x011a34, 3},
	{0x011a44, 1},
	{0x011a4c, 2},
	{0x012000, 20},
	{0x012088, 22},
	{0x012100, 18},
	{0x012200, 36},
	{0x012300, 3},
	{0x012310, 1},
	{0x012320, 1},
	{0x012330, 1},
	{0x012340, 1},
	{0x012350, 1},
	{0x012360, 1},
	{0x012370, 1},
	{0x012400, 166},
	{0x0126a0, 1},
	{0x012700, 32},
	{0x012800, 1},
	{0x012808, 1},
	{0x012810, 1},
	{0x012818, 1},
	{0x012820, 1},
	{0x012828, 1},
	{0x012830, 1},
	{0x012838, 1},
	{0x012850, 1},
	{0x012858, 1},
	{0x012860, 1},
	{0x012868, 1},
	{0x012870, 1},
	{0x012878, 1},
	{0x012880, 1},
	{0x012888, 1},
	{0x012890, 5},
	{0x012900, 10},
	{0x012940, 6},
	{0x012960, 6},
	{0x012980, 5},
	{0x012a00, 6},
	{0x012a40, 5},
	{0x012a80, 9},
	{0x012ac0, 19},
	{0x012b10, 2},
	{0x014000, 1},
	{0x014010, 10},
	{0x014104, 1},
	{0x01410c, 4},
	{0x014200, 1},
	{0x014210, 4},
	{0x014300, 4},
	{0x014400, 3},
	{0x014410, 1},
	{0x014444, 1},
	{0x01444c, 4},
	{0x014480, 2},
	{0x014500, 35},
	{0x014590, 1},
	{0x014600, 32},
	{0x014704, 1},
	{0x01470c, 1},
	{0x014804, 2},
	{0x014814, 3},
	{0x014824, 2},
	{0x014834, 3},
	{0x014844, 2},
	{0x014854, 5},
	{0x014a00, 4},
	{0x014a14, 2},
	{0x015004, 1},
	{0x01500c, 4},
	{0x015204, 1},
	{0x015214, 4},
	{0x015228, 1},
	{0x015244, 1},
	{0x015250, 4},
	{0x015274, 1},
	{0x01527c, 1},
	{0x0152a4, 1},
	{0x015300, 4},
	{0x015314, 1},
	{0x01531c, 3},
	{0x015404, 2},
	{0x015430, 20},
	{0x015484, 2},
	{0x015498, 5},
	{0x0154bc, 1},
	{0x0154c8, 1},
	{0x0154d4, 2},
	{0x01550c, 3},
	{0x015524, 3},
	{0x015608, 3},
	{0x015618, 4},
	{0x015680, 16},
	{0x015704, 2},
	{0x015730, 20},
	{0x015784, 3},
	{0x015794, 3},
	{0x0157c0, 6},
	{0x015800, 52},
	{0x0158d4, 1},
	{0x0158dc, 1},
	{0x0158e4, 3},
	{0x0158f4, 1},
	{0x0158fc, 52},
	{0x015b04, 2},
	{0x015b14, 3},
	{0x015b24, 2},
	{0x015b38, 6},
	{0x015b60, 5},
	{0x015c04, 1},
	{0x015cb8, 18},
	{0x015d98, 1},
	{0x015e00, 1},
	{0x015e08, 1},
	{0x015e10, 1},
	{0x016000, 32},
	{0x016100, 7},
	{0x016120, 2},
	{0x016144, 3},
	{0x016204, 1},
	{0x01620c, 1},
	{0x016224, 1},
	{0x01622c, 1},
	{0x016234, 1},
	{0x01623c, 1},
	{0x016244, 1},
	{0x01624c, 1},
	{0x016254, 1},
	{0x01625c, 2},
	{0x016270, 1},
	{0x016280, 1},
	{0x016290, 1},
	{0x0162a0, 1},
	{0x0162b0, 1},
	{0x0162c0, 1},
	{0x0162d0, 1},
	{0x0162e0, 1},
	{0x0162f0, 1},
	{0x016300, 1},
	{0x016310, 1},
	{0x016320, 1},
	{0x016330, 1},
	{0x016340, 32},
	{0x016400, 19},
	{0x016454, 7},
	{0x016484, 1},
	{0x01648c, 1},
	{0x016800, 9},
	{0x016840, 16},
	{0x017800, 1},
	{0x017828, 1},
	{0x017850, 1},
	{0x017878, 1},
	{0x0178a0, 12},
	{0x0179ac, 1},
	{0x0179d8, 4},
	{0x017a00, 6},
	{0x017a20, 6},
	{0x017a40, 3},
	{0x017c00, 6},
	{0x017c20, 6},
	{0x017c40, 6},
	{0x017c60, 6},
	{0x017c80, 6},
	{0x017ca0, 6},
	{0x017cc0, 6},
	{0x017ce0, 6},
	{0x017d00, 3},
	{0x017d10, 5},
	{0x018400, 11},
	{0x018430, 2},
	{0x018440, 4},
	{0x018604, 1},
	{0x018618, 2},
	{0x018640, 4},
	{0x018660, 2},
	{0x018800, 1},
	{0x018810, 4},
	{0x018844, 1},
	{0x01884c, 4},
	{0x018880, 2},
	{0x01a400, 12},
	{0x01a444, 1},
	{0x01a44c, 1},
	{0x01a800, 1},
	{0x01a814, 3},
	{0x01a844, 1},
	{0x01a84c, 4},
	{0x01c400, 7},
	{0x01c500, 8},
	{0x01c544, 1},
	{0x01c554, 3},
	{0x01c564, 3},
	{0x01c574, 3},
	{0x01c604, 1},
	{0x01c60c, 3},
	{0x01c800, 1},
	{0x01c814, 3},
	{0x01c844, 1},
	{0x01c84c, 4},
	{0x01f000, 1},
	{0x01f400, 1},
	{0x01f408, 4},
	{0x01f41c, 3},
	{0x01f500, 2},
	{0x01f800, 1},
	{0x01f814, 3},
	{0x01f844, 1},
	{0x01f84c, 4},
	{0x01f880, 3},
	{0x020004, 1},
	{0x02000c, 4},
	{0x020080, 9},
	{0x020800, 16},
	{0x020900, 192},
	{0x021004, 1},
	{0x02100c, 4},
	{0x021400, 5},
	{0x021418, 5},
	{0x021480, 1},
	{0x021704, 2},
	{0x02173c, 17},
	{0x021804, 1},
	{0x02180c, 4},
	{0x021840, 2},
	{0x022280, 12},
	{0x022408, 6},
	{0x022444, 1},
	{0x022454, 3},
	{0x022504, 1},
	{0x02250c, 4},
	{0x022624, 1},
	{0x02262c, 3},
	{0x02263c, 1},
	{0x022804, 1},
	{0x02280c, 4},
	{0x022904, 4},
	{0x022924, 4},
	{0x024000, 36},
	{0x0240c0, 21},
	{0x024120, 11},
	{0x024200, 5},
	{0x024220, 1},
	{0x024230, 8},
	{0x024258, 1},
	{0x024260, 1},
	{0x024270, 9},
	{0x0242a0, 1},
	{0x0242b0, 4},
	{0x0242c8, 2},
	{0x024300, 5},
	{0x024318, 2},
	{0x02439c, 25},
	{0x024424, 4},
	{0x024464, 13},
	{0x0244a4, 1},
	{0x0244ac, 3},
	{0x0244c0, 2},
	{0x0244d0, 3},
	{0x0244e0, 3},
	{0x0244f0, 3},
	{0x024500, 3},
	{0x024510, 3},
	{0x024520, 3},
	{0x024530, 3},
	{0x024884, 11},
	{0x0248b4, 3},
	{0x0248c4, 1},
	{0x0248cc, 1},
	{0x0248d4, 1},
	{0x0248dc, 1},
	{0x0248f0, 2},
	{0x024908, 6},
	{0x024928, 6},
	{0x024968, 6},
	{0x024984, 3},
	{0x024994, 1},
	{0x02499c, 6},
	{0x0249b8, 7},
	{0x024a08, 6},
	{0x024a28, 6},
	{0x024a68, 6},
	{0x024a84, 1},
	{0x024a8c, 1},
	{0x024a94, 1},
	{0x024a9c, 13},
	{0x024ae0, 6},
	{0x024b00, 7},
	{0x024b20, 6},
	{0x024b40, 8},
	{0x024c00, 2},
	{0x024c24, 3},
	{0x024c34, 3},
	{0x025004, 6},
	{0x025800, 37},
	{0x025904, 1},
	{0x02590c, 1},
	{0x026000, 99},
	{0x026200, 1},
	{0x026800, 7},
	{0x026824, 6},
	{0x026840, 2},
	{0x026864, 1},
	{0x02686c, 1},
	{0x026874, 3},
	{0x026884, 1},
	{0x0268a4, 7},
	{0x026904, 1},
	{0x02690c, 4},
	{0x026940, 1},
	{0x026980, 33},
	{0x026a0c, 7},
	{0x026a30, 1},
	{0x026a44, 4},
	{0x026a60, 1},
	{0x026a70, 1},
	{0x026b00, 9},
	{0x026b44, 2},
	{0x026b68, 6},
	{0x026b84, 2},
	{0x026ba8, 14},
	{0x026c00, 16},
	{0x026c44, 1},
	{0x026c4c, 1},
	{0x026c84, 1},
	{0x026c8c, 4},
	{0x026cc4, 1},
	{0x026ccc, 4},
	{0x026d00, 2},
	{0x028800, 2},
	{0x028844, 1},
	{0x02884c, 4},
	{0x029004, 7},
	{0x029b30, 2},
	{0x029b50, 4},
	{0x02a004, 1},
	{0x02a00c, 4},
	{0x02a040, 6},
	{0x02a200, 1},
	{0x02a210, 11},
	{0x02a240, 8},
	{0x02a484, 1},
	{0x02a4c0, 16},
	{0x02a780, 1},
	{0x02a7a0, 4},
	{0x02a7c0, 1},
	{0x02a900, 1},
	{0x02aa04, 1},
	{0x02aa0c, 4},
	{0x02ab00, 40},
	{0x02aba4, 1},
	{0x02abac, 1},
	{0x02abb4, 1},
	{0x02abbc, 1},
	{0x02abc4, 1},
	{0x02abcc, 1},
	{0x02abe0, 4},
	{0x02abf4, 2},
	{0x02ac00, 2},
	{0x02ac10, 3},
	{0x02ac44, 3},
	{0x02ad00, 1},
	{0x02ad08, 1},
	{0x02ad10, 1},
	{0x02ad18, 1},
	{0x02ad20, 1},
	{0x02ad28, 1},
	{0x02ad30, 1},
	{0x02ad38, 1},
	{0x02ad40, 1},
	{0x02ad48, 1},
	{0x02ad50, 1},
	{0x02ad58, 1},
	{0x02ad60, 1},
	{0x02ad68, 1},
	{0x02ad70, 1},
	{0x02ad78, 1},
	{0x02ad80, 1},
	{0x02ad88, 1},
	{0x02ad90, 1},
	{0x02ad98, 1},
	{0x02ada0, 1},
	{0x02ada8, 1},
	{0x02adb0, 1},
	{0x02adb8, 1},
	{0x02adc0, 1},
	{0x02adc8, 1},
	{0x02add0, 1},
	{0x02add8, 1},
	{0x02ade0, 1},
	{0x02ade8, 1},
	{0x02adf0, 1},
	{0x02adf8, 1},
	{0x02ae00, 6},
	{0x02ae20, 5},
	{0x02ae40, 4},
	{0x02c000, 6},
	{0x02c100, 5},
	{0x02c204, 1},
	{0x02c214, 3},
	{0x02c224, 1},
	{0x02c22c, 4},
	{0x02c244, 2},
	{0x02c250, 5},
	{0x02c400, 2},
	{0x02c428, 2},
	{0x02c450, 2},
	{0x02c478, 2},
	{0x02c4a0, 24},
	{0x02c5ac, 1},
	{0x02c5d8, 4},
	{0x02c600, 6},
	{0x02c620, 6},
	{0x02c640, 6},
	{0x02c660, 6},
	{0x02c680, 3},
	{0x02c800, 3},
	{0x02c820, 6},
	{0x02c840, 6},
	{0x02c860, 6},
	{0x02c880, 6},
	{0x02c8a0, 6},
	{0x02c8c0, 6},
	{0x02c8e0, 6},
	{0x02c900, 6},
	{0x02c920, 6},
	{0x02c940, 6},
	{0x02c960, 6},
	{0x02c980, 6},
	{0x02c9a0, 6},
	{0x02c9c0, 6},
	{0x02c9e0, 6},
	{0x02ca00, 6},
	{0x02ca20, 6},
	{0x02ca40, 6},
	{0x02ca60, 6},
	{0x02ca80, 6},
	{0x02caa0, 6},
	{0x02cac0, 6},
	{0x02cae0, 6},
	{0x02cb00, 6},
	{0x02cb20, 6},
	{0x02cc48, 4},
	{0x02cd00, 9},
	{0x02cd40, 17},
	{0x030004, 2},
	{0x030034, 19},
	{0x030084, 2},
	{0x0300bc, 17},
	{0x030104, 2},
	{0x030138, 27},
	{0x030300, 7},
	{0x030340, 2},
	{0x03034c, 2},
	{0x030384, 1},
	{0x0303c0, 16},
	{0x030404, 1},
	{0x03040c, 4},
	{0x030804, 1},
	{0x03080c, 4},
	{0x030c04, 1},
	{0x030c0c, 4},
	{0x030c40, 4},
	{0x031000, 11},
	{0x031100, 11},
	{0x031200, 17},
	{0x031280, 6},
	{0x031304, 1},
	{0x03130c, 5},
	{0x031400, 6},
	{0x031420, 1},
	{0x031444, 2},
	{0x031454, 3},
	{0x031464, 2},
	{0x031474, 11},
	{0x031500, 7},
	{0x031520, 6},
	{0x031540, 8},
	{0x031600, 13},
	{0x031640, 6},
	{0x031700, 2},
	{0x034200, 24},
	{0x034280, 10},
	{0x0342ac, 2},
	{0x0342c0, 6},
	{0x0342f0, 39},
	{0x034600, 24},
	{0x034680, 10},
	{0x0346ac, 2},
	{0x0346c0, 6},
	{0x0346f0, 39},
	{0x034c00, 6},
	{0x034c20, 4},
	{0x034c40, 9},
	{0x034c80, 9},
	{0x034e04, 2},
	{0x034e14, 3},
	{0x034e44, 1},
	{0x034e4c, 4},
	{0x034e80, 6},
	{0x034f04, 1},
	{0x034f18, 11},
	{0x034f80, 2},
	{0x035000, 2},
	{0x035010, 3},
	{0x035044, 3},
	{0x035100, 66},
	{0x035210, 3},
	{0x035244, 3},
	{0x035300, 64},
	{0x035404, 1},
	{0x03540c, 8},
	{0x037000, 6},
	{0x03702c, 7},
	{0x037080, 10},
	{0x0370ac, 4},
	{0x037100, 4},
	{0x037200, 2},
	{0x037210, 3},
	{0x037244, 3},
	{0x037300, 70},
	{0x03742c, 7},
	{0x037480, 10},
	{0x0374ac, 4},
	{0x037500, 4},
	{0x037600, 2},
	{0x037610, 3},
	{0x037644, 3},
	{0x037700, 69},
	{0x037818, 4},
	{0x038000, 3},
	{0x038104, 3},
	{0x03813c, 2},
	{0x038150, 3},
	{0x038400, 2},
	{0x038428, 2},
	{0x038450, 2},
	{0x038478, 2},
	{0x0384a0, 22},
	{0x0385ac, 1},
	{0x0385d8, 4},
	{0x038600, 6},
	{0x038620, 6},
	{0x038640, 6},
	{0x038660, 6},
	{0x038680, 3},
	{0x038800, 3},
	{0x038820, 6},
	{0x038840, 6},
	{0x038860, 6},
	{0x038880, 6},
	{0x0388a0, 6},
	{0x0388c0, 6},
	{0x038900, 28},
	{0x038978, 1},
	{0x038a40, 25},
	{0x038ac0, 16},
	{0x039000, 35},
	{0x039090, 3},
	{0x039100, 35},
	{0x039190, 3},
	{0x039200, 35},
	{0x039290, 3},
	{0x039300, 35},
	{0x039390, 3},
	{0x039400, 35},
	{0x039490, 3},
	{0x039500, 1},
	{0x039800, 3},
	{0x039884, 1},
	{0x0398c0, 16},
	{0x039904, 2},
	{0x039934, 20},
	{0x039a04, 2},
	{0x039a10, 4},
	{0x039a24, 2},
	{0x039a30, 4},
	{0x039a44, 2},
	{0x039a50, 4},
	{0x039a64, 2},
	{0x039a70, 8},
	{0x039c00, 7},
	{0x039c20, 6},
	{0x039c40, 8},
	{0x039d00, 11},
	{0x039d40, 11},
	{0x039d84, 1},
	{0x039dc0, 26},
	{0x039e30, 2},
	{0x039e44, 3},
	{0x039e54, 1},
	{0x039e5c, 2},
	{0x039e80, 2},
	{0x039e90, 3},
	{0x039ea4, 1},
	{0x039eac, 3},
	{0x039ec0, 3},
	{0x039f00, 9},
	{0x039f40, 22},
	{0x039fa0, 6},
	{0x039fc0, 8},
	{0x03c000, 3},
	{0x03c010, 3},
	{0x03c020, 3},
	{0x03c040, 9},
	{0x03c068, 6},
	{0x03c090, 2},
	{0x03c0a0, 3},
	{0x03c0c0, 12},
	{0x03c0f4, 1},
	{0x03c100, 2},
	{0x03c110, 3},
	{0x03c120, 1},
	{0x03c130, 11},
	{0x03c160, 2},
	{0x03c180, 4},
	{0x03c194, 3},
	{0x03c1a4, 2},
	{0x03c1b0, 4},
	{0x03c2e0, 5},
	{0x03c2f8, 2},
	{0x03c30c, 13},
	{0x03c34c, 77},
	{0x03c48c, 18},
	{0x03c500, 32},
	{0x03c800, 16},
	{0x03c84c, 18},
	{0x03c8e0, 2},
	{0x03c8ec, 3},
	{0x03c900, 1},
	{0x03c910, 5},
	{0x03c930, 5},
	{0x03c950, 5},
	{0x03c970, 5},
	{0x03c9a4, 3},
	{0x03ca78, 34},
	{0x03cb3c, 18},
	{0x03cb94, 3},
	{0x03cba4, 3},
	{0x03cbf0, 1},
	{0x03cbf8, 10},
	{0x03cc30, 1},
	{0x03cc44, 4},
	{0x03cc60, 1},
	{0x03cc80, 1},
	{0x03cc90, 1},
	{0x03d004, 6},
	{0x03e004, 1},
	{0x03e00c, 4},
	{0x03e404, 1},
	{0x03e40c, 4},
	{0x03e604, 1},
	{0x03e60c, 4},
	{0x03e800, 1},
	{0x03f034, 19},
	{0x03f084, 2},
	{0x03f0c0, 16},
	{0x03f200, 1},
	{0x03f210, 1},
	{0x03f300, 6},
	{0x03f320, 6},
	{0x03f380, 9},
	{0x03f3c0, 16},
	{0x050000, 1},
	{0x050008, 2},
	{0x050044, 8},
	{0x050104, 1},
	{0x050178, 34},
	{0x050204, 1},
	{0x05020c, 1},
	{0x050214, 1},
	{0x050224, 1},
	{0x05022c, 1},
	{0x050234, 1},
	{0x05023c, 1},
	{0x050244, 1},
	{0x05024c, 1},
	{0x050254, 1},
	{0x050264, 3},
	{0x050280, 2},
	{0x0502b0, 2},
	{0x0502c4, 1},
	{0x0502cc, 1},
	{0x0502d4, 1},
	{0x0502dc, 1},
	{0x0502e4, 1},
	{0x0502ec, 1},
	{0x0502f4, 1},
	{0x0502fc, 1},
	{0x050304, 1},
	{0x05030c, 1},
	{0x050314, 3},
	{0x050324, 3},
	{0x050334, 6},
	{0x050380, 32},
	{0x050404, 1},
	{0x050438, 18},
	{0x050500, 2},
	{0x050544, 1},
	{0x05054c, 4},
	{0x050584, 2},
	{0x050598, 2},
	{0x0505a4, 1},
	{0x0505b0, 4},
	{0x0505c4, 1},
	{0x0505cc, 1},
	{0x0505d4, 1},
	{0x0505e0, 1},
	{0x0505f0, 17},
	{0x050700, 2},
	{0x050800, 1},
	{0x050820, 9},
	{0x050850, 5},
	{0x050870, 5},
	{0x0508c4, 3},
	{0x0508d4, 3},
	{0x0508e4, 3},
	{0x0508f4, 4},
	{0x050910, 5},
	{0x050930, 4},
	{0x050944, 1},
	{0x05094c, 57},
	{0x050a40, 1},
	{0x050a50, 28},
	{0x050ac4, 1},
	{0x050acc, 2},
	{0x050ad8, 10},
	{0x050b04, 1},
	{0x050b0c, 1},
	{0x050b14, 1},
	{0x050b20, 9},
	{0x050b50, 4},
	{0x050b64, 1},
	{0x050b70, 6},
	{0x050b8c, 4},
	{0x050ba4, 4},
	{0x050bc0, 1},
	{0x050bd0, 1},
	{0x050be0, 1},
	{0x050bf0, 1},
	{0x050c00, 1},
	{0x050c14, 1},
	{0x050c1c, 1},
	{0x050c24, 1},
	{0x050c2c, 1},
	{0x050c34, 1},
	{0x050c3c, 1},
	{0x050c44, 1},
	{0x050c4c, 1},
	{0x050c54, 4},
	{0x050c70, 1},
	{0x050c80, 1},
	{0x050c90, 1},
	{0x050ca0, 1},
	{0x050cb0, 1},
	{0x050cc0, 1},
	{0x050cd0, 1},
	{0x050ce0, 1},
	{0x050cf0, 1},
	{0x050d00, 1},
	{0x050d10, 1},
	{0x050d20, 1},
	{0x050d30, 1},
	{0x050d40, 1},
	{0x050d50, 1},
	{0x050d60, 1},
	{0x050d70, 1},
	{0x050d80, 1},
	{0x050d90, 1},
	{0x050da0, 1},
	{0x052000, 192},
	{0x052800, 2},
	{0x053000, 51},
	{0x053400, 3},
	{0x053420, 6},
	{0x053440, 6},
	{0x053460, 6},
	{0x053480, 6},
	{0x0534a0, 6},
	{0x0534c0, 6},
	{0x0534e0, 6},
	{0x053500, 6},
	{0x053520, 6},
	{0x053540, 6},
	{0x053560, 6},
	{0x053600, 32},
	{0x0536c0, 16},
	{0x053704, 1},
	{0x053740, 18},
	{0x0537a0, 10},
	{0x0537e0, 29},
	{0x053860, 10},
	{0x054204, 1},
	{0x05420c, 1},
	{0x054214, 1},
	{0x054224, 1},
	{0x05422c, 1},
	{0x054234, 1},
	{0x054244, 1},
	{0x05424c, 3},
	{0x054304, 1},
	{0x054340, 16},
	{0x0543a0, 1},
	{0x0543b0, 1},
	{0x0543c0, 1},
	{0x0543d0, 4},
	{0x054400, 8},
	{0x054820, 3},
	{0x054928, 9},
	{0x054954, 4},
	{0x054970, 1},
	{0x054980, 1},
	{0x054990, 3},
	{0x0549c4, 13},
	{0x054a04, 1},
	{0x054a0c, 4},
	{0x054a40, 1},
	{0x054a50, 5},
	{0x054a70, 1},
	{0x054a80, 1},
	{0x054a90, 2},
	{0x054c04, 1},
	{0x054c0c, 4},
	{0x054c40, 1},
	{0x054c50, 4},
	{0x054d40, 1},
	{0x054d80, 7},
	{0x054e04, 1},
	{0x054e0c, 4},
	{0x054e40, 1},
	{0x054e50, 4},
	{0x054f40, 1},
	{0x054f80, 7},
	{0x055000, 11},
	{0x055050, 7},
	{0x055070, 16},
	{0x0550b8, 8},
	{0x055200, 13},
	{0x055240, 11},
	{0x055300, 16},
	{0x055400, 3},
	{0x055420, 6},
	{0x055440, 6},
	{0x055460, 6},
	{0x055480, 6},
	{0x0554a0, 6},
	{0x0554c0, 6},
	{0x0554e0, 6},
	{0x055500, 6},
	{0x055520, 6},
	{0x055540, 6},
	{0x055560, 6},
	{0x055580, 6},
	{0x0555a0, 6},
	{0x0555c0, 6},
	{0x0555e0, 6},
	{0x055600, 6},
	{0x055620, 6},
	{0x055640, 6},
	{0x055660, 6},
	{0x056000, 2},
	{0x056044, 1},
	{0x05604c, 4},
	{0x056800, 2},
	{0x056844, 1},
	{0x05684c, 4},
	{0x057004, 7},
	{0x057b20, 2},
	{0x057b30, 2},
	{0x057b50, 5},
	{0x058004, 1},
	{0x05800c, 4},
	{0x058800, 7},
	{0x058820, 6},
	{0x058844, 2},
	{0x058864, 9},
	{0x058890, 1},
	{0x0588a0, 1},
	{0x0588b0, 1},
	{0x0588c0, 1},
	{0x0588d0, 1},
	{0x058904, 3},
	{0x058978, 66},
	{0x058a84, 1},
	{0x058a8c, 1},
	{0x058a94, 1},
	{0x058aa0, 2},
	{0x058ff0, 4},
	{0x059004, 1},
	{0x05900c, 4},
	{0x059080, 39},
	{0x059120, 1},
	{0x059130, 1},
	{0x059140, 1},
	{0x059150, 1},
	{0x059160, 1},
	{0x059800, 1},
	{0x059810, 1},
	{0x059820, 1},
	{0x059830, 1},
	{0x059840, 1},
	{0x059850, 1},
	{0x059860, 1},
	{0x059870, 1},
	{0x059880, 1},
	{0x059890, 1},
	{0x0598a0, 1},
	{0x0598d4, 2},
	{0x0598e0, 1},
	{0x059900, 3},
	{0x059980, 9},
	{0x0599c0, 32},
	{0x059a44, 3},
	{0x059a54, 1},
	{0x059a5c, 1},
	{0x059c00, 2},
	{0x059c28, 2},
	{0x059c50, 2},
	{0x059c78, 2},
	{0x059ca0, 20},
	{0x059dac, 1},
	{0x059dd8, 4},
	{0x059e00, 6},
	{0x059e20, 6},
	{0x059e40, 6},
	{0x059e60, 6},
	{0x059e80, 3},
	{0x05a000, 4},
	{0x05a020, 2},
	{0x05a030, 1},
	{0x05a04c, 2},
	{0x05a070, 4},
	{0x05a088, 3},
	{0x05a0a0, 1},
	{0x05a0b0, 1},
	{0x05a0c0, 1},
	{0x060020, 4},
	{0x060120, 4},
	{0x060144, 2},
	{0x060154, 7},
	{0x060174, 1},
	{0x06017c, 3},
	{0x06018c, 1},
	{0x0601a0, 4},
	{0x0601b8, 2},
	{0x0601c4, 2},
	{0x0601e4, 7},
	{0x060204, 1},
	{0x06020c, 4},
	{0x060240, 16},
	{0x060304, 1},
	{0x06030c, 1},
	{0x060400, 2},
	{0x060420, 3},
	{0x060430, 1},
	{0x060440, 1},
	{0x060484, 2},
	{0x0604b8, 18},
	{0x060504, 1},
	{0x06050c, 1},
	{0x060514, 1},
	{0x060900, 1},
	{0x060914, 1},
	{0x06091c, 2},
	{0x060930, 1},
	{0x060a00, 32},
	{0x060a84, 1},
	{0x060a8c, 1},
	{0x060a94, 1},
	{0x060a9c, 1},
	{0x060bf0, 3},
	{0x060c00, 11},
	{0x060c30, 3},
	{0x061004, 1},
	{0x061010, 1},
	{0x061018, 4},
	{0x061044, 2},
	{0x061054, 3},
	{0x061100, 32},
	{0x061200, 1},
	{0x061404, 1},
	{0x06140c, 4},
	{0x061440, 2},
	{0x061468, 2},
	{0x061478, 2},
	{0x06180c, 8},
	{0x061908, 6},
	{0x061928, 6},
	{0x061968, 6},
	{0x061988, 6},
	{0x0619a8, 6},
	{0x0619e8, 14},
	{0x062008, 5},
	{0x062024, 3},
	{0x062034, 1},
	{0x06203c, 1},
	{0x062044, 1},
	{0x06204c, 1},
	{0x062054, 1},
	{0x06205c, 3},
	{0x06206c, 1},
	{0x062080, 2},
	{0x062094, 1},
	{0x06209c, 2},
	{0x0620b0, 1},
	{0x0620c0, 1},
	{0x0620d0, 1},
	{0x0620e0, 1},
	{0x0620f4, 4},
	{0x0621c0, 2},
	{0x0621e4, 3},
	{0x0621f4, 3},
	{0x062404, 1},
	{0x06240c, 4},
	{0x062800, 12},
	{0x062834, 1},
	{0x06283c, 5},
	{0x062854, 1},
	{0x062900, 3},
	{0x062914, 1},
	{0x06291c, 1},
	{0x062924, 1},
	{0x06292c, 1},
	{0x062934, 1},
	{0x06293c, 2},
	{0x062950, 3},
	{0x062960, 2},
	{0x063000, 1},
	{0x063010, 4},
	{0x063024, 1},
	{0x06302c, 1},
	{0x063034, 1},
	{0x063044, 1},
	{0x06304c, 1},
	{0x063054, 1},
	{0x06305c, 2},
	{0x063070, 1},
	{0x063080, 1},
	{0x063090, 1},
	{0x0630a4, 3},
	{0x063100, 2},
	{0x063144, 1},
	{0x06314c, 1},
	{0x063154, 1},
	{0x063164, 1},
	{0x06316c, 1},
	{0x063174, 1},
	{0x063180, 16},
	{0x063208, 1},
	{0x063210, 1},
	{0x063218, 1},
	{0x063224, 3},
	{0x063240, 10},
	{0x063300, 8},
	{0x064004, 1},
	{0x06400c, 4},
	{0x064200, 3},
	{0x064300, 1},
	{0x064308, 6},
	{0x064324, 2},
	{0x064338, 2},
	{0x064380, 2},
	{0x064394, 1},
	{0x06439c, 2},
	{0x064400, 2},
	{0x064420, 3},
	{0x064430, 1},
	{0x064440, 1},
	{0x064484, 2},
	{0x0644b8, 18},
	{0x064504, 1},
	{0x06450c, 1},
	{0x064514, 1},
	{0x064804, 1},
	{0x064884, 2},
	{0x0648c8, 14},
	{0x065018, 2},
	{0x065080, 9},
	{0x0650c0, 16},
	{0x065104, 3},
	{0x065114, 1},
	{0x065200, 3},
	{0x065220, 6},
	{0x065240, 6},
	{0x065260, 6},
	{0x065280, 6},
	{0x065300, 1},
	{0x065f00, 2},
	{0x066000, 2},
	{0x066028, 2},
	{0x066050, 2},
	{0x066078, 2},
	{0x0660a0, 24},
	{0x0661ac, 1},
	{0x0661d8, 4},
	{0x066200, 6},
	{0x066220, 6},
	{0x066240, 6},
	{0x066260, 6},
	{0x066280, 3},
	{0x066400, 6},
	{0x066500, 17},
	{0x066550, 8},
	{0x066574, 3},
	{0x070004, 1},
	{0x07000c, 4},
	{0x070400, 7},
	{0x070420, 11},
	{0x070500, 5},
	{0x070524, 1},
	{0x07052c, 1},
	{0x070534, 1},
	{0x070540, 4},
	{0x070600, 142},
	{0x070884, 2},
	{0x0708b4, 19},
	{0x070904, 1},
	{0x070940, 16},
	{0x071000, 1},
	{0x071094, 1},
	{0x071104, 2},
	{0x071140, 16},
	{0x071204, 1},
	{0x07120c, 4},
	{0x071404, 1},
	{0x07140c, 1},
	{0x071414, 1},
	{0x07141c, 1},
	{0x071424, 1},
	{0x07142c, 1},
	{0x071434, 1},
	{0x07143c, 1},
	{0x071800, 2},
	{0x071904, 1},
	{0x071940, 16},
	{0x071a04, 1},
	{0x071a40, 16},
	{0x071b04, 1},
	{0x071b40, 16},
	{0x072000, 3},
	{0x072804, 1},
	{0x07280c, 4},
	{0x072880, 1},
	{0x072888, 3},
	{0x073000, 1},
	{0x073020, 1},
	{0x073040, 1},
	{0x073060, 1},
	{0x073080, 1},
	{0x0730a0, 1},
	{0x0730c0, 1},
	{0x0730e0, 1},
	{0x073100, 1},
	{0x073120, 1},
	{0x073140, 1},
	{0x073160, 1},
	{0x073180, 1},
	{0x0731a0, 1},
	{0x0731c0, 1},
	{0x0731e0, 1},
	{0x073200, 1},
	{0x073220, 1},
	{0x073240, 1},
	{0x073260, 1},
	{0x073280, 1},
	{0x0732a0, 1},
	{0x0732c0, 1},
	{0x0732e0, 1},
	{0x073300, 1},
	{0x073320, 1},
	{0x073340, 1},
	{0x073360, 1},
	{0x073380, 1},
	{0x0733a0, 1},
	{0x0733c0, 1},
	{0x0733e0, 1},
	{0x073800, 16},
	{0x074830, 4},
	{0x074884, 1},
	{0x074890, 4},
	{0x074900, 3},
	{0x074920, 6},
	{0x074940, 6},
	{0x074980, 9},
	{0x0749c0, 16},
	{0x074c00, 1},
	{0x074c28, 1},
	{0x074c50, 1},
	{0x074c78, 1},
	{0x074ca0, 12},
	{0x074dac, 1},
	{0x074dd8, 4},
	{0x074e00, 6},
	{0x074e20, 6},
	{0x074e40, 3},
	{0x075000, 3},
	{0x075010, 2},
	{0x075020, 3},
	{0x075100, 2},
	{0x078000, 20},
	{0x078054, 9},
	{0x07807c, 3},
	{0x07808c, 7},
	{0x078100, 5},
	{0x078118, 4},
	{0x078130, 2},
	{0x078204, 2},
	{0x078244, 15},
	{0x078284, 2},
	{0x0782c4, 17},
	{0x07830c, 4},
	{0x078320, 2},
	{0x078340, 2},
	{0x078360, 2},
	{0x078380, 2},
	{0x0783a0, 2},
	{0x0783d4, 1},
	{0x0783dc, 1},
	{0x078404, 1},
	{0x07840c, 4},
	{0x078804, 18},
	{0x078850, 4},
	{0x078884, 1},
	{0x07889c, 1},
	{0x0788a4, 23},
	{0x079000, 9},
	{0x079040, 16},
	{0x079804, 2},
	{0x079828, 6},
	{0x079844, 2},
	{0x079868, 6},
	{0x079884, 2},
	{0x0798a8, 7},
	{0x0798d0, 7},
	{0x079904, 1},
	{0x07990c, 1},
	{0x079914, 1},
	{0x079a04, 2},
	{0x079a40, 16},
	{0x079a84, 2},
	{0x079a90, 2},
	{0x079a9c, 13},
	{0x079b04, 2},
	{0x079b3c, 17},
	{0x079c04, 2},
	{0x079c40, 16},
	{0x079c84, 2},
	{0x079c98, 2},
	{0x079cc4, 13},
	{0x079cfc, 7},
	{0x079d24, 1},
	{0x079d2c, 1},
	{0x079d80, 2},
	{0x079d8c, 1},
	{0x079f00, 6},
	{0x079f20, 6},
	{0x079f40, 6},
	{0x079f60, 6},
	{0x079fa0, 3},
	{0x079fb0, 1},
	{0x07a000, 3},
	{0x07a010, 3},
	{0x07a020, 3},
	{0x07a040, 9},
	{0x07a068, 6},
	{0x07a090, 2},
	{0x07a0a0, 3},
	{0x07a0c0, 12},
	{0x07a0f4, 1},
	{0x07a100, 2},
	{0x07a110, 3},
	{0x07a120, 1},
	{0x07a130, 11},
	{0x07a160, 2},
	{0x07a180, 4},
	{0x07a194, 3},
	{0x07a1a4, 2},
	{0x07a1b0, 4},
	{0x07a2e0, 5},
	{0x07a2f8, 2},
	{0x07a30c, 13},
	{0x07a34c, 77},
	{0x07a48c, 18},
	{0x07a500, 32},
	{0x07a800, 16},
	{0x07a84c, 18},
	{0x07a8e0, 2},
	{0x07a8ec, 3},
	{0x07a900, 1},
	{0x07a910, 5},
	{0x07a930, 5},
	{0x07a950, 5},
	{0x07a970, 5},
	{0x07a9a4, 3},
	{0x07aa78, 34},
	{0x07ab3c, 18},
	{0x07ab94, 3},
	{0x07aba4, 3},
	{0x07abf0, 1},
	{0x07abf8, 10},
	{0x07ac30, 1},
	{0x07ac44, 4},
	{0x07ac60, 1},
	{0x07ac80, 1},
	{0x07ac90, 1},
	{0x07b004, 6},
	{0x080000, 29},
	{0x080340, 14},
	{0x08037c, 3},
	{0x08038c, 1},
	{0x0803c0, 2},
	{0x0803e4, 3},
	{0x0803f4, 3},
	{0x080404, 6},
	{0x080804, 2},
	{0x080874, 35},
	{0x081000, 129},
	{0x081210, 4},
	{0x081228, 3},
	{0x081240, 2},
	{0x081264, 2},
	{0x081274, 3},
	{0x081284, 2},
	{0x081298, 2},
	{0x0812a4, 1},
	{0x0812b4, 21},
	{0x081310, 8},
	{0x081344, 1},
	{0x08134c, 1},
	{0x081354, 1},
	{0x081364, 2},
	{0x081370, 4},
	{0x081384, 2},
	{0x081390, 4},
	{0x081404, 2},
	{0x081478, 34},
	{0x081504, 2},
	{0x081518, 14},
	{0x081580, 5},
	{0x081598, 2},
	{0x0815a4, 10},
	{0x082000, 29},
	{0x082340, 14},
	{0x08237c, 3},
	{0x08238c, 1},
	{0x0823c0, 2},
	{0x0823e4, 3},
	{0x0823f4, 3},
	{0x082404, 1},
	{0x08240c, 4},
	{0x082804, 2},
	{0x082874, 35},
	{0x082904, 2},
	{0x082974, 35},
	{0x083000, 129},
	{0x083210, 6},
	{0x083244, 2},
	{0x083254, 7},
	{0x083284, 1},
	{0x08328c, 1},
	{0x083294, 1},
	{0x0832a4, 1},
	{0x0832b4, 19},
	{0x083304, 2},
	{0x083310, 4},
	{0x083324, 2},
	{0x083330, 14},
	{0x084000, 29},
	{0x084340, 14},
	{0x08437c, 3},
	{0x08438c, 1},
	{0x0843c0, 2},
	{0x0843e4, 3},
	{0x0843f4, 3},
	{0x084404, 1},
	{0x08440c, 4},
	{0x084804, 2},
	{0x084874, 35},
	{0x084904, 2},
	{0x084974, 35},
	{0x085000, 32},
	{0x085200, 1},
	{0x085210, 7},
	{0x085240, 12},
	{0x085280, 2},
	{0x0852a4, 1},
	{0x0852b4, 3},
	{0x085304, 1},
	{0x08530c, 1},
	{0x085314, 1},
	{0x085324, 2},
	{0x085334, 3},
	{0x085344, 2},
	{0x085358, 2},
	{0x085364, 2},
	{0x085378, 2},
	{0x085384, 2},
	{0x085398, 2},
	{0x0853c0, 23},
	{0x086000, 2},
	{0x086020, 2},
	{0x086040, 1},
	{0x086400, 11},
	{0x086800, 3},
	{0x086820, 6},
	{0x086840, 6},
	{0x086860, 6},
	{0x086880, 6},
	{0x0868a0, 6},
	{0x0868c0, 6},
	{0x0868e0, 6},
	{0x086900, 9},
	{0x086940, 16},
	{0x087000, 26},
	{0x087100, 1},
	{0x087108, 1},
	{0x087110, 1},
	{0x087118, 1},
	{0x087120, 1},
	{0x087128, 1},
	{0x087130, 1},
	{0x087138, 1},
	{0x087140, 1},
	{0x087148, 1},
	{0x087150, 1},
	{0x087158, 1},
	{0x087160, 1},
	{0x087168, 1},
	{0x087170, 1},
	{0x087178, 1},
	{0x087180, 10},
	{0x0871b0, 9},
	{0x087200, 1},
	{0x087208, 1},
	{0x087210, 3},
	{0x090000, 17},
	{0x090060, 2},
	{0x09006c, 1},
	{0x090104, 1},
	{0x090140, 25},
	{0x0901a8, 2},
	{0x0901c0, 9},
	{0x0901e8, 2},
	{0x090204, 1},
	{0x090220, 24},
	{0x090300, 6},
	{0x090320, 9},
	{0x090348, 1},
	{0x090350, 1},
	{0x090400, 6},
	{0x090420, 9},
	{0x090448, 1},
	{0x090450, 1},
	{0x090500, 6},
	{0x090520, 6},
	{0x090540, 2},
	{0x090564, 2},
	{0x090578, 3},
	{0x091004, 3},
	{0x091800, 8},
	{0x091824, 2},
	{0x091830, 10},
	{0x091860, 6},
	{0x092000, 32},
	{0x093000, 1},
	{0x093020, 1},
	{0x093040, 1},
	{0x093060, 1},
	{0x093080, 1},
	{0x0930a0, 1},
	{0x0930c0, 1},
	{0x0930e0, 1},
	{0x093100, 1},
	{0x0931a0, 1},
	{0x0931c0, 1},
	{0x093200, 3},
	{0x093404, 1},
	{0x093440, 16},
	{0x093504, 1},
	{0x09353c, 28},
	{0x0935b0, 2},
	{0x0935c0, 3},
	{0x094000, 9},
	{0x094040, 19},
	{0x0940c0, 1},
	{0x094800, 1},
	{0x094828, 1},
	{0x094850, 1},
	{0x094878, 1},
	{0x0948a0, 8},
	{0x0949ac, 1},
	{0x0949d8, 4},
	{0x094a00, 6},
	{0x094a20, 6},
	{0x094a40, 3},
	{0x096000, 1},
	{0x096010, 4},
	{0x096028, 3},
	{0x096104, 1},
	{0x09610c, 7},
	{0x096204, 1},
	{0x09620c, 4},
	{0x096488, 1},
	{0x096498, 3},
	{0x0964b0, 4},
	{0x096504, 1},
	{0x09650c, 4},
	{0x096584, 4},
	{0x096600, 16},
	{0x096644, 2},
	{0x096658, 10},
	{0x096684, 2},
	{0x0966bc, 51},
	{0x096800, 11},
	{0x096884, 3},
	{0x0968a0, 12},
	{0x097100, 6},
	{0x097120, 1},
	{0x0a0000, 10},
	{0x0a0030, 4},
	{0x0a0080, 6},
	{0x0a00a0, 8},
	{0x0a0400, 7},
	{0x0a0420, 2},
	{0x0a0464, 3},
	{0x0a0480, 6},
	{0x0a04a0, 6},
	{0x0a0500, 30},
	{0x0a0580, 2},
	{0x0a0800, 14},
	{0x0a0840, 6},
	{0x0a0860, 18},
	{0x0a1004, 1},
	{0x0a100c, 4},
	{0x0a1044, 2},
	{0x0a1058, 2},
	{0x0a1064, 2},
	{0x0a1074, 6},
	{0x0a1090, 9},
	{0x0a1204, 1},
	{0x0a120c, 4},
	{0x0a1244, 2},
	{0x0a1254, 6},
	{0x0a1270, 6},
	{0x0a1300, 1},
	{0x0a1404, 1},
	{0x0a1440, 16},
	{0x0a1484, 1},
	{0x0a148c, 4},
	{0x0a14c4, 1},
	{0x0a14d0, 12},
	{0x0a1504, 1},
	{0x0a1510, 12},
	{0x0a1544, 1},
	{0x0a1550, 12},
	{0x0a1584, 1},
	{0x0a1590, 12},
	{0x0a15c4, 1},
	{0x0a15cc, 5},
	{0x0a15e4, 1},
	{0x0a15ec, 5},
	{0x0a1604, 1},
	{0x0a160c, 5},
	{0x0a1624, 1},
	{0x0a162c, 5},
	{0x0a1644, 1},
	{0x0a164c, 5},
	{0x0a1664, 1},
	{0x0a166c, 5},
	{0x0a1684, 1},
	{0x0a168c, 5},
	{0x0a16a4, 1},
	{0x0a16ac, 5},
	{0x0a16c4, 7},
	{0x0a16e4, 1},
	{0x0a16ec, 1},
	{0x0a16f4, 1},
	{0x0a16fc, 1},
	{0x0a1704, 17},
	{0x0a1754, 1},
	{0x0a175c, 1},
	{0x0a1764, 2},
	{0x0a1774, 3},
	{0x0a1800, 18},
	{0x0a1900, 3},
	{0x0a1948, 3},
	{0x0a1958, 6},
	{0x0a1974, 8},
	{0x0a2004, 1},
	{0x0a200c, 4},
	{0x0a2400, 2},
	{0x0a240c, 6},
	{0x0a2440, 1},
	{0x0a2450, 4},
	{0x0a2468, 3},
	{0x0a2480, 1},
	{0x0a24a0, 10},
	{0x0a24d0, 9},
	{0x0a2804, 1},
	{0x0a280c, 4},
	{0x0a2c00, 2},
	{0x0a2c0c, 2},
	{0x0a2c40, 1},
	{0x0a2c50, 4},
	{0x0a2c68, 3},
	{0x0a2c88, 2},
	{0x0a2cf0, 1},
	{0x0a3004, 1},
	{0x0a300c, 4},
	{0x0a3040, 2},
	{0x0a3064, 3},
	{0x0a3074, 4},
	{0x0a3200, 9},
	{0x0a3230, 2},
	{0x0a323c, 2},
	{0x0a3248, 4},
	{0x0a3400, 1},
	{0x0a3408, 3},
	{0x0a3418, 4},
	{0x0a3430, 2},
	{0x0a343c, 1},
	{0x0a3480, 1},
	{0x0a3490, 1},
	{0x0a3504, 1},
	{0x0a3510, 76},
	{0x0a4000, 48},
	{0x0a4100, 3},
	{0x0a4110, 6},
	{0x0a412c, 4},
	{0x0a4140, 1},
	{0x0a4304, 1},
	{0x0a4318, 10},
	{0x0a4804, 1},
	{0x0a480c, 4},
	{0x0a4840, 2},
	{0x0a4864, 3},
	{0x0a4874, 3},
	{0x0a4c04, 2},
	{0x0a4c10, 2},
	{0x0a4c1c, 6},
	{0x0a4c38, 2},
	{0x0a4c50, 8},
	{0x0a4c78, 19},
	{0x0a4d04, 2},
	{0x0a4d40, 21},
	{0x0a4da4, 1},
	{0x0a4dac, 1},
	{0x0a4db4, 1},
	{0x0a4dc0, 1},
	{0x0a5000, 14},
	{0x0a6000, 2},
	{0x0a6028, 2},
	{0x0a6050, 2},
	{0x0a6078, 2},
	{0x0a60a0, 35},
	{0x0a61ac, 1},
	{0x0a61d8, 4},
	{0x0a6200, 6},
	{0x0a6220, 6},
	{0x0a6240, 6},
	{0x0a6260, 6},
	{0x0a6280, 3},
	{0x0a6400, 3},
	{0x0a6420, 6},
	{0x0a6440, 6},
	{0x0a6460, 6},
	{0x0a6480, 6},
	{0x0a64a0, 6},
	{0x0a64c0, 6},
	{0x0a6500, 9},
	{0x0a6540, 18},
	{0x0a65c0, 4},
	{0x0a8000, 10},
	{0x0a802c, 15},
	{0x0a806c, 5},
	{0x0a8408, 5},
	{0x0a8424, 3},
	{0x0a8434, 6},
	{0x0a8450, 2},
	{0x0a845c, 5},
	{0x0a84c4, 1},
	{0x0a84cc, 4},
	{0x0a8604, 1},
	{0x0a860c, 4},
	{0x0a8700, 17},
	{0x0a8750, 4},
	{0x0a8800, 4},
	{0x0a8880, 1},
	{0x0a88a0, 1},
	{0x0a88c0, 1},
	{0x0a8900, 1},
	{0x0a8960, 1},
	{0x0a8980, 4},
	{0x0a8994, 1},
	{0x0a899c, 1},
	{0x0a89a4, 3},
	{0x0a89c0, 1},
	{0x0a8a00, 4},
	{0x0a8aa4, 1},
	{0x0a8aac, 1},
	{0x0a8ab4, 1},
	{0x0a8ad4, 10},
	{0x0a8b00, 2},
	{0x0a8b80, 4},
	{0x0a8c04, 6},
	{0x0a9000, 3},
	{0x0a9010, 21},
	{0x0a9080, 4},
	{0x0a9100, 10},
	{0x0a91f4, 3},
	{0x0a920c, 35},
	{0x0a92a4, 7},
	{0x0a930c, 39},
	{0x0a940c, 35},
	{0x0a949c, 9},
	{0x0a9500, 2},
	{0x0a950c, 3},
	{0x0a951c, 3},
	{0x0a954c, 13},
	{0x0a9804, 1},
	{0x0a980c, 4},
	{0x0a9c00, 2},
	{0x0a9c0c, 3},
	{0x0a9c1c, 2},
	{0x0a9c28, 1},
	{0x0a9c44, 1},
	{0x0a9c60, 17},
	{0x0a9d00, 1},
	{0x0a9d20, 8},
	{0x0a9d48, 3},
	{0x0a9d80, 1},
	{0x0a9dc4, 3},
	{0x0a9dd4, 3},
	{0x0a9de4, 3},
	{0x0a9df4, 4},
	{0x0a9e20, 8},
	{0x0a9e48, 3},
	{0x0a9e84, 1},
	{0x0a9e98, 1},
	{0x0a9ea0, 8},
	{0x0a9f00, 6},
	{0x0a9f20, 6},
	{0x0a9f40, 6},
	{0x0a9f60, 6},
	{0x0a9f80, 9},
	{0x0a9fc0, 31},
	{0x0aa204, 1},
	{0x0aa20c, 4},
	{0x0b0000, 2},
	{0x0b0010, 4},
	{0x0b1000, 15},
	{0x0b1040, 10},
	{0x0b1080, 20},
	{0x0b1100, 2},
	{0x0b1110, 2},
	{0x0b1120, 2},
	{0x0b1160, 2},
	{0x0b116c, 1},
	{0x0b1180, 3},
	{0x0b1190, 3},
	{0x0b11a0, 3},
	{0x0b11e0, 2},
	{0x0b11ec, 1},
	{0x0b1200, 2},
	{0x0b1210, 2},
	{0x0b1260, 2},
	{0x0b126c, 1},
	{0x0b1280, 3},
	{0x0b1290, 3},
	{0x0b12e0, 2},
	{0x0b12ec, 1},
	{0x0b1300, 2},
	{0x0b1310, 2},
	{0x0b1320, 2},
	{0x0b1360, 2},
	{0x0b136c, 1},
	{0x0b1380, 3},
	{0x0b1390, 3},
	{0x0b13e0, 2},
	{0x0b13ec, 1},
	{0x0b1400, 3},
	{0x0b1410, 3},
	{0x0b1460, 2},
	{0x0b146c, 1},
	{0x0b1480, 3},
	{0x0b1490, 7},
	{0x0b14b0, 4},
	{0x0b14e0, 2},
	{0x0b14ec, 1},
	{0x0b1500, 3},
	{0x0b1510, 3},
	{0x0b1560, 2},
	{0x0b156c, 1},
	{0x0b1580, 2},
	{0x0b1590, 2},
	{0x0b15e0, 2},
	{0x0b15ec, 1},
	{0x0b1600, 3},
	{0x0b1610, 3},
	{0x0b1660, 2},
	{0x0b166c, 1},
	{0x0b1680, 3},
	{0x0b1690, 3},
	{0x0b16e0, 2},
	{0x0b16ec, 1},
	{0x0b1700, 8},
	{0x0b1760, 2},
	{0x0b176c, 1},
	{0x0b1780, 3},
	{0x0b1790, 3},
	{0x0b17e0, 2},
	{0x0b17ec, 1},
	{0x0b1800, 9},
	{0x0b1840, 16},
	{0x0b2000, 2},
	{0x0b2010, 24},
	{0x0b2080, 9},
	{0x0b20c0, 18},
	{0x0b2110, 24},
	{0x0b2180, 9},
	{0x0b21c0, 18},
	{0x0b2210, 24},
	{0x0b2280, 9},
	{0x0b22c0, 18},
	{0x0b2310, 24},
	{0x0b2380, 9},
	{0x0b23c0, 18},
	{0x0b2410, 24},
	{0x0b2480, 9},
	{0x0b24c0, 18},
	{0x0b2510, 24},
	{0x0b2580, 9},
	{0x0b25c0, 18},
	{0x0b2610, 24},
	{0x0b2680, 9},
	{0x0b26c0, 18},
	{0x0b2710, 24},
	{0x0b2780, 9},
	{0x0b27c0, 16},
	{0x0b2900, 1},
	{0x0b2910, 1},
	{0x0b3000, 157},
	{0x0b3278, 26},
	{0x0b3300, 9},
	{0x0b3404, 1},
	{0x0b340c, 4},
	{0x0b3800, 73},
	{0x0b3940, 20},
	{0x0b39a0, 4},
	{0x0b39c4, 5},
	{0x0b3a00, 1},
	{0x0b3a20, 8},
	{0x0b3a44, 3},
	{0x0b4000, 7},
	{0x0b4040, 12},
	{0x0b4100, 9},
	{0x0b4140, 16},
	{0x0b5000, 2},
	{0x0b5028, 2},
	{0x0b5050, 2},
	{0x0b5078, 2},
	{0x0b50a0, 5},
	{0x0b51ac, 1},
	{0x0b51d8, 4},
	{0x0b5200, 6},
	{0x0b5220, 6},
	{0x0b5240, 6},
	{0x0b5260, 6},
	{0x0b5280, 3},
	{0x0b6800, 3},
	{0x0b6820, 6},
	{0x0b6840, 6},
	{0x0b6860, 6},
	{0x0b6880, 6},
	{0x0b68a0, 6},
	{0x0b68c0, 6},
	{0x0b68e0, 6},
	{0x0b6900, 6},
	{0x0d0000, 5},
	{0x0d0038, 4},
	{0x0d0050, 4},
	{0x0d0080, 3},
	{0x0d00a0, 6},
	{0x0d00c0, 6},
	{0x0d0100, 3},
	{0x0d0120, 6},
	{0x0d0140, 6},
	{0x0d0180, 14},
	{0x0d01c0, 6},
	{0x0d1000, 2},
	{0x0d100c, 1},
	{0x0d1020, 13},
	{0x0d1058, 1},
	{0x0d1060, 6},
	{0x0d1080, 1},
	{0x0d1100, 2},
	{0x0d110c, 1},
	{0x0d1120, 13},
	{0x0d1158, 1},
	{0x0d1160, 6},
	{0x0d1180, 1},
	{0x0d1200, 2},
	{0x0d120c, 1},
	{0x0d1220, 13},
	{0x0d1258, 1},
	{0x0d1260, 6},
	{0x0d1280, 1},
	{0x0d1300, 2},
	{0x0d130c, 1},
	{0x0d1320, 13},
	{0x0d1358, 1},
	{0x0d1360, 6},
	{0x0d1380, 1},
	{0x0d1400, 3},
	{0x0d1410, 10},
	{0x0d1440, 1},
	{0x0d1450, 6},
	{0x0d1484, 6},
	{0x0d14a0, 6},
	{0x0d1504, 25},
	{0x0d1580, 14},
	{0x0d15c0, 4},
	{0x0d1600, 6},
	{0x0d1620, 6},
	{0x0d1640, 3},
	{0x0d1660, 6},
	{0x0d1700, 2},
	{0x0d170c, 3},
	{0x0d1720, 6},
	{0x0d1800, 11},
	{0x0d1830, 7},
	{0x0d1850, 7},
	{0x0d1870, 9},
	{0x0d18a0, 2},
	{0x0d18c0, 2},
	{0x0d1a00, 1},
	{0x0d1a08, 13},
	{0x0d1a40, 11},
	{0x0d1c00, 24},
	{0x0d1c64, 5},
	{0x0d1c80, 3},
	{0x0d1c90, 2},
	{0x0d1d00, 15},
	{0x0d1d40, 16},
	{0x0d1d90, 4},
	{0x0d1dc0, 12},
	{0x0d1e00, 1},
	{0x0d1e20, 1},
	{0x0d1e28, 12},
	{0x0d1e5c, 7},
	{0x0d1f00, 14},
	{0x0d2000, 2},
	{0x0d200c, 1},
	{0x0d2020, 13},
	{0x0d2058, 1},
	{0x0d2060, 6},
	{0x0d2080, 1},
	{0x0d2100, 2},
	{0x0d210c, 1},
	{0x0d2120, 13},
	{0x0d2158, 1},
	{0x0d2160, 6},
	{0x0d2180, 1},
	{0x0d2200, 2},
	{0x0d220c, 1},
	{0x0d2220, 13},
	{0x0d2258, 1},
	{0x0d2260, 6},
	{0x0d2280, 1},
	{0x0d2300, 2},
	{0x0d230c, 1},
	{0x0d2320, 13},
	{0x0d2358, 1},
	{0x0d2360, 6},
	{0x0d2380, 1},
	{0x0d2400, 3},
	{0x0d2410, 10},
	{0x0d2440, 1},
	{0x0d2450, 6},
	{0x0d2484, 6},
	{0x0d24a0, 6},
	{0x0d2504, 25},
	{0x0d2580, 14},
	{0x0d25c0, 4},
	{0x0d2600, 6},
	{0x0d2620, 6},
	{0x0d2640, 3},
	{0x0d2660, 6},
	{0x0d2700, 2},
	{0x0d270c, 3},
	{0x0d2720, 6},
	{0x0d2800, 11},
	{0x0d2830, 7},
	{0x0d2850, 7},
	{0x0d2870, 9},
	{0x0d28a0, 2},
	{0x0d28c0, 2},
	{0x0d2a00, 1},
	{0x0d2a08, 13},
	{0x0d2a40, 11},
	{0x0d2c00, 24},
	{0x0d2c64, 5},
	{0x0d2c80, 3},
	{0x0d2c90, 2},
	{0x0d2d00, 15},
	{0x0d2d40, 16},
	{0x0d2d90, 4},
	{0x0d2dc0, 12},
	{0x0d2e00, 1},
	{0x0d2e20, 1},
	{0x0d2e28, 12},
	{0x0d2e5c, 7},
	{0x0d2f00, 14},
	{0x0d3000, 2},
	{0x0d3400, 1},
	{0x0d3428, 1},
	{0x0d3450, 1},
	{0x0d3478, 1},
	{0x0d34a0, 5},
	{0x0d35ac, 1},
	{0x0d35d8, 4},
	{0x0d3600, 6},
	{0x0d3620, 6},
	{0x0d3640, 3},
	{0x0d3e00, 3},
	{0x0d3e20, 6},
	{0x0d3e40, 6},
	{0x0d3e80, 3},
	{0x0d3e90, 1},
	{0x0d4000, 29},
	{0x0d4078, 4},
	{0x0d4090, 2},
	{0x0d40a0, 7},
	{0x0d40c0, 11},
	{0x0d4100, 14},
	{0x0d4140, 14},
	{0x0d4180, 61},
	{0x0d4278, 4},
	{0x0d4290, 2},
	{0x0d42a0, 7},
	{0x0d42c0, 11},
	{0x0d4300, 14},
	{0x0d4340, 14},
	{0x0d4380, 61},
	{0x0d4478, 4},
	{0x0d4490, 2},
	{0x0d44a0, 7},
	{0x0d44c0, 11},
	{0x0d4500, 14},
	{0x0d4540, 14},
	{0x0d4580, 61},
	{0x0d4678, 4},
	{0x0d4690, 2},
	{0x0d46a0, 7},
	{0x0d46c0, 11},
	{0x0d4700, 14},
	{0x0d4740, 14},
	{0x0d4780, 62},
	{0x0d487c, 1},
	{0x0d4a00, 8},
	{0x0d4a24, 15},
	{0x0d4a64, 30},
	{0x0d4b00, 4},
	{0x0d4b20, 3},
	{0x0d4c00, 6},
	{0x0d4c40, 14},
	{0x0d4c80, 9},
	{0x0d4d00, 9},
	{0x0d4d2c, 1},
	{0x0d4d40, 3},
	{0x0d4d60, 1},
	{0x0d4d80, 3},
	{0x0d4e00, 2},
	{0x0d4e0c, 1},
	{0x0d4e14, 5},
	{0x0d4e2c, 1},
	{0x0d4e34, 5},
	{0x0d4e4c, 1},
	{0x0d4e54, 5},
	{0x0d4e6c, 1},
	{0x0d4e74, 5},
	{0x0d4e8c, 1},
	{0x0d4e94, 5},
	{0x0d4eac, 1},
	{0x0d4eb4, 3},
	{0x0d5000, 29},
	{0x0d5078, 4},
	{0x0d5090, 2},
	{0x0d50a0, 7},
	{0x0d50c0, 11},
	{0x0d5100, 14},
	{0x0d5140, 14},
	{0x0d5180, 61},
	{0x0d5278, 4},
	{0x0d5290, 2},
	{0x0d52a0, 7},
	{0x0d52c0, 11},
	{0x0d5300, 14},
	{0x0d5340, 14},
	{0x0d5380, 61},
	{0x0d5478, 4},
	{0x0d5490, 2},
	{0x0d54a0, 7},
	{0x0d54c0, 11},
	{0x0d5500, 14},
	{0x0d5540, 14},
	{0x0d5580, 61},
	{0x0d5678, 4},
	{0x0d5690, 2},
	{0x0d56a0, 7},
	{0x0d56c0, 11},
	{0x0d5700, 14},
	{0x0d5740, 14},
	{0x0d5780, 62},
	{0x0d587c, 1},
	{0x0d5a00, 8},
	{0x0d5a24, 15},
	{0x0d5a64, 30},
	{0x0d5b00, 4},
	{0x0d5b20, 3},
	{0x0d5c00, 6},
	{0x0d5c40, 14},
	{0x0d5c80, 9},
	{0x0d5d00, 9},
	{0x0d5d2c, 1},
	{0x0d5d40, 3},
	{0x0d5d60, 1},
	{0x0d5d80, 3},
	{0x0d5e00, 2},
	{0x0d5e0c, 1},
	{0x0d5e14, 5},
	{0x0d5e2c, 1},
	{0x0d5e34, 5},
	{0x0d5e4c, 1},
	{0x0d5e54, 5},
	{0x0d5e6c, 1},
	{0x0d5e74, 5},
	{0x0d5e8c, 1},
	{0x0d5e94, 5},
	{0x0d5eac, 1},
	{0x0d5eb4, 3},
	{0x0d6000, 15},
	{0x0d6070, 3},
	{0x0d6080, 6},
	{0x0d6100, 9},
	{0x0d6204, 8},
	{0x0d6240, 13},
	{0x0d6280, 16},
	{0x0d6400, 8},
	{0x0d6424, 15},
	{0x0d6464, 15},
	{0x0d64a4, 15},
	{0x0d64e4, 30},
	{0x0d6580, 10},
	{0x0d65ac, 1},
	{0x0d65b4, 5},
	{0x0d65cc, 1},
	{0x0d65d4, 5},
	{0x0d65ec, 1},
	{0x0d65f4, 13},
	{0x0d6680, 7},
	{0x0d66a0, 5},
	{0x0d66c0, 5},
	{0x0d66e0, 4},
	{0x0d6800, 19},
	{0x0d6850, 10},
	{0x0d6880, 19},
	{0x0d68d0, 10},
	{0x0d6900, 19},
	{0x0d6950, 10},
	{0x0d6980, 19},
	{0x0d69d0, 10},
	{0x0d6a00, 19},
	{0x0d6a50, 10},
	{0x0d6a80, 19},
	{0x0d6ad0, 10},
	{0x0d6b00, 19},
	{0x0d6b50, 10},
	{0x0d6b80, 19},
	{0x0d6bd0, 10},
	{0x0d6c00, 19},
	{0x0d6c60, 6},
	{0x0d6c84, 1},
	{0x0d6c94, 8},
	{0x0d6cb8, 9},
	{0x0d6ce0, 4},
	{0x0d7000, 9},
	{0x0d7040, 16},
	{0x0d8000, 6},
	{0x0d8020, 3},
	{0x0d8030, 3},
	{0x0d8040, 6},
	{0x0d8060, 17},
	{0x0d80c0, 38},
	{0x0d8180, 1},
	{0x0d8400, 2},
	{0x0d8428, 2},
	{0x0d8450, 2},
	{0x0d8478, 2},
	{0x0d84a0, 16},
	{0x0d85ac, 1},
	{0x0d85d8, 4},
	{0x0d8600, 6},
	{0x0d8620, 6},
	{0x0d8640, 6},
	{0x0d8660, 6},
	{0x0d8680, 3},
	{0x0d8800, 2},
	{0x0d9000, 35},
	{0x0d9100, 26},
	{0x0d916c, 7},
	{0x0d91a0, 1},
	{0x0d91c0, 9},
	{0x0d91e8, 1},
	{0x0d9200, 6},
	{0x0d9220, 6},
	{0x0d9248, 4},
	{0x0d9280, 6},
	{0x0d929c, 1},
	{0x0d92a4, 2},
	{0x0d92b8, 9},
	{0x0d9304, 4},
	{0x0d9328, 3},
	{0x0d9340, 6},
	{0x0d9400, 1},
	{0x0d9408, 1},
	{0x0d9410, 2},
	{0x0d9424, 2},
	{0x0d9444, 1},
	{0x0d9480, 27},
	{0x0d9500, 6},
	{0x0d9520, 12},
	{0x0d9700, 7},
	{0x0d9744, 9},
	{0x0d976c, 2},
	{0x0d9780, 6},
	{0x0d97a0, 2},
	{0x0d9800, 35},
	{0x0d9900, 26},
	{0x0d996c, 7},
	{0x0d99a0, 1},
	{0x0d99c0, 9},
	{0x0d99e8, 1},
	{0x0d9a00, 6},
	{0x0d9a20, 6},
	{0x0d9a48, 4},
	{0x0d9a80, 6},
	{0x0d9a9c, 1},
	{0x0d9aa4, 2},
	{0x0d9ab8, 9},
	{0x0d9b04, 4},
	{0x0d9b28, 3},
	{0x0d9b40, 6},
	{0x0d9c00, 1},
	{0x0d9c08, 1},
	{0x0d9c10, 2},
	{0x0d9c24, 2},
	{0x0d9c44, 1},
	{0x0d9c80, 27},
	{0x0d9d00, 6},
	{0x0d9d20, 12},
	{0x0d9f00, 7},
	{0x0d9f44, 9},
	{0x0d9f6c, 2},
	{0x0d9f80, 6},
	{0x0d9fa0, 2},
	{0x0db000, 1028},
	{0x0dc018, 18},
	{0x0dc100, 4},
	{0x0dc118, 18},
	{0x0dc200, 12},
	{0x0dc300, 6},
	{0x0dc320, 5},
	{0x0dc340, 6},
	{0x0dc360, 5},
	{0x0dc380, 6},
	{0x0dc400, 9},
	{0x0dc440, 26},
	{0x0dc4c4, 1},
	{0x0dc4cc, 1},
	{0x0dc4d4, 1},
	{0x0dc50c, 7},
	{0x0dc544, 2},
	{0x0dc55c, 9},
	{0x0dc584, 7},
	{0x0dc5a4, 2},
	{0x0dc5b8, 2},
	{0x0dc5c4, 2},
	{0x0dc5d8, 2},
	{0x0dc600, 2},
	{0x0dcfbc, 15},
	{0x0dd000, 7},
	{0x0dd020, 6},
	{0x0dd040, 8},
	{0x0dd104, 1},
	{0x0dd10c, 1},
	{0x0dd200, 8},
	{0x0dd244, 2},
	{0x0dd268, 18},
	{0x0dd404, 1},
	{0x0dd440, 40},
	{0x0dd504, 3},
	{0x0dd514, 3},
	{0x0dd524, 3},
	{0x0dd534, 3},
	{0x0dd544, 3},
	{0x0dd554, 3},
	{0x0dd564, 3},
	{0x0dd574, 3},
	{0x0dd584, 3},
	{0x0dd594, 3},
	{0x0dd5a4, 3},
	{0x0dd5b4, 3},
	{0x0dd604, 2},
	{0x0dd640, 16},
	{0x0dd684, 3},
	{0x0dd704, 2},
	{0x0dd740, 18},
	{0x0ddc00, 4},
	{0x0ddc80, 1},
	{0x0ddd00, 6},
	{0x0ddd20, 6},
	{0x0ddd40, 6},
	{0x0ddd80, 1},
	{0x0dde00, 3},
	{0x0dde20, 10},
	{0x0dde50, 6},
	{0x0dde80, 9},
	{0x0ddec0, 16},
	{0x0de000, 123},
	{0x0de200, 2},
	{0x0de20c, 3},
	{0x0de220, 2},
	{0x0de22c, 3},
	{0x0de240, 2},
	{0x0de24c, 3},
	{0x0de260, 2},
	{0x0de26c, 3},
	{0x0de280, 2},
	{0x0de28c, 3},
	{0x0de2a0, 2},
	{0x0de2ac, 3},
	{0x0de2c0, 2},
	{0x0de2cc, 3},
	{0x0de2e0, 2},
	{0x0de2ec, 3},
	{0x0de300, 2},
	{0x0de30c, 3},
	{0x0de320, 2},
	{0x0de32c, 3},
	{0x0de340, 2},
	{0x0de34c, 3},
	{0x0de360, 2},
	{0x0de36c, 3},
	{0x0de380, 2},
	{0x0de38c, 3},
	{0x0de3a0, 2},
	{0x0de3ac, 3},
	{0x0de3c0, 2},
	{0x0de3cc, 3},
	{0x0de3e0, 2},
	{0x0de3ec, 3},
	{0x0de400, 123},
	{0x0de600, 2},
	{0x0de60c, 3},
	{0x0de620, 2},
	{0x0de62c, 3},
	{0x0de640, 2},
	{0x0de64c, 3},
	{0x0de660, 2},
	{0x0de66c, 3},
	{0x0de680, 2},
	{0x0de68c, 3},
	{0x0de6a0, 2},
	{0x0de6ac, 3},
	{0x0de6c0, 2},
	{0x0de6cc, 3},
	{0x0de6e0, 2},
	{0x0de6ec, 3},
	{0x0de700, 2},
	{0x0de70c, 3},
	{0x0de720, 2},
	{0x0de72c, 3},
	{0x0de740, 2},
	{0x0de74c, 3},
	{0x0de760, 2},
	{0x0de76c, 3},
	{0x0de780, 2},
	{0x0de78c, 3},
	{0x0de7a0, 2},
	{0x0de7ac, 3},
	{0x0de7c0, 2},
	{0x0de7cc, 3},
	{0x0de7e0, 2},
	{0x0de7ec, 3},
	{0x0de800, 123},
	{0x0dea00, 2},
	{0x0dea0c, 3},
	{0x0dea20, 2},
	{0x0dea2c, 3},
	{0x0dea40, 2},
	{0x0dea4c, 3},
	{0x0dea60, 2},
	{0x0dea6c, 3},
	{0x0dea80, 2},
	{0x0dea8c, 3},
	{0x0deaa0, 2},
	{0x0deaac, 3},
	{0x0deac0, 2},
	{0x0deacc, 3},
	{0x0deae0, 2},
	{0x0deaec, 3},
	{0x0deb00, 2},
	{0x0deb0c, 3},
	{0x0deb20, 2},
	{0x0deb2c, 3},
	{0x0deb40, 2},
	{0x0deb4c, 3},
	{0x0deb60, 2},
	{0x0deb6c, 3},
	{0x0deb80, 2},
	{0x0deb8c, 3},
	{0x0deba0, 2},
	{0x0debac, 3},
	{0x0debc0, 2},
	{0x0debcc, 3},
	{0x0debe0, 2},
	{0x0debec, 3},
	{0x0dec00, 123},
	{0x0dee00, 2},
	{0x0dee0c, 3},
	{0x0dee20, 2},
	{0x0dee2c, 3},
	{0x0dee40, 2},
	{0x0dee4c, 3},
	{0x0dee60, 2},
	{0x0dee6c, 3},
	{0x0dee80, 2},
	{0x0dee8c, 3},
	{0x0deea0, 2},
	{0x0deeac, 3},
	{0x0deec0, 2},
	{0x0deecc, 3},
	{0x0deee0, 2},
	{0x0deeec, 3},
	{0x0def00, 2},
	{0x0def0c, 3},
	{0x0def20, 2},
	{0x0def2c, 3},
	{0x0def40, 2},
	{0x0def4c, 3},
	{0x0def60, 2},
	{0x0def6c, 3},
	{0x0def80, 2},
	{0x0def8c, 3},
	{0x0defa0, 2},
	{0x0defac, 3},
	{0x0defc0, 2},
	{0x0defcc, 3},
	{0x0defe0, 2},
	{0x0defec, 3},
	{0x0df000, 123},
	{0x0df200, 2},
	{0x0df20c, 3},
	{0x0df220, 2},
	{0x0df22c, 3},
	{0x0df240, 2},
	{0x0df24c, 3},
	{0x0df260, 2},
	{0x0df26c, 3},
	{0x0df280, 2},
	{0x0df28c, 3},
	{0x0df2a0, 2},
	{0x0df2ac, 3},
	{0x0df2c0, 2},
	{0x0df2cc, 3},
	{0x0df2e0, 2},
	{0x0df2ec, 3},
	{0x0df300, 2},
	{0x0df30c, 3},
	{0x0df320, 2},
	{0x0df32c, 3},
	{0x0df340, 2},
	{0x0df34c, 3},
	{0x0df360, 2},
	{0x0df36c, 3},
	{0x0df380, 2},
	{0x0df38c, 3},
	{0x0df3a0, 2},
	{0x0df3ac, 3},
	{0x0df3c0, 2},
	{0x0df3cc, 3},
	{0x0df3e0, 2},
	{0x0df3ec, 3},
	{0x0df400, 2},
	{0x0e0000, 3},
	{0x0e0010, 4},
	{0x0e0028, 3},
	{0x0e0048, 2},
	{0x0e0058, 2},
	{0x0e0064, 32},
	{0x0e00f0, 1},
	{0x0e00fc, 35},
	{0x0e019c, 15},
	{0x0e01e0, 1},
	{0x0e01e8, 5},
	{0x0e0204, 5},
	{0x0e021c, 1},
	{0x0e0300, 16},
	{0x0e0400, 3},
	{0x0e0410, 4},
	{0x0e0428, 3},
	{0x0e0448, 2},
	{0x0e0458, 2},
	{0x0e0464, 32},
	{0x0e04f0, 1},
	{0x0e04fc, 35},
	{0x0e059c, 15},
	{0x0e05e0, 1},
	{0x0e05e8, 5},
	{0x0e0604, 5},
	{0x0e061c, 1},
	{0x0e0700, 16},
	{0x0e0800, 3},
	{0x0e0810, 4},
	{0x0e0828, 3},
	{0x0e0848, 2},
	{0x0e0858, 2},
	{0x0e0864, 32},
	{0x0e08f0, 1},
	{0x0e08fc, 35},
	{0x0e099c, 15},
	{0x0e09e0, 1},
	{0x0e09e8, 5},
	{0x0e0a04, 5},
	{0x0e0a1c, 1},
	{0x0e0b00, 16},
	{0x0e0c00, 3},
	{0x0e0c10, 4},
	{0x0e0c28, 3},
	{0x0e0c48, 2},
	{0x0e0c58, 2},
	{0x0e0c64, 32},
	{0x0e0cf0, 1},
	{0x0e0cfc, 35},
	{0x0e0d9c, 15},
	{0x0e0de0, 1},
	{0x0e0de8, 5},
	{0x0e0e04, 5},
	{0x0e0e1c, 1},
	{0x0e0f00, 16},
	{0x0e1000, 3},
	{0x0e1010, 4},
	{0x0e1028, 3},
	{0x0e1048, 2},
	{0x0e1058, 2},
	{0x0e1064, 32},
	{0x0e10f0, 1},
	{0x0e10fc, 35},
	{0x0e119c, 15},
	{0x0e11e0, 1},
	{0x0e11e8, 5},
	{0x0e1204, 5},
	{0x0e121c, 1},
	{0x0e1300, 16},
	{0x0e1400, 3},
	{0x0e1410, 4},
	{0x0e1428, 3},
	{0x0e1448, 2},
	{0x0e1458, 2},
	{0x0e1464, 32},
	{0x0e14f0, 1},
	{0x0e14fc, 35},
	{0x0e159c, 15},
	{0x0e15e0, 1},
	{0x0e15e8, 5},
	{0x0e1604, 5},
	{0x0e161c, 1},
	{0x0e1700, 16},
	{0x0e1800, 3},
	{0x0e1810, 4},
	{0x0e1828, 3},
	{0x0e1848, 2},
	{0x0e1858, 2},
	{0x0e1864, 32},
	{0x0e18f0, 1},
	{0x0e18fc, 35},
	{0x0e199c, 15},
	{0x0e19e0, 1},
	{0x0e19e8, 5},
	{0x0e1a04, 5},
	{0x0e1a1c, 1},
	{0x0e1b00, 16},
	{0x0e1c00, 3},
	{0x0e1c10, 4},
	{0x0e1c28, 3},
	{0x0e1c48, 2},
	{0x0e1c58, 2},
	{0x0e1c64, 32},
	{0x0e1cf0, 1},
	{0x0e1cfc, 35},
	{0x0e1d9c, 15},
	{0x0e1de0, 1},
	{0x0e1de8, 5},
	{0x0e1e04, 5},
	{0x0e1e1c, 1},
	{0x0e1f00, 16},
	{0x0e20c0, 8},
	{0x0e20ec, 5},
	{0x0e2108, 3},
	{0x0e2200, 5},
	{0x0e2218, 36},
	{0x0e2300, 6},
	{0x0e2330, 4},
	{0x0e2500, 3},
	{0x0e2510, 12},
	{0x0e26e0, 6},
	{0x0e2700, 6},
	{0x0e2720, 6},
	{0x0e2740, 3},
	{0x0e2780, 6},
	{0x0e27a0, 6},
	{0x0e27c0, 3},
	{0x0e2800, 67},
	{0x0e2a00, 6},
	{0x0e2a20, 6},
	{0x0e2a40, 3},
	{0x0e2a50, 3},
	{0x0e2a60, 1},
	{0x0e2a80, 17},
	{0x0e3020, 10},
	{0x0e3070, 1},
	{0x0e3080, 2},
	{0x0e308c, 1},
	{0x0e3440, 21},
	{0x0e34e4, 13},
	{0x0e3520, 6},
	{0x0e3540, 6},
	{0x0e3560, 6},
	{0x0e3580, 6},
	{0x0e35a0, 6},
	{0x0e35c0, 6},
	{0x0e35e0, 6},
	{0x0e3600, 16},
	{0x0e3804, 3},
	{0x0e3900, 33},
	{0x0e3a00, 6},
	{0x0e3a20, 2},
	{0x0e3a30, 1},
	{0x0e3a40, 8},
	{0x0e3a64, 5},
	{0x0e3c00, 1},
	{0x0e3c28, 1},
	{0x0e3c50, 1},
	{0x0e3c78, 1},
	{0x0e3ca0, 2},
	{0x0e3dac, 1},
	{0x0e3dd8, 4},
	{0x0e3e00, 6},
	{0x0e3e20, 6},
	{0x0e3e40, 3},
	{0x0e4010, 12},
	{0x0e4044, 3},
	{0x0e4084, 2},
	{0x0e40bc, 84},
	{0x0e4240, 18},
	{0x0e45f0, 4},
	{0x0e4604, 1},
	{0x0e4640, 16},
	{0x0e46f0, 4},
	{0x0e4704, 1},
	{0x0e4740, 16},
	{0x0e5000, 8},
	{0x0e6000, 9},
	{0x0e6040, 16},
	{0x0e8000, 9},
	{0x0e8080, 6},
	{0x0e80a0, 3},
	{0x0f0000, 3},
	{0x0f0014, 11},
	{0x0f004c, 3},
	{0x0f0060, 8},
	{0x0f00f0, 3},
	{0x0f0100, 1},
	{0x0f010c, 2},
	{0x0f0118, 1},
	{0x0f0130, 4},
	{0x0f0180, 3},
	{0x0f0190, 2},
	{0x0f01a8, 1},
	{0x0f01c0, 2},
	{0x0f01d0, 10},
	{0x0f0200, 61},
	{0x0f0404, 9},
	{0x0f0440, 12},
	{0x0f0480, 5},
	{0x0f04b8, 21},
	{0x0f0520, 1},
	{0x0f0528, 1},
	{0x0f0540, 2},
	{0x0f0580, 4},
	{0x0f05a0, 1},
	{0x0f05c0, 8},
	{0x0f0800, 17},
	{0x0f0850, 9},
	{0x0f0880, 9},
	{0x0f08b0, 9},
	{0x0f08e0, 9},
	{0x0f0920, 4},
	{0x0f093c, 5},
	{0x0f095c, 5},
	{0x0f097c, 5},
	{0x0f099c, 5},
	{0x0f09bc, 5},
	{0x0f09dc, 1},
	{0x0f0a90, 2},
	{0x0f0c00, 128},
	{0x0f0e04, 1},
	{0x0f0e14, 9},
	{0x0f0e3c, 1},
	{0x0f1000, 3},
	{0x0f1010, 12},
	{0x0f1080, 10},
	{0x0f10c0, 1},
	{0x0f10e0, 2},
	{0x0f10ec, 1},
	{0x0f10f4, 3},
	{0x0f1400, 6},
	{0x0f1420, 6},
	{0x0f1440, 6},
	{0x0f1460, 6},
	{0x0f1480, 6},
	{0x0f14a0, 6},
	{0x0f14c0, 6},
	{0x0f14e0, 6},
	{0x0f1500, 6},
	{0x0f1520, 6},
	{0x0f1540, 6},
	{0x0f1560, 6},
	{0x0f1580, 6},
	{0x0f15a0, 6},
	{0x0f15c0, 6},
	{0x0f15e0, 6},
	{0x0f1600, 6},
	{0x0f1620, 3},
	{0x0f1800, 3},
	{0x0f1840, 4},
	{0x0f1854, 3},
	{0x0f1864, 3},
	{0x0f1874, 3},
	{0x0f2000, 2},
	{0x0f200c, 3},
	{0x0f2020, 10},
	{0x0f2060, 6},
	{0x0f2080, 2},
	{0x0f208c, 3},
	{0x0f20a0, 10},
	{0x0f20e0, 6},
	{0x0f2100, 2},
	{0x0f210c, 3},
	{0x0f2120, 10},
	{0x0f2160, 6},
	{0x0f2180, 2},
	{0x0f218c, 3},
	{0x0f21a0, 10},
	{0x0f21e0, 6},
	{0x0f2200, 2},
	{0x0f220c, 3},
	{0x0f2220, 10},
	{0x0f2260, 6},
	{0x0f2280, 2},
	{0x0f228c, 3},
	{0x0f22a0, 10},
	{0x0f22e0, 6},
	{0x0f2300, 2},
	{0x0f230c, 3},
	{0x0f2320, 10},
	{0x0f2360, 6},
	{0x0f2380, 2},
	{0x0f238c, 3},
	{0x0f23a0, 10},
	{0x0f23e0, 6},
	{0x0f2400, 2},
	{0x0f240c, 3},
	{0x0f2420, 10},
	{0x0f2460, 6},
	{0x0f2480, 2},
	{0x0f248c, 3},
	{0x0f24a0, 10},
	{0x0f24e0, 6},
	{0x0f2500, 2},
	{0x0f250c, 3},
	{0x0f2520, 10},
	{0x0f2560, 6},
	{0x0f2580, 2},
	{0x0f258c, 3},
	{0x0f25a0, 10},
	{0x0f25e0, 6},
	{0x0f2600, 2},
	{0x0f260c, 3},
	{0x0f2620, 10},
	{0x0f2660, 6},
	{0x0f2680, 2},
	{0x0f268c, 3},
	{0x0f26a0, 10},
	{0x0f26e0, 6},
	{0x0f2700, 2},
	{0x0f270c, 3},
	{0x0f2720, 10},
	{0x0f2760, 6},
	{0x0f2780, 2},
	{0x0f278c, 3},
	{0x0f27a0, 10},
	{0x0f27e0, 6},
	{0x0f2800, 2},
	{0x0f280c, 3},
	{0x0f2820, 10},
	{0x0f2860, 6},
	{0x0f2880, 2},
	{0x0f288c, 3},
	{0x0f28a0, 10},
	{0x0f28e0, 6},
	{0x0f2900, 2},
	{0x0f290c, 3},
	{0x0f2920, 10},
	{0x0f2960, 6},
	{0x0f2980, 2},
	{0x0f298c, 3},
	{0x0f29a0, 10},
	{0x0f29e0, 6},
	{0x0f4000, 7},
	{0x0f4020, 4},
	{0x0f4204, 1},
	{0x0f4280, 35},
	{0x0f4310, 4},
	{0x0f4404, 1},
	{0x0f4480, 34},
	{0x0f4510, 10},
	{0x0f453c, 3},
	{0x0f4800, 7},
	{0x0f4820, 4},
	{0x0f4a04, 1},
	{0x0f4a80, 35},
	{0x0f4b10, 4},
	{0x0f4c04, 1},
	{0x0f4c80, 34},
	{0x0f4d10, 10},
	{0x0f4d3c, 3},
	{0x0f5000, 7},
	{0x0f5020, 4},
	{0x0f5204, 1},
	{0x0f5280, 35},
	{0x0f5310, 4},
	{0x0f5404, 1},
	{0x0f5480, 34},
	{0x0f5510, 10},
	{0x0f553c, 3},
	{0x0f5800, 7},
	{0x0f5820, 4},
	{0x0f5a04, 1},
	{0x0f5a80, 35},
	{0x0f5b10, 4},
	{0x0f5c04, 1},
	{0x0f5c80, 34},
	{0x0f5d10, 10},
	{0x0f5d3c, 3},
	{0x0f6000, 7},
	{0x0f6020, 4},
	{0x0f6204, 1},
	{0x0f6280, 35},
	{0x0f6310, 4},
	{0x0f6404, 1},
	{0x0f6480, 34},
	{0x0f6510, 10},
	{0x0f653c, 3},
	{0x0f6800, 7},
	{0x0f6820, 4},
	{0x0f6a04, 1},
	{0x0f6a80, 35},
	{0x0f6b10, 4},
	{0x0f6c04, 1},
	{0x0f6c80, 34},
	{0x0f6d10, 10},
	{0x0f6d3c, 3},
	{0x100000, 1},
	{0x100008, 1},
	{0x100010, 2},
	{0x100020, 1},
	{0x100028, 1},
	{0x100030, 2},
	{0x100040, 1},
	{0x100048, 1},
	{0x100050, 2},
	{0x100060, 1},
	{0x100068, 1},
	{0x100070, 2},
	{0x100080, 21},
	{0x100100, 21},
	{0x100180, 21},
	{0x100200, 21},
	{0x100284, 1},
	{0x1003b0, 5},
	{0x100400, 13},
	{0x100440, 13},
	{0x100480, 13},
	{0x1004c0, 13},
	{0x100500, 68},
	{0x100618, 1},
	{0x100804, 1},
	{0x10080c, 4},
	{0x100820, 9},
	{0x1008a0, 24},
	{0x100920, 24},
	{0x100a00, 48},
	{0x100b00, 59},
	{0x100d00, 1},
	{0x100d08, 2},
	{0x100d80, 4},
	{0x100da0, 6},
	{0x100e00, 4},
	{0x100e20, 1},
	{0x100e28, 7},
	{0x100e48, 7},
	{0x100e68, 7},
	{0x100e88, 6},
	{0x100ee0, 6},
	{0x100f00, 6},
	{0x100f1c, 10},
	{0x100f70, 8},
	{0x100f94, 4},
	{0x100fc0, 6},
	{0x100fe0, 6},
	{0x101400, 16},
	{0x101444, 1},
	{0x10145c, 10},
	{0x101504, 1},
	{0x10151c, 30},
	{0x101600, 1},
	{0x101628, 6},
	{0x101648, 6},
	{0x101680, 16},
	{0x1016e0, 16},
	{0x101780, 1},
	{0x101790, 16},
	{0x101a00, 14},
	{0x101a40, 3},
	{0x101a50, 2},
	{0x101a60, 2},
	{0x101a70, 2},
	{0x101a80, 2},
	{0x101a90, 1},
	{0x101a9c, 11},
	{0x101b0c, 5},
	{0x101c00, 34},
	{0x101d00, 3},
	{0x102000, 1},
	{0x102028, 1},
	{0x102050, 1},
	{0x102078, 1},
	{0x1020a0, 7},
	{0x1021ac, 1},
	{0x1021d8, 4},
	{0x102200, 6},
	{0x102220, 6},
	{0x102240, 3},
	{0x102560, 1},
	{0x102584, 10},
	{0x1025b0, 1},
	{0x1025fc, 1},
	{0x102604, 1},
	{0x1026ec, 69},
	{0x103000, 32},
	{0x103084, 5},
	{0x1030f8, 3},
	{0x103108, 3},
	{0x103118, 7},
	{0x103144, 1},
	{0x103160, 10},
	{0x103200, 32},
	{0x103284, 5},
	{0x1032f8, 3},
	{0x103308, 3},
	{0x103318, 7},
	{0x103344, 1},
	{0x103360, 10},
	{0x103400, 32},
	{0x103484, 5},
	{0x1034f8, 3},
	{0x103508, 3},
	{0x103518, 7},
	{0x103544, 1},
	{0x103560, 10},
	{0x103600, 32},
	{0x103684, 5},
	{0x1036f8, 3},
	{0x103708, 3},
	{0x103718, 7},
	{0x103744, 1},
	{0x103760, 10},
	{0x103800, 1},
	{0x10380c, 1},
	{0x103a00, 64},
	{0x104000, 3},
	{0x104020, 3},
	{0x104040, 3},
	{0x104060, 3},
	{0x104084, 1},
	{0x104090, 4},
	{0x1040a4, 1},
	{0x1040b0, 4},
	{0x1040c4, 1},
	{0x1040d0, 4},
	{0x1040e4, 1},
	{0x1040f0, 21},
	{0x104148, 18},
	{0x1041f0, 6},
	{0x104308, 9},
	{0x104330, 1},
	{0x104340, 16},
	{0x1043b4, 4},
	{0x1043c8, 4},
	{0x1043dc, 4},
	{0x1043f0, 4},
	{0x104404, 1},
	{0x104470, 36},
	{0x104504, 1},
	{0x104570, 36},
	{0x104604, 1},
	{0x104670, 36},
	{0x104704, 1},
	{0x104770, 50},
	{0x104840, 2},
	{0x10484c, 1},
	{0x104900, 1},
	{0x104908, 1},
	{0x104984, 1},
	{0x1049a0, 24},
	{0x104a08, 6},
	{0x104a28, 6},
	{0x104a48, 6},
	{0x104a68, 6},
	{0x104a88, 6},
	{0x104aa8, 6},
	{0x104ac8, 6},
	{0x104ae8, 15},
	{0x104b40, 16},
	{0x104c00, 6},
	{0x104c20, 6},
	{0x104c40, 6},
	{0x104c60, 6},
	{0x104c80, 6},
	{0x104ca0, 6},
	{0x104cc0, 6},
	{0x104ce0, 6},
	{0x104d00, 3},
	{0x104d20, 6},
	{0x104d40, 6},
	{0x105000, 448},
	{0x105704, 3},
	{0x105734, 1},
	{0x106000, 62},
	{0x106100, 32},
	{0x106184, 1},
	{0x107010, 1},
	{0x110000, 4},
	{0x110014, 2},
	{0x110020, 5},
	{0x110040, 5},
	{0x110060, 6},
	{0x110080, 5},
	{0x110098, 1},
	{0x1100a0, 4},
	{0x1100b8, 8},
	{0x1100e0, 6},
	{0x110200, 4},
	{0x110214, 2},
	{0x110220, 5},
	{0x110240, 5},
	{0x110260, 6},
	{0x110280, 5},
	{0x110298, 1},
	{0x1102a0, 4},
	{0x1102b8, 8},
	{0x1102e0, 6},
	{0x110400, 4},
	{0x110414, 2},
	{0x110420, 5},
	{0x110440, 5},
	{0x110460, 6},
	{0x110480, 5},
	{0x110498, 1},
	{0x1104a0, 4},
	{0x1104b8, 8},
	{0x1104e0, 6},
	{0x110600, 4},
	{0x110614, 2},
	{0x110620, 5},
	{0x110640, 5},
	{0x110660, 6},
	{0x110680, 5},
	{0x110698, 1},
	{0x1106a0, 4},
	{0x1106b8, 8},
	{0x1106e0, 6},
	{0x110800, 21},
	{0x110880, 15},
	{0x1108c0, 3},
	{0x1108d0, 2},
	{0x110900, 1},
	{0x111000, 1},
	{0x111028, 1},
	{0x111050, 1},
	{0x111078, 1},
	{0x1110a0, 5},
	{0x1111ac, 1},
	{0x1111d8, 4},
	{0x111200, 6},
	{0x111220, 6},
	{0x111240, 3},
	{0x111400, 3},
	{0x111480, 9},
	{0x1114c0, 25},
	{0x111540, 25},
	{0x1115c0, 25},
	{0x111640, 17},
	{0x11168c, 1},
	{0x118000, 29},
	{0x118078, 4},
	{0x118090, 2},
	{0x1180a0, 7},
	{0x1180c0, 11},
	{0x118100, 14},
	{0x118140, 14},
	{0x118180, 61},
	{0x118278, 4},
	{0x118290, 2},
	{0x1182a0, 7},
	{0x1182c0, 11},
	{0x118300, 14},
	{0x118340, 14},
	{0x118380, 61},
	{0x118478, 4},
	{0x118490, 2},
	{0x1184a0, 7},
	{0x1184c0, 11},
	{0x118500, 14},
	{0x118540, 14},
	{0x118580, 61},
	{0x118678, 4},
	{0x118690, 2},
	{0x1186a0, 7},
	{0x1186c0, 11},
	{0x118700, 14},
	{0x118740, 14},
	{0x118780, 62},
	{0x11887c, 1},
	{0x118a00, 8},
	{0x118a24, 15},
	{0x118a64, 30},
	{0x118b00, 4},
	{0x118b20, 3},
	{0x118c00, 6},
	{0x118c40, 14},
	{0x118c80, 9},
	{0x118d00, 9},
	{0x118d2c, 1},
	{0x118d40, 3},
	{0x118d60, 1},
	{0x118d80, 3},
	{0x118e00, 2},
	{0x118e0c, 1},
	{0x118e14, 5},
	{0x118e2c, 1},
	{0x118e34, 5},
	{0x118e4c, 1},
	{0x118e54, 5},
	{0x118e6c, 1},
	{0x118e74, 5},
	{0x118e8c, 1},
	{0x118e94, 5},
	{0x118eac, 1},
	{0x118eb4, 3},
	{0x119000, 29},
	{0x119078, 4},
	{0x119090, 2},
	{0x1190a0, 7},
	{0x1190c0, 11},
	{0x119100, 14},
	{0x119140, 14},
	{0x119180, 61},
	{0x119278, 4},
	{0x119290, 2},
	{0x1192a0, 7},
	{0x1192c0, 11},
	{0x119300, 14},
	{0x119340, 14},
	{0x119380, 61},
	{0x119478, 4},
	{0x119490, 2},
	{0x1194a0, 7},
	{0x1194c0, 11},
	{0x119500, 14},
	{0x119540, 14},
	{0x119580, 61},
	{0x119678, 4},
	{0x119690, 2},
	{0x1196a0, 7},
	{0x1196c0, 11},
	{0x119700, 14},
	{0x119740, 14},
	{0x119780, 62},
	{0x11987c, 1},
	{0x119a00, 8},
	{0x119a24, 15},
	{0x119a64, 30},
	{0x119b00, 4},
	{0x119b20, 3},
	{0x119c00, 6},
	{0x119c40, 14},
	{0x119c80, 9},
	{0x119d00, 9},
	{0x119d2c, 1},
	{0x119d40, 3},
	{0x119d60, 1},
	{0x119d80, 3},
	{0x119e00, 2},
	{0x119e0c, 1},
	{0x119e14, 5},
	{0x119e2c, 1},
	{0x119e34, 5},
	{0x119e4c, 1},
	{0x119e54, 5},
	{0x119e6c, 1},
	{0x119e74, 5},
	{0x119e8c, 1},
	{0x119e94, 5},
	{0x119eac, 1},
	{0x119eb4, 3},
	{0x11a000, 29},
	{0x11a078, 4},
	{0x11a090, 2},
	{0x11a0a0, 7},
	{0x11a0c0, 11},
	{0x11a100, 14},
	{0x11a140, 14},
	{0x11a180, 61},
	{0x11a278, 4},
	{0x11a290, 2},
	{0x11a2a0, 7},
	{0x11a2c0, 11},
	{0x11a300, 14},
	{0x11a340, 14},
	{0x11a380, 61},
	{0x11a478, 4},
	{0x11a490, 2},
	{0x11a4a0, 7},
	{0x11a4c0, 11},
	{0x11a500, 14},
	{0x11a540, 14},
	{0x11a580, 61},
	{0x11a678, 4},
	{0x11a690, 2},
	{0x11a6a0, 7},
	{0x11a6c0, 11},
	{0x11a700, 14},
	{0x11a740, 14},
	{0x11a780, 62},
	{0x11a87c, 1},
	{0x11aa00, 8},
	{0x11aa24, 15},
	{0x11aa64, 30},
	{0x11ab00, 4},
	{0x11ab20, 3},
	{0x11ac00, 6},
	{0x11ac40, 14},
	{0x11ac80, 9},
	{0x11ad00, 9},
	{0x11ad2c, 1},
	{0x11ad40, 3},
	{0x11ad60, 1},
	{0x11ad80, 3},
	{0x11ae00, 2},
	{0x11ae0c, 1},
	{0x11ae14, 5},
	{0x11ae2c, 1},
	{0x11ae34, 5},
	{0x11ae4c, 1},
	{0x11ae54, 5},
	{0x11ae6c, 1},
	{0x11ae74, 5},
	{0x11ae8c, 1},
	{0x11ae94, 5},
	{0x11aeac, 1},
	{0x11aeb4, 3},
	{0x11b000, 29},
	{0x11b078, 4},
	{0x11b090, 2},
	{0x11b0a0, 7},
	{0x11b0c0, 11},
	{0x11b100, 14},
	{0x11b140, 14},
	{0x11b180, 61},
	{0x11b278, 4},
	{0x11b290, 2},
	{0x11b2a0, 7},
	{0x11b2c0, 11},
	{0x11b300, 14},
	{0x11b340, 14},
	{0x11b380, 61},
	{0x11b478, 4},
	{0x11b490, 2},
	{0x11b4a0, 7},
	{0x11b4c0, 11},
	{0x11b500, 14},
	{0x11b540, 14},
	{0x11b580, 61},
	{0x11b678, 4},
	{0x11b690, 2},
	{0x11b6a0, 7},
	{0x11b6c0, 11},
	{0x11b700, 14},
	{0x11b740, 14},
	{0x11b780, 62},
	{0x11b87c, 1},
	{0x11ba00, 8},
	{0x11ba24, 15},
	{0x11ba64, 30},
	{0x11bb00, 4},
	{0x11bb20, 3},
	{0x11bc00, 6},
	{0x11bc40, 14},
	{0x11bc80, 9},
	{0x11bd00, 9},
	{0x11bd2c, 1},
	{0x11bd40, 3},
	{0x11bd60, 1},
	{0x11bd80, 3},
	{0x11be00, 2},
	{0x11be0c, 1},
	{0x11be14, 5},
	{0x11be2c, 1},
	{0x11be34, 5},
	{0x11be4c, 1},
	{0x11be54, 5},
	{0x11be6c, 1},
	{0x11be74, 5},
	{0x11be8c, 1},
	{0x11be94, 5},
	{0x11beac, 1},
	{0x11beb4, 3},
	{0x11c000, 19},
	{0x11c050, 10},
	{0x11c080, 19},
	{0x11c0d0, 10},
	{0x11c100, 19},
	{0x11c150, 10},
	{0x11c180, 19},
	{0x11c1d0, 10},
	{0x11c200, 19},
	{0x11c250, 10},
	{0x11c280, 19},
	{0x11c2d0, 10},
	{0x11c300, 19},
	{0x11c350, 10},
	{0x11c380, 19},
	{0x11c3d0, 10},
	{0x11c400, 19},
	{0x11c450, 10},
	{0x11c480, 19},
	{0x11c4d0, 10},
	{0x11c500, 19},
	{0x11c550, 10},
	{0x11c580, 19},
	{0x11c5d0, 10},
	{0x11c600, 19},
	{0x11c650, 10},
	{0x11c680, 19},
	{0x11c6d0, 10},
	{0x11c700, 19},
	{0x11c750, 10},
	{0x11c780, 19},
	{0x11c7d0, 10},
	{0x11c800, 19},
	{0x11c860, 6},
	{0x11c884, 1},
	{0x11c894, 22},
	{0x11c900, 7},
	{0x11d000, 7},
	{0x11d020, 15},
	{0x11d060, 15},
	{0x11d0a0, 15},
	{0x11d0e0, 15},
	{0x11d120, 15},
	{0x11d160, 15},
	{0x11d1a0, 15},
	{0x11d1e0, 15},
	{0x11d220, 15},
	{0x11d260, 15},
	{0x11d2a0, 15},
	{0x11d2e0, 15},
	{0x11d320, 15},
	{0x11d360, 15},
	{0x11d3a0, 15},
	{0x11d3e0, 17},
	{0x11d428, 3},
	{0x11d440, 5},
	{0x11d480, 9},
	{0x11d4a8, 3},
	{0x11d4c0, 5},
	{0x11d500, 9},
	{0x11d528, 3},
	{0x11d540, 5},
	{0x11d580, 9},
	{0x11d5a8, 3},
	{0x11d5c0, 5},
	{0x11d600, 6},
	{0x11d620, 6},
	{0x11d640, 6},
	{0x11d660, 6},
	{0x11d680, 6},
	{0x11d6a0, 6},
	{0x11d6c0, 6},
	{0x11d6e0, 6},
	{0x11d700, 12},
	{0x11d734, 1},
	{0x11d73c, 4},
	{0x11d750, 4},
	{0x11d764, 1},
	{0x11d800, 102},
	{0x11da00, 3},
	{0x11da10, 1},
	{0x11da18, 2},
	{0x11da24, 7},
	{0x11da50, 4},
	{0x11da80, 3},
	{0x11da90, 1},
	{0x11da98, 2},
	{0x11daa4, 7},
	{0x11dad0, 4},
	{0x11db00, 3},
	{0x11db10, 1},
	{0x11db18, 2},
	{0x11db24, 7},
	{0x11db50, 4},
	{0x11db80, 3},
	{0x11db90, 1},
	{0x11db98, 2},
	{0x11dba4, 7},
	{0x11dbd0, 4},
	{0x11dc00, 17},
	{0x11e000, 72},
	{0x11e200, 72},
	{0x11e400, 72},
	{0x11e600, 72},
	{0x11e800, 6},
	{0x11e820, 6},
	{0x11e840, 6},
	{0x11e860, 6},
	{0x11e880, 6},
	{0x11e8a0, 6},
	{0x11e8c0, 6},
	{0x11e8e0, 6},
	{0x11e900, 6},
	{0x11e920, 1},
	{0x11ea00, 6},
	{0x11ea20, 6},
	{0x11ea40, 6},
	{0x11ea60, 6},
	{0x11ea80, 6},
	{0x11eaa0, 6},
	{0x11eac0, 6},
	{0x11eae0, 6},
	{0x11eb00, 6},
	{0x11eb20, 1},
	{0x11ec00, 6},
	{0x11ec20, 6},
	{0x11ec40, 6},
	{0x11ec60, 6},
	{0x11ec80, 6},
	{0x11eca0, 6},
	{0x11ecc0, 6},
	{0x11ece0, 6},
	{0x11ed00, 6},
	{0x11ed20, 1},
	{0x11ee00, 6},
	{0x11ee20, 6},
	{0x11ee40, 6},
	{0x11ee60, 6},
	{0x11ee80, 6},
	{0x11eea0, 6},
	{0x11eec0, 6},
	{0x11eee0, 6},
	{0x11ef00, 6},
	{0x11ef20, 1},
	{0x11f000, 14},
	{0x11f040, 2},
	{0x11f080, 14},
	{0x11f0c0, 2},
	{0x11f100, 14},
	{0x11f140, 2},
	{0x11f180, 14},
	{0x11f1c0, 2},
	{0x11f400, 17},
	{0x11f448, 5},
	{0x11f460, 3},
	{0x11f470, 3},
	{0x11f480, 13},
	{0x11f4b8, 1},
	{0x11f500, 17},
	{0x11f548, 5},
	{0x11f560, 3},
	{0x11f570, 3},
	{0x11f580, 13},
	{0x11f5b8, 1},
	{0x11f600, 17},
	{0x11f648, 5},
	{0x11f660, 3},
	{0x11f670, 3},
	{0x11f680, 13},
	{0x11f6b8, 1},
	{0x11f700, 17},
	{0x11f748, 5},
	{0x11f760, 3},
	{0x11f770, 3},
	{0x11f780, 13},
	{0x11f7b8, 1},
	{0x11f800, 8},
	{0x11f824, 15},
	{0x11f864, 15},
	{0x11f8a4, 15},
	{0x11f8e4, 30},
	{0x11f980, 10},
	{0x11f9ac, 1},
	{0x11f9b4, 5},
	{0x11f9cc, 1},
	{0x11f9d4, 5},
	{0x11f9ec, 1},
	{0x11f9f4, 13},
	{0x11fa80, 7},
	{0x11faa0, 5},
	{0x11fac0, 5},
	{0x11fae0, 4},
	{0x11fc04, 8},
	{0x11fc40, 13},
	{0x11fc80, 16},
	{0x11fd00, 6},
	{0x11fd20, 3},
	{0x11fd30, 3},
	{0x11fd40, 19},
	{0x11fdc0, 3} };

enum {
	IFC_MAX_RETRIES = 2048
};

enum {
	PCI_CAP_PTR = 0x34,
	PCI_HDR_SIZE = 0x40,
	PCI_EXT_SPACE_ADDR = 0xff,

	PCI_CTRL_OFFSET = 0x4,
	PCI_COUNTER_OFFSET = 0x8,
	PCI_SEMAPHORE_OFFSET = 0xc,

	PCI_ADDR_OFFSET = 0x10,
	PCI_ADDR_BIT_LEN = 30,

	PCI_DATA_OFFSET = 0x14,

	PCI_FLAG_BIT_OFFS = 31,

	PCI_SPACE_BIT_OFFS = 0,
	PCI_SPACE_BIT_LEN = 16,

	PCI_SIZE_VLD_BIT_OFFS = 28,
	PCI_SIZE_VLD_BIT_LEN = 1,

	PCI_STATUS_BIT_OFFS = 29,
	PCI_STATUS_BIT_LEN = 3,
};

enum{
	CX4_DEVID = 0x209,
	CX4LX_DEVID = 0x20b,
};

struct mlx5_mst_dump {
	void *on_demand_dump;
	u32 dump_size;
	u32 vsec_addr;
	/*sync mst dump */
	struct mutex lock;
};

#define MLX5_PROTECTED_CR_SPCAE_DOMAIN 0x6
#define MLX5_PROTECTED_CR_SCAN_CRSPACE 0x7

#define BAD_ACCESS_VAL 0xbadacce5

int mlx5_pciconf_set_addr_space(struct mlx5_core_dev *dev, u16 space)
{
	int ret = 0;
	u32 val;

	ret = pci_read_config_dword(dev->pdev,
				    dev->mst_dump->vsec_addr +
				    PCI_CTRL_OFFSET,
				    &val);
	if (ret)
		goto out;

	val = MLX5_MERGE(val, space, PCI_SPACE_BIT_OFFS, PCI_SPACE_BIT_LEN);
	ret = pci_write_config_dword(dev->pdev,
				     dev->mst_dump->vsec_addr +
				     PCI_CTRL_OFFSET,
				     val);
	if (ret)
		goto out;

	ret = pci_read_config_dword(dev->pdev,
				    dev->mst_dump->vsec_addr +
				    PCI_CTRL_OFFSET,
				    &val);
	if (ret)
		goto out;

	if (MLX5_EXTRACT(val, PCI_STATUS_BIT_OFFS, PCI_STATUS_BIT_LEN) == 0)
		return -EINVAL;

	if ((space == MLX5_PROTECTED_CR_SCAN_CRSPACE || space == MLX5_PROTECTED_CR_SPCAE_DOMAIN) &&
	    (!MLX5_EXTRACT(val, PCI_SIZE_VLD_BIT_OFFS, PCI_SIZE_VLD_BIT_LEN))) {
		mlx5_core_warn(dev, "Failed to get protected cr space size, valid bit not set");
		return -EINVAL;
	}

	return 0;
out:
	return ret;
}

int mlx5_pciconf_set_protected_addr_space(struct mlx5_core_dev *dev,
					  u32 *ret_space_size) {
	int ret;

	if (!ret_space_size)
		return -EINVAL;

	*ret_space_size = 0;

	ret = mlx5_pciconf_set_addr_space(dev, MLX5_PROTECTED_CR_SCAN_CRSPACE);
	if (ret) {
		ret = mlx5_pciconf_set_addr_space(dev, MLX5_PROTECTED_CR_SPCAE_DOMAIN);
		if (ret)
			return ret;
		dev->priv.health.crdump->space = MLX5_PROTECTED_CR_SPCAE_DOMAIN;
	} else {
		dev->priv.health.crdump->space = MLX5_PROTECTED_CR_SCAN_CRSPACE;
	}

	ret = pci_read_config_dword(dev->pdev,
				    dev->mst_dump->vsec_addr +
				    PCI_ADDR_OFFSET,
				    ret_space_size);
	if (ret) {
		mlx5_core_warn(dev, "Failed to get read protected cr space size");
		return ret;
	}

	*ret_space_size = MLX5_EXTRACT(*ret_space_size, 0, PCI_ADDR_BIT_LEN);

	return 0;
}

int mlx5_pciconf_cap9_sem(struct mlx5_core_dev *dev, int state)
{
	u32 counter = 0;
	int retries = 0;
	u32 lock_val;
	int ret;

	if (state == UNLOCK) {
		ret = pci_write_config_dword(dev->pdev,
					     dev->mst_dump->vsec_addr +
					     PCI_SEMAPHORE_OFFSET,
					     UNLOCK);
		if (ret)
			goto out;
	} else {
		do {
			if (retries > IFC_MAX_RETRIES)
				return -EBUSY;
			ret = pci_read_config_dword(dev->pdev,
						    dev->mst_dump->vsec_addr +
						    PCI_SEMAPHORE_OFFSET,
						    &lock_val);
			if (ret)
				goto out;
			if (lock_val) {
				retries++;
				usleep_range(1000, 2000);
				continue;
			}
			ret = pci_read_config_dword(dev->pdev,
						    dev->mst_dump->vsec_addr +
						    PCI_COUNTER_OFFSET,
						    &counter);
			if (ret)
				goto out;
			ret = pci_write_config_dword(dev->pdev,
						     dev->mst_dump->vsec_addr +
						     PCI_SEMAPHORE_OFFSET,
						     counter);
			if (ret)
				goto out;
			ret = pci_read_config_dword(dev->pdev,
						    dev->mst_dump->vsec_addr +
						    PCI_SEMAPHORE_OFFSET,
						    &lock_val);
			if (ret)
				goto out;
			retries++;
		} while (counter != lock_val);
	}
	return 0;
out:
	return ret;
}

static int mlx5_pciconf_wait_on_flag(struct mlx5_core_dev *dev,
				     u8 expected_val)
{
	int retries = 0;
	u32 flag;
	int ret;

	do {
		if (retries > IFC_MAX_RETRIES)
			return -EBUSY;
		ret = pci_read_config_dword(dev->pdev,
					    dev->mst_dump->vsec_addr +
					    PCI_ADDR_OFFSET,
					    &flag);
		flag = MLX5_EXTRACT(flag, PCI_FLAG_BIT_OFFS, 1);
		retries++;
		if ((retries & 0xf) == 0)
			usleep_range(1000, 2000);
	} while (flag != expected_val);
	return 0;
}

static int mlx5_pciconf_read(struct mlx5_core_dev *dev,
			     unsigned int offset,
			     u32 *data)
{
	u32 address;
	int ret;

	if (MLX5_EXTRACT(offset, 31, 1))
		return -EINVAL;
	address = MLX5_MERGE(offset, 0, PCI_FLAG_BIT_OFFS, 1);
	ret = pci_write_config_dword(dev->pdev,
				     dev->mst_dump->vsec_addr +
				     PCI_ADDR_OFFSET,
				     address);
	if (ret)
		goto out;
	ret = mlx5_pciconf_wait_on_flag(dev, 1);
	if (ret)
		goto out;
	ret = pci_read_config_dword(dev->pdev,
				    dev->mst_dump->vsec_addr +
				    PCI_DATA_OFFSET,
				    data);
out:
	return ret;
}

static int mlx5_pciconf_read_fast(struct mlx5_core_dev *dev,
				  unsigned int read_addr,
				  unsigned int *next_read_addr,
				  u32 *data)
{
	int ret;

	ret = mlx5_pciconf_read(dev, read_addr, data);
	if (ret)
		goto out;

	ret = pci_read_config_dword(dev->pdev,
				    dev->mst_dump->vsec_addr +
				    PCI_ADDR_OFFSET,
				    next_read_addr);
	if (ret)
		goto out;

	*next_read_addr = MLX5_EXTRACT(*next_read_addr, 0, PCI_ADDR_BIT_LEN);

	if (*next_read_addr <= read_addr)
		ret = EINVAL;
out:
	return ret;
}

static int mlx5_pciconf_write(struct mlx5_core_dev *dev,
			      unsigned int offset,
			      u32 data)
{
	u32 address;
	int ret;

	if (MLX5_EXTRACT(offset, 31, 1))
		return -EINVAL;

	/* Set flag to 0x1 */
	address = MLX5_MERGE(offset, 1, PCI_FLAG_BIT_OFFS, 1);

	ret = pci_write_config_dword(dev->pdev,
				     dev->mst_dump->vsec_addr +
				     PCI_DATA_OFFSET,
				     data);
	if (ret)
		goto out;

	ret = pci_write_config_dword(dev->pdev,
				     dev->mst_dump->vsec_addr +
				     PCI_ADDR_OFFSET,
				     address);
	if (ret)
		goto out;

	/* Wait for the flag to be cleared */
	ret = mlx5_pciconf_wait_on_flag(dev, 0);

out:
	return ret;
}

int mlx5_block_op_pciconf(struct mlx5_core_dev *dev,
			  unsigned int offset, u32 *data,
			  int length)
{
	int read = length;
	int i;

	if (length % 4)
		return -EINVAL;
	for (i = 0; i < length; i += 4) {
		if (mlx5_pciconf_read(dev, offset + i, &data[(i >> 2)])) {
			read = i;
			goto cleanup;
		}
	}
cleanup:
	return read;
}

int mlx5_block_op_pciconf_fast(struct mlx5_core_dev *dev,
			       u32 *data,
			       int length)
{
	unsigned int next_read_addr = 0;
	unsigned int read_addr = 0;
	int i;

	if (length % 4)
		return -EINVAL;

	for (i = 0; i < (length / 4); i++)
		data[i] = BAD_ACCESS_VAL;

	while (read_addr < length) {
		if (mlx5_pciconf_read_fast(dev, read_addr, &next_read_addr, &data[(read_addr >> 2)]))
			return read_addr;

		read_addr = next_read_addr;
	}
	return length;
}

static int mlx5_read_reg_dword(struct mlx5_core_dev *dev, u32 addr, u32 *data)
{
	int ret = 0;

	ret = pci_write_config_dword(dev->pdev, MLX5_ADDR_REG, addr);
	if (ret)
		return ret;
	ret = pci_read_config_dword(dev->pdev, MLX5_DATA_REG, data);
	return ret;
}

static int mlx5_block_op_pciconf_old(struct mlx5_core_dev *dev,
				     unsigned int offset, u32 *data, int length)
{
	int read = length;
	int i;

	if (length % 4)
		return -EINVAL;
	for (i = 0; i < length ; i += 4) {
		if (mlx5_read_reg_dword(dev, offset + i, &data[(i >> 2)])) {
			read = i;
			goto cleanup;
		}
	}
cleanup:
	return read;
}

static int mlx5_read4_block_new(struct mlx5_core_dev *dev,
				unsigned int offset, u32 *data, int length)
{
	return mlx5_block_op_pciconf(dev, offset, data, length);
}

static int mlx5_read4_block_old(struct mlx5_core_dev *dev,
				unsigned int offset,
				u32 *data,
				int length)
{
	return mlx5_block_op_pciconf_old(dev, offset, data, length);
}

static int mlx5_get_vendor_cap_addr(struct mlx5_core_dev *dev)
{
	int vend_cap;
	int ret;

	vend_cap = pci_find_capability(dev->pdev, CAP_ID);
	if (!vend_cap)
		return 0;
	dev->mst_dump->vsec_addr = vend_cap;
	ret = mlx5_pciconf_cap9_sem(dev, LOCK);
	if (ret) {
		mlx5_core_warn(dev, "pciconf_cap9_sem locking failure\n");
		return 0;
	}
	if (mlx5_pciconf_set_addr_space(dev, MLX5_CR_SPACE_DOMAIN))
		vend_cap = 0;
	ret = mlx5_pciconf_cap9_sem(dev, UNLOCK);
	if (ret)
		mlx5_core_warn(dev, "pciconf_cap9_sem unlocking failure\n");
	return vend_cap;
}

int mlx5_mst_capture(struct mlx5_core_dev *dev)
{
	unsigned long (*blocks)[2];
	void *mst_data = NULL;
	u32 total_len = 0;
	int block_count;
	int size = 0;
	u32 *mst_out;
	int ret = 0;
	u32 i = 0;
	u32 j = 0;
	u32 *data;
	u32 addr;
	u32 hwid;
	u32 rc;

	if (!dev->mst_dump)
		return -ENODEV;

	mutex_lock(&dev->mst_dump->lock);
	if (dev->mst_dump->vsec_addr) {
		ret = mlx5_pciconf_cap9_sem(dev, LOCK);
		if (ret)
			goto unlock;
		ret = mlx5_pciconf_set_addr_space(dev, MLX5_CR_SPACE_DOMAIN);
		if (ret)
			goto unlock;
		ret = mlx5_read4_block_new(dev, MLX5_HWID_ADDR, &hwid, 4);
		mlx5_pciconf_cap9_sem(dev, UNLOCK);
	} else {
		ret = mlx5_read_reg_dword(dev, MLX5_HWID_ADDR, &hwid);
		if (ret)
			goto unlock;
	}

	if (hwid == CX4LX_DEVID) {
		block_count = MLX5_NUM_MST_OFFSETS_MT4117;
		blocks = (unsigned long (*)[2]) mlx5_mst_dump_regs_mt4117;
		total_len = MLX5_MST_DUMP_SIZE_BYTES_MT4117;
	} else if (hwid == CX4_DEVID) {
		block_count = MLX5_NUM_MST_OFFSETS_MT4115;
		blocks = (unsigned long (*)[2])mlx5_mst_dump_regs_mt4115;
		total_len = MLX5_MST_DUMP_SIZE_BYTES_MT4115;
	} else {
		ret = -ENODEV;
		goto unlock;
	}

	mst_data = kcalloc(total_len, sizeof(u32), GFP_KERNEL);
	if (!mst_data) {
		ret = -ENOMEM;
		goto unlock;
	}
	mst_out = mst_data;
	if (dev->mst_dump->vsec_addr) {
		ret = mlx5_pciconf_cap9_sem(dev, LOCK);
		if (ret)
			goto unlock;
	}
	for (i = 0; i < block_count; i++) {
		data = kcalloc(blocks[i][1], sizeof(u32), GFP_KERNEL);
		if (!data) {
			ret = -ENOMEM;
			goto unlock;
		}
		if (dev->mst_dump->vsec_addr) {
			ret = mlx5_pciconf_set_addr_space(dev,
							  MLX5_CR_SPACE_DOMAIN);
			if (ret)
				goto unlock;
			rc = mlx5_read4_block_new(dev,
						  blocks[i][0],
						  (u32 *)data,
						  blocks[i][1] * sizeof(u32));
		} else {
			rc = mlx5_read4_block_old(dev,
						  blocks[i][0],
						  (u32 *)data,
						  blocks[i][1] * sizeof(u32));
		}
		if (blocks[i][1] * sizeof(u32) != rc) {
			kfree(data);
			ret = -EINVAL;
			goto unlock;
		}
		for (j = 0; j < blocks[i][1]; j++) {
			addr = blocks[i][0] + (j * sizeof(u32));
			*mst_out = addr;
			mst_out++;
			*mst_out = ((u32 *)data)[j];
			mst_out++;
			size += 2 * sizeof(u32);
		}
		kfree(data);
	}

	if (dev->mst_dump->vsec_addr)
		mlx5_pciconf_cap9_sem(dev, UNLOCK);

	kfree(dev->mst_dump->on_demand_dump);
	dev->mst_dump->on_demand_dump = mst_data;
	dev->mst_dump->dump_size = size;
	ret = size;
unlock:
	mutex_unlock(&dev->mst_dump->lock);
	if (ret < 0)
		kfree(mst_data);
	return ret;
}

u32 mlx5_mst_dump(struct mlx5_core_dev *dev, void *buff, u32 buff_sz)
{
	u32 copy_sz = 0;

	if (!dev->mst_dump)
		return copy_sz;

	mutex_lock(&dev->mst_dump->lock);
	if (dev->mst_dump->on_demand_dump) {
		copy_sz = min(buff_sz, dev->mst_dump->dump_size);
		memcpy(buff, dev->mst_dump->on_demand_dump, copy_sz);
	}
	mutex_unlock(&dev->mst_dump->lock);

	return copy_sz;
}

void mlx5_mst_free_capture(struct mlx5_core_dev *dev)
{
	mutex_lock(&dev->mst_dump->lock);
	kfree(dev->mst_dump->on_demand_dump);
	dev->mst_dump->on_demand_dump = NULL;
	dev->mst_dump->dump_size = 0;
	mutex_unlock(&dev->mst_dump->lock);
}

void mlx5_mst_dump_cleanup(struct mlx5_core_dev *dev)
{
	if (!dev->mst_dump)
		return;
	kfree(dev->mst_dump->on_demand_dump);
	dev->mst_dump->on_demand_dump = NULL;
	dev->mst_dump->dump_size = 0;
	kfree(dev->mst_dump);
}

int mlx5_mst_dump_init(struct mlx5_core_dev *dev)
{
	dev->mst_dump = kzalloc(sizeof(*dev->mst_dump), GFP_KERNEL);
	if (!dev->mst_dump)
		return -ENOMEM;
	dev->mst_dump->vsec_addr = mlx5_get_vendor_cap_addr(dev);
	mutex_init(&dev->mst_dump->lock);
	return 0;
}

static int mlx5_icmd_get_max_mailbox_sz(struct mlx5_core_dev *dev,
					int *max_sz)
{
	return mlx5_pciconf_read(dev, MLX5_ICMD_MAILBOX_SZ, max_sz);
}

static int mlx5_icmd_trigger(struct mlx5_core_dev *dev,
			     int opcode)
{
	union {
		struct mlx5_icmd_ctrl_bits ctrl;
		u32 ctrl_in;
		u32 ctrl_out;
	} u;
	int retries = 0;
	int ret;

	memset(&u.ctrl_in, 0, sizeof(u));

	u.ctrl.opcode = cpu_to_be16(opcode);
	u.ctrl.busy = 1;

	/* Write opcode to ctrl and set busy bit */
	ret = mlx5_pciconf_write(dev, MLX5_ICMD_CTRL, cpu_to_be32(u.ctrl_in));
	if (ret)
		goto out;

	/* Read back ctrl and wait for busy bit to be cleared by hardware */
	do {
		if (retries > IFC_MAX_RETRIES)
			return -EBUSY;

		ret = mlx5_pciconf_read(dev, MLX5_ICMD_CTRL, &u.ctrl_out);
		if (ret)
			goto out;

		u.ctrl_out = cpu_to_be32(u.ctrl_out);

		retries++;
		if ((retries & 0xf) == 0)
			usleep_range(1000, 2000);

	} while (u.ctrl.busy != 0);

	if (u.ctrl.status)
		return -EINVAL;

	return 0;
out:
	return ret;
}

static int mlx5_icmd_send(struct mlx5_core_dev *dev,
			  int opcode, void *mailbox, int dword_sz)
{
	u32 *mail_in = mailbox;
	u32 *mail_out = mailbox;
	int ret;
	int i;

	/* Write mailbox input */
	for (i = 0; i < dword_sz; i++) {
		ret = mlx5_pciconf_write(dev,
					 MLX5_ICMD_MAILBOX + i * 4,
					 cpu_to_be32(*mail_in++));

		if (ret)
			goto out;
	}

	/* Trigger the cmd */
	mlx5_icmd_trigger(dev, opcode);

	/* Read mailbox output */
	for (i = 0; i < dword_sz; i++) {
		*mail_out = 0;
		ret = mlx5_pciconf_read(dev,
					MLX5_ICMD_MAILBOX + i * 4,
					mail_out);

		if (ret)
			goto out;

		*mail_out = cpu_to_be32(*mail_out);
		mail_out++;
	}

out:
	return ret;
}

int mlx5_icmd_access_register(struct mlx5_core_dev *dev,
			      int reg_id,
			      int method,
			      void *io_buff,
			      u32 io_buff_dw_sz)
{
	union {
		struct mlx5_icmd_access_reg_input_bits mailbox_in;
		struct mlx5_icmd_access_reg_output_bits mailbox_out;
		u32 b[7];
	} u;

	u32 *data_in = io_buff;
	u32 *data_out = io_buff;
	int ret = 0;
	int max_len;
	int i;

	memset(u.b, 0, sizeof(u));

	if (!dev->mst_dump)
		return -ENODEV;

	if (!dev->mst_dump->vsec_addr)
		return -ENODEV;

	if (io_buff_dw_sz > MLX5_ICMD_ACCESS_REG_DATA_DW_SZ)
		return -EINVAL;

	u.mailbox_in.constant_1_2 =  cpu_to_be16(0x1 << 11 | 0x4);
	u.mailbox_in.register_id = cpu_to_be16(reg_id);
	u.mailbox_in.method = method;
	u.mailbox_in.constant_3 = 0x1;
	u.mailbox_in.len = cpu_to_be16(0x3 << 11 | 0x3);

	for (i = 0; i < io_buff_dw_sz; i++)
		u.mailbox_in.reg_data[i] = *data_in++;

	ret = mlx5_pciconf_cap9_sem(dev, LOCK);
	if (ret)
		goto out;

	ret = mlx5_pciconf_set_addr_space(dev, MLX5_ICMD_SPACE_DOMAIN);
	if (ret)
		goto unlock;

	ret = mlx5_icmd_get_max_mailbox_sz(dev, &max_len);
	if (ret)
		goto unlock;

	if (unlikely(max_len < sizeof(struct mlx5_icmd_access_reg_input_bits)))
		return -EINVAL;

	/* Send access_register cmd */
	ret = mlx5_icmd_send(dev, MLX5_ICMD_ACCESS_REG, u.b, sizeof(u) / 4);
	if (ret)
		goto unlock;

	if (u.mailbox_out.status ||
	    u.mailbox_out.register_id != cpu_to_be16(reg_id)) {
		ret = u.mailbox_out.status;
		goto unlock;
	}

	/* Copy the output, length field takes 10 bits and unit is dword */
	if (method == MLX5_ICMD_QUERY)
		memcpy(data_out, u.mailbox_out.reg_data,
		       ((cpu_to_be16(u.mailbox_out.len) & 0x7FF) - 1) * 4);

unlock:
	mlx5_pciconf_cap9_sem(dev, UNLOCK);
out:
	return ret;
}

