;; Terminal-independent keypad and function key bindings.
;; Copyright (C) 1988, 1989 Free Software Foundation, Inc.

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.


;; This file handles definitions for keypad and function keys.  It is
;; independent of terminals and also works with the X Window System.

;; These keys are handled by a two-level mapping.  From the user's point
;; of view, symbols representing keys (e.g. 'function-1) are mapped
;; to functions.  This mapping is done through a keymap called
;; function-keymap.  To specify the definition of a key, use
;; keypad-define as in (keypad-define 'function-1 'end-of-line).
;; This file handles the mapping of symbol to function.

;; The lower level mapping between a symbol and the actual escape character
;; sequence generated by the key which that symbol represents is
;; performed in the keyboard-specific files term/*.el.  When X Windows
;; are in use, keypad and function keys are rebound to generate the
;; same escape sequences, so that the same files may be used.

;; The symbol names for the various keys have been chosen to be in
;; accordance with the ISO key definitions used by the X window system.
;; To see all the key symbols, look at the file term/x-win.el.

(defun keypad-define (char definition)
  (let ((exist (assq char (cdr function-keymap))))
    (if exist
	(setcdr exist definition)
      (setcdr function-keymap
	      (cons (cons char definition)
		    (cdr function-keymap))))))

(keypad-define 'left-arrow 'backward-char)
(keypad-define 'right-arrow 'forward-char)
(keypad-define 'up-arrow 'previous-line)
(keypad-define 'down-arrow 'next-line)

(keypad-define 'select 'set-mark-command)
(keypad-define 'find 're-search-forward)
(keypad-define 'search 're-search-forward)
(keypad-define 'delete 'delete-char)
(keypad-define 'remove 'kill-region)
(keypad-define 'home 'move-to-window-line)
(keypad-define 'help 'help-for-help)
(keypad-define 'enter 'open-line)
(keypad-define 'undo 'undo)
(keypad-define 'do 'execute-extended-command)

;; Also, there can be shifted function keys with no special names.
;; For them, use names such as S-erase, M-erase.

(keypad-define 'function-1 'beginning-of-line)
(keypad-define 'function-2 'end-of-line)
(keypad-define 'function-3 'isearch-forward)
(keypad-define 'function-4 'kill-line)
;; function-5 and so on are also usable.

;; On some terminals, left-1 and so on are used, and right-1 and so on.

(keypad-define 'keypad-period 'delete-char)
(keypad-define 'keypad-0 'yank)
(keypad-define 'keypad-1 'backward-word)
(keypad-define 'keypad-3 'forward-word)
(keypad-define 'keypad-7 'backward-paragraph)
(keypad-define 'keypad-9 'forward-paragraph)

(keypad-define 'delete-char 'delete-char)
(keypad-define 'next-page 'scroll-up)
(keypad-define 'previous-page 'scroll-down)
(keypad-define 'clear-screen 'recenter)

;; Here are the standard command meanings we give to the various
;; function key names.  Because this file is loaded after the user's
;; init file, we are careful to avoid overriding any definitions
;; already stored in function-keymap by the init file or (less often)
;; by the terminal-specific term/*.el file.

(defun setup-terminal-keymap (map translations)
  "Set up keymap MAP to forward to function-keymap according to TRANSLATIONS.
TRANSLATIONS is an alist; each element of it looks like (FROMSTRING . TOCHAR).
For each such pair, we define the key sequence FROMSTRING in MAP
to forward to the definition of character TOCHAR in function-keymap.
\"Forwarding\" means that subsequent redefinition of TOCHAR in
function-keymap will be seen automatically in MAP as well.

This function is used by files term/*.el to set up the mapping from the
escape sequences sent by function keys on particular terminals (FROMSTRINGs)
into Emacs standard function key slots (TOCHARs).
Generally, MAP is a prefix keymap which will be attached to a key
that is the common prefix sent by all function keys (often ESC O or ESC [)."
  (while translations
    (define-key map (car (car translations))
      (cons function-keymap (cdr (car translations))))
    (setq translations (cdr translations))))

(defun function-key-sequence (char)
  "Return key sequence for function key that on this terminal
translates into slot CHAR in function-keymap.
Or return nil if there is none."
  (car (where-is-internal (cons function-keymap char) (current-local-map)
			  nil t)))

(provide 'keypad)
