;;; footnote-cyrillic.el --- Cyrillic lettering for footnote mode

;; Copyright (C) 1997 by Free Software Foundation, Inc.

;; Author: Steven L Baur <steve@altair.xemacs.org>
;; Keywords: mule, news, mail
;; Version: 0.17

;; This file is part of XEmacs.

;; XEmacs is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; XEmacs is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with XEmacs; see the file COPYING.  If not, write to the Free
;; Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
;; 02111-1307, USA.

;;; Synched up with: Not in FSF

;;; Commentary:

;; This file provides support for Cyrillic lettering in footnote mode.
;; It requires an Emacs or XEmacs with MULE internationalization.

;;; Change Log:

;; Apr-05-1997:	Replicate letters when footnote numbers hit the end of
;;		the alphabet.
;; Apr-18-1997:	Added regexp to the footnote style alist.

;;; Code:

(require 'footnote)

(defconst footnote-cyrillic-lower "-L"-A
  "Cyrillic lower case alphabet.")

(defconst footnote-cyrillic-lower-regexp
  (concat "[" footnote-cyrillic-lower "]")
  "Regexp for lower case cyrillic.")

(defconst footnote-cyrillic-upper "-LƴĳŸö˷"-A
  "Cyrillic upper case alphabet.")

(defconst footnote-cyrillic-upper-regexp
  (concat "[" footnote-cyrillic-upper "]")
  "Regexp for upper case cyrillic.")

(defun Footnote-cyrillic-lower (n)
  "Return a Cyrillic lower case letter.
Wrapping around the alphabet implies successive repetitions of letters."
  (let* ((ltr (mod (1- n) (length footnote-cyrillic-lower)))
	 (rep (/ (1- n) (length footnote-cyrillic-lower)))
	 (chr (char-to-string (aref footnote-cyrillic-lower ltr)))
	 rc)
    (while (>= rep 0)
      (setq rc (concat rc chr))
      (setq rep (1- rep)))
    rc))

(defun Footnote-cyrillic-upper (n)
  "Return a Cyrillic upper case letter.
Wrapping around the alphabet implies successive repetitions of letters."
  (let* ((ltr (mod (1- n) (length footnote-cyrillic-upper)))
	 (rep (/ (1- n) (length footnote-cyrillic-upper)))
	 (chr (char-to-string (aref footnote-cyrillic-upper ltr)))
	 rc)
    (while (>= rep 0)
      (setq rc (concat rc chr))
      (setq rep (1- rep)))
    rc))

(setq footnote-style-alist
      (append (list `(cyrillic-lower
		      Footnote-cyrillic-lower
		      ,footnote-cyrillic-lower-regexp)) footnote-style-alist))

(setq footnote-style-alist
      (append (list `(cyrillic-upper
		      Footnote-cyrillic-upper
		      ,footnote-cyrillic-upper-regexp)) footnote-style-alist))

(provide 'footnote-cyrillic)

;;; footnote-cyrillic.el ends here
