;;; xwem-clswi.el --- Client switching package.

;; Copyright (C) 2003 by Free Software Foundation, Inc.

;; Author: Zajcev Evgeny <zevlg@yandex.ru>
;; Keywords: xwem
;; X-CVS: $Id: xwem-clswi.el,v 1.2 2004/05/05 22:43:08 lg Exp $

;; This file is NOT part of XEmacs.

;; XWEM is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; XWEM is distributed in the hope that it will be useful, but WITHOUT
;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
;; or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
;; License for more details.

;; You should have received a copy of the GNU General Public License
;; along with XEmacs; see the file COPYING.  If not, write to the Free
;; Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
;; 02111-1307, USA.

;;; Synched up with: Not in FSF

;;; Commentary:

;; To use this package add something like:
;;
;;     (autoload 'xwem-clswi-enable "xwem-clswi" "Enable client switching." t)
;;     (add-hook 'xwem-load-hook 'xwem-clswi-enable)
;;
;; to your xwemrc.el.  It will install new bindings `H-]' for
;; switching to next client and `H-[' for switching to previous
;; client.

;;; Code:

(defgroup xwem-clswi nil
  "Group to customize clients switcher."
  :prefix "xwem-clswi-"
  :group 'xwem)

(defcustom xwem-clswi-beep-on-error nil
  "*Non-nil mean beep on any error."
  :type 'boolean
  :group 'xwem-clswi)

(defcustom xwem-clswi-show-info nil
  "*Non-nil mean show info about client in xwem minibuffer after switch."
  :type 'boolean
  :group 'xwem-clswi)

(defvar xwem-clswi-enabled nil 
  "Non-nil means that xwem-clswi is enabled.
Do not change this value directly, use `xwem-clswi-enable',
`xwem-clswi-disable' or `xwem-clswi-toggle' instead.")


(define-xwem-command xwem-clswi-enable ()
  "Enables xwem-clswi."
  (xwem-interactive)

  (unless xwem-clswi-enabled
    (if xwem-started
	(progn
	  (xwem-global-set-key (xwem-kbd "H-]") 'xwem-clswi-next)
	  (xwem-global-set-key (xwem-kbd "H-[") 'xwem-clswi-prev))

      (define-key xwem-global-map (xwem-kbd "H-]") 'xwem-clswi-next)
      (define-key xwem-global-map (xwem-kbd "H-[") 'xwem-clswi-prev))

    (setq xwem-clswi-enabled t)))

(define-xwem-command xwem-clswi-disable ()
  "Disables xwem-clswi, uninstalls default keybindings."
  (xwem-interactive)

  (when xwem-clswi-enabled
    (if xwem-started
	(progn
	  (xwem-global-set-key (xwem-kbd "H-]") nil)
	  (xwem-global-set-key (xwem-kbd "H-[") nil))

      (define-key xwem-global-map (xwem-kbd "H-]") nil)
      (define-key xwem-global-map (xwem-kbd "H-[") nil))

    (setq xwem-clswi-enabled nil)))

(define-xwem-command xwem-clswi-toggle ()
  "Toggle xwem-clswi."
  (xwem-interactive)

  (if xwem-clswi-enabled
      (xwem-clswi-enable)
    (xwem-clswi-disable)
  ))

(define-xwem-command xwem-clswi-next (arg &optional win)
  "Switch to ARG next client in WIN.
If WIN is ommited then in selected window."
  (xwem-interactive "p")

  (let* ((swin (or win (xwem-win-selected)))
	 (ccl (xwem-win-cl swin))	;currently selected client
	 (cls (xwem-win-make-cl-list swin))
	 cclinx num scl)
    (if cls
	(progn
	  (setq cclinx (- (length cls) (length (memq ccl cls))))
	  (setq num (% (+ cclinx arg) (length cls)))
	  (setq scl (nth (if (natnump num) num (+ (length cls) num)) cls))

	  (xwem-manda-manage scl)

	  (when xwem-clswi-show-info
	    (xwem-client-info scl)))

      (xwem-message (if xwem-clswi-beep-on-error 'warn 'warn-nobeep)
		    "No clients to switch."))))

(define-xwem-command xwem-clswi-prev (arg &optional win)
  "Switch to ARG previous client in WIN.
If WIN is ommited then in selected window."
  (xwem-interactive "p")

  (xwem-clswi-next (- arg) win))


(provide 'xwem-clswi)

;;; xwem-clswi.el ends here
