/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997	Tim Janik	<timj@psynet.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include	"rcs.h"
RCS_ID("$Id: builder.c,v 1.7 1997/08/19 02:05:08 timj Exp $")


#define		__builder_c__

#include	"builder.h"
#include	"widgets.h"
#include	"defines.h"
#include	"ufunc.h"
#include	"windows.h"
#include	"write_cond.txt"
#include	"misc.h"
#include	<time.h>
#include	<errno.h>



/* --- global variables --- */
FILE		*Template_Stream;
const	gchar	*Template_Name;
guint		Template_Line;
FILE		*Target_Stream;
const	gchar	*Target_Base;
const	gchar	*Target_Postfix;
gboolean	Target_Dynamic;
struct	tm	*Target_tm;
guint		ChBufL=DFL_BUFFER_SIZE*2;
gchar		ChBuf[DFL_BUFFER_SIZE*2+1];



/* --- defines --- */
#define		UNKNOWN_MACRO	"ExpandMacro: unknown makro"



/* --- static prototypes --- */
static	gboolean	builder_expand_macro_char	(const	gchar	macro);


/* --- functions --- */
gint
template_get_char	(gboolean	check_eof)
{
	register gint	Char;
	
	Char=fgetc(Template_Stream);
	
	if (check_eof && Char==EOF)
		g_error("%s:%d: unexpected EOF",
			Template_Name,
			Template_Line);
	
	if (Char=='\n')
		Template_Line++;
	
	return Char;
}


void
builder_put_string	(const	gchar	*string)
{
	if (string && fputs(string, Target_Stream)==EOF)
		g_warning("\nfputs(\"%s\",): %s",
			string,
			g_strerror(errno));
}


gboolean
builder_expand_macro	(const	gchar	*macro)
{
	ChBuf[0]=0;

	if (strlen(macro)==1)
		return builder_expand_macro_char(macro[0]);
	
	if (strcmp(macro,				"builder_short")==0)
		snprintf(ChBuf, ChBufL, "%s",
			to_downcase(PRGNAME_SHORT));
	else if (strcmp(macro,				"Builder_Short")==0)
		snprintf(ChBuf, ChBufL, "%s",
			to_DownCase(PRGNAME_SHORT));
	else if (strcmp(macro,				"BUILDER_SHORT")==0)
		snprintf(ChBuf, ChBufL, "%s",
			PRGNAME_SHORT);
	else if (strcmp(macro,				"builder_long")==0)
		snprintf(ChBuf, ChBufL, "%s",
			to_downcase(PRGNAME_LONG));
	else if (strcmp(macro,				"Builder_Long")==0)
		snprintf(ChBuf, ChBufL, "%s",
			to_DownCase(PRGNAME_LONG));
	else if (strcmp(macro,				"BUILDER_LONG")==0)
		snprintf(ChBuf, ChBufL, "%s",
			PRGNAME_LONG);
	else if (strcmp(macro,				"fname_base")==0)
		snprintf(ChBuf, ChBufL, "%s",
			Target_Base);
	else if (strcmp(macro,				"fname_postf")==0)
		if (Target_Postfix)
			snprintf(ChBuf, ChBufL, "%s",
				Target_Postfix);
		else
			ChBuf[0]=0;
	else if (strcmp(macro,				"fname")==0)
		if (Target_Postfix)
			snprintf(ChBuf, ChBufL, "%s.%s",
				Target_Base,
				Target_Postfix);
		else
			snprintf(ChBuf, ChBufL, "%s",
				Target_Base);
	else if (strcmp(macro,				"fstate")==0)
		if (Target_Dynamic)
			snprintf(ChBuf, ChBufL, "dynamic");
		else
			snprintf(ChBuf, ChBufL, "static");
	else
		return FALSE;
	
	FLUSH_TARGET();
	ChBuf[0]=0;
	
	return TRUE;
}


gboolean
builder_expand_macro_char	(const	gchar	macro)
{
	switch (macro) {
	case	'_':
		snprintf(ChBuf, ChBufL, "%c%s%c", '$', "Id", '$');
		break;
		
	case	'h':
		snprintf(ChBuf, ChBufL, "%02d", Target_tm->tm_hour);
		break;
		
	case	'm':
		snprintf(ChBuf, ChBufL, "%02d", Target_tm->tm_min);
		break;
		
	case	's':
		snprintf(ChBuf, ChBufL, "%02d", Target_tm->tm_sec);
		break;
		
	case	'd':
		snprintf(ChBuf, ChBufL, "%02d", Target_tm->tm_mday);
		break;
		
	case	'o':
		snprintf(ChBuf, ChBufL, "%02d", Target_tm->tm_mon+1);
		break;
		
	case	'y':
		snprintf(ChBuf, ChBufL, "%d", Target_tm->tm_year);
		break;
		
	case	'Y':
		snprintf(ChBuf, ChBufL, "%d", Target_tm->tm_year+1900);
		break;
		
	default:
		return FALSE;
		break;
	};
	
	FLUSH_TARGET();
	ChBuf[0]=0;
	
	return TRUE;
}


void
builder_expand_file	(builder_expand_macro_func	MacroFunc)
{
	register gchar	Char=~EOF;
	register guint	macro_index;
	
	g_assert(MacroFunc);
	
	macro_index=0;
	
	
	/* main processing loop, this is only little bit more than
	 * copying all the chars from the template file to the 
	 * target file.
	 * this does piping chars from the source into destination
	 * and if an '@' sign is read, it reads all chars up to the next
	 * '@' sign, these chars identify a macro - look at templates/macros.doc
	 * for a description on the macros
	 * special case is '@' sign + '#' + optional chars + '\n' this is
	 * treated as a comment
	*/
	do {
		static gchar	Char_String[2]="\0\0";
		static gchar	macro_buffer[MAX_MACRO_LEN];
		
		Char=template_get_char(FALSE);
		
		
		/* read macro
		*/
		if (Char==MACRO_IDENT) {
			
			Char=template_get_char(TRUE);
			
			if (Char!='#') {


				/* read the macro name
				*/
				while (Char!=MACRO_IDENT && macro_index<MAX_MACRO_LEN) {
					macro_buffer[macro_index++]=Char;
					Char=template_get_char(TRUE);
				}
				
				
				/* need this check to avoid segfault
				*/
				if (macro_index>=MAX_MACRO_LEN) {
					g_warning("%s\n%s:%d: readmacro: overflow after %d chars",
						FUNCNAME,
						Template_Name,
						Template_Line,
						macro_index);
					macro_index=0;
				}
				macro_buffer[macro_index]=0;
				
				
				/* expand macro
				*/
				if (macro_index>0) {
					if ( (!builder_expand_macro(macro_buffer)) &&
					     ( (builder_expand_macro==MacroFunc) ||
					       (!(*MacroFunc)(macro_buffer)) ) ) {
						g_warning("%s\n%s:%d: %s: `%s'",
							FUNCNAME,
							Template_Name,
							Template_Line,
							UNKNOWN_MACRO,
							macro_buffer);
					}
				} else {
					Char_String[0]=MACRO_IDENT;
					builder_put_string(Char_String);
				}
				macro_index=0;
			} else
				
				
				/* read up all chars until next '\n'
				 * this is for "@#.*\n" comments
				*/
				while (Char!='\n')
					Char=template_get_char(TRUE);
		} else if (Char!=EOF) {
		
		
			/* pass all other chars into destination file
			*/
			Char_String[0]=Char;
			builder_put_string(Char_String);
		}
	} while (Char!=EOF);
}


gchar*
builder_init		(const	gchar		*template_dir,
			 const	gchar		*template_name,
			 const	gchar		*target_dir,
			 const	gchar		*target_base,
			 const	gchar		*target_postfix,
			 gboolean		update,
			 gboolean		dynamic,
			 gboolean		truncate)
{
	static	 time_t		target_time=0;
	register gchar		*file_name;
	register guint		file_name_len;
	register int		save_errno;
	
	g_assert(template_dir);
	g_assert(template_name);
	g_assert(target_dir);
	g_assert(target_base);
	
	
	/* reset global vars and close open descriptors
	*/
	builder_reset();
	
	
	/* set global variables
	*/
	Target_Base=g_strdup(target_base);
	if (target_postfix)
		Target_Postfix=g_strdup(target_postfix);
	Template_Name=g_strdup(template_name);
	Template_Line=1;
	Target_Dynamic=dynamic;
	
	
	/* determine wether to write the file or not
	*/
	file_name_len=strlen	(target_dir)+1+strlen(Target_Base) +
				 ( Target_Postfix ? 1+strlen(Target_Postfix) : 0) +1;
	file_name=g_new(gchar, file_name_len+1);
	if (Target_Postfix)
		snprintf(file_name,
			file_name_len,
			"%s/%s.%s",
			target_dir,
			Target_Base,
			Target_Postfix);
	else
		snprintf(file_name,
			file_name_len,
			"%s/%s",
			target_dir,
			Target_Base);
	
	if ( ! WRITE_TARGET_OK(get_stat_mode(file_name)!=0, update, dynamic, truncate) ) {
		builder_reset();
		return NULL;
	}
	
	
	/* open destination
	*/
	if (!(Target_Stream=fopen(file_name, "w"))) {
		save_errno=errno;
		builder_reset();
		errno=save_errno;
		return file_name;
	} else
		g_free(file_name);
	
	
	/* open template
	*/
	file_name_len=strlen(template_dir)+1+strlen(Template_Name)+1;
	file_name=g_new(gchar, file_name_len+1);
	snprintf(file_name,
		file_name_len,
		"%s/%s",
		template_dir,
		Template_Name);
	if (!(Template_Stream=fopen(file_name, "r"))) {
		save_errno=errno;
		builder_reset();
		errno=save_errno;
		return file_name;
	} else
		g_free(file_name);
	
	
	/* setup creation time
	 * hm, don't know for sure wether localtime() returns a
	 * pointer to a static buffer
	*/
	target_time=time(NULL);
	Target_tm=localtime(&target_time);
	
	return NULL;
}


void
builder_reset	(void)
{
	if (Template_Stream)
		fclose(Template_Stream);
	
	if (Target_Stream)
		fclose(Target_Stream);
	
	if (Template_Name)
		g_free((gchar*)Template_Name);
	
	g_free((gchar*)Target_Base);
	
	g_free((gchar*)Target_Postfix);

	Template_Stream=NULL;
	Template_Name=NULL;
	Template_Line=0;
	
	Target_Stream=NULL;
	Target_Base=NULL;
	Target_Postfix=NULL;
	Target_Dynamic=FALSE;
	
	Target_tm=NULL;
}


void
builder_get_config	(gboolean	*dynamic_only,
			 gboolean	*skip_handler,
			 gboolean	*trunc_override)
{
	if (dynamic_only)
		*dynamic_only=GB_wCAST(toggle_button, Builder_Toggle_Update)->init_state;
	if (skip_handler)
		*skip_handler=GB_wCAST(toggle_button, Builder_Toggle_Handler)->init_state;
	if (trunc_override)
		*trunc_override=GB_wCAST(toggle_button, Builder_Toggle_Overwrite)->init_state;
}


void
builder_set_config	(gboolean	dynamic_only,
			 gboolean	skip_handler,
			 gboolean	trunc_override)
{
	GB_wCAST(toggle_button, Builder_Toggle_Update)->init_state=dynamic_only!=0;
	GB_wCAST(toggle_button, Builder_Toggle_Handler)->init_state=skip_handler!=0;
	GB_wCAST(toggle_button, Builder_Toggle_Overwrite)->init_state=trunc_override!=0;
}


void
builder_create_window	(GtkObject	*parent_object)
{
	
	
	/* only raise the window if it already exists
	*/
	if (Builder_Window->widget) {
		_gtk_widget_raise(Builder_Window->widget);
		return;
	}
	
	
	/* initialize widgets
	*/
	GB_wCAST(entry, Builder_Entry_Directory)->init_text=DFL_TARGET_DIR;
	
	
	/* build window
	*/
	gb_window_build(Builder_Window);
	
	
	/* the window should be destroyed
	 * if it's parent gets destroyed.
	 * OTOH, this "destroy" signal handler must be
	 * removed from parent if the window gets destroyed.
	*/
	if (parent_object) {
		object_connect_disc_handler (
			GTK_OBJECT(Builder_Button_Cancel->widget),
			"destroy",
			parent_object,
			gtk_signal_connect_object (
				parent_object,
				"destroy",
				GTK_SIGNAL_FUNC(gtk_widget_destroy),
				GTK_OBJECT(Builder_Window->widget)
			)
		);
	}
	
	
	/* connect signal handlers
	*/
	gb_window_connect(Builder_Window);
	
	
	/* finaly show the window
	*/
	gtk_widget_show(GTK_WIDGET(Builder_Window->widget));
}


void
sigh_builder_Build_clicked	(GtkWidget	*widget,
				 gpointer	func_data)
{
	register gchar		*directory;
	register gint		err;
	
	
	directory=gtk_entry_get_text(GTK_ENTRY(Builder_Entry_Directory->widget));
	
	if (!strlen(directory)) {
		message_create_window(GTK_OBJECT(widget), "Directory name has zero length!");
		return;
	}
	
	if (!S_ISDIR(get_stat_mode(directory)) && mkdir(directory, 0700)) {
	    	message_create_window	(GTK_OBJECT(widget),
	    				 "Failed to create directory:\n%s:\n%s",
	    				 g_strerror(errno),
	    				 directory);
		return;
	}
	
	
	err=Write_c_source(directory,
			GTK_TOGGLE_BUTTON(Builder_Toggle_Update->widget)->active,
			GTK_TOGGLE_BUTTON(Builder_Toggle_Handler->widget)->active,
			GTK_TOGGLE_BUTTON(Builder_Toggle_Overwrite->widget)->active,
			GB_SOURCE_MAIN | GB_SOURCE_MAIN_DESC);
	if (err)
		message_create_window(NULL, "Failed to build source:\n%s", g_strerror(err));
	
	gtk_widget_destroy(Builder_Window->widget);
}
