/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997	Tim Janik	<timj@psynet.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include	"rcs.h"
RCS_ID("$Id: editor.c,v 1.10 1997/08/18 03:31:36 timj Exp $")


#define		__editor_c__

#include	"editor.h"
#include	"gsi.h"
#include	"structures.h"
#include	"browser.h"
#include	"widgets.h"
#include	"windows.h"
#include	"widdata.h"
#include	"ufunc.h"
#include	"special_fields.h"
#include	"misc.h"
#include	"defines.h"
#include	"config.h"


/* --- structures --- */
typedef	struct	editor_wid_list_S	editor_wid_list_S;
struct	editor_wid_list_S {
	gb_wdat_base_S	*widget_data;
	GtkWidget	*Label_symbol_name;
	GtkWidget	*Entry_symbol_name;
	guint		object_fields;
	GtkWidget	**object_widgets;
	guint		linkage_fields;
	GtkWidget	**linkage_widgets;
#ifdef	GTK_BUGS_WORKAROUND
	GtkWidget	*NOTEBOOK;
#endif	GTK_BUGS_WORKAROUND
	GtkWidget	*List_signal_handler;
	GtkWidget	*Button_SigH_Add;
	GtkWidget	*Button_SigH_Change;
	GtkWidget	*Button_SigH_Delete;
	GtkWidget	*Entry_signal;
	GtkWidget	*Entry_func;
	GtkWidget	*Entry_sig_data;
	GtkWidget	*Check_CONNECT;
	GtkWidget	*Check_CONNECT_OBJECT;
	GtkWidget	*Check_CONNECT_AFTER;
	GtkWidget	*Check_CONNECT_DATA_POINTER;
};
typedef	struct	wd_link_S	wd_link_S;
struct	wd_link_S {
	gb_struct_type_E	type;
	GtkWidget		*list_win;
	GtkWidget		*entry;
	gb_wdat_base_S		*widget_data;
};



/* --- prototypes --- */
static	GtkWidget*
		Editor_field_create		(gb_any_S		*Struct,
						 GtkBox			*parent,
						 const gsi_field_info_S	*field_info);
static	gboolean
		Editor_field_apply		(gb_any_S		*Struct,
						 GtkWidget		*widget);
static	void	Editor_field_refresh		(gb_any_S		*Struct,
						 GtkWidget		*widget);
static	void	Editor_SigH_List_refresh	(gb_wdat_base_S		*WidDat,
						 editor_wid_list_S	*widget_list);
static	editor_wid_list_S*
		Editor_fields_create		(gb_wdat_base_S		*WidDat,
						 GtkNotebook		*notebook);
static	void	Editor_SigH_List_create		(gb_wdat_base_S		*WidDat,
						 GtkNotebook		*notebook,
						 editor_wid_list_S	*widget_list);
static	GtkWidget*
		Editor_SigH_List_update_item	(GtkWidget		*item,
						 gchar			*signal_name,
						 gchar			*func_name,
						 gchar			*data_name,
						 guint			connect_options);
static	void	SigH_Editor_Apply_clicked	(GtkWidget		*widget,
						 editor_wid_list_S	*widget_list);
static	void	SigH_Editor_wd_link_clicked	(GtkWidget		*widget,
						 gb_any_S		*Struct);
static	void	SigH_Editor_SigH_List_selection_changed	(
						 GtkWidget		*widget,
						 editor_wid_list_S	*widget_list);
static	void	SigH_Editor_SigH_SimpleList_selection_changed	(
						 GtkWidget		*list,
						 GtkWidget		*entry);
static	void	SigH_Editor_SigH_AddChange_clicked	(
						 GtkWidget		*widget,
						 editor_wid_list_S	*widget_list);
static	void	SigH_Editor_SigH_Delete_clicked	(GtkWidget		*widget,
						 editor_wid_list_S	*widget_list);
static	void	SigH_Editor_SigH_List_clicked	(GtkWidget		*widget,
						 editor_wid_list_S	*widget_list);



/* --- variables --- */
static	const gchar	*wd_link_key="wd_link_key";
static	const gchar	*field_info_key="field_info_key";



/* --- functions --- */
void
Editor_create	(gb_wdat_base_S	*WidDat)
{
	if (!GUBI_DATA(WidDat)->editor) {
		register editor_wid_list_S	*widget_list;
		
		
		/* add linkage if needed,
		 * FIXME: move this into another place
		*/
		if (!WidDat->linkage)
			widget_data_linkage_new(WidDat);
		
		
		/* clone window
		*/
		gb_widget_data_clone(GB_wCAST(base, Editor_Window));
		
		
		/* build window
		*/
		gb_window_build(GB_wCAST(window, Editor_Window->clone));
		
		
		/* set GtkWindow* in the editor field and reset to NULL
		 * if the window is destroyed
		*/
		GUBI_DATA(WidDat)->editor=Editor_Window->clone->widget;
		GB_NULLIFY_ON_DESTROY(Editor_Window->clone->widget, &GUBI_DATA(WidDat)->editor);
		
		
		/* create generic fields
		 * and free the malloced widget_list upon destroy
		 *
		 * we grab a v-box as container, create the notebook in it
		 * and call Editor_bla_create() with the notebook as parent
		*/
		widget_list=Editor_fields_create(WidDat, GTK_NOTEBOOK(Editor_Notebook->clone->widget));
		Editor_SigH_List_create(WidDat, GTK_NOTEBOOK(Editor_Notebook->clone->widget), widget_list);
#ifdef	GTK_BUGS_WORKAROUND
		widget_list->NOTEBOOK=Editor_Notebook->clone->widget;
#endif	GTK_BUGS_WORKAROUND
		widget_list->Label_symbol_name=Editor_Label_symbol_name->clone->widget;
		widget_list->Entry_symbol_name=Editor_Entry_symbol_name->clone->widget;
		GB_GFREE_ON_DESTROY(Editor_Window->clone->widget, widget_list->object_widgets);
		GB_GFREE_ON_DESTROY(Editor_Window->clone->widget, widget_list->linkage_widgets);
		GB_GFREE_ON_DESTROY(Editor_Window->clone->widget, widget_list);
		gtk_object_set_user_data(GTK_OBJECT(Editor_Window->clone->widget),
					 widget_list);
		
		
		/* connect SigH to buttons
		*/
		gtk_signal_connect(GTK_OBJECT(Editor_Button_Apply->clone->widget),
				   "clicked",
				   GTK_SIGNAL_FUNC(SigH_Editor_Apply_clicked),
				   widget_list);
		gtk_signal_connect_object(GTK_OBJECT(Editor_Button_Close->clone->widget),
				  "clicked",
				  GTK_SIGNAL_FUNC(gtk_widget_destroy),
				  GTK_OBJECT(Editor_Window->clone->widget));
	
	
		/* show window
		*/
		gtk_widget_show(Editor_Window->clone->widget);
		
		
		/* build up contents
		*/
		Editor_refresh(WidDat);
	} else
		_gtk_widget_raise(GUBI_DATA(WidDat)->editor);
}


GtkWidget*
Editor_field_create	(gb_any_S		*Struct,
			 GtkBox			*parent,
			 const gsi_field_info_S	*field_info)
{
	register GtkWidget		*widget;
	
	g_assert(Struct);
	g_assert(parent);
	g_assert(field_info);
	
	switch (field_info->field_type) {
		register GtkWidget		*box;
		register GtkWidget		*h_box;
		register GtkWidget		*label;
		register GtkWidget		*entry;
		register GtkWidget		*toggle;
		register GtkWidget		*button;
		register GtkWidget		*frame;
		register GtkWidget		*omenu;
		register GtkWidget		*menu;
		register GtkWidget		*menuitem;
		register wd_link_S		*wd_link;
		register guint			i;
	
	case	GSI_FIELD_STRUCT_LINK:
		frame=gtk_frame_new(to_UpCase(field_info->field_name));
		gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
		gtk_box_pack_start(parent, frame, FALSE, FALSE, 0);
		gtk_widget_show(frame);
		box=gtk_vbox_new(FALSE, 0);
		gtk_container_border_width(GTK_CONTAINER(box), 5);
		gtk_container_add(GTK_CONTAINER(frame), box);
		
		h_box=gtk_hbox_new(FALSE, 0);
		gtk_container_border_width(GTK_CONTAINER(h_box), 0);
		gtk_container_add(GTK_CONTAINER(box), h_box);
		label=gtk_label_new("Link ");
		gtk_box_pack_start(GTK_BOX(h_box), label, FALSE, FALSE, 0);
		gtk_widget_show(label);
		entry=gtk_entry_new();
		gtk_widget_set_sensitive(entry, FALSE);
		gtk_box_pack_end(GTK_BOX(h_box), entry, TRUE, TRUE, 0);
		gtk_widget_show(entry);
		gtk_widget_show(h_box);
		
		h_box=gtk_hbox_new(FALSE, 0);
		gtk_container_border_width(GTK_CONTAINER(h_box), 0);
		gtk_container_add(GTK_CONTAINER(box), h_box);
		button=gtk_button_new_with_label("List");
		gtk_box_pack_start(GTK_BOX(h_box), button, TRUE, TRUE, 0);
		wd_link=g_new(wd_link_S, 1);
		wd_link->type=GB_STRUCT_NONE;
		wd_link->list_win=NULL;
		wd_link->entry=entry;
		wd_link->widget_data=(gb_wdat_base_S*)gsi_field_get_value(Struct, field_info).v_struct_link;
		/* ERASE gtk_object_set_user_data(GTK_OBJECT(button->parent), wd_link); */
		gtk_object_set_data(GTK_OBJECT(button), wd_link_key, wd_link);
		GB_GFREE_ON_DESTROY(button->parent, wd_link);
		gtk_signal_connect(GTK_OBJECT(button),
				   "clicked",
				   GTK_SIGNAL_FUNC(SigH_Editor_wd_link_clicked),
				   Struct);
		gtk_widget_show(button);
		gtk_widget_show(h_box);
		gtk_widget_show(box);
		widget=button;
		break;
	
	case	GSI_FIELD_STRING:
	case	GSI_FIELD_INT:
	case	GSI_FIELD_FLOAT:
		box=gtk_hbox_new(FALSE, 10);
		gtk_container_border_width(GTK_CONTAINER(box), 0);
		gtk_box_pack_start(parent, box, FALSE, FALSE, 0);
		gtk_widget_show(box);
		label=gtk_label_new(to_UpCase(field_info->field_name));
		gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);
		gtk_widget_show(label);
		entry=gtk_entry_new();
		if (field_info->field_type==GSI_FIELD_STRING)
			gtk_widget_set_usize(entry, FIELD_LENGTH_STRING, 25);
		if (field_info->field_type==GSI_FIELD_INT)
			gtk_widget_set_usize(entry, FIELD_LENGTH_INT, 25);
		if (field_info->field_type==GSI_FIELD_FLOAT)
			gtk_widget_set_usize(entry, FIELD_LENGTH_FLOAT, 25);
		gtk_box_pack_end(GTK_BOX(box), entry, FALSE, TRUE, 0);
		gtk_widget_show(entry);
		widget=entry;
		break;
	
	case	GSI_FIELD_BOOLEAN:
		toggle=gtk_check_button_new_with_label(to_UpCase(field_info->field_name));
		/*gtk_widget_set_usize(toggle, 150, 25);*/
		gtk_box_pack_start(GTK_BOX(parent), toggle, FALSE, TRUE, 0);
		gtk_widget_show(toggle);
		widget=toggle;
		break;

	case	GSI_FIELD_BITS:
		frame=gtk_frame_new(to_UpCase(field_info->field_name));
		gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
		gtk_box_pack_start(parent, frame, FALSE, FALSE, 0);
		gtk_widget_show(frame);
		box=gtk_vbox_new(FALSE, 0);
		gtk_container_border_width(GTK_CONTAINER(box), 5);
		gtk_container_add(GTK_CONTAINER(frame), box);
		g_assert(field_info->field_func);
		for (i=1; i<=gsi_field_n_bits(field_info); i++) {
			toggle=gtk_check_button_new_with_label((gchar*)
				(* ((GsiUNum2StringFunc*)field_info->field_func) )
					(gsi_field_bit_value(field_info, i), TRUE));
			gtk_object_set_user_data(GTK_OBJECT(toggle), (gpointer)i);
			/*gtk_widget_set_usize(toggle, 150, 25);*/
			gtk_box_pack_start(GTK_BOX(box), toggle, TRUE, TRUE, 0);
			gtk_widget_show(toggle);
		}
		gtk_widget_show(box);
		widget=box;
		break;
	
	case	GSI_FIELD_ENUM:
		box=gtk_hbox_new(FALSE, 10);
		gtk_container_border_width(GTK_CONTAINER(box), 0);
		gtk_box_pack_start(parent, box, FALSE, FALSE, 0);
		gtk_widget_show(box);
		label=gtk_label_new(to_UpCase(field_info->field_name));
		gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);
		gtk_widget_show(label);
		omenu=gtk_option_menu_new();
		gtk_widget_set_usize(omenu, FIELD_LENGTH_ENUM, 25);
		menu=gtk_menu_new();
		g_assert(field_info->field_func);
		for (i=field_info->minimum.v_enum; i<=field_info->maximum.v_enum; i++) {
			menuitem=gtk_menu_item_new_with_label((gchar*)
						(* ((GsiUNum2StringFunc*)field_info->field_func) )
							(i, TRUE));
			gtk_object_set_user_data(GTK_OBJECT(menuitem), (gpointer)i);
			gtk_menu_append(GTK_MENU(menu), menuitem);
			gtk_widget_show(menuitem);
		}
		gtk_option_menu_set_menu(GTK_OPTION_MENU(omenu), menu);
		gtk_box_pack_end(GTK_BOX(box), omenu, FALSE, TRUE, 0);
		gtk_widget_show(omenu);
		widget=omenu;
		break;
	
	default:
		widget=NULL;
		g_assert_not_reached();
		break;
	}
	
	gtk_object_set_data(GTK_OBJECT(widget), field_info_key, (gpointer)field_info);
	
	return widget;
}


void
Editor_field_refresh	(gb_any_S	*Struct,
			 GtkWidget	*widget)
{
	register const gsi_field_info_S	*field_info;
	
	g_assert(GB_IS_STRUCT(Struct));
	
	g_assert((field_info=gtk_object_get_data(GTK_OBJECT(widget), field_info_key)));
	
	switch (field_info->field_type) {
		register GtkWidget	*menu;
		register GtkWidget	*menuitem;
		register GList		*list, *free_list;
		register wd_link_S	*wd_link;
		register gchar		*string;
		register gsi_base_S	*struct_p;
	
	case	GSI_FIELD_STRUCT_LINK:
		wd_link=gtk_object_get_data(GTK_OBJECT(widget), wd_link_key);
		struct_p=gsi_field_get_value(Struct, field_info).v_struct_link;
		if (struct_p) {
			g_assert(GB_IS_WIDDAT(struct_p));
			
			string=(gchar*)widget_data_symbol_name_get(GB_wCAST(base, struct_p));
		} else
			string="";
		gtk_entry_set_text(GTK_ENTRY(wd_link->entry), string);
		break;
	
	case	GSI_FIELD_STRING:
		string=gsi_field_get_value(Struct, field_info).v_string;
		gtk_entry_set_text(GTK_ENTRY(widget), string ? string : "");
		break;
	
	case	GSI_FIELD_INT:
	case	GSI_FIELD_FLOAT:
		string=(gchar*)gsi_field_get_value_as_string(Struct, field_info);
		gtk_entry_set_text(GTK_ENTRY(widget), string ? string : "");
		break;
	
	case	GSI_FIELD_BOOLEAN:
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(widget),
			gsi_field_get_value(Struct, field_info).v_boolean);
		break;

	case	GSI_FIELD_BITS:
		free_list=list=gtk_container_children(GTK_CONTAINER(widget));
		while (list) {
			register GtkWidget	*toggle;
			register guint32	bit_val;
			register guint		bit_num;
			
			g_assert((toggle=list->data));
			g_assert(GTK_IS_TOGGLE_BUTTON(toggle));
			
			g_assert((bit_num=(guint)gtk_object_get_user_data(GTK_OBJECT(toggle))));
			bit_val=gsi_field_bit_value(field_info, bit_num);
			
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(toggle),
				( gsi_field_get_value(Struct, field_info).v_bits &
				  bit_val ) !=0);
			
			list=list->next;
		}
		g_list_free(free_list);
		break;
	
	case	GSI_FIELD_ENUM:
		menu=gtk_option_menu_get_menu(GTK_OPTION_MENU(widget));
		menuitem=gtk_menu_get_active(GTK_MENU(menu));
		gtk_menu_set_active(GTK_MENU(menu),
			gsi_field_get_value(Struct, field_info).v_enum -
				field_info->minimum.v_enum);
		gtk_option_menu_set_menu(GTK_OPTION_MENU(widget), menu);
		if (menuitem!=gtk_menu_get_active(GTK_MENU(menu)))
			gtk_widget_draw(widget, NULL);
		break;
	
	default:
		g_assert_not_reached();
		break;
	}
}


gboolean
Editor_field_apply	(gb_any_S	*Struct,
			 GtkWidget	*widget)
{
	register const gsi_field_info_S	*field_info;
	register gsi_value_U		old_value;
	
	g_assert(GB_IS_STRUCT(Struct));
	
	g_assert((field_info=gtk_object_get_data(GTK_OBJECT(widget), field_info_key)));
	
	old_value=gsi_field_get_value(Struct, field_info);
	if (field_info->field_type==GSI_FIELD_STRING)
		old_value.v_string=g_strdup(old_value.v_string);
	
	switch (field_info->field_type) {
		register GtkWidget	*menu;
		register GtkWidget	*menuitem;
		register GList		*list, *free_list;
		register gsi_value_U	value;
		register wd_link_S	*wd_link;
	
	case	GSI_FIELD_FLOAT:
	case	GSI_FIELD_INT:
	case	GSI_FIELD_STRING:
		gsi_field_set_value_from_string(Struct, field_info, gtk_entry_get_text(GTK_ENTRY(widget)));
		break;
	
	case	GSI_FIELD_STRUCT_LINK:
		wd_link=gtk_object_get_data(GTK_OBJECT(widget), wd_link_key);
		value.v_struct_link=GB_CAST(any, wd_link->widget_data);
		gsi_field_set_value(Struct, field_info, value);
		break;
		
	case	GSI_FIELD_BOOLEAN:
		value.v_boolean=GTK_TOGGLE_BUTTON(widget)->active;
		gsi_field_set_value(Struct, field_info, value);
		break;

	case	GSI_FIELD_BITS:
		free_list=list=gtk_container_children(GTK_CONTAINER(widget));
		while (list) {
			register GtkWidget	*toggle;
			register guint		bit_num;
			
			g_assert((toggle=list->data));
			g_assert(GTK_IS_TOGGLE_BUTTON(toggle));
			
			g_assert((bit_num=(guint)gtk_object_get_user_data(GTK_OBJECT(toggle))));
			
			gsi_field_set_bit(Struct,
					  field_info,
					  bit_num,
					  GTK_TOGGLE_BUTTON(toggle)->active);
			
			list=list->next;
		}
		g_list_free(free_list);
		break;
	
	case	GSI_FIELD_ENUM:
		menu=gtk_option_menu_get_menu(GTK_OPTION_MENU(widget));
		menuitem=gtk_menu_get_active(GTK_MENU(menu));
		value.v_enum=(guint)gtk_object_get_user_data(GTK_OBJECT(menuitem));
		gsi_field_set_value(Struct, field_info, value);
		break;
	
	default:
		g_assert_not_reached();
		break;
	}
	
	
	/* return wether rebuilding is required
	*/
	if (GSI_FIELD_OPT(field_info, GB_OPT_REBUILD)) {
		if (field_info->field_type==GSI_FIELD_STRING) {
			register gchar		*name;
			
			name=gsi_field_get_value(Struct, field_info).v_string;
			if (old_value.v_string && name) {
				register gboolean rebuild;
				
				rebuild=strcmp(old_value.v_string, name)!=0;
				g_free(old_value.v_string);
				return rebuild;
			} else {
				g_free(old_value.v_string);
				return old_value.v_string!=name;
			}
		} else
			return  !gsi_values_equal(field_info->field_type,
						field_info->field_options,
						old_value,
						gsi_field_get_value(Struct, field_info));
	}
	
	if (field_info->field_type==GSI_FIELD_STRING)
		g_free(old_value.v_string);
	
	return FALSE;
}


void
Editor_SigH_List_create		(gb_wdat_base_S		*WidDat,
				 GtkNotebook		*notebook,
				 editor_wid_list_S	*widget_list)
{
	register GtkWidget	*frame;
	register GtkWidget	*box;
	register GtkWidget	*label;
	register gb_wdat_base_S	*EdBoxDat;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	g_assert(notebook);
	
	frame=gtk_frame_new(to_UpCase(gb_signal_handler_struct_name));
	gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_container_border_width(GTK_CONTAINER(frame), 5);
	gtk_widget_show (frame);
	
	box=gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(box), 5);
	gtk_container_add(GTK_CONTAINER(frame), box);
	gtk_widget_show(box);
	
	/* clone & build auxillary window
	*/
	gb_widget_data_clone(GB_wCAST(base, SigHandler_AuxWindow));
	gb_window_build(GB_wCAST(window, SigHandler_AuxWindow->clone));
	EdBoxDat=SigHandler_Box->clone;
	
	/* remove EdBox from SigHandler_AuxWindow
	 * connect EdBox to notebook
	 * destroy SigHandler_AuxWindow
	*/
	gtk_container_remove(GTK_CONTAINER(SigHandler_AuxWindow->clone->widget), EdBoxDat->widget);
	gtk_container_add(GTK_CONTAINER(box), EdBoxDat->widget);
	gtk_widget_destroy(GTK_WIDGET(SigHandler_AuxWindow->clone->widget));
	
	label=gtk_label_new("Signals" /* to_UpCase(gb_signal_handler_struct_name) */ );
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);
	
	widget_list->List_signal_handler=SigHandler_List->clone->widget;
	widget_list->Button_SigH_Add=SigHandler_Button_Add->clone->widget;
	widget_list->Button_SigH_Change=SigHandler_Button_Change->clone->widget;
	widget_list->Button_SigH_Delete=SigHandler_Button_Delete->clone->widget;
	widget_list->Entry_signal=SigHandler_Entry_signal->clone->widget;
	widget_list->Entry_func=SigHandler_Entry_function->clone->widget;
	widget_list->Entry_sig_data=SigHandler_Entry_func_data->clone->widget;
	widget_list->Check_CONNECT=SigHandler_Check_CONNECT->clone->widget;
	widget_list->Check_CONNECT_OBJECT=SigHandler_Check_CONNECT_OBJECT->clone->widget;
	widget_list->Check_CONNECT_AFTER=SigHandler_Check_CONNECT_AFTER->clone->widget;
	widget_list->Check_CONNECT_DATA_POINTER=SigHandler_Check_CONNECT_DATA_POINTER->clone->widget;
	
	/* connect SigH to list and buttons
	*/
	gtk_signal_connect(GTK_OBJECT(widget_list->List_signal_handler),
			   "selection_changed",
			   GTK_SIGNAL_FUNC(SigH_Editor_SigH_List_selection_changed),
			   widget_list);
	gtk_signal_connect(GTK_OBJECT(widget_list->Button_SigH_Add),
			   "clicked",
			   GTK_SIGNAL_FUNC(SigH_Editor_SigH_AddChange_clicked),
			   widget_list);
	gtk_signal_connect(GTK_OBJECT(widget_list->Button_SigH_Change),
			   "clicked",
			   GTK_SIGNAL_FUNC(SigH_Editor_SigH_AddChange_clicked),
			   widget_list);
	gtk_signal_connect(GTK_OBJECT(widget_list->Button_SigH_Delete),
			   "clicked",
			   GTK_SIGNAL_FUNC(SigH_Editor_SigH_Delete_clicked),
			   widget_list);
	gtk_signal_connect(GTK_OBJECT(SigHandler_Button_List->clone->widget),
			   "clicked",
			   GTK_SIGNAL_FUNC(SigH_Editor_SigH_List_clicked),
			   widget_list);
	
	
	/* the selection_changed handler of the list does some things
	 * that are required for initialization also, so we just call
	 * it (list doesn't have a selection yet).
	*/
	SigH_Editor_SigH_List_selection_changed(widget_list->List_signal_handler, widget_list);
}


void
Editor_SigH_List_refresh	(gb_wdat_base_S		*WidDat,
				 editor_wid_list_S	*widget_list)
{
	register guint		indx;
	register GtkWidget	*sig_list;
	
	g_assert(WidDat);
	g_assert(widget_list);
	
	sig_list=widget_list->List_signal_handler;
	
	gtk_list_clear_items(GTK_LIST(sig_list), 0, -1);
	
	indx=0;
	if (WidDat->signal_handler_stack) {
		while (WidDat->signal_handler_stack[indx].signal_name) {
			register GtkWidget	*item;
			
			item=gtk_list_item_new();
			Editor_SigH_List_update_item(
				item,
				(gchar*) WidDat->signal_handler_stack[indx].signal_name,
				(gchar*) WidDat->signal_handler_stack[indx].handler_func,
				(gchar*) WidDat->signal_handler_stack[indx].data.func_data,
				         WidDat->signal_handler_stack[indx].connect_options);
			
			gtk_container_add(GTK_CONTAINER(sig_list), item);
			gtk_widget_show(item);
			
			indx++;
		}
	}
}


GtkWidget*
Editor_SigH_List_update_item	(GtkWidget	*item,
				 gchar		*signal_name,
				 gchar		*func_name,
				 gchar		*data_name,
				 guint		connect_options)
{
	register GtkWidget	*box;
	register GtkWidget	*label;
	static	 gchar		*string;
	static	 gchar		buffer[16];
	
	g_assert(item);
	
	box=gtk_hbox_new(TRUE, 5);
	gtk_container_border_width(GTK_CONTAINER(box), 0);
	gtk_widget_show(box);
	
	label=gtk_label_new(signal_name);
	gtk_label_get(GTK_LABEL(label), &string);
	gtk_object_set_data(GTK_OBJECT(item),
			gb_signal_handler_field_info[GB_SIGH_SIGNAL_NAME_INDX].key,
			string);
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_container_add(GTK_CONTAINER(box), label);
	gtk_widget_show(label);
	
	label=gtk_label_new(func_name);
	gtk_label_get(GTK_LABEL(label), &string);
	gtk_object_set_data(GTK_OBJECT(item),
			gb_signal_handler_field_info[GB_SIGH_FUNC_NAME_INDX].key,
			 string);
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_container_add(GTK_CONTAINER(box), label);
	gtk_widget_show(label);
	
	label=gtk_label_new(data_name);
	gtk_label_get(GTK_LABEL(label), &string);
	gtk_object_set_data(GTK_OBJECT(item),
			gb_signal_handler_field_info[GB_SIGH_DATA_NAME_INDX].key,
	 		string);
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_container_add(GTK_CONTAINER(box), label);
	gtk_widget_show(label);
	
	sprintf(buffer, " %s %s %s %s ",
		connect_options & GB_CONNECT              ? "C" : "-",
		connect_options & GB_CONNECT_OBJECT       ? "O" : "-",
		connect_options & GB_CONNECT_AFTER        ? "A" : "-",
		connect_options & GB_CONNECT_DATA_POINTER ? "P" : "-");
	label=gtk_label_new(buffer);
	gtk_object_set_data(GTK_OBJECT(item),
			gb_signal_handler_field_info[GB_SIGH_CONNECT_OPTIONS_INDX].key,
			(gpointer)connect_options);
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_container_add(GTK_CONTAINER(box), label);
	gtk_widget_show(label);
	
	if (GTK_BIN(item)->child)
		gtk_widget_destroy(GTK_BIN(item)->child);
	gtk_container_add(GTK_CONTAINER(item), box);
	
	
	return item;
}


editor_wid_list_S*
Editor_fields_create		(gb_wdat_base_S	*WidDat,
				 GtkNotebook	*notebook)
{
	register editor_wid_list_S		*widget_list;
	register guint				f_indx;
	register const gsi_struct_info_S	*struct_info;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	g_assert(notebook);
	
	/* initialize widget_list structure
	*/
	widget_list=g_new0(editor_wid_list_S, 1);
	widget_list->widget_data=WidDat;
	widget_list->Label_symbol_name=NULL;
	widget_list->Entry_symbol_name=NULL;
	
	
	/* put the object fields into notebook pages
	*/
	struct_info=GUBI_DATA(WidDat)->struct_info;
	widget_list->object_fields=gsi_struct_n_fields(struct_info);
	widget_list->object_widgets=g_new0(GtkWidget*, widget_list->object_fields);
	f_indx=0;
	while (struct_info) {
		register GtkWidget	*frame;
		register GtkWidget	*box;
		register GtkWidget	*label;
		register guint		i;
		
		if (!struct_info->n_fields) {
			struct_info=struct_info->parent;
			continue;
		}
		
		frame=gtk_frame_new(to_UpCase(struct_info->struct_name));
		gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
		gtk_container_border_width(GTK_CONTAINER(frame), 5);
		/* FIXME: gtk_widget_set_usize(frame, 200, 150); */
		gtk_widget_show (frame);
		
		box=gtk_vbox_new(FALSE, 0);
		gtk_container_border_width(GTK_CONTAINER(box), 5);
		gtk_container_add(GTK_CONTAINER(frame), box);
		gtk_widget_show(box);
		
		for (i=0; i<struct_info->n_fields; i++)
			widget_list->object_widgets[f_indx++]=Editor_field_create(
						GB_CAST(any, WidDat),
						GTK_BOX(box),
						&struct_info->fields[i]);
		
		label=gtk_label_new(to_UpCase(struct_info->struct_name));
		gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);
		
		struct_info=struct_info->parent;
	}
	if (WidDat->linkage) {
		struct_info=gsi_struct_info(WidDat->linkage->type);
		widget_list->linkage_fields=gsi_struct_n_fields(struct_info);
		widget_list->linkage_widgets=g_new0(GtkWidget*, widget_list->linkage_fields);
	}
	f_indx=0;
	while (struct_info) {
		register GtkWidget	*frame;
		register GtkWidget	*box;
		register GtkWidget	*label;
		register guint		i;
		register gchar		*name;
		
		if (!struct_info->n_fields) {
			struct_info=struct_info->parent;
			continue;
		}
		
		name=g_strdup(to_UpCase(struct_info->struct_name));
		
		/* ugly code, but the linkage name looks nicer ;)
		*/
		i=0;
		while (name[i]) {
			if (name[i]=='_')
				name[i]=0;
			else
				i++;
		}
		
		frame=gtk_frame_new(name);
		gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
		gtk_container_border_width(GTK_CONTAINER(frame), 5);
		/* FIXME: gtk_widget_set_usize(frame, 200, 150); */
		gtk_widget_show (frame);
		
		box=gtk_vbox_new(FALSE, 0);
		gtk_container_border_width(GTK_CONTAINER(box), 5);
		gtk_container_add(GTK_CONTAINER(frame), box);
		gtk_widget_show(box);
		
		for (i=0; i<struct_info->n_fields; i++)
			widget_list->linkage_widgets[f_indx++]=Editor_field_create(
						GB_CAST(any, WidDat->linkage),
						GTK_BOX(box),
						&struct_info->fields[i]);
		
		label=gtk_label_new(name);
		gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);
		
		g_free(name);
		
		struct_info=struct_info->parent;
	}
	
	return widget_list;
}


void
Editor_refresh		(gb_wdat_base_S	*WidDat)
{
	register guint			i;
	register editor_wid_list_S	*widget_list;
	register gchar			*name;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	if (!GUBI_DATA(WidDat)->editor)
		return;
	
	g_assert((widget_list=
		  gtk_object_get_user_data(GTK_OBJECT(GUBI_DATA(WidDat)->editor))));
	
	name=NULL;
	if (GUBI_DATA(WidDat)->category & WID_CAT_AUXILLARY) {
		register const gchar	*s_name;
		register gchar		*prefix="Auxillary: ";
		register guint		len;
		
		s_name=widget_data_symbol_name_get(WidDat);
		len=strlen(prefix)+strlen(s_name)+1;
		name=g_new(gchar, len+1);
		snprintf(name, len, "%s%s", prefix, s_name);
	} else
		name=g_strdup(widget_data_symbol_name_get(WidDat));
	
	if ( 1 ) {
		register gchar	*s_name;
		register guint	len;
		
		s_name=name;
		len=strlen(s_name)+strlen(GUBI_DATA(WidDat)->struct_info->struct_name)+11+1;
		name=g_new(gchar, len+1);
		snprintf(name, len, "%s  [ %s ]",
			 s_name,
			 to_UpCase(GUBI_DATA(WidDat)->struct_info->struct_name));
		g_free(s_name);
	}
	
	gtk_label_set(GTK_LABEL(widget_list->Label_symbol_name), name);
	g_free(name); name=NULL;
	gtk_entry_set_text(GTK_ENTRY(widget_list->Entry_symbol_name),
			   (gchar*)widget_data_symbol_name_get(WidDat));
	
	
	for (i=0; i<widget_list->object_fields; i++)
		Editor_field_refresh(GB_CAST(any, WidDat), widget_list->object_widgets[i]);
	for (i=0; i<widget_list->linkage_fields; i++)
		Editor_field_refresh(GB_CAST(any, WidDat->linkage), widget_list->linkage_widgets[i]);
	Editor_SigH_List_refresh(WidDat, widget_list);

#ifdef	GTK_BUGS_WORKAROUND
/* whatfor?
 * on some of the notebook pages we're using GtkOptionMenus that are reset
 * by gtk_option_menu_set_menu(). this causes container resizing walking up
 * the widget tree and calls gtk_real_container_need_resize() wich maps a
 * child of a notebook no matter wether it is the current page or not.
 * so we simply map/unmap all notebook pages until we are back at the current.
*/
{
	register GList	*children;
	register guint	c;
	
	children=GTK_NOTEBOOK(widget_list->NOTEBOOK)->children;
	c=gtk_notebook_current_page(GTK_NOTEBOOK(widget_list->NOTEBOOK));
	
	while (children) {
		register GtkNotebookPage	*page;
		
		page=children->data;
		gtk_widget_unmap(page->child);
		
		children=children->next;
	}
	
	GTK_NOTEBOOK(widget_list->NOTEBOOK)->cur_page=NULL;
	gtk_notebook_set_page(GTK_NOTEBOOK(widget_list->NOTEBOOK), c);
}
#endif	/*GTK_BUGS_WORKAROUND*/
}


void
SigH_Editor_wd_link_clicked	(GtkWidget	*widget,
				 gb_any_S	*Struct)
{
	g_assert(Struct);
	
	if (GB_IS_WIDDAT(Struct)) {
		register wd_link_S	*wd_link;
		register const gsi_field_info_S	*field_info;
		register field_func_WD_LINK	field_func;
		
		g_assert((field_info=gtk_object_get_data(GTK_OBJECT(widget), field_info_key)));
		
		wd_link=gtk_object_get_data(GTK_OBJECT(widget), wd_link_key);
		
		g_assert(!wd_link->list_win);
		if ((field_func=(field_func_WD_LINK)field_info->field_func))
			wd_link->list_win=field_func(
						GB_wCAST(base, Struct),
						GTK_OBJECT(widget),
						&(wd_link->list_win),
						GTK_SIGNAL_FUNC(SigH_Editor_wd_link_clicked),
						wd_link,
						&(wd_link->widget_data));
		
		if (wd_link->list_win) {
			gtk_widget_set_sensitive(widget, FALSE);
			gtk_signal_connect_object(GTK_OBJECT(wd_link->list_win),
						  "destroy",
						  GTK_SIGNAL_FUNC(_gtk_widget_set_sensitivity),
						  GTK_OBJECT(widget));
		}
	} else {
		register wd_link_S	*wd_link;
		
		g_assert((wd_link=(wd_link_S*)Struct));
		
		gtk_entry_set_text(GTK_ENTRY(wd_link->entry),
			wd_link->widget_data ?
			  (gchar*)widget_data_symbol_name_get(wd_link->widget_data) :
			  "");
	}
}


void
SigH_Editor_Apply_clicked	(GtkWidget		*widget,
				 editor_wid_list_S	*widget_list)
{
	register guint			i;
	register gchar			*string;
	register GList			*list, *free_list;
	register gboolean		name_changed;
	register gboolean		need_rebuild;
	register gb_wdat_base_S		*WidDat;
	
	g_assert(widget_list);
	WidDat=widget_list->widget_data;
	g_assert(GB_IS_WIDDAT(WidDat));
	
	
	/* apply symbol name
	*/
	string=gtk_entry_get_text(GTK_ENTRY(widget_list->Entry_symbol_name));
	name_changed=strcmp(string, widget_data_symbol_name_get(WidDat))!=0;
	if (name_changed)
		widget_data_symbol_name_set(WidDat, string);
	
	
	/* apply new values to widget fields
	*/
	need_rebuild=FALSE;
	for (i=0; i<widget_list->object_fields; i++)
		need_rebuild|=Editor_field_apply(GB_CAST(any, WidDat), widget_list->object_widgets[i]);
	for (i=0; i<widget_list->linkage_fields; i++)
		need_rebuild|=Editor_field_apply(GB_CAST(any, WidDat->linkage), widget_list->linkage_widgets[i]);
	
	
	/* set the signal handlers
	*/
	free_list=list=gtk_container_children(GTK_CONTAINER(widget_list->List_signal_handler));
	widget_data_handler_stack_remove(WidDat);
	while (list) {
		register GtkObject	*item;
		
		item=list->data;
		g_assert(GTK_IS_LIST_ITEM(item));
		
		widget_data_handler_add(
			WidDat,
			(gchar*)gtk_object_get_data(item,
				gb_signal_handler_field_info[GB_SIGH_SIGNAL_NAME_INDX].key),
			(gchar*)gtk_object_get_data(item,
				gb_signal_handler_field_info[GB_SIGH_FUNC_NAME_INDX].key),
			(gchar*)gtk_object_get_data(item,
				gb_signal_handler_field_info[GB_SIGH_DATA_NAME_INDX].key),
			(guint)gtk_object_get_data(item,
				gb_signal_handler_field_info[GB_SIGH_CONNECT_OPTIONS_INDX].key));
		
		list=list->next;
	}
	g_list_free(free_list);
	
	
	/* check for linkage specific stuff
	*/
	widget_data_linkage_check(WidDat);
	if (widget_data_children_linkage_check(WidDat)) {
		register GList	*list;
		
		list=GUBI_DATA(WidDat)->children;
		while (list) {
			register gb_wdat_base_S	*ChildDat=list->data;
			
			g_assert(GB_IS_WIDDAT(ChildDat));
			
			Editor_refresh(ChildDat);
			
			list=list->next;
		}
	}
	
	
	/* realize values in gtkwidgets and rebuild widget tree if neccessary
	*/
	if (need_rebuild)
	{
		tree_rebuild(GUBI_DATA(WidDat)->tree);
		tree_show(GUBI_DATA(WidDat)->tree);
	} else
		gb_widget_data_set(WidDat);
	
	
	/* refresh editor
	*/
	Editor_refresh(WidDat);
	
	
	/* if the symbol name changed, update all list items
	 * referring to this WidDat and update widget lists of
	 * all parents and immediate children.
	*/
	if (name_changed) {
		register GList		*list;
		
		widget_data_items_update(WidDat);
		
		list=GUBI_DATA(WidDat)->children;
		while (list) {
			register gb_wdat_base_S	*ChildDat;
			
			ChildDat=list->data;
			if (GUBI_DATA(ChildDat)->browser)
				Browser_refresh(ChildDat, TRUE);
			list=list->next;
		}
		while (WidDat) {
			if (GUBI_DATA(WidDat)->browser)
				Browser_refresh(WidDat, TRUE);
			WidDat=WidDat->parent;
		}
	}
}


void
SigH_Editor_SigH_List_selection_changed	(GtkWidget		*widget,
					 editor_wid_list_S	*widget_list)
{
	register GList	*selection;
	register gchar	*signal_name;
	register guint	connect_options;
	register gchar	*func_name;
	register gchar	*data_name;
	
	g_assert(widget_list);
	
	selection=GTK_LIST(widget_list->List_signal_handler)->selection;
	
	
	/* set sensitivity of buttons
	*/
	gtk_widget_set_sensitive(GTK_WIDGET(widget_list->Button_SigH_Add), selection==NULL);
	gtk_widget_set_sensitive(GTK_WIDGET(widget_list->Button_SigH_Change), selection!=NULL);
	gtk_widget_set_sensitive(GTK_WIDGET(widget_list->Button_SigH_Delete), selection!=NULL);
	
	
	/* retrive the current items data
	*/
	if (selection) {
		register GtkObject	*item;
		
		item=selection->data;
		
		g_assert(GTK_IS_LIST_ITEM(item));
		
		signal_name=(gchar*)gtk_object_get_data(item,
				gb_signal_handler_field_info[GB_SIGH_SIGNAL_NAME_INDX].key);
		func_name=(gchar*)gtk_object_get_data(item,
				gb_signal_handler_field_info[GB_SIGH_FUNC_NAME_INDX].key);
		data_name=(gchar*)gtk_object_get_data(item,
				gb_signal_handler_field_info[GB_SIGH_DATA_NAME_INDX].key);
		connect_options=(guint)gtk_object_get_data(item,
				gb_signal_handler_field_info[GB_SIGH_CONNECT_OPTIONS_INDX].key);
	} else {
		return;
		
		signal_name="";
		func_name="";
		data_name="";
		connect_options=0;
	}
	
	
	/* set the current signal handler data in editing widgets
	*/
	gtk_entry_set_text(GTK_ENTRY(widget_list->Entry_signal), signal_name);
	gtk_entry_set_text(GTK_ENTRY(widget_list->Entry_func), func_name);
	gtk_entry_set_text(GTK_ENTRY(widget_list->Entry_sig_data), data_name);
	gtk_toggle_button_set_state(
		GTK_TOGGLE_BUTTON(widget_list->Check_CONNECT),
		connect_options & GB_CONNECT);
	gtk_toggle_button_set_state(
		GTK_TOGGLE_BUTTON(widget_list->Check_CONNECT_OBJECT),
		connect_options & GB_CONNECT_OBJECT);
	gtk_toggle_button_set_state(
		GTK_TOGGLE_BUTTON(widget_list->Check_CONNECT_AFTER),
		connect_options & GB_CONNECT_AFTER);
	gtk_toggle_button_set_state(
		GTK_TOGGLE_BUTTON(widget_list->Check_CONNECT_DATA_POINTER),
		connect_options & GB_CONNECT_DATA_POINTER);
}


void
SigH_Editor_SigH_AddChange_clicked	(GtkWidget		*widget,
					 editor_wid_list_S	*widget_list)
{
	register GList		*selection;
	register gchar		*signal_name;
	register guint		connect_options;
	register gchar		*func_name;
	register gchar		*data_name;
	register GtkWidget	*item;
	
	g_assert(widget_list);
	
	selection=GTK_LIST(widget_list->List_signal_handler)->selection;
	
	
	/* retrive the current signal handler data from editing widgets
	*/
	signal_name=gtk_entry_get_text(GTK_ENTRY(widget_list->Entry_signal));
	func_name=gtk_entry_get_text(GTK_ENTRY(widget_list->Entry_func));
	data_name=gtk_entry_get_text(GTK_ENTRY(widget_list->Entry_sig_data));
	connect_options=0;
	if (GTK_TOGGLE_BUTTON(widget_list->Check_CONNECT)->active)
		connect_options|=GB_CONNECT;
	if (GTK_TOGGLE_BUTTON(widget_list->Check_CONNECT_OBJECT)->active)
		connect_options|=GB_CONNECT_OBJECT;
	if (GTK_TOGGLE_BUTTON(widget_list->Check_CONNECT_AFTER)->active)
		connect_options|=GB_CONNECT_AFTER;
	if (GTK_TOGGLE_BUTTON(widget_list->Check_CONNECT_DATA_POINTER)->active)
		connect_options|=GB_CONNECT_DATA_POINTER;
	
	
	/* we abort if the values are not valid
	*/
	if (signal_name[0]==0 || func_name[0]==0 || data_name[0]==0) {
		SigH_Editor_SigH_List_selection_changed(widget_list->List_signal_handler, widget_list);
		return;
	}
	
	
	/* if an item is selected change it, if not add one
	*/
	if (selection) {
		item=selection->data;
		g_assert(GTK_IS_LIST_ITEM(item));
	} else
		item=gtk_list_item_new();
	Editor_SigH_List_update_item(item, signal_name, func_name, data_name, connect_options);
	if (!selection) {
		gtk_container_add(GTK_CONTAINER(widget_list->List_signal_handler), item);
		gtk_widget_show(item);
	}
}


void
SigH_Editor_SigH_Delete_clicked	(GtkWidget		*widget,
				 editor_wid_list_S	*widget_list)
{
	register GList	*list;
	
	g_assert(widget_list);
	
	list=GTK_LIST(widget_list->List_signal_handler)->selection;
	if (list) {
		register GList	*clear_list;
		
		g_assert(!list->next);
		
		clear_list=g_list_alloc();
		clear_list->data=list->data;
		
		gtk_list_remove_items(GTK_LIST(widget_list->List_signal_handler), clear_list);
		gtk_widget_destroy(clear_list->data);
		
		g_list_free(clear_list);
	}

}


void
SigH_Editor_SigH_SimpleList_selection_changed	(GtkWidget		*list,
						 GtkWidget		*entry)
{
	register GList	*selection;
	
	g_assert(list);
	g_assert(entry);
	
	selection=GTK_LIST(list)->selection;
	
	if (selection) {
		register GtkObject	*item;
		
		item=selection->data;
		g_assert(GTK_IS_LIST_ITEM(item));
		
		gtk_entry_set_text(GTK_ENTRY(entry), gtk_object_get_user_data(item));
	}
}


void
SigH_Editor_SigH_List_clicked	(GtkWidget		*button,
				 editor_wid_list_S	*widget_list)
{
	register GtkWidget	*SimpleList;
	register GList		*item_list;
	register GtkWidget	*item;
	register guint		i;
	register gint		pos;
	register gchar		*signal_name;
	
	g_assert(button);
	g_assert(widget_list);
	
	
	/* create list items
	*/
	signal_name=gtk_entry_get_text(GTK_ENTRY(widget_list->Entry_signal));
	pos=-1;
	item_list=NULL;
	for (i=0; i<GTK_OBJECT(widget_list->widget_data->widget)->klass->nsignals; i++) {
		static	 gchar	*tmp_signal_name;
		
		tmp_signal_name=gtk_signal_name(GTK_OBJECT(widget_list->widget_data->widget)->klass->signals[i]);
		item=gtk_list_item_new_with_label(tmp_signal_name);
		gtk_label_get(GTK_LABEL(GTK_BIN(item)->child), &tmp_signal_name);
		gtk_object_set_user_data(GTK_OBJECT(item), tmp_signal_name);
		gtk_widget_show(GTK_WIDGET(item));
		if (strcmp(signal_name, tmp_signal_name)==0)
			pos=i+1;
		
		item_list=g_list_prepend(item_list, item);
	}
	if (pos!=-1)
		pos=i-pos;
	
	
	/* pop up the signal_name list
	*/
	SimpleList=SimpleList_create	(GTK_OBJECT(button),
				 NULL,
				 GTK_SIGNAL_FUNC(SigH_Editor_SigH_SimpleList_selection_changed),
				 widget_list->Entry_signal,
				 FALSE,
				 item_list,
				 pos,
				 NULL,
				 TRUE,
				 "Select signal for pasting:");
	
	
	/* set our button to insensitive, but get sensitive again if the
	 * list is destroyed
	*/
	if (SimpleList) {
		gtk_widget_set_sensitive(button, FALSE);
		gtk_signal_connect_object(GTK_OBJECT(SimpleList),
					  "destroy",
					  GTK_SIGNAL_FUNC(_gtk_widget_set_sensitivity),
					  GTK_OBJECT(button));
	}
}
