/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997  Tim Janik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include	"RCS.h"
RCS_ID("$Id: editor.c,v 1.9 1997/05/20 21:17:50 tim Exp $")


#define		__editor_c__

#include	"editor.h"
#include	"browser.h"
#include	"widgets.h"
#include	"windows.h"
#include	"widdata.h"
#include	"ufunc.h"
#include	"structures.h"
#include	"special_fields.h"
#include	"misc.h"
#include	"defines.h"
#include	"config.h"


/* --- structures --- */
typedef	struct	editwid_widget_list_S	editwid_widget_list_S;
struct	editwid_widget_list_S {
	gb_wdat_base_S	*widget_data;
	GtkWidget	*Label_symbol_name;
	GtkWidget	*Entry_symbol_name;
	guint		object_fields;
	GtkWidget	**object_widgets;
	guint		base_fields;
	GtkWidget	**base_widgets;
	guint		misc_fields;
	GtkWidget	**misc_widgets;
	guint		container_fields;
	GtkWidget	**container_widgets;
	guint		bin_fields;
	GtkWidget	**bin_widgets;
	guint		linkage_fields;
	GtkWidget	**linkage_widgets;
	guint		specific_fields;
	GtkWidget	**specific_widgets;
};
typedef	struct	wd_link_S	wd_link_S;
struct	wd_link_S {
	gb_struct_type_E	type;
	GtkWidget		*list_win;
	GtkWidget		*entry;
	gb_wdat_base_S		*widget_data;
};



/* --- prototypes --- */
static	GtkWidget*
		Editor_field_create		(gb_any_S	*Struct,
						 GtkBox		*parent,
						 const field_info_S *FldInf);
static	gboolean
		Editor_field_apply		(gb_any_S	*Struct,
						 GtkWidget	*widget);
static	void	Editor_field_refresh		(gb_any_S	*Struct,
						 GtkWidget	*widget);
static	editwid_widget_list_S*
		Editor_fields_create		(gb_wdat_base_S	*WidDat,
						 GtkBox		*object_box,
						 GtkBox		*base_box,
						 GtkBox		*misc_box,
						 GtkBox		*container_box,
						 GtkBox		*bin_box,
						 GtkBox		*linkage_box,
						 GtkBox		*specific_box);
static	void	SigH_Editor_Apply_clicked	(GtkWidget	*widget,
						 gb_wdat_base_S	*WidDat);
static	void	SigH_Editor_wd_link_clicked	(GtkWidget	*widget,
						 gb_any_S	*Struct);
static	void	SigH_Editor_Expand_clicked	(GtkWidget	*widget,
						 GtkWidget	*Box);



/* --- variables --- */



/* --- functions --- */
void
Editor_create	(gb_wdat_base_S	*WidDat)
{
	if (!GUBI_DATA(WidDat)->editor) {
		register editwid_widget_list_S	*widget_list;
		
		
		/* add linkage if needed,
		 * FIXME: move this into another place
		*/
		if (!WidDat->linkage)
			widget_data_linkage_new(WidDat);
		
		
		/* clone window
		*/
		gb_widget_data_clone(GB_wCAST(base, Editor_Window));
		
		
		/* build window
		*/
		gb_window_build(GB_wCAST(window, Editor_Window->clone));
		
		
		/* set GtkWindow* in the editor field and reset to NULL
		 * if the window is destroyed
		*/
		GUBI_DATA(WidDat)->editor=Editor_Window->clone->widget;
		GB_NULLIFY_ON_DESTROY(Editor_Window->clone->widget, &GUBI_DATA(WidDat)->editor);
	
	
		/* create generic fields
		 * and free the malloced widget_list upon destroy
		*/
		widget_list=Editor_fields_create(WidDat,
			/* object */	GTK_BOX(Editor_Box_Base->clone->widget),
			/* base */	GTK_BOX(Editor_Box_Base->clone->widget),
			/* misc */	GTK_BOX(Editor_Box_Specific->clone->widget),
			/* container */	GTK_BOX(Editor_Box_Base->clone->widget),
			/* bin */	GTK_BOX(Editor_Box_Specific->clone->widget),
			/* linkage */	GTK_BOX(Editor_Box_Linkage->clone->widget),
			/* specific */	GTK_BOX(Editor_Box_Specific->clone->widget));
		widget_list->Label_symbol_name=Editor_Label_symbol_name->clone->widget;
		widget_list->Entry_symbol_name=Editor_Entry_symbol_name->clone->widget;
		GB_GFREE_ON_DESTROY(Editor_Window->clone->widget, widget_list->object_widgets);
		GB_GFREE_ON_DESTROY(Editor_Window->clone->widget, widget_list->base_widgets);
		GB_GFREE_ON_DESTROY(Editor_Window->clone->widget, widget_list->misc_widgets);
		GB_GFREE_ON_DESTROY(Editor_Window->clone->widget, widget_list->container_widgets);
		GB_GFREE_ON_DESTROY(Editor_Window->clone->widget, widget_list->bin_widgets);
		GB_GFREE_ON_DESTROY(Editor_Window->clone->widget, widget_list->linkage_widgets);
		GB_GFREE_ON_DESTROY(Editor_Window->clone->widget, widget_list->specific_widgets);
		GB_GFREE_ON_DESTROY(Editor_Window->clone->widget, widget_list);
		gtk_object_set_user_data(GTK_OBJECT(Editor_Window->clone->widget),
					 widget_list);
		
		
		/* set init states for box visibilities
		*/
		if (!widget_list->object_fields &&
		    !widget_list->base_fields &&
		    !widget_list->container_fields) {
			gtk_toggle_button_set_state(
				GTK_TOGGLE_BUTTON(Editor_Toggle_Base->clone->widget),
				FALSE);
			gtk_widget_set_sensitive(Editor_Toggle_Base->clone->widget, FALSE);
		}
		if (!widget_list->linkage_fields) {
			gtk_toggle_button_set_state(
				GTK_TOGGLE_BUTTON(Editor_Toggle_Linkage->clone->widget),
				FALSE);
			gtk_widget_set_sensitive(Editor_Toggle_Linkage->clone->widget, FALSE);
		}
		if (!widget_list->misc_fields &&
		    !widget_list->bin_fields &&
		    !widget_list->specific_fields) {
			gtk_toggle_button_set_state(
				GTK_TOGGLE_BUTTON(Editor_Toggle_Specific->clone->widget),
				FALSE);
			gtk_widget_set_sensitive(Editor_Toggle_Specific->clone->widget, FALSE);
		}
		if (!GTK_TOGGLE_BUTTON(Editor_Toggle_Base->clone->widget)->active)
			gtk_widget_hide(Editor_Frame_Base->clone->widget);
		if (!GTK_TOGGLE_BUTTON(Editor_Toggle_Linkage->clone->widget)->active)
			gtk_widget_hide(Editor_Frame_Linkage->clone->widget);
		if (!GTK_TOGGLE_BUTTON(Editor_Toggle_Specific->clone->widget)->active)
			gtk_widget_hide(Editor_Frame_Specific->clone->widget);
		
		
		/* connect SigH to buttons
		*/
		gtk_signal_connect(GTK_OBJECT(Editor_Toggle_Base->clone->widget),
				   "clicked",
				   GTK_SIGNAL_FUNC(SigH_Editor_Expand_clicked),
				   Editor_Frame_Base->clone->widget);
		gtk_signal_connect(GTK_OBJECT(Editor_Toggle_Linkage->clone->widget),
				   "clicked",
				   GTK_SIGNAL_FUNC(SigH_Editor_Expand_clicked),
				   Editor_Frame_Linkage->clone->widget);
		gtk_signal_connect(GTK_OBJECT(Editor_Toggle_Specific->clone->widget),
				   "clicked",
				   GTK_SIGNAL_FUNC(SigH_Editor_Expand_clicked),
				   Editor_Frame_Specific->clone->widget);
		gtk_signal_connect(GTK_OBJECT(Editor_Button_Apply->clone->widget),
				   "clicked",
				   GTK_SIGNAL_FUNC(SigH_Editor_Apply_clicked),
				   WidDat);
		gtk_signal_connect_object(GTK_OBJECT(Editor_Button_Close->clone->widget),
				  "clicked",
				  GTK_SIGNAL_FUNC(gtk_widget_destroy),
				  GTK_OBJECT(Editor_Window->clone->widget));
	
	
		/* show window
		*/
		gtk_widget_show(Editor_Window->clone->widget);
		
		
		/* build up contents
		*/
		Editor_refresh(WidDat);
	} else
		_gtk_widget_raise(GUBI_DATA(WidDat)->editor);
}


GtkWidget*
Editor_field_create	(gb_any_S		*Struct,
			 GtkBox			*parent,
			 const	field_info_S	*FldInf)
{
	register GtkWidget		*widget;
	
	g_assert(GB_IS_STRUCT(Struct));
	g_assert(parent);
	g_assert(FldInf);
	
	switch (FldInf->field_type) {
		register GtkWidget		*box;
		register GtkWidget		*h_box;
		register GtkWidget		*label;
		register GtkWidget		*entry;
		register GtkWidget		*toggle;
		register GtkWidget		*button;
		register GtkWidget		*frame;
		register GtkWidget		*omenu;
		register GtkWidget		*menu;
		register GtkWidget		*menuitem;
		register wd_link_S		*wd_link;
		register guint			i;
	
	case	FIELD_WD_LINK:
		frame=gtk_frame_new(to_UpCase(FldInf->field_name));
		gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
		gtk_box_pack_start(parent, frame, FALSE, FALSE, 0);
		gtk_widget_show(frame);
		box=gtk_vbox_new(FALSE, 0);
		gtk_container_border_width(GTK_CONTAINER(box), 5);
		gtk_container_add(GTK_CONTAINER(frame), box);
		
		h_box=gtk_hbox_new(FALSE, 0);
		gtk_container_border_width(GTK_CONTAINER(h_box), 0);
		gtk_container_add(GTK_CONTAINER(box), h_box);
		label=gtk_label_new("Link ");
		gtk_box_pack_start(GTK_BOX(h_box), label, FALSE, FALSE, 0);
		gtk_widget_show(label);
		entry=gtk_entry_new();
		gtk_widget_set_sensitive(entry, FALSE);
		gtk_box_pack_end(GTK_BOX(h_box), entry, TRUE, TRUE, 0);
		gtk_widget_show(entry);
		gtk_widget_show(h_box);
		
		h_box=gtk_hbox_new(FALSE, 0);
		gtk_container_border_width(GTK_CONTAINER(h_box), 0);
		gtk_container_add(GTK_CONTAINER(box), h_box);
		button=gtk_button_new_with_label("List");
		gtk_box_pack_start(GTK_BOX(h_box), button, TRUE, TRUE, 0);
		wd_link=g_new(wd_link_S, 1);
		wd_link->type=GB_STRUCT_NONE;
		wd_link->list_win=NULL;
		wd_link->entry=entry;
		wd_link->widget_data=structure_field_get_value(Struct, FldInf).link;
		gtk_object_set_user_data(GTK_OBJECT(button->parent), wd_link);
		GB_GFREE_ON_DESTROY(button->parent, wd_link);
		gtk_signal_connect(GTK_OBJECT(button),
				   "clicked",
				   GTK_SIGNAL_FUNC(SigH_Editor_wd_link_clicked),
				   Struct);
		gtk_widget_show(button);
		gtk_widget_show(h_box);
		gtk_widget_show(box);
		widget=button;
		break;
	
	case	FIELD_POINTER:
	case	FIELD_STRING:
	case	FIELD_INT:
	case	FIELD_FLOAT:
		box=gtk_hbox_new(FALSE, 10);
		gtk_container_border_width(GTK_CONTAINER(box), 0);
		gtk_box_pack_start(parent, box, FALSE, FALSE, 0);
		gtk_widget_show(box);
		label=gtk_label_new(to_UpCase(FldInf->field_name));
		gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);
		gtk_widget_show(label);
		entry=gtk_entry_new();
		if (FldInf->field_type==FIELD_POINTER)
			gtk_widget_set_usize(entry, FIELD_LENGTH_POINTER, 25);
		if (FldInf->field_type==FIELD_STRING)
			gtk_widget_set_usize(entry, FIELD_LENGTH_STRING, 25);
		if (FldInf->field_type==FIELD_INT)
			gtk_widget_set_usize(entry, FIELD_LENGTH_INT, 25);
		if (FldInf->field_type==FIELD_FLOAT)
			gtk_widget_set_usize(entry, FIELD_LENGTH_FLOAT, 25);
		gtk_box_pack_end(GTK_BOX(box), entry, FALSE, TRUE, 0);
		gtk_widget_show(entry);
		widget=entry;
		break;
	
	case	FIELD_BOOLEAN:
		toggle=gtk_check_button_new_with_label(to_UpCase(FldInf->field_name));
		/*gtk_widget_set_usize(toggle, 150, 25);*/
		gtk_box_pack_start(GTK_BOX(parent), toggle, TRUE, TRUE, 0);
		gtk_widget_show(toggle);
		widget=toggle;
		break;

	case	FIELD_BITS:
		frame=gtk_frame_new(to_UpCase(FldInf->field_name));
		gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
		gtk_box_pack_start(parent, frame, FALSE, FALSE, 0);
		gtk_widget_show(frame);
		box=gtk_vbox_new(FALSE, 0);
		gtk_container_border_width(GTK_CONTAINER(box), 5);
		gtk_container_add(GTK_CONTAINER(frame), box);
		for (i=1; i<=structure_field_bits_count(FldInf); i++) {
			toggle=gtk_check_button_new_with_label((gchar*)
				(* ((enum_field_func)FldInf->field_func) )
					(structure_field_bit_value(FldInf, i), TRUE));
			gtk_object_set_user_data(GTK_OBJECT(toggle), (gpointer)i);
			/*gtk_widget_set_usize(toggle, 150, 25);*/
			gtk_box_pack_start(GTK_BOX(box), toggle, TRUE, TRUE, 0);
			gtk_widget_show(toggle);
		}
		gtk_widget_show(box);
		widget=box;
		break;
	
	case	FIELD_ENUM:
		box=gtk_hbox_new(FALSE, 10);
		gtk_container_border_width(GTK_CONTAINER(box), 0);
		gtk_box_pack_start(parent, box, FALSE, FALSE, 0);
		gtk_widget_show(box);
		label=gtk_label_new(to_UpCase(FldInf->field_name));
		gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);
		gtk_widget_show(label);
		omenu=gtk_option_menu_new();
		gtk_widget_set_usize(omenu, FIELD_LENGTH_ENUM, 25);
		menu=gtk_menu_new();
		for (i=(unsigned)FldInf->minimum; i<=(unsigned)FldInf->maximum; i++) {
			menuitem=gtk_menu_item_new_with_label((gchar*)
						(* ((enum_field_func)FldInf->field_func) )
							(i, TRUE));
			gtk_object_set_user_data(GTK_OBJECT(menuitem), (gpointer)i);
			gtk_menu_append(GTK_MENU(menu), menuitem);
			gtk_widget_show(menuitem);
		}
		gtk_option_menu_set_menu(GTK_OPTION_MENU(omenu), menu);
		gtk_box_pack_end(GTK_BOX(box), omenu, FALSE, TRUE, 0);
		gtk_widget_show(omenu);
		widget=omenu;
		break;
	
	default:
		widget=NULL;
		g_assert_not_reached();
		break;
	}
	
	gtk_object_set_user_data(GTK_OBJECT(widget), (gpointer)FldInf);
	
	return widget;
}


void
Editor_field_refresh	(gb_any_S	*Struct,
			 GtkWidget	*widget)
{
	register const	field_info_S	*FldInf;
	
	g_assert(GB_IS_STRUCT(Struct));
	
	g_assert((FldInf=gtk_object_get_user_data(GTK_OBJECT(widget))));
	
	switch (FldInf->field_type) {
		register GtkWidget	*menu;
		register GtkWidget	*menuitem;
		register GList		*list;
		register wd_link_S	*wd_link;
		register gchar		*string;
	
	case	FIELD_WD_LINK:
		wd_link=gtk_object_get_user_data(GTK_OBJECT(widget->parent));
		string=(gchar*)structure_field_get_value_str(Struct, FldInf);
		gtk_entry_set_text(GTK_ENTRY(wd_link->entry), string ? string : "");
		break;
	
	case	FIELD_POINTER:
	case	FIELD_STRING:
	case	FIELD_INT:
	case	FIELD_FLOAT:
		string=(gchar*)structure_field_get_value_str(Struct, FldInf);
		gtk_entry_set_text(GTK_ENTRY(widget), string ? string : "");
		break;
	
	case	FIELD_BOOLEAN:
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(widget),
			structure_field_get_value(Struct, FldInf).bool_val);
		break;

	case	FIELD_BITS:
		list=gtk_container_children(GTK_CONTAINER(widget));
		while (list) {
			register GtkWidget	*toggle;
			register guint32	bit_val;
			register guint		bit_num;
			
			g_assert((toggle=list->data));
			g_assert(GTK_IS_TOGGLE_BUTTON(toggle));
			
			g_assert((bit_num=(guint)gtk_object_get_user_data(GTK_OBJECT(toggle))));
			bit_val=structure_field_bit_value(FldInf, bit_num);
			
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(toggle),
				( structure_field_get_value(Struct, FldInf).bit_val &
				  bit_val ) !=0);
			
			list=list->next;
		}
		g_list_free(list);
		break;
	
	case	FIELD_ENUM:
		menu=gtk_option_menu_get_menu(GTK_OPTION_MENU(widget));
		menuitem=gtk_menu_get_active(GTK_MENU(menu));
		gtk_menu_set_active(GTK_MENU(menu),
			structure_field_get_value(Struct, FldInf).enum_val -
				(unsigned)FldInf->minimum);
		gtk_option_menu_set_menu(GTK_OPTION_MENU(widget), menu);
		if (menuitem!=gtk_menu_get_active(GTK_MENU(menu)))
			gtk_widget_draw(widget, NULL);
		break;
	
	default:
		g_assert_not_reached();
		break;
	}
}


gboolean
Editor_field_apply	(gb_any_S	*Struct,
			 GtkWidget	*widget)
{
	register const	field_info_S	*FldInf;
	register field_value_U		old_value;
	
	g_assert(GB_IS_STRUCT(Struct));
	
	g_assert((FldInf=gtk_object_get_user_data(GTK_OBJECT(widget))));
	
	old_value=structure_field_get_value(Struct, FldInf);
	if (FldInf->field_type==FIELD_STRING)
		old_value.string=g_strdup(old_value.string);
	
	switch (FldInf->field_type) {
		register GtkWidget	*menu;
		register GtkWidget	*menuitem;
		register GList		*list;
		register field_value_U	value;
		register wd_link_S	*wd_link;
	
	case	FIELD_FLOAT:
		value.string=gtk_entry_get_text(GTK_ENTRY(widget));
		value.float_val=atof(value.string);
		structure_field_set_value(Struct, value, FldInf);
		break;
	
	case	FIELD_INT:
		value.string=gtk_entry_get_text(GTK_ENTRY(widget));
		value.int_val=atoi(value.string);
		structure_field_set_value(Struct, value, FldInf);
		break;
	
	case	FIELD_WD_LINK:
		wd_link=gtk_object_get_user_data(GTK_OBJECT(widget->parent));
		value.link=wd_link->widget_data;
		structure_field_set_value(Struct, value, FldInf);
		break;
		
	case	FIELD_POINTER:
	case	FIELD_STRING:
		value.string=gtk_entry_get_text(GTK_ENTRY(widget));
		structure_field_set_value(Struct, value, FldInf);
		break;
	
	case	FIELD_BOOLEAN:
		value.bool_val=GTK_TOGGLE_BUTTON(widget)->active;
		structure_field_set_value(Struct, value, FldInf);
		break;

	case	FIELD_BITS:
		list=gtk_container_children(GTK_CONTAINER(widget));
		while (list) {
			register GtkWidget	*toggle;
			register guint		bit_num;
			
			g_assert((toggle=list->data));
			g_assert(GTK_IS_TOGGLE_BUTTON(toggle));
			
			g_assert((bit_num=(guint)gtk_object_get_user_data(GTK_OBJECT(toggle))));
			
			structure_field_bit_set(Struct,
						  FldInf,
						  bit_num,
						  GTK_TOGGLE_BUTTON(toggle)->active);
			
			list=list->next;
		}
		g_list_free(list);
		break;
	
	case	FIELD_ENUM:
		menu=gtk_option_menu_get_menu(GTK_OPTION_MENU(widget));
		menuitem=gtk_menu_get_active(GTK_MENU(menu));
		value.enum_val=(guint)gtk_object_get_user_data(GTK_OBJECT(menuitem));
		structure_field_set_value(Struct, value, FldInf);
		break;
	
	default:
		g_assert_not_reached();
		break;
	}
	
	
	/* return wether rebuilding is required
	*/
	if (FIELD_FLAGS(FldInf)&FOPT_BUILDING) {
		if (FldInf->field_type==FIELD_STRING) {
			register gchar		*name;
			
			name=structure_field_get_value(Struct, FldInf).string;
			if (old_value.string && name) {
				register gboolean rebuild;
				
				rebuild=strcmp(old_value.string, name)!=0;
				g_free(old_value.string);
				return rebuild;
			} else {
				g_free(old_value.string);
				return old_value.string!=name;
			}
		} else
			return  old_value.int_val !=
				  structure_field_get_value(Struct, FldInf).int_val;
	} else
		return FALSE;
}


editwid_widget_list_S*
Editor_fields_create		(gb_wdat_base_S	*WidDat,
				 GtkBox		*object_box,
				 GtkBox		*base_box,
				 GtkBox		*misc_box,
				 GtkBox		*container_box,
				 GtkBox		*bin_box,
				 GtkBox		*linkage_box,
				 GtkBox		*specific_box)
{
	register guint			i;
	register editwid_widget_list_S	*widget_list;
	register const struct_info_S	*StrInf=NULL;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	g_assert(object_box && base_box && misc_box && container_box && bin_box && linkage_box && specific_box);
	
	
	/* initialize widget_list structure
	*/
	widget_list=g_new0(editwid_widget_list_S, 1);
	widget_list->widget_data=WidDat;
	widget_list->Label_symbol_name=NULL;
	widget_list->Entry_symbol_name=NULL;
	
	
	/* create object fields
	*/
	StrInf=structure_info(GB_OBJECT);
	widget_list->object_fields=StrInf->field_count;
	widget_list->object_widgets=g_new0(GtkWidget*, widget_list->object_fields);
	for (i=0; i<widget_list->object_fields; i++)
		widget_list->object_widgets[i]=Editor_field_create(GB_CAST(any, WidDat),
								object_box,
								field_info(StrInf, i));
	
	
	/* create base fields
	*/
	StrInf=structure_info(GB_WIDGET_BASE);
	widget_list->base_fields=StrInf->field_count;
	widget_list->base_widgets=g_new0(GtkWidget*, widget_list->base_fields);
	for (i=0; i<widget_list->base_fields; i++)
		widget_list->base_widgets[i]=Editor_field_create(GB_CAST(any, WidDat),
								base_box,
								field_info(StrInf, i));
	
	
	/* create misc fields
	*/
	StrInf=structure_info(GB_WIDGET_MISC);
	if (GB_TYPE_IS_WIDDAT_MISC(WidDat->type))
		widget_list->misc_fields=StrInf->field_count;
	else
		widget_list->misc_fields=0;
	widget_list->misc_widgets=g_new0(GtkWidget*, widget_list->misc_fields);
	for (i=0; i<widget_list->misc_fields; i++)
		widget_list->misc_widgets[i]=Editor_field_create(GB_CAST(any, WidDat),
								misc_box,
								field_info(StrInf, i));
	
	
	/* create container fields
	*/
	StrInf=structure_info(GB_WIDGET_CONTAINER);
	if (GB_TYPE_IS_WIDDAT_CONTAINER(WidDat->type))
		widget_list->container_fields=StrInf->field_count;
	else
		widget_list->container_fields=0;
	widget_list->container_widgets=g_new0(GtkWidget*, widget_list->container_fields);
	for (i=0; i<widget_list->container_fields; i++)
		widget_list->container_widgets[i]=Editor_field_create(GB_CAST(any, WidDat),
								container_box,
								field_info(StrInf, i));
	
	
	/* create bin fields
	*/
	StrInf=structure_info(GB_WIDGET_BIN);
	if (GB_TYPE_IS_WIDDAT_BIN(WidDat->type))
		widget_list->bin_fields=StrInf->field_count;
	else
		widget_list->bin_fields=0;
	widget_list->bin_widgets=g_new0(GtkWidget*, widget_list->bin_fields);
	for (i=0; i<widget_list->bin_fields; i++)
		widget_list->bin_widgets[i]=Editor_field_create(GB_CAST(any, WidDat),
								bin_box,
								field_info(StrInf, i));
	
	
	/* create linkage fields
	*/
	StrInf=NULL;
	if (WidDat->linkage) {
		StrInf=structure_info(WidDat->linkage->type);
		widget_list->linkage_fields=StrInf->field_count;
	} else
		widget_list->linkage_fields=0;
	widget_list->linkage_widgets=g_new0(GtkWidget*, widget_list->linkage_fields);
	for (i=0; i<widget_list->linkage_fields; i++)
		widget_list->linkage_widgets[i]=Editor_field_create(WidDat->linkage,
								linkage_box,
								field_info(StrInf, i));
	
	
	/* create specific fields
	*/
	StrInf=structure_info(WidDat->type);
	widget_list->specific_fields=StrInf->field_count;
	widget_list->specific_widgets=g_new0(GtkWidget*, widget_list->specific_fields);
	for (i=0; i<widget_list->specific_fields; i++)
		widget_list->specific_widgets[i]=Editor_field_create(GB_CAST(any, WidDat),
								specific_box,
								field_info(StrInf, i));
	
	
	return widget_list;
}


void
Editor_refresh		(gb_wdat_base_S	*WidDat)
{
	register guint			i;
	register editwid_widget_list_S	*widget_list;
	register gchar			*name;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	if (!GUBI_DATA(WidDat)->editor)
		return;
	
	g_assert((widget_list=
		  gtk_object_get_user_data(GTK_OBJECT(GUBI_DATA(WidDat)->editor))));
	
	if (GUBI_DATA(WidDat)->category & WID_CAT_AUXILLARY) {
		register gchar	*s_name;
		register gchar	*prefix="Auxillary: ";
		
		s_name=g_strdup(widget_data_symbol_name_get(WidDat));
		name=g_new(gchar, strlen(prefix)+strlen(s_name)+1);
		sprintf(name, "%s%s", prefix, s_name);
		g_free(s_name);
	} else
		name=g_strdup(widget_data_symbol_name_get(WidDat));
	
	gtk_label_set(GTK_LABEL(widget_list->Label_symbol_name), name);
	g_free(name); name=NULL;
	gtk_entry_set_text(GTK_ENTRY(widget_list->Entry_symbol_name),
			   (gchar*)widget_data_symbol_name_get(WidDat));
	
	
	for (i=0; i<widget_list->object_fields; i++)
		Editor_field_refresh(GB_CAST(any, WidDat), widget_list->object_widgets[i]);
	for (i=0; i<widget_list->base_fields; i++)
		Editor_field_refresh(GB_CAST(any, WidDat), widget_list->base_widgets[i]);
	for (i=0; i<widget_list->misc_fields; i++)
		Editor_field_refresh(GB_CAST(any, WidDat), widget_list->misc_widgets[i]);
	for (i=0; i<widget_list->container_fields; i++)
		Editor_field_refresh(GB_CAST(any, WidDat), widget_list->container_widgets[i]);
	for (i=0; i<widget_list->bin_fields; i++)
		Editor_field_refresh(GB_CAST(any, WidDat), widget_list->bin_widgets[i]);
	for (i=0; i<widget_list->linkage_fields; i++)
		Editor_field_refresh(WidDat->linkage, widget_list->linkage_widgets[i]);
	for (i=0; i<widget_list->specific_fields; i++)
		Editor_field_refresh(GB_CAST(any, WidDat), widget_list->specific_widgets[i]);
}


void
SigH_Editor_wd_link_clicked	(GtkWidget	*widget,
				 gb_any_S	*Struct)
{
	g_assert(Struct);
	
	
	if (GB_IS_WIDDAT(Struct)) {
		register wd_link_S	*wd_link;
		register field_info_S	*FldInf;
		register field_func_WD_LINK	field_func;
		
		g_assert((FldInf=gtk_object_get_user_data(GTK_OBJECT(widget))));
		
		wd_link=gtk_object_get_user_data(GTK_OBJECT(widget->parent));
		
		g_assert(!wd_link->list_win);
		if ((field_func=(field_func_WD_LINK)FldInf->field_func))
			wd_link->list_win=field_func(
						GB_wCAST(base, Struct),
						GTK_OBJECT(widget),
						&(wd_link->list_win),
						GTK_SIGNAL_FUNC(SigH_Editor_wd_link_clicked),
						wd_link,
						&(wd_link->widget_data));
		
		if (wd_link->list_win) {
			gtk_widget_set_sensitive(widget, FALSE);
			gtk_signal_connect_object(GTK_OBJECT(wd_link->list_win),
						  "destroy",
						  GTK_SIGNAL_FUNC(_gtk_widget_set_sensitivity),
						  GTK_OBJECT(widget));
		}
	} else {
		register wd_link_S	*wd_link;
		
		g_assert((wd_link=(wd_link_S*)Struct));
		
		gtk_entry_set_text(GTK_ENTRY(wd_link->entry),
			wd_link->widget_data ?
			  (gchar*)widget_data_symbol_name_get(wd_link->widget_data) :
			  "");
	}
}


void
SigH_Editor_Apply_clicked	(GtkWidget	*widget,
				 gb_wdat_base_S	*WidDat)
{
	register guint			i;
	register editwid_widget_list_S	*widget_list;
	register gchar			*string;
	register gboolean		name_changed;
	register gboolean		need_rebuild;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	g_assert((widget_list=
		  gtk_object_get_user_data(GTK_OBJECT(GUBI_DATA(WidDat)->editor))));
	
	
	string=gtk_entry_get_text(GTK_ENTRY(widget_list->Entry_symbol_name));
	name_changed=strcmp(string, widget_data_symbol_name_get(WidDat))!=0;
	if (name_changed)
		widget_data_symbol_name_set(WidDat, string);
	
	
	/* apply new values to widget fields
	*/
	need_rebuild=FALSE;
	for (i=0; i<widget_list->object_fields; i++)
		need_rebuild|=Editor_field_apply(GB_CAST(any, WidDat), widget_list->object_widgets[i]);
	for (i=0; i<widget_list->base_fields; i++)
		need_rebuild|=Editor_field_apply(GB_CAST(any, WidDat), widget_list->base_widgets[i]);
	for (i=0; i<widget_list->misc_fields; i++)
		need_rebuild|=Editor_field_apply(GB_CAST(any, WidDat), widget_list->misc_widgets[i]);
	for (i=0; i<widget_list->container_fields; i++)
		need_rebuild|=Editor_field_apply(GB_CAST(any, WidDat), widget_list->container_widgets[i]);
	for (i=0; i<widget_list->bin_fields; i++)
		need_rebuild|=Editor_field_apply(GB_CAST(any, WidDat), widget_list->bin_widgets[i]);
	for (i=0; i<widget_list->linkage_fields; i++)
		need_rebuild|=Editor_field_apply(WidDat->linkage, widget_list->linkage_widgets[i]);
	for (i=0; i<widget_list->specific_fields; i++)
		need_rebuild|=Editor_field_apply(GB_CAST(any, WidDat), widget_list->specific_widgets[i]);
	
	
	/* check for linkage specific stuff
	*/
	widget_data_linkage_check(WidDat);
	if (widget_data_children_linkage_check(WidDat)) {
		register GList	*list;
		
		list=GUBI_DATA(WidDat)->children;
		while (list) {
			register gb_wdat_base_S	*ChildDat=list->data;
			
			g_assert(GB_IS_WIDDAT(ChildDat));
			
			Editor_refresh(ChildDat);
			
			list=list->next;
		}
	}
	
	
	/* refbuild widget tree if neccessary
	*/
	if (need_rebuild)
	{
		tree_rebuild(GUBI_DATA(WidDat)->tree);
		tree_show(GUBI_DATA(WidDat)->tree);
	} else
		gb_widget_data_set(WidDat);
	
	
	/* refresh editor
	*/
	Editor_refresh(WidDat);
	
	
	/* if the symbol name changed, update all list items
	 * referring to this WidDat and update widget lists of
	 * all parents and immediate children.
	*/
	if (name_changed) {
		register GList		*list;
		
		widget_data_items_update(WidDat);
		
		list=GUBI_DATA(WidDat)->children;
		while (list) {
			register gb_wdat_base_S	*ChildDat;
			
			ChildDat=list->data;
			if (GUBI_DATA(ChildDat)->browser)
				Browser_refresh(ChildDat, TRUE);
			list=list->next;
		}
		while (WidDat) {
			if (GUBI_DATA(WidDat)->browser)
				Browser_refresh(WidDat, TRUE);
			WidDat=WidDat->parent;
		}
	}
}


void
SigH_Editor_Expand_clicked	(GtkWidget	*widget,
				 GtkWidget	*Box)
{
	register GtkWidget	*parent;
	
	g_assert(widget);
	g_assert(Box);
	
	if (GTK_TOGGLE_BUTTON(widget)->active)
		gtk_widget_show(Box);
	else
		gtk_widget_hide(Box);
	
	parent=Box->parent;
	do {
		parent=parent->parent;
	} while (parent->parent);
	
	gtk_widget_set_usize(parent, -1, -1);
	gtk_widget_set_usize(parent, 0, 0);
}
