/*
 * This program is an implementation of the ISAKMP Internet Standard.
 * Copyright (C) 1997 Angelos D. Keromytis.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * This code was written while the author was in Greece, in May/June
 * 1997.
 *
 * You may contact the author by:
 *   e-mail: angelos@dsl.cis.upenn.edu
 *  US-mail: Angelos D. Keromytis
 *           Distributed Systems Lab
 *           Computer and Information Science Department
 *           University of Pennsylvania
 *           Moore Building
 *           200 South 33rd Street
 *           Philadelphia, PA 19104	   
 */

#include <stdio.h>
#include <signal.h>
#include <sys/time.h>
#include <fcntl.h>
#include "sha1.h"
#include "constants.h"
#include "defs.h"

#define RANDOM_POOL_SIZE   20
static u_char random_pool[RANDOM_POOL_SIZE];
static volatile u_char i, j, k;

/*
 * Signal handler.
 */
void
rnd_handler()
{
    k <<= 1;        /* Shift left by 1 */
    j++;           
    k |= (i & 0x1); /* Get lsbit of counter */

    if (j != 8)
      signal(SIGVTALRM, rnd_handler);
}

/*
 * Do the clock skew trick.
 */
static u_char
generate_rnd_byte(void)
{
    struct itimerval tmval, ntmval;
#ifdef linux
    u_char c;
#endif

/*
 * #ifdef linux
 *  int mask = siggetmask();
 *
 *  mask |= SIGVTALRM;
 *  sigsetmask(mask);
 * #endif
 */

#ifdef linux
    if (read(linux_random, &c, sizeof(c)) == -1)
      exit_log("read() failed in get_rnd_byte()", 0, 0, 0);
    
    return c;
#endif

    i = 0;
    j = 0;

    ntmval.it_interval.tv_sec = 0;
    ntmval.it_interval.tv_usec = 1;
    ntmval.it_value.tv_sec = 0;
    ntmval.it_value.tv_usec = 1;
    signal(SIGVTALRM, rnd_handler);
    setitimer(ITIMER_VIRTUAL, &ntmval, &tmval);
    
    while (j != 8)
      i++;
    
    setitimer(ITIMER_VIRTUAL, &tmval, &ntmval);
    signal(SIGVTALRM, SIG_IGN);

/*
 * #ifdef linux
 *  mask ^= SIGVTALRM;
 *  sigsetmask(mask);
 * #endif
 */
    return k;
}

/*
 * Get a single random byte.
 */
u_char
get_rnd_byte(void)
{
    SHA1_CTX ctx;
    
    SHA1Init(&ctx);

    random_pool[RANDOM_POOL_SIZE - 1] = generate_rnd_byte();
    random_pool[0] = generate_rnd_byte();

    SHA1Update(&ctx, random_pool, RANDOM_POOL_SIZE);
    SHA1Final(random_pool, &ctx);
    
    return random_pool[0];
}

/*
 * A random number generator based on clock skew. I first heard of
 * the idea by John Ioannidis, who heard it from Matt Blaze and/or
 * Jack Lacy. Seems to work.
 * If we're on OpenBSD, use the arc4random() generator instead; if
 * on Linux, use /dev/random.
 */
void
get_rnd_bytes(u_char *buffer, int length)
{
    int i;
#ifdef __OpenBSD__
    for (i = 0; i < length; i++)
      buffer[i] = arc4random() % 256;
#else    
    for (i = 0; i < length; i++)
      buffer[i] = get_rnd_byte();
#endif
}

/*
 * Initialize the random pool.
 */
void
init_rnd_pool(void)
{
    int i;
    SHA1_CTX ctx;
    
    SHA1Init(&ctx);

#ifdef linux
#ifdef DEBUG
    log(0, "opening %s", LINUX_RANDOM_FILE);
#endif
    linux_random = open(LINUX_RANDOM_FILE, O_RDONLY);
    if (linux_random == -1)
      exit_log("open() failed in init_rnd_pool()", 0, 0, 0);
#endif

    get_rnd_bytes(random_pool, RANDOM_POOL_SIZE);

    SHA1Update(&ctx, random_pool, RANDOM_POOL_SIZE);
    SHA1Final(random_pool, &ctx);
}
