/*
 * This program is an implementation of the ISAKMP Internet Standard.
 * Copyright (C) 1997 Angelos D. Keromytis.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * This code was written while the author was in Greece, in May/June
 * 1997.
 *
 * You may contact the author by:
 *   e-mail: angelos@dsl.cis.upenn.edu
 *  US-mail: Angelos D. Keromytis
 *           Distributed Systems Lab
 *           Computer and Information Science Department
 *           University of Pennsylvania
 *           Moore Building
 *           200 South 33rd Street
 *           Philadelphia, PA 19104	   
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <netdb.h>
#include <unistd.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include "constants.h"
#include "state.h"
#include "defs.h"

/*
 *  Server main loop and socket initialization routines.
 */

/*
 * Initialize the kernel socket.
 * XXX For now, just get a port.
 */
int
init_kernelfd(void)
{
    struct sockaddr_in sin;
    int s;
    
    s = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if (s == -1)
      exit_log("socket() in init_kernelfd()", 0, 0, 0);
    
    sin.sin_family = AF_INET;
    sin.sin_port = htons(our_port + 1);
    sin.sin_addr.s_addr = INADDR_ANY;

#ifdef DEBUG
    log(0, "init_kernelfd(): listening to port %d", our_port + 1, 0, 0);
#endif

    if (bind(s, (struct sockaddr *)&sin, sizeof(sin)) < 0)
      exit_log("bind() in init_kernelfd()", 0, 0, 0);
   
    return s;
}

/*
 * Initialize the sockets.
 */
int
init_socket(int *socks)
{
    struct protoent *proto;
    struct sockaddr_in sin, *sin2;
    int sock, i, j, k;
    struct ifconf ifconf; 
    char buf[1024];

    /* Get a UDP socket */

    proto = getprotobyname("udp");
    
    if (proto == (struct protoent *) NULL)
      socks[0] = socket(PF_INET, SOCK_DGRAM, 0);
    else
      socks[0] = socket(PF_INET, SOCK_DGRAM, proto->p_proto);
    
    if (socks[0] == -1)
      exit_log("socket() failed in init_socket()", 0, 0, 0);

    sin.sin_family = AF_INET;
    sin.sin_port = htons(our_port);
    sin.sin_addr.s_addr = INADDR_ANY;

#ifdef DEBUG
    log(0, "init_socket(): listening to port %d", our_port, 0, 0);
#endif

    k = 1;
    if (setsockopt(socks[0], SOL_SOCKET, SO_REUSEADDR, (void *)&k, 
		   sizeof(k)) < 0)
      exit_log("setsockopt() in init_socket()", 0, 0, 0);

    /* bind the socket */
    if (bind(socks[0], (struct sockaddr *)&sin, sizeof(sin)) < 0)
      exit_log("bind() failed in init_socket()", 0, 0, 0);

    /* Get local interfaces */
    ifconf.ifc_len = 1024; 
    ifconf.ifc_buf = buf; 
    bzero(buf, 1024); 
 
    if (ioctl(socks[0], SIOCGIFCONF, &ifconf) == -1) 
      exit_log("ioctl() in init_socket()", 0, 0, 0); 

    /* This loop by Niels Provos */
    for (i = 1, j = 0; j < ifconf.ifc_len; j += IFNAMSIZ + 
#if defined(__NetBSD__) || defined(__OpenBSD__)
	     buf[IFNAMSIZ + j] 
#else 
	     sizeof(struct sockaddr) 
#endif 
	     , i++)
    {
	sin2 = (struct sockaddr_in *) &buf[IFNAMSIZ + j]; 
	
	if (sin2->sin_family != AF_INET)
	{
	    i--;
	    continue;
	}
	
	if ((socks[i] = socket(PF_INET, SOCK_DGRAM, proto->p_proto)) < 0)
	  exit_log("socket() in init_socket()", 0, 0, 0); 

	if (setsockopt(socks[i], SOL_SOCKET, SO_REUSEADDR, (void *)&k, 
		       sizeof(k)) < 0)
	  exit_log("setsockopt() in init_socket()", 0, 0, 0);

	bzero((void *)&sin, sizeof(sin));
	sin.sin_port = htons(our_port);
	sin.sin_addr = sin2->sin_addr; 
	sin.sin_family = AF_INET; 
	
	if (bind(socks[i], (struct sockaddr *)&sin, sizeof(sin)) < 0)
	  exit_log("bind() in init_socket()", 0, 0,0); 

#ifdef DEBUG
	log(0, "listening at %s", inet_ntoa(sin2->sin_addr), 0, 0);
#endif
    }
    
#ifdef DEBUG
    log(0, "listening at %d interfaces", i - 1, 0, 0);
#endif

    return i;
}

/*
 * This routine listens for incoming ISAKMP packets, kernel PF_KEY requests,
 * handles packet retransmitts and other events.
 */
void
call_server(void)
{
    struct sockaddr_in sin;
    int kernelfd;
    int sock[256];
    int numsocks;
    fd_set readfds;
    struct timeval tm;
    int i, maxfd, k;
    
    numsocks = init_socket(sock);
    kernelfd = init_kernelfd();
    
    while (1)
    {
	for (k = 1; k < numsocks; k++)
	  FD_SET(sock[k], &readfds);
	FD_SET(kernelfd, &readfds);

	maxfd = (sock[numsocks - 1] > kernelfd ? sock[numsocks - 1] : 
		 kernelfd) + 1;

	i = next_event();   /* Is there any pending event ? */
	if (i == -1)
	  i = select(maxfd, &readfds, NULL, NULL, NULL);
	else
	{
	    tm.tv_sec = i;
	    tm.tv_usec = 0;
	    i = select(maxfd, &readfds, NULL, NULL, &tm);
	}

	if (i == -1)
	  exit_log("select() failed in call_server()", 0, 0, 0);

	for (k = 1; k < numsocks; k++)
	  if (FD_ISSET(sock[k], &readfds))
#ifdef DEBUG
	  {
	      log(0, "received packet", 0, 0, 0);
#endif
	      comm_handle(kernelfd, sock[k]);
#ifdef DEBUG
	  }
#endif

	if (FD_ISSET(kernelfd, &readfds))
#ifdef DEBUG
	{
	    log(0, "received kernel message", 0, 0, 0);
#endif
	  kernel_handle(kernelfd, sock[0]);
#ifdef DEBUG
	}
#endif
	
	if (i == 0)   /* Event */
#ifdef DEBUG
	{
	    log(0, "time to handle event", 0, 0, 0);
#endif
	  event_handle(kernelfd, sock[0]);
#ifdef DEBUG	    
	}
#endif
    }
}
