/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AmountString,
  Duration,
  PaymentInsufficientBalanceDetails,
  TalerErrorCode,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { CoinConfig, defaultCoinConfig } from "../harness/denomStructures.js";
import {
  GlobalTestState,
  generateRandomPayto,
  setupDb,
} from "../harness/harness.js";
import { createSimpleTestkudosEnvironmentV3, withdrawViaBankV3 } from "../harness/helpers.js";

export async function runWalletInsufficientBalanceTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const coinConfig: CoinConfig[] = defaultCoinConfig.map((x) => x("TESTKUDOS"));

  let {
    bankClient,
    exchange,
    merchant,
    walletService,
    walletClient,
  } = await createSimpleTestkudosEnvironmentV3(t, coinConfig, {
    skipWireFeeCreation: true,
  });

  await merchant.addInstanceWithWireAccount({
    id: "default",
    name: "Default Instance",
    paytoUris: [generateRandomPayto("merchant-default")],
    defaultWireTransferDelay: Duration.toTalerProtocolDuration(
      Duration.fromSpec({ minutes: 1 }),
    ),
  });

  await merchant.addInstanceWithWireAccount({
    id: "minst1",
    name: "minst1",
    paytoUris: [generateRandomPayto("minst1")],
    defaultWireTransferDelay: Duration.toTalerProtocolDuration(
      Duration.fromSpec({ minutes: 1 }),
    ),
  });

  await walletClient.client.call(WalletApiOperation.InitWallet, {
    config: {
      testing: {
        skipDefaults: true,
      },
    },
  });

  const wres = await withdrawViaBankV3(t, {
    amount: "TESTKUDOS:10",
    bankClient,
    exchange,
    walletClient,
  });
  await wres.withdrawalFinishedCond;

  const exc = await t.assertThrowsTalerErrorAsync(async () => {
    await walletClient.call(WalletApiOperation.PrepareDeposit, {
      amount: "TESTKUDOS:5" as AmountString,
      depositPaytoUri: "payto://x-taler-bank/localhost/foobar",
    });
  });

  t.assertDeepEqual(
    exc.errorDetail.code,
    TalerErrorCode.WALLET_DEPOSIT_GROUP_INSUFFICIENT_BALANCE,
  );

  const insufficientBalanceDetails: PaymentInsufficientBalanceDetails =
    exc.errorDetail.insufficientBalanceDetails;

  t.assertAmountEquals(
    insufficientBalanceDetails.balanceAvailable,
    "TESTKUDOS:9.72",
  );
  t.assertAmountEquals(
    insufficientBalanceDetails.balanceExchangeDepositable,
    "TESTKUDOS:0",
  );
}

runWalletInsufficientBalanceTest.suites = ["wallet"];
