/*
 *	Copyright 1990 by Rayan S. Zachariassen, all rights reserved.
 *	This will be free software, but only when it is finished.
 */

/*
 * To really understand how headers (and their converted versions)
 * are processed you do need to draw a diagram.
 * Basically:
 *    rp->desc->headers[]    is index to ALL of the headers, and
 *    rp->desc->headerscvt[] is index to ALL of the CONVERTED headers.
 * Elements on these arrays are  "char *strings[]" which are the
 * actual headers.
 * There are multiple-kind headers depending upon how they have been
 * rewritten, and those do tack together for each recipients (rp->)
 * There
 *    rp->newmsgheader    is a pointer to an element on  rp->desc->headers[]
 *    rp->newmsgheadercvt is prespectively an elt on  rp->desc->headerscvt[]
 *
 * The routine-collection   mimeheaders.c  creates converted headers,
 * if the receiving system needs them. Converted data is created only
 * once per  rewrite-rule group, so there should not be messages which
 * report  "Received: ... convert XXXX convert XXXX convert XXXX; ..."
 * for as many times as there there are recipients for the message.
 * [mea@utu.fi] - 25-Jul-94
 */


#include <stdio.h>
#include "hostenv.h"
#include <ctype.h>
#include <sys/stat.h>
#include <sys/file.h>
#ifdef FCNTL_H
# include FCNTL_H
#endif
#include <sysexits.h>
#include <mail.h>
#include "malloc.h"
#include "ta.h"

extern int markoff(), cistrcmp(), lockaddr();
extern void warning();
extern char *routermxes();


#ifndef	MAXPATHLEN
#define	MAXPATHLEN 1024
#endif	/* !MAXPATHLEN */


void
ctlfree(dp,anyp)
	struct ctldesc *dp;
	void *anyp;
{
	unsigned long lowlim = (unsigned long) dp->contents;
	unsigned long highlim = lowlim + dp->contentsize;
	if ((unsigned long)anyp < lowlim ||
	    (unsigned long)anyp > highlim)
	  free(anyp);	/* It isn't within DP->CONTENTS data.. */
}

void *
ctlrealloc(dp,anyp,size)
	struct ctldesc *dp;
	void *anyp;
	size_t size;
{
	unsigned long lowlim = (unsigned long) dp->contents;
	unsigned long highlim = lowlim + dp->contentsize;
	void *anyp2;

	if ((unsigned long)anyp < lowlim ||
	    (unsigned long)anyp > highlim)
	  return realloc(anyp,size);	/* It isn't within
					   DP->CONTENTS data.. */
	/* It is within the data, allocate a new storage.. */
	anyp2 = (void*) malloc(size);
	if (anyp2)
	  memcpy(anyp2,anyp,size);
	return anyp2;
}

void
ctlclose(dp)
	struct ctldesc *dp;
{
	struct address *ap, *nextap;
	struct rcpt *rp, *nextrp;
	char ***msghpp;

	for (rp = dp->recipients; rp != NULL; rp = rp->next) {
	  if (rp->lockoffset == 0)
	    continue;
	  diagnostic(rp, EX_TEMPFAIL, "address was left locked!!");
	}
	if (dp->ctlfd >= 0)
	  close(dp->ctlfd);
	if (dp->msgfd >= 0)
	  close(dp->msgfd);
	if (dp->contents != NULL)
	  free(dp->contents);
	if (dp->offset != NULL)
	  free((char *)dp->offset);
	for (ap = dp->senders; ap != NULL; ap = nextap) {
	  nextap = ap->link;
	  ap->link = NULL; /* XX: Excessive caution.. */
	  free((char *)ap);
	}
	dp->senders = NULL; /* XX: Excessive caution.. */
	for (rp = dp->recipients; rp != NULL; rp = nextrp) {
	  nextrp = rp->next;
	  rp->next = NULL; /* XX: Excessive caution.. */
	  ap = rp->addr;
	  if (ap->routermxes) {
	    free((char *)ap->routermxes);
	    ap->routermxes = NULL; /* XX: Excessive caution.. */
	  }
	  free((char *)rp);
	}
	dp->recipients = NULL; /* XX: Excessive caution.. */
	/* Free ALL dp->msgheader's, if they have been reallocated.
	   Don't free on individual recipients, only on this global set.. */
	for (msghpp = dp->msgheaders; msghpp &&  *msghpp; ++msghpp) {
	  char **msghp = *msghpp;
	  for ( ; msghp && *msghp ; ++msghp )
	    ctlfree(dp,*msghp);
	  free(*msghpp);
	}
	free(dp->msgheaders);
	dp->msgheaders = NULL; /* XX: Excessive caution.. */
}


/* as in: SKIPWHILE(isascii,cp) */
#define	SKIPWHILE(X,Y)	while (*Y != '\0' && isascii(*Y) && X(*Y)) { ++Y; }


struct address *
ctladdr(cp)
	char *cp;
{
	struct address *ap;

	ap = (struct address *)emalloc(sizeof (struct address));
	if (ap == NULL)
		return NULL;
	ap->link = NULL;
	SKIPWHILE(isspace,cp);
	ap->channel = cp;
	SKIPWHILE(!isspace,cp);
	*cp++ = '\0';
	SKIPWHILE(isspace,cp);
	ap->host = cp;
	ap->routermxes = NULL;
	if (*cp == '(')
	  cp = routermxes(cp,ap);  /* It is a bit complex.. */
	else {
	  SKIPWHILE(!isspace,cp);
	  *cp++ = '\0';
	}
	SKIPWHILE(isspace,cp);
	ap->user = cp;
	/* the user value is allowed to have embedded whitespace */
	while (*cp != '\0')
	  ++cp;
	--cp;
	while (isascii(*cp) && isdigit(*cp))
	  --cp;
	if (isascii(*cp) && *cp == '-')
	  --cp;
	ap->misc = cp+1;
	while (isascii(*cp) && isspace(*cp))
	  --cp;
	*++cp = '\0';
	return ap;
}

struct ctldesc *
ctlopen(file, channel, host, exitflagp, selectaddr, matchrouter)
	char *file, *channel, *host;
	int *exitflagp;
	int (*selectaddr)();
	int (*matchrouter)();
{
	register char *s;
	char *mfpath;
	int i, n;
	static struct ctldesc d;
	struct address *ap;
	struct rcpt *rp;
	struct stat stbuf;
	char ***msgheaders = NULL;
	char ***msgheaderscvt = NULL;
	int headers_cnt = 0;
	int headers_spc = 0;
	int largest_headersize = 80; /* Some magic minimum.. */

	if ((d.ctlfd = open(file, O_RDWR, 0)) < 0) {
	  char cwd[MAXPATHLEN], buf[MAXPATHLEN+MAXPATHLEN+100];

#ifdef	USE_GETWD
	  getwd(cwd);
#else
	  getcwd(cwd,MAXPATHLEN);
#endif
	  sprintf(buf,
		  "Cannot open control file \"%%s\" from \"%s\" as uid %d! (%%m)",
		  cwd, geteuid());
	  warning(buf, file);
	  return NULL;
	}
	if (fstat(d.ctlfd, &stbuf) < 0) {
	  warning("Cannot stat control file \"%s\"! (%m)", file);
	  close(d.ctlfd);
	  return NULL;
	}
	if (!S_ISREG(stbuf.st_mode)) {
	  warning("Control file \"%s\" is not a regular file!", file);
	  close(d.ctlfd);
	  return NULL;
	}
	/* 4 is the minimum number of characters per line */
	n = sizeof (long) * (stbuf.st_size / 4);
	if ((d.contents = emalloc((u_int)stbuf.st_size+1)) == NULL
	    || (d.offset = (long *)emalloc((u_int)n)) == NULL) {
	  warning("Out of virtual memory!", (char *)NULL);
	  exit(EX_SOFTWARE);
	}
	d.contentsize = stbuf.st_size;
	d.contents[d.contentsize] = 0; /* Treat it as a long string.. */
	if (read(d.ctlfd, d.contents, d.contentsize) != d.contentsize) {
	  warning("Wrong size read from control file \"%s\"! (%m)",
		  file);
	  free(d.contents);
	  free((char *)d.offset);
	  close(d.ctlfd);
	  return NULL;
	}
	n = markoff(d.contents, d.contentsize, d.offset, file);
	if (n < 4) {
	  /*
	   * If it is less than the minimum possible number of control
	   * lines, then there is something wrong...
	   */
	  free(d.contents);
	  free((char *)d.offset);
	  close(d.ctlfd);
	  warning("Truncated or illegal control file \"%s\"!", file);
	  /* exit(EX_PROTOCOL); */
	  sleep(60);
	  return NULL;
	}

	d.ctlid = stbuf.st_ino;
	d.senders = NULL;
	d.recipients = NULL;
	d.logident = "none";
	d.verbose = NULL;

	/* run through the file and set up the information we need */
	for (i = 0; i < n; ++i) {
	  if (*exitflagp && d.recipients == NULL)
	    break;
	  s = d.contents + d.offset[i];
	  switch (*s) {
	  case _CF_SENDER:
	    ap = ctladdr(s+2);
	    ap->link = d.senders;
	    d.senders = ap;
	    break;
	  case _CF_RECIPIENT:
	    if (*++s != _CFTAG_NORMAL || d.senders == NULL)
	      break;
	    if ((ap = ctladdr(s+1)) == NULL) {
	      warning("Out of virtual memory!", (char *)NULL);
	      *exitflagp = 1;
	      break;
	    }
	    /* [mea] understand  'host' of type:
	       "((mxer)(mxer mxer))"   */
	    if ((channel != NULL
		 && strcmp(channel, ap->channel) != 0)
#if 1
		|| (ap->routermxes != NULL && matchrouter != NULL
		    && !(*matchrouter)(host, ap))
#endif
		|| (ap->routermxes == NULL && selectaddr != NULL
		    && !(*selectaddr)(host, ap->host))
		|| (ap->routermxes == NULL && selectaddr == NULL
		    && host != NULL && cistrcmp(host,ap->host) !=0)
		|| !lockaddr(d.ctlfd, d.offset[i]+1,
			     _CFTAG_NORMAL, _CFTAG_LOCK)) {
	      if (ap->routermxes)
		free((char *)ap->routermxes);
	      ap->routermxes = NULL;
	      free((char *)ap);
	      break;
	    }
	    ap->link = d.senders; /* point at sender address */
	    rp = (struct rcpt *)emalloc(sizeof (struct rcpt));
	    if (rp == NULL) {
	      lockaddr(d.ctlfd, d.offset[i]+1,
		       _CFTAG_LOCK, _CFTAG_DEFER);
	      warning("Out of virtual memory!", (char *)NULL);
	      *exitflagp = 1;
	      break;
	    }
	    rp->addr = ap;
	    rp->id = d.offset[i];
	    /* XX: XOR locks are different */
	    rp->lockoffset = rp->id + 1;
	    rp->next = d.recipients;
	    rp->desc = &d;
	    d.recipients = rp;
	    rp->status = EX_OK;
	    rp->newmsgheader = NULL;
	    break;
	  case _CF_MSGHEADERS:
	    {
	      char **msgheader = NULL;
	      char ***mmp;
	      char *ss;
	      int  headerlines = 0;
	      int  headerspace = 0;
	      int  headersize  = strlen(s);

	      if (headersize > largest_headersize)
		largest_headersize = headersize;
	      /* position pointer at start of the header */
	      while (*s && *s != '\n')
		++s;
	      ++s;

	      /* Collect all the header lines into
		 individual lines.. [mea] */
	      while (*s) {
		if (headerlines == 0) {
		  msgheader = (char **)malloc(sizeof(char**)*8);
		  headerspace = 7;
		}
		if (headerlines+1 >= headerspace) {
		  headerspace += 8;
		  msgheader =
		    (char**)realloc(msgheader,
				    sizeof(char**)*headerspace);
		}
		ss = s;
		while (*ss && *ss != '\n') ++ss;
		if (*ss == '\n') *ss++ = '\0';
		msgheader[headerlines++] = s;
		msgheader[headerlines] = NULL;
		s = ss;
	      }

	      /* And the global connection.. */
	      if (headers_cnt == 0) {
		msgheaders = (char***)malloc(sizeof(char***)*8);
		msgheaderscvt = (char***)malloc(sizeof(char***)*8);
		headers_spc = 7;
	      }
	      if (headers_cnt+1 >= headers_spc) {
		headers_spc += 8;
		msgheaders =
		  (char***)realloc(msgheaders,
				   sizeof(char***)*headers_spc);
		msgheaderscvt =
		  (char***)realloc(msgheaderscvt,
				   sizeof(char***)*headers_spc);
	      }
	      msgheaders   [headers_cnt] = msgheader;
	      msgheaderscvt[headers_cnt] = NULL;

	      /* fill in header * of recent recipients */
	      for (rp = d.recipients;
		   rp != NULL && rp->newmsgheader == NULL;
		   rp = rp->next) {
		rp->newmsgheader    = &msgheaders   [headers_cnt];
		rp->newmsgheadercvt = &msgheaderscvt[headers_cnt];
	      }

	      msgheaders   [++headers_cnt] = NULL;
	      msgheaderscvt[  headers_cnt] = NULL;
	    }
	    break;
	  case _CF_MESSAGEID:
	    d.msgfile = s+2;
	    break;
	  case _CF_BODYOFFSET:
	    d.msgbodyoffset = (long)atoi(s+2);
	    break;
	  case _CF_LOGIDENT:
	    d.logident = s+2;
	    break;
	  case _CF_VERBOSE:
	    d.verbose = s+2;
	    break;
	  default:		/* We don't use them all... */
	    break;
	  }
	}

	d.msgheaders    = msgheaders;		/* Original headers	*/
	d.msgheaderscvt = msgheaderscvt;	/* Modified set		*/

	if (d.recipients == NULL) {
	  ctlclose(&d);
	  return NULL;
	}

	mfpath = emalloc((u_int)5+sizeof(QUEUEDIR)+strlen(d.msgfile));
	sprintf(mfpath, "../%s/%s", QUEUEDIR, d.msgfile);
	if ((d.msgfd = open(mfpath, O_RDONLY, 0)) < 0) {
	  for (rp = d.recipients; rp != NULL; rp = rp->next) {
	    diagnostic(rp, EX_UNAVAILABLE,
		       "message file is missing(!) -- possibly due to delivery scheduler restart.  Consider resending your message");
	  }
	  warning("Cannot open message file \"%s\"! (%m)", mfpath);
	  free(mfpath);
	  ctlclose(&d);
	  return NULL;
	}
	if (fstat(d.msgfd,&stbuf) < 0) {
	  stbuf.st_mode = S_IFCHR; /* Make it to be something what it
				      clearly can't be.. */
	}
	if (!S_ISREG(stbuf.st_mode)) {
	  for (rp = d.recipients; rp != NULL; rp = rp->next) {
	    diagnostic(rp, EX_UNAVAILABLE,
		       "Message file is not a regular file!");
	  }
	  warning("Cannot open message file \"%s\"! (%m)", mfpath);
	  free(mfpath);
	  ctlclose(&d);
	  close(d.msgfd);
	  return NULL;
	}

	d.msgsizeestimate = stbuf.st_size + largest_headersize;
	/* A nice fudge factor, usually this is enough..                 */
	/* Add 3% for CRLFs.. -- assume average line length of 35 chars. */
	d.msgsizeestimate += (3 * d.msgsizeestimate) / 100;

	return &d;
}

int
ctlsticky(spec_host, addr_host)
	char *spec_host, *addr_host;
{
	static char *hostref = NULL;

	if (hostref == NULL) {
	  if (spec_host != NULL)
	    hostref = spec_host;
	  else
	    hostref = addr_host;
	}
	if (spec_host == NULL && addr_host == NULL) {
	  hostref = NULL;
	  return 0;
	}
	return cistrcmp(hostref, addr_host) == 0;
}
