/* -*- C -*-
 *
 * Program:	ximap
 * File:        readwin.c -- reading a message or sequence of messages.
 *
 * Author:	Kevin Brock
 *	        Symbolic Systems Resources Group
 *		Stanford University
 *              MSOB x241
 *		Stanford, CA 94305
 *		Internet: brock@CAMIS.Stanford.Edu
 *
 * Date:	07 September 1992
 *
 * Copyright 1992 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and
 * THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL WARRANTIES, EXPRESS OR 
 * IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN NO EVENT 
 * SHALL THE LELAND STANFORD JUNIOR UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT 
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, 
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) 
 * OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 *
 * Last edited:
 *     14 Octobre 92 - WYeager : Enlever la function "SaveToFile". Celle-ci peut
 *     planter l'ordinateur!
 *
 */
#include <string.h>
#include <netdb.h>
#include <signal.h>
#include <ctype.h>

#include <stdio.h>
#include <memory.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <Client/osdep.h>
#include <Client/mail.h>
#include <Client/misc.h>
#include <Client/rfc822.h>
 
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include <X11/Xaw/Text.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/AsciiSrc.h>

#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>

#include <Xb/Browser.h>

#include "structures.h"

#include "message.h"
#include "globals.h"
#include "buttons.h"
#include "resources.h"
#include "textutil.h"
#include "addrutil.h"
#include "util.h"
#include "browserutil.h"
#include "ximap.h"
#include "readwin.h"
#include "compose.h"
#include "mailbox.h"
#include "filebrowser.h"
#include "modal.h"

/*  This section contains stuff for manipulating the BODY stack. */

static BODY*      BodyStack_pop(/* BodyStack* */);
static void       BodyStack_push(/* BODY*, BodyStack* */);
static BodyStack* BodyStack_new();
static void       BodyStack_delete(/* BodyStack* */);
static int        BodyStack_empty(/* BodyStack* */);
static BODY*      BodyStack_top(/* BodyStack* */);

static BODY* BodyStack_top(stack)
     BodyStack* stack;
{
    if( stack->gStackSize )
	return(*stack->gStackTop);

    return(NULL);
}

static int BodyStack_empty(stack)
     BodyStack* stack;
{
    return( !stack->gStackSize );
}

static void BodyStack_delete(stack)
     BodyStack* stack;
{
    free( stack->gStackBottom );
    free( stack );
}

static BodyStack* BodyStack_new()
{
    BodyStack* ret = (BodyStack*) malloc (sizeof(BodyStack));

    ret->gStackSize = 0;
    ret->gStackCapacity = 1;
    ret->gStackBottom = (BODY**) malloc (sizeof(BODY*));
    ret->gStackTop = ret->gStackBottom;

    return(ret);
}

static BODY* BodyStack_pop(stack)
     BodyStack *stack;
{
    BODY *ret = NULL;


    if(stack->gStackSize)            
    {
	ret = *stack->gStackTop;
	stack->gStackSize--;
	*stack->gStackTop = NULL;
	if(stack->gStackSize)
	    stack->gStackTop--;
    }
	
    return(ret);
}

static void  BodyStack_push(body, stack)
     BODY *body;
     BodyStack *stack;
{
    if( !stack || !body )
	return;
 
    if(stack->gStackSize)
    {
	if(stack->gStackSize < stack->gStackCapacity)
	{
	    stack->gStackTop++;
	}
	else
	{
	    stack->gStackCapacity *= 2;
	    stack->gStackBottom = 
		(BODY**) realloc (stack->gStackBottom, stack->gStackCapacity * sizeof(BODY*));
	    stack->gStackTop = stack->gStackBottom + stack->gStackSize;
	}
    }

    *stack->gStackTop = body;
    stack->gStackSize++;
}

static PartPosition* PartPosition_new();
static void          PartPosition_add();
static void          PartPosition_up();
static void          PartPosition_increment();
static void          PartPosition_decrement();
static char*         PartPosition_retrieve();
static void          PartPosition_clear();

static void PartPosition_clear(pp)
     PartPosition *pp;
{
    if(pp->gPositionArray)
	free(pp->gPositionArray);

    pp->gArraySize = 0;
    pp->gArrayCapacity = 0;
    pp->gPositionArray = NULL;
    pp->gPositionArrayBottom = NULL;
}

static char* PartPosition_retrieve(pp)
     PartPosition *pp;
{
    char *ret = NULL;
    char *temp = NULL;

    if (pp->gArraySize)
    {
	int i = pp->gArraySize;
	ret = malloc(4*i + 1);
	temp = ret;
	sprintf(temp, "%d", pp->gPositionArray[0]);
	for(i = 1; i < pp->gArraySize; i++)
	{
	    strcat(ret, ".");
	    temp += strlen(temp);
	    sprintf(temp,"%d", pp->gPositionArray[i]);
	}
    }

    return(ret);
}

static void PartPosition_increment(pp)
     PartPosition *pp;
{
    if( pp->gArraySize )
	++*pp->gPositionArrayBottom;
}

static void PartPosition_decrement(pp)
     PartPosition *pp;
{
    if( pp->gArraySize )
	--*pp->gPositionArrayBottom;
}

static void PartPosition_up(pp)
     PartPosition *pp;
{
    if(pp->gArraySize)
    {
	pp->gArraySize--;
	*pp->gPositionArrayBottom-- = 0;
    }
}

static void PartPosition_add(pp, d)
     PartPosition *pp;
     int d;
{
    pp->gArraySize++;

    if(pp->gArrayCapacity == 0)                   /* First addition */
    {
	pp->gArrayCapacity = 1;
	pp->gPositionArray = (int*) malloc( pp->gArrayCapacity * sizeof(int*));
    }
    else if(pp->gArraySize > pp->gArrayCapacity)
    {
	pp->gArrayCapacity *= 2;
	pp->gPositionArray = (int*) realloc(pp->gPositionArray, pp->gArrayCapacity * sizeof(int*));
    }

    pp->gPositionArrayBottom = pp->gPositionArray + pp->gArraySize - 1;
    *pp->gPositionArrayBottom = d;
}

static PartPosition* PartPosition_new()
{
    PartPosition *ret = (PartPosition*) malloc (sizeof(PartPosition));

    ret->gArraySize = 0;
    ret->gArrayCapacity = 0;
    ret->gPositionArray = NULL;
    ret->gPositionArrayBottom = NULL;
    
    return(ret);
}

void read_markDeleted(/* Widget, READWINDOW*, XEvent* */);
void read_markUndeleted(/* Widget, READWINDOW*, XEvent* */);
void read_markSeen(/* Widget, READWINDOW*, XEvent* */);
void read_markUnseen(/* Widget, READWINDOW*, XEvent* */);
void read_markRecent(/* Widget, READWINDOW*, XEvent* */);
void read_markOld(/* Widget, READWINDOW*, XEvent* */);
void read_markFlagged(/* Widget, READWINDOW*, XEvent* */);
void read_markUnflagged(/* Widget, READWINDOW*, XEvent* */);
void read_markAnswered(/* Widget, READWINDOW*, XEvent* */);
void read_markUnanswered(/* Widget, READWINDOW*, XEvent* */);

void read_setFlag(/* READWINDOW*, char* */);
void read_clearFlag(/* READWINDOW*, char* */);
void read_setKeyword(/* Widget, READWINDOW*, void* */);
void read_clearKeyword(/* Widget, READWINDOW*, void* */);

static void NextMessage(/* Widget, READWINDOW* , XEvent* */);
static void Quit(/* Widget, READWINDOW* , XEvent* */);
static void PreviousMessage(/* Widget, READWINDOW* , XEvent* */);
static void ReplyToSender(/* Widget, READWINDOW* , XEvent* */);
static void ReplyToAll(/* Widget, READWINDOW* , XEvent* */);
static void ForwardMessage(/* Widget, READWINDOW* , XEvent* */);
static void RemailMessage(/* Widget, READWINDOW* , XEvent* */);

static void PrintMessage(/* Widget, READWINDOW* , XEvent* */);
static void CopyMessage(/* Widget, READWINDOW* , XEvent* */);
static void MoveMessage(/* Widget, READWINDOW* , XEvent* */);
static void ToggleHeader(/* Widget, READWINDOW* , XEvent* */);
static void KillMessage();

static void DisplayText();
static void DisplayBody();
void ChangeTextPortion();
void nomimeChangeTextPortion();
static char* GetCurrentText(/*  READWINDOW*  */);
static char* nomimeGetCurrentText(/*  READWINDOW*  */);

static void SaveToFile(/* Widget, READWINDOW* , XEvent* */);
static void ViewPart(/* Widget, READWINDOW* , XEvent* */);

#ifdef JUMPREADY  /* not ready for prime time... */
static void JumpTo(/* Widget, READWINDOW* , XEvent* */);
static void JumpToPart(/* Widget, READWINDOW* , XEvent* */);
#endif

static void Down(/* Widget, READWINDOW* , XbListSelection* */);

static void Up(/* Widget, READWINDOW* , XEvent* */);
static void DownNext(/* Widget, READWINDOW* , XEvent* */);
static void ShowRaw();

static void ResizeBrowser();
static void ResizeText();

#ifdef JUMPREADY  /* not ready for prime time... */
void read_setUpJumpToMessage(/* READWINDOW* */);
void read_setUpJumpToPart(/* READWINDOW* */);
ButtonStruct *jump_to_part_menu; 
ButtonStruct *jump_to_message_menu; 
#endif

void read_setupKeywords(/* READWINDOW* */);
static ButtonStruct  read_clear_menu[30];
static ButtonStruct  read_set_menu[30];

ButtonStruct read_keywords_menu[]=
{
  { "clear",NULL, read_clear_menu , NEVER},
  { "set",NULL, read_set_menu,  NEVER},
  NULL,
};

ButtonStruct read_flags_menu[] = 
{

  { "seen",      read_markSeen     ,  NULL,  NEVER },
  { "unseen",    read_markUnseen   ,  NULL , NEVER },
  { "recent",    read_markRecent ,    NULL , NEVER },
  { "old",       read_markOld ,       NULL  ,  NEVER },
  { "answered",  read_markAnswered ,  NULL ,  NEVER },
  { "unanswered",read_markUnanswered, NULL  , NEVER },
  { "flagged",   read_markFlagged,    NULL,  NEVER },
  { "unflagged", read_markUnflagged,  NULL , NEVER  },
  NULL,
};

ButtonStruct read_reply_menu[] =
{
  { "sender",   ReplyToSender,  NULL,  NEVER },
  { "all",      ReplyToAll,  NULL ,  NEVER},
  NULL,
};

ButtonStruct read_response_menu[] = 
{
  { "Reply", NULL, read_reply_menu,  NEVER  },
  { "Remail", RemailMessage, NULL, NEVER },
  { "Forward", ForwardMessage, NULL, NEVER },
  NULL,
};

ButtonStruct read_copy_menu[] = 
{
  { "Copy", CopyMessage, NULL, NEVER },
  { "Move", MoveMessage, NULL, NEVER },
  { "LocalCopy", SaveToFile, NULL, NEVER },
  NULL,
};

ButtonStruct read_delete_menu[] =
{
  { "Delete",   read_markDeleted  ,  NULL ,  NEVER },
  { "Undelete", read_markUndeleted,  NULL ,  NEVER },
  NULL,
};

ButtonStruct rwinbuttons[] = 
{
  { "Quit",    Quit,  NULL,  NEVER },
  { "Delete",  NULL,  read_delete_menu,  NEVER},
  { "Print",  PrintMessage, NULL,  NEVER},
  { "Copy",    NULL,   read_copy_menu,  NEVER },
  { "Reply",   NULL,     read_response_menu , NEVER},
  { "Flags",   NULL,     read_flags_menu,  NEVER },     
  { "Keywords",NULL,  read_keywords_menu,  NEVER },
  { "HeaderMode", ToggleHeader, NULL,  NEVER },
  { "ShowRaw",    ShowRaw,  NULL, NOMIME},
  NULL,
};

ButtonStruct panelbuttons[] = 
{
  { "ViewPart",      ViewPart,    NULL,  NOHANDLER}, 
  NULL,
};

ButtonStruct partbuttons[] = 
{
  { "Next",       NextMessage,     NULL,  NEVER}, 
  { "Previous",   PreviousMessage, NULL,  FIRSTMESSAGE },
  { "Kill",       KillMessage,     NULL,  NEVER },
  { "Up",         Up,              NULL,  TOPLEVEL },
#ifdef JUMPREADY
  { "JumpToPart", JumpToPart,      NULL,  NOBODYPART }, 
  { "JumpTo",     JumpTo,          NULL,  NEVER },
#endif
  NULL,
};

static void ResizePanes(/*  READWINDOW*  */);

static Boolean ReadWindow_verifyMessage(/*  READWINDOW*  */);
static Boolean SetNextMessage(/*  READWINDOW*  */);
static Boolean SetPreviousMessage(/*  READWINDOW*  */);

static char* GetHead();
static char* GetFilteredHead(/*  READWINDOW*  */);

void exitRead(/*  MAILSTATUS*, READWINDOW*  */);

static void readPrintAct();
static void readToggleHeaderAct();
static void readNextAct();
static void readPrevAct();
static void readKillAct();
static void readUndeleteAct();
static void readDeleteAct();
static void readCopyAct();
static void readMoveAct();
static void readLocalCopyAct();
static void readQuitAct();
static void readForwardAct();
static void readRemailAct();
static void readReplyAllAct();
static void readReplySenderAct();
static void readFlagAct();
static void readUnflagAct();
static void readDownAct();
static void readUpAct();

static XtActionsRec rwin_actions[] = {
  { "rwin_unflag",     readUnflagAct },
  { "rwin_flag",       readFlagAct },
  { "rwin_undelete",   readUndeleteAct },
  { "rwin_delete",     readDeleteAct },
  { "rwin_kill",       readKillAct },
  { "rwin_prev",       readPrevAct },
  { "rwin_next",       readNextAct },
  { "rwin_copy",       readCopyAct },
  { "rwin_move",       readMoveAct },
  { "rwin_local",      readLocalCopyAct },
  { "rwin_close",      readQuitAct },
  { "rwin_forward",    readForwardAct },
  { "rwin_remail",     readRemailAct },
  { "rwin_replyall",   readReplyAllAct },
  { "rwin_replysender",readReplySenderAct },
  { "rwin_togglehead", readToggleHeaderAct },
  { "rwin_print",      readPrintAct },
  { "rwin_down",       readDownAct },
  { "rwin_up",         readUpAct },
};

char *typeNames[] =
{
    "TEXT",
    "MULTIPART",
    "MESSAGE",
    "APPLICATION",
    "AUDIO",
    "IMAGE",
    "VIDEO",
    "OTHER"
};

extern AppResources res;             
extern XtAppContext app_context;
extern char *typeAcronyms[];

Arg warg[ARGLISTSIZE];
int n = 0;

void addReadWindowActions(ac)
     XtAppContext ac;
{
    XtAppAddActions(ac, rwin_actions, XtNumber(rwin_actions));
}

READWINDOW* createReadWindow(w, nodelist, ms)
     Widget w;
     NodeList* nodelist;
     MailBox*  ms;
{
    XbListSelectionPtr dummyList = NULL;

    READWINDOW* rwin = (READWINDOW*) XtMalloc (sizeof(READWINDOW));
    
    rwin->name = (char*) XtMalloc ((res.line_length+1)*sizeof(char));

    rwin->ms = ms;
    rwin->nmsgs = nodelist->length;
    rwin->messages = nodelist->nodes;
    rwin->current = nodelist->nodes;

    rwin->text = NULL;
    rwin->user_height = 0;

    rwin->currpart = NULL;
    rwin->panes = NULL;
    rwin->buttons = NULL;

    rwin->state = FIRSTMESSAGE | TOPLEVEL | NOBODYPART | NOENCODEDDATA;
    rwin->panel_state = NOHANDLER;

    rwin->text_height = 0;
    rwin->next = 0;
    rwin->currtext = NULL;
    rwin->filterHead = ms->filter_heads;
    rwin->rawText = FALSE;

    rwin->dform = NULL;
    rwin->browser = NULL;

    rwin->handle = mail_makehandle(ms->mailbox);
    (*rwin->current)->visible++;

    rwin->bodystack = BodyStack_new();
    rwin->partposition = PartPosition_new();

    while(!ReadWindow_verifyMessage(rwin))
	SetNextMessage(rwin);

    if( ReadWindow_verifyMessage(rwin) )
    { 
	sprintf(rwin->name,"Message %4d",(*rwin->current)->msgno);
	XtSetArg(warg[n], XtNtitle, rwin->name); n++;
	XtSetArg(warg[n], XtNiconName, rwin->name); n++;
	XtSetArg(warg[n], XtNallowShellResize, TRUE); n++;
	rwin->shell = XtCreatePopupShell("read_shell",topLevelShellWidgetClass,
					 w, warg, n); n = 0;
	
	rwin->panes = XtCreateManagedWidget("read_panes", panedWidgetClass,
					    rwin->shell, warg, n); n = 0;

	XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
	XtSetArg(warg[n], XtNresizeToPreferred, TRUE); n++;
	XtSetArg(warg[n], XtNskipAdjust, TRUE); n++;
	rwin->buttons = XtCreateManagedWidget("read_buttons", boxWidgetClass,
					      rwin->panes,  warg, n); n = 0;
	
	XtSetArg(warg[n], XtNscrollVertical, XawtextScrollAlways); n++;
	XtSetArg(warg[n], XtNeditType, XawtextRead); n++;
	XtSetArg(warg[n], XtNwrap, XawtextWrapWord); n++;
	XtSetArg(warg[n], XtNdisplayCaret, FALSE); n++;
	XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
	XtSetArg(warg[n], XtNallowResize, TRUE); n++;
	rwin->text =  XtCreateManagedWidget("read_text", asciiTextWidgetClass,
					    rwin->panes, warg, n); n = 0;
	
	XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
	XtSetArg(warg[n], XtNskipAdjust, TRUE); n++;
	XtSetArg(warg[n], XtNresizeToPreferred, TRUE); n++;
	rwin->part_buttons = XtCreateManagedWidget("part_buttons", boxWidgetClass,
						   rwin->panes,  warg, n); n = 0;


	XtSetArg(warg[n], XtNallowResize, TRUE); n++;
	XtSetArg(warg[n], XtNskipAdjust, TRUE); n++;
	XtSetArg(warg[n], XtNresizeToPreferred, TRUE); n++;
	XtSetArg(warg[n], XtNborderWidth, 0); n++;
	rwin->dform = XtCreateManagedWidget("display_form", formWidgetClass,
				     rwin->panes, warg, n); n = 0;

	XtSetArg(warg[n], XtNborderWidth, 0); n++;
	XtSetArg(warg[n], XtNlabel, "                                          "); n++;
	rwin->panel_label = XtCreateWidget("panel_label", 
					   labelWidgetClass, rwin->dform, warg, n); n = 0;

	XtSetArg(warg[n], XtNorientation, XtorientHorizontal); n++;
	XtSetArg(warg[n], XtNborderWidth, 0); n++;
	XtSetArg(warg[n], XtNfromVert, rwin->panel_label); n++;
	rwin->panel_buttons = XtCreateWidget("panel_buttons", boxWidgetClass,
					      rwin->dform,  warg, n); n = 0;
	
	dummyList = ximap_newlistmap(1);
	dummyList->entries[0].index = 0;
	dummyList->entries[0].entry = cpystr("");
	XtSetArg(warg[n], XtNlist, dummyList); n++;
	XtSetArg(warg[n], XtNinitialViewHt, 5); n++;
	XtSetArg(warg[n], XtNresizable, TRUE); n++;
	rwin->browser = XtCreateWidget("read_browser", XbbrowserWidgetClass,
				       rwin->dform, warg, n); n = 0;

	XtAddCallback(rwin->browser, XtNdoubleClick, Down, rwin);

	XtSetArg(warg[n], XtNborderWidth, 0); n++;
	XtSetArg(warg[n], XtNlabel, "                                          "); n++;
	XtCreateManagedWidget("dummy", labelWidgetClass, rwin->dform, warg, n); n = 0;
	
	read_setupKeywords(rwin);

	createButtons(rwin->buttons, 
		      rwin,
		      rwinbuttons);

	rwin->panel_states = createButtons(rwin->panel_buttons, 
					  rwin,
					  panelbuttons);

	rwin->states = createButtons(rwin->part_buttons, 
				     rwin,
				     partbuttons);

	(*rwin->current)->visible += 1;

	PartPosition_add(rwin->partposition, 1);
	ResizePanes(rwin);
	rwin->display = DisplayBody;
	rwin->change_text = ChangeTextPortion;
	rwin->current_text = GetCurrentText;

	/* If there's no body, we're not taliking to a MIME compliant server... */
	if(!(*rwin->current)->body)
	{
	    rwin->state |= NOMIME;
	    rwin->display = DisplayText;
	    rwin->change_text = nomimeChangeTextPortion;
	    rwin->current_text = nomimeGetCurrentText;
	}

	rwin->display(rwin, (*rwin->current)->body);

	XtPopup(rwin->shell, XtGrabNone);
	checkButtonStates(rwin->states, rwin->state);
	XtSetKeyboardFocus( rwin->shell, rwin->text );

	return( rwin );
    }
    else
    {
	Quit( w, rwin, NULL );
	return( NULL );
    }
    
}

static void 
Quit(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent *event;
{
    READWINDOW* rtmp = rwin->ms->rwinList;
    char *sequence = NULL;
    int i = 0;
    NodeList*   nodelist = ximap_newnodelist(rwin->messages, rwin->nmsgs);

    
    if ( rwin->ms )
    {
	for( i = 0; i < rwin->nmsgs; i++)
	{
	    if (rwin->messages[i])
		rwin->messages[i]->locked--;
	}
	
	if ( rwin->nmsgs != 0 )
	    sequence = updateSequence( nodelist );
	
	if ( sequence != NULL
	    && !rwin->ms->mailbox->lock)
		updateHeaders( rwin->ms, nodelist );
    }

    if(rwin->states)
	free_statelist(rwin->states);

    if (rwin->shell)
	XtDestroyWidget(rwin->shell);

    for( i = 0; read_set_menu[i].name; i++)
	XtFree(read_set_menu[i].name);

    for( i = 0; read_clear_menu[i].name; i++)
	XtFree(read_clear_menu[i].name);

    BodyStack_delete(rwin->bodystack);
    if( rtmp )
    {   
	if (rtmp == rwin)
	{
	    rwin->ms->rwinList = rwin->next;
	}
	else
	{
	    while( rtmp->next && rtmp->next !=  rwin )
		rtmp = rtmp->next;
	    
	    if ( rtmp->next )
		rtmp->next = rwin->next;
	}
    }
    XtFree(rwin);
}

static void NextMessage(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent *event;
{
    BODY *body = NULL;
    if ( rwin != NULL )
    {
	rwin->rawText = FALSE;
	/* If we're in a multi-part, navigate through parts, not messages */
	if(!(rwin->state & NOMIME))
	{
	    if(body = BodyStack_top(rwin->bodystack))
	    {
		PART *part = body->contents.part;
		PART *cp  = rwin->currpart;
		
		while( part && cp != part )
		    part = part->next;
		
		if( part && part->next )    /* If there's a next part, read it. */
		{
		    rwin->currpart = part->next;
		    PartPosition_increment(rwin->partposition);
		    rwin->display(rwin, &part->next->body);
		}
		else                /* Else go up a level, and on to the next piece. */
		{
		    BodyStack_pop(rwin->bodystack);
		    PartPosition_up(rwin->partposition);
		    rwin->currpart = NULL;
		    NextMessage(w, rwin, NULL);
		}
		return;
	    }
	}
	
	if (SetNextMessage( rwin ))
	{
	    PartPosition_clear(rwin->partposition);  /* Top level, whole new part list.    */
	    PartPosition_add(rwin->partposition,1);  /* We always start out with part 1... */
	    rwin->currpart = NULL;
	    rwin->state |= NOBODYPART;
	    rwin->display(rwin, (*rwin->current)->body);
	    return;
	}
	else
	{
	    mm_log("Last Message...", NIL);
	    Quit(w, rwin, event);
	}
    }
}

static void PreviousMessage(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent *event;
{
    BODY *body;
    if ( rwin != NULL )
    {
	if(!(rwin->state & NOMIME))
	{
	    if(body = BodyStack_top(rwin->bodystack))
	    {
		PART *part = body->contents.part;
		PART *cp   = rwin->currpart;

		if( cp == part )   /* If cp is the first part, upsy-daisy! */
		{
		    BodyStack_pop(rwin->bodystack);
		    PartPosition_up(rwin->partposition);
		    rwin->currpart = NULL;
		    PreviousMessage(w, rwin, NULL);
		}
		
		while( part && cp != part->next )
		    part = part->next;
		
		if( part )         /* If there's a next part, read it. */
		{
		    rwin->currpart = part;
		    PartPosition_decrement(rwin->partposition);
		    rwin->display(rwin, &part->body);
		}

		return;
	    }
	}

	if (SetPreviousMessage(rwin))
	{
	    PartPosition_clear(rwin->partposition);
	    PartPosition_add(rwin->partposition,1);
	    rwin->currpart = NULL;
 	    rwin->display(rwin, (*rwin->current)->body);
	}
	else
	{
	    checkButtonStates(rwin->states, rwin->state);
	}
    }
}

static  void ReplyToSender(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent *event;
{
    if (!(*rwin->current)->elt)
      return;

    (*rwin->current)->status &= ~INRESPONSE;
    (*rwin->current)->status  |= REPLYSENDER;

    doReply(rwin, rwin->ms, ximap_newnodelist(rwin->current, 1));

    (*rwin->current)->status &= ~INRESPONSE;
}

static  void ReplyToAll(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent *event;
{
    if (!(*rwin->current)->elt)
      return;

    (*rwin->current)->status &= ~INRESPONSE;
    (*rwin->current)->status  |= REPLYALL;

    doReply(rwin, rwin->ms, ximap_newnodelist(rwin->current, 1));

    (*rwin->current)->status &= ~INRESPONSE;
}

static  void RemailMessage(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent *event;
{
    if (!(*rwin->current)->elt)
      return;

    (*rwin->current)->status &= ~INRESPONSE;
    (*rwin->current)->status  |= REMAILMSG;

    doRemail(rwin->ms, ximap_newnodelist(rwin->current, 1));

    (*rwin->current)->status &= ~INRESPONSE;
}

static void PrintMessage(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent *event;
{
    char printit;

    if (!(*rwin->current)->elt)
      return;

    printit = XbConfirm(rwin->shell,
			"Print current message...", 
			TRUE);

    if ( printit == TRUE )
	doPrint(rwin->ms, ximap_newnodelist(rwin->current, 1));
}

static void CopyMessage(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent *event;
{
  if (!(*rwin->current)->elt)
    return;
  doCopy( rwin->ms, ximap_newnodelist(rwin->current, 1));
}

static void MoveMessage(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent *event;
{
    if (!(*rwin->current)->elt)
      return;
  doMove( rwin->ms, ximap_newnodelist(rwin->current, 1));
}

static void ForwardMessage(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent *event;
{
    if (!(*rwin->current)->elt)
      return;

  doForward( rwin->ms, ximap_newnodelist(rwin->current, 1));
}

static  void ResizePanes(rwin)
     READWINDOW* rwin;
{
    Dimension width = 0;
    XFontStruct *font;

    XtSetArg(warg[n], XtNfont, &font); n++;
    XtGetValues(rwin->text, warg, n); n = 0;
	
    width = (font->max_bounds.width) * 85;

    XtSetArg(warg[n], XtNwidth, width); n++;
    XtSetValues(rwin->panes, warg, n); n = 0;

    XtRealizeWidget(rwin->shell);
}

static  void ResizeBrowser(rwin)
     READWINDOW* rwin;
{
    XtWidgetGeometry ask, reply;

    Dimension current_height;

    XtSetArg(warg[n], XtNheight, &current_height); n++;
    XtGetValues(rwin->browser, warg, n); n = 0;
	
    ask.request_mode = CWHeight | XtCWQueryOnly;
    XtQueryGeometry(rwin->browser, &ask, &reply);

    if(current_height < reply.height)
    {
	XtSetArg(warg[n], XtNheight, reply.height); n++;
	XtSetValues(rwin->browser, warg, n); n = 0;
    }
}

static  void ResizeText(rwin, headerOnly)
     READWINDOW* rwin;
     int         headerOnly;
{
    Dimension text_height = 0;
    Dimension current_height = 0;
    Dimension default_height = 0;

    XFontStruct *font;
    char* str = NULL;
    char* temp = NULL;
    int i = 0;

    XawPanedSetRefigureMode(rwin->panes, False);

    XtSetArg(warg[n], XtNfont, &font); n++;
    XtSetArg(warg[n], XtNheight, &current_height); n++;
    XtSetArg(warg[n], XtNstring, &str); n++;
    XtGetValues(rwin->text, warg, n); n = 0;

    while((temp = strchr(str,'\n')) && i < 20)
    {
	i++;
	str = temp+1;
    }

    /* More lines than newlines...*/
    i+=2;

    text_height=i*(font->max_bounds.ascent+font->max_bounds.descent);
    default_height=20*(font->max_bounds.ascent+font->max_bounds.descent);

    if(current_height > default_height)
	rwin->user_height = current_height;

    if(headerOnly)
	default_height = text_height;
    else if( rwin->user_height )
	default_height =  rwin->user_height;

    XtSetArg(warg[n], XtNheight, default_height); n++;
    XtSetValues(rwin->text, warg, n); n = 0;

    XawPanedSetRefigureMode(rwin->panes, True);
}

static Boolean ReadWindow_verifyMessage(rwin)
     READWINDOW* rwin;
{
    if(rwin                               /* We have a read window          */
       && (*rwin->current)                /* There exists a current message */
       && (*rwin->current)->elt != NULL) /* The message envelope exists    */
	return( TRUE );
    else
	return( FALSE );
}

static Boolean SetNextMessage(rwin)
     READWINDOW* rwin;
{
    MessageNode  **m = ++rwin->current;
    MessageNode   *temp = NULL;
    int i = 0;

    /* 
     * This should take care of the fact that the c-Client can delete
     * the MESSAGECACHE at any time, without notifying the ReadWindow
     */
    if (m < rwin->messages + rwin->nmsgs)
    {
	(*m)->visible--;
	while (*m                                      /* there is a message               */
	       && m < rwin->messages + rwin->nmsgs     /* and it's a valid message         */
	       && !((*m)->elt))                       /* But it *doesn't* have a cache... */
	{
	    /*
	     * if the bad message isn't the last one, remove the bad message 
	     * from the MessageList.  The fact that the cache has been set to 
	     * NULL *should* indicate that that node has been expunged...
	     *
	     */
	    temp = *m;
	    temp->locked--;
	    if (temp->locked <= 0)
		free_node(temp);

	    for(i = m - rwin->messages; i < rwin->nmsgs - 1; i++)
		rwin->messages[i] = rwin->messages[i+1];
	    rwin->messages[i] = NULL;

	    rwin->nmsgs--;
	}	  
	
	if ( rwin->current - rwin->messages < rwin->nmsgs )
	{
	    (*rwin->current)->visible++;
	    rwin->state &= ~FIRSTMESSAGE;
	    return( TRUE );  /* there is a message */
	}
    }

    return FALSE; /* last message */
}


static Boolean SetPreviousMessage(rwin)
     READWINDOW* rwin;
{
    MessageNode **m = rwin->current;
    MessageNode  *temp = NULL;

    int i = 0;

    /* 
     * This should take care of the fact that the c-Client can delete
     * the MESSAGECACHE at any time, without notifying the ReadWindow
     */
    if (m > rwin->messages)
    {
	m = --rwin->current;
	(*m)->visible--;
	while (*m                                /* there is a message */
	       && m >= rwin->messages            /* and it has a previous  */
	       && !((*m)->elt))             /* but the message node of the 
					            prev. has no MESSAGECACHE* */
	{
	    /* if the previous message has a previous, remove the bad message 
	     * from the MessageList 
	     */
	    temp = *m;
	    temp->locked--;
	    if (temp->locked <= 0)
		free_node(temp);

	    for(i = m - rwin->messages; i < rwin->nmsgs - 1; i++)
		rwin->messages[i] = rwin->messages[i+1];

	    rwin->messages[i] = NULL;
	    rwin->nmsgs--;
	}	  
	
	if ( rwin->current - rwin->messages < rwin->nmsgs )
	{
	    (*rwin->current)->visible++;
	    if(rwin->current != rwin->messages)
		rwin->state &= ~FIRSTMESSAGE;
	    else
		rwin->state |= FIRSTMESSAGE;
	    
	    return( TRUE );  /* there is a message */
	}
    }

    rwin->state |= FIRSTMESSAGE;
    return (FALSE);
}

static char* GetCurrentText(rwin, body)
     READWINDOW *rwin;
     BODY       *body;
{
    char *ret = NULL;
    char *temp, *header;

    long size;

    if(rwin->filterHead)
	header = GetFilteredHead(rwin);
    else
	header = GetHead(rwin);

    if(body->type != TYPEMULTIPART)
	mail_fetchbody(rwin->ms->mailbox, 
		       (*rwin->current)->msgno,
		       PartPosition_retrieve(rwin->partposition),
		       &size);

    if(body->type == TYPETEXT)
    {
	if(rwin->rawText || strcasecmp(body->subtype, "richtext"))
	    ret = gettext_for_display((char*)body->contents.text);
    }
    else if(body->type == TYPEMESSAGE)
    {
	ret = gettext_for_display((char*)body->contents.msg.text);
    }
    else if(rwin->rawText)
    {
	ret = gettext_for_display((char*)body->contents.text);
    }
    
    if(ret)
    {
	temp = malloc( strlen(ret) + strlen(header) + 3 );
	if(temp) {
	  sprintf(temp, "%s\n\n%s", header, ret);
	  rwin->headerlen += 2;
	}
	free(ret);
    }
    else
    {
	temp = malloc(strlen(header) + 1);
	if(temp)
	  strcpy(temp,header);
    }

    free(header); 
    return(temp);
}

static char* nomimeGetCurrentText(rwin, body)
     READWINDOW *rwin;
     BODY       *body;
{
    char *ret = NULL;
    char *temp, *header, *text;

    if(rwin->filterHead)
	header = GetFilteredHead(rwin);
    else
	header = GetHead(rwin);

    text = mail_fetchtext(rwin->ms->mailbox,(*rwin->current)->msgno);
    ret = gettext_for_display(text);
    fs_give(&text);

    if(ret)
    {
	temp = malloc( strlen(ret) + strlen(header) + 3 );
	if(temp) {
	  sprintf(temp, "%s\n\n%s", header, ret);
	  rwin->headerlen += 2;
	}
	free(ret);
    }
    else
    {
	temp = malloc(strlen(header) + 1);
	if(temp)
	  strcpy(temp,header);
    }

    free(header); 
    return(temp);
}

static char* GetHead(rwin)
     READWINDOW* rwin;
{
    MessageNode* m = (*rwin->current);
    char *temp = NULL;

    if (m)
    {
      if ( m->header == NULL )
	m->header = gettext_for_display((char*)mail_fetchheader(rwin->ms->mailbox,
								m->msgno));
      temp = malloc(1+strlen(m->header));
      if(temp != NULL) {
	strcpy(temp, m->header);
	rwin->headerlen = strlen(temp);
	return(temp);
      }
    }
    rwin->headerlen = 0;
    return(NULL);
}

static char* GetFilteredHead(rwin)
     READWINDOW* rwin;
{
    char datetemp[32];
    MessageNode* m = (*rwin->current);
    char *fullHeader = NULL;
    char *temp = NULL, *temp2 = NULL;
    char *to = NULL;
    int maxsize = 0;
    int place = 0;
    
    if (m != NULL)
    {
	if(!m->header)
	    m->header =
		gettext_for_display((char*)mail_fetchheader(rwin->ms->mailbox,
						     m->msgno));
	fullHeader = m->header;
	maxsize = strlen(fullHeader);
	temp = XtMalloc(maxsize + 1);
	
	sprintf( temp, "%s %s\n", "Date:   ", mail_date(datetemp, m->elt));
	if( m->env->to )
	{
	    strcat( temp, "To:      " );
	    place = strlen(temp);
	    temp2 = address_to_text(m->env->to, maxsize-place);
	    strcpy( temp+place, temp2 );
	    XtFree(temp2);
	}
	else
	{
	    to = parse_news_newsgroups(rwin->ms->mailbox, 
				       m->msgno,
				       MAXLINELENGTH);
	    if ( to )
		strcat( temp, to );
	}
	
	strcat( temp, "\n" );
	strcat( temp, "From:    " );
	
	place = strlen(temp);
	mail_fetchfrom(temp+place, rwin->ms->mailbox, m->msgno, FROMLEN);

	strcat( temp, "\n" );
	strcat( temp, "Subject: ");
	place = strlen(temp);
	mail_fetchsubject(temp+place,  rwin->ms->mailbox, m->msgno, RSUBJECTLEN);

	temp2 = malloc(1+strlen(temp));
	strcpy(temp2,temp);
	XtFree(temp);
	rwin->headerlen = strlen(temp2);
	return( temp2 );
    }
    else 
    {
      rwin->headerlen = 0;
	return( NULL );
    }
}

static void KillMessage(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent* event;
{
    if ((*rwin->current)->elt)
      read_markDeleted( w, rwin, event );
    NextMessage( w, rwin, event );
}

void read_markDeleted(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent* event;
{
  read_setFlag( rwin, DELETED_FLAG );
}

 
void read_markUndeleted(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent* event;
{
  read_clearFlag( rwin, DELETED_FLAG );
}


void read_markSeen(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent* event;
{
  read_setFlag( rwin, SEEN_FLAG );
}

 
void read_markUnseen(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent* event;
{
  read_clearFlag( rwin, SEEN_FLAG );
}


void read_markRecent(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent* event;
{
  read_setFlag( rwin, RECENT_FLAG );
}

 
void read_markOld(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent* event;
{
  read_clearFlag( rwin, RECENT_FLAG );
}


void read_markAnswered(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent* event;
{
  read_setFlag( rwin, ANSWERED_FLAG );
}

 
void read_markUnanswered(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent* event;
{
  read_clearFlag( rwin, ANSWERED_FLAG );
}


void read_markFlagged(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent* event;
{
  read_setFlag( rwin, FLAGGED_FLAG );
}

 
void read_markUnflagged(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent* event;
{
  read_clearFlag( rwin, FLAGGED_FLAG );
}

void read_setFlag(rwin, flag)
     READWINDOW* rwin;
     char       *flag;
{
    if (flag != NULL)
    {
        /*
	 * May have been expunged not valid but still visible
         * in the read window, eg, the last message in the read window. */
        if (ReadWindow_verifyMessage(rwin))
	  doSetFlag(rwin->ms, ximap_newnodelist(rwin->current, 1), flag);
    }
}

void read_clearFlag(rwin, flag)
     READWINDOW* rwin;
     char       *flag;
{
    if (flag != NULL)
    {
        if (ReadWindow_verifyMessage(rwin))
	  doClearFlag(rwin->ms, ximap_newnodelist(rwin->current, 1), flag);
    }
}

void read_setupKeywords(rwin)
     READWINDOW* rwin;
{
    char **keywords = rwin->ms->mailbox->user_flags;
    char **temp = keywords;
    
    int  num_keys = 0;
    
    while ( num_keys < NUSERFLAGS && *temp++ != NULL )
	++num_keys;
    
    temp = keywords;
    
    if (*temp)
    {
	read_clear_menu[num_keys].name = NULL;
	read_set_menu[num_keys].name = NULL;

	while(num_keys--)
	{
	    read_clear_menu[num_keys].name = cpystr(temp[num_keys]);
	    read_clear_menu[num_keys].function = read_clearKeyword;
	    read_set_menu[num_keys].name = cpystr(temp[num_keys]);
	    read_set_menu[num_keys].function = read_setKeyword;
	    read_clear_menu[num_keys].menu = NULL;
	    read_set_menu[num_keys].menu = NULL;
	    read_clear_menu[num_keys].state = NEVER;
	    read_set_menu[num_keys].state = NEVER;
	}
    }
    else
    {
	read_clear_menu[0].name = NULL;
	read_clear_menu[0].function = NULL;
	read_set_menu[0].name = NULL;
	read_set_menu[0].function = NULL;
	read_clear_menu[0].menu = NULL;
	read_set_menu[0].menu = NULL;

	read_keywords_menu[0].state = ~NEVER;
	read_keywords_menu[1].state = ~NEVER;
    }
}

void read_setKeyword(w, rwin, call)
     Widget w;
     READWINDOW* rwin;
     void *call;
{
    char *temp = NULL;
    
    XtSetArg(warg[n], XtNlabel, &temp); n++;
    XtGetValues( w, warg, n); n = 0;
    
    if (temp != NULL)
    {
	read_setFlag(rwin, temp);
    }
}

void read_clearKeyword(w, rwin, call)
     Widget w;
     READWINDOW* rwin;
     void *call;
{
  char *temp = NULL;
  
  XtSetArg(warg[n], XtNlabel, &temp); n++;
  XtGetValues( w, warg, n); n = 0;

  if (temp != NULL)
    {
      read_clearFlag(rwin, temp);
    }
}

/***********************************************************


  Actions


***********************************************************/

static void readDownAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    DownNext(w, rwin, event);
}



static void readUpAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    Up(w, rwin, event);
}



static void readKillAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    KillMessage(w, rwin, event);
}


static void readFlagAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    if (!(*rwin->current)->elt)
      return;
    read_markFlagged(w, rwin, event);
}


static void readUnflagAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    if (!(*rwin->current)->elt)
      return;
    read_markUnflagged(w, rwin, event);
}


static void readDeleteAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    if (!(*rwin->current)->elt)
      return;
    read_markDeleted(w, rwin, event);
}


static void readUndeleteAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    if (!(*rwin->current)->elt)
      return;
    read_markUndeleted(w, rwin, event);
}


static void readCopyAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    /* Insure there is an messagecache elt. This is a symptom fix. 
     * We have entered here without an elt. Not sure why just yet.
     * BUT this will happen if the message has been expunged but
     * the readwindow is still open. */
    if (!(*rwin->current)->elt)
      return;

    CopyMessage(w, rwin, event);
}


static void readMoveAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    /* Insure there is an messagecache elt. This is a symptom fix. 
     * We have entered here without an elt. Not sure why just yet.
     * BUT this will happen if the message has been expunged but
     * the readwindow is still open. */
    if (!(*rwin->current)->elt)
      return;

    MoveMessage(w, rwin, event);
}

static void readLocalCopyAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    /* Insure there is an messagecache elt. This is a symptom fix. 
     * We have entered here without an elt. Not sure why just yet.
     * BUT this will happen if the message has been expunged but
     * the readwindow is still open. */
    if (!(*rwin->current)->elt)
      return;

    SaveToFile(w, rwin, event);
}

static void readNextAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    NextMessage(w, rwin, event);
}


static void readPrevAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    PreviousMessage(w, rwin, event);
}


static void readQuitAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    Quit(w, rwin, event);
}


static void readForwardAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;

    if (!(*rwin->current)->elt)
      return;

    ForwardMessage(w, rwin, event);
}


static void readRemailAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;

    RemailMessage(w, rwin, event);
}


static void readReplySenderAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    if (!(*rwin->current)->elt)
      return;

    ReplyToSender(w, rwin, event);
}

static void readReplyAllAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    if (!(*rwin->current)->elt)
      return;

    ReplyToAll(w, rwin, event);
}

static void readToggleHeaderAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    ToggleHeader(w, rwin, event);
}

static void readPrintAct( w, event )
     Widget w;
     XEvent *event;
{
    READWINDOW* rwin = getRwinFromWidget(w);

    if (!rwin) return;
    PrintMessage(w, rwin, event);
}

static void DisplayBody( rwin, body )
     READWINDOW  *rwin;
     BODY        *body;
{
    char temp[80];
    XbListSelection *partList = NULL;
    int type = body->type;

    static int lastType= -1;		/* initialize this dude 1-dec-92 */

    rwin->change_text(rwin, body);
    rwin->state &= ENCODEDDATA;

    if(rwin->rawText)
	type = TYPETEXT;

    switch( type )
    {
      case TYPETEXT: /* Most common case.  Just display the text */
      /*
       * Make sure the MIME parse worked, and we know the subtype.
       * Formerly, body->subtype == NULL *did* cause problems :-)
       *        Bill Yeager -- 1-dec-92 */
      if (body->subtype && !strcasecmp(body->subtype, "richtext"))
	{
	  if(getHandler(body))
	    displayMultiMedia(body);
	}
      else
	{
	  rwin->state |= NOENCODEDDATA;
	}
      break;
      case TYPEMESSAGE: /* This could be a pain... just text for now. */
	break;
      case TYPEMULTIPART: /* probably the second most common.  Display
			     a browser of the body parts of the message.  The
 			     user should be able to save any individual body part
			     to a file after decoding. */
	if(BodyStack_empty(rwin->bodystack))
	    PartPosition_clear(rwin->partposition);

	rwin->state &= HASBODYPART;

	partList = makeMultiPartHeaderList(body->contents.part);
	XbBrowserChangeItems(rwin->browser, partList);
	XtManageChild(rwin->browser);
	ResizeBrowser(rwin);
	break;
      case TYPEAPPLICATION: /* If we know the application, start it, else
			       offer to save to a file. */
      case TYPEAUDIO: /* Give window with 'play' button, some way to save to file. */
      case TYPEIMAGE: /* Give window with 'display' button, some way to save to file. */
      case TYPEVIDEO: /* Offer to save to file. */
      case TYPEOTHER: /* do absolutely nothing except offer to save to file. 
		         and offer the user a chance to view it if we have the
			 proper handler. */
	if(body->description)
	    sprintf(temp, "Body Part: Type %s\nDescription: %s", 
		    body->subtype, body->description); 
	else
	    sprintf(temp, "Body Part: Type %s", body->subtype); 
	if(getHandler(body))
	    rwin->panel_state &= HASHANDLER;
	else
	    rwin->panel_state |= NOHANDLER;

	XtSetArg(warg[n], XtNlabel, cpystr(temp)); n++;
	XtSetValues(rwin->panel_label, warg, n); n = 0;

	XtManageChild(rwin->panel_label);
	XtManageChild(rwin->panel_buttons);
	checkButtonStates(rwin->panel_states, rwin->panel_state);
	/* XtUnmanageChild(rwin->dform); */
	break;
      default:
	break;
    }
    
    if( lastType != body->type )
    {
	switch (lastType)
	{
	  case TYPETEXT:
	  case TYPEMESSAGE:
	    /* XtUnmanageChild(rwin->text); */
	    break;
	  case TYPEMULTIPART:
	    XtUnmanageChild(rwin->browser);
	    break;
	  default:
	    if( body->type == TYPETEXT
	       || body->type == TYPEMESSAGE 
	       || body->type == TYPEMULTIPART )
	    {
		XtUnmanageChild(rwin->panel_label);
		XtUnmanageChild(rwin->panel_buttons);
	    }
	    break;
	}
    }
    
    lastType = body->type;

    sprintf( rwin->name, "Message %4d: part %s", 
	    (*rwin->current)->msgno, PartPosition_retrieve(rwin->partposition));

    XtSetArg(warg[n], XtNtitle, rwin->name); n++;
    XtSetValues(rwin->shell, warg, n); n = 0;
    
    checkButtonStates(rwin->states, rwin->state);
}

/* body may be NULL for this function call... */
static void DisplayText( rwin, body )
     READWINDOW  *rwin;
     BODY        *body;
{
    rwin->change_text(rwin, body);

    sprintf( rwin->name, "Message %4d: part %s", 
	    (*rwin->current)->msgno, PartPosition_retrieve(rwin->partposition));

    XtSetArg(warg[n], XtNtitle, rwin->name); n++;
    XtSetValues(rwin->shell, warg, n); n = 0;
    
    checkButtonStates(rwin->states, rwin->state);
}

static void SaveToFile(w, rwin, event)
     Widget w;
     READWINDOW *rwin;
     XEvent *event;
{
    char *filename = NULL;
    char temp[TMPLEN];
    FILE *ofp = NULL;
    BODY *body = NULL;
    struct stat sb;

    if (!(*rwin->current)->elt)
      return;

    filename = getFilename(rwin->shell, getHomeDir(), 1, 0);
    if(!filename)
      return;

    /* Only overwrite existing files with permission */
    if (stat(filename, &sb) == 0) {
      if (XbConfirm(rwin->shell,
		    "Overwrite an existing file?",
		    TRUE) == FALSE) {
	free(filename);
	return;
      }
    }

    if(!(rwin->state & NOMIME))
      {
	if(rwin->currpart)
	  body = &rwin->currpart->body;
	else
	  body = (*rwin->current)->body;
      }
    
    if((ofp = fopen(filename, "w+")) != NULL)
      {
	void *rawdata = NULL;
	unsigned long leng;
	
	if(body)
	  {
	    if(body->encoding != ENC7BIT)
	      {
		rawdata = rfc822_base64(body->contents.text, body->size.bytes, &leng);
	      }
	    else
	      {
		rawdata = body->contents.text;
		fixnl(rawdata);		/* CRLF follie-- B Yeager: 3 dec 92 */
		leng = strlen(rawdata);
	      }
	  }
	else
	  {
	    rawdata = mail_fetchtext(rwin->ms->mailbox,
				     (*rwin->current)->msgno);
	    fixnl(rawdata);		/* CRLF follie-- B Yeager: 3 dec 92 */
	    leng = strlen(rawdata);
	  }
	
	if(rawdata && leng)
	  {
	    size_t nbytes = fwrite(rawdata, 1, leng, ofp);
	    if (nbytes != leng) {
	      sprintf(temp, "Error writing %s", filename);
	      XbWarning(rwin->shell, temp);
	    }
	    free(rawdata);
	  }
	
	fclose(ofp);
      }
    else
      {
	sprintf(temp, "Cannot open file: %s", filename);
	XbWarning(rwin->shell, temp);
      }
    free(filename);			/* Bill Yeager -- 27 Oct 92 */
}

static void ViewPart(w, rwin, event)
     Widget w;
     READWINDOW *rwin;
     XEvent *event;
{
    long size;
    BODY *body = NULL;

    if(rwin->currpart)
	body = &rwin->currpart->body;
    else
	body = (*rwin->current)->body;

    if( !body->contents.text )
	mail_fetchbody(rwin->ms->mailbox,
		       (*rwin->current)->msgno,
		       PartPosition_retrieve(rwin->partposition),
		       &size);
    
    displayMultiMedia(body);
}

static void ShowRaw(w, rwin, event)
     Widget w;
     READWINDOW *rwin;
     XEvent *event;
{
    char* label;
    BODY *body = NULL;

    /* Insure there is an messagecache elt. This is a symptom fix. 
     * We have entered here without an elt. Not sure why just yet.
     * BUT this will happen if the message has been expunged but
     * the readwindow is still open. */
    if (!(*rwin->current)->elt)
      return;

    if(rwin->currpart)
	body = &rwin->currpart->body;
    else
	body = (*rwin->current)->body;

    if(rwin->rawText == TRUE)
	rwin->rawText = FALSE;
    else
	rwin->rawText = TRUE;


    rwin->display(rwin, body);

    if(rwin->rawText)
	label = cpystr("Show as MIME");
    else
	label = cpystr("ShowRaw");

    XtSetArg(warg[n], XtNlabel, label); n++;
    XtSetValues(w, warg, n); n = 0; 
}

#ifdef JUMPREADY

static void JumpTo(w, rwin, event)
     Widget w;
     READWINDOW *rwin;
     XEvent *event;
{

    if (!(*rwin->current)->elt)
      return;


}

static void JumpToPart(w, rwin, event)
     Widget w;
     READWINDOW *rwin;
     XEvent *event;
{

    if (!(*rwin->current)->elt)
      return;


}

#endif /* JUMPREADY */

static void Up(w, rwin, event)
     Widget w;
     READWINDOW *rwin;
     XEvent *event;
{
    PART *part;
    BODY *body;

    if (!(*rwin->current)->elt)
      return;

    if(body = BodyStack_top(rwin->bodystack))
    {
	BodyStack_pop(rwin->bodystack);
	PartPosition_up(rwin->partposition);
	rwin->currpart = NULL;

	/* If there's still something in the stack, we need to know
	   which part of it we're in right now. */
	if(!BodyStack_empty(rwin->bodystack))
	{
	    part = (BodyStack_top(rwin->bodystack))->contents.part;
	    while( part && &part->body != body )
		part = part->next;
	    rwin->currpart = part;
	}
	else
	{
	    rwin->state |= TOPLEVEL;
	}

	rwin->display(rwin,body);
    }
}

static void DownNext(w, rwin, event)
     Widget w;
     READWINDOW *rwin;
     XEvent *event;
{
    BODY * body = NULL;

    if (!(*rwin->current)->elt)
      return;

    if(!(rwin->state & NOMIME))
    {
	if(rwin->currpart) 
	    body = &rwin->currpart->body;
	else
	    body = (*rwin->current)->body;
	
	if(body->type == TYPEMULTIPART)
	{
	    XbListSelection* sel = ximap_newlistmap(1);
	    
	    sel->entries[0].index = 1;
	    Down(w, rwin, sel);
	    return;
	}
    }

    NextMessage(w,rwin,event);
}

static void Down( w, rwin, map)
     Widget              w;
     READWINDOW         *rwin;
     XbListSelectionPtr  map;
{
    PART *temp = NULL;
    int i = 0, partno = 0;

    if(map && map->size)
    {
	partno = map->entries->index;

	PartPosition_add(rwin->partposition, partno);
	if(BodyStack_empty(rwin->bodystack))
	    BodyStack_push((*rwin->current)->body, rwin->bodystack);
	else
	    BodyStack_push(&rwin->currpart->body, rwin->bodystack);

	temp = (BodyStack_top(rwin->bodystack))->contents.part;

	while(++i < partno)
	    temp = temp->next;
	
	rwin->currpart = temp;
	rwin->state &= INMULTI;
	rwin->rawText = FALSE;
	rwin->display(rwin, &temp->body);
    }
}

void doRead(ms, nodelist)
     MailBox*   ms;
     NodeList*  nodelist;
{ 
    struct read_window* tmp;
    int i = 0;

    fetch_unfetched(ms, nodelist);
    
    while ( i < nodelist->length )
	nodelist->nodes[i++]->locked++;

    tmp = createReadWindow( ms->mailWindow, nodelist, ms);
    if(tmp)
    {
	tmp->next = ms->rwinList;
	ms->rwinList = tmp;
    }
}

extern MAILSESSION id;
READWINDOW* getRwinFromWidget(w)
     Widget w;
{
    register STREAMARRAY *temp = id.mailboxes;
    READWINDOW* rwin = NULL;
    int number = 0;

    Widget sh = getShell( w );

    while (number++ < id.no_boxes        
	   && temp)
    {
	rwin = temp->status->rwinList;
	while(rwin)
	{
	    if( rwin->shell == sh )
		return( rwin );

	    rwin = rwin->next;
	}
	temp = temp->next;
    }
    return ( NULL );
}


#ifdef JUMPREADY

void read_setUpJumpToMessage(rwin)
     READWINDOW* rwin;
{
    if (!(*rwin->current)->elt)
      return;

}

void read_setUpJumpToPart(rwin)
     READWINDOW* rwin;
{

    if (!(*rwin->current)->elt)
      return;

}

#endif /* JUMPREADY */

void nomimeChangeTextPortion(rwin,dummy)
     READWINDOW *rwin;
     void       *dummy;
{
    Widget newsource;
    Widget oldsource;

    char *text = rwin->current_text(rwin, dummy);
    XtSetArg(warg[n], XtNstring, text); n++;
    
    newsource = XtCreateWidget(text, asciiSrcObjectClass,
			       rwin->text, warg, n); n = 0;
	
    oldsource = XawTextGetSource(rwin->text);
    XawTextSetSource(rwin->text, newsource, 0);
    XtDestroyWidget(oldsource);

    ResizeText(rwin, 0);
    XtFree(text);
}

void ChangeTextPortion(rwin, body)
     READWINDOW *rwin;
     BODY *body;
{
    Widget newsource;
    Widget oldsource;

    char *text = rwin->current_text(rwin, body);
    XtSetArg(warg[n], XtNstring, text); n++;
    
    newsource = XtCreateWidget(text, asciiSrcObjectClass,
			       rwin->text, warg, n); n = 0;
	
    oldsource = XawTextGetSource(rwin->text);
    XawTextSetSource(rwin->text, newsource, 0);
    XtDestroyWidget(oldsource);

    if(   body->type == TYPETEXT 
       || body->type == TYPEMESSAGE 
       || rwin->rawText)
	ResizeText(rwin, 0);
    else
	ResizeText(rwin, 1);

    XtFree(text);
}

static void ToggleHeader(w, rwin, event)
     Widget w;
     READWINDOW* rwin;
     XEvent *event;
{
    BODY *body;

    /* Insure there is an messagecache elt. This is a symptom fix. 
     * We have entered here without an elt. Not sure why just yet.
     * BUT this will happen if the message has been expunged but
     * the readwindow is still open. */
    if (!(*rwin->current)->elt)
      return;

    /* Insure there is an messagecache elt. This is a symptom fix. 
     * We have entered here without an elt. Not sure why just yet.
     * BUT this will happen if the message has been expunged but
     * the readwindow is still open. */
    if (!(*rwin->current)->elt)
      return;

    if ( rwin->filterHead == TRUE )
	rwin->filterHead = FALSE;
    else
	rwin->filterHead = TRUE;
    
    if(rwin->currpart)
	body = &rwin->currpart->body;
    else
	body = (*rwin->current)->body;

    rwin->change_text(rwin, body);
}

