#include "tkgs.h"
#include "tkgsInt.h"

/* FIXME: include our own? We need this for strcasecmp */
#include <tkPort.h>

/*
 * Procedures used only in this file.
 */

static TkGSSubFontId	CanUseFallbackWithAliases _ANSI_ARGS_((TkGS_Drawable d,
			    TkGS_Font font, TkGSMultiFont multiFont, 
			    char *faceName, Tcl_UniChar ch, 
			    Tcl_DString *nameTriedPtr));
static int		SeenName _ANSI_ARGS_((CONST char *name,
			    Tcl_DString *dsPtr));


extern Tcl_Encoding unicodeEncoding;


/*
 * Hardcoded font aliases.  These are used to describe (mostly) identical
 * fonts whose names differ from platform to platform.  If the
 * user-supplied font name matches any of the names in one of the alias
 * lists, the other names in the alias list are also automatically tried.
 */

static char *timesAliases[] = {
    "Times",			/* Unix. */
    "Times New Roman",		/* Windows. */
    "New York",			/* Mac. */
    NULL
};

static char *helveticaAliases[] = {
    "Helvetica",		/* Unix. */
    "Arial",			/* Windows. */
    "Geneva",			/* Mac. */
    NULL
};

static char *courierAliases[] = {
    "Courier",			/* Unix and Mac. */
    "Courier New",		/* Windows. */
    NULL
};

static char *minchoAliases[] = {
    "mincho",			/* Unix. */
    "\357\274\255\357\274\263 \346\230\216\346\234\235",
				/* Windows (MS mincho). */
    "\346\234\254\346\230\216\346\234\235\342\210\222\357\274\255",
				/* Mac (honmincho-M). */
    NULL
};

static char *gothicAliases[] = {
    "gothic",			/* Unix. */
    "\357\274\255\357\274\263 \343\202\264\343\202\267\343\203\203\343\202\257",
				/* Windows (MS goshikku). */
    "\344\270\270\343\202\264\343\202\267\343\203\203\343\202\257\342\210\222\357\274\255",
				/* Mac (goshikku-M). */
    NULL    
};

static char *dingbatsAliases[] = {
    "dingbats", "zapfdingbats", "itc zapfdingbats",
				/* Unix. */
				/* Windows. */
    "zapf dingbats",		/* Mac. */
    NULL
};

static char **fontAliases[] = {
    timesAliases,
    helveticaAliases,
    courierAliases,
    minchoAliases,
    gothicAliases,
    dingbatsAliases,
    NULL
};  

/*
 * Hardcoded font classes.  If the character cannot be found in the base 
 * font, the classes are examined in order to see if some other similar 
 * font should be examined also.  
 */

static char *systemClass[] = {
    "fixed",				/* Unix. */
					/* Windows. */
    "chicago", "osaka", "sistemny",	/* Mac. */
    NULL
};

static char *serifClass[] = {
    "times", "palatino", "mincho",	/* All platforms. */
    "song ti",				/* Unix. */
    "ms serif", "simplified arabic", 	/* Windows. */
    "latinski",				/* Mac. */
    NULL
};

static char *sansClass[] = {
    "helvetica", "gothic",		/* All platforms. */
					/* Unix. */
    "ms sans serif", "traditional arabic",
					/* Windows. */
    "bastion",				/* Mac. */
    NULL
};

static char *monoClass[] = {
    "courier", "gothic",		/* All platforms. */
    "fangsong ti",			/* Unix. */
    "simplified arabic fixed",		/* Windows. */
    "monaco", "pryamoy",		/* Mac. */
    NULL
};

static char *symbolClass[] = {
    "symbol", "dingbats", "wingdings", NULL
};

static char **fontFallbacks[] = {
    systemClass,
    serifClass,
    sansClass,
    monoClass,
    symbolClass,
    NULL
};

/*
 * Global fallbacks.  If the character could not be found in the preferred
 * fallback list, this list is examined.  If the character still cannot be
 * found, all font families in the system are examined. 
 */

static char *globalFontClass[] = {
    "symbol",			/* All platforms. */
    "newspaper",		/* Unix. *//* TODO: optimize loading of big fonts anyway. */
    "lucida sans unicode",	/* Windows. */
    "bitstream cyberbit",	/* Windows popular CJK font */
    "chicago",			/* Mac. */
    NULL
};




/*
 * FontFaces Obj type
 */

static TkGS_FreeBaseProc FreeFontFaces;
static TkGS_DupBaseProc DupFontFaces;

TkGS_BaseType TkGSFontFacesBaseType = {
    "FontFaces",
    sizeof(TkGS_FontFaces_),
    3,				/* nbIntRep */

    FreeFontFaces,		/* freeBaseProc */
    DupFontFaces		/* dupBaseProc */
};

TkGS_BaseType *
TkGS_GetFontFacesBaseType()
{
    return &TkGSFontFacesBaseType;
}



static void
FreeFontFaces(objPtr)
    TkGS_Obj *objPtr;
{
    ckfree((char *) TkGSFontFaces_Names(objPtr));
    ckfree(TkGSFontFaces_Buffer(objPtr));
}

static void
DupFontFaces(srcPtr, dupPtr)
    TkGS_Obj *srcPtr;
    TkGS_Obj *dupPtr;
{
    register int nbNames = TkGSFontFaces_NbNames(srcPtr);
    register char **names = TkGSFontFaces_Names(srcPtr);

    if (nbNames) {
	register char *buf, *p, **nms;
	int length;
	int i;

	for (i=0, length=0; i < nbNames; i++) {
	    length += strlen(names[i])+1;
	}
    
	/* Allocate families array & buffer */
	nms = (char **) ckalloc(sizeof(char *) * nbNames);
	buf = ckalloc(length);

	for (i=0, p=buf; i < nbNames; i++, p += length) {
	    length = strlen(names[i])+1;
	    strcpy(p, names[i]);
	    nms[i] = p;
	}

	TkGSFontFaces_NbNames(dupPtr) = nbNames;
	TkGSFontFaces_Names(dupPtr)   = nms;
	TkGSFontFaces_Buffer(dupPtr)  = buf;

    } else {
	/* No family, use the default font family */

	/* TODO? */
	TkGSFontFaces_NbNames(dupPtr) = 0;
	TkGSFontFaces_Names(dupPtr)   = NULL;
	TkGSFontFaces_Buffer(dupPtr)  = NULL;
    }
}




TkGS_FontFaces_ *
TkGSNewFontFace(name)
    char *name;
{
    if (name) {
	return TkGSNewFontFaces(1, &name);
    } else {
	/* NULL family means use the default font family */
	return TkGSNewFontFaces(0, NULL);
    }
}

TkGS_FontFaces_ *
TkGSNewFontFaces(nbNames, names)
    int  nbNames;
    char **names;
{
    register TkGS_FontFaces_ *ffPtr = (TkGS_FontFaces_*) TkGSNewObj(&TkGSFontFacesBaseType);

    if (nbNames) {
	register char *buf, *p, **nms;
	int length;
	int i;

	for (i=0, length=0; i < nbNames; i++) {
	    length += strlen(names[i])+1;
	}
    
	/* Allocate names array & buffer */
	nms = (char **) ckalloc(sizeof(char *) * nbNames);
	buf = ckalloc(length);

	for (i=0, p=buf; i < nbNames; i++, p += length) {
	    length = strlen(names[i])+1;
	    strcpy(p, names[i]);
	    nms[i] = p;
	}

	TkGSFontFaces_NbNames(ffPtr) = nbNames;
	TkGSFontFaces_Names(ffPtr)   = nms;
	TkGSFontFaces_Buffer(ffPtr)  = buf;

    } else {
	/* No family, use the default font family */

	/* TODO? */
	TkGSFontFaces_NbNames(ffPtr) = 0;
	TkGSFontFaces_Names(ffPtr)   = NULL;
	TkGSFontFaces_Buffer(ffPtr)  = NULL;
    }
    return ffPtr;
}


/*
 * Font Obj type
 */

static TkGS_DupBaseProc DupFont;
static TkGS_FreeBaseProc FreeFont;

TkGS_BaseType TkGSFontBaseType = {
    "Font",
    sizeof(TkGS_Font_),
    3,				/* nbIntRep */

    FreeFont,			/* freeBaseProc */
    DupFont			/* dupBaseProc */
};

TkGS_BaseType *
TkGS_GetFontBaseType()
{
    return &TkGSFontBaseType;
}


static void
FreeFont(objPtr)
    TkGS_Obj *objPtr;
{
    register TkGS_FontAttributes fa = TkGSFont_Attributes(objPtr);

    TkGS_DecrRefCount((TkGS_Obj *) fa.faces);
}

static void
DupFont(srcPtr, dupPtr)
    TkGS_Obj *srcPtr;
    TkGS_Obj *dupPtr;
{
    register TkGS_FontAttributes fa = TkGSFont_Attributes(srcPtr);

    TkGS_IncrRefCount((TkGS_Obj *) fa.faces);
    TkGSFont_Attributes(dupPtr) = fa;
}




TkGS_Font_ *
TkGSNewFont(faPtr)
    TkGS_FontAttributes *faPtr;
{
    register TkGS_Font_ *fontPtr = (TkGS_Font_*) TkGSNewObj(&TkGSFontBaseType);

    TkGS_IncrRefCount((TkGS_Obj *) faPtr->faces);
    TkGSFont_Attributes(fontPtr) = *faPtr;

    return fontPtr;
}



/*
 * Fonts and text primitives
 */

/* Unicode version */

int
TkGS_MeasureCharsUni(d, string, length, maxPixels, flags, lengthPtr)
    TkGS_Drawable     d;
    CONST Tcl_UniChar *string;
    int               length;
    int               maxPixels;
    int               flags;
    int               *lengthPtr;
{
    int result;

    TkGSUpdateDrawableState(d, (TkGS_GCFont));

    if (TKGS_GETDEVICEMEMBER(d, measureCharsUni)) {
	/* Call Unicode version */
	result = TKGS_GETDEVICEMEMBER(d, measureCharsUni)(d, string, length, 
		    maxPixels, flags, lengthPtr);

    } else if (TKGS_GETDEVICEMEMBER(d, measureCharsUtf)) {
	/* Convert string to UTF-8 */
	Tcl_DString ds;
	char *stringUtf;
	int lengthUtf;

	Tcl_DStringInit(&ds);
	stringUtf = Tcl_UniCharToUtfDString(string, length, &ds);
	lengthUtf = Tcl_DStringLength(&ds);

	/* Call UTF-8 version */
	result = TKGS_GETDEVICEMEMBER(d, measureCharsUtf)(d, stringUtf, 
		    lengthUtf, maxPixels, flags, lengthPtr);

	/* Free converted string */
	Tcl_DStringFree(&ds);

    } else if (TKGS_GETDEVICEMEMBER(d, measureCharsSys)) {
	/* Convert string to system encoding */
	Tcl_DString dsUtf, dsSys;
	char *stringUtf, *stringSys;
	int lengthUtf, lengthSys;

	Tcl_DStringInit(&dsUtf);
	Tcl_DStringInit(&dsSys);

	stringUtf = Tcl_UniCharToUtfDString(string, length, &dsUtf);
	lengthUtf = Tcl_DStringLength(&dsUtf);
	stringSys = Tcl_UtfToExternalDString(NULL /* system encoding */, 
			stringUtf, lengthUtf, &dsSys);
	lengthSys = Tcl_DStringLength(&dsSys);

	/* Call system-specific version */
	result = TKGS_GETDEVICEMEMBER(d, measureCharsSys)(d, stringSys, 
		    lengthSys, maxPixels, flags, lengthPtr);

	/* Free converted strings */
	Tcl_DStringFree(&dsUtf);
	Tcl_DStringFree(&dsSys);

    } else {
	/* Function not available */
	*lengthPtr = 0;
	result = 0;
    }
    return result;
}

int
TkGS_TextWidthUni(d, string, length)
    TkGS_Drawable     d;
    CONST Tcl_UniChar *string;
    int               length;
{
    int result;

    TkGSUpdateDrawableState(d, (TkGS_GCFont));

    if (TKGS_GETDEVICEMEMBER(d, textWidthUni)) {
	/* Call Unicode version */
	result = TKGS_GETDEVICEMEMBER(d, textWidthUni)(d, string, length);

    } else if (TKGS_GETDEVICEMEMBER(d, textWidthUtf)) {
	/* Convert string to UTF-8 */
	Tcl_DString ds;
	char *stringUtf;
	int lengthUtf;

	Tcl_DStringInit(&ds);
	stringUtf = Tcl_UniCharToUtfDString(string, length, &ds);
	lengthUtf = Tcl_DStringLength(&ds);

	/* Call UTF-8 version */
	result = TKGS_GETDEVICEMEMBER(d, textWidthUtf)(d, stringUtf, 
		    lengthUtf);

	/* Free converted string */
	Tcl_DStringFree(&ds);

    } else if (TKGS_GETDEVICEMEMBER(d, textWidthSys)) {
	/* Convert string to system encoding */
	Tcl_DString dsUtf, dsSys;
	char *stringUtf, *stringSys;
	int lengthUtf, lengthSys;

	Tcl_DStringInit(&dsUtf);
	Tcl_DStringInit(&dsSys);

	stringUtf = Tcl_UniCharToUtfDString(string, length, &dsUtf);
	lengthUtf = Tcl_DStringLength(&dsUtf);
	stringSys = Tcl_UtfToExternalDString(NULL /* system encoding */, 
			stringUtf, lengthUtf, &dsSys);
	lengthSys = Tcl_DStringLength(&dsSys);

	/* Call system-specific version */
	result = TKGS_GETDEVICEMEMBER(d, textWidthUtf)(d, stringSys, 
		    lengthSys);

	/* Free converted strings */
	Tcl_DStringFree(&dsUtf);
	Tcl_DStringFree(&dsSys);

    } else {
	/* Function not available, try with TkGS_MeasureChars */
	TkGS_MeasureCharsUni(d, string, length, -1, 0, &result);
    }
    return result;
}

void
TkGS_DrawCharsUni(d, string, length, x, y, widthPtr)
    TkGS_Drawable     d;
    CONST Tcl_UniChar *string;
    int               length;
    int               x;
    int               y;
    int               *widthPtr;
{
    TkGSUpdateDrawableState(d, (TkGS_GCForeground|TkGS_GCBackground|TkGS_GCFont));

    if (TKGS_GETDEVICEMEMBER(d, drawCharsUni)) {
	/* Call Unicode version */
	TKGS_GETDEVICEMEMBER(d, drawCharsUni)(d, string, length, x, y, 
	    widthPtr);

    } else if (TKGS_GETDEVICEMEMBER(d, drawCharsUtf)) {
	/* Convert string to UTF-8 */
	Tcl_DString ds;
	char *stringUtf;
	int lengthUtf;

	Tcl_DStringInit(&ds);
	stringUtf = Tcl_UniCharToUtfDString(string, length, &ds);
	lengthUtf = Tcl_DStringLength(&ds);

	/* Call UTF-8 version */
	TKGS_GETDEVICEMEMBER(d, drawCharsUtf)(d, stringUtf, lengthUtf, x, y, 
	    widthPtr);

	/* Free converted string */
	Tcl_DStringFree(&ds);

    } else if (TKGS_GETDEVICEMEMBER(d, drawCharsSys)) {
	/* Convert string to system encoding */
	Tcl_DString dsUtf, dsSys;
	char *stringUtf, *stringSys;
	int lengthUtf, lengthSys;

	Tcl_DStringInit(&dsUtf);
	Tcl_DStringInit(&dsSys);

	stringUtf = Tcl_UniCharToUtfDString(string, length, &dsUtf);
	lengthUtf = Tcl_DStringLength(&dsUtf);
	stringSys = Tcl_UtfToExternalDString(NULL /* system encoding */, 
			stringUtf, lengthUtf, &dsSys);
	lengthSys = Tcl_DStringLength(&dsSys);

	/* Call system-specific version */
	TKGS_GETDEVICEMEMBER(d, drawCharsSys)(d, stringSys, lengthSys, x, y, 
	    widthPtr);

	/* Free converted strings */
	Tcl_DStringFree(&dsUtf);
	Tcl_DStringFree(&dsSys);
    }
}

void 
TkGS_UnderlineCharsUni(d, string, length, x, y, first, last)
    TkGS_Drawable     d;
    CONST Tcl_UniChar *string;
    int               length;
    int               x;
    int               y;
    int               first;
    int               last;
{
    TkGSUpdateDrawableState(d, (TkGS_GCForeground|TkGS_GCFont));

    if (TKGS_GETDEVICEMEMBER(d, underlineCharsUni)) {
	/* Call Unicode version */
	TKGS_GETDEVICEMEMBER(d, underlineCharsUni)(d, string, length, 
	    x, y, first, last);

    } else if (TKGS_GETDEVICEMEMBER(d, underlineCharsUtf)) {
	/* Convert string to UTF-8 */
	Tcl_DString ds;
	char *stringUtf;
	int lengthUtf;

	Tcl_DStringInit(&ds);
	stringUtf = Tcl_UniCharToUtfDString(string, length, &ds);
	lengthUtf = Tcl_DStringLength(&ds);

	/* Call UTF-8 version */
	TKGS_GETDEVICEMEMBER(d, underlineCharsUtf)(d, stringUtf, lengthUtf, 
	    x, y, first, last);

	/* Free converted string */
	Tcl_DStringFree(&ds);

    } else if (TKGS_GETDEVICEMEMBER(d, underlineCharsSys)) {
	/* Convert string to system encoding */
	Tcl_DString dsUtf, dsSys;
	char *stringUtf, *stringSys;
	int lengthUtf, lengthSys;

	Tcl_DStringInit(&dsUtf);
	Tcl_DStringInit(&dsSys);

	stringUtf = Tcl_UniCharToUtfDString(string, length, &dsUtf);
	lengthUtf = Tcl_DStringLength(&dsUtf);
	stringSys = Tcl_UtfToExternalDString(NULL /* system encoding */, 
			stringUtf, lengthUtf, &dsSys);
	lengthSys = Tcl_DStringLength(&dsSys);

	/* Call system-specific version */
	TKGS_GETDEVICEMEMBER(d, underlineCharsSys)(d, stringSys, lengthSys, 
	    x, y, first, last);

	/* Free converted strings */
	Tcl_DStringFree(&dsUtf);
	Tcl_DStringFree(&dsSys);
    }
}


/* Utf-8 version */

int 
TkGS_MeasureCharsUtf(d, string, length, maxPixels, flags, lengthPtr)
    TkGS_Drawable     d;
    CONST char        *string;
    int               length;
    int               maxPixels;
    int               flags;
    int               *lengthPtr;
{
    int result;

    TkGSUpdateDrawableState(d, (TkGS_GCFont));

    if (TKGS_GETDEVICEMEMBER(d, measureCharsUtf)) {
	/* Call UTF-8 version */
	result = TKGS_GETDEVICEMEMBER(d, measureCharsUtf)(d, string, length, 
		    maxPixels, flags, lengthPtr);

    } else if (TKGS_GETDEVICEMEMBER(d, measureCharsUtf)) {
	/* Convert string to Unicode */
	Tcl_DString ds;
	Tcl_UniChar *stringUni;
	int lengthUni;

	Tcl_DStringInit(&ds);
	stringUni = Tcl_UtfToUniCharDString(string, length, &ds);
	lengthUni = Tcl_DStringLength(&ds)/sizeof(Tcl_UniChar);

	/* Call Unicode version */
	result = TKGS_GETDEVICEMEMBER(d, measureCharsUni)(d, stringUni, 
		    lengthUni, maxPixels, flags, lengthPtr);

	/* Free converted string */
	Tcl_DStringFree(&ds);

    } else if (TKGS_GETDEVICEMEMBER(d, measureCharsSys)) {
	/* Convert string to system encoding */
	Tcl_DString ds;
	char *stringSys;
	int lengthSys;

	Tcl_DStringInit(&ds);
	stringSys = Tcl_UtfToExternalDString(NULL /* system encoding */, 
			string, length, &ds);
	lengthSys = Tcl_DStringLength(&ds);

	/* Call system-specific version */
	result = TKGS_GETDEVICEMEMBER(d, measureCharsSys)(d, stringSys, 
		    lengthSys, maxPixels, flags, lengthPtr);

	/* Free converted string */
	Tcl_DStringFree(&ds);

    } else {
	/* Function not available */
	*lengthPtr = 0;
	result = 0;
    }
    return result;
}

int 
TkGS_TextWidthUtf(d, string, length)
    TkGS_Drawable     d;
    CONST char        *string;
    int               length;
{
    int result;

    TkGSUpdateDrawableState(d, (TkGS_GCFont));

    if (TKGS_GETDEVICEMEMBER(d, textWidthUtf)) {
	/* Call UTF-8 version */
	result = TKGS_GETDEVICEMEMBER(d, textWidthUtf)(d, string, length);

    } else if (TKGS_GETDEVICEMEMBER(d, textWidthUtf)) {
	/* Convert string to Unicode */
	Tcl_DString ds;
	Tcl_UniChar *stringUni;
	int lengthUni;

	Tcl_DStringInit(&ds);
	stringUni = Tcl_UtfToUniCharDString(string, length, &ds);
	lengthUni = Tcl_DStringLength(&ds)/sizeof(Tcl_UniChar);

	/* Call Unicode version */
	result = TKGS_GETDEVICEMEMBER(d, textWidthUni)(d, stringUni, 
		    lengthUni);

	/* Free converted string */
	Tcl_DStringFree(&ds);

    } else if (TKGS_GETDEVICEMEMBER(d, textWidthSys)) {
	/* Convert string to system encoding */
	Tcl_DString ds;
	char *stringSys;
	int lengthSys;

	Tcl_DStringInit(&ds);
	stringSys = Tcl_UtfToExternalDString(NULL /* system encoding */, 
			string, length, &ds);
	lengthSys = Tcl_DStringLength(&ds);

	/* Call system-specific version */
	result = TKGS_GETDEVICEMEMBER(d, textWidthSys)(d, stringSys, 
		    lengthSys);

	/* Free converted string */
	Tcl_DStringFree(&ds);

    } else {
	/* Function not available, try with TkGS_MeasureChars */
	TkGS_MeasureCharsUtf(d, string, length, -1, 0, &result);
    }
    return result;
}

void 
TkGS_DrawCharsUtf(d, string, length, x, y, widthPtr)
    TkGS_Drawable     d;
    CONST char        *string;
    int               length;
    int               x;
    int               y;
    int               *widthPtr;
{
    TkGSUpdateDrawableState(d, (TkGS_GCForeground|TkGS_GCBackground|TkGS_GCFont));

    if (TKGS_GETDEVICEMEMBER(d, drawCharsUtf)) {
	/* Call UTF-8 version */
	TKGS_GETDEVICEMEMBER(d, drawCharsUtf)(d, string, length, x, y, 
	    widthPtr);

    } else if (TKGS_GETDEVICEMEMBER(d, drawCharsUni)) {
	/* Convert string to Unicode */
	Tcl_DString ds;
	Tcl_UniChar *stringUni;
	int lengthUni;

	Tcl_DStringInit(&ds);
	stringUni = Tcl_UtfToUniCharDString(string, length, &ds);
	lengthUni = Tcl_DStringLength(&ds)/sizeof(Tcl_UniChar);

	/* Call Unicode version */
	TKGS_GETDEVICEMEMBER(d, drawCharsUni)(d, stringUni, lengthUni, x, y, 
	    widthPtr);

	/* Free converted string */
	Tcl_DStringFree(&ds);

    } else if (TKGS_GETDEVICEMEMBER(d, drawCharsSys)) {
	/* Convert string to system encoding */
	Tcl_DString ds;
	char *stringSys;
	int lengthSys;

	Tcl_DStringInit(&ds);
	stringSys = Tcl_UtfToExternalDString(NULL /* system encoding */, 
			string, length, &ds);
	lengthSys = Tcl_DStringLength(&ds);

	/* Call system-specific version */
	TKGS_GETDEVICEMEMBER(d, drawCharsSys)(d, stringSys, lengthSys, x, y, 
	    widthPtr);

	/* Free converted string */
	Tcl_DStringFree(&ds);
    }
}

void 
TkGS_UnderlineCharsUtf(d, string, length, x, y, first, last)
    TkGS_Drawable     d;
    CONST char        *string;
    int               length;
    int               x;
    int               y;
    int               first;
    int               last;
{
    TkGSUpdateDrawableState(d, (TkGS_GCForeground|TkGS_GCFont));

    if (TKGS_GETDEVICEMEMBER(d, underlineCharsUtf)) {
	/* Call UTF-8 version */
	TKGS_GETDEVICEMEMBER(d, underlineCharsUtf)(d, string, length, x, y, 
	    first, last);

    } else if (TKGS_GETDEVICEMEMBER(d, underlineCharsUni)) {
	/* Convert string to Unicode */
	Tcl_DString ds;
	Tcl_UniChar *stringUni;
	int lengthUni;

	Tcl_DStringInit(&ds);
	stringUni = Tcl_UtfToUniCharDString(string, length, &ds);
	lengthUni = Tcl_DStringLength(&ds)/sizeof(Tcl_UniChar);

	/* Call Unicode version */
	TKGS_GETDEVICEMEMBER(d, underlineCharsUni)(d, stringUni, lengthUni, 
	    x, y, first, last);

    } else if (TKGS_GETDEVICEMEMBER(d, underlineCharsSys)) {
	/* Convert string to system encoding */
	Tcl_DString ds;
	char *stringSys;
	int lengthSys;

	Tcl_DStringInit(&ds);
	stringSys = Tcl_UtfToExternalDString(NULL /* system encoding */, 
			string, length, &ds);
	lengthSys = Tcl_DStringLength(&ds);

	/* Call system-specific version */
	TKGS_GETDEVICEMEMBER(d, underlineCharsSys)(d, stringSys, lengthSys, 
	    x, y, first, last);

	/* Free converted string */
	Tcl_DStringFree(&ds);
    }
}



/* System-specific version */

int 
TkGS_MeasureCharsSys(d, string, length, maxPixels, flags, lengthPtr)
    TkGS_Drawable     d;
    CONST char        *string;
    int               length;
    int               maxPixels;
    int               flags;
    int               *lengthPtr;
{
    int result;

    TkGSUpdateDrawableState(d, (TkGS_GCFont));

    if (TKGS_GETDEVICEMEMBER(d, measureCharsSys)) {
	/* Call system-specific version */
	result = TKGS_GETDEVICEMEMBER(d, measureCharsSys)(d, string, length, 
		    maxPixels, flags, lengthPtr);

    } else if (TKGS_GETDEVICEMEMBER(d, measureCharsUtf)) {
	/* Convert string to UTF-8 */
	Tcl_DString ds;
	char *stringUtf;
	int lengthUtf;

	Tcl_DStringInit(&ds);
	stringUtf = Tcl_ExternalToUtfDString(NULL /* system encoding */, 
			string, length, &ds);
	lengthUtf = Tcl_DStringLength(&ds);

	/* Call UTF-8 version */
	result = TKGS_GETDEVICEMEMBER(d, measureCharsUtf)(d, stringUtf, 
		    lengthUtf, maxPixels, flags, lengthPtr);

	/* Free converted string */
	Tcl_DStringFree(&ds);

    } else if (TKGS_GETDEVICEMEMBER(d, measureCharsUni)) {
	/* Convert string to Unicode */
	Tcl_DString dsUtf, dsUni;
	char *stringUtf;
	Tcl_UniChar *stringUni;
	int lengthUtf, lengthUni;

	Tcl_DStringInit(&dsUtf);
	Tcl_DStringInit(&dsUni);

	stringUtf = Tcl_ExternalToUtfDString(NULL /* system encoding */, 
			string, length, &dsUtf);
	lengthUtf = Tcl_DStringLength(&dsUtf);
	stringUni = Tcl_UtfToUniCharDString(stringUtf, lengthUtf, &dsUni);
	lengthUni = Tcl_DStringLength(&dsUni)/sizeof(Tcl_UniChar);

	/* Call Unicode version */
	result = TKGS_GETDEVICEMEMBER(d, measureCharsUni)(d, stringUni, 
		    lengthUni, maxPixels, flags, lengthPtr);

	/* Free converted strings */
	Tcl_DStringFree(&dsUtf);
	Tcl_DStringFree(&dsUni);

    } else {
	/* Function not available */
	*lengthPtr = 0;
	result = 0;
    }
    return result;
}

int 
TkGS_TextWidthSys(d, string, length)
    TkGS_Drawable     d;
    CONST char        *string;
    int               length;
{
    int result;

    TkGSUpdateDrawableState(d, (TkGS_GCFont));

    if (TKGS_GETDEVICEMEMBER(d, textWidthSys)) {
	/* Call system-specific version */
	result = TKGS_GETDEVICEMEMBER(d, textWidthSys)(d, string, length);

    } else if (TKGS_GETDEVICEMEMBER(d, textWidthUtf)) {
	/* Convert string to UTF-8 */
	Tcl_DString ds;
	char *stringUtf;
	int lengthUtf;

	Tcl_DStringInit(&ds);
	stringUtf = Tcl_ExternalToUtfDString(NULL /* system encoding */, 
			string, length, &ds);
	lengthUtf = Tcl_DStringLength(&ds);

	/* Call UTF-8 version */
	result = TKGS_GETDEVICEMEMBER(d, textWidthUtf)(d, stringUtf, 
		    lengthUtf);

	/* Free converted string */
	Tcl_DStringFree(&ds);

    } else if (TKGS_GETDEVICEMEMBER(d, textWidthUni)) {
	/* Convert string to Unicode */
	Tcl_DString dsUtf, dsUni;
	char *stringUtf;
	Tcl_UniChar *stringUni;
	int lengthUtf, lengthUni;

	Tcl_DStringInit(&dsUtf);
	Tcl_DStringInit(&dsUni);

	stringUtf = Tcl_ExternalToUtfDString(NULL /* system encoding */, 
			string, length, &dsUtf);
	lengthUtf = Tcl_DStringLength(&dsUtf);
	stringUni = Tcl_UtfToUniCharDString(stringUtf, lengthUtf, &dsUni);
	lengthUni = Tcl_DStringLength(&dsUni)/sizeof(Tcl_UniChar);

	/* Call Unicode version */
	result = TKGS_GETDEVICEMEMBER(d, textWidthUni)(d, stringUni, 
		    lengthUni);

	/* Free converted strings */
	Tcl_DStringFree(&dsUtf);
	Tcl_DStringFree(&dsUni);

    } else {
	/* Function not available, try with TkGS_MeasureChars */
	TkGS_MeasureCharsSys(d, string, length, -1, 0, &result);
    }
    return result;
}

void 
TkGS_DrawCharsSys(d, string, length, x, y, widthPtr)
    TkGS_Drawable     d;
    CONST char        *string;
    int               length;
    int               x;
    int               y;
    int               *widthPtr;
{
    TkGSUpdateDrawableState(d, (TkGS_GCForeground|TkGS_GCBackground|TkGS_GCFont));

    if (TKGS_GETDEVICEMEMBER(d, drawCharsSys)) {
	/* Call system-specific version */
	TKGS_GETDEVICEMEMBER(d, drawCharsSys)(d, string, length, x, y, 
	    widthPtr);

    } else if (TKGS_GETDEVICEMEMBER(d, drawCharsUtf)) {
	/* Convert string to UTF-8 */
	Tcl_DString ds;
	char *stringUtf;
	int lengthUtf;

	Tcl_DStringInit(&ds);
	stringUtf = Tcl_ExternalToUtfDString(NULL /* system encoding */, 
			string, length, &ds);
	lengthUtf = Tcl_DStringLength(&ds);

	/* Call UTF-8 version */
	TKGS_GETDEVICEMEMBER(d, drawCharsUtf)(d, stringUtf, lengthUtf, x, y, 
	    widthPtr);

	/* Free converted string */
	Tcl_DStringFree(&ds);

    } else if (TKGS_GETDEVICEMEMBER(d, drawCharsUni)) {
	/* Convert string to Unicode */
	Tcl_DString dsUtf, dsUni;
	char *stringUtf;
	Tcl_UniChar *stringUni;
	int lengthUtf, lengthUni;

	Tcl_DStringInit(&dsUtf);
	Tcl_DStringInit(&dsUni);

	stringUtf = Tcl_ExternalToUtfDString(NULL /* system encoding */, 
			string, length, &dsUtf);
	lengthUtf = Tcl_DStringLength(&dsUtf);
	stringUni = Tcl_UtfToUniCharDString(stringUtf, lengthUtf, &dsUni);
	lengthUni = Tcl_DStringLength(&dsUni)/sizeof(Tcl_UniChar);

	/* Call Unicode version */
	TKGS_GETDEVICEMEMBER(d, drawCharsUni)(d, stringUni, lengthUni, x, y, 
	    widthPtr);

	/* Free converted strings */
	Tcl_DStringFree(&dsUtf);
	Tcl_DStringFree(&dsUni);
    }
}

void 
TkGS_UnderlineCharsSys(d, string, length, x, y, first, last)
    TkGS_Drawable     d;
    CONST char        *string;
    int               length;
    int               x;
    int               y;
    int               first;
    int               last;
{
    TkGSUpdateDrawableState(d, (TkGS_GCForeground|TkGS_GCFont));

    if (TKGS_GETDEVICEMEMBER(d, underlineCharsSys)) {
	/* Call system-specific version */
	TKGS_GETDEVICEMEMBER(d, underlineCharsSys)(d, string, length, x, y, 
	    first, last);

    } else if (TKGS_GETDEVICEMEMBER(d, underlineCharsUtf)) {
	/* Convert string to UTF-8 */
	Tcl_DString ds;
	char *stringUtf;
	int lengthUtf;

	Tcl_DStringInit(&ds);
	stringUtf = Tcl_ExternalToUtfDString(NULL /* system encoding */, 
			string, length, &ds);
	lengthUtf = Tcl_DStringLength(&ds);

	/* Call UTF-8 version */
	TKGS_GETDEVICEMEMBER(d, underlineCharsUtf)(d, stringUtf, lengthUtf, 
	    x, y, first, last);

	/* Free converted string */
	Tcl_DStringFree(&ds);

    } else if (TKGS_GETDEVICEMEMBER(d, underlineCharsUni)) {
	/* Convert string to Unicode */
	Tcl_DString dsUtf, dsUni;
	char *stringUtf;
	Tcl_UniChar *stringUni;
	int lengthUtf, lengthUni;

	Tcl_DStringInit(&dsUtf);
	Tcl_DStringInit(&dsUni);

	stringUtf = Tcl_ExternalToUtfDString(NULL /* system encoding */, 
			string, length, &dsUtf);
	lengthUtf = Tcl_DStringLength(&dsUtf);
	stringUni = Tcl_UtfToUniCharDString(stringUtf, lengthUtf, &dsUni);
	lengthUni = Tcl_DStringLength(&dsUni)/sizeof(Tcl_UniChar);

	/* Call Unicode version */
	TKGS_GETDEVICEMEMBER(d, underlineCharsUni)(d, stringUni, lengthUni, 
	    x, y, first, last);

	/* Free converted strings */
	Tcl_DStringFree(&dsUtf);
	Tcl_DStringFree(&dsUni);
    }
}

/*
 *-------------------------------------------------------------------------
 *
 * TkGSFontGetAliasList --
 *
 *	Given a font name, find the list of all aliases for that font
 *	name.  One of the names in this list will probably be the name
 *	that this platform expects when asking for the font.
 *
 * Results:
 *	As above.  The return value is NULL if the font name has no 
 *	aliases.
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */
	
char **
TkGSFontGetAliasList(faceName)
    CONST char *faceName;	/* Font name to test for aliases. */
{   
    int i, j;

    for (i = 0; fontAliases[i] != NULL; i++) {
	for (j = 0; fontAliases[i][j] != NULL; j++) {
	    if (strcasecmp(faceName, fontAliases[i][j]) == 0) {
		return fontAliases[i];
	    }
	}
    }
    return NULL;
}

/*
 *-------------------------------------------------------------------------
 *
 * TkGSFontGetFallbacks --
 *
 *	Get the list of font fallbacks that the platform-specific code
 *	can use to try to find the closest matching font the name 
 *	requested.
 *
 * Results:
 *	As above.
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */
	
char ***
TkGSFontGetFallbacks()
{
    return fontFallbacks;
}

/*
 *-------------------------------------------------------------------------
 *
 * TkGSFontGetGlobalClass --
 *
 *	Get the list of fonts to try if the requested font name does not
 *	exist and no fallbacks for that font name could be used either.
 *	The names in this list are considered preferred over all the other
 *	font names in the system when looking for a last-ditch fallback.
 *
 * Results:
 *	As above.
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */
	
char **
TkGSFontGetGlobalClass()
{
    return globalFontClass;
}

/*
 *-------------------------------------------------------------------------
 *
 * TkGSFontGetSymbolClass --
 *
 *	Get the list of fonts that are symbolic; used if the operating 
 *	system cannot apriori identify symbolic fonts on its own.
 *
 * Results:
 *	As above.
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */
	
char **
TkGSFontGetSymbolClass()
{
    return symbolClass;
}

/*
 *---------------------------------------------------------------------------
 *
 *  TkGSMultiFontMeasureCharsUni --
 *
 *	Determine the number of characters from the string that will fit
 *	in the given horizontal span.  The measurement is done under the
 *	assumption that TkGSMultiFontDrawCharsUni() will be used to actually 
 *	display the characters.
 *
 * Results:
 *	The return value is the number of characters from string that
 *	fit into the span that extends from 0 to maxPixels.  *lengthPtr is
 *	filled with the x-coordinate of the right edge of the last
 *	character that did fit.
 *
 * Side effects:
 *	Extra information may be loaded in the current font object.
 *
 *---------------------------------------------------------------------------
 */

int
TkGSMultiFontMeasureCharsUni(d, font, multiFont, string, length, maxPixels, 
    flags, lengthPtr)
    TkGS_Drawable d;		/* Drawable on which font will be used. */
    TkGS_Font font;		/* Font with which the character will be 
				 * displayed. */
    TkGSMultiFont multiFont;	/* Multi-font info for the above font. */
    CONST Tcl_UniChar *string;	/* Unicode string to be displayed.  Need not be
				 * '\0' terminated. */
    int length;			/* Maximum number of characters to consider
				 * from string string. */
    int maxPixels;		/* If >= 0, maxPixels specifies the longest
				 * permissible line length in pixels; don't
				 * consider any character that would cross
				 * this x-position.  If < 0, then line length
				 * is unbounded and the flags argument is
				 * ignored. */
    int flags;			/* Various flag bits OR-ed together:
				 * TK_PARTIAL_OK means include the last char
				 * which only partially fit on this line.
				 * TK_WHOLE_WORDS means stop on a word
				 * boundary, if possible.
				 * TK_AT_LEAST_ONE means return at least one
				 * character even if no characters fit. */
    int *lengthPtr;		/* Filled with the number of pixels occupied 
				 * by the number of characters returned */
{
    TkGSSubFontId lastSubFontId;
    int curX, curChar;

    lastSubFontId = TKGS_BASE_SUBFONT;

    if (length == 0) {
	curX = 0;
	curChar = 0;
    } else if (maxPixels < 0) {
	CONST Tcl_UniChar *p, *end;
	TkGSSubFontId thisSubFontId;

	/*
	 * A two step process:
	 * 1. Find a contiguous range of characters that can all be 
	 *    represented by a single screen font.
	 * 2. Measure converted chars.
	 */

	curX = 0;
	end = string + length;
	for (p = string; p < end; p++) {
	    thisSubFontId = TkGSFindSubFontForChar(d, font, multiFont, *p);
	    if (thisSubFontId != lastSubFontId) {
/*TODO: avoid calling primitive when length is null. See TkGSMultiFontDrawCharsUni. 
  Ditto for similar procs */

		TkGSSetDrawableSubFont(d, lastSubFontId);
		curX += TkGSSubFontTextWidthUni(d, string, p - string);
		lastSubFontId = thisSubFontId;
		string = p;
	    }
	}
	TkGSSetDrawableSubFont(d, lastSubFontId);
	curX += TkGSSubFontTextWidthUni(d, string, p - string);
	curChar = length;
    } else {
	CONST Tcl_UniChar *p, *end;
	TkGSSubFontId thisSubFontId;
	CONST Tcl_UniChar *stringOrig;
        int width, widthLeft;
	CONST Tcl_UniChar *rest = NULL;
	int len;

	/*
	 * How many chars will fit in the space allotted? 
	 */

        widthLeft = maxPixels; 
        stringOrig = string;
        end = string + length;      
	for (p = string; p < end; p++) {
  	    thisSubFontId = TkGSFindSubFontForChar(d, font, multiFont, *p);
  	    if (thisSubFontId != lastSubFontId) {
  	        if (p > string) {
		    TkGSSetDrawableSubFont(d, lastSubFontId);
  	            len = TkGSSubFontMeasureCharsUni(d, string, p - string, 
			    widthLeft, flags, &width);
		    widthLeft -= width;
  	            flags &= ~TK_AT_LEAST_ONE;
		    if (len == p - string) {
			rest = NULL;
		    } else {
			rest = string+len;
  	                p = string;
  	                break;
		    }
  	        }
                lastSubFontId = thisSubFontId;
                string = p;
            }
        }
        
        if (p > string) {
	    TkGSSetDrawableSubFont(d, lastSubFontId);
            len = TkGSSubFontMeasureCharsUni(d, string, p - string, widthLeft, 
		    flags, &width);
	    widthLeft -= width;
	    if (len == p - string) {
		rest = NULL;
	    } else {
		rest = string+len;
	    }
        }
        
	if (rest == NULL) {
            curChar = length;
        } else {
            curChar = rest - stringOrig;
        }
        curX = maxPixels - widthLeft;
    }

    *lengthPtr = curX;
    return curChar;
}

/*
 *---------------------------------------------------------------------------
 *
 *  TkGSMultiFontMeasureCharsUtf --
 *
 *	Determine the number of characters from the string that will fit
 *	in the given horizontal span.  The measurement is done under the
 *	assumption that TkGSMultiFontDrawCharsUni() will be used to actually 
 *	display the characters.
 *
 * Results:
 *	The return value is the number of characters from string that
 *	fit into the span that extends from 0 to maxPixels.  *lengthPtr is
 *	filled with the x-coordinate of the right edge of the last
 *	character that did fit.
 *
 * Side effects:
 *	Extra information may be loaded in the current font object.
 *
 *---------------------------------------------------------------------------
 */

int
TkGSMultiFontMeasureCharsUtf(d, font, multiFont, string, length, maxPixels, 
    flags, lengthPtr)
    TkGS_Drawable d;		/* Drawable on which font will be used. */
    TkGS_Font font;		/* Font with which the character will be 
				 * displayed. */
    TkGSMultiFont multiFont;	/* Multi-font info for the above font. */
    CONST char *string;		/* UTF-8 string to be displayed.  Need not be
				 * '\0' terminated. */
    int length;			/* Maximum number of bytes to consider
				 * from string string. */
    int maxPixels;		/* If >= 0, maxPixels specifies the longest
				 * permissible line length in pixels; don't
				 * consider any character that would cross
				 * this x-position.  If < 0, then line length
				 * is unbounded and the flags argument is
				 * ignored. */
    int flags;			/* Various flag bits OR-ed together:
				 * TK_PARTIAL_OK means include the last char
				 * which only partially fit on this line.
				 * TK_WHOLE_WORDS means stop on a word
				 * boundary, if possible.
				 * TK_AT_LEAST_ONE means return at least one
				 * character even if no characters fit. */
    int *lengthPtr;		/* Filled with the number of pixels occupied 
				 * by the number of characters returned */
{
    TkGSSubFontId lastSubFontId;
    int curX, curByte;

    lastSubFontId = TKGS_BASE_SUBFONT;

    if (length == 0) {
	curX = 0;
	curByte = 0;
    } else if (maxPixels < 0) {
	CONST char *p, *end, *next;
	Tcl_UniChar ch;
	TkGSSubFontId thisSubFontId;

	/*
	 * A two step process:
	 * 1. Find a contiguous range of characters that can all be 
	 *    represented by a single screen font.
	 * 2. Measure converted chars.
	 */

	curX = 0;
	end = string + length;
	for (p = string; p < end; p = next) {
	    next = p + Tcl_UtfToUniChar(p, &ch);
	    thisSubFontId = TkGSFindSubFontForChar(d, font, multiFont, ch);
	    if (thisSubFontId != lastSubFontId) {
		TkGSSetDrawableSubFont(d, lastSubFontId);
		curX += TkGSSubFontTextWidthUtf(d, string, p - string);
		lastSubFontId = thisSubFontId;
		string = p;
	    }
	}
	TkGSSetDrawableSubFont(d, lastSubFontId);
	curX += TkGSSubFontTextWidthUtf(d, string, p - string);
	curByte = length;
    } else {
	Tcl_UniChar ch;
	CONST char *p, *end, *next;
	TkGSSubFontId thisSubFontId;
	CONST char *stringOrig;
        int width, widthLeft;
	CONST char *rest = NULL;
	int len;

	/*
	 * How many chars will fit in the space allotted? 
	 */

        widthLeft = maxPixels; 
        stringOrig = string;
        end = string + length;      
	for (p = string; p < end; p = next) {
	    next = p + Tcl_UtfToUniChar(p, &ch);
	    thisSubFontId = TkGSFindSubFontForChar(d, font, multiFont, ch);
  	    if (thisSubFontId != lastSubFontId) {
  	        if (p > string) {
		    TkGSSetDrawableSubFont(d, lastSubFontId);
  	            len = TkGSSubFontMeasureCharsUtf(d, string, p - string, 
			    widthLeft, flags, &width);
		    widthLeft -= width;
  	            flags &= ~TK_AT_LEAST_ONE;
		    if (len == p - string) {
			rest = NULL;
		    } else {
			rest = string+len;
  	                p = string;
  	                break;
		    }
  	        }
                lastSubFontId = thisSubFontId;
                string = p;
            }
        }
        
        if (p > string) {
	    TkGSSetDrawableSubFont(d, lastSubFontId);
  	    len = TkGSSubFontMeasureCharsUtf(d, string, p - string, 
		    widthLeft, flags, &width);
	    widthLeft -= width;
	    if (len == p - string) {
		rest = NULL;
	    } else {
		rest = string+len;
	    }
        }
        
	if (rest == NULL) {
            curByte = length;
        } else {
            curByte = rest - stringOrig;
        }
        curX = maxPixels - widthLeft;
    }

    *lengthPtr = curX;
    return curByte;
}

/*
 *-------------------------------------------------------------------------
 *
 * TkGSMultiFontDrawCharsUni --
 *
 *	Draws characters, using the various screen fonts in font and
 *	multiFont to draw multilingual characters.
 *	Note: No bidirectional support.
 *
 * Results:
 *	If widthPtr is non null, will contain the total width.
 *
 * Side effects:
 *	Information gets drawn on the screen.  
 *	Contents of multiFont may be modified if more subfonts were loaded 
 *	in order to draw all the multilingual characters in the given 
 *	string.
 *
 *-------------------------------------------------------------------------
 */

void
TkGSMultiFontDrawCharsUni(d, font, multiFont, string, length, x, y, widthPtr)
    TkGS_Drawable d;		/* Drawable on which to draw. */
    TkGS_Font font;		/* Font with which the character will be 
				 * displayed. */
    TkGSMultiFont multiFont;	/* Multi-font info for the above font. */
    CONST Tcl_UniChar *string;	/* Potentially multilingual Unicode string. */
    int length;			/* Length of string in characters. */
    int x, y;			/* Coordinates at which to place origin *
				 * of string when drawing. */
    int *widthPtr;		/* For returning the total width */
{
    TkGSSubFontId thisSubFontId, lastSubFontId;
    CONST Tcl_UniChar *p, *end;
    int xStart, w;

    lastSubFontId = TKGS_BASE_SUBFONT;
    xStart = x;

    end = string + length;
    for (p = string; p < end; p++) {
	thisSubFontId = TkGSFindSubFontForChar(d, font, multiFont, *p);
	if (thisSubFontId != lastSubFontId) {
            if (p > string) {
		TkGSSetDrawableSubFont(d, lastSubFontId);
		TkGSSubFontDrawCharsUni(d, string, p - string, x, y, &w);
		x += w;
	    }
            lastSubFontId = thisSubFontId;
            string = p;
	}
    }

    if (p > string) {
	TkGSSetDrawableSubFont(d, lastSubFontId);
	if (widthPtr) {
	    TkGSSubFontDrawCharsUni(d, string, p - string, x, y, &w);
	    x += w;
	} else {
	    TkGSSubFontDrawCharsUni(d, string, p - string, x, y, NULL);
	}
    }
    if (widthPtr) {
	*widthPtr = (x-xStart);
    }
}

/*
 *-------------------------------------------------------------------------
 *
 * TkGSMultiFontDrawCharsUtf --
 *
 *	Draws characters, using the various screen fonts in font and
 *	multiFont to draw multilingual characters.
 *	Note: No bidirectional support.
 *
 * Results:
 *	If widthPtr is non null, will contain the total width.
 *
 * Side effects:
 *	Information gets drawn on the screen.  
 *	Contents of multiFont may be modified if more subfonts were loaded 
 *	in order to draw all the multilingual characters in the given 
 *	string.
 *
 *-------------------------------------------------------------------------
 */

void
TkGSMultiFontDrawCharsUtf(d, font, multiFont, string, length, x, y, widthPtr)
    TkGS_Drawable d;		/* Drawable on which to draw. */
    TkGS_Font font;		/* Font with which the character will be 
				 * displayed. */
    TkGSMultiFont multiFont;	/* Multi-font info for the above font. */
    CONST char *string;		/* Potentially multilingual UTF-8 string. */
    int length;			/* Length of string in bytes. */
    int x, y;			/* Coordinates at which to place origin *
				 * of string when drawing. */
    int *widthPtr;		/* For returning the total width */
{
    TkGSSubFontId thisSubFontId, lastSubFontId;
    CONST char *p, *end, *next;
    Tcl_UniChar ch;
    int xStart, w;

    lastSubFontId = TKGS_BASE_SUBFONT;
    xStart = x;

    end = string + length;
    for (p = string; p < end; p = next) {
	next = p + Tcl_UtfToUniChar(p, &ch);
	thisSubFontId = TkGSFindSubFontForChar(d, font, multiFont, ch);
	if (thisSubFontId != lastSubFontId) {
            if (p > string) {
		TkGSSetDrawableSubFont(d, lastSubFontId);
		TkGSSubFontDrawCharsUtf(d, string, p - string, x, y, &w);
		x += w;
	    }
            lastSubFontId = thisSubFontId;
            string = p;
	}
    }

    if (p > string) {
	TkGSSetDrawableSubFont(d, lastSubFontId);
	if (widthPtr) {
	    TkGSSubFontDrawCharsUtf(d, string, p - string, x, y, &w);
	    x += w;
	} else {
	    TkGSSubFontDrawCharsUtf(d, string, p - string, x, y, NULL);
	}
    }
    if (widthPtr) {
	*widthPtr = (x-xStart);
    }
}

/*
 *-------------------------------------------------------------------------
 *
 * TkGSFindSubFontForChar --
 *
 *	Determine which screen font is necessary to use to display the 
 *	given character.  If the font object does not have a screen font 
 *	that can display the character, another screen font may be loaded 
 *	into the font object, following a set of preferred fallback rules.
 *
 * Results:
 *	The return value is the TkGSSubFont to use to display the given 
 *	character. 
 *
 * Side effects:
 *	The contents of fontPtr are modified to cache the results
 *	of the lookup and remember any TkGSSubFonts that were dynamically 
 *	loaded.
 *
 *-------------------------------------------------------------------------
 */

/*
 * The following structure is passed as the ClientData when calling the font
 * enumeration procedure to determine if a font can support the given
 * character.
 */

typedef struct CanUse {
    TkGS_Font font;
    TkGSMultiFont multiFont;
    Tcl_DString *nameTriedPtr;
    Tcl_UniChar ch;
    TkGSSubFontId subFontId;
} CanUse;

static TkGS_FontFamiliesEnumProc FontCanUseProc;

TkGSSubFontId
TkGSFindSubFontForChar(d, font, multiFont, ch)
    TkGS_Drawable d;		/* Drawable on which font will be used. */
    TkGS_Font font;		/* The font object with which the character
				 * will be displayed. */
    TkGSMultiFont multiFont;	/* Multi-font info for the above font. */
    Tcl_UniChar ch;		/* The Unicode character to be displayed. */
{
    int i, j, k;
    char *faceName, *fallbackName;
    char **aliases, **anyFallbacks;
    char ***fontFallbacks;
    Tcl_DString faceNames;
    TkGSSubFontId subFontId;
    CanUse canUse;

    subFontId = TkGSCanUseFont(d, font, multiFont, ch);
    if (subFontId) {
	return subFontId;
    }

    /*
     * Keep track of all face names that we check, so we don't check some
     * name multiple times if it can be reached by multiple paths.
     */
     
    Tcl_DStringInit(&faceNames);

    /* 
     * TODO: manage multiple font family specifiers. For now only take 
     * the first.
     */

    faceName = TkGSFontFaces_Names(font->attributes.faces)[0];

    /*
     * Are there any other fonts with the same face name as the base
     * font that could display this character?
     */

    if (TkGSTryHomonymousFonts(d)) {
	/*
	 * Under some systems, there can be distinct fonts with the same 
	 * face name. For example, under X, if the font is 
	 * adobe:fixed:iso8859-1, we might be able to use misc:fixed:iso8859-8 
	 * or sony:fixed:jisx0208.1983-0
	 */

	if (SeenName(faceName, &faceNames) == 0) {
	    subFontId = TkGSCanUseFallback(d, font, multiFont, faceName, ch);
	    if (subFontId) {
		goto end;
	    }
	}
    }

    aliases = TkGSFontGetAliasList(faceName);

    subFontId = 0;
    fontFallbacks = TkGSFontGetFallbacks();
    for (i = 0; fontFallbacks[i] != NULL; i++) {
	for (j = 0; fontFallbacks[i][j] != NULL; j++) {
	    fallbackName = fontFallbacks[i][j];
	    if (strcasecmp(fallbackName, faceName) == 0) {
		/*
		 * If the base font has a fallback...
		 */

		goto tryfallbacks;
	    } else if (aliases != NULL) {
		/* 
		 * Or if an alias for the base font has a fallback...
		 */

		for (k = 0; aliases[k] != NULL; k++) {
		    if (strcasecmp(fallbackName, aliases[k]) == 0) {
			goto tryfallbacks;
		    }
		}
	    }
	}
	continue;

	/* 
	 * ...then see if we can use one of the fallbacks, or an
	 * alias for one of the fallbacks.
	 */

	tryfallbacks:
	for (j = 0; fontFallbacks[i][j] != NULL; j++) {
	    fallbackName = fontFallbacks[i][j];
	    subFontId = CanUseFallbackWithAliases(d, font, multiFont, 
		    fallbackName, ch, &faceNames);
	    if (subFontId) {
		goto end;
	    }
	}
    }

    /*
     * See if we can use something from the global fallback list. 
     */

    anyFallbacks = TkGSFontGetGlobalClass();
    for (i = 0; anyFallbacks[i] != NULL; i++) {
	fallbackName = anyFallbacks[i];
	subFontId = CanUseFallbackWithAliases(d, font, multiFont, fallbackName, 
		ch, &faceNames);
	if (subFontId) {
	    goto end;
	}
    }

    /*
     * Try all face names available in the whole system until we
     * find one that can be used.
     */

    canUse.font = font;
    canUse.multiFont = multiFont;
    canUse.nameTriedPtr = &faceNames;
    canUse.ch = ch;
    canUse.subFontId = 0;
    TkGS_EnumerateFontFamilies(d, NULL,
	(TkGS_FontFamiliesEnumProc *) FontCanUseProc, (ClientData) &canUse);
    subFontId = canUse.subFontId;

    end:
    Tcl_DStringFree(&faceNames);

    if (!subFontId) {
	/*
         * No font can display this character so use the default font.
	 */

	subFontId = TkGSDefaultSubFont(d, font, multiFont);
	TkGSFontMapInsert(d, TkGSGetSubFont(d, multiFont, subFontId), ch);
    }
    return subFontId;
}

static int
FontCanUseProc(d, clientData, name)
    TkGS_Drawable d;
    ClientData clientData;
    CONST char *name;
{
    CanUse *canUsePtr;
    TkGSSubFontId subFontId;

    canUsePtr	 = (CanUse *) clientData;

    if (SeenName(name, canUsePtr->nameTriedPtr) == 0) {
	subFontId = TkGSCanUseFallback(d, canUsePtr->font, canUsePtr->multiFont, 
	    name, canUsePtr->ch);
	if (subFontId) {
	    canUsePtr->subFontId = subFontId;
	    return 0;
	}
    }
    return 1;
}


/*
 *---------------------------------------------------------------------------
 *
 * CanUseFallbackWithAliases --
 *
 *	Helper function for TkGSFindSubFontForChar.  Determine if the
 *	specified face name (or an alias of the specified face name)
 *	can be used to construct a screen font that can display the
 *	given character.
 *
 * Results:
 *	See TkGSCanUseFallback().
 *
 * Side effects:
 *	If the name and/or one of its aliases was rejected, the
 *	rejected string is recorded in nameTriedPtr so that it won't
 *	be tried again.
 *
 *---------------------------------------------------------------------------
 */

static TkGSSubFontId
CanUseFallbackWithAliases(d, font, multiFont, faceName, ch, nameTriedPtr)
    TkGS_Drawable d;		/* Drawable on which font will be used. */
    TkGS_Font font;		/* The font object that will own the new
				 * screen font. */
    TkGSMultiFont multiFont;	/* Multi-font info for the above font. */
    char *faceName;		/* Desired face name for new screen font. */
    Tcl_UniChar ch;		/* The Unicode character that the new
				 * screen font must be able to display. */
    Tcl_DString *nameTriedPtr;	/* Records face names that have already
				 * been tried.  It is possible for the same
				 * face name to be queried multiple times when
				 * trying to find a suitable screen font. */
{
    TkGSSubFontId subFontId;
    char **aliases;
    int i;
    
    if (SeenName(faceName, nameTriedPtr) == 0) {
	subFontId = TkGSCanUseFallback(d, font, multiFont, faceName, ch);
	if (subFontId) {
	    return subFontId;
	}
    }
    aliases = TkGSFontGetAliasList(faceName);
    if (aliases != NULL) {
	for (i = 0; aliases[i] != NULL; i++) {
	    if (SeenName(aliases[i], nameTriedPtr) == 0) {
		subFontId = TkGSCanUseFallback(d, font, multiFont, aliases[i], 
		    ch);
		if (subFontId) {
		    return subFontId;
		}
	    }
	}
    }
    return 0;
}

/*
 *---------------------------------------------------------------------------
 *
 * SeenName --
 *
 *	Used to determine we have already tried and rejected the given
 *	face name when looking for a screen font that can support some
 *	Unicode character.
 *
 * Results:
 *	The return value is 0 if this face name has not already been seen,
 *	non-zero otherwise.
 *
 * Side effects:
 *	If not present, the given name is appended to dsPtr.
 *
 *---------------------------------------------------------------------------
 */

static int
SeenName(name, dsPtr)
    CONST char *name;		/* The name to check. */
    Tcl_DString *dsPtr;		/* Contains names that have already been
				 * seen. */
{
    CONST char *seen, *end;

    seen = Tcl_DStringValue(dsPtr);
    end = seen + Tcl_DStringLength(dsPtr);
    while (seen < end) {
	if (strcasecmp(seen, name) == 0) {
	    return 1;
	}
	seen += strlen(seen) + 1;
    }
    Tcl_DStringAppend(dsPtr, (char *) name, (int) (strlen(name) + 1));
    return 0;
}

/*
 *-------------------------------------------------------------------------
 *
 * TkGSFontMapInitPage --
 *
 *	Allocate and initialize font mapping page for the given row if 
 *	needed.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	New pages are added to the font mapping cache whenever the
 *	character belongs to a page that hasn't been seen before.
 *	When a page is loaded, information about all the characters on
 *	that page is stored, not just for the single character in
 *	question.
 *
 *-------------------------------------------------------------------------
 */

void
TkGSFontMapInitPage(subFont, row)
    TkGSSubFont subFont;	/* Contains font mapping cache to be queried
				 * and possibly updated. */
    int row;			/* Row to initialize. */
{
    subFont->fontMap[row] = (char *) ckalloc(FONTMAP_BITSPERPAGE / 8);
    memset(subFont->fontMap[row], 0, FONTMAP_BITSPERPAGE / 8);
}

/*
 *-------------------------------------------------------------------------
 *
 * TkGSFontMapLookup --
 *
 *	See if the screen font can display the given character.
 *
 * Results:
 *	The return value is 0 if the screen font cannot display the
 *	character, non-zero otherwise.
 *
 * Side effects:
 *	New pages are added to the font mapping cache whenever the
 *	character belongs to a page that hasn't been seen before.
 *	When a page is loaded, information about all the characters on
 *	that page is stored, not just for the single character in
 *	question.
 *
 *-------------------------------------------------------------------------
 */

int
TkGSFontMapLookup(d, subFont, ch)
    TkGS_Drawable d;
    TkGSSubFont subFont;	/* Contains font mapping cache to be queried
				 * and possibly updated. */
    int ch;			/* Character to be tested. */
{
    int row, bitOffset;

    row = ch >> FONTMAP_SHIFT;
    if (subFont->fontMap[row] == NULL) {
	TkGSFontMapInitPage(subFont, row);
	TkGSFontMapLoadPage(d, subFont, row);
    }
    bitOffset = ch & (FONTMAP_BITSPERPAGE - 1);
    return (subFont->fontMap[row][bitOffset >> 3] >> (bitOffset & 7)) & 1;
}

/*
 *-------------------------------------------------------------------------
 *
 * TkGSFontMapInsert --
 *
 *	Tell the font mapping cache that the given screen font should be
 *	used to display the specified character.  This is called when no
 *	font on the system can be be found that can display that 
 *	character; we lie to the font and tell it that it can display
 *	the character, otherwise we would end up re-searching the entire
 *	fallback hierarchy every time that character was seen.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	New pages are added to the font mapping cache whenever the
 *	character belongs to a page that hasn't been seen before.
 *	When a page is loaded, information about all the characters on
 *	that page is stored, not just for the single character in
 *	question.
 *
 *-------------------------------------------------------------------------
 */

void
TkGSFontMapInsert(d, subFont, ch)
    TkGS_Drawable d;
    TkGSSubFont subFont;	/* Contains font mapping cache to be queried
				 * and possibly updated. */
    int ch;			/* Character to be added to cache. */
{
    int row, bitOffset;

    row = ch >> FONTMAP_SHIFT;
    if (subFont->fontMap[row] == NULL) {
	TkGSFontMapInitPage(subFont, row);
	TkGSFontMapLoadPage(d, subFont, row);
    }
    bitOffset = ch & (FONTMAP_BITSPERPAGE - 1);
    subFont->fontMap[row][bitOffset >> 3] |= 1 << (bitOffset & 7);
}

/*
 *-------------------------------------------------------------------------
 *
 * TkGSFreeMultiFont --
 * 
 *	Helper proc called to release a multi-font. The caller is 
 *	responsible for freeing the memory used by the font itself.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory is freed.
 *
 *---------------------------------------------------------------------------
 */

void
TkGSFreeMultiFont(multiFontPtr, stride, freeSubFontProc)
    TkGSMultiFont_ *multiFontPtr;	
				/* The font to free. */
    int stride;			/* Number of bytes between consecutive
				 * subfonts in subFontArray. */
    TkGSFreeSubFontProc freeSubFontProc;
				/* Called to delete device-specific info for
				 * each subfont. */
{
    int i;
    register char *p;
    register TkGSSubFont_ *subFontPtr;

    for (i = 0, p = (char *) multiFontPtr->subFontArray; 
	 i < multiFontPtr->numSubFonts; 
	 i++, p += stride) {

	/* 
	 * Free subfont 
	 */

	subFontPtr = (TkGSSubFont_*) p;

	/* Device-specific */
	freeSubFontProc(multiFontPtr, subFontPtr);
    }
    ckfree((char *) multiFontPtr->subFontArray);
}

void
TkGSFreeSubFont(subFontPtr, freeFontFamilyProc)
    TkGSSubFont_ *subFontPtr;	/* The subfont to free. */
    TkGSFreeFontFamilyProc freeFontFamilyProc;
				/* Called to delete device-specific info for
				 * the font family. */
{
    int i;
    register TkGSFontFamily_ *familyPtr;

    familyPtr = subFontPtr->family;
    if (familyPtr != NULL) {
	familyPtr->refCount--;
	if (familyPtr->refCount <= 0) {
	    for (i = 0; i < FONTMAP_PAGES; i++) {
		if (familyPtr->fontMap[i] != NULL) {
		    ckfree(familyPtr->fontMap[i]);
		}
	    }
	    Tcl_FreeEncoding(familyPtr->encoding);

	    /* Device-specific */
	    freeFontFamilyProc(familyPtr);

	    ckfree((char *) familyPtr);
	}
    }
}

/*
 *-------------------------------------------------------------------------
 *
 * TkGSGetClosestFaceName --
 *
 *	Return the name of the closest existing font to the specified
 *	face name.
 *
 * Results:
 *	The return value is NULL if no suitable font family exists, or
 *	an existing font family name otherwise.
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */

char *
TkGSGetClosestFaceName(d, faceName)
    TkGS_Drawable d;		/* Drawable on which font names will be 
				 * queried. */
    CONST char *faceName;	/* UTF-8 name of font family to query. */
{
    int i, j;
    char ***fontFallbacks;
    char *fallback, *actualName;

    /*
     * Algorithm to get the closest font to the name requested.
     *
     * try fontname
     * try all aliases for fontname
     * foreach fallback for fontname
     *	    try the fallback
     *	    try all aliases for the fallback
     */

    if (faceName == NULL) {
	return NULL;
    }
    actualName = TkGSFontFamilyOrAliasExists(d, faceName);
    if (actualName != NULL) {
	return actualName;
    }
    fontFallbacks = TkGSFontGetFallbacks();
    for (i = 0; fontFallbacks[i] != NULL; i++) {
	for (j = 0; (fallback = fontFallbacks[i][j]) != NULL; j++) {
	    if (strcasecmp(faceName, fallback) == 0) {
		break;
	    }
	}
	if (fallback != NULL) {
	    for (j = 0; (fallback = fontFallbacks[i][j]) != NULL; j++) {
		actualName = TkGSFontFamilyOrAliasExists(d, fallback);
		if (actualName != NULL) {
		    return actualName;
		}
	    }
	}
    }
    return NULL;
}

/*
 *-------------------------------------------------------------------------
 *
 * TkGSFontFamilyOrAliasExists --
 *
 *	Determines if any physical screen font exists on the system with 
 *	the given family name.  If the family exists, then it should be
 *	possible to construct some physical screen font with that family
 *	name.
 *
 * Results:
 *	The return value is NULL if no suitable font family exists, or
 *	an existing font family name otherwise.
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */

char *
TkGSFontFamilyOrAliasExists(d, faceName)
    TkGS_Drawable d;		/* Drawable on which font names will be 
				 * queried. */
    CONST char *faceName;	/* UTF-8 name of font family to query. */
{
    char **aliases;
    int i;

    if (TkGS_FontFamilyExists(d, faceName) != 0) {
	return (char *) faceName;
    }
    aliases = TkGSFontGetAliasList(faceName);
    if (aliases != NULL) {
	for (i = 0; aliases[i] != NULL; i++) {
	    if (TkGS_FontFamilyExists(d, aliases[i]) != 0) {
		return aliases[i];
	    }
	}
    }
    return NULL;
}






void		
TkGSInitMultiFont(multiFont)
    TkGSMultiFont multiFont;
{
    multiFont->numSubFonts = 0;
    multiFont->subFontArray = NULL;
}


void		
TkGSInitSubFont(subFont, family)
    TkGSSubFont subFont;
    TkGSFontFamily family;
{
    subFont->family = family;
    subFont->fontMap = subFont->family->fontMap;
}


void		
TkGSInitFontFamily(family, encoding)
    TkGSFontFamily family;
    Tcl_Encoding encoding;
{
    memset(family, 0, sizeof(TkGSFontFamily_));

    /* 
     * An initial refCount of 1 means that FontFamily information will
     * persist even when the SubFont that loaded the FontFamily is released.
     * Change it to 0 to cause FontFamilies to be unloaded when not in use.
     */

    family->refCount = 1;

    family->encoding = encoding;
}


