/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997	Tim Janik	<timj@psynet.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* gsi.h: generic structure inteface
 * RCS_ID("$Id: gsi.h,v 1.1 1997/08/18 03:37:09 timj Exp $")
*/


#ifndef		__gsi_h__
#define		__gsi_h__

#ifdef		__cplusplus
  extern	"C"	{
#endif

#include	<glib.h>



/* --- remarks --- */
/* PRIVATE NOTE:
 * using this interface, we try to provide an easy to
 * use access interface to generic structures by using
 * their (string) named elements.
 * also, this supports inheritance for structure types.
 *
 * hm, more to come here...
*/
/* GUBI: few parts needed to be extended (changed),
 * those are ifdef'ed.
 *
 * NOTES:
 *	the gsi_struct_info_S.gb_widget_child_count field is only valid
 *	in the first struct_info, parents don't interfear with this!
*/
#define	GUBI	(1)
#ifdef	GUBI
	#include	"gbuild.h"
#endif	/*GUBI*/



/* --- macros --- */
/* macro used to mask out (test) structure flags and field options
*/
#define	GSI_STRUCT_FLAGS(struct_info,mask)		\
	(((struct_info)->struct_flags)&(mask))
#define	GSI_FIELD_OPT(field_info,mask)			\
	(((field_info)->field_options)&(mask))


/* macros used for initialization of various fields in gsi_field_info_S
*/
#define	_GSI_VU_DOUBLE_(value)		 ((gsi_value_U)((gdouble)    (value)))
#define	_GSI_VU_FLOAT_(value)		 ((gsi_value_U)((gfloat)     (value)))
#define	_GSI_VU_LONG_(value)		 ((gsi_value_U)((glong)      (value)))
#define	_GSI_VU_INT_(value)		 ((gsi_value_U)((gint)       (value)))
#define	_GSI_VU_ENUM_(value)		 ((gsi_value_U)((guint32)    (value)))
#define	_GSI_VU_BITS_(value)		 ((gsi_value_U)((guint32)    (value)))
#define	_GSI_VU_BOOLEAN_(value)		 ((gsi_value_U)((gboolean)   (value)))
#define	_GSI_VU_STRING_(value)		 ((gsi_value_U)((gchar)      (value)))
#define	_GSI_VU_POINTER_(value)		 ((gsi_value_U)((gpointer)   (value)))
#define	_GSI_VU_STRUCT_LINK_(value)	 ((gsi_value_U)((gsi_base_S*)(value)))
#define	_GSI_FIELD_FUNC_(value)		 ((GsiGenericFieldFunc*)(value))
#define	_GSI_OFFSET_(struct_type,struct_member)	\
		((gsi_offset_U)((guint)&((struct_type*)0)->struct_member))



/* --- typedefs --- */
#ifndef	GUBI
	typedef	guint32				gsi_struct_type_E;
	typedef	struct	gsi_base_S		gsi_base_S;
#else	/*GUBI*/
	typedef	gb_struct_type_E		gsi_struct_type_E;
	typedef	gb_any_S			gsi_base_S;
#endif	/*GUBI*/


enum {
	GSI_FIELD_DOUBLE,
	GSI_FIELD_FLOAT,
	GSI_FIELD_LONG,
	GSI_FIELD_INT,
	GSI_FIELD_ENUM,
	GSI_FIELD_BITS,
	GSI_FIELD_BOOLEAN,
	GSI_FIELD_STRING,
	GSI_FIELD_POINTER,
	GSI_FIELD_STRUCT_LINK
};
typedef	guint32				gsi_field_type_E;

enum {
	GSI_FLAG_NONE		= 0,
	GSI_FLAG_MASK		= 0x0000	/* mask for structure flags
						 * that are used by gsi
						*/,
#ifdef	GUBI
	GB_FLAG_BASE_TYPE	= 1 <<  8	/* structure flag indicating
						 * base types
						*/,
	GB_FLAG_HIDDEN		= 1 <<  9	/* structure flag hides struct
						 * from widget list
						*/
#endif	/*GUBI*/
};
typedef	guint32				gsi_struct_flags_E;

enum {
	GSI_OPT_NONE		= 0,
	GSI_OPT_STRING_NONNULL	= 1 <<  0	/* on GSI_FIELD_STRING fields
						 * this requires a pointer to
						 * an empty string instead of
						 * a NULL pointer
						*/,
	GSI_OPT_STRING_DEFAULTS	= 1 <<  1	/* on GSI_FIELD_STRING fields
						 * this resets the string to
						 * it's default if empty or
						 * NULL
						*/,
	GSI_OPT_STRING_FUNCTION	= 1 <<  2	/* on GSI_FIELD_STRING fields
						 * the default value of a
						 * string is determined by a
						 * call to the field_func as
						 * GsiStringFieldFunc
						*/,
	GSI_OPT_STRING_UNQUOTED	= 1 <<  3	/* this determines wether the
						 * string will be enclosed in
						 * doublequotes or not
						 * on the conversion of
						 * GSI_FIELD_STRING fields
						*/,
	GSI_OPT_POINTER_UPCASED	= 1 <<  4	/* this determines which one of
						 * "0x%x" or "0X%X" will be
						 * used in the string
						 * conversion of
						 * GSI_FIELD_POINTER fields
						*/,
	GSI_OPT_MASK		= 0x001F	/* mask for field options that
						 * are used by gsi
						*/,
#ifdef	GUBI
	GB_OPT_REBUILD		= 1 <<  8	/* field option indicating the
						 * neccessity of a gtk-tree
						 * rebuild.
						*/,
	GB_OPT_STRING_IS_CHILD	= 1 <<  9	/* if the field is non NULL
						 * this counts as child
						 * on widgets
						*/
#endif	/*GUBI*/
};
typedef	guint32				gsi_field_options_E;

typedef	enum {
	GSI_SORT_NONE,
	GSI_SORT_BY_TYPE,
	GSI_SORT_BY_NAME
} gsi_sort_type_E;

typedef	union	gsi_offset_U		gsi_offset_U;
typedef	union	gsi_value_U		gsi_value_U;
typedef	struct	gsi_struct_info_S	gsi_struct_info_S;
typedef	struct	gsi_field_info_S	gsi_field_info_S;

typedef	void	(GsiReferenceFunc)	(gsi_base_S		*struct_p,
					 gsi_base_S		*link_p,
					 gboolean		add_link);
typedef	gchar*
		(GsiGenericFieldFunc)	(void);
typedef	gchar*
		(GsiUNum2StringFunc)	(const gulong		numeric_value,
					 const gboolean		descriptive);
typedef	gchar*
		(GsiStringFieldFunc)	(gsi_base_S		*struct_p,
					 gchar			**string_field);



/* --- structures --- */


/* BASE structure
*/
#ifndef	GUBI
struct	gsi_base_S {
	gsi_struct_type_E	type
				/* unique structure identifier
				*/;
};
#endif	/*GUBI*/


/* union to force macro usage on access
*/
union	gsi_offset_U {
	guint	pos;
};


/* union providing all valid acces types to structure fields
*/
union	gsi_value_U {
	gdouble		v_double;
	gfloat		v_float;
	glong		v_long;
	gint		v_int;
	guint32		v_enum;
	guint32		v_bits;
	gboolean	v_boolean;
	gchar		*v_string;
	gpointer	v_pointer;
	gsi_base_S	*v_struct_link;
	/* privatly used (more or less ;)
	*/
	gchar		_char;
};


/* this structure contains all the needed information about a generic
 * structure, not including information about the various fields.
*/
struct	gsi_struct_info_S {
	const gsi_struct_type_E		struct_type;
	const gchar * const		struct_name;
	const gsi_struct_flags_E	struct_flags;
	const guint			struct_size;
	const gsi_struct_info_S		*parent;
	const guint			n_fields;
	const gsi_field_info_S		*fields;
#ifdef	GUBI
	gint				gb_widget_child_count;
#endif	/*GUBI*/
};


/* this structure contains all the needed information about a field of a
 * generic structure.
*/
struct	gsi_field_info_S {
	const gsi_field_type_E		field_type;
	const gchar * const		field_name;
	const gsi_field_options_E	field_options;
	const gsi_value_U		minimum
					/* minimum value for
					 * GSI_FIELD_DOUBLE,
					 * GSI_FIELD_FLOAT,
					 * GSI_FIELD_LONG,
					 * GSI_FIELD_INT and
					 * GSI_FIELD_ENUM
					*/;
	const gsi_value_U		v_default
					/* default value for
					 * GSI_FIELD_DOUBLE,
					 * GSI_FIELD_FLOAT,
					 * GSI_FIELD_LONG,
					 * GSI_FIELD_INT,
					 * GSI_FIELD_ENUM,
					 * GSI_FIELD_BITS,
					 * GSI_FIELD_BOOLEAN,
					 * GSI_FIELD_STRING and
					 * GSI_FIELD_POINTER
					*/;
	const gsi_value_U		maximum
					/* maximum value for
					 * GSI_FIELD_DOUBLE,
					 * GSI_FIELD_FLOAT,
					 * GSI_FIELD_LONG,
					 * GSI_FIELD_INT and
					 * GSI_FIELD_ENUM.
					 * with GSI_FIELD_BITS this specifies
					 * the maximum of *settable* bits
					*/;
	GsiGenericFieldFunc		*field_func
					/* function to return a string for
					 * GSI_FIELD_ENUM,
					 * GSI_FIELD_BITS,
					 * GSI_FIELD_BOOLEAN and
					 * GSI_FIELD_STRING
					*/;
	const gsi_offset_U		field_offset
					/* field offset within structures
					 * derived from gsi_base_S
					*/;
};



/* --- prototypes --- */

/* the gsi_initialize() function will set up and initialize the
 * generic structure interface. this must be called prior to all
 * other gsi_* functions.
 * info_array is a NULL terminated array of gsi_struct_info_S pointers.
*/
void	gsi_initialize			(const gsi_struct_info_S **info_array);


/* the gsi_struct_info_add() function will inform gsi about a new
 * generic structure.
 * the pointer struct_info is asserted to be non NULL.
*/
void	gsi_struct_info_add		(const gsi_struct_info_S *struct_info);


/* the gsi_struct_info_remove() function will delete the structure information
 * from the gsi pool.
 * the pointer struct_info is asserted to be non NULL.
*/
void	gsi_struct_info_remove		(const gsi_struct_info_S *struct_info);


/* the gsi_set_reference_func() functions resets the function pointer used
 * to invoke the function that needs to be called to (un)reference structure
 * linkes on GSI_FIELD_STRUCT_LINK fields.
*/
void	gsi_set_reference_func		(GsiReferenceFunc	func_p);


/* the gsi_struct_info() function returns a pointer to the gsi_struct_info_S
 * for a specific structure type.
 * returns:
 * a pointer to a constant gsi_struct_info_S or NULL.
*/
const gsi_struct_info_S*
	gsi_struct_info			(const gsi_struct_type_E struct_type);


/* the gsi_struct_info_by_name() function returns a pointer to a
 * gsi_struct_info_S just like gsi_struct_info() with the structure
 * specified by it's name.
 * returns:
 * a pointer to a constant gsi_struct_info_S or NULL.
*/
const gsi_struct_info_S*
	gsi_struct_info_by_name		(const gchar		*struct_name);


/* the gsi_struct_info_list() function returns a linked list with
 * the struct_info pointers in GList.data of all structures sorted by
 * sort_type.
 * the GList* needs to be freed by a call to g_list_free() after usage.
*/
GList	*gsi_struct_info_list		(const gsi_sort_type_E	sort_type);


/* the gsi_struct_info_is_a() functions returns wether struct_info itself
 * or one of it's parent is of type struct_type.
*/
gboolean
	gsi_struct_info_is_a		(const gsi_struct_info_S *struct_info,
					 const gsi_struct_type_E struct_type);


/* the gsi_struct_info_n_parents() function returns the number of parents
 * a structure has.
*/
guint	gsi_struct_info_n_parents	(const gsi_struct_info_S *struct_info);


/* the gsi_struct_field_list() function returns a linked list with
 * the field_info pointers in GList.data of all fields.
 * the GList* needs to be freed by a call to g_list_free() after usage.
*/
GList	*gsi_struct_field_list		(const gsi_struct_info_S *struct_info);


/* the gsi_struct_field_list_from_to() function returns a linked list with
 * the field_info pointers in GList.data just like gsi_struct_field_list().
 * the field_info structures that got introduced by ancestors previous to
 * to_struct_type or after from_struct_type are left out.
 * the GList* needs to be freed by a call to g_list_free() after usage.
*/
GList	*gsi_struct_field_list_from_to	(const gsi_struct_info_S *struct_info,
					 const gsi_struct_type_E from_struct_type,
					 const gsi_struct_type_E to_struct_type);


/* the gsi_struct_n_fields() function returns the number of fields that
 * the gsi knows about a specific structure. this also indicates the valid
 * index for gsi_field_info() plus one.
*/
const guint
	gsi_struct_n_fields		(const gsi_struct_info_S *struct_info)
					#ifdef		__GNUC__
						__attribute__ ((const))
					#endif		/*__GNUC__*/
					;


/* the gsi_field_info() function returns a pointer to the gsi_field_info_S
 * for a specific field of a given structure specified by struct_info.
 * field_indx starts with zero, see gsi_struct_n_fields() also.
 * returns:
 * a pointer to a constant gsi_field_info_S or NULL.
*/
const gsi_field_info_S*
	gsi_field_info			(const gsi_struct_info_S *struct_info,
					 const guint		field_indx);


/* the gsi_field_info_by_name() function returns a pointer to a
 * gsi_field_info_S just like gsi_field_info() with the field
 * specified by it's name.
 * returns:
 * a pointer to a constant gsi_field_info_S or NULL.
*/
const gsi_field_info_S*
	gsi_field_info_by_name		(const gsi_struct_info_S *struct_info,
					 const gchar		*field_name);


/* the gsi_field_info_by_name2() function returns a pointer to a
 * gsi_field_info_S just like gsi_field_info() with the field and structure
 * specified by their names.
 * returns:
 * a pointer to a constant gsi_field_info_S or NULL.
*/
const gsi_field_info_S*
	gsi_field_info_by_name2		(const gchar		*struct_name,
					 const gchar		*field_name);


/* the gsi_field_set_value() function sets the specified field
 * in *struct_p to value, taking care for it's validity.
 * the pointers struct_p and field_info are asserted to be non NULL.
 * returns:
 * the boolean value returned indicates wether the field's value was changed.
*/
gboolean
	gsi_field_set_value		(gsi_base_S		*struct_p,
					 const gsi_field_info_S	*field_info,
					 gsi_value_U		value);


/* the gsi_field_set_value_by_string() function calls gsi_field_set_value()
 * after extracting the valid values from value_as_string. note that on some
 * field types the conversion facilities are very limited, this might result
 * in not even calling gsi_field_set_value().
*/
gboolean
	gsi_field_set_value_from_string	(gsi_base_S		*struct_p,
					 const gsi_field_info_S	*field_info,
					 const gchar		*value_as_string);


/* the gsi_field_get_value() function returns the current value of the field
 * specified by field_info out of *struct_p.
 * the pointers struct_p and field_info are asserted to be non NULL.
*/
gsi_value_U
	gsi_field_get_value		(gsi_base_S		*struct_p,
					 const gsi_field_info_S	*field_info);


/* the gsi_field_get_value_as_string() function is just a wrapper around
 * gsi_field_get_value() which does a value to-string conversion afterwards.
 * this function always returns a valid string, therefore GSI_FIELD_STRING
 * pointers that are in fact NULL, will be returned as
 * the empty string "".
 * for GSI_FIELD_POINTER fields the value will be in the form "0xHEXVALUE"
 * (see also GSI_OPT_POINTER_UPCASED).
 * the pointers struct_p and field_info are asserted to be non NULL.
 * returns:
 * a pointer to a constant string.
 * the returned string can only be used until the next call to this function.
*/
const gchar*
	gsi_field_get_value_as_string	(gsi_base_S		*struct_p,
					 const gsi_field_info_S	*field_info);


/* the gsi_struct_init_fields() function sets all fields of Struct
 * to their default values.
 * in contrast to structure_field_set_default() this function doesn't
 * care about any existing values.
 * the pointers struct_info and struct_p are asserted to be non NULL.
*/
void	gsi_struct_init_fields		(const gsi_struct_info_S *struct_info,
					 gsi_base_S		*struct_p);


/* the gsi_field_set_default() function calls gsi_field_set_value()
 * with the respective default value for the specified field.
 * the pointers struct_p and field_info are asserted to be non NULL.
*/
void	gsi_field_set_default		(gsi_base_S		*struct_p,
					 const gsi_field_info_S	*field_info);


/* the gsi_struct_free_fields() function frees any data that got previously
 * allocated through field assignments.
 * do not rely on any field values after this function.
 * the pointers struct_info and struct_p are asserted to be non NULL.
*/
void	gsi_struct_free_fields		(const gsi_struct_info_S *struct_info,
					 gsi_base_S		*struct_p);


/* the gsi_field_has_default() function returns wether the specified field
 * has been changed since the last call to gsi_field_set_default().
 * the pointers struct_p and field_info are asserted to be non NULL.
*/
gboolean
	gsi_field_has_default		(gsi_base_S		*struct_p,
					 const gsi_field_info_S	*field_info);


/* the gsi_struct_init_fields() function allocates the space needed for a
 * structure and initializes it's fields by calling g_malloc0() and
 * gsi_struct_init_fields().
*/
gsi_base_S*
	gsi_struct_new0			(const gsi_struct_info_S *struct_info);


/* the gsi_struct_free() function frees all the space immediatly allocated
 * by the structure through calls to gsi_struct_free_fields() and g_free().
*/
void	gsi_struct_free			(gsi_base_S		*struct_p);


/* the gsi_field_n_bits() function computes the amount of usable bits in a
 * GSI_FIELD_BITS.
 * the pointer field_info is asserted to be non NULL.
*/
guint	gsi_field_n_bits		(const gsi_field_info_S	*field_info);


/* the gsi_field_bit_value() function returns the unsigned integer value
 * representing the nth_bit of a field as returned by gsi_field_n_bits().
 * the pointer field_info is asserted to be non NULL.
 * returns:
 * the corresponding bit value or 0 if unspecified.
*/
guint32	gsi_field_bit_value		(const gsi_field_info_S	*field_info,
					 const guint		nth_bit);


/* the gsi_field_set_bit() function sets the nth_bit as returned by
 * gsi_field_n_bits() of the field specified by field_info out of *struct_p
 * to the specified state wich may be either of TRUE or FALSE.
 * the pointers struct_p and field_info are asserted to be non NULL.
*/
void	gsi_field_set_bit		(gsi_base_S		*struct_p,
					 const gsi_field_info_S	*field_info,
					 const guint		nth_bit,
					 gboolean		state);


/* the gsi_values_equal() function returns wether the given values are
 * equal according to the field type and possible field options.
*/
gboolean
	gsi_values_equal		(const gsi_field_type_E	field_type,
					 const gsi_field_options_E field_options,
					 gsi_value_U		value1,
					 gsi_value_U		value2);


/* the gsi_struct_unlink() function resets all fields of type
 * GSI_FIELD_STRUCT_LINK from *linking_struct_p that refer to struct_p
 * to their default values.
 * this automatically erases the corresponding entries from the reference
 * list of struct_p by calling the GsiReferenceFunc (if this has been set
 * through gsi_set_reference_func()).
*/
void	gsi_struct_unlink      		(gsi_base_S     *struct_p,
	        	                 gsi_base_S     *linking_struct_p);






#ifdef		__cplusplus
  }
#endif
#endif		/*__gsi_h__*/
