/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998, 1999 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	<gle/gleconfig.h>

#include	"gleshell.h"
#include	"gleted.h"
#include	"gleeditor.h"
#include	"gledumper.h"
#include	"glewpalette.h"
#include	"glefiledialog.h"
#include	"glercargs.h"
#include	"glercpref.h"
#include	"gleparser.h"
#include	"glestock.h"
#include	"gleprivate.h"
#include	"gledocu.inc"
#include	<gdk/gdk.h>
#include	<string.h>
#include	<stdio.h>



/* --- signals --- */
enum
{
  SIGNAL_LAST
};


/* --- prototypes --- */
static void		gle_shell_class_init		(GleShellClass	*class);
static void		gle_shell_init			(GleShell	*shell);
static void		gle_shell_destroy		(GtkObject	*object);
static void		gle_shell_finalize		(GtkObject	*object);
static void		gle_shell_realize		(GtkWidget	*widget);
static void		gle_shell_unrealize		(GtkWidget	*widget);
static gint		gle_shell_configure_event	(GtkWidget	*widget,
							 GdkEventConfigure *event);
static gint		gle_shell_delete_event		(GtkWidget	*widget,
							 GdkEventAny	*dummy);
static void		gle_shell_do_selection		(GleShell	*shell);
static void		gle_shell_candidate_check	(GtkObject	*caller,
							 GtkWidget     **new_candidate,
							 gint		*candidate_ok);
static gint		gle_shell_plist_button_press	(GleShell	*shell,
							 GdkEventButton *event);
static GQuark		gle_shell_plist_check_tag	(GleShell	*shell,
							 GleGObject	*gobject);
static void		gle_shell_popup_finish		(GleShell	*shell);
static gboolean		gle_shell_popup_check		(GleShell	*shell,
							 GleGWidget	*gwidget,
							 guint		 pop);
static void		gle_shell_popup_operation	(GleShell	*shell,
							 guint		 shell_op,
							 GtkWidget	*item);


/* --- variables --- */
static GtkWindowClass	*parent_class = NULL;
static GleShellClass	*gle_shell_class = NULL;
static GleShell		*gle_shell = NULL;
static gchar		*gle_rc_file = NULL;
static const gchar *gle_rc_string =
( "style'GLE-DefaultStyle'"
  "{"
  "font='-adobe-helvetica-medium-r-normal--*-120-*-*-*-*-*-*'\n"
  "fg[NORMAL]={0.,0.,0.}"
  "fg[ACTIVE]={0.,0.,0.}"
  "fg[PRELIGHT]={0.,0.,0.}"
  "fg[SELECTED]={1.,1.,1.}"
  "fg[INSENSITIVE]={.457,.457,.457}"
  "bg[NORMAL]={.83,.83,.87}"
  "bg[ACTIVE]={.76,.76,.80}"
  "bg[PRELIGHT]={.91,.91,.95}"
  "bg[SELECTED]={0.,0.,.65}"
  "bg[INSENSITIVE]={.83,.83,.87}"
  "base[NORMAL]={1.,1.,1.}"
  "base[PRELIGHT]={1.,0.,0.}"
  "base[INSENSITIVE]={.83,.83,.87}"
  "text[INSENSITIVE]={.83,.83,.87}"
  "}"
  /* feature normal GLE dialogs */
  "widget'GLE-*'style'GLE-DefaultStyle'"
  /* feature ordinary GLE menus */
  "widget'GtkWindow.GLE-*'style'GLE-DefaultStyle'"
  /* feature GLE item factory menus */
  "widget'GtkWindow.<GLE-*'style'GLE-DefaultStyle'"
  "\n" );

/* --- menus --- */
static gchar	*gle_shell_factories_path = "<GLE-GleShell>";
static GtkItemFactoryEntry menubar_entries[] =
{
#define SHELL_OP(shell_op)	(gle_shell_operation), (GLE_SHELL_OP_ ## shell_op)
  { "/_File/_New",	"<ctrl>N",	SHELL_OP (NONE),	"<Item>" },
  { "/File/_Open...",	"<ctrl>O",	SHELL_OP (OPEN),	"<Item>" },
  { "/File/_Save...",	"<ctrl>S",	SHELL_OP (SAVE),	"<Item>" },
  { "/File/-----",	"",		NULL, 0,		"<Separator>" },
  { "/File/_Preferences...", "",	SHELL_OP (RC_PREF),	"<Item>" },
  { "/File/-----",	"",		NULL, 0,		"<Separator>" },
  { "/File/_Quit GLE",	"<ctrl>Q",	SHELL_OP (DELETE),	"<Item>" },
  { "/_Dialogs/_Widget Palette...", "",	SHELL_OP (WPALETTE),	"<Item>" },
  { "/_Proxy/_New Window", "",		SHELL_OP (NEW_WINDOW),	NULL,	},
  { "/_Help",		"",		NULL,	0,		"<LastBranch>" },
  { "/Help/_Intro...",	"",		SHELL_OP (HELP_INTRO),	"<Item>" },
  { "/Help/_About...",	"",		SHELL_OP (HELP_ABOUT),	"<Item>" },
#undef	SHELL_OP
};
static guint n_menubar_entries = sizeof (menubar_entries) / sizeof (menubar_entries[0]);
enum
{
  POPUP_OP_NONE,
  POPUP_OP_P_TED,
  POPUP_OP_P_EDITOR,
  POPUP_OP_P_INSTANTIATE,
  POPUP_OP_P_DISASSOCIATE,
  POPUP_OP_P_DUMP,
  POPUP_OP_P_PP,
  POPUP_OP_P_DESTROY,
  POPUP_OP_W_HIDE,
  POPUP_OP_W_SHOW,
  POPUP_OP_W_DESTROY,
  POPUP_OP_LAST
};
static GtkItemFactoryEntry popup_entries[] =
{
#define POP(popup_op)	(gle_shell_popup_operation), (POPUP_OP_ ## popup_op)
  { "/Proxy Operations", "",		POP (NONE),			"<Title>" },
  { "/---",		  "",		NULL, 0,			"<Separator>" },
  { "/Tree Editor...",	"<ctrl>T",	POP (P_TED),			NULL },
  { "/Widget Editor...", "<ctrl>E",	POP (P_EDITOR),			NULL },
  { "/Instantiate",	NULL,		POP (P_INSTANTIATE),		NULL },
  { "/Disassociate",	NULL,		POP (P_DISASSOCIATE),		NULL },
  { "/Stdout Dump",	NULL,		POP (P_DUMP),			NULL },
  { "/PP",		NULL,		POP (P_PP),			NULL },
  { "/Destroy",		NULL,		POP (P_DESTROY),		NULL },
  { "/---",		"",		NULL, 0,			"<Separator>" },
  { "/Widget Operations", "",		POP (NONE),			"<Title>" },
  { "/---",		"",		NULL, 0,			"<Separator>" },
  { "/Show Widget",	"<ctrl>S",	POP (W_SHOW),			NULL },
  { "/Hide Widget",	"<ctrl>H",	POP (W_HIDE),			NULL },
  { "/Destroy Widget",	NULL,		POP (W_DESTROY),		NULL },
#undef	POP
};
static guint n_popup_entries = (sizeof (popup_entries) / sizeof (popup_entries[0]));


/* --- functions --- */
GtkType
gle_shell_get_type (void)
{
  static GtkType shell_type = 0;
  
  if (!shell_type)
    {
      GtkTypeInfo shell_info =
      {
	"GleShell",
	sizeof (GleShell),
	sizeof (GleShellClass),
	(GtkClassInitFunc) gle_shell_class_init,
	(GtkObjectInitFunc) gle_shell_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
	(GtkClassInitFunc) NULL,
      };
      
      shell_type = gtk_type_unique (GTK_TYPE_WINDOW, &shell_info);
    }
  
  return shell_type;
}

static void
gle_shell_class_init (GleShellClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;
  GtkWindowClass *window_class;
  
  gle_shell_class = class;
  parent_class = gtk_type_class (GTK_TYPE_WINDOW);
  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;
  container_class = (GtkContainerClass*) class;
  window_class = (GtkWindowClass*) class;
  
  object_class->destroy = gle_shell_destroy;
  object_class->finalize = gle_shell_finalize;
  
  widget_class->realize = gle_shell_realize;
  widget_class->unrealize = gle_shell_unrealize;
  widget_class->delete_event = gle_shell_delete_event;
  widget_class->configure_event = gle_shell_configure_event;
  
  class->factories_path = gle_shell_factories_path;
  
  gtk_rc_parse_string (gle_rc_string);
}

static void
gle_shell_init (GleShell *shell)
{
  GtkWidget *shell_vbox;
  GtkWidget *main_vbox;
  GtkWidget *hbox;
  GtkWidget *button;
  GtkWidget *any;
  gchar *string;
  
  if (gle_shell)
    g_error ("GleShell works only solely");
  else
    gle_shell = shell;
  
  GLE_TAG (shell, "GLE-Shell");
  
  /* structure setup */
  shell->plist = NULL;
  shell->update_button = NULL;
  shell->selector_button = NULL;
  shell->selector = NULL;
  shell->menubar_factory = NULL;
  shell->popup_factory = NULL;
  shell->popup_data = NULL;
  shell->gtoplevel_hook = 0;
  shell->rc_pref = NULL;
  shell->wpalette = NULL;
  shell->file_open_dialog = NULL;
  shell->file_save_dialog = NULL;
  shell->help_intro = NULL;
  shell->help_about = NULL;
  
  /* GLE rc parsing
   */
  if (!gle_rc_file)
    {
      gle_rc_args_init ();
      gle_rc_file = g_strconcat (g_get_home_dir (), "/.glerc", NULL);
    }
  gle_rc_parse (gle_rc, gle_rc_file);
  
  /* create the popup factory
   */
  shell->popup_factory =
    gtk_item_factory_new (GTK_TYPE_MENU,
			  gle_shell_class->factories_path,
			  NULL);
  GLE_TAG_WEAK (shell->popup_factory);
  gtk_signal_connect (GTK_OBJECT (shell->popup_factory),
		      "destroy",
		      GTK_SIGNAL_FUNC (gtk_widget_destroyed),
		      &shell->popup_factory);
  gtk_item_factory_create_items (shell->popup_factory,
				 n_popup_entries,
				 popup_entries,
				 shell);
  gtk_window_add_accel_group (GTK_WINDOW (shell), shell->popup_factory->accel_group);
  
  /* create GUI
   */
  string = g_strconcat ("GLE-Shell", "	 [", gle_get_prgname (), "]", NULL);
  gtk_widget_set (GTK_WIDGET (shell),
		  "GtkWindow::title", string,
		  "GtkWindow::type", GTK_WINDOW_TOPLEVEL,
		  "GtkWindow::allow_shrink", TRUE,
		  "GtkWindow::allow_grow", TRUE,
		  "GtkWindow::auto_shrink", FALSE,
		  NULL);
  g_free (string);
  gtk_widget_set_usize (GTK_WIDGET (shell),
			GLE_RCVAL_SHELL_WIDTH > 0 ? GLE_RCVAL_SHELL_WIDTH : -1,
			GLE_RCVAL_SHELL_HEIGHT > 0 ? GLE_RCVAL_SHELL_HEIGHT : -1);
  if (GLE_RCVAL_SHELL_USE_POSITION)
    gtk_widget_set_uposition (GTK_WIDGET (shell),
			      GLE_RCVAL_SHELL_INIT_X > 0 ? GLE_RCVAL_SHELL_INIT_X : -1,
			      GLE_RCVAL_SHELL_INIT_Y > 0 ? GLE_RCVAL_SHELL_INIT_Y : -1);
  shell_vbox =
    gtk_widget_new (GTK_TYPE_VBOX,
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 0,
		    "GtkContainer::border_width", 0,
		    "GtkWidget::parent", shell,
		    "GtkWidget::visible", TRUE,
		    NULL);
  main_vbox =
    gtk_widget_new (GTK_TYPE_VBOX,
		    "homogeneous", FALSE,
		    "spacing", 5,
		    "border_width", 5,
		    "parent", shell_vbox,
		    "visible", TRUE,
		    NULL);
  gtk_box_set_child_packing (GTK_BOX (shell_vbox), main_vbox, TRUE, TRUE, 0, GTK_PACK_START);
  
  /* toplevel proxy list
   */
  shell->plist = (GlePList*)
    gtk_widget_new (GLE_TYPE_PLIST,
		    "visible", TRUE,
		    "object_signal::button-press-event", gle_shell_plist_button_press, shell,
		    "object_signal::check-tag", gle_shell_plist_check_tag, shell,
		    "signal::destroy", gtk_widget_destroyed, &shell->plist,
		    NULL);
  any = gtk_widget_new (GTK_TYPE_SCROLLED_WINDOW,
			"visible", TRUE,
			"hscrollbar_policy", GTK_POLICY_AUTOMATIC,
			"vscrollbar_policy", GTK_POLICY_AUTOMATIC,
			"parent", main_vbox,
			NULL);
  gtk_container_add (GTK_CONTAINER (any), GTK_WIDGET (shell->plist));
  
  /* buttons
   */
  hbox =
    gtk_widget_new (GTK_TYPE_HBOX,
		    "visible", TRUE,
		    "GtkBox::homogeneous", TRUE,
		    "GtkBox::spacing", 5,
		    "border_width", 0,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, TRUE, 0);
  shell->update_button =
    gtk_widget_new (GTK_TYPE_BUTTON,
		    "visible", FALSE,
		    "label", "Update",
		    "object_signal::clicked", gle_plist_update_all, shell->plist,
		    "parent", hbox,
		    NULL);
  shell->selector_button =
    gtk_widget_new (GTK_TYPE_BUTTON,
		    "visible", TRUE,
		    "label", GLE_RCVAL_SHELL_SELECTOR_NAME,
		    "object_signal::clicked", gle_shell_do_selection, shell,
		    "parent", hbox,
		    NULL);
  any =
    gtk_widget_new (GTK_TYPE_HSEPARATOR,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (shell_vbox), any, FALSE, TRUE, 0);
  button =
    gtk_widget_new (GTK_TYPE_BUTTON,
		    "visible", TRUE,
		    "parent", shell_vbox,
		    "label", "Close",
		    "border_width", 5,
		    "can_default", TRUE,
		    "has_default", TRUE,
		    "object_signal::clicked", gle_shell_delete_event, shell,
		    NULL);
  gtk_box_set_child_packing (GTK_BOX (shell_vbox), button, FALSE, TRUE, 0, GTK_PACK_START);
  
  /* MenuBar
   */
  shell->menubar_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR,
						 gle_shell_factories_path,
						 NULL);
  GLE_TAG_WEAK (shell->menubar_factory);
  gtk_window_add_accel_group (GTK_WINDOW (shell),
			      GTK_ITEM_FACTORY (shell->menubar_factory)->accel_group);
  gtk_item_factory_create_items (GTK_ITEM_FACTORY (shell->menubar_factory),
				 n_menubar_entries,
				 menubar_entries,
				 shell);
  gtk_container_add_with_args (GTK_CONTAINER (shell_vbox),
			       GTK_ITEM_FACTORY (shell->menubar_factory)->widget,
			       "expand", FALSE,
			       "position", 0,
			       NULL);
  gtk_widget_show (GTK_ITEM_FACTORY (shell->menubar_factory)->widget);
  gtk_signal_connect (GTK_OBJECT (shell->menubar_factory),
		      "destroy",
		      GTK_SIGNAL_FUNC (gtk_widget_destroyed),
		      &shell->menubar_factory);
  
  /* setup the GLE selector
   */
  shell->selector = gle_selector_new ("GLE Selector", "Select a window by clicking");
  GLE_TAG (shell->selector, "GLE-Shell-Selector");
  GLE_TAG_WEAK (GLE_SELECTOR (shell->selector)->warning_window);
  gtk_widget_set (shell->selector,
		  "object_signal::candidate_check", gle_shell_candidate_check, shell,
		  "signal::destroy", gtk_widget_destroyed, &shell->selector,
		  NULL);
}

static void
gle_shell_destroy (GtkObject *object)
{
  GleShell *shell;
  
  g_return_if_fail (object != NULL);
  g_return_if_fail (GLE_IS_SHELL (object));
  
  shell = GLE_SHELL (object);
  g_assert (shell == gle_shell);
  
  gle_set_flash_widget (NULL);
  
  gdk_flush ();
  
  gle_plist_clear (shell->plist);

  /*
    for (list = gle_gtoplevel_get_list (); list; list = list->next)
    if (GLE_GOBJECT_CHECK_NOTIFIER_F (list->data, gle_bomb_shell,))
    GLE_NOTIFIER_REMOVE_F (list->data, gle_bomb_shell);
  */
  
  /* get rid of the menu factories */
  gtk_object_unref (GTK_OBJECT (shell->popup_factory));
  gtk_object_unref (GTK_OBJECT (shell->menubar_factory));
  
  /* destroy associated dialogs */
  if (shell->rc_pref)
    gtk_widget_destroy (shell->rc_pref);
  if (shell->wpalette)
    gtk_widget_destroy (shell->wpalette);
  if (shell->file_open_dialog)
    gtk_widget_destroy (shell->file_open_dialog);
  if (shell->file_save_dialog)
    gtk_widget_destroy (shell->file_save_dialog);
  
  if (shell->selector)
    gtk_widget_destroy (shell->selector);
  
  GTK_OBJECT_CLASS (parent_class)->destroy (object);
  
  gle_shell = NULL;
}

static void
gle_shell_finalize (GtkObject *object)
{
  GleShell *shell;
  
  g_return_if_fail (object != NULL);
  g_return_if_fail (GLE_IS_SHELL (object));
  
  shell = GLE_SHELL (object);
  
  GTK_OBJECT_CLASS (parent_class)->finalize (object);
  
  /* save ~/.glerc
   */
  if (GLE_RCVAL_AUTO_SAVE_GLERC)
    gle_rc_dump (gle_rc, gle_rc_file);
}

static gint
gle_shell_delete_event (GtkWidget   *widget,
			GdkEventAny *dummy)
{
  g_return_val_if_fail (widget != NULL, TRUE);
  g_return_val_if_fail (GLE_IS_SHELL (widget), TRUE);
  
  /* FIXME: need "want to save" popup here */
  
  gle_shell_operation (GLE_SHELL (widget), GLE_SHELL_OP_DELETE);
  
  return TRUE;
}

static gint
gle_shell_configure_event (GtkWidget	       *widget,
			   GdkEventConfigure   *event)
{
  if (GLE_RCVAL_SHELL_TRACK_MOVEMENTS)
    {
      gint x, y;
      
      gdk_window_get_root_origin (widget->window, &x, &y);
      gle_rc_set_long (gle_rc, _GLE_RCARG_SHELL_INIT_X, x);
      gle_rc_set_long (gle_rc, _GLE_RCARG_SHELL_INIT_Y, y);
    }
  
  if (GTK_WIDGET_CLASS (parent_class)->configure_event)
    return GTK_WIDGET_CLASS (parent_class)->configure_event (widget, event);
  else
    return FALSE;
}

static void
gle_shell_gtoplevel_hook (GleGToplevel	 *gtoplevel,
			  gpointer	  data)
{
  GleShell *shell;
  
  shell = GLE_SHELL (data);
  gle_plist_prepend (shell->plist, GLE_GOBJECT (gtoplevel));
}

static void
gle_shell_realize (GtkWidget *widget)
{
  GleShell *shell;
  GList *list;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GLE_IS_SHELL (widget));
  
  shell = GLE_SHELL (widget);
  
  GTK_WIDGET_CLASS (parent_class)->realize (widget);
  
  /* fill in the plist */
  for (list = g_list_last (gle_gtoplevel_get_list ()); list; list = list->prev)
    gle_plist_prepend (shell->plist, list->data);
  shell->gtoplevel_hook = gle_gtoplevel_add_init_hook (gle_shell_gtoplevel_hook, shell);
}

static void
gle_shell_unrealize (GtkWidget *widget)
{
  GleShell *shell;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GLE_IS_SHELL (widget));
  
  shell = GLE_SHELL (widget);
  
  gle_gtoplevel_remove_init_hook (shell->gtoplevel_hook);
  shell->gtoplevel_hook = 0;
  gle_plist_clear (shell->plist);
  
  GTK_WIDGET_CLASS (parent_class)->unrealize (widget);
}

GleShell*
gle_shell_popup (void)
{
  if (!gle_shell)
    gtk_widget_new (GLE_TYPE_SHELL, NULL);
  
  gtk_widget_show (GTK_WIDGET (gle_shell));
  gdk_window_raise (GTK_WIDGET (gle_shell)->window);
  
  return gle_shell;
}

gboolean
gle_shell_is_popped_up (void)
{
  return gle_shell != NULL;
}

static void
gle_shell_candidate_check (GtkObject  *caller,
			   GtkWidget **new_candidate,
			   gint	      *candidate_ok)
{
  GleShell *shell;
  GtkWidget *toplevel;
  
  g_return_if_fail (caller != NULL);
  g_return_if_fail (GLE_IS_SHELL (caller));
  
  shell = GLE_SHELL (caller);
  
  toplevel = *new_candidate ? gtk_widget_get_toplevel (*new_candidate) : NULL;
  if (!toplevel || GLE_HAS_TAG (toplevel))
    *candidate_ok = FALSE;
  
  *new_candidate = toplevel;
  
  if (*candidate_ok)
    {
      *candidate_ok = (*new_candidate &&
		       *new_candidate != GTK_WIDGET (shell) &&
		       !GLE_HAS_TAG (*new_candidate));
      
      if (*candidate_ok)
	{
	  GtkItemFactory *ifactory;
	  
	  ifactory = gtk_item_factory_from_widget (*new_candidate);
	  if (ifactory && GLE_HAS_TAG (ifactory))
	    *candidate_ok = FALSE;
	}
    }
  
  if (!*candidate_ok)
    {
      gle_plist_mark (shell->plist, NULL);
    }
  else
    {
      GleGObject *gobject;

      gobject = gle_object_get_gobject (GTK_OBJECT (*new_candidate));
      gle_plist_mark (shell->plist, gobject);
    }
}

static void
gle_shell_do_selection (GleShell *shell)
{
  gboolean do_selection;
  GtkWidget *customer;
  
  g_return_if_fail (shell != NULL);
  g_return_if_fail (GLE_IS_SHELL (shell));
  
  do_selection = !gle_selector_in_selection (NULL);
  
  if (do_selection && gdk_pointer_is_grabbed () &&
      !GLE_SELECTOR (shell->selector)->warning_visible)
    {
      gle_selector_popup_warning (GLE_SELECTOR (shell->selector));
      do_selection = FALSE;
    }
  
  if (do_selection)
    {
      if (!gdk_pointer_is_grabbed () &&
	  !GTK_WIDGET_VISIBLE (shell))
	{
	  gtk_widget_show (GTK_WIDGET (shell));
	  gdk_window_raise (GTK_WIDGET (shell)->window);
	}
      
      gtk_object_ref (GTK_OBJECT (shell));
      
      customer = gle_selector_make_selection (GLE_SELECTOR (shell->selector));
      
      if (!GTK_OBJECT_DESTROYED (shell) &&
	  shell->selector)
	{
	  if (customer && GTK_OBJECT (customer)->ref_count > 1)
	    gle_shell_add_toplevel (shell, customer);
	  else
	    gle_plist_mark (shell->plist, NULL);
	  
	  gle_selector_reset (GLE_SELECTOR (shell->selector));

	  if (!GTK_WIDGET_VISIBLE (shell))
	    gtk_widget_show (GTK_WIDGET (shell));
	}
      
      gtk_object_unref (GTK_OBJECT (shell));
    }
}

static void
gle_bomb_shell (GleGObject *gobject)
{
  if (GLE_GOBJECT_IS_INSTANTIATED (gobject) &&
      !GTK_OBJECT_DESTROYED (gobject->object))
    {
    }
  else
    {
      gle_gobject_destroy (gobject);
      if (gle_shell)
	gtk_widget_destroy (GTK_WIDGET (gle_shell));
    }
}

void
gle_shell_add_toplevel (GleShell  *shell,
			GtkWidget *widget)
{
  GleGWidget *gwidget;

  g_return_if_fail (shell != NULL);
  g_return_if_fail (GLE_IS_SHELL (shell));
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_WINDOW (widget));

  gwidget = gle_widget_get_gwidget (widget);

  if (!gwidget)
    {
      gle_plist_freeze (shell->plist);

      gwidget = gle_toplevel_make_gwidget (widget);

      gle_shell_own_gtoplevel (gwidget);

      gle_gwidget_unref (gwidget);

      gle_plist_mark (shell->plist, GLE_GOBJECT (gwidget));

      gle_plist_thaw (shell->plist);
    }
}

void
gle_shell_own_gtoplevel (GleGWidget *gwidget)
{
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GTOPLEVEL (gwidget));

  if (!GLE_GOBJECT_SHELL_OWNED (gwidget))
    {
      gle_gwidget_ref (gwidget);

      GLE_GOBJECT_SET_FLAG (gwidget, GLE_GOBJECT_SHELL_OWNED);
      
      GLE_NOTIFIER_INSTALL (gwidget, "pre_disassociate", gle_bomb_shell, NULL);
      GLE_NOTIFIER_INSTALL (gwidget, "pre_destroy", gle_bomb_shell, NULL);

      if (gle_shell && !GTK_OBJECT_DESTROYED (gle_shell))
	gle_plist_update (gle_shell->plist, GLE_GOBJECT (gwidget));
    }
}

void
gle_shell_give_up_gtoplevel (GleGWidget *gwidget)
{
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GTOPLEVEL (gwidget));

  if (GLE_GOBJECT_SHELL_OWNED (gwidget))
    {
      GLE_GOBJECT_UNSET_FLAG (gwidget, GLE_GOBJECT_SHELL_OWNED);

      if (GLE_NOTIFIER_CHECK_MFD (gwidget, "pre_disassociate", gle_bomb_shell, NULL))
	GLE_NOTIFIER_REMOVE_MFD (gwidget, "pre_disassociate", gle_bomb_shell, NULL);
      if (GLE_NOTIFIER_CHECK_MFD (gwidget, "pre_destroy", gle_bomb_shell, NULL))
	GLE_NOTIFIER_REMOVE_MFD (gwidget, "pre_destroy", gle_bomb_shell, NULL);

      if (gle_shell && !GTK_OBJECT_DESTROYED (gle_shell))
	gle_plist_update (gle_shell->plist, GLE_GOBJECT (gwidget));

      gle_gwidget_unref (gwidget);
    }
}

GList*
gle_shell_list_selected (void)
{
  if (gle_shell && !GTK_OBJECT_DESTROYED (gle_shell))
    return gle_plist_list_selected (gle_shell->plist);
  else
    return NULL;
}

GList*
gle_shell_list_all (void)
{
  if (gle_shell && !GTK_OBJECT_DESTROYED (gle_shell))
    return gle_plist_list_all (gle_shell->plist);
  else
    return NULL;
}

static GQuark
gle_shell_plist_check_tag (GleShell	  *shell,
			   GleGObject	  *gobject)
{
  if (!GLE_GOBJECT_SHELL_OWNED (gobject))
    return g_quark_try_string (GLE_XPM_CODE);
  else if (GLE_NOTIFIER_CHECK_F (gobject, gle_bomb_shell))
    return g_quark_try_string (GLE_XPM_BOMB);
  else if (1)
    return g_quark_try_string (GLE_XPM_SHELL_OWNED);
  else
    return 0;
}

static void
gle_shell_popup_finish (GleShell *shell)
{
  guint i;
  
  g_return_if_fail (shell != NULL);
  g_return_if_fail (GLE_IS_SHELL (shell));
  
  shell->popup_data = NULL;
  
  if (GTK_OBJECT_DESTROYED (shell))
    return;
  
  for (i = 0; i < POPUP_OP_LAST; i++)
    gle_item_factory_adjust (shell->popup_factory,
			     i,
			     TRUE,
			     NULL);
}

static gint
gle_shell_plist_button_press (GleShell	     *shell,
			      GdkEventButton *event)
{
  gboolean handled;
  
  g_return_val_if_fail (shell != NULL, FALSE);
  g_return_val_if_fail (GLE_IS_SHELL (shell), FALSE);
  
  handled = FALSE;
  if (event->type == GDK_BUTTON_PRESS)
    {
      gint row = 0;
      gint column;
      
      if (gle_clist_selection_info (GTK_CLIST (shell->plist),
				    event->window,
				    event->x, event->y,
				    &row, &column))
	{
	  if (event->button == 3)
	    {
	      GleGObject *gobject;
	      guint i;
	      
	      handled = TRUE;
	      
	      gobject = gle_plist_get_gobject (shell->plist, row);
	      g_return_val_if_fail (gobject != NULL, TRUE);
	      g_return_val_if_fail (GLE_IS_GWIDGET (gobject), TRUE);
	      
	      /* we temporarily adjust the popup menu for the gwidget
	       * gle_shell_popup_finish() cares about readjustment
	       */
	      shell->popup_data = gobject;
	      for (i = 0; i < POPUP_OP_LAST; i++)
		gle_item_factory_adjust (shell->popup_factory,
					 i,
					 gle_shell_popup_check (shell, GLE_GWIDGET (gobject), i),
					 NULL);
	      gtk_item_factory_popup_with_data (shell->popup_factory,
						shell,
						(GtkDestroyNotify) gle_shell_popup_finish,
						event->x_root,
						event->y_root,
						event->button,
						event->time);
	    }
	  else if (column == GLE_PLIST_COL_TAG)
	    {
	      GleGObject *gobject;
	      
	      gobject = gle_plist_get_gobject (shell->plist, row);
	      g_return_val_if_fail (gobject != NULL, TRUE);
	      g_return_val_if_fail (GLE_IS_GWIDGET (gobject), TRUE);

	      handled = TRUE;
	      
	      if (event->button == 1)
		{
		  if (GLE_GOBJECT_SHELL_OWNED (gobject))
		    {
		      if (GLE_NOTIFIER_CHECK_F (gobject, gle_bomb_shell))
			GLE_NOTIFIER_REMOVE_F (gobject, gle_bomb_shell);
		      else
			{
			  GLE_NOTIFIER_INSTALL (gobject, "pre_disassociate", gle_bomb_shell, NULL);
			  GLE_NOTIFIER_INSTALL (gobject, "pre_destroy", gle_bomb_shell, NULL);
			}
		      
		      gle_plist_update (shell->plist, gobject);
		    }
		}
	      else if (event->button == 2)
		{
		  if (GLE_GOBJECT_SHELL_OWNED (gobject))
		    gle_shell_give_up_gtoplevel (GLE_GWIDGET (gobject));
		  else
		    gle_shell_own_gtoplevel (GLE_GWIDGET (gobject));
		}
	    }
	}
    }
  
  if (handled)
    gtk_signal_emit_stop_by_name (GTK_OBJECT (shell->plist), "button-press-event");
  
  return handled;
}

static void
gle_shell_popup_operation (GleShell	  *shell,
			   guint	   popup_op,
			   GtkWidget	  *item)
{
  GList *gwidget_list, *list;
  
  g_return_if_fail (shell != NULL);
  g_return_if_fail (GLE_IS_SHELL (shell));
  g_return_if_fail (popup_op < POPUP_OP_LAST);
  
  if (shell == gtk_item_factory_popup_data_from_widget (item))
    {
      /* we are invoked from the popup menu */
      gle_gwidget_ref (shell->popup_data);
      gwidget_list = g_list_prepend (NULL, shell->popup_data);
    }
  else
    {
      /* we are invoked via accelerator, in this case
       * we group-operate on all selected gwidgets
       */
      
      gwidget_list = gle_plist_list_selected (shell->plist);
      for (list = gwidget_list; list; list = list->next)
	gle_gwidget_ref (list->data);
    }
  
  gtk_widget_ref (GTK_WIDGET (shell));
  
  for (list = gwidget_list; list; list = list->next)
    {
      GleGWidget *gwidget;
      GtkWidget *widget;
      
      gwidget = list->data;
      if (!gle_shell_popup_check (shell, gwidget, popup_op))
	{
	  gle_gwidget_unref (gwidget);
	  continue;
	}
      
      widget = GLE_GOBJECT_IS_INSTANTIATED (gwidget) ? GLE_GWIDGET_WIDGET (gwidget) : NULL;
      
      switch (popup_op)
	{
	  GtkWidget *dialog;
	  
	case POPUP_OP_NONE:
	  break;
	case POPUP_OP_P_TED:
	  if (GLE_IS_GCONTAINER (gwidget))
	    {
	      dialog = (GtkWidget*) gle_ted_from_gwidget (gwidget);
	      if (!dialog)
		dialog = gle_ted_new (gwidget);
	      gtk_widget_show (dialog);
	      gdk_window_raise (dialog->window);
	    }
	  break;
	case POPUP_OP_P_EDITOR:
	  dialog = (GtkWidget*) gle_editor_from_gobject (GLE_GOBJECT (gwidget));
	  if (!dialog)
	    dialog = gle_editor_new (GLE_GOBJECT (gwidget));
	  gtk_widget_show (dialog);
	  gdk_window_raise (dialog->window);
	  break;
	case POPUP_OP_P_INSTANTIATE:
	  gle_gtoplevel_instantiate (GLE_GTOPLEVEL (gwidget));
	  break;
	case POPUP_OP_P_DISASSOCIATE:
	  gle_gtoplevel_disassociate (GLE_GTOPLEVEL (gwidget));
	  break;
	case POPUP_OP_P_DUMP:
	  gle_dump_gwidget (stdout, "", FALSE,
			    gwidget,
			    GLE_DUMP_ALL_ARGS);
	  fputc ('\n', stdout);
	  break;
	case POPUP_OP_P_PP:
	  g_message ("(%s*)%p", gtk_type_name (GLE_GOBJECT_PROXY_TYPE (gwidget)), gwidget);
	  break;
	case POPUP_OP_P_DESTROY:
	  gle_gwidget_destroy (gwidget);
	  break;
	case POPUP_OP_W_HIDE:
	  gtk_widget_hide (widget);
	  break;
	case POPUP_OP_W_SHOW:
	  gtk_widget_show (widget);
	  break;
	case POPUP_OP_W_DESTROY:
	  gtk_widget_destroy (widget);
	  break;
	default:
	  fprintf (stderr, "GleShell: invalid popup operation: %u\n", popup_op);
	  break;
	}
      
      gle_gwidget_unref (gwidget);
    }
  g_list_free (gwidget_list);
  
  gtk_widget_unref (GTK_WIDGET (shell));
}

static gboolean
gle_shell_popup_check (GleShell	  *shell,
		       GleGWidget *gwidget,
		       guint	   pop)
{
  GtkWidget *widget;
  
  g_return_val_if_fail (gwidget != NULL, FALSE);
  g_return_val_if_fail (GLE_IS_GTOPLEVEL (gwidget), FALSE);
  
  if (GLE_GOBJECT_DESTROYED (gwidget))
    return FALSE;
  
  widget = GLE_GWIDGET_WIDGET (gwidget);
  
  switch (pop)
    {
    case POPUP_OP_P_TED:
      return gle_gwidget_is_shell_owned (gwidget);
    case POPUP_OP_P_EDITOR:
      return gle_gwidget_is_shell_owned (gwidget);
    case POPUP_OP_P_INSTANTIATE:
      return gle_gwidget_is_shell_owned (gwidget) && !GLE_GOBJECT_IS_INSTANTIATED (gwidget);
    case POPUP_OP_P_DISASSOCIATE:
      return gle_gwidget_is_shell_owned (gwidget) && GLE_GOBJECT_IS_INSTANTIATED (gwidget);
    case POPUP_OP_P_DUMP:
      return TRUE;
    case POPUP_OP_P_PP:
      return TRUE;
    case POPUP_OP_P_DESTROY:
      return gle_gwidget_is_shell_owned (gwidget);
    case POPUP_OP_W_HIDE:
      return widget && GTK_WIDGET_VISIBLE (widget);
    case POPUP_OP_W_SHOW:
      return widget && !GTK_WIDGET_VISIBLE (widget);
    case POPUP_OP_W_DESTROY:
      return widget != NULL;
    default:
      return FALSE;
    }
}

void
gle_shell_operation (GleShell	    *shell,
		     GleShellOps     shell_op)
{
  g_return_if_fail (shell != NULL);
  g_return_if_fail (GLE_IS_SHELL (shell));
  g_return_if_fail (shell_op < GLE_SHELL_OP_LAST);
  
  gtk_widget_ref (GTK_WIDGET (shell));
  
  switch (shell_op)
    {
    case    GLE_SHELL_OP_NONE:
      break;
    case  GLE_SHELL_OP_DELETE:
      gtk_widget_destroy (GTK_WIDGET (shell));
      break;
    case  GLE_SHELL_OP_SELECTOR:
      gle_shell_do_selection (shell);
      break;
    case  GLE_SHELL_OP_UPDATE:
      gle_plist_update_all (shell->plist);
      break;
    case  GLE_SHELL_OP_NEW_WINDOW:
      gle_shell_add_toplevel (shell, gtk_widget_new (GTK_TYPE_WINDOW,
						     "title", "User Window",
						     "visible", TRUE,
						     NULL));
      break;
    case  GLE_SHELL_OP_WPALETTE:
      if (!shell->wpalette)
	{
	  shell->wpalette =
	    gtk_widget_new (GLE_TYPE_WPALETTE,
			    "signal::destroy", gtk_widget_destroyed, &shell->wpalette,
			    NULL);
	}
      gtk_widget_show (shell->wpalette);
      gdk_window_raise (shell->wpalette->window);
      break;
    case  GLE_SHELL_OP_RC_PREF:
      if (!shell->rc_pref)
	{
	  shell->rc_pref = gle_rc_pref_dialog_new (gle_rc, "GLE-RcPreferences");
	  GLE_TAG (shell->rc_pref, "GLE-RcPreferences");
	  gtk_signal_connect (GTK_OBJECT (shell->rc_pref),
			      "destroy",
			      gtk_widget_destroyed,
			      &shell->rc_pref);
	}
      gtk_widget_show (shell->rc_pref);
      gdk_window_raise (shell->rc_pref->window);
      break;
    case  GLE_SHELL_OP_OPEN:
      if (!shell->file_open_dialog)
	{
	  shell->file_open_dialog = gle_file_dialog_new_open ("*.glr");
	  gtk_signal_connect (GTK_OBJECT (shell->file_open_dialog),
			      "destroy",
			      gtk_widget_destroyed,
			      &shell->file_open_dialog);
	}
      gtk_widget_show (shell->file_open_dialog);
      gdk_window_raise (shell->file_open_dialog->window);
      break;
    case  GLE_SHELL_OP_SAVE:
      if (!shell->file_save_dialog)
	{
	  shell->file_save_dialog = gle_file_dialog_new_save ("unnamed.glr");
	  gtk_signal_connect (GTK_OBJECT (shell->file_save_dialog),
			      "destroy",
			      gtk_widget_destroyed,
			      &shell->file_save_dialog);
	}
      gtk_widget_show (shell->file_save_dialog);
      gdk_window_raise (shell->file_save_dialog->window);
      break;
    case  GLE_SHELL_OP_HELP_INTRO:
      if (!shell->help_intro)
	{
	  shell->help_intro = gle_info_window ("GLE Introduction", gle_docus[0].text);
	  GLE_TAG (shell->help_intro, "GLE-Editor");
          gtk_signal_connect (GTK_OBJECT (shell->help_intro),
			      "destroy",
			      gtk_widget_destroyed,
			      &shell->help_intro);
	}
      gtk_widget_show (shell->help_intro);
      gdk_window_raise (shell->help_intro->window);
      break;
    case  GLE_SHELL_OP_HELP_ABOUT:
    default:
      fprintf (stderr, "GleShellOps: unimplemented %u\n", shell_op);
      break;
    }
  
  gtk_widget_unref (GTK_WIDGET (shell));
}

void
gle_shell_manage_object (GtkObject	*object)
{
  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_OBJECT (object));
  g_return_if_fail (GLE_HAS_TAG (object));
  
  if (!gle_shell)
    g_error ("No GleShell present");
  
  gtk_signal_connect_object_while_alive (GTK_OBJECT (gle_shell),
					 "destroy",
					 gtk_object_destroy,
					 object);
}
