/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include "buffer_flat.h"


#define DATA   (&flat_buffer->data)
#define VALID  (flat_buffer->valid)
#define VALID2 (flat_buffer->validating)




static void
gimp_flat_buffer_delete (GimpBuffer * buffer)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  d_uninit (DATA);
  gimp_buffer_uninit (buffer);
  g_free (flat_buffer);
}


static gboolean
gimp_flat_buffer_alloc (GimpBuffer   *buffer,
                        GimpArea     *area,
                        Alloc         how)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);
  gint n = buffer->_x.len * buffer->_y.len * buffer->_z.len;

  switch (how)
    {
    case ALLOC_ALLOC:
      g_return_val_if_fail (d_alloc (DATA, n) == TRUE, FALSE);
      return TRUE;

    case ALLOC_UNALLOC:
      g_return_val_if_fail (d_unalloc (DATA) == TRUE, FALSE);
      VALID = FALSE;
      return TRUE;

    case ALLOC_NONE:
      g_warning ("bad value");
      return FALSE;
    }

  return FALSE;
}


static gboolean
gimp_flat_buffer_validate (GimpBuffer  *buffer,
                           GimpArea    *area, 
                           Validate     how)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  switch (how)
    {
    case VALIDATE_VALIDATE:
      if (VALID == TRUE)
        return TRUE;

      if (! d_is_alloced (DATA))
        {
          if (gimp_flat_buffer_alloc (buffer, area,
                                      ALLOC_ALLOC) != TRUE)
            {
              g_warning ("flatbuffer autoalloc failed");
              return FALSE;
            }
        }

      g_return_val_if_fail (d_write (DATA) == TRUE, FALSE);
      if (buffer->vfunc != NULL)
        {
          if (VALID2 == TRUE)
            {
              g_warning ("recursive validate");
            }
          else
            {
              VALID2 = TRUE;
              buffer->vfunc (buffer, area, guchar_d (DATA));
              VALID2 = FALSE;
            }
        }
      else
        {
          memset (guchar_d (DATA), 128, (buffer->_x.len *
                                         buffer->_y.len *
                                         buffer->_z.len));
        }
      g_return_val_if_fail (d_release (DATA) == TRUE, FALSE);

      VALID = TRUE;
      return TRUE;


    case VALIDATE_INVALIDATE:
      g_return_val_if_fail (d_usecount (DATA) == 0, FALSE);
      VALID = FALSE;
      return TRUE;


    case VALIDATE_NONE:
      g_warning ("bad value");
      return FALSE;
    }

  return FALSE;
}



static gboolean
gimp_flat_buffer_use (GimpBuffer  *buffer,
                      GimpPortion *portion, 
                      Use          how)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  if ((VALID != TRUE) && ((how == USE_READ) ||
                          (how == USE_UPDATE) ||
                          (how == USE_WRITE)))
    {
      if (gimp_flat_buffer_validate (buffer, &portion->bounds,
                                     VALIDATE_VALIDATE) != TRUE)
        {
          g_warning ("flatbuffer autovalidate failed");
          return FALSE;
        }
    }

  switch (how)
    {
    case USE_READ:
      g_return_val_if_fail (d_read (DATA) == TRUE, FALSE);
      return TRUE;

    case USE_UPDATE:
      g_return_val_if_fail (d_update (DATA) == TRUE, FALSE);
      return TRUE;

    case USE_WRITE:
      g_return_val_if_fail (d_write (DATA) == TRUE, FALSE);
      return TRUE;

    case USE_RELEASE:
      g_return_val_if_fail (d_release (DATA) == TRUE, FALSE);
      return TRUE;

    case USE_NONE:
      g_warning ("bad value");
      return FALSE;
    }

  return FALSE;
}


static gboolean
gimp_flat_buffer_query (GimpBuffer    *buffer,
                        GimpPortion   *portion,
                        GimpMemStatus *status)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  status->alloced   = (d_is_alloced (DATA) ? TRUE : FALSE);
  status->valid     = VALID;
  status->usecount  = (d_usecount (DATA));

  return TRUE;
}


static gboolean
gimp_flat_buffer_data (GimpBuffer      *buffer,
                       GimpPortion     *portion,
                       GimpPixelArray  *array)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  /* describe the memory layout */
  array->tag       = buffer->_z.tag;
  array->width     = MIN (buffer->_x.len,
                          portion->focus.b.x) - portion->focus.a.x;
  array->height    = MIN (buffer->_y.len,
                          portion->focus.b.y) - portion->focus.a.y;
  array->pixstride = buffer->_z.len;
  array->rowstride = buffer->_z.len * buffer->_x.len;

  /* get a pointer to the memory */
  array->data = DATA;

  /* and save an offset to the first pixel */
  array->offset = (portion->focus.a.y * array->rowstride) +
                  (portion->focus.a.x * array->pixstride);

  return TRUE;
}


static GimpFlatBufferClass my_class =
{
  {
    BUFFER_FLAT,
    gimp_flat_buffer_delete,
    gimp_flat_buffer_alloc,
    NULL, /* map */
    gimp_flat_buffer_validate,
    gimp_flat_buffer_use,
    gimp_flat_buffer_query,
    gimp_flat_buffer_data
  }
};






GimpFlatBuffer *
gimp_flat_buffer_new (Tag   tag,
                      gint  width,
                      gint  height)
{
  GimpFlatBuffer * flat_buffer;

  flat_buffer = g_new (GimpFlatBuffer, 1);

  gimp_buffer_init (GIMP_BUFFER (flat_buffer), tag,
                    width, width, 0,
                    height, height, 0);

  d_init (DATA);
  VALID = FALSE;
  VALID2 = FALSE;

  GIMP_BUFFER (flat_buffer)->klass = (void*) &my_class;

  return flat_buffer;
}


