/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifndef __BUFFER_H__
#define __BUFFER_H__

#include "cow.h"


#define GIMP_BUFFER(b)        ((GimpBuffer*)(b))
#define GIMP_BUFFER_CLASS(b)  ((GimpBufferClass*)(b))

/* need to reconcile Tag with ColorModel */
typedef guint Tag;
#define tag_valid(t) TRUE
#define tag_bytes(t) (t)



typedef struct _GimpBuffer GimpBuffer;
typedef struct _GimpBufferClass GimpBufferClass;
typedef struct _GimpPoint GimpPoint;
typedef struct _GimpArea GimpArea;
typedef struct _GimpPortion GimpPortion;
typedef struct _GimpPixelArray GimpPixelArray;
typedef struct _GimpMemStatus GimpMemStatus;



typedef enum
{
  BUFFER_NONE,
  BUFFER_FLAT,
  BUFFER_TILED,
  BUFFER_SHM,
  BUFFER_PTR
} BufferType;


typedef enum
{
  ALLOC_NONE,
  ALLOC_ALLOC,
  ALLOC_UNALLOC
} Alloc;


typedef enum
{
  VALIDATE_NONE,
  VALIDATE_VALIDATE,
  VALIDATE_INVALIDATE
} Validate;


typedef enum
{
  USE_NONE,
  USE_READ,
  USE_WRITE,
  USE_UPDATE,
  USE_RELEASE
} Use;



/* ----------------------------------------------------------- */

struct _GimpPoint {
  gint x;
  gint y;
};



/* ----------------------------------------------------------- */

struct _GimpArea {
  GimpPoint a;
  GimpPoint b;
};

#define ga_width(ga)  ((ga)->b.x - (ga)->a.x)
#define ga_height(ga) ((ga)->b.y - (ga)->a.y)


/* ----------------------------------------------------------- */

struct _GimpPortion
{
  /* the original area passed to gimp_buffer_focus */
  GimpArea orig;

  /* shrunk to whole tiles */
  GimpArea min;
  
  /* expanded to whole tiles */
  GimpArea max;
  
  /* clipped to single tile */
  GimpArea focus;

  /* clipped to single tile and expanded to whole tile */
  GimpArea bounds;
};

#define gp_orig(gp)    (&(gp)->orig)
#define gp_min(gp)     (&(gp)->min)
#define gp_max(gp)     (&(gp)->max)
#define gp_focus(gp)   (&(gp)->focus)
#define gp_bounds(gp)  (&(gp)->bounds)



/* ----------------------------------------------------------- */

struct _GimpPixelArray
{
  /* the layout of each element */
  Tag tag;

  /* the array bounds */
  gint width;
  gint height;

  /* the element spacing */
  gint pixstride;
  gint rowstride;

  /* reference to first element */
  GimpMemPtr * data;
  gint offset;
};

#define gpa_tag(gpa)         ((gpa)->tag)
#define gpa_width(gpa)       ((gpa)->width)
#define gpa_height(gpa)      ((gpa)->height)
#define gpa_pixstride(gpa)   ((gpa)->pixstride)
#define gpa_rowstride(gpa)   ((gpa)->rowstride)
#define gpa_data(gpa)        ((gpa)->data)
#define gpa_offset(gpa)      ((gpa)->offset)

#define gpa_dataptr(gpa)     ((void*) \
                              (guchar_d((gpa)->data) + \
                               (gpa)->offset))




/* ----------------------------------------------------------- */

struct _GimpMemStatus
{
  /* does the portion have memory allocated? */
  gboolean alloced;
  
  /* does the memory contain valid data? */
  gboolean valid;

  /* how many places are using it? */
  gint usecount;
};



/* ----------------------------------------------------------- */

/* custom validator.  the void ptr is the raw memory located inside
   area in the buffer */
typedef void (*GimpValidator) (GimpBuffer *, GimpArea *, void *);


/* the member vars have funny names to avoid surprises from the
   TileManager typedef */
struct _GimpBuffer
{
  /* the class object */
  void * klass;

  /* the xy dimensions */
  struct {
    
    /* size of image data in pixels */
    gint len;
    
    /* size of tile */
    gint period;

    /* offset of image data into first tile */
    gint offset;

  } _x, _y;    

  /* the z dimension */
  struct {
    
    /* size in bytes */
    gint len;

    /* pixel layout. merge with jays ColorModel concept */
    Tag tag;

  } _z;
  
  /* custom validator */
  GimpValidator vfunc;
  void * vdata;
  
  /* temporary, gimp expects buffers to have a position */
  gint qq_x;
  gint qq_y;
};


struct _GimpBufferClass
{
  /* subclass type */
  BufferType type;

  /* virtual method table */
  void      (* delete)  (GimpBuffer        *buffer);
  gboolean  (* alloc)   (GimpBuffer        *buffer,
                         GimpArea          *area,
                         Alloc              how);
  gboolean  (* map)     (GimpBuffer        *dst,
                         GimpArea          *darea,
                         GimpBuffer        *src,
                         GimpArea          *sarea);
  gboolean  (* validate)(GimpBuffer        *buffer,
                         GimpArea          *area,
                         Validate           how);
  gboolean  (* use)     (GimpBuffer        *buffer,
                         GimpPortion       *portion, 
                         Use                how);
  gboolean  (* query)   (GimpBuffer        *buffer,
                         GimpPortion       *portion,
                         GimpMemStatus     *status);
  gboolean  (* data)    (GimpBuffer        *buffer,
                         GimpPortion       *portion,
                         GimpPixelArray    *array);
};


gint        gimp_buffer_width    (GimpBuffer        *buffer);
gint        gimp_buffer_height   (GimpBuffer        *buffer);
gint        gimp_buffer_depth    (GimpBuffer        *buffer);

gint        gimp_buffer_tilenum  (GimpBuffer        *buffer,
                                  GimpPoint         *point);

void        gimp_buffer_delete   (GimpBuffer        *buffer);

gboolean    gimp_buffer_focus    (GimpBuffer        *buffer,
                                  GimpPortion       *portion,
                                  GimpArea          *roi);

gboolean    gimp_buffer_alloc    (GimpBuffer        *buffer,
                                  GimpArea          *roi,
                                  Alloc              how);

gboolean    gimp_buffer_map      (GimpBuffer        *dst,
                                  GimpArea          *darea,
                                  GimpBuffer        *src,
                                  GimpArea          *sarea);

gboolean    gimp_buffer_validate (GimpBuffer        *buffer,
                                  GimpArea          *area,
                                  Validate           how);

gboolean    gimp_buffer_use      (GimpBuffer        *buffer,
                                  GimpPortion       *portion,
                                  Use                how);

gboolean    gimp_buffer_query    (GimpBuffer        *buffer,
                                  GimpPortion       *portion,
                                  GimpMemStatus     *status);

gboolean    gimp_buffer_data     (GimpBuffer        *buffer,
                                  GimpPortion       *portion,
                                  GimpPixelArray    *array);

/* for use by subclasses */
void        gimp_buffer_init     (GimpBuffer *buffer,
                                  Tag         tag,
                                  gint        width,
                                  gint        height,
                                  gint        x_period,
                                  gint        x_offset,
                                  gint        y_period,
                                  gint        y_offset);

void        gimp_buffer_uninit   (GimpBuffer *buffer);


#endif /* __CANVAS_H__ */
