static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * Copyright 2010 Danny Robson <danny@blubinc.net>                            \n"
" * Based off 2006 Anat Levin                                                  \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"                                                                              \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"property_int   (epsilon, _(\"Epsilon\"), -6)                                  \n"
"   description (_(\"Log of the error weighting\"))                            \n"
"   value_range (-9, -1)                                                       \n"
"                                                                              \n"
"property_int    (radius, _(\"Radius\"), 1)                                    \n"
"   description  (_(\"Radius of the processing window\"))                      \n"
"   value_range  (1, 3)                                                        \n"
"                                                                              \n"
"property_double (threshold, _(\"Threshold\"), 0.02)                           \n"
"   description  (_(\"Alpha threshold for multilevel processing\"))            \n"
"   value_range  (0.0, 0.1)                                                    \n"
"                                                                              \n"
"property_double (lambda, _(\"Lambda\"), 100.0)                                \n"
"   value_range  (0.0, 100.0)                                                  \n"
"   description  (_(\"Trimap influence factor\"))                              \n"
"                                                                              \n"
"property_int    (levels, _(\"Levels\"), 4)                                    \n"
"   description  (_(\"Number of downsampled levels to use\"))                  \n"
"   value_range  (0, 8)                                                        \n"
"                                                                              \n"
"property_int    (active_levels, _(\"Active levels\"), 2)                      \n"
"   description  (_(\"Number of levels to perform solving\"))                  \n"
"   value_range  (0, 8)                                                        \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_COMPOSER                                                      \n"
"#define GEGL_OP_NAME matting_levin                                            \n"
"#define GEGL_OP_C_SOURCE matting-levin.c                                      \n"
"                                                                              \n"
"#include \"gegl-op.h\"                                                        \n"
"#include \"gegl-debug.h\"                                                     \n"
"                                                                              \n"
"#include <stdlib.h>                                                           \n"
"#include <stdio.h>                                                            \n"
"#include <math.h>                                                             \n"
"                                                                              \n"
"/* XXX: We have two options for the two common installation locations of      \n"
" * UMFPACK. Ideally this would be sorted out purely in autoconf; see          \n"
" * configure.ac for the issues.                                               \n"
" */                                                                           \n"
"#if defined(HAVE_UMFPACK_H)                                                   \n"
"#include <umfpack.h>                                                          \n"
"#elif defined (HAVE_SUITESPARSE_UMFPACK_H)                                    \n"
"#include <suitesparse/umfpack.h>                                              \n"
"#endif                                                                        \n"
"                                                                              \n"
"#include \"matting-levin-cblas.h\"                                            \n"
"                                                                              \n"
"                                                                              \n"
"/* We don't use the babl_format_get_n_components function for these values,   \n"
" * as literal constants can be used for stack allocation of array sizes. They \n"
" * are double checked in matting_process.                                     \n"
" */                                                                           \n"
"#define COMPONENTS_AUX    2                                                   \n"
"#define COMPONENTS_INPUT  3                                                   \n"
"#define COMPONENTS_OUTPUT 1                                                   \n"
"#define COMPONENTS_COEFF  4                                                   \n"
"                                                                              \n"
"#define CONVOLVE_RADIUS   2                                                   \n"
"#define CONVOLVE_LEN     ((CONVOLVE_RADIUS * 2) + 1)                          \n"
"                                                                              \n"
"                                                                              \n"
"/* A simple structure holding a compressed column sparse matrix. Data fields  \n"
" * correspond directly to the expected format used by UMFPACK. This restricts \n"
" * us to using square matrices.                                               \n"
" */                                                                           \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  guint    elems,                                                             \n"
"           columns,                                                           \n"
"           rows;                                                              \n"
"  glong   *col_idx,                                                           \n"
"          *row_idx;                                                           \n"
"  gdouble *values;                                                            \n"
"} sparse_t;                                                                   \n"
"                                                                              \n"
"                                                                              \n"
"/* All channels use double precision. Despite it being overly precise, slower,\n"
" * and larger; it's much more convenient:                                     \n"
" *   - Input R'G'B' needs to be converted into doubles later when calculating \n"
" *     the matting laplacian, as the extra precision is actually useful here, \n"
" *     and UMFPACK requires doubles.                                          \n"
" *   - AUX Y' is easier to use as a double when dealing with the matting      \n"
" *     laplacian which is already in doubles.                                 \n"
" */                                                                           \n"
"                                                                              \n"
"static const gchar *FORMAT_AUX    = \"Y'A double\";                           \n"
"static const gchar *FORMAT_INPUT  = \"R'G'B' double\";                        \n"
"static const gchar *FORMAT_OUTPUT = \"Y' double\";                            \n"
"                                                                              \n"
"static const guint   AUX_VALUE = 0;                                           \n"
"static const guint   AUX_ALPHA = 1;                                           \n"
"static const gdouble AUX_BACKGROUND = 0.1;                                    \n"
"static const gdouble AUX_FOREGROUND = 0.9;                                    \n"
"                                                                              \n"
"                                                                              \n"
"/* Threshold below which we consider the Y channel to be undefined (or        \n"
" * masked). This is a binary specification, either fully masked or fully      \n"
" * defined.                                                                   \n"
" */                                                                           \n"
"static const gdouble TRIMAP_ALPHA_THRESHOLD = 0.01;                           \n"
"                                                                              \n"
"                                                                              \n"
"/* The smallest dimension of a buffer which we allow during downsampling.     \n"
" * This must allow sufficient room for CONVOLVE_RADIUS convolutions to work   \n"
" * usefully.                                                                  \n"
" */                                                                           \n"
"static const gint MIN_LEVEL_DIAMETER = 30;                                    \n"
"                                                                              \n"
"                                                                              \n"
"/* Round upwards with performing `x / y' */                                   \n"
"static guint                                                                  \n"
"ceil_div (gint x, gint y)                                                     \n"
"{                                                                             \n"
"  return (x + y - 1) / y;                                                     \n"
"}                                                                             \n"
"                                                                              \n"
"/* Perform a floating point comparison, returning true if the values are      \n"
" * within the percentage tolerance specified in FLOAT_TOLERANCE. Note: this   \n"
" * is different to GEGL_FLOAT_EQUAL which specifies an absolute delta. This   \n"
" * won't work with very small values, however our approach can slower.        \n"
" */                                                                           \n"
"static gboolean                                                               \n"
"float_cmp (gfloat a, gfloat b)                                                \n"
"{                                                                             \n"
"  static const gfloat FLOAT_TOLERANCE = 0.0001f;                              \n"
"                                                                              \n"
"  return (a - b) <= FLOAT_TOLERANCE * fabsf (a) ||                            \n"
"         (a - b) <= FLOAT_TOLERANCE * fabsf (b);                              \n"
"}                                                                             \n"
"                                                                              \n"
"/* Return the offset for the integer coordinates (X, Y), in surface of        \n"
" * dimensions R, which has C channels. Does not take into account the channel \n"
" * width, so should be used for indexing into properly typed arrays/pointers. \n"
" *                                                                            \n"
" * Quite expensive without inlining (~15% runtime).                           \n"
" */                                                                           \n"
"static inline off_t                                                           \n"
"offset (gint                 x,                                               \n"
"        gint                 y,                                               \n"
"        const GeglRectangle *restrict roi,                                    \n"
"        gint                 components)                                      \n"
"{                                                                             \n"
"  return (x + y * roi->width) * components;                                   \n"
"}                                                                             \n"
"                                                                              \n"
"/* Similar to `offset', inlining buys us a good speedup (though is less       \n"
" * frequently used than the general purpose `offset').                        \n"
" */                                                                           \n"
"static inline gboolean                                                        \n"
"trimap_masked (const gdouble       *restrict trimap,                          \n"
"               gint                 x,                                        \n"
"               gint                 y,                                        \n"
"               const GeglRectangle *restrict roi)                             \n"
"{                                                                             \n"
"  gdouble value = trimap[offset (x, y, roi, COMPONENTS_AUX) + AUX_ALPHA];     \n"
"  return  value < TRIMAP_ALPHA_THRESHOLD;                                     \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static const char*                                                            \n"
"matting_umf_error_to_string (guint err)                                       \n"
"{                                                                             \n"
"                                                                              \n"
"  switch (err)                                                                \n"
"    {                                                                         \n"
"      case UMFPACK_OK:                                                        \n"
"          return \"UMFPACK_OK\";                                              \n"
"      case UMFPACK_WARNING_singular_matrix:                                   \n"
"          return \"UMFPACK_WARNING_singular_matrix\";                         \n"
"      case UMFPACK_WARNING_determinant_underflow:                             \n"
"          return \"UMFPACK_WARNING_determinant_underflow\";                   \n"
"      case UMFPACK_WARNING_determinant_overflow:                              \n"
"          return \"UMFPACK_WARNING_determinant_overflow\";                    \n"
"      case UMFPACK_ERROR_out_of_memory:                                       \n"
"          return \"UMFPACK_ERROR_out_of_memory\";                             \n"
"      case UMFPACK_ERROR_invalid_Numeric_object:                              \n"
"          return \"UMFPACK_ERROR_invalid_Numeric_object\";                    \n"
"      case UMFPACK_ERROR_invalid_Symbolic_object:                             \n"
"          return \"UMFPACK_ERROR_invalid_Symbolic_object\";                   \n"
"      case UMFPACK_ERROR_argument_missing:                                    \n"
"          return \"UMFPACK_ERROR_argument_missing\";                          \n"
"      case UMFPACK_ERROR_n_nonpositive:                                       \n"
"          return \"UMFPACK_ERROR_n_nonpositive\";                             \n"
"      case UMFPACK_ERROR_invalid_matrix:                                      \n"
"          return \"UMFPACK_ERROR_invalid_matrix\";                            \n"
"      case UMFPACK_ERROR_different_pattern:                                   \n"
"          return \"UMFPACK_ERROR_different_pattern\";                         \n"
"      case UMFPACK_ERROR_invalid_system:                                      \n"
"          return \"UMFPACK_ERROR_invalid_system\";                            \n"
"      case UMFPACK_ERROR_invalid_permutation:                                 \n"
"          return \"UMFPACK_ERROR_invalid_permutation\";                       \n"
"      case UMFPACK_ERROR_internal_error:                                      \n"
"          return \"UMFPACK_ERROR_internal_error\";                            \n"
"      case UMFPACK_ERROR_file_IO:                                             \n"
"          return \"UMFPACK_ERROR_file_IO\";                                   \n"
"                                                                              \n"
"      default:                                                                \n"
"          g_return_val_if_reached (\"Unknown UMFPACK error\");                \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"matting_prepare (GeglOperation *operation)                                    \n"
"{                                                                             \n"
"  gegl_operation_set_format (operation, \"input\",  babl_format (FORMAT_INPUT));\n"
"  gegl_operation_set_format (operation, \"aux\",    babl_format (FORMAT_AUX));\n"
"  gegl_operation_set_format (operation, \"output\", babl_format (FORMAT_OUTPUT));\n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"matting_get_required_for_output (GeglOperation       *operation,              \n"
"                                 const gchar         *input_pad,              \n"
"                                 const GeglRectangle *roi)                    \n"
"{                                                                             \n"
"  GeglRectangle result = *gegl_operation_source_get_bounding_box (operation,  \n"
"                                                                  \"input\"); \n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"matting_get_cached_region (GeglOperation * operation,                         \n"
"                           const GeglRectangle * roi)                         \n"
"{                                                                             \n"
"  return *gegl_operation_source_get_bounding_box (operation, \"input\");      \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* An element-wise subtraction on two 3x3 matrices. */                        \n"
"static void                                                                   \n"
"matting_matrix3_matrix3_sub (gdouble _in1[3][3],                              \n"
"                             gdouble _in2[3][3],                              \n"
"                             gdouble _out[3][3])                              \n"
"{                                                                             \n"
"  const gdouble *in1 = (gdouble*)_in1,                                        \n"
"                *in2 = (gdouble*)_in2;                                        \n"
"        gdouble *out = (gdouble*)_out;                                        \n"
"  guint i;                                                                    \n"
"                                                                              \n"
"  for (i = 0; i < 3 * 3; ++i)                                                 \n"
"    out[i] = in1[i] - in2[i];                                                 \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* An element-wise division on one 3x3 matrix, by one scalar */               \n"
"static void                                                                   \n"
"matting_matrix3_scalar_div (gdouble  _in[3][3],                               \n"
"                            gdouble        arg,                               \n"
"                            gdouble _out[3][3])                               \n"
"{                                                                             \n"
"  const gdouble *in  = (gdouble*)_in;                                         \n"
"        gdouble *out = (gdouble*)_out;                                        \n"
"  guint i;                                                                    \n"
"                                                                              \n"
"  for (i = 0; i < 3 * 3; ++i)                                                 \n"
"    out[i] = in[i] / arg;                                                     \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Shortcut for a 3x3 matrix inversion. Assumes the matrix is stored in row   \n"
" * major format. Parameters 'in' and 'out' must not overlap. The output       \n"
" * matrix may be overwritten on error. Returns TRUE if an inversion exists.   \n"
" *                                                                            \n"
" * If the matrix consists of column vectors, A = (v_0, v_1, v_2)              \n"
" *                                                                            \n"
" *           1   / (v_1 x v_2)' \\                                            \n"
" * inv(A) = ___  | (v_2 x v_0)' |                                             \n"
" *          det  \\ (v_0 x v_1)' /                                            \n"
" *                                                                            \n"
" */                                                                           \n"
"static gboolean                                                               \n"
"matting_matrix3_inverse (gdouble  in[3][3],                                   \n"
"                         gdouble out[3][3])                                   \n"
"{                                                                             \n"
"  gdouble determinant;                                                        \n"
"                                                                              \n"
"  /* Save the column vector cross products straight into the output matrix */ \n"
"  out[0][0] = in[1][1] * in[2][2] - in[1][2] * in[2][1];                      \n"
"  out[1][0] = in[1][2] * in[2][0] - in[1][0] * in[2][2];                      \n"
"  out[2][0] = in[1][0] * in[2][1] - in[1][1] * in[2][0];                      \n"
"                                                                              \n"
"  out[0][1] = in[2][1] * in[0][2] - in[2][2] * in[0][1];                      \n"
"  out[1][1] = in[2][2] * in[0][0] - in[2][0] * in[0][2];                      \n"
"  out[2][1] = in[2][0] * in[0][1] - in[2][1] * in[0][0];                      \n"
"                                                                              \n"
"  out[0][2] = in[0][1] * in[1][2] - in[0][2] * in[1][1];                      \n"
"  out[1][2] = in[0][2] * in[1][0] - in[0][0] * in[1][2];                      \n"
"  out[2][2] = in[0][0] * in[1][1] - in[0][1] * in[1][0];                      \n"
"                                                                              \n"
"  /* For a 3x3 matrix, det = v_0 . (v_1 x v_2)                                \n"
"   * We use the cross product that was previously stored into row zero of the \n"
"   * output matrix.                                                           \n"
"   */                                                                         \n"
"                                                                              \n"
"  determinant = in[0][0] * out[0][0] +                                        \n"
"                in[0][1] * out[1][0] +                                        \n"
"                in[0][2] * out[2][0];                                         \n"
"  if (determinant == 0)                                                       \n"
"    return FALSE;                                                             \n"
"                                                                              \n"
"  /* Scale the output by the determinant*/                                    \n"
"  matting_matrix3_scalar_div (out, determinant, out);                         \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Expanded form for 4x4 matrix inversion, derived from adjugate matrix and   \n"
" * division by determinant. Extensively uses values of 2x2 submatrix          \n"
" * determinants.                                                              \n"
" *                                                                            \n"
" * Implementation based on David Eberly's article `The Laplace Expansion      \n"
" * Theorem: Computing the Determinants and Inverses of Matrices'.             \n"
" *                                                                            \n"
" * Input and output are in row-major format. Input and output must not        \n"
" * overlap. Output will not be altered on error. Returns TRUE if the inverse  \n"
" * exists.                                                                    \n"
" */                                                                           \n"
"static gboolean                                                               \n"
"matting_matrix4_inverse (gdouble  in[4][4],                                   \n"
"                         gdouble out[4][4])                                   \n"
"{                                                                             \n"
"  gdouble s[6], c[6];                                                         \n"
"  gdouble det;                                                                \n"
"                                                                              \n"
"  s[0] = in[0][0] * in[1][1] - in[1][0] * in[1][0];                           \n"
"  s[1] = in[0][0] * in[2][1] - in[2][0] * in[0][1];                           \n"
"  s[2] = in[0][0] * in[3][1] - in[3][0] * in[0][1];                           \n"
"  s[3] = in[1][0] * in[2][1] - in[2][0] * in[1][1];                           \n"
"  s[4] = in[1][0] * in[3][1] - in[3][0] * in[1][1];                           \n"
"  s[5] = in[2][0] * in[3][1] - in[3][0] * in[2][1];                           \n"
"                                                                              \n"
"  c[5] = in[2][2] * in[3][3] - in[3][2] * in[2][3];                           \n"
"  c[4] = in[1][2] * in[3][3] - in[3][2] * in[1][3];                           \n"
"  c[3] = in[1][2] * in[2][3] - in[2][2] * in[1][3];                           \n"
"  c[2] = in[0][2] * in[3][3] - in[3][2] * in[0][3];                           \n"
"  c[1] = in[0][2] * in[2][3] - in[2][2] * in[0][3];                           \n"
"  c[0] = in[0][2] * in[1][3] - in[1][2] * in[0][3];                           \n"
"                                                                              \n"
"  det = s[0] * c[5] -                                                         \n"
"        s[1] * c[4] +                                                         \n"
"        s[2] * c[3] +                                                         \n"
"        s[3] * c[2] -                                                         \n"
"        s[4] * c[1] +                                                         \n"
"        s[5] * c[0];                                                          \n"
"                                                                              \n"
"  /* The determinant can be extremely small in real cases (eg, 1e-15). So     \n"
"   * existing checks like GEGL_FLOAT_IS_ZERO are no-where near precise enough \n"
"   * in the general case.                                                     \n"
"   * Most of the time we assume there is an inverse, so the lack of precision \n"
"   * in here isn't a dealbreaker, and we just compare against an actual zero  \n"
"   * to avoid divide-by-zero errors.                                          \n"
"   */                                                                         \n"
"  if (det == 0.0)                                                             \n"
"    return FALSE;                                                             \n"
"  det = 1.0 / det;                                                            \n"
"                                                                              \n"
"  out[0][0] = (  in[1][1] * c[5] - in[2][1] * c[4] + in[3][1] * c[3]) * det;  \n"
"  out[0][1] = ( -in[1][0] * c[5] + in[2][0] * c[4] - in[3][0] * c[3]) * det;  \n"
"  out[0][2] = (  in[1][3] * s[5] - in[2][3] * s[4] + in[3][3] * s[3]) * det;  \n"
"  out[0][3] = ( -in[1][2] * s[5] + in[2][2] * s[4] - in[3][2] * s[3]) * det;  \n"
"                                                                              \n"
"  out[1][0] = ( -in[0][1] * c[5] + in[2][1] * c[2] - in[3][1] * c[1]) * det;  \n"
"  out[1][1] = (  in[0][0] * c[5] - in[2][0] * c[2] + in[3][0] * c[1]) * det;  \n"
"  out[1][2] = ( -in[0][3] * s[5] + in[2][3] * s[2] - in[3][3] * s[1]) * det;  \n"
"  out[1][3] = (  in[0][2] * s[5] - in[2][2] * s[2] + in[3][2] * s[1]) * det;  \n"
"                                                                              \n"
"  out[2][0] = (  in[0][1] * c[4] - in[1][1] * c[2] + in[3][1] * c[0]) * det;  \n"
"  out[2][1] = ( -in[0][0] * c[4] + in[1][0] * c[2] - in[3][0] * c[0]) * det;  \n"
"  out[2][2] = (  in[0][3] * s[4] - in[1][3] * s[2] + in[3][3] * s[0]) * det;  \n"
"  out[2][3] = ( -in[0][2] * s[4] + in[1][2] * s[2] - in[3][2] * s[0]) * det;  \n"
"                                                                              \n"
"  out[3][0] = ( -in[0][1] * c[3] + in[1][1] * c[1] - in[2][1] * c[0]) * det;  \n"
"  out[3][1] = (  in[0][0] * c[3] - in[1][0] * c[1] + in[2][0] * c[0]) * det;  \n"
"  out[3][2] = ( -in[0][3] * s[3] + in[1][3] * s[1] - in[2][3] * s[0]) * det;  \n"
"  out[3][3] = (  in[0][2] * s[3] - in[1][2] * s[1] + in[2][2] * s[0]) * det;  \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Takes a vector and multiplies by its transpose to form a matrix in row     \n"
" * major format.                                                              \n"
" */                                                                           \n"
"static void                                                                   \n"
"matting_vector3_self_product (gdouble in[3],                                  \n"
"                              gdouble out[3][3])                              \n"
"{                                                                             \n"
"  out[0][0] = in[0] * in[0];                                                  \n"
"  out[1][0] = in[0] * in[1];                                                  \n"
"  out[2][0] = in[0] * in[2];                                                  \n"
"                                                                              \n"
"  out[0][1] = in[1] * in[0];                                                  \n"
"  out[1][1] = in[1] * in[1];                                                  \n"
"  out[2][1] = in[1] * in[2];                                                  \n"
"                                                                              \n"
"  out[0][2] = in[2] * in[0];                                                  \n"
"  out[1][2] = in[2] * in[1];                                                  \n"
"  out[2][2] = in[2] * in[2];                                                  \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Perform an erosion on the last component of `pixels'. If all neighbour     \n"
" * pixels are greater than low and lesser than 1 - high, keep the pixel       \n"
" * value, otherwise set it to NAN.                                            \n"
" *                                                                            \n"
" * Note, the condition is NOT low < pixel < high. Setting high to negative    \n"
" * expands the non-masking range.                                             \n"
" * XXX: This could probably be done with seperable passes, however there are  \n"
" * more immediate performance bottlenecks.                                    \n"
" */                                                                           \n"
"static gdouble *                                                              \n"
"matting_erode_range (const gdouble       *restrict pixels,                    \n"
"                     const GeglRectangle *restrict region,                    \n"
"                     guint                components,                         \n"
"                     guint                radius,                             \n"
"                     gdouble              low,                                \n"
"                     gdouble              high)                               \n"
"{                                                                             \n"
"  gdouble *new_pixels;                                                        \n"
"  guint    x, y, i, j,                                                        \n"
"           diameter = radius * 2 + 1;                                         \n"
"                                                                              \n"
"  new_pixels = g_new0 (gdouble, region->width * region->height);              \n"
"  for (y = radius; y < region->height - radius; ++y)                          \n"
"    {                                                                         \n"
"      for (x = radius; x < region->width - radius; ++x)                       \n"
"        {                                                                     \n"
"          gdouble home = pixels[offset (x, y,                                 \n"
"                                        region,                               \n"
"                                        components) + components - 1],        \n"
"                  value;                                                      \n"
"          if (home == 0.0)                                                    \n"
"            continue;                                                         \n"
"                                                                              \n"
"          if (home < 0.0 + low || home > 1.00 - high)                         \n"
"            goto masked;                                                      \n"
"                                                                              \n"
"          for (i = 0; i < diameter; ++i)                                      \n"
"            {                                                                 \n"
"              for (j = 0; j < diameter; ++j)                                  \n"
"                {                                                             \n"
"                  value = pixels[offset (x - radius + i,                      \n"
"                                         y - radius + j,                      \n"
"                                         region,                              \n"
"                                         components) + components - 1];       \n"
"                  if (value < low || value > 1.0 - high)                      \n"
"                    goto masked;                                              \n"
"                }                                                             \n"
"            }                                                                 \n"
"                                                                              \n"
"          new_pixels[offset (x, y, region, 1)] = home;                        \n"
"          continue;                                                           \n"
"  masked:                                                                     \n"
"          new_pixels[offset (x, y, region, 1)] = NAN;                         \n"
"        }                                                                     \n"
"    }                                                                         \n"
"  return new_pixels;                                                          \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Fill the borders of an image with the pixels from the first row/column     \n"
" * outside of `radius'. Does not expand the image. Operates in place.         \n"
" */                                                                           \n"
"static void                                                                   \n"
"matting_fill_borders (gdouble             *restrict image,                    \n"
"                      const GeglRectangle *restrict region,                   \n"
"                      const gint           components,                        \n"
"                      const gint           radius)                            \n"
"{                                                                             \n"
"  gint x, y, c;                                                               \n"
"                                                                              \n"
"  g_return_if_fail (image  != NULL);                                          \n"
"  g_return_if_fail (region != NULL);                                          \n"
"  g_return_if_fail (components > 0);                                          \n"
"  g_return_if_fail (radius     > 0);                                          \n"
"                                                                              \n"
"  /* Radius shouldn't be greater than the region radius. */                   \n"
"  g_return_if_fail (radius < region->width  / 2);                             \n"
"  g_return_if_fail (radius < region->height / 2);                             \n"
"                                                                              \n"
"  /* Extend the edges of the convolution outwards */                          \n"
"  for (y = 0; y <= radius; ++y)                                               \n"
"    {                                                                         \n"
"      /* Copy the first convolved line into the top `radius' rows */          \n"
"      memcpy (&image[offset (0,      y, region, components)],                 \n"
"              &image[offset (0, radius + 1, region, components)],             \n"
"              region->width * sizeof (image[0]) * components);                \n"
"      /* Copy the last convolved line into the last `radius' rows */          \n"
"      memcpy (&image[offset (0, region->height -      y - 1, region, components)],\n"
"              &image[offset (0, region->height - radius - 2, region, components)],\n"
"              region->width * sizeof (image[0]) * components);                \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (y = radius; y < region->height - radius; ++y)                          \n"
"    {                                                                         \n"
"      for (x = 0; x <= radius; ++x)                                           \n"
"        {                                                                     \n"
"          for (c = 0; c < components; ++c)                                    \n"
"            {                                                                 \n"
"              image[offset (x, y, region, components) + c] =                  \n"
"                  image[offset (radius + 1, y, region, components) + c];      \n"
"              image[offset (region->width - x - 1, y, region, components) + c] =\n"
"                  image[offset (region->width - radius - 2, y, region, components) + c];\n"
"            }                                                                 \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Calculate the coefficients needed to upsample a previously computed output \n"
" * alpha map. Returns a surface of 4*doubles which correspond to:             \n"
" *    red * out[0] + green * out[1] + blue * out[2] + out[3]                  \n"
" */                                                                           \n"
"static gdouble *                                                              \n"
"matting_get_linear_coefficients (const gdouble       *restrict image,         \n"
"                                 const gdouble       *restrict alpha,         \n"
"                                 const GeglRectangle *restrict rect,          \n"
"                                 const gdouble        epsilon,                \n"
"                                 const gint           radius)                 \n"
"{                                                                             \n"
"  gint     diameter     = radius * 2 + 1,                                     \n"
"           window_elems = diameter * diameter,                                \n"
"           image_elems  = rect->width * rect->height;                         \n"
"  gdouble *coeffs       = g_new0 (gdouble, image_elems * (COMPONENTS_INPUT + 1));\n"
"  gint     x, y, i, j;                                                        \n"
"                                                                              \n"
"  gdouble window  [window_elems + COMPONENTS_INPUT][COMPONENTS_INPUT + 1],    \n"
"          winprod [COMPONENTS_INPUT + 1][COMPONENTS_INPUT + 1],               \n"
"          inverse [COMPONENTS_INPUT + 1][COMPONENTS_INPUT + 1],               \n"
"          invprod [COMPONENTS_INPUT + 1][window_elems + COMPONENTS_INPUT],    \n"
"          alphmat [window_elems + COMPONENTS_INPUT][1];                       \n"
"                                                                              \n"
"  g_return_val_if_fail (image, NULL);                                         \n"
"  g_return_val_if_fail (alpha, NULL);                                         \n"
"  g_return_val_if_fail (rect,  NULL);                                         \n"
"                                                                              \n"
"  g_return_val_if_fail (epsilon != 0.0, NULL);                                \n"
"  g_return_val_if_fail (radius   > 0,   NULL);                                \n"
"                                                                              \n"
"  g_return_val_if_fail (COMPONENTS_INPUT + 1 == COMPONENTS_COEFF, NULL);      \n"
"                                                                              \n"
"  /* Zero out the main window matrix, and pre-set the lower window identity   \n"
"   * matrix, ones, and zeroes.                                                \n"
"   */                                                                         \n"
"  memset (window,  0, sizeof (window));                                       \n"
"  memset (alphmat, 0, sizeof (alphmat));                                      \n"
"  for (i = 0; i < COMPONENTS_INPUT; ++i)                                      \n"
"    window[window_elems + i][i] = sqrtf (epsilon);                            \n"
"  for (i = 0; i < window_elems; ++i)                                          \n"
"    window[i][COMPONENTS_INPUT] = 1.0;                                        \n"
"                                                                              \n"
"  /* Calculate window's coefficients */                                       \n"
"  for (x = radius; x < rect->width - radius; ++x)                             \n"
"    {                                                                         \n"
"      for (y = radius; y < rect->height - radius; ++y)                        \n"
"        {                                                                     \n"
"          /*          / I_r, I_g, I_b, 1 \\                                   \n"
"           *          | ...  ...  ...  1 |                                    \n"
"           * window = | eps   0    0   0 |                                    \n"
"           *          |  0   eps   0   0 |                                    \n"
"           *          \\  0    0   eps  0 /                                   \n"
"           */                                                                 \n"
"          for (j = 0; j < diameter; ++j)                                      \n"
"            for (i = 0; i < diameter; ++i)                                    \n"
"              {                                                               \n"
"                guint image_offset  =  x - radius + i;                        \n"
"                      image_offset += (y - radius + j) * rect->width;         \n"
"                      image_offset *= COMPONENTS_INPUT;                       \n"
"                                                                              \n"
"                window[i + j * diameter][0] = image[image_offset + 0];        \n"
"                window[i + j * diameter][1] = image[image_offset + 1];        \n"
"                window[i + j * diameter][2] = image[image_offset + 2];        \n"
"              }                                                               \n"
"                                                                              \n"
"          /* window' x window */                                              \n"
"          cblas_dgemm (CblasRowMajor, CblasTrans, CblasNoTrans,               \n"
"                       COMPONENTS_INPUT + 1,                                  \n"
"                       COMPONENTS_INPUT + 1,                                  \n"
"                       window_elems + COMPONENTS_INPUT, 1.0,                  \n"
"                       (gdouble *)window, COMPONENTS_INPUT + 1,               \n"
"                       (gdouble *)window, COMPONENTS_INPUT + 1,               \n"
"                       0.0, (gdouble *)winprod, COMPONENTS_INPUT + 1);        \n"
"                                                                              \n"
"          /* inv ($_) */                                                      \n"
"          matting_matrix4_inverse (winprod, inverse);                         \n"
"                                                                              \n"
"          /* $_ x window' */                                                  \n"
"          cblas_dgemm (CblasRowMajor, CblasNoTrans, CblasTrans,               \n"
"                       COMPONENTS_INPUT + 1,                                  \n"
"                       window_elems + COMPONENTS_INPUT,                       \n"
"                       COMPONENTS_INPUT + 1, 1.0,                             \n"
"                       (gdouble *)inverse, COMPONENTS_INPUT + 1,              \n"
"                       (gdouble *)window,  COMPONENTS_INPUT + 1,              \n"
"                       0.0, (gdouble*)invprod, window_elems + COMPONENTS_INPUT);\n"
"                                                                              \n"
"          /* alphmat = | a[x,y], .., a[x+d,y+d], 0, 0, 0, 0 | */              \n"
"          for (j = 0; j < diameter; ++j)                                      \n"
"            {                                                                 \n"
"              for (i = 0; i < diameter; ++i)                                  \n"
"                {                                                             \n"
"                  alphmat[i + j * diameter][0] = alpha[offset (x - radius + i,\n"
"                                                               y - radius + j,\n"
"                                                               rect, 1)];     \n"
"                }                                                             \n"
"            }                                                                 \n"
"                                                                              \n"
"          /* $_ x alphmat = | w, x, y, z |  */                                \n"
"          cblas_dgemm (CblasRowMajor, CblasNoTrans, CblasNoTrans,             \n"
"                       COMPONENTS_INPUT + 1, 1,                               \n"
"                       window_elems + COMPONENTS_INPUT, 1.0,                  \n"
"                       (gdouble *)invprod, window_elems + COMPONENTS_INPUT,   \n"
"                       (gdouble *)alphmat, 1,                                 \n"
"                       0.0, coeffs + offset (x, y, rect, COMPONENTS_INPUT + 1), 1);\n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  matting_fill_borders (coeffs, rect, COMPONENTS_COEFF, radius);              \n"
"  return coeffs;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/*                                                                            \n"
" * Convolves with a seperable 5 element kernel. Modifies the input data in    \n"
" * place.                                                                     \n"
" */                                                                           \n"
"static void                                                                   \n"
"matting_convolve5 (gdouble             *restrict pixels,                      \n"
"                   const GeglRectangle *restrict region,                      \n"
"                   guint                components,                           \n"
"                   const gdouble        kernel[CONVOLVE_LEN])                 \n"
"{                                                                             \n"
"  gint     x, y, i;                                                           \n"
"  guint    c;                                                                 \n"
"  gdouble *temp = g_new0 (gdouble, region->width * region->height * components);\n"
"                                                                              \n"
"  g_return_if_fail (CONVOLVE_LEN % 2 == 1);                                   \n"
"  /* Horizontal convolution */                                                \n"
"  for (y = 0; y < region->height; ++y)                                        \n"
"    {                                                                         \n"
"      for (x = CONVOLVE_RADIUS; x < region->width - CONVOLVE_RADIUS; ++x)     \n"
"        {                                                                     \n"
"          for (i = 0; i < CONVOLVE_LEN; ++i)                                  \n"
"            {                                                                 \n"
"              for (c = 0; c < components; ++c)                                \n"
"                {                                                             \n"
"                  temp  [offset (                      x, y, region, components) + c] +=\n"
"                  pixels[offset (x + i - CONVOLVE_RADIUS, y, region, components) + c] * kernel[i];\n"
"                }                                                             \n"
"            }                                                                 \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* Vertical convolution */                                                  \n"
"  memset (pixels, 0, (sizeof (pixels[0]) *                                    \n"
"                      region->width      *                                    \n"
"                      region->height     *                                    \n"
"                      components));                                           \n"
"                                                                              \n"
"  for (y = CONVOLVE_RADIUS; y < region->height - CONVOLVE_RADIUS; ++y)        \n"
"    {                                                                         \n"
"      for (x = 0; x < region->width; ++x)                                     \n"
"        {                                                                     \n"
"          for (i = 0; i < CONVOLVE_LEN; ++i)                                  \n"
"            {                                                                 \n"
"              for (c = 0; c < components; ++c)                                \n"
"                {                                                             \n"
"                  pixels[offset (x, y     - CONVOLVE_RADIUS, region, components) + c] +=\n"
"                  temp  [offset (x, y + i - CONVOLVE_RADIUS, region, components) + c] * kernel[i];\n"
"                }                                                             \n"
"            }                                                                 \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (temp);                                                              \n"
"                                                                              \n"
"  matting_fill_borders (pixels, region, components, CONVOLVE_RADIUS + 1);     \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static gdouble *                                                              \n"
"matting_downsample (gdouble             *restrict pixels,                     \n"
"                    const GeglRectangle *restrict input,                      \n"
"                    GeglRectangle       *restrict output,                     \n"
"                    guint                components)                          \n"
"{                                                                             \n"
"  /* Downsamples a buffer by a factor of two, and performs a gaussian blur.   \n"
"   * Returns the output size via the provided pointer; this is not respected as\n"
"   * an input parameter.                                                      \n"
"   */                                                                         \n"
"  static const gdouble DOWNSAMPLE_KERNEL[] =                                  \n"
"    { 0.0625, 0.25, 0.375, 0.25, 0.0625 };                                    \n"
"                                                                              \n"
"  gint     x, y;                                                              \n"
"  guint    c;                                                                 \n"
"  gdouble *down,                                                              \n"
"          *copy;                                                              \n"
"                                                                              \n"
"  g_return_val_if_fail (input->x == 0 && input->y == 0, NULL);                \n"
"  output->x      = input->x;                                                  \n"
"  output->y      = input->y;                                                  \n"
"  output->width  = ceil_div (input->width,  2);                               \n"
"  output->height = ceil_div (input->height, 2);                               \n"
"                                                                              \n"
"  /* convolve a copy of the pixels */                                         \n"
"  copy = g_new (gdouble, input->width  * input->height  * components);        \n"
"  memcpy (copy, pixels, sizeof (pixels[0]) *                                  \n"
"                        input->width       *                                  \n"
"                        input->height      *                                  \n"
"                        components);                                          \n"
"  matting_convolve5 (copy, input, components, DOWNSAMPLE_KERNEL);             \n"
"                                                                              \n"
"  /* downscale the copy into a new buffer */                                  \n"
"  down = g_new (gdouble, output->width * output->height * components);        \n"
"  for (x = 0; x < input->width; x += 2)                                       \n"
"    {                                                                         \n"
"      for (y = 0; y < input->height; y += 2)                                  \n"
"        {                                                                     \n"
"          guint down_offset = (offset (x / 2 , y / 2, output, components)),   \n"
"                copy_offset = (offset (x     ,     y,  input, components));   \n"
"                                                                              \n"
"          for (c = 0; c < components; ++c)                                    \n"
"            down[down_offset + c] = copy[copy_offset + c];                    \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (copy);                                                              \n"
"  return down;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static gdouble *                                                              \n"
"matting_upsample (const gdouble       *restrict pixels,                       \n"
"                  const GeglRectangle *restrict input,                        \n"
"                  const GeglRectangle *restrict output,                       \n"
"                  guint                         components)                   \n"
"{                                                                             \n"
"  /* Upsample to the size given in output, which must equate to a factor of ~2.\n"
"   * Copies in input pixels into the corresponding output locations, leaving  \n"
"   * the gaps black. Then performs a gaussian blur with a double weighted     \n"
"   * kernel.                                                                  \n"
"   */                                                                         \n"
"  static const gdouble  UPSAMPLE_KERNEL[] =                                   \n"
"    { 0.125, 0.5, 0.75, 0.5, 0.125 };                                         \n"
"                                                                              \n"
"  gint     x, y;                                                              \n"
"  guint    c;                                                                 \n"
"  gdouble *newpix = NULL;                                                     \n"
"                                                                              \n"
"  g_return_val_if_fail (pixels, NULL);                                        \n"
"  g_return_val_if_fail (input,  NULL);                                        \n"
"  g_return_val_if_fail (output, NULL);                                        \n"
"  g_return_val_if_fail (input->width < output->width &&                       \n"
"                        input->height < output->height, NULL);                \n"
"  g_return_val_if_fail (abs (output->width  - 2 * input->width ) <= 1, NULL); \n"
"  g_return_val_if_fail (abs (output->height - 2 * input->height) <= 1, NULL); \n"
"                                                                              \n"
"  newpix  = g_new0 (gdouble, output->width * output->height * components);    \n"
"                                                                              \n"
"  for (y = 1; y < output->height; y += 2)                                     \n"
"    {                                                                         \n"
"      for (x = 1; x < output->width; x += 2)                                  \n"
"        {                                                                     \n"
"          guint newoff = (x     +  y      * output->width) * components,      \n"
"                oldoff = (x / 2 + (y / 2) * input->width ) * components;      \n"
"                                                                              \n"
"          for (c = 0; c < components; ++c)                                    \n"
"            newpix[newoff + c] = pixels[oldoff + c];                          \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  matting_convolve5 (newpix, output, components, UPSAMPLE_KERNEL);            \n"
"  return newpix;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Upsample a previously computed alpha mat, using linear coefficients taken  \n"
" * from the source image. Resizes from small_rect to large_rect, and assumes  \n"
" * the factor is 2x +/- 1pixel.                                               \n"
" */                                                                           \n"
"static gdouble *                                                              \n"
"matting_upsample_alpha (const gdouble       *restrict small_pixels,           \n"
"                        const gdouble       *restrict large_pixels,           \n"
"                        const gdouble       *restrict small_alpha,            \n"
"                        const GeglRectangle *restrict small_rect,             \n"
"                        const GeglRectangle *restrict large_rect,             \n"
"                        gdouble              epsilon,                         \n"
"                        guint                radius)                          \n"
"{                                                                             \n"
"  gdouble *small_coeff = NULL,                                                \n"
"          *large_coeff = NULL,                                                \n"
"          *new_alpha = NULL;                                                  \n"
"  gint     i;                                                                 \n"
"                                                                              \n"
"  small_coeff = matting_get_linear_coefficients (small_pixels, small_alpha,   \n"
"                                                 small_rect, epsilon,         \n"
"                                                 radius);                     \n"
"  if (!small_coeff)                                                           \n"
"    goto cleanup;                                                             \n"
"                                                                              \n"
"  large_coeff = matting_upsample (small_coeff, small_rect, large_rect, COMPONENTS_COEFF);\n"
"  if (!large_coeff)                                                           \n"
"    goto cleanup;                                                             \n"
"                                                                              \n"
"  new_alpha = g_new (gdouble, large_rect->width * large_rect->height);        \n"
"  for (i = 0; i < large_rect->width * large_rect->height; ++i)                \n"
"    {                                                                         \n"
"      new_alpha[i]  = large_coeff[i * COMPONENTS_COEFF + 3];                  \n"
"      new_alpha[i] += large_coeff[i * COMPONENTS_COEFF + 0] * large_pixels[i * COMPONENTS_INPUT + 0];\n"
"      new_alpha[i] += large_coeff[i * COMPONENTS_COEFF + 1] * large_pixels[i * COMPONENTS_INPUT + 1];\n"
"      new_alpha[i] += large_coeff[i * COMPONENTS_COEFF + 2] * large_pixels[i * COMPONENTS_INPUT + 2];\n"
"    }                                                                         \n"
"                                                                              \n"
"cleanup:                                                                      \n"
"  g_free (small_coeff);                                                       \n"
"  g_free (large_coeff);                                                       \n"
"  return new_alpha;                                                           \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static sparse_t *                                                             \n"
"matting_sparse_new (guint cols, guint rows, guint elems)                      \n"
"{                                                                             \n"
"  sparse_t *s = g_new (sparse_t, 1);                                          \n"
"  s->columns  = cols;                                                         \n"
"  s->rows     = rows;                                                         \n"
"  s->col_idx  = g_new  (SuiteSparse_long, cols + 1);                          \n"
"  s->row_idx  = g_new  (SuiteSparse_long, elems);                             \n"
"  s->values   = g_new0 (gdouble, elems);                                      \n"
"                                                                              \n"
"  return s;                                                                   \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"matting_sparse_free (sparse_t *s)                                             \n"
"{                                                                             \n"
"  if (!s)                                                                     \n"
"      return;                                                                 \n"
"                                                                              \n"
"  g_free (s->row_idx);                                                        \n"
"  g_free (s->col_idx);                                                        \n"
"  g_free (s->values);                                                         \n"
"  g_free (s);                                                                 \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static guint                                                                  \n"
"matting_sparse_elems (const sparse_t *s)                                      \n"
"{                                                                             \n"
"  return s->col_idx[s->columns];                                              \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Debugging function which ensures the sparse matrix fields are consistent   \n"
" * with what UMFPACK, and the matting algorithm, would expect.                \n"
" *                                                                            \n"
" * Returns FALSE, using glib debugging routines, if there is an error. Else,  \n"
" * returns TRUE.                                                              \n"
" */                                                                           \n"
"static gboolean                                                               \n"
"matting_verify (const sparse_t *s)                                            \n"
"{                                                                             \n"
"  guint     i, j;                                                             \n"
"  gboolean *rows;                                                             \n"
"                                                                              \n"
"  /* Must be a square matrix */                                               \n"
"  g_return_val_if_fail (s->columns == s->rows, FALSE);                        \n"
"  g_return_val_if_fail (s->col_idx[0] == 0,    FALSE);                        \n"
"                                                                              \n"
"  for (i = 1; i < s->columns; ++i)                                            \n"
"    {                                                                         \n"
"      /* Strictly ascending column indices */                                 \n"
"      guint col = s->col_idx[i];                                              \n"
"      g_return_val_if_fail (s->col_idx[i - 1] <= col, FALSE);                 \n"
"                                                                              \n"
"      for (j = s->col_idx[i] + 1; j < s->col_idx[i + 1]; ++j)                 \n"
"        {                                                                     \n"
"          /* Strictly ascending row indices, within a column */               \n"
"          guint row = s->row_idx[j];                                          \n"
"          g_return_val_if_fail (s->row_idx[j - 1] < row, FALSE);              \n"
"          g_return_val_if_fail (row < s->rows,           FALSE);              \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* We expect to have entries for each column in the matrix. Note: this is   \n"
"   * not a requirement of the UMFPACK format; rather, something we expect of  \n"
"   * the matrix from the matting algorithm.                                   \n"
"   */                                                                         \n"
"  rows = g_new (gboolean, s->rows);                                           \n"
"                                                                              \n"
"  for (i = 0; i < s->rows; ++i)                                               \n"
"    rows [i] = FALSE;                                                         \n"
"  for (i = 0; i < matting_sparse_elems (s); ++i)                              \n"
"    {                                                                         \n"
"      guint row = s->row_idx[i];                                              \n"
"      if (!(row < s->rows))                                                   \n"
"        {                                                                     \n"
"          g_free (rows);                                                      \n"
"          g_return_val_if_reached (FALSE);                                    \n"
"        }                                                                     \n"
"      rows[row] = TRUE;                                                       \n"
"    }                                                                         \n"
"  for (i = 0; i < s->rows; ++i)                                               \n"
"    {                                                                         \n"
"      if (!rows[i])                                                           \n"
"        {                                                                     \n"
"          g_free (rows);                                                      \n"
"          g_return_val_if_reached (FALSE);                                    \n"
"        }                                                                     \n"
"      }                                                                       \n"
"                                                                              \n"
"  g_free (rows);                                                              \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Calculate the matting laplacian for an image, given a user trimap.         \n"
" * We accumulate entries in a sparse banded matrix, for a radius around each  \n"
" * pixel in the image.                                                        \n"
" *                                                                            \n"
" * We construct a triplet form of the matrix initially, then transform to     \n"
" * compressed column. This is much simpler than directly constructing the     \n"
" * compressed column form, and does not appear to cause a performance         \n"
" * bottleneck (though does consume additional memory).                        \n"
" */                                                                           \n"
"static sparse_t*                                                              \n"
"matting_get_laplacian (const gdouble       *restrict image,                   \n"
"                       const gdouble       *restrict trimap,                  \n"
"                       const GeglRectangle *restrict roi,                     \n"
"                       const gint           radius,                           \n"
"                       const gdouble        epsilon,                          \n"
"                       const gdouble        lambda)                           \n"
"{                                                                             \n"
"  gint      diameter     = radius * 2 + 1,                                    \n"
"            window_elems = diameter * diameter,                               \n"
"            image_elems  = roi->width * roi->height,                          \n"
"            i, j, k, x, y,                                                    \n"
"            status;                                                           \n"
"  SuiteSparse_long *trip_col,                                                 \n"
"                   *trip_row;                                                 \n"
"  glong     trip_nz = 0,                                                      \n"
"            trip_cursor = 0,                                                  \n"
"            trip_masked = 0;                                                  \n"
"  gdouble  *trip_val;                                                         \n"
"  sparse_t *laplacian;                                                        \n"
"                                                                              \n"
"  gdouble       mean[COMPONENTS_INPUT],                                       \n"
"         mean_matrix[COMPONENTS_INPUT][COMPONENTS_INPUT],                     \n"
"          covariance[COMPONENTS_INPUT][COMPONENTS_INPUT],                     \n"
"             inverse[COMPONENTS_INPUT][COMPONENTS_INPUT],                     \n"
"              window[COMPONENTS_INPUT][window_elems],                         \n"
"             winxinv[COMPONENTS_INPUT][window_elems],                         \n"
"                  values[window_elems][window_elems];                         \n"
"                                                                              \n"
"  g_return_val_if_fail (radius > 0, NULL);                                    \n"
"  g_return_val_if_fail (COMPONENTS_INPUT == 3, NULL);                         \n"
"                                                                              \n"
"  for (j = radius; j < roi->height - radius; ++j)                             \n"
"    {                                                                         \n"
"      for (i = radius; i < roi->width - radius; ++i)                          \n"
"        {                                                                     \n"
"          if (trimap_masked (trimap, i, j, roi))                              \n"
"            trip_masked++;                                                    \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  trip_nz   = trip_masked * window_elems * window_elems;                      \n"
"  trip_nz  += image_elems; // Sparse diagonal and row summing at conclusion   \n"
"                                                                              \n"
"  trip_col  = g_new  (SuiteSparse_long, trip_nz);                             \n"
"  trip_row  = g_new  (SuiteSparse_long, trip_nz);                             \n"
"  trip_val  = g_new0 (gdouble, trip_nz);                                      \n"
"                                                                              \n"
"  /* Compute the contribution of each pixel in the image to the laplacian */  \n"
"  for (i = radius; i < roi->width - radius; ++i)                              \n"
"    {                                                                         \n"
"      for (j = radius; j < roi->height - radius; ++j)                         \n"
"        {                                                                     \n"
"          /* Skip if the pixel is valid in the the trimap */                  \n"
"          if (!trimap_masked (trimap, i, j, roi))                             \n"
"            continue;                                                         \n"
"          trip_masked--;                                                      \n"
"          g_return_val_if_fail (trip_masked >= 0, FALSE);                     \n"
"                                                                              \n"
"          /* Calculate window's component means, and their vector product     \n"
"           * (which we will use later to calculate the covariance matrix).    \n"
"           * Store the values into the window matrix as we go.                \n"
"           */                                                                 \n"
"          mean[0] = mean[1] = mean[2] = 0.0;                                  \n"
"          k = 0;                                                              \n"
"          for (y = j - radius; y <= j + radius; ++y)                          \n"
"            for (x = i - radius; x <= i + radius; ++x)                        \n"
"              {                                                               \n"
"                mean[0] += window[0][k] = image[(x + y * roi->width) * COMPONENTS_INPUT + 0];\n"
"                mean[1] += window[1][k] = image[(x + y * roi->width) * COMPONENTS_INPUT + 1];\n"
"                mean[2] += window[2][k] = image[(x + y * roi->width) * COMPONENTS_INPUT + 2];\n"
"                ++k;                                                          \n"
"              }                                                               \n"
"                                                                              \n"
"          mean[0] /= window_elems;                                            \n"
"          mean[1] /= window_elems;                                            \n"
"          mean[2] /= window_elems;                                            \n"
"                                                                              \n"
"          matting_vector3_self_product (mean, mean_matrix);                   \n"
"                                                                              \n"
"          /*                                                                  \n"
"           * Calculate inverse covariance matrix.                             \n"
"           */                                                                 \n"
"                                                                              \n"
"          /* Multiply the 'component x window' matrix with its transpose to   \n"
"           * form a 3x3 matrix which is the first component of the covariance \n"
"           * matrix.                                                          \n"
"           */                                                                 \n"
"          cblas_dgemm (CblasRowMajor, CblasNoTrans, CblasTrans,               \n"
"                       COMPONENTS_INPUT, COMPONENTS_INPUT, window_elems,      \n"
"                       1.0 / window_elems,                                    \n"
"                       (gdouble *)window, window_elems,                       \n"
"                       (gdouble *)window, window_elems,                       \n"
"                       0.0,  (gdouble *)covariance, COMPONENTS_INPUT);        \n"
"                                                                              \n"
"          /* Subtract the mean to create the covariance matrix, then add the  \n"
"           * epsilon term and invert.                                         \n"
"           */                                                                 \n"
"          matting_matrix3_matrix3_sub (covariance, mean_matrix, covariance);  \n"
"          covariance[0][0] += epsilon / window_elems;                         \n"
"          covariance[1][1] += epsilon / window_elems;                         \n"
"          covariance[2][2] += epsilon / window_elems;                         \n"
"          matting_matrix3_inverse     (covariance, inverse);                  \n"
"                                                                              \n"
"          /* Subtract each component's mean from the pixels */                \n"
"          for (k = 0; k < window_elems; ++k)                                  \n"
"            {                                                                 \n"
"              window[0][k] -= mean[0];                                        \n"
"              window[1][k] -= mean[1];                                        \n"
"              window[2][k] -= mean[2];                                        \n"
"            }                                                                 \n"
"                                                                              \n"
"          /* Calculate the values for the matting matrix */                   \n"
"          cblas_dgemm (CblasRowMajor, CblasNoTrans, CblasNoTrans,             \n"
"                       COMPONENTS_INPUT, window_elems, COMPONENTS_INPUT,      \n"
"                       1.0,                                                   \n"
"                       (gdouble *)inverse, COMPONENTS_INPUT,                  \n"
"                       (gdouble *) window, window_elems,                      \n"
"                       0.0, (gdouble *)winxinv, window_elems);                \n"
"                                                                              \n"
"          cblas_dgemm (CblasRowMajor, CblasTrans, CblasNoTrans,               \n"
"                       window_elems, window_elems, COMPONENTS_INPUT,          \n"
"                       1.0,                                                   \n"
"                       (gdouble *) window, window_elems,                      \n"
"                       (gdouble *)winxinv, window_elems,                      \n"
"                       0.0, (gdouble *)values, window_elems);                 \n"
"                                                                              \n"
"          /* Store the values and coordinates */                              \n"
"          for (y = 0; y < window_elems; ++y)                                  \n"
"            for (x = 0; x < window_elems; ++x)                                \n"
"              {                                                               \n"
"                SuiteSparse_long yx = y % diameter,                           \n"
"                                 yy = y / diameter,                           \n"
"                                 xx = x % diameter,                           \n"
"                                 xy = x / diameter;                           \n"
"                                                                              \n"
"                g_return_val_if_fail (trip_cursor < trip_nz, FALSE);          \n"
"                trip_col[trip_cursor] = (i - radius + yx) + (j - radius + yy) * roi->width,\n"
"                trip_row[trip_cursor] = (i - radius + xx) + (j - radius + xy) * roi->width,\n"
"                trip_val[trip_cursor] = (1.0 + values[y][x]) / window_elems;  \n"
"                ++trip_cursor;                                                \n"
"              }                                                               \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  {                                                                           \n"
"    gdouble *row_sum = g_new (gdouble, image_elems);                          \n"
"                                                                              \n"
"    /* Calculate the sum of all the elements in each row */                   \n"
"    for (i = 0; i < image_elems; ++i)                                         \n"
"      row_sum[i] = 0.0;                                                       \n"
"    for (i = 0; i < trip_cursor; ++i)                                         \n"
"      row_sum[trip_row[i]]  += trip_val[i];                                   \n"
"                                                                              \n"
"    /* Negate each entry of the matrix. This partially implements a           \n"
"     * subtraction from the diagonal matrix:                                  \n"
"     *    [lambda + sum, lambda + sum, ..., lambda + sum]                     \n"
"     */                                                                       \n"
"    for (i = 0; i < trip_cursor; ++i)                                         \n"
"      trip_val[i] = -trip_val[i];                                             \n"
"                                                                              \n"
"    /* Set the diagonal such that the sum of the row equals `lambda' if the   \n"
"     * trimap entry is valid                                                  \n"
"     */                                                                       \n"
"    for (i = 0; i < image_elems; ++i)                                         \n"
"      {                                                                       \n"
"        trip_col[trip_cursor] = i;                                            \n"
"        trip_row[trip_cursor] = i;                                            \n"
"        trip_val[trip_cursor] = row_sum[i];                                   \n"
"                                                                              \n"
"        if (!trimap_masked (trimap, i, 0, roi))                               \n"
"          trip_val[trip_cursor] += lambda;                                    \n"
"        trip_cursor++;                                                        \n"
"      }                                                                       \n"
"                                                                              \n"
"    /* Double check that each row equals either 0.0 or lambda */              \n"
"    for (i = 0; i < image_elems; ++i)                                         \n"
"      row_sum[i] = 0.0;                                                       \n"
"    for (i = 0; i < trip_cursor; ++i)                                         \n"
"      row_sum[trip_row[i]]  += trip_val[i];                                   \n"
"    for (i = 0; i < image_elems; ++i)                                         \n"
"      {                                                                       \n"
"        g_warn_if_fail (float_cmp (row_sum [i], 0.0) ||                       \n"
"                        float_cmp (row_sum [i], lambda));                     \n"
"      }                                                                       \n"
"                                                                              \n"
"    g_free (row_sum);                                                         \n"
"  }                                                                           \n"
"                                                                              \n"
"  g_warn_if_fail (trip_cursor == trip_nz);                                    \n"
"                                                                              \n"
"  /* Convert to the compressed column format expected by UMFPACK */           \n"
"  laplacian = matting_sparse_new (image_elems, image_elems, trip_cursor);     \n"
"  status    = umfpack_dl_triplet_to_col (laplacian->rows,                     \n"
"                                         laplacian->columns,                  \n"
"                                         trip_cursor,                         \n"
"                                         trip_row, trip_col, trip_val,        \n"
"                                         laplacian->col_idx,                  \n"
"                                         laplacian->row_idx,                  \n"
"                                         laplacian->values,                   \n"
"                                         NULL);                               \n"
"                                                                              \n"
"  g_free (trip_col);                                                          \n"
"  g_free (trip_row);                                                          \n"
"  g_free (trip_val);                                                          \n"
"                                                                              \n"
"  g_return_val_if_fail (status == UMFPACK_OK, FALSE);                         \n"
"  return laplacian;                                                           \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static gboolean                                                               \n"
"matting_solve_laplacian (gdouble             *restrict trimap,                \n"
"                         sparse_t            *restrict laplacian,             \n"
"                         gdouble             *restrict solution,              \n"
"                         const GeglRectangle *restrict roi,                   \n"
"                         gdouble              lambda)                         \n"
"{                                                                             \n"
"  void    *symbolic = NULL,                                                   \n"
"          *numeric  = NULL;                                                   \n"
"  gint     status;                                                            \n"
"  guint    image_elems, i;                                                    \n"
"  gboolean success = FALSE;                                                   \n"
"  gdouble  umfcontrol[UMFPACK_CONTROL],                                       \n"
"           umfinfo[UMFPACK_INFO];                                             \n"
"                                                                              \n"
"  g_return_val_if_fail (trimap,    FALSE);                                    \n"
"  g_return_val_if_fail (laplacian, FALSE);                                    \n"
"  g_return_val_if_fail (solution,  FALSE);                                    \n"
"                                                                              \n"
"  g_return_val_if_fail (roi,       FALSE);                                    \n"
"  g_return_val_if_fail (!gegl_rectangle_is_empty (roi), FALSE);               \n"
"  image_elems = roi->width * roi->height;                                     \n"
"                                                                              \n"
"  g_return_val_if_fail (laplacian->columns == image_elems, FALSE);            \n"
"  g_return_val_if_fail (laplacian->rows    == image_elems, FALSE);            \n"
"                                                                              \n"
"  matting_verify (laplacian);                                                 \n"
"                                                                              \n"
"  umfpack_di_defaults (umfcontrol);                                           \n"
"  /* Pre-process the matrix */                                                \n"
"  if ((status = umfpack_dl_symbolic (laplacian->rows,                         \n"
"                                     laplacian->columns,                      \n"
"                                     laplacian->col_idx,                      \n"
"                                     laplacian->row_idx,                      \n"
"                                     laplacian->values,                       \n"
"                                     &symbolic,                               \n"
"                                     umfcontrol, umfinfo)) < 0)               \n"
"    {                                                                         \n"
"      symbolic = NULL;                                                        \n"
"      goto cleanup;                                                           \n"
"    }                                                                         \n"
"                                                                              \n"
"  if ((status = umfpack_dl_numeric (laplacian->col_idx,                       \n"
"                                    laplacian->row_idx,                       \n"
"                                    laplacian->values,                        \n"
"                                    symbolic, &numeric,                       \n"
"                                    umfcontrol, umfinfo)) < 0)                \n"
"    {                                                                         \n"
"      numeric = NULL;                                                         \n"
"      goto cleanup;                                                           \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* Solve and exit */                                                        \n"
"  {                                                                           \n"
"    gdouble *residual = g_new (gdouble, image_elems);                         \n"
"    for (i = 0; i < image_elems; ++i)                                         \n"
"      {                                                                       \n"
"        if (trimap_masked (trimap, i, 0, roi))                                \n"
"          residual[i] = 0;                                                    \n"
"        else                                                                  \n"
"          residual[i] = lambda * trimap[i * COMPONENTS_AUX + AUX_VALUE];      \n"
"      }                                                                       \n"
"                                                                              \n"
"    status = umfpack_dl_solve (UMFPACK_A,                                     \n"
"                               laplacian->col_idx,                            \n"
"                               laplacian->row_idx,                            \n"
"                               laplacian->values,                             \n"
"                               solution,                                      \n"
"                               residual,                                      \n"
"                               numeric,                                       \n"
"                               umfcontrol, umfinfo);                          \n"
"                                                                              \n"
"    /* Positive numbers are warnings. We don't care if the matrix is          \n"
"     * singular, as the computed result is still usable, so just check for    \n"
"     * errors.                                                                \n"
"     */                                                                       \n"
"    g_free (residual);                                                        \n"
"    if (status < 0)                                                           \n"
"      goto cleanup;                                                           \n"
"  }                                                                           \n"
"                                                                              \n"
"  /* Courtesy clamping of the solution to normal alpha range */               \n"
"  for (i = 0; i < image_elems; ++i)                                           \n"
"    solution[i] = CLAMP (solution[i], 0.0, 1.0);                              \n"
"                                                                              \n"
"  success = TRUE;                                                             \n"
"cleanup:                                                                      \n"
"  /* Singular matrices appear to work correctly, provided that we clamp the   \n"
"   * results (which needs to be done regardless). I'm not sure if this is a   \n"
"   * result of an incorrect implementation of the algorithm, or if it's       \n"
"   * inherent to the design; either way it seems to work.                     \n"
"   */                                                                         \n"
"  if (status != UMFPACK_OK && status != UMFPACK_WARNING_singular_matrix)      \n"
"    g_warning (\"%s\", matting_umf_error_to_string (status));                 \n"
"                                                                              \n"
"  if (numeric)                                                                \n"
"    umfpack_dl_free_numeric  (&numeric);                                      \n"
"  if (symbolic)                                                               \n"
"    umfpack_dl_free_symbolic (&symbolic);                                     \n"
"                                                                              \n"
"  return success;                                                             \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Recursively downsample, solve, then upsample the matting laplacian.        \n"
" * Perform up to `levels' recursions (provided the image remains large        \n"
" * enough), with up to `active_levels' number of full laplacian solves (not   \n"
" * just extrapolation).                                                       \n"
" */                                                                           \n"
"static gdouble *                                                              \n"
"matting_solve_level (gdouble             *restrict pixels,                    \n"
"                     gdouble             *restrict trimap,                    \n"
"                     const GeglRectangle *restrict region,                    \n"
"                     guint                active_levels,                      \n"
"                     guint                levels,                             \n"
"                     guint                radius,                             \n"
"                     gdouble              epsilon,                            \n"
"                     gdouble              lambda,                             \n"
"                     gdouble              threshold)                          \n"
"{                                                                             \n"
"  gint     i;                                                                 \n"
"  gdouble *new_alpha    = NULL,                                               \n"
"          *eroded_alpha = NULL;                                               \n"
"  gdouble value;                                                              \n"
"                                                                              \n"
"  /* add alpha to trimap because algo is need it */                           \n"
"  for (i = 0; i < region->width * region->height; ++i)                        \n"
"  {                                                                           \n"
"    value = trimap[i * COMPONENTS_AUX + AUX_VALUE];                           \n"
"    if (value > AUX_BACKGROUND && value < AUX_FOREGROUND)                     \n"
"        trimap[i * COMPONENTS_AUX + AUX_ALPHA] = 0.0;                         \n"
"  }                                                                           \n"
"                                                                              \n"
"  if (region->width  < MIN_LEVEL_DIAMETER ||                                  \n"
"      region->height < MIN_LEVEL_DIAMETER)                                    \n"
"    {                                                                         \n"
"      GEGL_NOTE (GEGL_DEBUG_PROCESS,                                          \n"
"                 \"skipping subdivision with level %dx%d\\n\",                \n"
"                region->width, region->height);                               \n"
"      levels = 0;                                                             \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (levels > 0)                                                             \n"
"    {                                                                         \n"
"      GeglRectangle  small_region;                                            \n"
"      gdouble       *small_pixels,                                            \n"
"                    *small_trimap;                                            \n"
"      gdouble       *small_alpha;                                             \n"
"                                                                              \n"
"      /* Downsample, solve, then upsample again */                            \n"
"      small_pixels = matting_downsample (pixels, region, &small_region,       \n"
"                                         COMPONENTS_INPUT);                   \n"
"      small_trimap = matting_downsample (trimap, region, &small_region,       \n"
"                                         COMPONENTS_AUX);                     \n"
"      for (i = 0; i < small_region.width  *                                   \n"
"                      small_region.height *                                   \n"
"                      COMPONENTS_AUX; ++i)                                    \n"
"        {                                                                     \n"
"          small_trimap[i] = roundf (small_trimap[i]);                         \n"
"        }                                                                     \n"
"                                                                              \n"
"      small_alpha = matting_solve_level (small_pixels, small_trimap,          \n"
"                                         &small_region, active_levels,        \n"
"                                         levels - 1, radius, epsilon,         \n"
"                                         lambda, threshold);                  \n"
"                                                                              \n"
"      new_alpha = matting_upsample_alpha (small_pixels, pixels, small_alpha,  \n"
"                                          &small_region, region, epsilon,     \n"
"                                          radius);                            \n"
"                                                                              \n"
"      /* Erode the result:                                                    \n"
"       * If the trimap alpha has not been set high (ie, valid), update the    \n"
"       * trimap value with our computed result.                               \n"
"       * If it was eroded, then set the trimap pixel as valid by setting      \n"
"       * alpha high.                                                          \n"
"       * Set all trimap values as either high or low.                         \n"
"       */                                                                     \n"
"      eroded_alpha = matting_erode_range (new_alpha, region, 1, radius,       \n"
"                                          threshold, threshold);              \n"
"                                                                              \n"
"      g_free (small_pixels);                                                  \n"
"      g_free (small_trimap);                                                  \n"
"      g_free (small_alpha);                                                   \n"
"                                                                              \n"
"      for (i = 0; i < region->width * region->height; ++i)                    \n"
"        {                                                                     \n"
"          if (trimap[i * COMPONENTS_AUX + AUX_ALPHA] < TRIMAP_ALPHA_THRESHOLD)\n"
"            trimap[i * COMPONENTS_AUX + AUX_VALUE] = new_alpha[i];            \n"
"                                                                              \n"
"          if (isnan (eroded_alpha[i]))                                        \n"
"            trimap[i * COMPONENTS_AUX + AUX_ALPHA] = 1.0;                     \n"
"                                                                              \n"
"          trimap[i * COMPONENTS_AUX + AUX_VALUE] *= roundf (trimap[i * COMPONENTS_AUX + AUX_VALUE]) *\n"
"                                                    trimap[i * COMPONENTS_AUX + AUX_ALPHA];\n"
"        }                                                                     \n"
"      g_free (eroded_alpha);                                                  \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* Ordinary solution of the matting laplacian */                            \n"
"  if (active_levels >= levels || levels == 0)                                 \n"
"    {                                                                         \n"
"      sparse_t *laplacian;                                                    \n"
"      g_free (new_alpha);                                                     \n"
"                                                                              \n"
"      if (!(laplacian = matting_get_laplacian (pixels, trimap, region,        \n"
"              radius, epsilon, lambda)))                                      \n"
"        {                                                                     \n"
"          g_warning (\"unable to construct laplacian matrix\");               \n"
"          return NULL;                                                        \n"
"        }                                                                     \n"
"                                                                              \n"
"      new_alpha = g_new (gdouble, region->width * region->height);            \n"
"      matting_solve_laplacian (trimap, laplacian, new_alpha, region, lambda); \n"
"      matting_sparse_free (laplacian);                                        \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_return_val_if_fail (new_alpha != NULL, NULL);                             \n"
"  return new_alpha;                                                           \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Simple wrapper around matting_solve_level, which extracts the relevant     \n"
" * pixel data and writes the solution to output.                              \n"
" */                                                                           \n"
"static gboolean                                                               \n"
"matting_process (GeglOperation       *operation,                              \n"
"                 GeglBuffer          *input_buf,                              \n"
"                 GeglBuffer          *aux_buf,                                \n"
"                 GeglBuffer          *output_buf,                             \n"
"                 const GeglRectangle *result,                                 \n"
"                 gint                 level)                                  \n"
"{                                                                             \n"
"  const GeglProperties *o = GEGL_PROPERTIES (operation);                      \n"
"  gdouble          *input   = NULL,                                           \n"
"                   *trimap  = NULL;                                           \n"
"  gdouble          *output  = NULL;                                           \n"
"  gboolean          success = FALSE;                                          \n"
"                                                                              \n"
"  g_return_val_if_fail (babl_format_get_n_components (babl_format (FORMAT_INPUT )) == COMPONENTS_INPUT,  FALSE);\n"
"  g_return_val_if_fail (babl_format_get_n_components (babl_format (FORMAT_AUX   )) == COMPONENTS_AUX,    FALSE);\n"
"  g_return_val_if_fail (babl_format_get_n_components (babl_format (FORMAT_OUTPUT)) == COMPONENTS_OUTPUT, FALSE);\n"
"                                                                              \n"
"  g_return_val_if_fail (operation,  FALSE);                                   \n"
"  g_return_val_if_fail (input_buf,  FALSE);                                   \n"
"  g_return_val_if_fail (aux_buf,    FALSE);                                   \n"
"  g_return_val_if_fail (output_buf, FALSE);                                   \n"
"  g_return_val_if_fail (result,     FALSE);                                   \n"
"                                                                              \n"
"  input  = g_new (gdouble, result->width * result->height * COMPONENTS_INPUT);\n"
"  trimap = g_new (gdouble, result->width * result->height * COMPONENTS_AUX);  \n"
"                                                                              \n"
"  gegl_buffer_get (input_buf, result, 1.0, babl_format (FORMAT_INPUT), input, GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);\n"
"  gegl_buffer_get (  aux_buf, result, 1.0, babl_format (FORMAT_AUX),  trimap, GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);\n"
"                                                                              \n"
"  output = matting_solve_level (input, trimap, result,                        \n"
"                                MIN (o->active_levels, o->levels), o->levels, \n"
"                                o->radius, powf (10, o->epsilon), o->lambda,  \n"
"                                o->threshold);                                \n"
"  gegl_buffer_set (output_buf, result, 0, babl_format (FORMAT_OUTPUT), output,\n"
"                   GEGL_AUTO_ROWSTRIDE);                                      \n"
"                                                                              \n"
"  success = TRUE;                                                             \n"
"                                                                              \n"
"  g_free (input);                                                             \n"
"  g_free (trimap);                                                            \n"
"  g_free (output);                                                            \n"
"                                                                              \n"
"  return success;                                                             \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GeglOperationClass         *operation_class;                                \n"
"  GeglOperationComposerClass *composer_class;                                 \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"  composer_class  = GEGL_OPERATION_COMPOSER_CLASS (klass);                    \n"
"                                                                              \n"
"  composer_class->process = matting_process;                                  \n"
"                                                                              \n"
"  operation_class->prepare                 = matting_prepare;                 \n"
"  operation_class->get_required_for_output = matting_get_required_for_output; \n"
"  operation_class->get_cached_region       = matting_get_cached_region;       \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"  \"name\",         \"gegl:matting-levin\",                                   \n"
"  \"title\",        _(\"Matting Levin\"),                                     \n"
"  \"categories\",   \"matting\",                                              \n"
"  \"description\",                                                            \n"
"        _(\"Given a sparse user supplied tri-map and an input image, create a \"\n"
"          \"foreground alpha mat. Set white as selected, black as unselected, \"\n"
"          \"for the tri-map.\"),                                              \n"
"        NULL);                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"#endif                                                                        \n"
;
