/******************************************************************************
* File: lua_funcs.c                                                           *
* C functions callable from Lua scripts.                                      *
*                                                                             *
* This file is Copyright (C) 1997 by Joachim Pileborg, all rights reserved.   *
******************************************************************************/

#define __LUA_FUNCS_C__

#include <lua.h>
#include <lualib.h>

#include "conf.h"
#include "sysdep.h"

#include "structs.h"
#include "utils.h"
#include "triggers.h"
#include "interpreter.h"
#include "db.h"
#include "handler.h"

/*****************************************************************************/

extern void             *_me;
extern int               _room;
extern struct obj_data  *_object;
extern struct char_data *_char;
extern int               _source;

/*****************************************************************************/

void lm_number(void)
{
	lua_Object from, to;
	int res;

	from = lua_getparam(1);
	to   = lua_getparam(2);

	res = number(lua_getnumber(from), lua_getnumber(to));

	lua_pushnumber(res);
}

/*****************************************************************************/

void lm_dice(void)
{
	lua_Object from, to;
	int res;

	from = lua_getparam(1);
	to   = lua_getparam(2);

	res = dice(lua_getnumber(from), lua_getnumber(to));

	lua_pushnumber(res);
}

/*****************************************************************************/

void lm_say(void)
{
	char *str = NULL;

	ACMD(do_say);

	if (_source != TS_MOBILE && _source != TS_PLAYER && _source != TS_CHAR)
	{
		return;
	}

	str = lua_getstring(lua_getparam(1));

	do_say((struct char_data *) _me, str, 0, 0);
}

/*****************************************************************************/

void lm_isroom(void)
{
	struct user_data *d;
	lua_Object        o;

	o = lua_getparam(1);
	if (!lua_isuserdata(o))
	{
		lua_pushnil();
		return;
	}

	d = (struct user_data *) lua_getuserdata(o);
	if (!d)
	{
		lua_pushnil();
		return;
	}

	if (d->type == TS_ROOM)
		lua_pushnumber(1);
	else
		lua_pushnil();
}

void lm_isobject(void)
{
	struct user_data *d;
	lua_Object        o;

	o = lua_getparam(1);
	if (!lua_isuserdata(o))
	{
		lua_pushnil();
		return;
	}

	d = (struct user_data *) lua_getuserdata(o);
	if (!d)
	{
		lua_pushnil();
		return;
	}

	if (d->type == TS_OBJECT)
		lua_pushnumber(1);
	else
		lua_pushnil();
}

void lm_ischar(void)
{
	struct user_data *d;
	lua_Object        o;

	o = lua_getparam(1);
	if (!lua_isuserdata(o))
	{
		lua_pushnil();
		return;
	}

	d = (struct user_data *) lua_getuserdata(o);
	if (!d)
	{
		lua_pushnil();
		return;
	}

	if (d->type == TS_MOBILE || d->type == TS_PLAYER || d->type == TS_CHAR)
		lua_pushnumber(1);
	else
		lua_pushnil();
}

void lm_isplayer(void)
{
	struct user_data *d;
	lua_Object        o;

	o = lua_getparam(1);
	if (!lua_isuserdata(o))
	{
		lua_pushnil();
		return;
	}

	d = (struct user_data *) lua_getuserdata(o);
	if (!d)
	{
		lua_pushnil();
		return;
	}

	if ((d->type == TS_PLAYER || d->type == TS_CHAR) &&
			!IS_NPC((struct char_data *) d->data))
		lua_pushnumber(1);
	else
		lua_pushnil();
}

void lm_isnpc(void)
{
	struct user_data *d;
	lua_Object        o;

	o = lua_getparam(1);
	if (!lua_isuserdata(o))
	{
		lua_pushnil();
		return;
	}

	d = (struct user_data *) lua_getuserdata(o);
	if (!d)
	{
		lua_pushnil();
		return;
	}

	if ((d->type == TS_MOBILE || d->type == TS_CHAR) &&
			!IS_NPC((struct char_data *) d->data))
		lua_pushnumber(1);
	else
		lua_pushnil();
}
/*****************************************************************************/

void lm_tell(void)
{
	struct char_data *ch = NULL;
	lua_Object to, what;

	void perform_tell(struct char_data *, struct char_data *, char *);

	if (_source != TS_MOBILE && _source != TS_PLAYER && _source != TS_CHAR)
		return;

	if (lua_isnil(to = lua_getparam(1)))
		return;
	if (lua_isnil(what = lua_getparam(2)) || (!lua_isstring(what)))
		return;

	if (lua_isstring(to))
	{
		ch = get_char_vis((struct char_data *) _me, lua_getstring(to));
	}
	else if (lua_isuserdata(to))
	{
		if (((struct user_data *) lua_getuserdata(to))->type == TS_MOBILE ||
				((struct user_data *) lua_getuserdata(to))->type == TS_PLAYER ||
				((struct user_data *) lua_getuserdata(to))->type == TS_CHAR)
		{
			ch = (struct char_data *)
				((struct user_data *) lua_getuserdata(to))->data;
		}
	}

	if (ch)
		perform_tell((struct char_data *) _me, ch, lua_getstring(what));
}

/*****************************************************************************/

void lm_ask(void)
{
	struct char_data *ch = NULL;
	char *name = NULL;
	lua_Object to, what;

	ACMD(do_spec_comm);

	if (_source != TS_MOBILE && _source != TS_PLAYER && _source != TS_CHAR)
		return;

	if (lua_isnil(to = lua_getparam(1)))
		return;
	if (lua_isnil(what = lua_getparam(2)) || (!lua_isstring(what)))
		return;

	if (lua_isstring(to))
	{
		name = lua_getstring(to);
	}
	else if (lua_isuserdata(to))
	{
		if (((struct user_data *) lua_getuserdata(to))->type == TS_MOBILE ||
				((struct user_data *) lua_getuserdata(to))->type == TS_PLAYER ||
				((struct user_data *) lua_getuserdata(to))->type == TS_CHAR)
		{
			ch = (struct char_data *)
				((struct user_data *) lua_getuserdata(to))->data;
			name = (IS_NPC(ch) ? fname(ch->player.name) : GET_NAME(ch));
		}
	}

	if (name)
	{
		sprintf(buf, " %s %s", name, lua_getstring(what));
		do_spec_comm((struct char_data *) _me, buf, 0, SCMD_ASK);
	}
}

/*****************************************************************************/

void lm_whisper(void)
{
	struct char_data *ch = NULL;
	char *name = NULL;
	lua_Object to, what;

	ACMD(do_spec_comm);

	if (_source != TS_MOBILE && _source != TS_PLAYER && _source != TS_CHAR)
		return;

	if (lua_isnil(to = lua_getparam(1)))
		return;
	if (lua_isnil(what = lua_getparam(2)) || (!lua_isstring(what)))
		return;

	if (lua_isstring(to))
	{
		name = lua_getstring(to);
	}
	else if (lua_isuserdata(to))
	{
		if (((struct user_data *) lua_getuserdata(to))->type == TS_MOBILE ||
				((struct user_data *) lua_getuserdata(to))->type == TS_PLAYER ||
				((struct user_data *) lua_getuserdata(to))->type == TS_CHAR)
		{
			ch = (struct char_data *)
				((struct user_data *) lua_getuserdata(to))->data;
			name = (IS_NPC(ch) ? fname(ch->player.name) : GET_NAME(ch));
		}
	}

	if (name)
	{
		sprintf(buf, " %s %s", name, lua_getstring(what));
		do_spec_comm((struct char_data *) _me, buf, 0, SCMD_WHISPER);
	}
}

/*****************************************************************************/

void lm_getname(void)
{
	lua_Object who;
	struct char_data *ch = NULL;
	struct obj_data  *ob = NULL;
	char *name = NULL;

	if (lua_isnil(who = lua_getparam(1)))
	{
		name = "";
	}
	else if (lua_isstring(who))
	{
		name = lua_getstring(who);
	}
	else if (lua_isuserdata(who))
	{
		if (((struct user_data *) lua_getuserdata(who))->type == TS_MOBILE ||
				((struct user_data *) lua_getuserdata(who))->type == TS_PLAYER ||
				((struct user_data *) lua_getuserdata(who))->type == TS_CHAR)
		{
			ch = (struct char_data *)
				((struct user_data *) lua_getuserdata(who))->data;
			name = (IS_NPC(ch) ? fname(ch->player.name) : GET_NAME(ch));
		}
		else if (((struct user_data *) lua_getuserdata(who))->type == TS_OBJECT)
		{
			ob = (struct obj_data *)
				((struct user_data *) lua_getuserdata(who))->data;
			name = fname(ob->name);
		}
	}
	else
	{
		name = "";
	}

	lua_pushstring(name);
}

/*****************************************************************************/

void lm_getshort(void)
{
	lua_Object who;
	struct char_data *ch = NULL;
	struct obj_data  *ob = NULL;
	char *descr = NULL;

	if (lua_isnil(who = lua_getparam(1)))
	{
		descr = "";
	}
	else if (lua_isstring(who))
	{
		descr = lua_getstring(who);
	}
	else if (lua_isuserdata(who))
	{
		if (((struct user_data *) lua_getuserdata(who))->type == TS_MOBILE ||
				((struct user_data *) lua_getuserdata(who))->type == TS_PLAYER ||
				((struct user_data *) lua_getuserdata(who))->type == TS_CHAR)
		{
			ch = (struct char_data *)
				((struct user_data *) lua_getuserdata(who))->data;
			descr = GET_NAME(ch);
		}
		else if (((struct user_data *) lua_getuserdata(who))->type == TS_OBJECT)
		{
			ob = (struct obj_data *)
				((struct user_data *) lua_getuserdata(who))->data;
			descr = ob->short_description;
		}
	}
	else
	{
		descr = "";
	}

	lua_pushstring(descr);
}

/*****************************************************************************/

void lm_cap(void)
{
	lua_Object str;

	str = lua_getparam(1);

	if (lua_isstring(str))
		lua_pushstring(CAP(lua_getstring(str)));
	else
		lua_pushobject(str);
}

/*****************************************************************************/

void lm__print(void)
{
	lua_Object str;

	if (lua_isstring(str = lua_getparam(1)))
		fprintf(stderr, "%s\n", lua_getstring(str));
}

/*****************************************************************************/

static struct lua_reg lm_functable[] = {
	{ "_print"   , lm__print   },
	{ "number"   , lm_number   },
	{ "dice"     , lm_dice     },
	{ "say"      , lm_say      },
	{ "isroom"   , lm_isroom   },
	{ "isobject" , lm_isobject },
	{ "ischar"   , lm_ischar   },
	{ "isplayer" , lm_isplayer },
	{ "isnpc"    , lm_isnpc    },
	{ "tell"     , lm_tell     },
	{ "ask"      , lm_ask      },
	{ "whisper"  , lm_whisper  },
	{ "get_name" , lm_getname  },
	{ "get_short", lm_getshort },
	{ "cap"      , lm_cap      }
};

void lm_funcs_init(void)
{
  luaI_openlib(lm_functable, (sizeof(lm_functable) / sizeof(struct lua_reg)));
}

/*****************************************************************************/
