/******************************************************************************
* File: lua_script.c                                                          *
* All functions for interfacing between CircleMUD and the Lua scriting system.*
*                                                                             *
* This file is Copyright (C) 1997 by Joachim Pileborg, all rights reserved.   *
******************************************************************************/

#define __LUA_SCRIPT_C__

#include <lua.h>
#include <lualib.h>

#include "conf.h"
#include "sysdep.h"

#include "structs.h"
#include "utils.h"
#include "db.h"
#include "comm.h"
#include "triggers.h"

/*****************************************************************************/

void lm_funcs_init(void);  /* from lua_funcs.c */

/*****************************************************************************/

#define SCRIPT_DIR "scripts"

void             *_me     = NULL;
int               _room   = NOWHERE;
struct obj_data  *_object = NULL;
struct char_data *_char   = NULL;
char             *_text   = NULL;

int   _source = -1;
char *_script = NULL;
char *_func   = NULL;

struct user_data *_userdata_list = NULL;

/*****************************************************************************/

/*
 * FUNCTION:    init_lua()
 * DESCRIPTION: Initialize the interface between CircleMUD and the
 *              Lua script system.
 */
int init_lua(void)
{
	iolib_open();
	mathlib_open();
	strlib_open();

	/* initialize the mud specific Lua functions */
	lm_funcs_init();

	return 0;
}

/*****************************************************************************/

/*
 * FUNCTION:    mud_print()
 * DESCRIPTION: Print a string to all players in the current room.
 */
void mud_print(char *str)
{
	int room;
	struct obj_data *o;

	sprintf(buf, "%s\r\n", str);

	switch (_source)
	{
	case TS_ROOM:
		send_to_room(buf, (int) _me);
		break;
	case TS_OBJECT:
		o = (struct obj_data *) _me;

		if (o->in_room != NOWHERE)
			room = o->in_room;
		else if (o->carried_by)
			room = o->carried_by->in_room;
		else if (o->worn_by)
			room = o->worn_by->in_room;
		else
			room = NOWHERE;

		send_to_room(buf, room);
		break;
	case TS_MOBILE:
	case TS_PLAYER:
		send_to_room(buf, ((struct char_data *) _me)->in_room);
		break;
	default:
		printf(buf);
	}
}

/*****************************************************************************/

/*
 * FUNCTION:
 * ARGUMENTS:
 * DESCRIPTION: Load a Lua script, and eventually call a function in it.
 * RETURNS:     -1 if something went wrong
 *               0 if no function was called or the called function
 *                 didn't return anything
 *              else it returns what the called function returned
 */
int execute_script(const char *script, const char *func, int source, void *me,
									 int room, struct obj_data *obj, struct char_data *ch,
									 char *txt)
{
	int err = 0;
	lua_Object result;

	if (!script)
	{
		sprintf(buf, "[SCRIPT] lua_script.c:execute_script() : script == NULL");
		log(buf);
		return -1;
	}

	_script = (char *) script;
	_func   = (char *) func;
	_source = source;
	_me     = me;
	_room   = room;
	_object = obj;
	_char   = ch;
	_text   = (txt ? strdup(txt) : NULL);

	if (!func)
	{
		/* ALLWAYS add the global variable "me" */
		if (_me)
			(source == TS_ROOM ? lua_pushnumber((int) _me) :
			 _lua_pushuserdata(_me, source));
		else
			lua_pushnil();
		lua_storeglobal("me");

		if (_room != NOWHERE)
		{
			lua_pushnumber(_room);
			lua_storeglobal("room");
		}
		if (_object != NULL)
		{
			_lua_pushuserdata(_object, TS_OBJECT);
			lua_storeglobal("object");
		}
		if (_char != NULL)
		{
			_lua_pushuserdata(_char, TS_CHAR);
			lua_storeglobal("ch");
		}
		if (_text != NULL)
		{
			_lua_pushuserdata(_text, TS_TEXT);
			lua_storeglobal("text");
		}
	}

	sprintf(buf, "%s/%s", SCRIPT_DIR, script);
	if ((err = lua_dofile(buf)) != 0)
	{
		switch (err)
		{
		case 2:
			sprintf(buf, "[SCRIPT] Could not call script %s: No such file.", script);
			break;
		default:
			sprintf(buf, "[SCRIPT] Could not call script %s: Unknown error %d.",
							script, err);
			break;
		}

		log(buf);
		return -1;
	}

	if (!func)
	{
		/* no function to call, just return */
		return 0;
	}

	/* me is allways the first argument */
	_lua_pushuserdata(_me, _source);

	(_room   == NOWHERE ? : _lua_pushuserdata((void *) _room, TS_ROOM));
	(_object == NULL    ? : _lua_pushuserdata(_object, TS_OBJECT));
	(_char   == NULL    ? : _lua_pushuserdata(_char, TS_CHAR));
	(_text   == NULL    ? : _lua_pushuserdata(_text, TS_TEXT));

	lua_call((char *) func);

	result = lua_getresult(1);

	if (_text)
		free(_text);

	return (lua_isnumber(result) ? (int) lua_getnumber(result) : 0);
}

/*****************************************************************************/


void _lua_pushuserdata(void *data, int type)
{
	struct user_data *d;

	CREATE(d, struct user_data, 1);
	d->data = data;
	d->type = type;

	d->next = _userdata_list;
	_userdata_list = d;

	lua_pushuserdata(d);
}

/*****************************************************************************/
