/*  forms.c: The way to read information from an HTML form. */

/*  Copyright (c) 1995 Universal Access, Inc
    Author: Brian J. Fox (bfox@ua.com) Sat May 20 11:38:02 1995.  */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>

#include "forms.h"
#include "xmalloc.h"

static void cleanup (char *string);
static int parse_hex_pair (char *pair_start);

extern char **environ;

/* Read the input data from all of the available sources.  This means
   the environment variables PATH_INFO and QUERY_STRING, the contents
   of standard input, if there is any, and the arguments passed into
   the CGI program.  A single string is returned.  The arguments are
   returned in the item PROGRAM-ARGUMENTS. */
POSTED_ITEM **
forms_input_data (int argc, char *argv[])
{
  register int i;
  int content_length = 0;
  char *env_item;
  POSTED_ITEM **items = (POSTED_ITEM **)NULL;
  int items_slots = 0, items_index = 0;
  char *data_string = (char *)NULL;
  int data_string_size = 0, data_string_index = 0;

  /* First, get the program arguments if there are any. */
  if (argc != 1)
    {
      POSTED_ITEM *item;

      item = (POSTED_ITEM *)xmalloc (sizeof (POSTED_ITEM));
      item->name = strdup ("PROGRAM-ARGUMENTS");
      item->values = (char **)xmalloc (argc * sizeof (char *));

      for (i = 1; i < argc; i++)
	item->values[i - 1] = strdup (argv[i]);

      item->values[i - 1] = (char *)NULL;

      items = (POSTED_ITEM **)xmalloc
	((items_slots = 10) * sizeof (POSTED_ITEM *));
      items[0] = item;
      items[1] = (POSTED_ITEM *)NULL;
      items_index = 1;
    }

  /* Now get all of the environment variables. */
  for (i = 0; environ != (char **)NULL && environ[i] != (char *)NULL; i++)
    {
      char *name, *value;
      POSTED_ITEM *item;

      name = strdup (environ[i]);
      value = strchr (name, '=');

      if (value)
	{
	  *value = '\0';
	  value++;
	  value = strdup (value);
	}

      item = (POSTED_ITEM *)xmalloc (sizeof (POSTED_ITEM));
      item->name = name;
      item->values = (char **)xmalloc (2 * sizeof (char *));
      item->values[0] = value;
      item->values[1] = (char *)NULL;

      if ((items_index + 2) > items_slots)
	items = (POSTED_ITEM **)xrealloc
	  (items, (items_slots += 10) * sizeof (POSTED_ITEM *));

      items[items_index++] = item;
      items[items_index] = (POSTED_ITEM *)NULL;
    }

  /* If there is any input available from standard input, then read it
     now. */
  if (((env_item = getenv ("CONTENT_LENGTH")) != (char *)NULL) &&
      ((content_length = atoi (env_item)) != 0))
    {
      data_string = (char *)xmalloc (1 + (data_string_size = content_length));
      read (fileno (stdin), data_string, content_length);

      data_string[content_length] = '\0';
      data_string_index = content_length;
    }

  /* If any input is coming from QUERY_STRING or PATH_INFO, get it now. */
  {
    char *names[3] = { "QUERY_STRING", "PATH_INFO", (char *)NULL };

    for (i = 0; names[i]; i++)
      {
	if (((env_item = getenv (names[i])) != (char *)NULL) &&
	    (*env_item != '\0'))
	  {
	    int max_len;

	    if ((strcmp (names[i], "PATH_INFO") == 0) &&
		*env_item == '/')
	      env_item++;

	    max_len = data_string_index + strlen (env_item) + 3;

	    if (max_len > data_string_size)
	      data_string = (char *)xrealloc
		(data_string, (data_string_size = max_len));

	    if (data_string_index != 0)
	      data_string[data_string_index++] = '&';

	    strcpy (data_string + data_string_index, env_item);
	    data_string_index += strlen (env_item);
	    data_string[data_string_index] = '\0';
	  }
      }
  }

  /* DATA_STRING may contain any number of name/value pairs, including
     none.  If there are any, add them to the list of ITEMS. */
  if (data_string)
    {
      items = forms_parse_data_string
	(data_string, items, &items_index, &items_slots);
      free (data_string);
    }

  return (items);
}

/* Read name/value pairs from BUFFER.  The pairs are delimited with ampersand
   (`&') or end of data.  The name is separated from the value by an equals
   sign (`=').  Space characters are encoded as plus signs (`+').  A percent
   sign (`%') is used to introduce two hex digits, which when coerced to an
   ASCII character is the result.  This mechanism is used to get plus signs
   into the name or value string, for example. */
POSTED_ITEM **
forms_parse_data_string (char *buffer, POSTED_ITEM **array, int *array_indexp,
			 int *array_sizep)
{
  register int i, j, start, length;
  int array_index = *array_indexp;
  int array_size = *array_sizep;

  /* Strip out the name/value pairs, and add them to the array. */
  length = strlen (buffer);
  start = 0;

  while (start < length)
    {
      int equal = -1;
      POSTED_ITEM *item = (POSTED_ITEM *)NULL;
      char *name, *value;

      for (i = start; buffer[i] != '\0' && buffer[i] != '&'; i++)
	if (buffer[i] == '=')
	  equal =  i;

      /* If there was no equal sign found, we are done parsing the pairs,
	 so simply return the current array. */
      if (equal < 0)
	return (array);

      /* `EQUAL' is the offset of the name/value separator.
	 `I' is the offset of the delimiter between pairs. */
      buffer[equal] = '\0';
      buffer[i] = '\0';

      name = strdup (buffer + start);
      value = strdup (buffer + equal + 1);

      cleanup (name);
      cleanup (value);

      /* Get the appropriate item pointer. */
      for (j = 0, item = (POSTED_ITEM *)NULL; j < array_index; j++)
	{
	  if (strcasecmp (array[j]->name, name) == 0)
	    {
	      item = array[j];
	      break;
	    }
	}

      if (!item)
	{
	  item = (POSTED_ITEM *)xmalloc (sizeof (POSTED_ITEM));
	  item->name = name;
	  item->values = (char **)xmalloc (2 * sizeof (char *));
	  item->values[0] = value;
	  item->values[1] = (char *)NULL;
	  name = (char *)NULL;

	  if (array_index + 2 > array_size)
	    array = (POSTED_ITEM **)xrealloc
	      (array, (array_size += 20) * sizeof (POSTED_ITEM *));

	  array[array_index++] = item;
	  array[array_index] = (POSTED_ITEM *)NULL;
	}
      else
	{
	  /* Item already in existence.  Add this value to the end of the
	     values array. */
	  for (j = 0; item->values[j] != (char *)NULL; j++);
	  item->values = (char **)xrealloc
	    (item->values, (j + 3) * sizeof (char *));

	  item->values[j++] = value;
	  item->values[j] = (char *)NULL;
	}

      start = i + 1;
    }

  *array_indexp = array_index;
  *array_sizep = array_size;
  return (array);
}

/* Turn ITEMS into a string suitable for appending onto a URL.
   This means that we encode special characters, and write name
   value pairs into a new string.
   A newly allocated string is returned. */
char *
forms_unparse_items (POSTED_ITEM **items)
{
  register int i;
  POSTED_ITEM *item;
  char *result = (char *)NULL;
  int result_index = 0;
  int result_size = 0;

  if (items == (POSTED_ITEM **)NULL)
    return ((char *)NULL);

  for (i = 0; (item = items[i]) != (POSTED_ITEM *)NULL; i++)
    {
      register int j;

      if (item->values == (char **)NULL)
	continue;

      for (j = 0; item->values[j] != (char *)NULL; j++)
	{
	  register int from, to, c;
	  char *pair;

	  pair = (char *)xmalloc
	    (2 + strlen (item->name) + (3 * strlen (item->values[j])));

	  for (from = 0, to = 0; (c = item->name[from]) != '\0'; from++)
	    {
	      if ((isalnum (c)) || (strchr (".-_@:", c) != (char *)NULL))
		pair[to++] = c;
	      else if (c == ' ')
		pair[to++] = '+';
	      else
		{
		  sprintf (pair + to, "%%%02X", c);
		  to += 3;
		}
	    }

	  pair[to++] = '=';

	  for (from = 0; (c = item->values[j][from]) != '\0'; from++)
	    {
	      if ((isalnum (c)) || (strchr (".-_@:", c) != (char *)NULL))
		pair[to++] = c;
	      else if (c == ' ')
		pair[to++] = '+';
	      else
		{
		  sprintf (pair + to, "%%%02X", c);
		  to += 3;
		}
	    }

	  pair[to] = '\0';

	  /* Add this pair to our string. */
	  if ((result_index + strlen (pair) + 2) > result_size)
	    result = (char *)xrealloc
	      (result, (result_size += 100 + strlen (pair)));

	  /* If there is already a pair present, separate it from this
	     one with an ampersand. */
	  if (result_index != 0)
	    result[result_index++] = '&';

	  strcpy (result + result_index, pair);
	  result_index += strlen (pair);
	  result[result_index] = '\0';
	  free (pair);
	}
    }
  return (result);
}

/* Do the `%FF' and `+' hacking on string.  We can do this hacking in
   place, since the resultant string cannot be longer than the input
   string. */
static void
cleanup (char *string)
{
  register int i, j, len;
  char *dest;

  len = strlen (string);
  dest = (char *)alloca (1 + len);

  for (i = 0, j = 0; i < len; i++)
    {
      switch (string[i])
	{
	case '%':
	  dest[j++] = parse_hex_pair (string + i + 1);
	  i += 2;
	  break;

	case '+':
	  dest[j++] = ' ';
	  break;

	default:
	  dest[j++] = string[i];
	}
    }

  dest[j] = '\0';
  strcpy (string, dest);
}

static int
parse_hex_pair (char *pair_start)
{
  int value = 0;
  int char1, char2;

  char1 = char2 = 0;

  char1 = *pair_start;

  if (char1)
    char2 = (pair_start[1]);

  if (isupper (char1))
    char1 = tolower (char1);

  if (isupper (char2))
    char2 = tolower (char2);

  if (isdigit (char1))
    value = char1 - '0';
  else if ((char1 <= 'f') && (char1 >= 'a'))
    value = 10 + (char1 - 'a');

  if (isdigit (char2))
    value = (value * 16) + (char2 - '0');
  else if ((char2 <= 'f') && (char2 >= 'a'))
    value = (value * 16) + (10 + (char2 - 'a'));

  return (value);
}

/* Search through ITEMS for TAG, and return the associated array of
   character strings.  If TAG isn't present, return a NULL pointer. */
char **
forms_find_tag (POSTED_ITEM **items, char *tag)
{
  register int i;

  if (items)
    {
      for (i = 0; items[i] != (POSTED_ITEM *)NULL; i++)
	if (strcasecmp (items[i]->name, tag) == 0)
	  return (items[i]->values);
    }

  return ((char **)NULL);
}

static int
array_len (char **array)
{
  register int i = 0;

  if (array != (char **)NULL)
    for (i = 0; array[i] != (char *)NULL; i++);

  return (i);
}

/* Using the list in ITEMS, give TAG an additional value of VALUE.
   Returns the ITEMS, or if ITEMS was NULL, a new list.
   MUST_BE_UNIQUE says don't add VALUE if there is already a value
   that matches. */
static POSTED_ITEM **
add_value (POSTED_ITEM **items, char *tag, char *value, int must_be_unique)
{
  register int i;
  int dont_add = 0;
  POSTED_ITEM **list = items;
  int list_size, list_index;
  POSTED_ITEM *item = (POSTED_ITEM *)NULL;

  if (!list)
    {
      list = (POSTED_ITEM **)xmalloc (sizeof (POSTED_ITEM *));
      list[0] = (POSTED_ITEM *)NULL;
    }

  list_size = array_len ((char **)list);
  list_index = list_size;

  /* Try to locate TAG. */
  for (i = 0; list[i] != (POSTED_ITEM *)NULL; i++)
    {
      if (strcasecmp (tag, list[i]->name) == 0)
	  {
	    item = list[i];
	    break;
	  }
    }

  /* If not found, create a new item. */
  if (item == (POSTED_ITEM *)NULL)
    {
      item = (POSTED_ITEM *)xmalloc (sizeof (POSTED_ITEM));
      item->name = strdup (tag);
      item->values = (char **)NULL;

      list = (POSTED_ITEM **)xrealloc
	(list, ((list_size += 2) * sizeof (POSTED_ITEM *)));

      list[list_index++] = item;
      list[list_index] = (POSTED_ITEM *)NULL;
    }

  /* If we are not allowing duplicate values to appear in a single
     item, then check for that case now. */
  if (must_be_unique && item->values != (char **)NULL && value != (char *)NULL)
    {
      for (i = 0; item->values[i] != (char *)NULL; i++)
	if (strcmp (value, item->values[i]) == 0)
	  {
	    dont_add++;
	    break;
	  }
    }

  /* If the value is empty, then don't add it. */
  if (value == (char *)NULL)
    dont_add++;

  if (!dont_add)
    {
      /* We have the item.  Append value to the list of values for
	 this item. */
      list_index = array_len (item->values);
      list_size = 2 + list_index;
      item->values = (char **)xrealloc
	(item->values, (list_size * sizeof (char *)));

      item->values[list_index++] = strdup (value);
      item->values[list_index] = (char *)NULL;
    }

  return (list);
}

static POSTED_ITEM **
merge_one (POSTED_ITEM **collector, POSTED_ITEM **items)
{
  register int i, j;

  if (items != (POSTED_ITEM **)NULL)
    {
      for (i = 0; items[i] != (POSTED_ITEM *)NULL; i++)
	{
	  POSTED_ITEM *item = items[i];
	  char **values = item->values;

	  if (values != (char **)NULL)
	    for (j = 0; values[j] != (char *)NULL; j++)
	      collector = add_value (collector, item->name, values[j], 1);
	}
    }

  return (collector);
}

/* Using the list in ITEMS, remove from TAG an the value of VALUE.
   Returns ITEMS. */
POSTED_ITEM **
forms_remove_value (POSTED_ITEM **items, char *tag, char *value)
{
  char **values;

  values = forms_find_tag (items, tag);

  if (values)
    {
      register int i;

      for (i = 0; values[i] != (char *)NULL; i++)
	if (strcasecmp (values[i], value) == 0)
	  {
	    register int j;

	    /* Delete the item. */
	    for (j = i + 1; values[j] != (char *)NULL; j++, i++)
	      values[i] = values[j];

	    values[i] = (char *)NULL;
	    break;
	  }
    }

  return (items);
}

/* Using the list in ITEMS, give TAG an additional value of VALUE.
   Returns the ITEMS, or if ITEMS was NULL, a new list. */
POSTED_ITEM **
forms_add_value (POSTED_ITEM **items, char *tag, char *value)
{
  return (add_value (items, tag, value, 0));
}

/* Totally remove from ITEMS the variable named TAG, and all of the
   associated values. */
POSTED_ITEM **
forms_delete_item (POSTED_ITEM **items, char *tag)
{
  if (items != (POSTED_ITEM **)NULL)
    {
      register int i;

      for (i = 0; items[i] != (POSTED_ITEM *)NULL; i++)
	if (strcmp (items[i]->name, tag) == 0)
	  {
	    free (items[i]->name);

	    if (items[i]->values != (char **)NULL)
	      {
		register int j;
		char **values = items[i]->values;

		for (j = 0; values[j] != (char *)NULL; j++)
		  free (values[j]);

		free (values);
	      }

	    free (items[i]);
	    while ((items[i] = items[i + 1]) != (POSTED_ITEM *)NULL) i++;
	  }
    }
  return (items);
}

/* Given 2 arrays of POSTED_ITEM *, merge the contents, returning a
   new array.  */
POSTED_ITEM **
forms_merge_data (POSTED_ITEM **list1, POSTED_ITEM **list2)
{
  POSTED_ITEM **result = (POSTED_ITEM **)NULL;

  result = merge_one (result, list1);
  result = merge_one (result, list2);

  return (result);
}

/* Free the data associated with ITEMS. */
void
forms_free_items (POSTED_ITEM **items)
{
  if (items != (POSTED_ITEM **)NULL)
    {
      register int i;
      POSTED_ITEM *item;

      for (i = 0; (item = items[i]) != (POSTED_ITEM *)NULL; i++)
	{
	  if (item->name != (char *)NULL)
	    free (item->name);

	  if (item->values != (char **)NULL)
	    {
	      register int j;

	      for (j = 0; item->values[j] != (char *)NULL; j++)
		free (item->values[j]);

	      free (item->values);
	    }

	  free (item);
	}

      free (items);
    }
}
