/*
 * Filename: safestring.h
 * Project:  SafeString
 *
 * Function: SafeStrings are ANSI-C objects that contain strings.
 *           The string contents is be accessed and manipulated
 *           through the use of C functions.
 *
 *           No more malloc errors!  This code should be useful
 *           to all C programmers, especially beginners without
 *           much experience with pointer manipulation.  SafeStrings
 *           make string manipulations easy.
 *
 * Author:   Pascal Forget <pascal@wsc.com>
 *
 * Copyright (C) 1995-1996 Pascal Forget.  All Rights Reserved.
 *
 * PASCAL FORGET MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY
 * OF THIS SOFTWARE FOR ANY PURPOSE.  IT IS SUPPLIED "AS IS"
 * WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for any purpose and without fee is
 * hereby granted, provided that the above copyright notice
 * appear in all copies of the software.
 */

#ifndef SAFESTRING_H__
#define SAFESTRING_H__

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

typedef struct {
    char *buffer;
    unsigned int length;
    unsigned int capacity;
} SafeString;


/* Basic functions */

void ss_append_character(SafeString *str, char c);
void ss_append_string(SafeString *str, SafeString *other_str);
void ss_append_c_string(SafeString *str, const char *cstring);
void ss_append_c_string_length(SafeString *str, const char *cstring, int length);
void ss_fast_append_c_string_length(SafeString *str,const char *cstring,int length);
char ss_character_at_index(SafeString *str, int index);
int ss_compare_strings(SafeString *str1, SafeString *str2);
void ss_copy_string(SafeString *to_str, SafeString *from_str);
void ss_copy_c_string(SafeString *str, const char *cstring);
SafeString *ss_create_string(void);
SafeString *ss_create_string_from_file(const char *filename);
SafeString *ss_create_string_with_capacity(unsigned int capacity);
SafeString *ss_create_string_with_c_string(const char *cstring);
void ss_destroy_string(SafeString *str);
SafeString *ss_duplicate_string(SafeString *str);
void ss_insert_c_string_at_index(SafeString *str, const char *cstring, int index);
void ss_save_string_to_file(SafeString *str, const char *filename);
int ss_strings_are_equal(SafeString *str1, SafeString *str2);
int ss_string_length(SafeString *str);
void ss_set_string_capacity(SafeString *str, unsigned int cap);

/* String values */

const char *ss_string_value(SafeString *str);
int ss_int_value(SafeString *str);
float ss_float_value(SafeString *str);
double ss_double_value(SafeString *str);
void ss_remove_count_at_index(SafeString *str, int count, int index);
void ss_set_string_value(SafeString *str, const char *new_value);
void ss_set_int_value(SafeString *str, int new_value);
void ss_set_float_value(SafeString *str, float new_value);
void ss_set_double_value(SafeString *str, double new_value);

/* Other goodies */

void ss_capitalize_string(SafeString *str);  /* "new york" -> "New York" */
int ss_has_prefix(SafeString *str, SafeString *prefix);
int ss_has_case_prefix(SafeString *str, SafeString *prefix);/* case insensitive*/
int ss_has_suffix(SafeString *str, SafeString *suffix);
SafeString *ss_last_path_component(SafeString *str);
SafeString *ss_path_extension(SafeString *str);
void ss_replace_substring(SafeString *str, const char *old, const char *new);
void ss_remove_trailing_zeroes(SafeString *str);
SafeString *ss_string_by_deleting_last_path_component(SafeString *str);
SafeString *ss_string_by_deleting_path_extension(SafeString *str);

/* Trim leading and trailing blanks */
void ss_trim_string(SafeString *str);

/* replacement for strdup() */
char *safe_c_string_copy(const char *s);
void safe_c_string_free(char *s);

#endif
