/*
** RESULTS.C - This is the ODBC sample driver code for
** returning results and information about results.
**
**	This code is furnished on an as-is basis as part of the ODBC SDK and is
**	intended for example purposes only.
**
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <assert.h>

#define MIN(A,B) (A) < (B) ? (A) : (B)
//	-	-	-	-	-	-	-	-	-

#include "gorta.h"

#include <msql.h>

//	-	-	-	-	-	-	-	-	-

//	This returns the number of columns associated with the database
//	attached to "hstmt".

RETCODE SQL_API SQLNumResultCols(
	HSTMT	  hstmt,
	SWORD FAR *pccol)
{
	LPSTMT lpstmt;
	m_result* MResult;
	RETCODE retcode = SQL_SUCCESS;
    
    lpstmt = (LPSTMT)hstmt;
	assert(lpstmt->lMagic == STATEMENT_MAGIC);
    
    if (!(lpstmt && lpstmt->pvdMResult))
    {
    	/*
    	 * I don't do a separate check to ensure that 
    	 * lpstmt is valid. The driver manager should
    	 * do that.
    	 */
    	*pccol = 0; /* No result Columns */
    }
    else
    {
    	MResult = lpstmt->pvdMResult;
    	*pccol = MResult->numFields;
    }
	retcode = SQL_SUCCESS;

Exit:
	return(retcode);
}

//	-	-	-	-	-	-	-	-	-

//	Return information about the database column the user wants
//	information about.

RETCODE SQL_API SQLDescribeCol(
	HSTMT	   hstmt,
	UWORD	   icol,
	UCHAR  FAR *szColName,
	SWORD	   cbColNameMax,
	SWORD  FAR *pcbColName,
	/* UNALIGNED */ SWORD  FAR *pfSqlType,
	/* UNALIGNED */ UDWORD FAR *pcbColDef,
	/* UNALIGNED */ SWORD  FAR *pibScale,
	SWORD  FAR *pfNullable)
{
	LPSTMT lpstmt;
	m_result* MResult;
	RETCODE retcode = SQL_SUCCESS;
	m_fdata *MFData;
	unsigned int iLoop;
    
    /*
     * TBD - check for NULL arguments and set errors
     */
    lpstmt = (LPSTMT)hstmt;
	assert(lpstmt->lMagic == STATEMENT_MAGIC);
    if (!(lpstmt && lpstmt->pvdMResult))
    {
    	goto Exit;
    }
    MResult = lpstmt->pvdMResult;
    
	/*
	 * Find the required column. The columns are indexed from
	 * 1 (0 is a bookmark column.)
	 */
	for (iLoop = 1, MFData = MResult->fieldData; (iLoop < icol) && MFData; iLoop++)
		MFData = MFData->next;

	if (!MFData)
		goto Exit;
	    
	if (szColName && cbColNameMax > 0)
	{
		sprintf(szColName, "%.*s", cbColNameMax, MFData->field.name);
	}
	if (pcbColName)
		*pcbColName = strlen(MFData->field.name);
	if (pfNullable)
		*pfNullable = (MFData->field.flags & NOT_NULL_FLAG) ?
			SQL_NO_NULLS : SQL_NULLABLE;
	switch (MFData->field.type)
	{
		case SHORT_TYPE:
			pfSqlType && (*pfSqlType = SQL_SMALLINT);
			pcbColDef && (*pcbColDef = 0);
			pibScale  && (*pibScale = 0);
			break;	
		case INT_TYPE:
			pfSqlType && (*pfSqlType = SQL_INTEGER);
			pcbColDef && (*pcbColDef = 0);
			pibScale  && (*pibScale = 0);
			break;
		case CHAR_TYPE:
			pfSqlType && (*pfSqlType = SQL_VARCHAR);
			pcbColDef && (*pcbColDef = MFData->field.length);
			pibScale  && (*pibScale = 0);
			break;
		default:
			goto Exit; /* Cannot handle it */
	}
     
Exit:
	return(retcode);
}

//	-	-	-	-	-	-	-	-	-

//	Returns result column descriptor information for a result set.

RETCODE SQL_API SQLColAttributes(
	HSTMT	   hstmt,
	UWORD	   icol,
	UWORD	   fDescType,
	PTR 	   rgbDesc,
	SWORD	   cbDescMax,
	SWORD  FAR *pcbDesc,
	SDWORD FAR *pfDesc)
{
	LPSTMT lpstmt;
	m_result* MResult;
	RETCODE retcode = SQL_SUCCESS;
	m_fdata *MFData;
	unsigned int iLoop;
    
    /*
     * TBD - check for NULL arguments and set errors
     */
    lpstmt = (LPSTMT)hstmt;
    if (!(lpstmt && lpstmt->pvdMResult))
    {
    	goto Exit;
    }
	assert(lpstmt->lMagic == STATEMENT_MAGIC);
    MResult = lpstmt->pvdMResult;

	if (fDescType != SQL_COLUMN_COUNT)
	{    
		/*
	 	 * Find the required column. The columns are indexed from
		 * 1 (0 is a bookmark column.)
		 */
		for (iLoop = 1, MFData = MResult->fieldData; (iLoop < icol) && MFData; iLoop++)
			MFData = MFData->next;

		if (!MFData)
			goto Exit;
	}
	    
	switch (fDescType)
	{
	case SQL_COLUMN_AUTO_INCREMENT:
		*pfDesc = FALSE; // Not supported
		pcbDesc && (*pcbDesc = sizeof(int));
		break;
	case SQL_COLUMN_CASE_SENSITIVE:
		if (MFData->field.type == CHAR_TYPE)
			*pfDesc = TRUE;
		else
			*pfDesc = FALSE; // Numeric
		pcbDesc && (*pcbDesc = sizeof(int));
		break;
	case SQL_COLUMN_COUNT:
		*pfDesc = MResult->numFields;
		pcbDesc && (*pcbDesc = sizeof(int));
		break;
	case SQL_COLUMN_DISPLAY_SIZE:
		if (MFData->field.type == CHAR_TYPE)
			*pfDesc = MFData->field.length;
		else
			*pfDesc = 10;
		pcbDesc && (*pcbDesc = sizeof(int));
		break;
	case SQL_COLUMN_LENGTH:
		*pfDesc = MFData->field.length;
		pcbDesc && (*pcbDesc = sizeof(int));
		break;		
	case SQL_COLUMN_MONEY:
		*pfDesc = FALSE; // Not supported
		pcbDesc && (*pcbDesc = sizeof(int));
		break;
	case SQL_COLUMN_LABEL:
	case SQL_COLUMN_NAME:
	case SQL_COLUMN_TABLE_NAME:
		sprintf((char *)pfDesc, "%.*s", (int)cbDescMax, MFData->field.name);	
		pcbDesc && (*pcbDesc = strlen(MFData->field.name));
		break;
	case SQL_COLUMN_NULLABLE:
		*pfDesc = (MFData->field.flags & NOT_NULL_FLAG) ?
			SQL_NO_NULLS : SQL_NULLABLE;
		pcbDesc && (*pcbDesc = sizeof(int));
		break;
	case SQL_COLUMN_OWNER_NAME:
		*(char *)pfDesc = '\0';
		pcbDesc && (*pcbDesc = 0);
		break;
	case SQL_COLUMN_PRECISION:
		*(int *)pfDesc = 0; // Not supported
		pcbDesc && (*pcbDesc = sizeof(int));
		break;
	case SQL_COLUMN_QUALIFIER_NAME:
		*(char *)pfDesc = '\0';
		pcbDesc && (*pcbDesc = 0);
		break;
	case SQL_COLUMN_SCALE:
		*(int *)pfDesc = 0; // Not supported
		pcbDesc && (*pcbDesc = sizeof(int));
		break;
	case SQL_COLUMN_SEARCHABLE:
		if (MFData->field.type == CHAR_TYPE)
			*(int *)pfDesc = SQL_SEARCHABLE;
		else
			*(int *)pfDesc = SQL_ALL_EXCEPT_LIKE;
		pcbDesc && (*pcbDesc = strlen((char *)pfDesc));
		break;
	case SQL_COLUMN_TYPE:
		if (MFData->field.type == CHAR_TYPE)
			*(int *)pfDesc = SQL_CHAR;
		else
			*(int *)pfDesc = SQL_INTEGER;
		pcbDesc && (*pcbDesc = sizeof(int));
		break;
	case SQL_COLUMN_TYPE_NAME:
		if (MFData->field.type == CHAR_TYPE)
			sprintf((char *)pfDesc, "%.*s", (int)cbDescMax, "CHAR");	
		else
			sprintf((char *)pfDesc, "%.*s", (int)cbDescMax, "INTEGER");	
		pcbDesc && (*pcbDesc = strlen((char *)pfDesc));
		break;
	case SQL_COLUMN_UNSIGNED:
		*(int *)pfDesc = FALSE;
		pcbDesc && (*pcbDesc = sizeof(int));
		break;
	case SQL_COLUMN_UPDATABLE:
		*pfDesc = SQL_COLUMN_UPDATABLE;
		pcbDesc && (*pcbDesc = sizeof(int));
		break;
	}
Exit:
	return SQL_SUCCESS;
}

//	-	-	-	-	-	-	-	-	-

//	Associate a user-supplied buffer with a database column.

RETCODE SQL_API SQLBindCol(
	HSTMT	   hstmt,
	UWORD	   icol,
	SWORD	   fCType,
	PTR 	   rgbValue,
	SDWORD	   cbValueMax,
	SDWORD FAR *pcbValue)
{
	LPSTMT lpstmt;
	m_result* MResult;
	RETCODE retcode = SQL_SUCCESS;

    /*
     * TBD - check for NULL arguments and set errors
     */
    lpstmt = (LPSTMT)hstmt;
    if (!(lpstmt && lpstmt->pvdMResult))
    {
    	goto Exit;
    }
	assert(lpstmt->lMagic == STATEMENT_MAGIC);
    MResult = lpstmt->pvdMResult;
    /*
     * If the bindings array is not yet allocated...
     */
    if (!lpstmt->pBindings)
    {
    	lpstmt->pBindings = calloc(MResult->numFields, sizeof(*lpstmt->pBindings));
    	assert(lpstmt->pBindings);
    }
	/*
	 * Initialise required field (offset by 1 for bookmark)
	 */
	assert((int)icol <= MResult->numFields);
	lpstmt->pBindings[icol -1].pvdBuffer = rgbValue;
    lpstmt->pBindings[icol -1].plResultLen = pcbValue;
    lpstmt->pBindings[icol -1].lMaxLen = cbValueMax;
    lpstmt->pBindings[icol -1].iType = fCType;
    
Exit:
	return(retcode);
}

//	-	-	-	-	-	-	-	-	-

//	Returns data for bound columns in the current row ("hstmt->iCursor"),
//	advances the cursor.

RETCODE SQL_API SQLFetch(
	HSTMT	hstmt)
{
	m_fdata	*curField;
	m_row curRow;
	int iField;
	LPSTMT lpstmt;
	m_result* MResult;
	RETCODE retcode = SQL_SUCCESS;

    /*
     * TBD - check for NULL arguments and set errors
     */
    lpstmt = (LPSTMT)hstmt;
    if (!(lpstmt && lpstmt->pvdMResult))
    {
		retcode = SQL_NO_DATA_FOUND;
    	goto Exit;
    }
	assert(lpstmt->lMagic == STATEMENT_MAGIC);
    MResult = lpstmt->pvdMResult;

	if (MResult->cursor && MResult->cursor->data == NULL)
	{
		/* No more data left */
		retcode = SQL_NO_DATA_FOUND;
		goto Exit;
	}

	if (!MResult->cursor)
		MResult->cursor = MResult->queryData;		/* Set the cursor to first position */
	else
		MResult->cursor = MResult->cursor->next;	/* Increment the cursor */		
	
	if (!MResult->cursor || !(curRow = MResult->cursor->data))
	{
		/* No more data left */
		retcode = SQL_NO_DATA_FOUND;
		goto Exit;
	}

	curField = MResult->fieldData; 

	if (lpstmt->pBindings)
	{
		for (iField=0; iField < MResult->numFields; iField++, curField = curField->next)
		{
			if (lpstmt->pBindings[iField].pvdBuffer)
			{
				if (!curRow[iField]) /* NULL value */
				{
					if (lpstmt->pBindings[iField].plResultLen) /* a length buffer available */
						*lpstmt->pBindings[iField].plResultLen = SQL_NULL_DATA;												
				}
                else 
                {
                	struct BindInfo *pBI;
                	pBI = &(lpstmt->pBindings[iField]);
					switch (curField->field.type)
					{
					case SHORT_TYPE:
						if (pBI->iType == SQL_C_SHORT || 
							pBI->iType == SQL_C_DEFAULT)
						{
							*((short *)pBI->pvdBuffer) = atoi(curRow[iField]);
							if (pBI->plResultLen)
								*pBI->plResultLen = sizeof(short);
						}
						break;						
					case INT_TYPE:
						if (pBI->iType == SQL_C_LONG || 
							pBI->iType == SQL_C_DEFAULT)
						{
							*((long *)pBI->pvdBuffer) = atol(curRow[iField]);
							if (pBI->plResultLen)
								*pBI->plResultLen = sizeof(long);
						}
						else if (pBI->iType == SQL_C_CHAR)
						{
							sprintf((char *)pBI->pvdBuffer, "%.*s",
								(int)pBI->lMaxLen, curRow[iField]);
							if (pBI->plResultLen)
								*pBI->plResultLen = strlen(curRow[iField]);
					}
						break;						
					case CHAR_TYPE:
						if (pBI->iType == SQL_C_CHAR || 
							pBI->iType == SQL_C_DEFAULT)
						{
							if (pBI->plResultLen)
								*pBI->plResultLen = 
									sprintf((char *)pBI->pvdBuffer, "%.*s",
									(int)pBI->lMaxLen, curRow[iField]);
							else
								sprintf((char *)pBI->pvdBuffer, "%.*s",
									(int)pBI->lMaxLen, curRow[iField]);
						}
					default:
						break;
					}
				}
			}
		}
	}

Exit:	
	return(retcode);
}

//	Returns result data for a single column in the current row.

RETCODE SQL_API SQLGetData(
	HSTMT	   hstmt,
	UWORD	   icol,
	SWORD	   fCType,
	PTR 	   rgbValue,
	SDWORD	   cbValueMax,
	SDWORD FAR *pcbValue)
{
	m_fdata	*curField;
	m_row curRow;
	unsigned int iField;
	LPSTMT lpstmt;
	m_result* MResult;
	RETCODE retcode = SQL_SUCCESS;

    /*
     * TBD - check for NULL arguments and set errors
     */
    lpstmt = (LPSTMT)hstmt;
    if (!(lpstmt && lpstmt->pvdMResult))
    {
		retcode = SQL_NO_DATA_FOUND;
    	goto Exit;
    }
	assert(lpstmt->lMagic == STATEMENT_MAGIC);
    MResult = lpstmt->pvdMResult;

	if (!MResult->cursor || !(curRow = MResult->cursor->data))
	{
		/* No more data left */
		retcode = SQL_NO_DATA_FOUND;
		goto Exit;
	}

	curField = MResult->fieldData; 

	/*
	 * Find the required field number. icol numbers begin from 1,
	 * allowing 0 to be used for a bookmark.
	 */
	for (iField = 0; iField < (unsigned)MResult->numFields && iField < icol - 1; iField++)
		curField = curField->next;

	if (iField == icol)
	{
		/* Outside the valid range - TBD correct error message*/
		retcode = SQL_NO_DATA_FOUND;
		goto Exit;
	}
	
	if (curRow[iField])
	{
		/* TBD - Type convertions etc */
		switch (curField->field.type)
		{
		case SHORT_TYPE:
			if (fCType == SQL_C_SSHORT || fCType == SQL_C_SHORT
				|| fCType == SQL_C_DEFAULT)
			{
				*((short *)rgbValue) = atoi(curRow[iField]);
				pcbValue && (*pcbValue = sizeof(short));
			}
			else if (fCType == SQL_C_CHAR)
			{
				sprintf((char *)rgbValue, "%.*s", (int)cbValueMax,
					(char *)curRow[iField]);
				pcbValue && (*pcbValue = strlen((char *)curRow[iField]));
			}
			break;						
		case INT_TYPE:
			if (fCType == SQL_C_LONG || fCType == SQL_C_DEFAULT)
			{
				*((long *)rgbValue) = atol(curRow[iField]);
				pcbValue && (*pcbValue = sizeof(long));
			}
			else if (fCType == SQL_C_CHAR)
			{
				sprintf((char *)rgbValue, "%.*s", (int)cbValueMax,
					(char *)curRow[iField]);
				pcbValue && (*pcbValue = strlen((char *)curRow[iField]));
			}
			break;						
		case CHAR_TYPE:
			if (fCType == SQL_C_CHAR || fCType == SQL_C_DEFAULT)
			{
				sprintf((char *)rgbValue, "%.*s", (int)cbValueMax,
					(char *)curRow[iField]);
				pcbValue && (*pcbValue = strlen((char *)curRow[iField]));
			}
			break;
		default:
			break;
		}
	}
	else
	{
		/* NULL value */
		pcbValue && (*pcbValue = SQL_NULL_DATA);
	}

Exit:	
	return(retcode);
}

//	-	-	-	-	-	-	-	-	-

//	This determines whether there are more results sets available for
//	the "hstmt".

RETCODE SQL_API SQLMoreResults(
	HSTMT	hstmt)
{
	LPSTMT lpstmt;
	m_result* MResult;
	RETCODE retcode = SQL_SUCCESS;
    
    lpstmt = (LPSTMT)hstmt;
	assert(lpstmt->lMagic == STATEMENT_MAGIC);
    
    /*
     * TBD - check for NULL arguments and set errors
     */
    if (!(lpstmt && lpstmt->pvdMResult))
    {	
    	retcode = SQL_NO_DATA_FOUND;
    	goto Exit;
    }
    MResult = lpstmt->pvdMResult;
    if (!(MResult->cursor && MResult->cursor->data))
    {	
    	retcode = SQL_NO_DATA_FOUND;
    	goto Exit;
    }

   	retcode = SQL_SUCCESS;

Exit:
	return(retcode);
}


//	-	-	-	-	-	-	-	-	-
//	Return 1 in most cases this is the right answer.
//	may need to alter the Mini-SQL server to return this
//	information.

RETCODE SQL_API SQLRowCount(
	HSTMT	   hstmt,
	SDWORD FAR *pcrow)
{
	LPSTMT lpstmt;
	RETCODE retcode = SQL_SUCCESS;
    
    lpstmt = (LPSTMT)hstmt;
	assert(lpstmt->lMagic == STATEMENT_MAGIC);
    
   	*pcrow = 1;
	retcode = SQL_SUCCESS;
  	goto Exit;

Exit:
	return(retcode);
}

//	-	-	-	-	-	-	-	-	-

//	This positions the cursor within a block of data.

RETCODE SQL_API SQLSetPos(
	HSTMT	hstmt,
	UWORD	irow,
	UWORD	fOption,
	UWORD	fLock)
{
	return SQL_SUCCESS;
}

//	-	-	-	-	-	-	-	-	-

//	This fetchs a block of data (rowset).

RETCODE SQL_API SQLExtendedFetch(
	HSTMT	   hstmt,
	UWORD	   fFetchType,
	SDWORD	   irow,
	UDWORD FAR *pcrow,
	UWORD  FAR *rgfRowStatus)
{
	return SQL_SUCCESS;
}

//	-	-	-	-	-	-	-	-	-

//	Returns the next SQL error information.

RETCODE SQL_API SQLError(
	LPENV	   lpenv,
	LPDBC	   lpdbc,
	HSTMT	   hstmt,
	UCHAR  FAR *szSqlState,
	SDWORD FAR *pfNativeError,
	UCHAR  FAR *szErrorMsg,
	SWORD	   cbErrorMsgMax,
	SWORD  FAR *pcbErrorMsg)
{
	RETCODE retcode = SQL_NO_DATA_FOUND;
	LPEI *plpei = NULL, lpei = NULL;

	if (hstmt)
	{
		LPSTMT lpstmt;
		lpstmt = hstmt;
		assert(lpstmt->lMagic == STATEMENT_MAGIC);
		plpei=&(lpstmt->lpei);
	}
	else if (lpdbc)
	{
		plpei=&(lpdbc->lpei);
	}
	else if (lpenv)
	{
		plpei=&(lpenv->lpei);
	}
	if (plpei && *plpei)
	{
		lpei = *plpei;
		*plpei = lpei->pNext;

		*pcbErrorMsg = strlen(lpei->szErrorMessage);
		*pfNativeError = lpei->lErrorCode;
		strncpy(szSqlState, lpei->szSqlState, 6);
		sprintf(szErrorMsg, "%.*s", MIN(*pcbErrorMsg, cbErrorMsgMax),
		 	lpei->szErrorMessage);
		if (*pcbErrorMsg > cbErrorMsgMax)
		{
			retcode = SQL_SUCCESS_WITH_INFO; /* Truncation occurred */
		}
		else
		{
			retcode = SQL_SUCCESS;
		}
		free(lpei);
	}
	else
	{
		retcode = SQL_NO_DATA_FOUND;
		strncpy(szSqlState, "00000", 6);
		*pcbErrorMsg = 0;
		if (cbErrorMsgMax > 0)
			*szErrorMsg = '\0';
	}
	return(retcode);
}

//	-	-	-	-	-	-	-	-	-
